/*
   Copyright (C) 2004 Paul Mackerras <paulus@samba.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#include "mc_include.h"

#define dis	VG_(print_codegen)

static int helper_offsets_loaded;
UInt offset_value_check[5];

static void load_helper_offsets(void)
{
    offset_value_check[0]
	= VG_(helper_offset)((Addr) &MC_(helperc_value_check0_fail));
    offset_value_check[1]
	= VG_(helper_offset)((Addr) &MC_(helperc_value_check1_fail));
    offset_value_check[2]
	= VG_(helper_offset)((Addr) &MC_(helperc_value_check2_fail));
    offset_value_check[3]
	= VG_(helper_offset)((Addr) &MC_(helperc_value_check3_fail));
    offset_value_check[4]
	= VG_(helper_offset)((Addr) &MC_(helperc_value_check4_fail));
}

static void tag1_op(TagOp op, int reg)
{
    int dreg, sreg;

    sreg = VG_(real_source_reg)(reg);
    dreg = VG_(real_dest_reg)(reg);
    switch (op) {
    case Tag_PCast40:
	VG_(emit_instr)(XFORM31(sreg, dreg, 0, MO_CNTLZW, 0));
	VG_(emit_instr)(MFORM(RLWINM, dreg, dreg, 27, 31, 31, 0));
	VG_(emit_instr)(XFORM31(dreg, dreg, dreg, MO_NOR, 0));
	if (dis)
	    VG_(printf)("\tcntlzw r%d,r%d\n\trlwinm r%d,r%d,27,31,31\n"
			"\tnot r%d,r%d\n", dreg, sreg, dreg, dreg, dreg, dreg);
	break;
    case Tag_PCast20:
	VG_(emit_instr)(MFORM(RLWINM, sreg, dreg, 0, 16, 31, 0));
	VG_(emit_instr)(XFORM31(dreg, dreg, 0, MO_CNTLZW, 0));
	VG_(emit_instr)(MFORM(RLWINM, dreg, dreg, 27, 31, 31, 0));
	VG_(emit_instr)(XFORM31(dreg, dreg, dreg, MO_NOR, 0));
	if (dis)
	    VG_(printf)("\trlwinm r%d,r%d,0,16,31\n\tcntlzw r%d,r%d\n"
			"\trlwinm r%d,r%d,27,31,31\n\tnot r%d,r%d\n",
			dreg, sreg, dreg, dreg, dreg, dreg, dreg, dreg);
	break;
    case Tag_PCast10:
	VG_(emit_instr)(MFORM(RLWINM, dreg, dreg, 0, 24, 31, 0));
	VG_(emit_instr)(XFORM31(dreg, dreg, 0, MO_CNTLZW, 0));
	VG_(emit_instr)(MFORM(RLWINM, dreg, dreg, 27, 31, 31, 0));
	VG_(emit_instr)(XFORM31(dreg, dreg, dreg, MO_NOR, 0));
	if (dis)
	    VG_(printf)("\trlwinm r%d,r%d,0,24,31\n\tcntlzw r%d,r%d\n"
			"\trlwinm r%d,r%d,27,31,31\n\tnot r%d,r%d\n",
			dreg, sreg, dreg, dreg, dreg, dreg, dreg, dreg);
	break;
    case Tag_PCast04:
	VG_(emit_instr)(DFORM(ANDIR, sreg, dreg, 1));
	VG_(emit_instr)(XFORM31(dreg, dreg, 0, MO_NEG, 0));
	if (dis)
	    VG_(printf)("\tandi. r%d,r%d,1\n\tneg r%d,r%d\n",
			dreg, sreg, dreg, dreg);
	break;
    case Tag_PCast02:
	VG_(emit_instr)(DFORM(ANDIR, sreg, dreg, 1));
	VG_(emit_instr)(XFORM31(dreg, dreg, 0, MO_NEG, 0));
	VG_(emit_instr)(DFORM(ORIS, dreg, dreg, 0xffff));
	if (dis)
	    VG_(printf)("\tandi. r%d,r%d,1\n\tneg r%d,r%d\n"
			"\toris r%d,r%d,0xffff\n",
			dreg, sreg, dreg, dreg, dreg, dreg);
	break;
    case Tag_PCast01:
	VG_(emit_instr)(DFORM(ANDIR, sreg, dreg, 1));
	VG_(emit_instr)(XFORM31(dreg, dreg, 0, MO_NEG, 0));
	VG_(emit_instr)(DFORM(ORIS, dreg, dreg, 0xffff));
	VG_(emit_instr)(DFORM(ORI, dreg, dreg, 0xff00));
	if (dis)
	    VG_(printf)("\tandi. r%d,r%d,1\n\tneg r%d,r%d\n"
			"\toris r%d,r%d,0xffff\n\tori r%d,r%d,0xff00\n",
			dreg, sreg, dreg, dreg, dreg, dreg, dreg, dreg);
	break;
    case Tag_PCast14:
	VG_(emit_instr)(MFORM(RLWINM, sreg, dreg, 0, 24, 31, 0));
	VG_(emit_instr)(XFORM31(dreg, dreg, 0, MO_CNTLZW, 0));
	VG_(emit_instr)(MFORM(RLWINM, dreg, dreg, 27, 31, 31, 0));
	VG_(emit_instr)(DFORM(ADDI, dreg, dreg, 0xffff));
	if (dis)
	    VG_(printf)("\trlwinm r%d,r%d,0,24,31\n\tcntlzw r%d,r%d\n"
			"\twlrinm r%d,r%d,27,31,31\n\taddi r%d,r%d,-1\n",
			dreg, sreg, dreg, dreg, dreg, dreg, dreg, dreg);
	break;
    case Tag_PCast12:
	VG_(emit_instr)(MFORM(RLWINM, sreg, dreg, 0, 24, 31, 0));
	VG_(emit_instr)(XFORM31(dreg, dreg, 0, MO_CNTLZW, 0));
	VG_(emit_instr)(MFORM(RLWINM, dreg, dreg, 27, 31, 31, 0));
	VG_(emit_instr)(DFORM(ADDI, dreg, dreg, 0xffff));
	VG_(emit_instr)(DFORM(ORIS, dreg, dreg, 0xffff));
	if (dis)
	    VG_(printf)("\trlwinm r%d,r%d,0,24,31\n\tcntlzw r%d,r%d\n"
			"\twlrinm r%d,r%d,27,31,31\n\taddi r%d,r%d,-1\n"
			"\toris r%d,r%d,0xffff\n",
			dreg, sreg, dreg, dreg, dreg, dreg, dreg, dreg,
			dreg, dreg);
	break;
    case Tag_PCast11:
	VG_(emit_instr)(MFORM(RLWINM, sreg, dreg, 0, 24, 31, 0));
	VG_(emit_instr)(XFORM31(dreg, dreg, 0, MO_CNTLZW, 0));
	VG_(emit_instr)(MFORM(RLWINM, dreg, dreg, 27, 31, 31, 0));
	VG_(emit_instr)(DFORM(ADDI, dreg, dreg, 0xffff));
	VG_(emit_instr)(DFORM(ORIS, dreg, dreg, 0xffff));
	VG_(emit_instr)(DFORM(ORI, dreg, dreg, 0xff00));
	if (dis)
	    VG_(printf)("\trlwinm r%d,r%d,0,24,31\n\tcntlzw r%d,r%d\n"
			"\twlrinm r%d,r%d,27,31,31\n\taddi r%d,r%d,-1\n"
			"\toris r%d,r%d,0xffff\n\tori r%d,r%d,0xff00\n",
			dreg, sreg, dreg, dreg, dreg, dreg, dreg, dreg,
			dreg, dreg, dreg, dreg);
	break;

    case Tag_PCast4CR:
	VG_(emit_instr)(XFORM31(sreg, dreg, 0, MO_CNTLZW, 0));
	VG_(emit_instr)(MFORM(RLWINM, dreg, dreg, 23, 3, 3, 0));
	VG_(emit_instr)(DFORM(ADDI, dreg, dreg, 0xffff));
	VG_(emit_instr)(DFORM(LWZ, 0, BB, VG_(shadow_reg_offset)(R_XER)));
	VG_(emit_instr)(MFORM(RLWIMI, 0, dreg, 29, 3, 3, 0));
	if (dis)
	    VG_(printf)("\tcntlzw r%d,r%d\n\trlwinm r%d,r%d,23,3,3\n"
			"\taddi r%d,r%d,-1\n\tlwz r0,%d(r%d)\n"
			"\trlwimi r%d,r0,29,3,3\n",
			dreg, sreg, dreg, dreg, dreg, dreg,
			VG_(shadow_reg_offset)(R_XER), BB, dreg);
	break;

    case Tag_PCast44:
	VG_(emit_instr)(XFORM31(sreg, dreg, 0, MO_CNTLZW, 0));
	VG_(emit_instr)(MFORM(RLWINM, dreg, dreg, 27, 31, 31, 0));
	VG_(emit_instr)(DFORM(ADDI, dreg, dreg, 0xffff));
	if (dis)
	    VG_(printf)("\tcntlzw r%d,r%d\n\trlwinm r%d,r%d,27,31,31\n"
			"\taddi r%d,r%d,-1\n",
			dreg, sreg, dreg, dreg, dreg, dreg);
	break;

    case Tag_Left4:
    case Tag_Left2:
    case Tag_Left1:
	VG_(emit_instr)(XFORM31(0, sreg, 0, MO_NEG, 0));
	VG_(emit_instr)(XFORM31(0, dreg, sreg, MO_OR, 0));
	if (dis)
	    VG_(printf)("\tneg r0,r%d\n\tor r%d,r0,r%d\n",
			sreg, dreg, sreg);
	break;

    case Tag_SWiden14:
	VG_(emit_instr)(XFORM31(sreg, dreg, 0, MO_EXTSB, 0));
	if (dis)
	    VG_(printf)("\textsb r%d,r%d\n", dreg, sreg);
	break;
    case Tag_SWiden24:
	VG_(emit_instr)(XFORM31(sreg, dreg, 0, MO_EXTSH, 0));
	if (dis)
	    VG_(printf)("\textsh r%d,r%d\n", dreg, sreg);
	break;
    case Tag_SWiden12:
	VG_(emit_instr)(XFORM31(sreg, dreg, 0, MO_EXTSB, 0));
	VG_(emit_instr)(DFORM(ORIS, dreg, dreg, 0xffff));
	if (dis)
	    VG_(printf)("\textsb r%d,r%d\n\toris r%d,r%d,0xffff\n",
			dreg, sreg, dreg, dreg);
	break;

    case Tag_ZWiden14:
	VG_(emit_instr)(MFORM(RLWINM, sreg, dreg, 0, 24, 31, 0));
	if (dis)
	    VG_(printf)("\trlwinm r%d,r%d,0,24,31\n", dreg, sreg);
	break;
    case Tag_ZWiden24:
	VG_(emit_instr)(MFORM(RLWINM, sreg, dreg, 0, 16, 31, 0));
	if (dis)
	    VG_(printf)("\trlwinm r%d,r%d,0,16,31\n", dreg, sreg);
	break;
    case Tag_ZWiden12:
	VG_(emit_instr)(MFORM(RLWINM, sreg, dreg, 0, 24, 31, 0));
	VG_(emit_instr)(DFORM(ORIS, dreg, dreg, 0xffff));
	if (dis)
	    VG_(printf)("\trlwinm r%d,r%d,0,24,31\n\toris r%d,r%d,0xffff\n",
			dreg, sreg, dreg, dreg);
	break;
    default:
	VG_(skin_panic)("tag1_op unimp");
    }
}

static void tag2_op(TagOp op, int sreg, int dreg)
{
    int treg;

    sreg = VG_(real_source_reg)(sreg);
    treg = VG_(real_source_reg)(dreg);
    dreg = VG_(real_dest_reg)(dreg);

    switch (op) {
    case Tag_UifU4:
    case Tag_UifU2:
    case Tag_UifU1:
    case Tag_UifU0:
	VG_(emit_instr)(XFORM31(sreg, dreg, treg, MO_OR, 0));
	if (dis)
	    VG_(printf)("\tor r%d,r%d,r%d\n", dreg, sreg, treg);
	break;

    case Tag_DifD4:
    case Tag_DifD2:
    case Tag_DifD1:
	VG_(emit_instr)(XFORM31(sreg, dreg, treg, MO_AND, 0));
	if (dis)
	    VG_(printf)("\tand r%d,r%d,r%d\n", dreg, sreg, treg);
	break;

    case Tag_ImproveAND4_TQ:
	VG_(emit_instr)(XFORM31(sreg, dreg, treg, MO_OR, 0));
	if (dis)
	    VG_(printf)("\tor r%d,r%d,r%d\n", dreg, sreg, treg);
	break;
    case Tag_ImproveAND2_TQ:
	VG_(emit_instr)(XFORM31(sreg, dreg, treg, MO_OR, 0));
	VG_(emit_instr)(DFORM(ORIS, dreg, dreg, 0xffff));
	if (dis)
	    VG_(printf)("\tor r%d,r%d,r%d\n\toris r%d,r%d,0xffff\n",
			dreg, sreg, treg, dreg, dreg);
	break;
    case Tag_ImproveAND1_TQ:
	VG_(emit_instr)(XFORM31(sreg, dreg, treg, MO_OR, 0));
	VG_(emit_instr)(DFORM(ORIS, dreg, dreg, 0xffff));
	VG_(emit_instr)(DFORM(ORI, dreg, dreg, 0xff00));
	if (dis)
	    VG_(printf)("\tor r%d,r%d,r%d\n\toris r%d,r%d,0xffff\n"
			"\tori r%d,r%d,0xff00\n",
			dreg, sreg, treg, dreg, dreg, dreg, dreg);
	break;

    case Tag_ImproveOR4_TQ:
	VG_(emit_instr)(XFORM31(treg, dreg, sreg, MO_ORC, 0));
	if (dis)
	    VG_(printf)("\torc r%d,r%d,r%d\n", dreg, treg, sreg);
	break;
    case Tag_ImproveOR2_TQ:
	VG_(emit_instr)(XFORM31(treg, dreg, sreg, MO_ORC, 0));
	VG_(emit_instr)(DFORM(ORIS, dreg, dreg, 0xffff));
	if (dis)
	    VG_(printf)("\torc r%d,r%d,r%d\n\toris r%d,r%d,0xffff\n",
			dreg, treg, sreg, dreg, dreg);
	break;
    case Tag_ImproveOR1_TQ:
	VG_(emit_instr)(XFORM31(treg, dreg, sreg, MO_ORC, 0));
	VG_(emit_instr)(DFORM(ORIS, dreg, dreg, 0xffff));
	VG_(emit_instr)(DFORM(ORI, dreg, dreg, 0xff00));
	if (dis)
	    VG_(printf)("\torc r%d,r%d,r%d\n\toris r%d,r%d,0xffff\n"
			"\tori r%d,r%d,0xff00\n",
			dreg, treg, sreg, dreg, dreg, dreg, dreg);
	break;

    case Tag_Max4_QT:
	VG_(emit_instr)(XFORM31(treg, dreg, sreg, MO_OR, 0));
	if (dis)
	    VG_(printf)("\tor r%d,r%d,r%d\n", dreg, treg, sreg);
	break;
    case Tag_Min4_QT:
	VG_(emit_instr)(XFORM31(treg, dreg, sreg, MO_ANDC, 0));
	if (dis)
	    VG_(printf)("\tandc r%d,r%d,r%d\n", dreg, treg, sreg);
	break;

    case Tag_Cmp0_TQ:
	/* Compute CR value bits from the value in sreg (t) and
	   its valid bits in treg (q). */
	/* andc r0,t,q; r0 contains t_min */
	VG_(emit_instr)(XFORM31(sreg, 0, treg, MO_ANDC, 0));
	/* cmpwi cr0,q,0 */
	VG_(emit_instr)(DFORM(CMPI, 0, treg, 0));
	/* cmpwi cr1,r0,0 */
	VG_(emit_instr)(DFORM(CMPI, 1*4, 0, 0));
	/* Now:
	   cr0.lt is set iff the MSB of q is set
	   cr0.eq is set iff q == 0
	   cr1.eq is set iff t_min == 0
	   LT is invalid iff the MSB of q is set
	   EQ is invalid iff q != 0 and tmin == 0
	   We take GT to be invalid if either LT or EQ is invalid.
	   This misses the case when q == 0x80000000 and t_min == 0.
	   In this case we know GT = 0 and is therefore valid.
	   This would take quite a few instructions to compute,
	   so we ignore it and hope it doesn't happen very often.
	   We won't miss errors this way but we may report an error
	   where there is none.
	*/
	VG_(emit_instr)(XFORM19(0*4+2, 1*4+2, 0*4+2, CRANDC));
	VG_(emit_instr)(XFORM19(0*4+1, 0*4+0, 0*4+2, CROR));
	/* Get the XER shadow so we can extract the SO valid bit */
	VG_(emit_instr)(DFORM(LWZ, 0, BB, VG_(shadow_reg_offset)(R_XER)));
	/* Get cr0, which now contains the valid bits for LT, GT, EQ */
	VG_(emit_instr)(XFORM31(dreg, 0, 0, MO_MFCR, 0));
	/* Insert the SO valid bit */
	VG_(emit_instr)(MFORM(RLWIMI, 0, dreg, 29, 3, 3, 0));
	/* paranoia: mark all other bits as invalid */
	VG_(emit_instr)(DFORM(ORIS, dreg, dreg, 0x0fff));
	VG_(emit_instr)(DFORM(ORI, dreg, dreg, 0xffff));
	if (dis) {
	    VG_(printf)("\tandc r0,%d,%d\n\tcmpwi %d,0\n\tcmpwi cr1,r0,0\n",
			sreg, treg, treg);
	    VG_(printf)("\tcrandc 4*cr0+eq,4*cr1+eq,4*cr0+eq\n");
	    VG_(printf)("\tcror 4*cr0+gt,4*cr0+lt,4*cr0+eq\n");
	    VG_(printf)("\tlwz r0,%d(r%d)\n\tmfcr r%d\n",
			VG_(shadow_reg_offset)(R_XER), BB, dreg);
	    VG_(printf)("\trlwimi r%d,r0,29,3,3\n", dreg);
	    VG_(printf)("\toris r%d,r%d,0xfff\n\tori r%d,r%d,0xffff\n",
			dreg, dreg, dreg, dreg);
	}
	break;

    case Tag_CLZ4_TQ:
	/* Compute valid bits for the result of a cntlzw */
	VG_(emit_instr)(XFORM31(treg, dreg, 0, MO_CNTLZW, 0));
	VG_(emit_instr)(XFORM31(dreg, sreg, dreg, MO_SUBF, 0));
	VG_(emit_instr)(DFORM(ADDI, dreg, dreg, 0xffff));
	VG_(emit_instr)(MFORM(RLWINM, dreg, dreg, 26, 26, 31, 0));
	if (dis) {
	    VG_(printf)("\tcntlzw r%d,r%d\n\tsubf r%d,r%d,r%d\n",
			dreg, treg, dreg, sreg, dreg);
	    VG_(printf)("\tsubi r%d,r%d,1\n\tsrwi r%d,r%d,26\n",
			dreg, dreg, dreg, dreg);
	}
	break;

    default:
	VG_(skin_panic)("tag2_op");
    }
}

static Addr load_helpers[4] = {
    (Addr) &MC_(helperc_LOADV1),
    (Addr) &MC_(helperc_LOADV2),
    (Addr) &MC_(helperc_LOADV3),
    (Addr) &MC_(helperc_LOADV4)
};

static void synth_loadv(int size, int reg1, int reg2)
{
    Addr helper;

    sk_assert(1 <= size && size <= 4);
    reg1 = VG_(real_source_reg)(reg1);
    reg2 = VG_(real_dest_reg)(reg2);
    helper = load_helpers[size-1];
    sk_assert(helper != 0);
    VG_(save_live_regs)();
    VG_(emit_instr)(XFORM31(reg1, 3, reg1, MO_OR, 0));
    if (dis)
	VG_(printf)("\tmr r3,r%d\n", reg1);
    VG_(emit_call)(VG_(helper_offset)(helper), False);
    VG_(emit_instr)(XFORM31(3, reg2, 3, MO_OR, 0));
    if (dis)
	VG_(printf)("\tmr r%d,r3\n", reg2);
}

static Addr store_helpers[4] = {
    (Addr) &MC_(helperc_STOREV1),
    (Addr) &MC_(helperc_STOREV2),
    (Addr) &MC_(helperc_STOREV3),
    (Addr) &MC_(helperc_STOREV4)
};

static void synth_storev(UInstr *u)
{
    Addr helper;
    int reg1, reg2;

    sk_assert(1 <= u->size && u->size <= 4);
    sk_assert(u->tag1 == RealReg || u->tag1 == Literal);
    sk_assert(u->tag2 == RealReg);
    reg1 = (u->tag1 == RealReg)? VG_(real_source_reg)(u->val1): -1;
    reg2 = VG_(real_source_reg)(u->val2);

    helper = store_helpers[u->size - 1];
    sk_assert(helper != 0);
    VG_(save_live_regs)();

    VG_(emit_instr)(XFORM31(reg2, 3, reg2, MO_OR, 0));
    if (dis)
	VG_(printf)("\tmr r3,r%d\n", reg2);
    if (u->tag1 == Literal)
	VG_(emit_loadlit)(4, u->lit32);
    else {
	VG_(emit_instr)(XFORM31(reg1, 4, reg1, MO_OR, 0));
	if (dis)
	    VG_(printf)("\tmr r4,r%d\n", reg1);
    }
    VG_(emit_call)(VG_(helper_offset)(helper), False);
}

unsigned int setv_sz[5] = {
    ~1U, ~0xffU, ~0xffffU, ~0xffffffU, 0
};

void SK_(emit_XUInstr)(UInstr *u, RRegSet regs_live_before)
{
    int reg, b;

    switch (u->opcode) {
    case GETV:
	sk_assert(u->tag1 == ArchReg);
	sk_assert(u->tag2 == RealReg);
	reg = VG_(real_dest_reg)(u->val2);
	VG_(emit_instr)(DFORM(LWZ, reg, BB, VG_(shadow_reg_offset)(u->val1)));
	if (dis)
	    VG_(printf)("\tlwz r%d,%d(r%d)\n",
			reg, VG_(shadow_reg_offset)(u->val1), BB);
	if (u->size <= 2) {
	    VG_(emit_instr)(DFORM(ORIS, reg, reg, 0xffff));
	    if (dis)
		VG_(printf)("\toris r%d,r%d,0xffff\n", reg, reg);
	}
	if (u->size == 1) {
	    VG_(emit_instr)(DFORM(ORI, reg, reg, 0xff00));
	    if (dis)
		VG_(printf)("\tori r%d,r%d,0xff00\n", reg, reg);
	}
	break;

    case PUTV:
	sk_assert(u->tag2 == ArchReg);
	sk_assert(u->size == 4);
	switch (u->tag1) {
	case RealReg:
	    reg = VG_(real_source_reg)(u->val1);
	    break;
	case Literal:
	    sk_assert(u->lit32 == 0);
	    VG_(emit_instr)(DFORM(ADDI, 0, 0, 0));
	    if (dis)
		VG_(printf)("\tli r0,0\n");
	    reg = 0;
	    break;
	default:
	    VG_(skin_panic)("PUTV unimp");
	}
	VG_(emit_instr)(DFORM(STW, reg, BB, VG_(shadow_reg_offset)(u->val2)));
	if (dis)
	    VG_(printf)("\tstw r%d,%d(r%d)\n",
			reg, VG_(shadow_reg_offset)(u->val2), BB);
	break;

    case TESTV:
	switch (u->tag1) {
	case ArchReg:
	    reg = 0;
	    VG_(emit_instr)(DFORM(LWZ, reg, BB,
				  VG_(shadow_reg_offset)(u->val1)));
	    if (dis)
		VG_(printf)("\tlwz r%d,%d(r%d)\n",
			    reg, VG_(shadow_reg_offset)(u->val1), BB);
	    break;
	case RealReg:
	    reg = VG_(real_source_reg)(u->val1);
	    break;
	default:
	    VG_(skin_panic)("TESTV unimp");
	}
	switch (u->size) {
	case 0:
	    VG_(emit_instr)(DFORM(ANDIR, reg, 0, 1));
	    if (dis)
		VG_(printf)("\tandi. r0,r%d,1\n", reg);
	    break;
	case 1:
	    VG_(emit_instr)(DFORM(ANDIR, reg, 0, 0xff));
	    if (dis)
		VG_(printf)("\tandi. r0,r%d,0xff\n", reg);
	    break;
	case 2:
	    VG_(emit_instr)(DFORM(ANDIR, reg, 0, 0xffff));
	    if (dis)
		VG_(printf)("\tandi. r0,r%d,0xffff\n", reg);
	    break;
	case 4:
	    VG_(emit_instr)(DFORM(CMPI, 0, reg, 0));
	    if (dis)
		VG_(printf)("\tcmpwi r%d,0\n", reg);
	    break;
	default:
	    VG_(skin_panic)("TESTV size unimp");
	}
	if (!helper_offsets_loaded)
	    load_helper_offsets();
	VG_(save_live_regs)();
	b = VG_(emit_instr)(BFORM(BC, 13, 2, 0, 0));
	if (dis)
	    VG_(printf)("\tbeq+ 1f\n");
	VG_(emit_call)(offset_value_check[u->size], True);
	if (dis)
	    VG_(printf)("1:\n");
	VG_(patch_branch)(b);
	break;

    case TAG1:
	tag1_op(u->val3, u->val1);
	break;

    case TAG2:
	tag2_op(u->val3, u->val1, u->val2);
	break;

    case LOADV:
	sk_assert(u->tag1 == RealReg);
	sk_assert(u->tag2 == RealReg);
	synth_loadv(u->size, u->val1, u->val2);
	break;

    case STOREV:
	sk_assert(u->tag1 == RealReg || u->tag1 == Literal);
	sk_assert(u->tag2 == RealReg);
	synth_storev(u);
	break;

    case SETV:
	sk_assert(u->tag1 == RealReg);
	reg = VG_(real_dest_reg)(u->val1);
	VG_(emit_loadlit)(reg, setv_sz[u->size]);
	break;

    case GETVF:
	/* for this we basically get the shadow_XER.CA bit */
	sk_assert(u->tag1 == RealReg);
	reg = VG_(real_dest_reg)(u->val1);
	VG_(emit_instr)(DFORM(LWZ, reg, BB, VG_(shadow_reg_offset)(R_XER)));
	VG_(emit_instr)(MFORM(RLWINM, reg, reg, 3, 31, 31, 0));
	/* may not be necessary... */
	VG_(emit_instr)(DFORM(ADDI, reg, reg, 0xfffe));
	if (dis)
	    VG_(printf)("\tlwz r%d,%d(r%d)\n\trlwinm r%d,r%d,3,31,31\n"
			"\taddi r%d,r%d,-2\n",
			reg, VG_(shadow_reg_offset)(R_XER), BB,
			reg, reg, reg, reg);
	break;

    case PUTVF:
	/* just update the shadow_XER.CA bit */
	sk_assert(u->tag1 == RealReg);
	reg = VG_(real_source_reg)(u->val1);
	VG_(emit_instr)(DFORM(LWZ, 0, BB, VG_(shadow_reg_offset)(R_XER)));
	VG_(emit_instr)(MFORM(RLWIMI, reg, 0, 29, 2, 2, 0));
	VG_(emit_instr)(DFORM(STW, 0, BB, VG_(shadow_reg_offset)(R_XER)));
	if (dis)
	    VG_(printf)("\tlwz r0,%d(r%d)\n\trlwimi r0,r%d,29,2,2\n"
			"\tstw r0,%d(r%d)\n",
			VG_(shadow_reg_offset)(R_XER), BB, reg,
			VG_(shadow_reg_offset)(R_XER), BB);
	break;

    default:
	VG_(printf)("emit_XUInstr: unhandled extension instruction:\n");
	VG_(pp_UInstr)(0, u);
	VG_(skin_panic)("emit_XUInstr");
    }
}
