/* Copyright 2007, 2008, 2009, 2010
   Free Software Foundation, Inc.

   This file is part of the GNU opcodes library.

   This library is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3, or (at your option)
   any later version.

   It is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
   License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street - Fifth Floor, Boston,
   MA 02110-1301, USA.  */

#include "sysdep.h"
#include <stdio.h>
#include <errno.h>
#include "getopt.h"
#include "libiberty.h"
#include "hashtab.h"
#include "safe-ctype.h"

#include "i386-opc.h"

#include <libintl.h>
#define _(String) gettext (String)

static const char *program_name = NULL;
static int debug = 0;

typedef struct initializer
{
  const char *name;
  const char *init;
} initializer;

static initializer cpu_flag_init[] =
{
  { "CPU_UNKNOWN_FLAGS",
    "~CpuL1OM" },
  { "CPU_GENERIC32_FLAGS",
    "Cpu186|Cpu286|Cpu386" },
  { "CPU_GENERIC64_FLAGS", 
    "Cpu186|Cpu286|Cpu386|Cpu486|Cpu586|Cpu686|CpuClflush|Cpu387|Cpu687|CpuMMX|CpuSSE|CpuSSE2|CpuLM" },
  { "CPU_NONE_FLAGS",
   "0" },
  { "CPU_I186_FLAGS",
    "Cpu186" },
  { "CPU_I286_FLAGS",
    "Cpu186|Cpu286" },
  { "CPU_I386_FLAGS",
    "Cpu186|Cpu286|Cpu386" },
  { "CPU_I486_FLAGS",
    "Cpu186|Cpu286|Cpu386|Cpu486" },
  { "CPU_I586_FLAGS",
    "Cpu186|Cpu286|Cpu386|Cpu486|Cpu586|Cpu387" },
  { "CPU_I686_FLAGS",
    "Cpu186|Cpu286|Cpu386|Cpu486|Cpu586|Cpu686|Cpu387|Cpu687" },
  { "CPU_P2_FLAGS",
    "Cpu186|Cpu286|Cpu386|Cpu486|Cpu586|Cpu686|Cpu387|Cpu687|CpuMMX" },
  { "CPU_P3_FLAGS",
    "Cpu186|Cpu286|Cpu386|Cpu486|Cpu586|Cpu686|Cpu387|Cpu687|CpuMMX|CpuSSE" },
  { "CPU_P4_FLAGS",
    "Cpu186|Cpu286|Cpu386|Cpu486|Cpu586|Cpu686|CpuClflush|Cpu387|Cpu687|CpuMMX|CpuSSE|CpuSSE2" },
  { "CPU_NOCONA_FLAGS",
    "Cpu186|Cpu286|Cpu386|Cpu486|Cpu586|Cpu686|CpuClflush|Cpu387|Cpu687|CpuFISTTP|CpuMMX|CpuSSE|CpuSSE2|CpuSSE3|CpuLM" },
  { "CPU_CORE_FLAGS",
    "Cpu186|Cpu286|Cpu386|Cpu486|Cpu586|Cpu686|CpuClflush|Cpu387|Cpu687|CpuFISTTP|CpuMMX|CpuSSE|CpuSSE2|CpuSSE3" },
  { "CPU_CORE2_FLAGS",
    "Cpu186|Cpu286|Cpu386|Cpu486|Cpu586|Cpu686|CpuClflush|Cpu387|Cpu687|CpuFISTTP|CpuMMX|CpuSSE|CpuSSE2|CpuSSE3|CpuSSSE3|CpuLM" },
  { "CPU_COREI7_FLAGS",
    "Cpu186|Cpu286|Cpu386|Cpu486|Cpu586|Cpu686|CpuClflush|Cpu387|Cpu687|CpuFISTTP|CpuMMX|CpuSSE|CpuSSE2|CpuSSE3|CpuSSSE3|CpuSSE4_1|CpuSSE4_2|CpuRdtscp|CpuLM" },
  { "CPU_K6_FLAGS",
    "Cpu186|Cpu286|Cpu386|Cpu486|Cpu586|CpuSYSCALL|Cpu387|CpuMMX" },
  { "CPU_K6_2_FLAGS",
    "Cpu186|Cpu286|Cpu386|Cpu486|Cpu586|CpuSYSCALL|Cpu387|CpuMMX|Cpu3dnow" },
  { "CPU_ATHLON_FLAGS",
    "Cpu186|Cpu286|Cpu386|Cpu486|Cpu586|Cpu686|CpuSYSCALL|Cpu387|Cpu687|CpuMMX|Cpu3dnow|Cpu3dnowA" },
  { "CPU_K8_FLAGS",
    "Cpu186|Cpu286|Cpu386|Cpu486|Cpu586|Cpu686|CpuSYSCALL|CpuRdtscp|Cpu387|Cpu687|CpuMMX|Cpu3dnow|Cpu3dnowA|CpuSSE|CpuSSE2|CpuLM" },
  { "CPU_AMDFAM10_FLAGS",
    "Cpu186|Cpu286|Cpu386|Cpu486|Cpu586|Cpu686|CpuSYSCALL|CpuRdtscp|Cpu387|Cpu687|CpuFISTTP|CpuMMX|Cpu3dnow|Cpu3dnowA|CpuSSE|CpuSSE2|CpuSSE3|CpuSSE4a|CpuABM|CpuLM" },
  { "CPU_BDVER1_FLAGS",
    "Cpu186|Cpu286|Cpu386|Cpu486|Cpu586|Cpu686|CpuSYSCALL|CpuRdtscp|Cpu387|Cpu687|CpuFISTTP|CpuMMX|Cpu3dnow|Cpu3dnowA|CpuSSE|CpuSSE2|CpuSSE3|CpuSSE4a|CpuABM|CpuLM|CpuFMA4|CpuXOP|CpuLWP" },
  { "CPU_8087_FLAGS",
    "Cpu8087" },
  { "CPU_287_FLAGS",
    "Cpu287" },
  { "CPU_387_FLAGS",
    "Cpu387" },
  { "CPU_ANY87_FLAGS",
    "Cpu8087|Cpu287|Cpu387|Cpu687|CpuFISTTP" },
  { "CPU_CLFLUSH_FLAGS",
    "CpuClflush" },
  { "CPU_SYSCALL_FLAGS",
    "CpuSYSCALL" },
  { "CPU_MMX_FLAGS",
    "CpuMMX" },
  { "CPU_SSE_FLAGS",
    "CpuMMX|CpuSSE" },
  { "CPU_SSE2_FLAGS",
    "CpuMMX|CpuSSE|CpuSSE2" },
  { "CPU_SSE3_FLAGS",
    "CpuMMX|CpuSSE|CpuSSE2|CpuSSE3" },
  { "CPU_SSSE3_FLAGS",
    "CpuMMX|CpuSSE|CpuSSE2|CpuSSE3|CpuSSSE3" },
  { "CPU_SSE4_1_FLAGS",
    "CpuMMX|CpuSSE|CpuSSE2|CpuSSE3|CpuSSSE3|CpuSSE4_1" },
  { "CPU_SSE4_2_FLAGS",
    "CpuMMX|CpuSSE|CpuSSE2|CpuSSE3|CpuSSSE3|CpuSSE4_1|CpuSSE4_2" },
  { "CPU_ANY_SSE_FLAGS",
    "CpuSSE|CpuSSE2|CpuSSE3|CpuSSSE3|CpuSSE4_1|CpuSSE4_2|CpuSSE4a|CpuAVX" },
  { "CPU_VMX_FLAGS",
    "CpuVMX" },
  { "CPU_SMX_FLAGS",
    "CpuSMX" },
  { "CPU_XSAVE_FLAGS",
    "CpuXsave" },
  { "CPU_XSAVEOPT_FLAGS",
    "CpuXsaveopt" },
  { "CPU_AES_FLAGS",
    "CpuMMX|CpuSSE|CpuSSE2|CpuSSE3|CpuSSSE3|CpuSSE4_1|CpuSSE4_2|CpuAES" },
  { "CPU_PCLMUL_FLAGS",
    "CpuMMX|CpuSSE|CpuSSE2|CpuSSE3|CpuSSSE3|CpuSSE4_1|CpuSSE4_2|CpuPCLMUL" },
  { "CPU_FMA_FLAGS",
    "CpuMMX|CpuSSE|CpuSSE2|CpuSSE3|CpuSSSE3|CpuSSE4_1|CpuSSE4_2|CpuAVX|CpuFMA" },
  { "CPU_FMA4_FLAGS",
    "CpuMMX|CpuSSE|CpuSSE2|CpuSSE3|CpuSSSE3|CpuSSE4_1|CpuSSE4_2|CpuAVX|CpuFMA4" },
  { "CPU_XOP_FLAGS",
    "CpuMMX|CpuSSE|CpuSSE2|CpuSSE3|CpuSSSE3|CpuSSE4_1|CpuSSE4_2|CpuSSE4a|CpuABM|CpuAVX|CpuFMA4|CpuXOP" },
  { "CPU_LWP_FLAGS",
    "CpuLWP" },
  { "CPU_MOVBE_FLAGS",
    "CpuMovbe" },
  { "CPU_RDTSCP_FLAGS",
    "CpuRdtscp" },
  { "CPU_EPT_FLAGS",
    "CpuEPT" },
  { "CPU_FSGSBASE_FLAGS",
    "CpuFSGSBase" },
  { "CPU_RDRND_FLAGS",
    "CpuRdRnd" },
  { "CPU_F16C_FLAGS",
    "CpuF16C" },
  { "CPU_3DNOW_FLAGS",
    "CpuMMX|Cpu3dnow" },
  { "CPU_3DNOWA_FLAGS",
    "CpuMMX|Cpu3dnow|Cpu3dnowA" },
  { "CPU_PADLOCK_FLAGS",
    "CpuPadLock" },
  { "CPU_SVME_FLAGS",
    "CpuSVME" },
  { "CPU_SSE4A_FLAGS",
    "CpuMMX|CpuSSE|CpuSSE2|CpuSSE3|CpuSSE4a" },
  { "CPU_ABM_FLAGS",
    "CpuABM" },
  { "CPU_AVX_FLAGS",
    "CpuMMX|CpuSSE|CpuSSE2|CpuSSE3|CpuSSSE3|CpuSSE4_1|CpuSSE4_2|CpuAVX" },
  { "CPU_ANY_AVX_FLAGS",
    "CpuAVX" },
  { "CPU_L1OM_FLAGS",
    "unknown" },
};

static initializer operand_type_init[] =
{
  { "OPERAND_TYPE_NONE",
    "0" },
  { "OPERAND_TYPE_REG8",
    "Reg8" },
  { "OPERAND_TYPE_REG16",
    "Reg16" },
  { "OPERAND_TYPE_REG32",
    "Reg32" },
  { "OPERAND_TYPE_REG64",
    "Reg64" },
  { "OPERAND_TYPE_IMM1",
    "Imm1" },
  { "OPERAND_TYPE_IMM8",
    "Imm8" },
  { "OPERAND_TYPE_IMM8S",
    "Imm8S" },
  { "OPERAND_TYPE_IMM16",
    "Imm16" },
  { "OPERAND_TYPE_IMM32",
    "Imm32" },
  { "OPERAND_TYPE_IMM32S",
    "Imm32S" },
  { "OPERAND_TYPE_IMM64",
    "Imm64" },
  { "OPERAND_TYPE_BASEINDEX",
    "BaseIndex" },
  { "OPERAND_TYPE_DISP8",
    "Disp8" },
  { "OPERAND_TYPE_DISP16",
    "Disp16" },
  { "OPERAND_TYPE_DISP32",
    "Disp32" },
  { "OPERAND_TYPE_DISP32S",
    "Disp32S" },
  { "OPERAND_TYPE_DISP64",
    "Disp64" },
  { "OPERAND_TYPE_INOUTPORTREG",
    "InOutPortReg" },
  { "OPERAND_TYPE_SHIFTCOUNT",
    "ShiftCount" },
  { "OPERAND_TYPE_CONTROL",
    "Control" },
  { "OPERAND_TYPE_TEST",
    "Test" },
  { "OPERAND_TYPE_DEBUG",
    "FloatReg" },
  { "OPERAND_TYPE_FLOATREG",
    "FloatReg" },
  { "OPERAND_TYPE_FLOATACC",
    "FloatAcc" },
  { "OPERAND_TYPE_SREG2",
    "SReg2" },
  { "OPERAND_TYPE_SREG3",
    "SReg3" },
  { "OPERAND_TYPE_ACC",
    "Acc" },
  { "OPERAND_TYPE_JUMPABSOLUTE",
    "JumpAbsolute" },
  { "OPERAND_TYPE_REGMMX",
    "RegMMX" },
  { "OPERAND_TYPE_REGXMM",
    "RegXMM" },
  { "OPERAND_TYPE_REGYMM",
    "RegYMM" },
  { "OPERAND_TYPE_ESSEG",
    "EsSeg" },
  { "OPERAND_TYPE_ACC32",
    "Reg32|Acc|Dword" },
  { "OPERAND_TYPE_ACC64",
    "Reg64|Acc|Qword" },
  { "OPERAND_TYPE_INOUTPORTREG",
    "InOutPortReg" },
  { "OPERAND_TYPE_REG16_INOUTPORTREG",
    "Reg16|InOutPortReg" },
  { "OPERAND_TYPE_DISP16_32",
    "Disp16|Disp32" },
  { "OPERAND_TYPE_ANYDISP",
    "Disp8|Disp16|Disp32|Disp32S|Disp64" },
  { "OPERAND_TYPE_IMM16_32",
    "Imm16|Imm32" },
  { "OPERAND_TYPE_IMM16_32S",
    "Imm16|Imm32S" },
  { "OPERAND_TYPE_IMM16_32_32S",
    "Imm16|Imm32|Imm32S" },
  { "OPERAND_TYPE_IMM32_32S_DISP32",
    "Imm32|Imm32S|Disp32" },
  { "OPERAND_TYPE_IMM64_DISP64",
    "Imm64|Disp64" },
  { "OPERAND_TYPE_IMM32_32S_64_DISP32",
    "Imm32|Imm32S|Imm64|Disp32" },
  { "OPERAND_TYPE_IMM32_32S_64_DISP32_64",
    "Imm32|Imm32S|Imm64|Disp32|Disp64" },
  { "OPERAND_TYPE_VEC_IMM4",
    "Vec_Imm4" },
};

typedef struct bitfield
{
  int position;
  int value;
  const char *name;
} bitfield;

#define BITFIELD(n) { n, 0, #n }

static bitfield cpu_flags[] =
{
  BITFIELD (Cpu186),
  BITFIELD (Cpu286),
  BITFIELD (Cpu386),
  BITFIELD (Cpu486),
  BITFIELD (Cpu586),
  BITFIELD (Cpu686),
  BITFIELD (CpuClflush),
  BITFIELD (CpuSYSCALL),
  BITFIELD (Cpu8087),
  BITFIELD (Cpu287),
  BITFIELD (Cpu387),
  BITFIELD (Cpu687),
  BITFIELD (CpuFISTTP),
  BITFIELD (CpuMMX),
  BITFIELD (CpuSSE),
  BITFIELD (CpuSSE2),
  BITFIELD (CpuSSE3),
  BITFIELD (CpuSSSE3),
  BITFIELD (CpuSSE4_1),
  BITFIELD (CpuSSE4_2),
  BITFIELD (CpuAVX),
  BITFIELD (CpuL1OM),
  BITFIELD (CpuSSE4a),
  BITFIELD (Cpu3dnow),
  BITFIELD (Cpu3dnowA),
  BITFIELD (CpuPadLock),
  BITFIELD (CpuSVME),
  BITFIELD (CpuVMX),
  BITFIELD (CpuSMX),
  BITFIELD (CpuABM),
  BITFIELD (CpuXsave),
  BITFIELD (CpuXsaveopt),
  BITFIELD (CpuAES),
  BITFIELD (CpuPCLMUL),
  BITFIELD (CpuFMA),
  BITFIELD (CpuFMA4),
  BITFIELD (CpuXOP),
  BITFIELD (CpuLWP),
  BITFIELD (CpuLM),
  BITFIELD (CpuMovbe),
  BITFIELD (CpuEPT),
  BITFIELD (CpuRdtscp),
  BITFIELD (CpuFSGSBase),
  BITFIELD (CpuRdRnd),
  BITFIELD (CpuF16C),
  BITFIELD (Cpu64),
  BITFIELD (CpuNo64),
#ifdef CpuUnused
  BITFIELD (CpuUnused),
#endif
};

static bitfield opcode_modifiers[] =
{
  BITFIELD (D),
  BITFIELD (W),
  BITFIELD (S),
  BITFIELD (Modrm),
  BITFIELD (ShortForm),
  BITFIELD (Jump),
  BITFIELD (JumpDword),
  BITFIELD (JumpByte),
  BITFIELD (JumpInterSegment),
  BITFIELD (FloatMF),
  BITFIELD (FloatR),
  BITFIELD (FloatD),
  BITFIELD (Size16),
  BITFIELD (Size32),
  BITFIELD (Size64),
  BITFIELD (IgnoreSize),
  BITFIELD (DefaultSize),
  BITFIELD (No_bSuf),
  BITFIELD (No_wSuf),
  BITFIELD (No_lSuf),
  BITFIELD (No_sSuf),
  BITFIELD (No_qSuf),
  BITFIELD (No_ldSuf),
  BITFIELD (FWait),
  BITFIELD (IsString),
  BITFIELD (IsLockable),
  BITFIELD (RegKludge),
  BITFIELD (FirstXmm0),
  BITFIELD (Implicit1stXmm0),
  BITFIELD (ToDword),
  BITFIELD (ToQword),
  BITFIELD (AddrPrefixOp0),
  BITFIELD (IsPrefix),
  BITFIELD (ImmExt),
  BITFIELD (NoRex64),
  BITFIELD (Rex64),
  BITFIELD (Ugh),
  BITFIELD (Vex),
  BITFIELD (VexVVVV),
  BITFIELD (VexW),
  BITFIELD (VexOpcode),
  BITFIELD (VexSources),
  BITFIELD (VexImmExt),
  BITFIELD (SSE2AVX),
  BITFIELD (NoAVX),
  BITFIELD (OldGcc),
  BITFIELD (ATTMnemonic),
  BITFIELD (ATTSyntax),
  BITFIELD (IntelSyntax),
};

static bitfield operand_types[] =
{
  BITFIELD (Reg8),
  BITFIELD (Reg16),
  BITFIELD (Reg32),
  BITFIELD (Reg64),
  BITFIELD (FloatReg),
  BITFIELD (RegMMX),
  BITFIELD (RegXMM),
  BITFIELD (RegYMM),
  BITFIELD (Imm1),
  BITFIELD (Imm8),
  BITFIELD (Imm8S),
  BITFIELD (Imm16),
  BITFIELD (Imm32),
  BITFIELD (Imm32S),
  BITFIELD (Imm64),
  BITFIELD (BaseIndex),
  BITFIELD (Disp8),
  BITFIELD (Disp16),
  BITFIELD (Disp32),
  BITFIELD (Disp32S),
  BITFIELD (Disp64),
  BITFIELD (InOutPortReg),
  BITFIELD (ShiftCount),
  BITFIELD (Control),
  BITFIELD (Debug),
  BITFIELD (Test),
  BITFIELD (SReg2),
  BITFIELD (SReg3),
  BITFIELD (Acc),
  BITFIELD (FloatAcc),
  BITFIELD (JumpAbsolute),
  BITFIELD (EsSeg),
  BITFIELD (RegMem),
  BITFIELD (Mem),
  BITFIELD (Byte),
  BITFIELD (Word),
  BITFIELD (Dword),
  BITFIELD (Fword),
  BITFIELD (Qword),
  BITFIELD (Tbyte),
  BITFIELD (Xmmword),
  BITFIELD (Ymmword),
  BITFIELD (Unspecified),
  BITFIELD (Anysize),
  BITFIELD (Vec_Imm4),
#ifdef OTUnused
  BITFIELD (OTUnused),
#endif
};

static const char *filename;

static int
compare (const void *x, const void *y)
{
  const bitfield *xp = (const bitfield *) x;
  const bitfield *yp = (const bitfield *) y;
  return xp->position - yp->position;
}

static void
fail (const char *message, ...)
{
  va_list args;
  
  va_start (args, message);
  fprintf (stderr, _("%s: Error: "), program_name);
  vfprintf (stderr, message, args);
  va_end (args);
  xexit (1);
}

static void
process_copyright (FILE *fp)
{
  fprintf (fp, "/* This file is automatically generated by i386-gen.  Do not edit!  */\n\
/* Copyright 2007, 2008, 2009, 2010\n\
   Free Software Foundation, Inc.\n\
\n\
   This file is part of the GNU opcodes library.\n\
\n\
   This library is free software; you can redistribute it and/or modify\n\
   it under the terms of the GNU General Public License as published by\n\
   the Free Software Foundation; either version 3, or (at your option)\n\
   any later version.\n\
\n\
   It is distributed in the hope that it will be useful, but WITHOUT\n\
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY\n\
   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public\n\
   License for more details.\n\
\n\
   You should have received a copy of the GNU General Public License\n\
   along with this program; if not, write to the Free Software\n\
   Foundation, Inc., 51 Franklin Street - Fifth Floor, Boston,\n\
   MA 02110-1301, USA.  */\n");
}

/* Remove leading white spaces.  */

static char *
remove_leading_whitespaces (char *str)
{
  while (ISSPACE (*str))
    str++;
  return str;
}

/* Remove trailing white spaces.  */

static void
remove_trailing_whitespaces (char *str)
{
  size_t last = strlen (str);

  if (last == 0)
    return;

  do
    {
      last--;
      if (ISSPACE (str [last]))
	str[last] = '\0';
      else
	break;
    }
  while (last != 0);
}

/* Find next field separated by SEP and terminate it. Return a
   pointer to the one after it.  */

static char *
next_field (char *str, char sep, char **next, char *last)
{
  char *p;

  p = remove_leading_whitespaces (str);
  for (str = p; *str != sep && *str != '\0'; str++);

  *str = '\0';
  remove_trailing_whitespaces (p);

  *next = str + 1; 

  if (p >= last)
    abort ();

  return p;
}

static void
set_bitfield (const char *f, bitfield *array, int value,
	      unsigned int size, int lineno)
{
  unsigned int i;

  if (strcmp (f, "CpuFP") == 0)
    {
      set_bitfield("Cpu387", array, value, size, lineno);
      set_bitfield("Cpu287", array, value, size, lineno);
      f = "Cpu8087";
    }
  else if (strcmp (f, "Mmword") == 0)
    f= "Qword";
  else if (strcmp (f, "Oword") == 0)
    f= "Xmmword";

  for (i = 0; i < size; i++)
    if (strcasecmp (array[i].name, f) == 0)
      {
	array[i].value = value;
	return;
      }

  if (value)
    {
      const char *v = strchr (f, '=');

      if (v)
	{
	  size_t n = v - f;
	  char *end;

	  for (i = 0; i < size; i++)
	    if (strncasecmp (array[i].name, f, n) == 0)
	      {
		value = strtol (v + 1, &end, 0);
		if (*end == '\0')
		  {
		    array[i].value = value;
		    return;
		  }
		break;
	      }
	}
    }

  if (lineno != -1)
    fail (_("%s: %d: Unknown bitfield: %s\n"), filename, lineno, f);
  else
    fail (_("Unknown bitfield: %s\n"), f);
}

static void
output_cpu_flags (FILE *table, bitfield *flags, unsigned int size,
		  int macro, const char *comma, const char *indent)
{
  unsigned int i;

  fprintf (table, "%s{ { ", indent);

  for (i = 0; i < size - 1; i++)
    {
      fprintf (table, "%d, ", flags[i].value);
      if (((i + 1) % 20) == 0)
	{
	  /* We need \\ for macro.  */
	  if (macro)
	    fprintf (table, " \\\n    %s", indent);
	  else
	    fprintf (table, "\n    %s", indent);
	}
    }

  fprintf (table, "%d } }%s\n", flags[i].value, comma);
}

static void
process_i386_cpu_flag (FILE *table, char *flag, int macro,
		       const char *comma, const char *indent,
		       int lineno)
{
  char *str, *next, *last;
  unsigned int i;
  bitfield flags [ARRAY_SIZE (cpu_flags)];

  /* Copy the default cpu flags.  */
  memcpy (flags, cpu_flags, sizeof (cpu_flags));

  if (strcasecmp (flag, "unknown") == 0)
    {
      /* We turn on everything except for cpu64 in case of
	 CPU_UNKNOWN_FLAGS.  */
      for (i = 0; i < ARRAY_SIZE (flags); i++)
	if (flags[i].position != Cpu64)
	  flags[i].value = 1;
    }
  else if (flag[0] == '~')
    {
      last = flag + strlen (flag);

      if (flag[1] == '(')
	{
	  last -= 1;
	  next = flag + 2;
	  if (*last != ')')
	    fail (_("%s: %d: Missing `)' in bitfield: %s\n"), filename,
		  lineno, flag);
	  *last = '\0';
	}
      else
	next = flag + 1;

      /* First we turn on everything except for cpu64.  */
      for (i = 0; i < ARRAY_SIZE (flags); i++)
	if (flags[i].position != Cpu64)
	  flags[i].value = 1;

      /* Turn off selective bits.  */
      for (; next && next < last; )
	{
	  str = next_field (next, '|', &next, last);
	  if (str)
	    set_bitfield (str, flags, 0, ARRAY_SIZE (flags), lineno);
	}
    }
  else if (strcmp (flag, "0"))
    {
      /* Turn on selective bits.  */
      last = flag + strlen (flag);
      for (next = flag; next && next < last; )
	{
	  str = next_field (next, '|', &next, last);
	  if (str)
	    set_bitfield (str, flags, 1, ARRAY_SIZE (flags), lineno);
	}
    }

  output_cpu_flags (table, flags, ARRAY_SIZE (flags), macro,
		    comma, indent);
}

static void
output_opcode_modifier (FILE *table, bitfield *modifier, unsigned int size)
{
  unsigned int i;

  fprintf (table, "    { ");

  for (i = 0; i < size - 1; i++)
    {
      fprintf (table, "%d, ", modifier[i].value);
      if (((i + 1) % 20) == 0)
	fprintf (table, "\n      ");
    }

  fprintf (table, "%d },\n", modifier[i].value);
}

static void
process_i386_opcode_modifier (FILE *table, char *mod, int lineno)
{
  char *str, *next, *last;
  bitfield modifiers [ARRAY_SIZE (opcode_modifiers)];

  /* Copy the default opcode modifier.  */
  memcpy (modifiers, opcode_modifiers, sizeof (modifiers));

  if (strcmp (mod, "0"))
    {
      last = mod + strlen (mod);
      for (next = mod; next && next < last; )
	{
	  str = next_field (next, '|', &next, last);
	  if (str)
	    set_bitfield (str, modifiers, 1, ARRAY_SIZE (modifiers),
			  lineno);
	}
    }
  output_opcode_modifier (table, modifiers, ARRAY_SIZE (modifiers));
}

static void
output_operand_type (FILE *table, bitfield *types, unsigned int size,
		     int macro, const char *indent)
{
  unsigned int i;

  fprintf (table, "{ { ");

  for (i = 0; i < size - 1; i++)
    {
      fprintf (table, "%d, ", types[i].value);
      if (((i + 1) % 20) == 0)
	{
	  /* We need \\ for macro.  */
	  if (macro)
	    fprintf (table, "\\\n%s", indent);
	  else
	    fprintf (table, "\n%s", indent);
	}
    }

  fprintf (table, "%d } }", types[i].value);
}

static void
process_i386_operand_type (FILE *table, char *op, int macro,
			   const char *indent, int lineno)
{
  char *str, *next, *last;
  bitfield types [ARRAY_SIZE (operand_types)];

  /* Copy the default operand type.  */
  memcpy (types, operand_types, sizeof (types));

  if (strcmp (op, "0"))
    {
      last = op + strlen (op);
      for (next = op; next && next < last; )
	{
	  str = next_field (next, '|', &next, last);
	  if (str)
	    set_bitfield (str, types, 1, ARRAY_SIZE (types), lineno);
	}
    }
  output_operand_type (table, types, ARRAY_SIZE (types), macro,
		       indent);
}

static void
output_i386_opcode (FILE *table, const char *name, char *str,
		    char *last, int lineno)
{
  unsigned int i;
  char *operands, *base_opcode, *extension_opcode, *opcode_length;
  char *cpu_flags, *opcode_modifier, *operand_types [MAX_OPERANDS];

  /* Find number of operands.  */
  operands = next_field (str, ',', &str, last);

  /* Find base_opcode.  */
  base_opcode = next_field (str, ',', &str, last);

  /* Find extension_opcode.  */
  extension_opcode = next_field (str, ',', &str, last);

  /* Find opcode_length.  */
  opcode_length = next_field (str, ',', &str, last);

  /* Find cpu_flags.  */
  cpu_flags = next_field (str, ',', &str, last);

  /* Find opcode_modifier.  */
  opcode_modifier = next_field (str, ',', &str, last);

  /* Remove the first {.  */
  str = remove_leading_whitespaces (str);
  if (*str != '{')
    abort ();
  str = remove_leading_whitespaces (str + 1);

  i = strlen (str);

  /* There are at least "X}".  */
  if (i < 2)
    abort ();

  /* Remove trailing white spaces and }. */
  do
    {
      i--;
      if (ISSPACE (str[i]) || str[i] == '}')
	str[i] = '\0';
      else
	break;
    }
  while (i != 0);

  last = str + i;

  /* Find operand_types.  */
  for (i = 0; i < ARRAY_SIZE (operand_types); i++)
    {
      if (str >= last)
	{
	  operand_types [i] = NULL;
	  break;
	}

      operand_types [i] = next_field (str, ',', &str, last);
      if (*operand_types[i] == '0')
	{
	  if (i != 0)
	    operand_types[i] = NULL;
	  break;
	}
    }

  fprintf (table, "  { \"%s\", %s, %s, %s, %s,\n",
	   name, operands, base_opcode, extension_opcode,
	   opcode_length);

  process_i386_cpu_flag (table, cpu_flags, 0, ",", "    ", lineno);

  process_i386_opcode_modifier (table, opcode_modifier, lineno);

  fprintf (table, "    { ");

  for (i = 0; i < ARRAY_SIZE (operand_types); i++)
    {
      if (operand_types[i] == NULL || *operand_types[i] == '0')
	{
	  if (i == 0)
	    process_i386_operand_type (table, "0", 0, "\t  ", lineno);
	  break;
	}

      if (i != 0)
	fprintf (table, ",\n      ");

      process_i386_operand_type (table, operand_types[i], 0,
				 "\t  ", lineno);
    }
  fprintf (table, " } },\n");
}

struct opcode_hash_entry
{
  struct opcode_hash_entry *next;
  char *name;
  char *opcode;
  int lineno;
};

/* Calculate the hash value of an opcode hash entry P.  */

static hashval_t
opcode_hash_hash (const void *p)
{
  struct opcode_hash_entry *entry = (struct opcode_hash_entry *) p;
  return htab_hash_string (entry->name);
}

/* Compare a string Q against an opcode hash entry P.  */

static int
opcode_hash_eq (const void *p, const void *q)
{
  struct opcode_hash_entry *entry = (struct opcode_hash_entry *) p;
  const char *name = (const char *) q;
  return strcmp (name, entry->name) == 0;
}

static void
process_i386_opcodes (FILE *table)
{
  FILE *fp;
  char buf[2048];
  unsigned int i, j;
  char *str, *p, *last, *name;
  struct opcode_hash_entry **hash_slot, **entry, *next;
  htab_t opcode_hash_table;
  struct opcode_hash_entry **opcode_array;
  unsigned int opcode_array_size = 1024;
  int lineno = 0;

  filename = "i386-opc.tbl";
  fp = fopen (filename, "r");

  if (fp == NULL)
    fail (_("can't find i386-opc.tbl for reading, errno = %s\n"),
	  xstrerror (errno));

  i = 0;
  opcode_array = (struct opcode_hash_entry **)
    xmalloc (sizeof (*opcode_array) * opcode_array_size);

  opcode_hash_table = htab_create_alloc (16, opcode_hash_hash,
					 opcode_hash_eq, NULL,
					 xcalloc, free);

  fprintf (table, "\n/* i386 opcode table.  */\n\n");
  fprintf (table, "const insn_template i386_optab[] =\n{\n");

  /* Put everything on opcode array.  */
  while (!feof (fp))
    {
      if (fgets (buf, sizeof (buf), fp) == NULL)
	break;

      lineno++;

      p = remove_leading_whitespaces (buf);

      /* Skip comments.  */
      str = strstr (p, "//");
      if (str != NULL)
	str[0] = '\0';

      /* Remove trailing white spaces.  */
      remove_trailing_whitespaces (p);

      switch (p[0])
	{
	case '#':
	  /* Ignore comments.  */
	case '\0':
	  continue;
	  break;
	default:
	  break;
	}

      last = p + strlen (p);

      /* Find name.  */
      name = next_field (p, ',', &str, last);

      /* Get the slot in hash table.  */
      hash_slot = (struct opcode_hash_entry **)
	htab_find_slot_with_hash (opcode_hash_table, name,
				  htab_hash_string (name),
				  INSERT);

      if (*hash_slot == NULL)
	{
	  /* It is the new one.  Put it on opcode array.  */
	  if (i >= opcode_array_size)
	    {
	      /* Grow the opcode array when needed.  */
	      opcode_array_size += 1024;
	      opcode_array = (struct opcode_hash_entry **)
		xrealloc (opcode_array,
			  sizeof (*opcode_array) * opcode_array_size);
	    }

	  opcode_array[i] = (struct opcode_hash_entry *)
	    xmalloc (sizeof (struct opcode_hash_entry));
	  opcode_array[i]->next = NULL;
	  opcode_array[i]->name = xstrdup (name);
	  opcode_array[i]->opcode = xstrdup (str);
	  opcode_array[i]->lineno = lineno;
	  *hash_slot = opcode_array[i];
	  i++;
	}
      else
	{
	  /* Append it to the existing one.  */
	  entry = hash_slot;
	  while ((*entry) != NULL)
	    entry = &(*entry)->next;
	  *entry = (struct opcode_hash_entry *)
	    xmalloc (sizeof (struct opcode_hash_entry));
	  (*entry)->next = NULL;
	  (*entry)->name = (*hash_slot)->name;
	  (*entry)->opcode = xstrdup (str);
	  (*entry)->lineno = lineno;
	}
    }

  /* Process opcode array.  */
  for (j = 0; j < i; j++)
    {
      for (next = opcode_array[j]; next; next = next->next)
	{
	  name = next->name;
	  str = next->opcode;
	  lineno = next->lineno;
	  last = str + strlen (str);
	  output_i386_opcode (table, name, str, last, lineno);
	}
    }

  fclose (fp);

  fprintf (table, "  { NULL, 0, 0, 0, 0,\n");

  process_i386_cpu_flag (table, "0", 0, ",", "    ", -1);

  process_i386_opcode_modifier (table, "0", -1);
 
  fprintf (table, "    { ");
  process_i386_operand_type (table, "0", 0, "\t  ", -1);
  fprintf (table, " } }\n");

  fprintf (table, "};\n");
}

static void
process_i386_registers (FILE *table)
{
  FILE *fp;
  char buf[2048];
  char *str, *p, *last;
  char *reg_name, *reg_type, *reg_flags, *reg_num;
  char *dw2_32_num, *dw2_64_num;
  int lineno = 0;

  filename = "i386-reg.tbl";
  fp = fopen (filename, "r");
  if (fp == NULL)
    fail (_("can't find i386-reg.tbl for reading, errno = %s\n"),
	  xstrerror (errno));

  fprintf (table, "\n/* i386 register table.  */\n\n");
  fprintf (table, "const reg_entry i386_regtab[] =\n{\n");

  while (!feof (fp))
    {
      if (fgets (buf, sizeof (buf), fp) == NULL)
	break;

      lineno++;

      p = remove_leading_whitespaces (buf);

      /* Skip comments.  */
      str = strstr (p, "//");
      if (str != NULL)
	str[0] = '\0';

      /* Remove trailing white spaces.  */
      remove_trailing_whitespaces (p);

      switch (p[0])
	{
	case '#':
	  fprintf (table, "%s\n", p);
	case '\0':
	  continue;
	  break;
	default:
	  break;
	}

      last = p + strlen (p);

      /* Find reg_name.  */
      reg_name = next_field (p, ',', &str, last);

      /* Find reg_type.  */
      reg_type = next_field (str, ',', &str, last);

      /* Find reg_flags.  */
      reg_flags = next_field (str, ',', &str, last);

      /* Find reg_num.  */
      reg_num = next_field (str, ',', &str, last);

      fprintf (table, "  { \"%s\",\n    ", reg_name);

      process_i386_operand_type (table, reg_type, 0, "\t", lineno);

      /* Find 32-bit Dwarf2 register number.  */
      dw2_32_num = next_field (str, ',', &str, last);

      /* Find 64-bit Dwarf2 register number.  */
      dw2_64_num = next_field (str, ',', &str, last);

      fprintf (table, ",\n    %s, %s, { %s, %s } },\n",
	       reg_flags, reg_num, dw2_32_num, dw2_64_num);
    }

  fclose (fp);

  fprintf (table, "};\n");

  fprintf (table, "\nconst unsigned int i386_regtab_size = ARRAY_SIZE (i386_regtab);\n");
}

static void
process_i386_initializers (void)
{
  unsigned int i;
  FILE *fp = fopen ("i386-init.h", "w");
  char *init;

  if (fp == NULL)
    fail (_("can't create i386-init.h, errno = %s\n"),
	  xstrerror (errno));

  process_copyright (fp);

  for (i = 0; i < ARRAY_SIZE (cpu_flag_init); i++)
    {
      fprintf (fp, "\n#define %s \\\n", cpu_flag_init[i].name);
      init = xstrdup (cpu_flag_init[i].init);
      process_i386_cpu_flag (fp, init, 1, "", "  ", -1);
      free (init);
    }

  for (i = 0; i < ARRAY_SIZE (operand_type_init); i++)
    {
      fprintf (fp, "\n\n#define %s \\\n  ", operand_type_init[i].name);
      init = xstrdup (operand_type_init[i].init);
      process_i386_operand_type (fp, init, 1, "      ", -1);
      free (init);
    }
  fprintf (fp, "\n");

  fclose (fp);
}

/* Program options.  */
#define OPTION_SRCDIR	200

struct option long_options[] = 
{
  {"srcdir",  required_argument, NULL, OPTION_SRCDIR},
  {"debug",   no_argument,       NULL, 'd'},
  {"version", no_argument,       NULL, 'V'},
  {"help",    no_argument,       NULL, 'h'},
  {0,         no_argument,       NULL, 0}
};

static void
print_version (void)
{
  printf ("%s: version 1.0\n", program_name);
  xexit (0);
}

static void
usage (FILE * stream, int status)
{
  fprintf (stream, "Usage: %s [-V | --version] [-d | --debug] [--srcdir=dirname] [--help]\n",
	   program_name);
  xexit (status);
}

int
main (int argc, char **argv)
{
  extern int chdir (char *);
  char *srcdir = NULL;
  int c;
  FILE *table;
  
  program_name = *argv;
  xmalloc_set_program_name (program_name);

  while ((c = getopt_long (argc, argv, "vVdh", long_options, 0)) != EOF)
    switch (c)
      {
      case OPTION_SRCDIR:
	srcdir = optarg;
	break;
      case 'V':
      case 'v':
	print_version ();
	break;
      case 'd':
	debug = 1;
	break;
      case 'h':
      case '?':
	usage (stderr, 0);
      default:
      case 0:
	break;
      }

  if (optind != argc)
    usage (stdout, 1);

  if (srcdir != NULL) 
    if (chdir (srcdir) != 0)
      fail (_("unable to change directory to \"%s\", errno = %s\n"),
	    srcdir, xstrerror (errno));

  /* Check the unused bitfield in i386_cpu_flags.  */
#ifndef CpuUnused
  c = CpuNumOfBits - CpuMax - 1;
  if (c)
    fail (_("%d unused bits in i386_cpu_flags.\n"), c);
#endif

  /* Check the unused bitfield in i386_operand_type.  */
#ifndef OTUnused
  c = OTNumOfBits - OTMax - 1;
  if (c)
    fail (_("%d unused bits in i386_operand_type.\n"), c);
#endif

  qsort (cpu_flags, ARRAY_SIZE (cpu_flags), sizeof (cpu_flags [0]),
	 compare);

  qsort (opcode_modifiers, ARRAY_SIZE (opcode_modifiers),
	 sizeof (opcode_modifiers [0]), compare);

  qsort (operand_types, ARRAY_SIZE (operand_types),
	 sizeof (operand_types [0]), compare);

  table = fopen ("i386-tbl.h", "w");
  if (table == NULL)
    fail (_("can't create i386-tbl.h, errno = %s\n"),
	  xstrerror (errno));

  process_copyright (table);

  process_i386_opcodes (table);
  process_i386_registers (table);
  process_i386_initializers ();

  fclose (table);

  exit (0);
}
