#include <unistd.h>
#include <sys/stat.h>
#include <errno.h>
#undef ENOSYS
#include <cyg/infra/diag.h>
#include <cyg/io/io.h>
#include "libio.h"

/* Maximum number of IO handler.  */
#define NBR_IO_HANDLER	4
/* Maximum number of file descriptor.  */
#define NBR_FD	16

#define FLAGS_USED	(1 << 0)

struct fd_xlat_t
{
  unsigned int flags;
  void *data;
  struct libio_handler_t *handler;
};

static struct fd_xlat_t fd_xlat[NBR_FD];

/* Ecos io handler.  */
static int
ecos_read (int fd, void *buffer, cyg_uint32 count)
{
  cyg_uint32 len;
  Cyg_ErrNo err;

  len = count;
  err = cyg_io_read (*(cyg_io_handle_t *)fd_xlat[fd].data, buffer, &len);
  if (err < 0)
    {
      diag_printf ("read (%d): %s\n", fd, strerror (err));
      return -1;
    }
  return len;
}

static int
ecos_write (int fd, const void *buffer, cyg_uint32 count)
{
  cyg_uint32 len;
  Cyg_ErrNo err;

  len = count;
  err = cyg_io_write (*(cyg_io_handle_t *)fd_xlat[fd].data, buffer, &len);
  if (err < 0)
    {
      diag_printf ("write (%d): %s\n", fd, strerror (err));
      return -1;
    }
  return count;
}

static int
ecos_ioctl (int fd, cyg_uint32 command, void *buffer)
{
  diag_write_string ("function ecos_ioctl called\a\n");
  return -1;
}

static int
ecos_close (int fd)
{
  diag_write_string ("function ecos_close called\a\n");
  return -1;
}

static struct libio_handler_t ecos_handler = 
{
  ecos_read, ecos_write, ecos_ioctl, ecos_close
};

/* General functions.  */
/* Get a new fd for a registred handler.  */
int
libio_get_fd (struct libio_handler_t *handler, void *data)
{
  int i;
  for (i = 0; i < NBR_FD; i++)
    if (!(fd_xlat[i].flags & FLAGS_USED))
      {
	fd_xlat[i].flags = FLAGS_USED;
	fd_xlat[i].data = data;
	fd_xlat[i].handler = handler;
	return i;
      }
  errno = EMFILE;
  return -1;
}

/* Get the data from an fd.  */
/* Return NULL if fd is bad.  */
void *
libio_get_data (int fd)
{
  if (fd < 0 || fd > NBR_FD || !(fd_xlat[fd].flags & FLAGS_USED))
    return NULL;
  return fd_xlat[fd].data;
}

static cyg_io_handle_t console_handle;
void
libio_initialize (void)
{
  Cyg_ErrNo err;
  int i;

  /* fd 0, 1, 2 are ttys.  */
  err = cyg_io_lookup ("/dev/tty0", &console_handle);
  if (err < 0)
    {
      diag_printf ("opening console: %d\n", err);
      return;
    }
  for (i = 0; i < 3; i++)
    {
      if (libio_get_fd (&ecos_handler, &console_handle) != i)
	diag_printf ("libio_initialize: bad for fd %d\n", i);
    }
}

#define IS_FD_OK(FD) ((FD) >= 0 && (FD) < NBR_FD && (fd_xlat[(FD)].flags & FLAGS_USED))
#define TEST_FD(FD) 				\
do 						\
{						\
  if (!(IS_FD_OK (FD)))				\
    {						\
      errno = EBADF;				\
      return -1;				\
    }						\
}						\
while (0)

int
isatty (int fd)
{
  TEST_FD (fd);
  return fd < 3;
}

int
write (int fd, const void *buf, size_t size)
{
  TEST_FD (fd);
  return (*fd_xlat[fd].handler->write)(fd, buf, size);
}

int
read (int fd, void *buf, size_t size)
{
  TEST_FD (fd);
  return (*fd_xlat[fd].handler->read)(fd, buf, size);
}

off_t
lseek (int fd, off_t pos, int whence)
{
  diag_write_string ("function lseek called\a\n");
  return 0;
}

int
fstat (int fd, struct stat *buf)
{
  diag_printf ("function fstat %d called\n", fd);
  return 0;
}

int
close (int fd)
{
  TEST_FD (fd);
  return (*fd_xlat[fd].handler->close)(fd);
}

int
ioctl (int fd, unsigned long cmd, void *buf)
{
  TEST_FD (fd);
  return (*fd_xlat[fd].handler->ioctl)(fd, cmd, buf);
}

/* Stupid memory allocator.  */
static char memory[256 * 1024];
static char *lastmem = memory;
void *
sbrk (size_t incr)
{
  void *res = lastmem;

  if (lastmem + incr > memory + sizeof memory)
    {
      diag_write_string ("sbrk: memory exhausted\n");
      errno = ENOMEM;
      return (void *)-1;
    }
  lastmem += incr;
  if (lastmem < memory)
    diag_write_string ("sbrk: memory underflow\n");
  return res;
}
