/*       tQL - relational database interface */
/*       Copyright (C) 1999 Timothy Fisken (tim@rose-cottage.demon.co.uk) */
     
/*       This program is free software; you can redistribute it and/or modify */
/*       it under the terms of the GNU General Public License as published by */
/*       the Free Software Foundation; either version 1, or (at your option) */
/*       any later version. */
    
/*       This program is distributed in the hope that it will be useful, */
/*       but WITHOUT ANY WARRANTY; without even the implied warranty of */
/*       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */
/*       GNU General Public License for more details. */
     
/*       You should have received a copy of the GNU General Public License */
/*       along with this program; if not, write to the Free Software */
/*       Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

/*
 * PGtQL - tQL interface to the Postgres95 database
 */

#include <tcl.h>
#include <pgsql/libpq-fe.h>
#include <stdio.h>
#include <malloc.h>

/*
 * Wrapping up PGconn as a TCL object
 */

/* Prototypes for the type functions */
void PGConn_FreeInternalRep(Tcl_Obj*);
void PGConn_DupInternalRep(Tcl_Obj*, Tcl_Obj*);
void PGConn_UpdateString(Tcl_Obj*);
int PGConn_SetFromAny(Tcl_Interp*, Tcl_Obj*);

/* Type definition structure */
Tcl_ObjType PGConn_ObjType = {
  "PGConn",
  PGConn_FreeInternalRep,
  PGConn_DupInternalRep,
  PGConn_UpdateString,
  PGConn_SetFromAny
};

/* Function definitions */

/* Closes the conection when the object is freed */
void PGConn_FreeInternalRep(Tcl_Obj* obj)
{
  PQfinish((PGconn*) obj->internalRep.otherValuePtr);
}

/* Duplicate the connection - the two objects simply share a pointer */
void PGConn_DupInternalRep(Tcl_Obj* src, Tcl_Obj* dup)
{
  dup->internalRep.otherValuePtr = src->internalRep.otherValuePtr;
}

/* Make a new string representation of the PQConn* */
void PGConn_UpdateString(Tcl_Obj* src)
{
  char* buffer;

  asprintf(&buffer, "PGConn@%p", src->internalRep.otherValuePtr);
  
  src->length = strlen(buffer);
  src->bytes = Tcl_Alloc(src->length + 1);
  strcpy(src->bytes, buffer);
  free(buffer);
}

/* Attempt to convert a string to a PGConn* */
int PGConn_SetFromAny(Tcl_Interp* interp, Tcl_Obj* obj)
{
  PGconn* conn;

  if( sscanf(obj->bytes, "PGConn@%p", &conn) != 1 )
    {
      Tcl_Obj *ret;
      char* msg;
      ret = Tcl_GetObjResult (interp);

      asprintf(&msg, "%s is not a PGConn object", obj->bytes);
      Tcl_SetStringObj(ret, msg, strlen(msg));
      free(msg);
      return TCL_ERROR;
    }

  obj->internalRep.otherValuePtr = conn;
  return TCL_OK;
}

/* Sets the internal representation to an existing PGconn* */
void PGConn_SetObj(Tcl_Obj* obj, PGconn* c)
{
  obj->typePtr = &PGConn_ObjType;
  obj->internalRep.otherValuePtr = c;
  Tcl_InvalidateStringRep(obj);
}

/* Create a new PGConn object (from an existing connection) */
Tcl_Obj* PGConn_NewObj(PGconn* c)
{
  Tcl_Obj* obj;

  obj = Tcl_NewObj();
  PGConn_SetObj(obj, c);
  return obj;
}

/* Extracts the relevant connection stuff */
PGconn* PGConn_GetFromObj(Tcl_Interp* interp, Tcl_Obj* obj)
{
  if( obj->typePtr != &PGConn_ObjType )
    if( Tcl_ConvertToType(interp, obj, &PGConn_ObjType) != TCL_OK )
      return NULL;

  return (PGconn*) obj->internalRep.otherValuePtr;
}



/*
 * Wrap up a PQresult as a Tcl_Obj.
 */

/* Prototypes for the type functions */

void PGResult_FreeInternalRep(Tcl_Obj*);
void PGResult_DupInternalRep(Tcl_Obj*, Tcl_Obj*);
void PGResult_UpdateString(Tcl_Obj*);
int PGResult_SetFromAny(Tcl_Interp*, Tcl_Obj*);

Tcl_ObjType PGResult_ObjType = {
  "PGResult",
  PGResult_FreeInternalRep,
  PGResult_DupInternalRep,
  PGResult_UpdateString,
  PGResult_SetFromAny
};

/* Function definitions */

/* Frees the result */
void PGResult_FreeInternalRep(Tcl_Obj* obj)
{
  PQclear((PGresult*) obj->internalRep.otherValuePtr);
}

/* Share the pointer between two objects */
void PGResult_DupInternalRep(Tcl_Obj* src, Tcl_Obj* dup)
{
  dup->internalRep.otherValuePtr = src->internalRep.otherValuePtr;
}

/* Make a string representation of PQresult */
void PGResult_UpdateString(Tcl_Obj* src)
{
  char* buffer;

  asprintf(&buffer, "PGResult@%p", src->internalRep.otherValuePtr);

  src->length = strlen(buffer);
  src->bytes = Tcl_Alloc(src->length + 1);
  strcpy(src->bytes, buffer);
  free(buffer);
}

/* Attempt to convert a string to a PGresult* */
int PGResult_SetFromAny(Tcl_Interp* interp, Tcl_Obj* obj)
{
  PGresult* result;

  if( sscanf(obj->bytes, "PGResult@%p", &result) != 1 )
    {
      Tcl_Obj *ret;
      char* msg;
      ret = Tcl_GetObjResult (interp);

      asprintf(&msg, "%s is not a PGResult object", obj->bytes);
      Tcl_SetStringObj(ret, msg, strlen(msg));
      free(msg);
      return TCL_ERROR;
    }

  obj->internalRep.otherValuePtr = result;
  return TCL_OK;
}

/* Sets an object to represent a PGresult* */
void PGResult_SetObj(Tcl_Obj* obj, PGresult* r)
{
  obj->typePtr = &PGResult_ObjType;
  obj->internalRep.otherValuePtr = r;
  Tcl_InvalidateStringRep(obj);
}

/* Create a new Tcl_Obj representing a PGresult */
Tcl_Obj* PGResult_NewObj(PGresult* r)
{
  Tcl_Obj* ret;

  ret = Tcl_NewObj();
  PGResult_SetObj(ret, r);
  return ret;
}

/* Extract the actual PGresult* */
PGresult* PGResult_GetFromObj(Tcl_Interp* interp, Tcl_Obj* obj)
{
    if( obj->typePtr != &PGResult_ObjType )
    if( Tcl_ConvertToType(interp, obj, &PGResult_ObjType) != TCL_OK )
      return NULL;

  return (PGresult*) obj->internalRep.otherValuePtr;
}



/*
 * Functions to be called from tQL scripts
 */

/* Open a connection to a database, returning it wrapped up as a Tcl_Object */
int PGtQL_OpenDB(ClientData clientData, Tcl_Interp* interp, int objc, Tcl_Obj** objv)
{
  char* host, *port, *dbname, *options, *errorString = "Unknown error";
  int i;
  Tcl_Obj* result;
  PGconn* conn;
  
  host = port = dbname = options = NULL;
  result = Tcl_GetObjResult(interp);
  
  for( i = 1; i < objc; i++ )
    {
      char* s;

      s = Tcl_GetStringFromObj(objv[i], NULL);

      if( !strcmp(s, "-host") )
	{
	  if( ++i >= objc )
	    {
	      errorString = "Specify a value after -host";
	      goto error;
	    }
	  host = Tcl_GetStringFromObj(objv[i], NULL);
	}
      else if( !strcmp(s, "-port") )
	{
	  if( ++i >= objc )
	    {
	      errorString = "Specify a value after -port";
	      goto error;
	    }
	  port = Tcl_GetStringFromObj(objv[i], NULL);
	}
      else if( !strcmp(s, "-db") )
	{
	  if( ++i >= objc )
	    {
	      errorString = "Specify a value after -db";
	      goto error;
	    }
	  dbname = Tcl_GetStringFromObj(objv[i], NULL);
	}
      else
	{
	  errorString = "Unsupported option passed";
	  goto error;
	}
    }

  conn = PQsetdb(host, port, NULL, NULL, dbname);
  if( PQstatus(conn) == CONNECTION_BAD )
    {
      errorString = PQerrorMessage(conn);
      PQfinish(conn);
      goto error;
    }

  PGConn_SetObj(result, conn);
  return TCL_OK;
  
error:
  Tcl_SetStringObj(result, errorString, strlen(errorString));
  return TCL_ERROR;
}

/* Execute a query on the server */
int PGtQL_Exec(ClientData clientData, Tcl_Interp* interp, int objc, Tcl_Obj** objv)
{
  char* errorString = "Unknown error", * query;
  Tcl_Obj* ret;
  PGconn* conn;
  PGresult* result;
  
  ret = Tcl_GetObjResult(interp);
  
  if( objc == 3 )
    {
      if( (conn = PGConn_GetFromObj(interp, objv[1])) == NULL )
	{
	  errorString = "First argument must be a database connection";
	  goto error;
	}
      query = Tcl_GetStringFromObj(objv[2], NULL);
    }
  else
    {
      errorString = "Exec requires 2 arguments";
      goto error;
    }

  if( (result = PQexec(conn, query)) == NULL )
    {
      errorString = PQerrorMessage(conn);
      goto error;
    }
  
  switch( PQresultStatus(result) )
    {
    case PGRES_TUPLES_OK:
      PGResult_SetObj(ret, result);
      return TCL_OK;
    case PGRES_COMMAND_OK:
      PGResult_SetObj(ret, result);
      return TCL_OK;
    case PGRES_EMPTY_QUERY:
      errorString = "Query must contain some data!";
      PQclear(result);
      goto error;
    default:
      errorString = "Bad query";
      PQclear(result);
      goto error;
    };
  
error:
  Tcl_SetStringObj(ret, errorString, strlen(errorString));
  return TCL_ERROR;
}

/* Prints a result structure, with nice formatting */
int PGtQL_Print(ClientData clientData, Tcl_Interp* interp, int objc, Tcl_Obj** objv)
{
  PGresult* result;
  char* errorString;
  Tcl_Obj* ret;
  PQprintOpt opt = { 1, 1, 0, 0, 0, 0, " | ", NULL, NULL };
  
  ret = Tcl_GetObjResult(interp);
  
  if( objc != 2 )
    {
      errorString =  "Wrong number of args";
      goto error;
      return TCL_ERROR;
    }

  if( (result = PGResult_GetFromObj(interp, objv[1])) == NULL )
    {
      errorString = "First argument must be a result";
      goto error;
      return TCL_ERROR;
    }

  PQprint(stdout, result, &opt);

  Tcl_SetStringObj(ret, "", 0);
  return TCL_OK;

error:
  Tcl_SetStringObj(ret, errorString, strlen(errorString));
  return TCL_ERROR;
}
       
/* Extract information from the PGresult structure */

/* No. of rows in a result */
int PGtQL_Rows(ClientData clientData, Tcl_Interp* interp, int objc, Tcl_Obj** objv)
{
 PGresult* result;
 Tcl_Obj* ret;
 char* errorString = "Unknown error";

 ret = Tcl_GetObjResult(interp);
 
 if( objc != 2 )
   {
     errorString = "Rows requires one argument";
     goto error;
   }
 if( (result = PGResult_GetFromObj(interp, objv[1])) == NULL )
   {
     errorString = "Argument 1 must be a result";
     goto error;
   }

 Tcl_SetIntObj(ret, PQntuples(result));
 return TCL_OK;

error:
  Tcl_SetStringObj(ret, errorString, strlen(errorString));
  return TCL_ERROR;
}

/* No. of fields in a result */
int PGtQL_Fields(ClientData clientData, Tcl_Interp* interp, int objc, Tcl_Obj** objv)
{
 PGresult* result;
 Tcl_Obj* ret;
 char* errorString = "Unknown error";

 ret = Tcl_GetObjResult(interp);
 
 if( objc != 2 )
   {
     errorString = "Fields requires one argument";
     goto error;
   }
 if( (result = PGResult_GetFromObj(interp, objv[1])) == NULL )
   {
     errorString = "Argument 1 must be a result";
     goto error;
   }

 Tcl_SetIntObj(ret, PQnfields(result));
 return TCL_OK;

error:
  Tcl_SetStringObj(ret, errorString, strlen(errorString));
  return TCL_ERROR;
}

/* The name associated with a given field */
int PGtQL_Fieldname(ClientData data, Tcl_Interp* interp, int objc, Tcl_Obj** objv)
{
 PGresult* result;
 int fieldno;
 Tcl_Obj* ret;
 char* errorString = "Unknown error", *name;

 ret = Tcl_GetObjResult(interp);
 
 if( objc != 3 )
   {
     errorString = "Fieldname requires two arguments";
     goto error;
   }
 if( (result = PGResult_GetFromObj(interp, objv[1])) == NULL )
   {
     errorString = "Argument 1 must be a result";
     goto error;
   }
 if( Tcl_GetIntFromObj(interp, objv[2], &fieldno) == TCL_ERROR )
   {
     errorString = "Argument 2 must be an integer";
     goto error;
   }
 if( (name = PQfname(result, fieldno)) == NULL )
   {
     errorString = "Field number out of range";
     goto error;
   }
 
 Tcl_SetStringObj(ret, name, strlen(name));
 return TCL_OK;

error:
  Tcl_SetStringObj(ret, errorString, strlen(errorString));
  return TCL_ERROR;
}  

/* Return a value from the result structure (don't use binary cursors) */
int PGtQL_Value(ClientData clientData, Tcl_Interp* interp, int objc, Tcl_Obj** objv)
{
 PGresult* result;
 int fieldno, rowno;
 Tcl_Obj* ret;
 char* errorString = "Unknown error", *value;

 ret = Tcl_GetObjResult(interp);
 
 if( objc != 4 )
   {
     errorString = "Value requires three arguments";
     goto error;
   }
 if( (result = PGResult_GetFromObj(interp, objv[1])) == NULL )
   {
     errorString = "Argument 1 must be a result";
     goto error;
   }
 if( Tcl_GetIntFromObj(interp, objv[2], &rowno) == TCL_ERROR )
   {
     errorString = "Argument 2 must be an integer";
   }
  if( Tcl_GetIntFromObj(interp, objv[3], &fieldno) == TCL_ERROR )
   {
     errorString = "Argument 3 must be an integer";
     goto error;
   }
 if( (value = PQgetvalue(result, rowno, fieldno)) == NULL )
   {
     errorString = "Row/Field number out of range";
     goto error;
   }
 
 Tcl_SetStringObj(ret, value, strlen(value));
 return TCL_OK;

error:
  Tcl_SetStringObj(ret, errorString, strlen(errorString));
  return TCL_ERROR;
}  

/* Extract the status information from a PGresult */
int PGtQL_Status(ClientData clientData, Tcl_Interp* interp, int objc, struct Tcl_Obj* CONST objv[])
{
 PGresult* result;
 Tcl_Obj* ret;
 char* errorString = "Unknown error", *status;

 ret = Tcl_GetObjResult(interp);
 
 if( objc != 2 )
   {
     errorString = "Status requires one argument";
     goto error;
   }
 if( (result = PGResult_GetFromObj(interp, objv[1])) == NULL )
   {
     errorString = "Argument 1 must be a result";
     goto error;
   }
 if( (status = PQcmdStatus(result)) == NULL )
   {
     errorString = "Bad result - not a command status";
     goto error;
   }
 
 Tcl_SetStringObj(ret, status, strlen(status));
 
 return TCL_OK;

error:
  Tcl_SetStringObj(ret, errorString, strlen(errorString));
  return TCL_ERROR;
}  

/* Initialize the PQtQL interface, called when the shared library is loaded */
int Pgtql_Init(Tcl_Interp* interp)
{
  if( Tcl_CreateObjCommand(interp, "::Postgres::OpenDB", (Tcl_ObjCmdProc*) PGtQL_OpenDB,
			   NULL, NULL) == NULL )
    return TCL_ERROR;

  if( Tcl_CreateObjCommand(interp, "::Postgres::Exec",
			   (Tcl_ObjCmdProc*) PGtQL_Exec,
			   NULL, NULL) == NULL )
    return TCL_ERROR;

  if( Tcl_CreateObjCommand(interp, "::Postgres::Print",
			   (Tcl_ObjCmdProc*) PGtQL_Print,
			   NULL, NULL) == NULL )
    return TCL_ERROR;

  if( Tcl_CreateObjCommand(interp, "::Postgres::Rows",
			   (Tcl_ObjCmdProc*) PGtQL_Rows,
			   NULL, NULL) == NULL )
    return TCL_ERROR;
  
  if( Tcl_CreateObjCommand(interp, "::Postgres::Fields",
			   (Tcl_ObjCmdProc*) PGtQL_Fields,
			   NULL, NULL) == NULL )
    return TCL_ERROR;
  
  if( Tcl_CreateObjCommand(interp, "::Postgres::Fieldname",
			   (Tcl_ObjCmdProc*) PGtQL_Fieldname,
			   NULL, NULL) == NULL )
    return TCL_ERROR;
  
  if( Tcl_CreateObjCommand(interp, "::Postgres::Value",
			   (Tcl_ObjCmdProc*) PGtQL_Value,
			   NULL, NULL) == NULL )
    return TCL_ERROR;
  
  if( Tcl_CreateObjCommand(interp, "::Postgres::Status",
			   (Tcl_ObjCmdProc*) PGtQL_Status,
			   NULL, NULL) == NULL )
    return TCL_ERROR;
  
  return TCL_OK;
}

  

