#ifndef _TIMER_H
#define _TIMER_H
/* --------------------------------------------------------------------------
 * Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
 * All rights reserved.
 *
 * You can use and distribute this software under the terms of the license
 * you should have received along with this software; either version 1.1 of
 * the license, or (at your option) any later version.
 * For a copy of the license or for additional information about this software,
 * write to Xcc Software, Durlacher Allee 53, D-76131 Karlsruhe, Germany;
 * Email: obst@xcc-ka.de.
 * --------------------------------------------------------------------------
 * Simple class for measuring and processing times.
 * --------------------------------------------------------------------------
 * ORIGINAL: D. Theobald				DATE: 14/2/93
 * --------------------------------------------------------------------------
 */
#define OBST_IMP_TIME
#define OBST_IMP_PROCRESOURCE
#include "obst_stdinc.h"

#ifdef __cplusplus
extern "C" {
#endif

typedef void* 					  Timer_t;
typedef enum { TP_WALLCLOCK, TP_SYSTEM, TP_USER } time_kind;

extern Timer_t tmr_new    __FSIG_(());
extern void    tmr_delete __FSIG_((Timer_t));

extern void    tmr_read_timer 	  __FSIG_((Timer_t));
extern void    tmr_write_duration __FSIG_((Timer_t, int, char*, time_kind));
extern void    tmr_print	  __FSIG_((Timer_t));
extern void    tmr_print_time	  __FSIG_((Timer_t, time_kind));

extern Timer_t tmr_assign   __FSIG_((Timer_t, Timer_t));
extern Timer_t tmr_negate   __FSIG_((Timer_t));
extern Timer_t tmr_add	    __FSIG_((Timer_t, Timer_t));
extern Timer_t tmr_sub	    __FSIG_((Timer_t, Timer_t));
extern Timer_t tmr_add_to   __FSIG_((Timer_t, Timer_t));
extern Timer_t tmr_sub_from __FSIG_((Timer_t, Timer_t));
extern double  tmr_2double  __FSIG_((Timer_t, time_kind));

#ifdef __cplusplus
}
#include <stream.h>

class Timer
{ public:
     void       read_timer ();
     void	write_duration (ostream&, const int, const char* = "s",
				time_kind = TP_WALLCLOCK) const;

     Timer  operator- () const;

     Timer  operator+ (const Timer&) const;
     Timer  operator- (const Timer&) const;

     Timer& operator += (const Timer&);
     Timer& operator -= (const Timer&);

     double cvt2double(time_kind = TP_WALLCLOCK) const;

     ostream& print (ostream&, time_kind = TP_WALLCLOCK) const;

#ifndef TEST_TIMER
  private:
#endif
     struct timeval tv[3];
     int	    negative[3];

#ifdef TIMER_PROCRESOURCES
  public:
     int	    blocks_in, blocks_out;
#endif

     friend ostream& operator<< (ostream&, const Timer&);
};

// --------------------------- realisation ---------------------------

inline double Timer::cvt2double(time_kind k /* = TP_WALLCLOCK */) const
{  double result = tv[k].tv_sec + ((double)tv[k].tv_usec / (double)1000000);
   return (negative[k]) ? -result : result;
}

inline void Timer::read_timer()
{
#if defined(__GNUG__) && __GNUG__ == 1
   struct c_proto_timezone tz;
#else
   struct timezone tz;
#endif
   gettimeofday(&tv[TP_WALLCLOCK],(struct timezone *)&tz);
   
   struct rusage rusage;
   getrusage(RUSAGE_SELF, &rusage);
   tv[TP_USER]   = rusage.ru_utime;
   tv[TP_SYSTEM] = rusage.ru_stime;

   negative[TP_WALLCLOCK] = 0;
   negative[TP_USER]      = 0;
   negative[TP_SYSTEM]    = 0;

#ifdef TIMER_PROCRESOURCES
   blocks_in  = rusage.ru_inblock;
   blocks_out = rusage.ru_oublock;
#endif
}

inline Timer Timer::operator- () const
{  Timer neg = *this;
   neg.negative[TP_WALLCLOCK] = !negative[TP_WALLCLOCK];
   neg.negative[TP_SYSTEM]    = !negative[TP_SYSTEM];
   neg.negative[TP_USER]      = !negative[TP_USER];

#ifdef TIMER_PROCRESOURCES
   neg.blocks_in  = -blocks_in;
   neg.blocks_out = -blocks_out;
#endif

   return neg;
}

inline Timer Timer::operator+ (const Timer &tp) const
{  Timer result = *this;
   result += tp;
   return result;
}

inline Timer Timer::operator- (const Timer &tp) const
{  return (*this) + (-tp);
}

inline Timer& Timer::operator-= (const Timer &tp)
{  return (*this) += (-tp);
}

#endif __cplusplus

#endif
