/* --------------------------------------------------------------------------
 * Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
 * All rights reserved.
 *
 * You can use and distribute this software under the terms of the license
 * you should have received along with this software; either version 1.1 of
 * the license, or (at your option) any later version.
 * For a copy of the license or for additional information about this software,
 * write to Xcc Software, Durlacher Allee 53, D-76131 Karlsruhe, Germany;
 * Email: obst@xcc-ka.de.
 *
 * --------------------------------------------------------------------------
 * Implementation of the C --> OBST Interface
 *
 * --> error handling.
 * --> container operations.
 * --> iterators.
 * --------------------------------------------------------------------------
 * ORIGINAL: D. Theobald				DATE: 14/2/93
 * --------------------------------------------------------------------------
 */
/* tclOBST LIBRARY MODULE */

#define OBST_IMP_STDCONST
#include "obst_stdinc.h"

#include "obst_progstd.h"
#include "mta_use.h"

#include "_c2obst.h"


// --------------------------------------------------------------------
// DEFINITIONS: stream-like interface
// --------------------------------------------------------------------
#ifdef c2o_STREAM_IF

EXPORT c2o_CallManip c2o_CallManip::manip;

#endif


// --------------------------------------------------------------------
// IMPLEMENTATION: error handling
// --------------------------------------------------------------------

EXPORT void obst_err_raise (err_class cl, err_msg msg,
					  err_msg where, int copy)
{  T_PROC ("obst_err_raise")
   TT (c2o_M, T_ENTER);

   err_raise (cl, msg, where, copy);

   TT (c2o_M, T_LEAVE);
}

EXPORT int obst_err_occurred (err_class cl)
{  T_PROC ("obst_err_occurred")
   TT (c2o_M, T_ENTER; T_LEAVE);

   return err_occurred (cl);
}
EXPORT void obst_err_reset()
{  T_PROC ("obst_err_reset")
   TT (c2o_M, T_ENTER);

   err_reset();

   TT (c2o_M, T_LEAVE);
}
EXPORT err_class obst_err_last_class()
{  T_PROC ("obst_err_last_class")
   TT (c2o_M, T_ENTER; T_LEAVE);

   return err_last_class();
}
EXPORT err_msg obst_err_last_raised()
{  T_PROC ("obst_err_last_raised")
   TT (c2o_M, T_ENTER; T_LEAVE);

   return err_last_raised();
}
EXPORT err_msg obst_err_last_origin()
{  T_PROC ("obst_err_last_origin")
   TT (c2o_M, T_ENTER; T_LEAVE);

   return err_last_origin();
}
EXPORT void obst_err_blockend()
{  T_PROC ("obst_err_blockend")
   TT (c2o_M, T_ENTER);

   err_pop_env();

   TT (c2o_M, T_LEAVE);
}
EXPORT obst_proc* obst_err_sethandler (obst_proc* newhandler)
{  T_PROC ("obst_err_sethandler")
   TT (c2o_M, T_ENTER);

   obst_proc* old_handler = (obst_proc*) err_output_handler;
   err_output_handler = newhandler;

   TT (c2o_M, T_LEAVE);
   return old_handler;
}

EXPORT int obst_err_blockstart()
{  T_PROC ("obst_err_blockstart")
   TT (c2o_M, T_ENTER);

   err_push_env();

   TT (c2o_M, T_LEAVE);
   return !sigsetjmp(*err_top_env,0);
}


// --------------------------------------------------------------------
// IMPLEMENTATION: container operations
// --------------------------------------------------------------------

// --------- querying -------------------------------------------------

EXPORT sos_Object obst_cnt_root_object (sos_Container ct)
{  T_PROC ("obst_cnt_root_object")
   TT (c2o_M, T_ENTER);

   sos_Object result;
   ERR_HANDLING_WRAPPER (result = ct.root_object();,
			 result = NO_OBJECT;)
   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT sos_Container_status obst_cnt_status (sos_Container ct)
{  T_PROC ("obst_cnt_status")
   TT (c2o_M, T_ENTER);

   sos_Container_status result;
   ERR_HANDLING_WRAPPER (result = ct.status();,
			 result = UNAVAILABLE;)
   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT sos_Bool obst_cnt_modified (sos_Container ct)
{  T_PROC ("obst_cnt_modified")
   TT (c2o_M, T_ENTER);

   sos_Bool result;
   ERR_HANDLING_WRAPPER (result = ct.modified();,
			 result = FALSE;)
   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT sos_Bool obst_cnt_exists (sos_Container ct)
{  T_PROC ("obst_cnt_exists")
   TT (c2o_M, T_ENTER);

   sos_Bool result;
   ERR_HANDLING_WRAPPER (result = ct.exists();,
			 result = FALSE;)
   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT sos_Int obst_cnt_occupied (sos_Container ct)
{  T_PROC ("obst_cnt_occupied")
   TT (c2o_M, T_ENTER);

   sos_Int result;
   ERR_HANDLING_WRAPPER (result = ct.occupied();,
			 result = 0;)
   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT sos_Existing_status obst_cnt_object_exists (sos_Container ct,
						   sos_Offset    os,
						   sos_Int 	 sz)
{  T_PROC ("obst_cnt_object_exists")
   TT (c2o_M, T_ENTER);

   sos_Existing_status result;
   ERR_HANDLING_WRAPPER (result = ct.object_exists (os,sz);,
			 result = PERHAPS_EXISTING;)
   TT (c2o_M, T_LEAVE);
   return result;
}


// --------- creation, destruction, ... -------------------------------

EXPORT sos_Container obst_cnt_create ()
{  T_PROC ("obst_cnt_create")
   TT (c2o_M, T_ENTER);

   sos_Container result;
   ERR_HANDLING_WRAPPER (result = sos_Container::create();,
			 result = UNUSED_CONTAINER;)
   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT void obst_cnt_destroy (sos_Container ct)
{  T_PROC ("obst_cnt_destroy")
   TT (c2o_M, T_ENTER);

   ERR_HANDLING_WRAPPER (ct.destroy();, ;);

   TT (c2o_M, T_LEAVE);
}

EXPORT void obst_cnt_clear (sos_Container ct)
{  T_PROC ("obst_cnt_clear")
   TT (c2o_M, T_ENTER);

   ERR_HANDLING_WRAPPER (ct.clear();, ;);

   TT (c2o_M, T_LEAVE);
}

EXPORT void obst_cnt_update (sos_Container ct)
{  T_PROC ("obst_cnt_update")
   TT (c2o_M, T_ENTER);

   ERR_HANDLING_WRAPPER (ct.update();, ;);

   TT (c2o_M, T_LEAVE);
}

EXPORT void obst_cnt_compress (sos_Container ct)
{  T_PROC ("obst_cnt_compress")
   TT (c2o_M, T_ENTER);

   ERR_HANDLING_WRAPPER (ct.compress();, ;);

   TT (c2o_M, T_LEAVE);
}

EXPORT void obst_cnt_squeeze (sos_Container ct)
{  T_PROC ("obst_cnt_squeeze")
   TT (c2o_M, T_ENTER);

   ERR_HANDLING_WRAPPER (ct.squeeze();, ;);

   TT (c2o_M, T_LEAVE);
}


// --------- opening, locking -----------------------------------------

EXPORT sos_Open_result obst_cnt_open (sos_Container   ct,
				      sos_Access_mode am, sos_Sync_mode sm)
{  T_PROC ("obst_cnt_open")
   TT (c2o_M, T_ENTER);

   sos_Open_result result;
   ERR_HANDLING_WRAPPER (result = ct.open (am, sm);,
			 result = UNACCESSIBLE;)
   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT sos_Open_result obst_cnt_access (sos_Container   ct,
				        sos_Access_mode am, sos_Sync_mode sm)
{  T_PROC ("obst_cnt_access")
   TT (c2o_M, T_ENTER);

   sos_Open_result result;
   ERR_HANDLING_WRAPPER (result = ct.access (am, sm);,
			 result = UNACCESSIBLE;)
   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT void obst_cnt_commit (sos_Container ct)
{  T_PROC ("obst_cnt_commit")
   TT (c2o_M, T_ENTER);

   ERR_HANDLING_WRAPPER (ct.commit();, ;);

   TT (c2o_M, T_LEAVE);
}

EXPORT void obst_cnt_reset (sos_Container ct)
{  T_PROC ("obst_cnt_reset")
   TT (c2o_M, T_ENTER);

   ERR_HANDLING_WRAPPER (ct.reset();, ;);

   TT (c2o_M, T_LEAVE);
}

EXPORT void obst_cnt_close (sos_Container ct)
{  T_PROC ("obst_cnt_close")
   TT (c2o_M, T_ENTER);

   ERR_HANDLING_WRAPPER (ct.close();, ;);

   TT (c2o_M, T_LEAVE);
}


// --------------------------------------------------------------------
// IMPLEMENTATION: container set operations
// --------------------------------------------------------------------

// --------- constructors, desctructors -------------------------------

EXPORT obst_Container_set obst_cset_create ()
{  T_PROC ("obst_cset_create")
   TT (c2o_M, T_ENTER);

   obst_Container_set result;
   ERR_HANDLING_WRAPPER (result = (obst_Container_set) new sos_Container_set;,
			 result = NULL;)
   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT void obst_cset_destroy (obst_Container_set cs)
{  T_PROC ("obst_cset_destroy")
   TT (c2o_M, T_ENTER);

   ERR_HANDLING_WRAPPER (delete (sos_Container_set*)cs;, ;);

   TT (c2o_M, T_LEAVE);
}

// --------- querying -------------------------------------------------

EXPORT obst_Container_set obst_cset_open_containers (sos_Container_status s)
{  T_PROC ("obst_cset_open_containers")
   TT (c2o_M, T_ENTER);

   obst_Container_set result;
   ERR_HANDLING_WRAPPER (
      result = (obst_Container_set)&sos_Container_set::open_containers (s);,
      result = NULL;)

   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT obst_Container_set obst_cset_open_containers_mod
 						(sos_Container_mod_status ms)
{  T_PROC ("obst_cset_open_containers_mod")
   TT (c2o_M, T_ENTER);

   obst_Container_set result;
   ERR_HANDLING_WRAPPER (
      result = (obst_Container_set)&sos_Container_set::open_containers (ms);,
      result = NULL;)

   TT (c2o_M, T_LEAVE);
   return result;
}

// --------- set manipulation -----------------------------------------

#if OBSTVERSIONNO >= 336
EXPORT void obst_cset_clear (obst_Container_set cs)
{  T_PROC ("obst_cset_insert")
   TT (c2o_M, T_ENTER);

   ERR_HANDLING_WRAPPER (((sos_Container_set*)cs)->clear();, ;);
   
   TT (c2o_M, T_LEAVE);
}
#endif

EXPORT obst_Container_set obst_cset_insert (obst_Container_set cs,
					    sos_Container      ct)
{  T_PROC ("obst_cset_insert")
   TT (c2o_M, T_ENTER);

   obst_Container_set result;
   ERR_HANDLING_WRAPPER (
      result = (obst_Container_set)&( (* (sos_Container_set*)cs) += ct );,
      result = NULL;)

   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT obst_Container_set obst_cset_remove (obst_Container_set cs,
					    sos_Container      ct)
{  T_PROC ("obst_cset_remove")
   TT (c2o_M, T_ENTER);

   obst_Container_set result;
   ERR_HANDLING_WRAPPER (
      result = (obst_Container_set)&( (* (sos_Container_set*)cs) -= ct );,
      result = NULL;)

   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT obst_Container_set obst_cset_add (obst_Container_set cs,
					 obst_Container_set a_cs)
{  T_PROC ("obst_cset_add")
   TT (c2o_M, T_ENTER);

   obst_Container_set result;
   ERR_HANDLING_WRAPPER (
      result = (obst_Container_set)&( (* (sos_Container_set*)cs)
				     += (* (sos_Container_set*)a_cs) );,
      result = NULL;)

   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT obst_Container_set obst_cset_sub (obst_Container_set cs,
					 obst_Container_set a_cs)
{  T_PROC ("obst_cset_sub")
   TT (c2o_M, T_ENTER);

   obst_Container_set result;
   ERR_HANDLING_WRAPPER (
      result = (obst_Container_set)&( (* (sos_Container_set*)cs)
				      -= (* (sos_Container_set*)a_cs) );,
      result = NULL;)

   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT int obst_cset_card (obst_Container_set cs)
{  T_PROC ("obst_cset_card")
   TT (c2o_M, T_ENTER);

   int result;
   ERR_HANDLING_WRAPPER (result = ((sos_Container_set*)cs)->card();,
			 result = 0;)
   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT sos_Container obst_cset_at (obst_Container_set cs, register int idx)
{  T_PROC ("obst_cset_at")
   TT (c2o_M, T_ENTER);

   sos_Container	result;
   sos_Container_cursor c;
   register
   sos_Container_set*   csptr = (sos_Container_set*)cs;

   if (idx < 0 || idx >= csptr->card())
   {  err_raise(err_USE, "c2obst - obst_cset_at", "index out of range",FALSE);
      result = UNUSED_CONTAINER;
   }
   else
      ERR_HANDLING_WRAPPER (
         if (csptr->is_valid (c = csptr->open_cursor()))
	    while (idx--)
	       if (! csptr->to_succ (c))
	          break;
         result = (csptr->is_valid (c)) ? csptr->get (c)
			 	        : UNUSED_CONTAINER;,

	 result = UNUSED_CONTAINER;)

   TT (c2o_M, T_LEAVE);
   return result;
}

// --------- locking, unlocking ---------------------------------------

EXPORT sos_Open_result obst_cset_open_sets (obst_Container_set rcs,
					    obst_Container_set ccs,
					    obst_Container_set wcs,
					    sos_Sync_mode      sm)
{  T_PROC ("obst_cset_open_sets")
   TT (c2o_M, T_ENTER);

   sos_Open_result result;
   ERR_HANDLING_WRAPPER (
      result = sos_Container_set::open (*(sos_Container_set*)rcs,
				        *(sos_Container_set*)ccs,
				        *(sos_Container_set*)wcs, sm);,
      result = UNACCESSIBLE;)

   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT sos_Open_result obst_cset_open (obst_Container_set cs,
				       sos_Access_mode    am, sos_Sync_mode sm)
{  T_PROC ("obst_cset_open")
   TT (c2o_M, T_ENTER);

   sos_Open_result result;
   ERR_HANDLING_WRAPPER (result = ((sos_Container_set*)cs)->open (am, sm);,
			 result = UNACCESSIBLE;)

   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT void obst_cset_close (obst_Container_set cs)
{  T_PROC ("obst_cset_close")
   TT (c2o_M, T_ENTER);

   ERR_HANDLING_WRAPPER (((sos_Container_set*)cs)->close();, ;);

   TT (c2o_M, T_LEAVE);
}

EXPORT void obst_cset_commit (obst_Container_set cs)
{  T_PROC ("obst_cset_commit")
   TT (c2o_M, T_ENTER);

   ERR_HANDLING_WRAPPER (((sos_Container_set*)cs)->commit();, ;);

   TT (c2o_M, T_LEAVE);
}

EXPORT void obst_cset_reset (obst_Container_set cs)
{  T_PROC ("obst_cset_reset")
   TT (c2o_M, T_ENTER);

   ERR_HANDLING_WRAPPER (((sos_Container_set*)cs)->reset();, ;);

   TT (c2o_M, T_LEAVE);
}


// --------------------------------------------------------------------
// IMPLEMENTATION: iterators
// --------------------------------------------------------------------

struct IterDescr
{  union
   {  sos_Object_sos_Object_Association assoc;
      sos_Object_Collection	        coll;	
   } u;
   int	      is_collection, forward_scan;
   sos_Cursor cursor;
};

EXPORT void* obst_iter_start (const sos_Object* agg, sos_Bool forward)
{  T_PROC ("obst_iter_start")
   TT (c2o_M, T_ENTER);

   if (*(sos_Object*)agg == NO_OBJECT)
   {  TT (c2o_M, T_LEAVE);
      return NULL;
   }
   register IterDescr* descr = new IterDescr;

   descr->forward_scan = forward;
   ERR_HANDLING_WRAPPER (
      if (descr->is_collection
	  = ((sos_Object*)agg)->is_some (sos_Object_Collection_type))
      {  descr->u.coll = sos_Object_Collection::make (*agg);
	 descr->cursor = descr->u.coll.open_cursor (TEMP_CONTAINER);
	 if (! (descr->u.coll.is_valid (descr->cursor)
	        && (forward || descr->u.coll.to_last (descr->cursor))))
	 {  descr->u.coll.close_cursor (descr->cursor);
	    delete descr;
	    descr = NULL;
	 }
      }
      else
      {  descr->u.assoc = sos_Object_sos_Object_Association::make (*agg);
	 descr->cursor  = descr->u.assoc.open_cursor (TEMP_CONTAINER);
	 if (! (descr->u.assoc.is_valid (descr->cursor)
	        && (forward || descr->u.assoc.to_last (descr->cursor))))
	 {  descr->u.assoc.close_cursor (descr->cursor);
	    delete descr; 
	    descr = NULL;
	 }
      },

      delete descr; descr = NULL;)

   TT (c2o_M, T_LEAVE);
   return descr;
}

EXPORT void* obst_iter_next (void* _descr)
{  T_PROC ("obst_iter_next")
   TT (c2o_M, T_ENTER);

   register IterDescr* descr = (IterDescr*)_descr;
   ERR_HANDLING_WRAPPER (
      if (descr->is_collection)
      {  if (descr->forward_scan)
	 {  if (!descr->u.coll.to_succ (descr->cursor))
	       _descr = NULL;
	 }
	 else if (!descr->u.coll.to_pred (descr->cursor))
	    _descr = NULL;
      }
      else if (descr->forward_scan)
      {  if (!descr->u.assoc.to_succ (descr->cursor))
	    _descr = NULL;
      }
      else if (!descr->u.assoc.to_pred (descr->cursor))
	 _descr = NULL;,

      _descr = NULL;)

   if (!_descr)
   {  if (descr->is_collection)
	 descr->u.coll.close_cursor (descr->cursor);
      else
	 descr->u.assoc.close_cursor (descr->cursor);
      delete descr;
   }
   TT (c2o_M, T_LEAVE);
   return _descr;
}

EXPORT void obst_iter_finish (void* _descr)
{  T_PROC ("obst_iter_finish")
   TT (c2o_M, T_ENTER);

   if (_descr)
   {  register IterDescr* descr = (IterDescr*)_descr;
      ERR_HANDLING_WRAPPER (
	 if (descr->is_collection)
	    descr->u.coll.close_cursor (descr->cursor);
	 else
	    descr->u.assoc.close_cursor (descr->cursor);, ;)
      delete descr;
   }
   TT (c2o_M, T_LEAVE);
}

EXPORT char* obst_iter_elemtp (void* _descr, int pno)
{  T_PROC ("obst_iter_elemtp")
   TT (c2o_M, T_ENTER);

   if (_descr)
   {  sos_Type elem_tp;
      ERR_HANDLING_WRAPPER (
	 if (((IterDescr*)_descr)->is_collection)
	    elem_tp = sos_Class_type::make(((IterDescr*)_descr)->u.coll.type())
	       	      .generic_param (sos_Class_type::make (
				       sos_Object_Collection_type),
			       	      pno);
	 else
	    elem_tp = sos_Class_type::make(((IterDescr*)_descr)->u.coll.type())
	       	      .generic_param (sos_Class_type::make (
				       sos_Object_sos_Object_Association_type),
			       	      pno);
	 , ;)
      TT (c2o_M, T_LEAVE);
      return c2obst_encode_type (elem_tp);
   }
   TT (c2o_M, T_LEAVE);
   return NULL;
}

EXPORT sos_Object obst_iter_role1 (register void* _descr)
{  T_PROC ("obst_iter_role1")
   TT (c2o_M, T_ENTER);

   sos_Object result;

   if (((IterDescr*)_descr)->is_collection)
   {  err_raise (err_USE, "not applicable to collections",
		 	  "c2obst - obst_iter_role1", FALSE);
      result = NO_OBJECT;
   }
   else
      ERR_HANDLING_WRAPPER (
         result = ((IterDescr*)_descr)->u.assoc.get_role1 (
					       ((IterDescr*)_descr)->cursor);,
	 result = NO_OBJECT;)

   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT sos_Object obst_iter_role2 (register void* _descr)
{  T_PROC ("obst_iter_role2")
   TT (c2o_M, T_ENTER);

   sos_Object result;

   if (((IterDescr*)_descr)->is_collection)
   {  err_raise (err_USE, "not applicable to collections",
		 	  "c2obst - obst_iter_role2", FALSE);
      result = NO_OBJECT;
   }
   else
      ERR_HANDLING_WRAPPER (
         result = ((IterDescr*)_descr)->u.assoc.get_role2 (
					       ((IterDescr*)_descr)->cursor);,
	 result = NO_OBJECT;)

   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT sos_Object obst_iter_get (register void* _descr)
{  T_PROC ("obst_iter_get")
   TT (c2o_M, T_ENTER);

   sos_Object result;

   if (((IterDescr*)_descr)->is_collection)
   {  ERR_HANDLING_WRAPPER (
         result = ((IterDescr*)_descr)->u.coll.get (
					  ((IterDescr*)_descr)->cursor);,
	 result = NO_OBJECT;)
   }
   else
   {  err_raise (err_USE, "not applicable to associations",
		          "c2obst - obst_iter_get", FALSE);
      result = NO_OBJECT;
   }
   TT (c2o_M, T_LEAVE);
   return result;
}

// --------------------------------------------------------------------
// IMPLEMENTATION: stream-like interface
// --------------------------------------------------------------------
#ifdef c2o_STREAM_IF

EXPORT c2o_CallManip& c2o_eval ()
{  T_PROC ("c2o_CallManip::c2o_eval")
   TT (c2o_H, T_ENTER);

   c2o_CallManip& res = c2o_CallManip::manip;

   res.eval = TRUE;

   TT (c2o_H, T_LEAVE);
   return res;
}

EXPORT c2o_CallManip& c2o_method (const sos_Object& receiver,
				  const char*	    mnm,
				  unsigned	    argc /* = 0 */)
{  T_PROC ("c2o_CallManip::c2o_method(obj,m)")
   TT (c2o_H, T_ENTER);

   c2o_CallManip& res = c2o_CallManip::manip;

   res.eval	 = FALSE;
   res.class_nm  = NULL;
   res.method_nm = mnm;
   res.receiver  = receiver;
   res.argc	 = argc;

   TT (c2o_H, T_LEAVE);
   return res;
}

EXPORT c2o_CallManip& c2o_method (const char* cnm,
				  const char* mnm, unsigned argc /* = 0 */)
{  T_PROC ("c2o_CallManip::c2o_method(cl,m)")
   TT (c2o_L, T_ENTER);

   c2o_CallManip& res = c2o_CallManip::manip;

   res.eval	 = FALSE;
   res.class_nm  = cnm;
   res.method_nm = mnm;
   res.argc	 = argc;

   TT (c2o_H, T_LEAVE);
   return res;
}

EXPORT void c2o_Call::eval()
{  T_PROC ("c2o_Call::eval")
   TT (c2o_H, T_ENTER);

   if (class_nm)
      result_obj = obst_call_classmethod(class_nm, method_nm, argc_curr, argv);
   else
      result_obj = obst_call_method (&receiver, method_nm, argc_curr, argv);

   TT (c2o_H, T_LEAVE);
}

EXPORT c2o_Call& c2o_Call::operator<< (const c2o_CallManip& cm)
{  T_PROC ("c2o_Call::<<(c2o_CallManip)")
   TT (c2o_H, T_ENTER);

   if (cm.eval)
      eval();
   else
   {  method_nm = cm.method_nm;
      argc_curr = cm.argc;

      if (!(class_nm = cm.class_nm))
	 receiver = cm.receiver;
   }
   return *this;
   TT (c2o_H, T_LEAVE);
}

EXPORT c2o_Call& c2o_Call::operator<< (const sos_Object& arg)
{  T_PROC ("c2o_Call::<<(sos_Object)")
   TT (c2o_H, T_ENTER);

   if (argc_curr == argc_max)
   {  if (argc_max)
	 argv = (sos_Object*) REALLOC (argv,
				       (argc_max += 2) * sizeof(sos_Object));
      else
	 argv = (sos_Object*) malloc ((argc_max = 2) * sizeof(sos_Object));
   }
   argv[argc_curr ++] = arg;

   TT (c2o_H, T_LEAVE);
   return *this;
}

// --------------------------------------------------------------------

EXPORT c2o_Call& operator<< (c2o_Call& c, const sos_Int i)
{  T_PROC ("c2o_Call::<<(sos_Int)")
   TT (c2o_H, T_ENTER);

   sos_Object o = make_sos_Int_object (i);
   c << o;

   TT (c2o_H, T_LEAVE);
   return c;
}
EXPORT c2o_Call& operator<< (c2o_Call& c, const sos_Bool b)
{  T_PROC ("c2o_Call::<<(sos_Bool)")
   TT (c2o_H, T_ENTER);

   sos_Object o = make_sos_Bool_object (b);
   c << o;

   TT (c2o_H, T_LEAVE);
   return c;
}
EXPORT c2o_Call& operator<< (c2o_Call& c, const sos_Cstring str)
{  T_PROC ("c2o_Call::<<(sos_Cstring)")
   TT (c2o_H, T_ENTER);

   sos_Object o = make_sos_Cstring_object (str);
   c << o;

   TT (c2o_H, T_LEAVE);
   return c;
}
EXPORT c2o_Call& operator<< (c2o_Call& c, const sos_Container ct)
{  T_PROC ("c2o_Call::<<(sos_Container)")
   TT (c2o_H, T_ENTER);

   sos_Object o = make_sos_Container_object (ct);
   c << o;

   TT (c2o_H, T_LEAVE);
   return c;
}

EXPORT c2o_Call& operator>> (c2o_Call& c, sos_Int& i)
{  T_PROC ("c2o_Call::>>(sos_Int)")
   TT (c2o_H, T_ENTER);

   c.eval();
   i = obst_obj2int (&c.result());

   TT (c2o_H, T_LEAVE);
   return c;
}
EXPORT c2o_Call& operator>> (c2o_Call& c, sos_Bool& b)
{  T_PROC ("c2o_Call::>>(sos_Bool)")
   TT (c2o_H, T_ENTER);

   c.eval();

   b = FALSE;
   if (c.result() == NO_OBJECT)
      err_raise (err_USE, err_NO_OBJECT, "c2o_Call::>>(sos_Bool)", FALSE);
   else
      ERR_HANDLING_WRAPPER (b = make_sos_Bool(c.result());, ;)

   TT (c2o_H, T_LEAVE);
   return c;
}
EXPORT c2o_Call& operator>> (c2o_Call& c, sos_Cstring& str)
{  T_PROC ("c2o_Call::>>(sos_Cstring)")
   TT (c2o_H, T_ENTER);

   c.eval();
   str = obst_obj2str (&c.result());

   TT (c2o_H, T_LEAVE);
   return c;
}
EXPORT c2o_Call& operator>> (c2o_Call& c, sos_Container& ct)
{  T_PROC ("c2o_Call::>>(sos_Container)")
   TT (c2o_H, T_ENTER);

   c.eval();
   ct = obst_obj2cnt (&c.result());

   TT (c2o_H, T_LEAVE);
   return c;
}

#endif /* c2o_STREAM_IF */
