# --------------------------------------------------------------------------
# Copyright 1993-1994 by Forschungszentrum Informatik (FZI)
# All rights reserved.
#
# You can use and distribute this software under the terms of the license
# you should have received along with this software; either version 1.1 of
# the license, or (at your option) any later version.
# For a copy of the license or for additional information about this software,
# write to Xcc Software, Durlacher Allee 53, D-76131 Karlsruhe, Germany;
# Email: obst@xcc-ka.de.
# --------------------------------------------------------------------------
# Module: graph_obst.tcl
# Tcl version: 7.3 (Tcl/Tk/XF)
# Tk version: 3.6
# XF version: 2.2
#

# module contents
global moduleList
global autoLoadList
set moduleList(graph_obst.tcl) { db.init db.graph.init db.graph.save db.graph.clone db.graph.exit db.file.do db.graph.node db.graph.edge db.graph.ID_create db.cnt.access db.cnt.savepoint}
set autoLoadList(graph_obst.tcl) {0}

# procedures to show toplevel windows


# User defined procedures


# ---------
proc db.init {} {
#
# Initializes the OBST access module of the Graph Editor. This includes
# finding and opening the container which holds the graph to be edited.
#
# The editor may be invoked with an optional container argument. If present,
# this container is opened and the graph contained therein is edited.
#
# Otherwise, the editor will use a container whose name is registered in the
# "tclOBST Graph Editor" slot of the OBST root directory. This container/slot
# will then be created if not yet present.
#
# Afterwards, the graph is read from the container and displayed.
#
   global db_container db_container_orig db_container_new edit_mode edit_mode_orig main_self argv argc

   set db_container	 ""
   set db_container_orig ""
   set db_container_new  0

   if {$argc > 0} {
      set err  0
      set arg1 [lindex $argv 0]

      if {$argc == 1} {
	 if {$arg1 == "new"} {
	    set db_container_orig ""
	    set db_container	  [cnt create]
	    edit.mode.set	  new
	 } else {
	    set err 1
	 }
      } elseif {$argc != 2 || [catch {cnt exists $arg1} cnt_exists]} {
	 set err 1
      } elseif {! $cnt_exists} {
	 puts stderr "*** $main_self: can't open container '$arg1'"
	 cb.main.close 1 1
      } else {
	 set db_container_orig $arg1
	 set db_container      $arg1
	 edit.mode.set	       [lindex $argv 1]
	 case $edit_mode in {
	    { read-only copy-on-write write } {}
	    default 			      { set err 1 }
	 }
      }
      if {$err} {
	 puts stderr "*** usage: $main_self ( new | <container> (read-only|copy-on-write|write) )"
	 cb.main.close 1 1
      }

   } else {
      edit.mode.set    write
      set db_container ""
      set str	       [OBST tmpstr "tclOBST Graph Editor"]
      set dir	       [mcall sos_Directory::root]
      set link	       [mcall $dir {[]} $str]
      set dir_cnt      [$dir container]
      
      if {$link != [OBST const NO_OBJECT]} {
	 set db_container [$link make_Cstring]
	 
	 if {![cnt exists $db_container]} {
	    set db_container ""
	    db.cnt.access $dir_cnt
	    
	    $dir insert $str [OBST const NO_OBJECT]
	    $link destroy
      }}
      if {[set db_container_orig $db_container] == ""} {
	 set db_container [cnt create]
	 
	 db.cnt.access $dir_cnt
	 $dir insert $str [mcall sos_String::create $dir_cnt $db_container]
      }
      $str destroy
   }
   set db_container_new 0
   set edit_mode_orig	$edit_mode

   db.cnt.access $db_container 0
   db.graph.init
}


# ---------
proc db.graph.init {} {
#
# Reads the contents of db_container and displays them in the editor canvas.
# db_graph is initialized with the root object of the persistent graph.
# If db_container is empty, a new root object is created.
#
# The persistent representation is admittedly crude and was mainly chosen in
# order to use just the type implementations provided by the OBST library.
# Hence, this sample program can be run by any tclOBST interpreter.
# An important reason for choosing this format was that the persistent
# representation contained in the graph container <cnt> can be readily viewed
# by invoking "obst-gsh <cnt>".
#
# Each graph is stored in a single container and the root object of this
# container is as well the root object of the graph.
# o The root object is of type Directory<sos_Object> and holds as well nodes as
#   edges.
# o Node/edge/graph attributes are simply stored as a Tcl list (converted into
#   a sos_String).
#   o Node attributes are: X, Y coordinate, and node text (in that order).
#   o Edge attributes are: the persistent IDs of the two connected nodes (start
#     and then end node), edge type (in that order) plus width, color, and
#     stipple pattern.
#   o The following graph attributes are stored in the "graph" slot:
#      o Defaults for edge type, width, color, and stipple pattern.
# o There are persistent node/edge IDs: the contents of the directory keys.
#   Node IDs start with 'n', and edge IDs start with 'e'.
#
# A mapping from persistent node/edge IDs to those used by the 'arrows' module
# is provided by means of the db_IDmap array.
#
   global db_container db_graph db_IDmap edit_mode edit_attrs edit_canvas

   if {[info exists db_IDmap]} {
      unset db_IDmap
   }
   if {[cnt occupied $db_container] == 0} {
      db.cnt.access $db_container
      set db_graph [mcall sos_Directory::create $db_container]
   } else {
      set db_graph [cnt root_object $db_container]
   }
   set attrs [concat type $edit_attrs(arrows)]

   agg loop $db_graph {
      set key [[agg current role1] make_Cstring]
      if {[string range $key 0 0] == "n"} {
	 set attr [[agg current role2] make_Cstring]
	 set ID   [cb.edit.node_create [lindex $attr 0] [lindex $attr 1] [lindex $attr 2] 0]
	 set db_IDmap(p,$key) $ID
	 set db_IDmap(a,$ID)  $key

      } elseif {$key == "graph"} {
	 set attr [[agg current role2] make_Cstring]
	 set idx  -1
	 foreach opt $edit_attrs(graph) {
	    arr_option $opt [lindex $attr [incr idx]]
	 }
      }
   }
   foreach opt $attrs {
      set option($opt) [arr_option $opt]
   }
   agg loop $db_graph {
      set key [[agg current role1] make_Cstring]
      if {[string range $key 0 0] == "e"} {
	 set attr [[agg current role2] make_Cstring]
	 set idx  1
	 foreach opt $attrs {
	    arr_option $opt [lindex $attr [incr idx]]
	 }
	 set ID [arr_Edge_create $edit_canvas $db_IDmap(p,[lindex $attr 0]) $db_IDmap(p,[lindex $attr 1])]

	 set db_IDmap(p,$key) $ID
	 set db_IDmap(a,$ID)  $key
      }
   }
   foreach opt $attrs {
      arr_option $opt $option($opt)
   }
   catch {[SymbolicName options::scale] set [arr_option width]}
}


# ---------
proc db.graph.save {} {
#
# Stores global graph attributes in the "graph" slot.
# The slot will be created if it does not yet exist.
#
   global db_container db_graph edit_attrs

   db.cnt.access $db_container

   set key   [OBST tmpstr graph]
   set attrs [$db_graph {[]} $key]
   if {$attrs == [OBST const NO_OBJECT]} {
      set attrs [mcall sos_String::create $db_container]
      $db_graph insert $key $attrs
   }
   $key destroy

   foreach opt $edit_attrs(graph) {
      lappend attrlist [arr_option $opt]
   }
   $attrs assign_Cstring $attrlist
}


# ---------
proc db.graph.clone { cnt} {
#
# Clones the graph rooted at $db_graph to the container $cnt. $cnt must have
# been newly created, must not contain any object, and write access to this
# container must have been granted.
#
   global db_graph db_container

   set new_graph [mcall sos_Directory::create $cnt]
   agg loop $db_graph {
      $new_graph insert [agg current role1]  [mcall sos_Object::clone [agg current role2] $cnt]
   }
}


# ---------
proc db.graph.exit { reset} {
#
# Cleans up the graph container, prepares for program termination, and writes
# out the program result. All open containers are closed. Prior to that they
# are either reset ($reset == 1), or comitted ($reset == 0).
#
   global db_container db_container_new

   db.cnt.savepoint 1 $reset

   if {$db_container_new} {
      puts stdout $db_container
   }
}


# ---------
proc db.file.do { op filename} {
#
# Either writes (op == "write") or reads (op == "read") the graph to/from the
# named file.
# The file representation uses tcl lists as storage format.
# Reading a graph discards the current graph contents and is hence not allowed
# in "read-only" mode.
#
   global main_self db_graph db_container

   if {$op == "write"} {
      if {[catch {open $filename w+} file]} {
	 infobox::displayText $file
	 return
      }
      puts $file "# $filename by $main_self on [exec date]\n"

      agg loop $db_graph {
	 set key   [mcall [agg current role1] make_Cstring]
	 set value [mcall [agg current role2] make_Cstring]

	 puts $file "[list $key $value]"
      }

   } else {
      if {[catch {open $filename r} file]} {
	 infobox::displayText $file
	 return
      }

      db.cnt.access $db_container
      agg loop $db_graph {
	 mcall [agg current role2] destroy
      }
      mcall $db_graph clear

      while {[gets $file line] != -1} {
	 if {[llength $line] == 2} {
	    set key   [OBST tmpstr [lindex $line 0]]
	    set value [mcall sos_String::create $db_container [lindex $line 1]]
	    mcall $db_graph insert $key $value
	 }
      }
      db.graph.init
   }
   close $file
}

# ---------
proc db.graph.node { op nodeID} {
#
# Adapts the database according the operation $op on node $nodeID.
# Operations are:
#  o "create" - node has been just created,
#  o "update" - node attributes may have changed,
#  o "delete" - node is about to be deleted.
#
   global edit_canvas db_container db_IDmap db_graph

   db.cnt.access $db_container

   if {$op == "create"} {
      set key  [OBST tmpstr [db.graph.ID_create $nodeID n]]
      set attr [mcall sos_String::create $db_container]

      $db_graph insert $key $attr
   } else {
      set key  [OBST tmpstr $db_IDmap(a,$nodeID)]
      set attr [$db_graph {[]} $key]
   }
   if {$op == "delete"} {
      $db_graph remove $key
      $attr destroy
   } else {
      set coords [$edit_canvas coords $nodeID]
      set text   [lindex [$edit_canvas itemconfigure $nodeID -text] 4]
      $attr assign_Cstring [list [lindex $coords 0] [lindex $coords 1] [set text]]
   }
}


# ---------
proc db.graph.edge { op node1 node2} {
#
# Adapts the database according the operation $op on the edge between the two
# given nodes. Operations are:
#  o "create" - edge from $node1 to $node2 has been just created,
#  o "delete" - edge is about to be deleted.
#
   global edit_canvas edit_attrs db_container db_IDmap db_graph

   set edgeID [arr_Edge_ID $edit_canvas $node1 $node2]

   db.cnt.access $db_container

   if {$op == "create"} {
      set key  [OBST tmpstr [db.graph.ID_create $edgeID e]]
      set attr [mcall sos_String::create $db_container]

      $db_graph insert $key $attr
   } else {
      set key  [OBST tmpstr $db_IDmap(a,$edgeID)]
      set attr [$db_graph {[]} $key]
   }
   if {$op == "delete"} {
      $db_graph remove $key
      $attr destroy
   } else {
      lappend attrlist $db_IDmap(a,$node1) $db_IDmap(a,$node2) [arr_Edge_type $edit_canvas $node1 $node2]
      foreach opt $edit_attrs(canvas) {
	 lappend attrlist [lindex [$edit_canvas itemconfigure $edgeID $opt] 4]
      }
      $attr assign_Cstring $attrlist
   }
}


# ---------
proc db.graph.ID_create { tmp_ID prefix} {
#
# Yields a new persistent ID for the arrows node/edge ID $tmp_ID and sets up
# db_IDmap accordingly. This new ID will start with $prefix.
#
   global db_IDmap db_graph

   set key [OBST tmpstr "$prefix[set _key [agg card $db_graph]]"]
   while {[$db_graph is_key $key] == "TRUE"} {
      $key assign_Cstring "$prefix[incr _key]"
   }
   $key destroy

   set new_ID		   "$prefix$_key"
   set db_IDmap(a,$tmp_ID) $new_ID
   set db_IDmap(p,$new_ID) $tmp_ID

   return $new_ID
}


# ---------
proc db.cnt.access { cnt {write "1"}} {
#
# Gains write ($write == 1, the default) or read ($write == 0) access to the
# given container.
# If write access is requested for the graph container in edit mode
# "copy-on-write", the graph is cloned to a new container which is then edited
# in edit mode "write".
#
   global db_container db_graph edit_mode main_self

   if {$write} {
      if {$edit_mode == "read-only"} {
	 puts stderr "*** $main_self: INTERNAL ERROR - modify in read-only mode"
	 cb.main.close 1 1

      } elseif {$cnt == $db_container && $edit_mode == "copy-on-write"} {
	 edit.cursor 0

	 db.graph.clone [set new_container [cnt create]]
	 set db_container $new_container
	 set db_graph	  [cnt root_object $db_container]
	 edit.mode.set write

	 edit.cursor 1
      }
   }
   set access_mode [expr {$write ? "WRITING" : "READING" }]

   switch [cnt status $cnt] {
      UNAVAILABLE  { cnt open $access_mode WAITING $cnt }
      WRITEABLE    -
      WRITABLE     { }
      default	   { if {$write} {
	 		cnt access $access_mode WAITING $cnt
      		   }}
   }
}


# ---------
proc db.cnt.savepoint { {close "0"} {reset "0"}} {
#
# Writes a savepoint ($reset == 0) or resets to the previous one ($reset == 1).
# If $close == 1, all open containers are closed afterwards.
#
# Global graph attributes are written if the editor is not operating in the
# read-only mode.
#
# If a savepoint is written for a newly created container, db_container_new is
# set since that container is now known to persist. The edit mode will then
# change to "write" for this new container.
#
# If the program is to be reset, there is a new container, and no savepoint was
# written so far for that container, the editor will recurse to the original
# container and the original edit mode. The new container will then be
# cleared if there is an original container, destroyed otherwise.
#
   global edit_mode edit_mode_orig db_container db_container_orig db_container_new

   if {$reset} {
      cnt reset [cnt open_containers MODIFIABLE]

      if {!$db_container_new && $db_container_orig != $db_container} {
	 if {$close || $db_container_orig != ""} {
	    cnt destroy $db_container
	 
	    set db_container $db_container_orig
	 } else {
	    cnt clear $db_container
	 }
	 edit.mode.set $edit_mode_orig
      }
   } else {
      if {$edit_mode != "read-only"} {
	 db.graph.save

	 if {$db_container != $db_container_orig} {
	    set db_container_new 1
	    edit.mode.set write
	 }
      }
      if {! $close} {
	 cnt commit [cnt open_containers MODIFIABLE]
      }
   }
   if {$close} {
      cnt close [cnt open_containers OPEN]
   }
}


# Internal procedures

# eof
#

