# --------------------------------------------------------------------------
# Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
# All rights reserved.
#
# You can use and distribute this software under the terms of the license
# you should have received along with this software; either version 1.1 of
# the license, or (at your option) any later version.
# For a copy of the license or for additional information about this software,
# write to Xcc Software, Durlacher Allee 53, D-76131 Karlsruhe, Germany;
# Email: obst@xcc-ka.de.
# --------------------------------------------------------------------------
# Module: arrows.tcl
# Tcl version: 7.0 (Tcl/Tk/XF)
# Tk version: 3.3
# XF version: 2.2
#

# module contents
global moduleList
global autoLoadList
set moduleList(arrows.tcl) { arr_doc arr_init arr_coordfunc arr_set_coordfuncs arr_option arr_Set arr_Node_add arr_Node_move arr_Node_remove arr_Node_list arr_Node_neighbours arr_Edge_create arr_Edge_remove arr_Edge_ID arr_Edge_type arr_Edge_coords arr_Drag arr_set_nodeattrs arr_gridded_rect_Tcl arr_continuous_rect_Tcl arr_continuous_oval_Tcl arr_isct arr_create_bidir arr_move_bidir arr_remove_bidir arr_create_unidir arr_move_unidir arr_remove_unidir arr_create_nodir arr_move_nodir arr_remove_nodir}
set autoLoadList(arrows.tcl) {0}

# procedures to show toplevel windows


# User defined procedures


# ---------
 proc arr_doc {} {
#
# arr_:
#  (Internal) array holding the state of the arrows module.
#
#  The following fields hold display parameters:
#    AttachMode
#	Builtin modes are either "continuous" (the default) or "gridded"
#	denoting edges which smoothly circulate the bounding box of a node when
#	moved, or edges which are connected only to a few points along the box,
#	respectively.
#    NodeShape
#	Default node shape. Builtin shapes are "rect" and "oval".
#	It defaults to "rect".
#    EdgeType
#	Default edge type. It defaults to "unidir".
#
#  The following fields record coordinate computation procedures:
#    C,<mode>,<shape>
#	Procedure for the given movement mode and node shape. There will always
#	be entries with indices 'C,<mode>,rect', and 'C,gridded,<shape>',
#	respectively.
#    Modes
#	Set (tcl list) with edge attachment modes seen so far.
#    Shapes
#	Set (tcl list) with node shapes seen so far.
#
#  For each given $nodeID, there are the following fields <f> stored at the
#  indices arr_($nodeID,<f>):
#    Nodes:
#	List of IDs of adjacent nodes which are connected via an edge to this
#	node. 
#    X, Y, W, H:
#	Boundary box of the node in the canvas coordinate system. (X,Y) is the
#	center of the boundary box.
#    S: Node shape.
#    M: Edge attachment mode for this node, "" if global mode is to be used.
#    Mfield:
#	Field of arr_ which holds the edge attachment mode for this node:
#	either '$nodeID,M', or 'AttachMode'.
#
#  For each edge with ID $edgeID, there	are the following fields stored at
#  the indices arr_($edgeID,<f>):
#    S: edge style as suitable for invoking a drawing procedure.
#
#  For the management of sets, there are the following fields arr(<f>,$setID)
#  for a set identified by $setID:
#    S: set represented as a tcl list
#
#  Auxiliary (internal) fields for the drag handlers:
#    dragX, dragY: Last recorded absolute position, in order to determine the
#		   movement relative to that position.
#    dragCalls   : Code that is to be executed after each movement.
#
#  The existence of a node/edge/set may be tested by testing if one of the
#  corresponding array entries exists.
# --------------------------------------------------------------------------
# proc arr_init {}
#
# in/out: void
#
# Initializes the arrow module by setting storing the default values for
# all display options into the arr_ array and by computing coordinate
# functions for all builtin movement modes and builtin node shapes.
# --------------------------------------------------------------------------
# proc arr_coordfunc { mode shape}
#
# in : edge attachment mode and node shape
# out: coordinate function for the given parameters
#
# This function does hide how replacements for missing coordinate functions
# are made.
# --------------------------------------------------------------------------
# proc arr_set_coordfuncs { mode shape}
#
# in : edge attachment mode and node shape, exactly one must be undefined ("")
# out: void
#
# Computes the coordinate functions for the defined argument and all currently
# known values of the other argument (i.e. arr_(Modes), or arr_(Shapes)).
# The defined argument is finally stored in arr_(Modes), or arr_(Shapes)
# if it is not yet contained there.
# --------------------------------------------------------------------------
# proc arr_option { option value}
#
# out: option value
#
# Procedure to set a display parameter to the given value: parameters are the
# edge attachment mode ("attach", see below), the default edge type ("type"),
# edge width ("width"), edge color ("color"), and stipple pattern ("stipple").
#
# If no value is given (the default), the current parameter setting is just
# returned. The value "" (re)sets the named display parameter to its default
# value (which may be "", too).
# The function yields the current/new parameter setting. This will be "" if
# the display parameter is so far undefined and no new value is given.
# --------------------------------------------------------------------------
# --------------------------------------------------------------------------
# proc arr_Set { setID mode value}
#
# out: set value after operation
#
# Maintains a set represented as a tcl list and identified by $setID.
# Valid operation modes are:
# . "insert" to insert element $value into the set,
# . "remove" to remove $value from the set if present, and
# . "all" to access the set in total: if $value is given, it must be a
#   duplicate free tcl list that will be taken as the new set value.
#   Otherwise, the current set is just returned.
# "insert" and "all" (with value parameter) may be used to create a new set.
# --------------------------------------------------------------------------
# --------------------------------------------------------------------------
# proc arr_Node_add { window nodeID mode shape mode x y width height}
#
# out: void
#
# Initializes a Tk item as a node. The item is identified by nodeID in the Tk
# window with pathname window. If no such item exists, nothing will happen.
# Mode must be either "new" or "refresh":
#  - "new" creates a new node for an item if nodeID is so far unknown.
#  - "refresh" can be used to re-initialize the parameters of an existing
#    node, if the coordinates, height or width of the item refering to this
#    node have changed.
# Shape defines the internal node shape, i.e. where edges start/end. It
# defaults to the current default node shape.
# Mode defines the edge attachment mode for this node. It default to the
# current global edge attachment mode.
# X, y set the center of the rectangle region, surrounding the item, where the 
# edges are to start and end. Height and width set the size of this region.
# If all these coordinates are undefined, a default bounding box is computed.
# --------------------------------------------------------------------------
# proc arr_Node_move { window nodeID goX goY}
#
# out: void
#
# Moves the edges of the Tk item with tag nodeID in the Tk window with
# pathname window relative to the former position by (goX,goY).
# An item which is moved by "$window move $nodeID $goX $goY" must afterwards
# by moved by arr_Node_add with the same parameters in order to move the edges
# of the item correctly.
# If the node is not moved (i.e. goX == 0, goY == 0), the edges will just be
# refreshed. This may be useful e.g. after changing the node size.
# --------------------------------------------------------------------------
# proc arr_Node_remove { window nodeID}
#
# out: void
#
# Removes the node corresponding to the Tk item with tag nodeID in the Tk
# window with pathname window.
# All edges starting or ending at this item are deleted.
# --------------------------------------------------------------------------
# arr_Node_list { window}
#
# out: List of IDs of all nodes in the given Tk window. The resulting list
#      might be empty.
# --------------------------------------------------------------------------
# arr_Node_neighbours { window nodeID}
#
# out: List of IDs of all nodes directly connected to $nodeID,
#      or an empty list if either $nodeID is not initialized as a node or if
#      there are no such edges.
# --------------------------------------------------------------------------
# --------------------------------------------------------------------------
# arr_Edge_create { window startID destID type}
#
# out: ID of newly created edge, or "" if no edge was created
#
# Creates an edge between two Tk items in the Tk window with pathname window
# and the tags startID and destID. Both items must have been initialized as
# nodes with the command arr_Node_add.
# Nothing will happen if there exists already an edge between the two items or
# if startID == destID, or if any of the nodes is no proper node.
# Type sets the type of the edge. If omitted, the default edge type is used.
# The three standard types are: lines with no direction (mode nodir),
# unidirectional lines (mode unidir) and bidirectional lines (mode bidir).
# Further edge types may be used provided their implementation is present.
# --------------------------------------------------------------------------
# proc arr_Edge_remove { window ID_1 ID_2}
#
# out: void
#
# Removes an existing edge between two Tk items in the Tk window with 
# pathname window and the tags ID_1 and ID_2.
# Nothing will happen, if there exists no edge between the two items or if
# at least one of the items is not initialized as a node.
# --------------------------------------------------------------------------
# proc arr_Edge_ID { window ID_1 ID_2}
#
# out: ID of the edge between the two given Tk items with tags ID_1 and ID_2,
#      or "" if there is no such edge.
# --------------------------------------------------------------------------
# proc arr_Edge_type { window ID_1 ID_2 type}
#
# out: "" if type is undefined or "" if there is no such edge and the current
#      edge type otherwise.
#
# Queries or modifies the type of the edge connecting the two given nodes.
# --------------------------------------------------------------------------
# --------------------------------------------------------------------------
# proc arr_Drag { mode window x y ID}
#
# out: void
#
# Standard drag handler which does move either the single node denoted by
# $ID, or all nodes contained in set $ID, respectively.
# Aany edges connected to these nodes are moved too. All moved nodes must
# actually exist in the given window.
# Mode indicates if dragging is to be initialized ("init"), if the nodes are
# in motion ("move"), or if dragging is finished ("done"), respectively.
# The ID argument is ignored in modes "move" and "done", and the x,y arguments
# are ignored in mode "done". The "done" call may be omitted in which case
# the cleanup is performed when initializing the next dragging operation for
# that window.
# Inner edges are NOT moved by the 'move' procedure of the respective edge
# type.
# No changes must be made to node/edge attributes between initialization and
# the end of dragging.
# If mouse button 1 is to be used for dragging, event bindings might be set
# up as follows for any of the dragged nodes:
#   $window bind $ID <ButtonPress-1>   "arr_Drag init %W %x %y $ID"
#   $window bind $ID <B1-Motion>       "arr_Drag move %W %x %y"
#   $window bind $ID <ButtonRelease-1> "arr_Drag done %W"
#
# (The procedure contains an inlined version of arr_Node_move.)
# --------------------------------------------------------------------------
# --------------------------------------------------------------------------
# proc arr_Edge_coords { startID destID}
#
# Returns a list of five elements, or an empty list if at least one of the
# given items is not initialized as a node:
#
# [0] = The positon of destID relatively to the position of startID.
#	This position can be N,S,E,W in move mode "gridded" or NE,NW,SW,SE in
#	mode "continuous".
#	It can be used e.g. to make the look of an edge depend on the relative
#	position of two items.
# [1] = The x-coordinate of the begin of an edge starting at the bounding box
#	of startID.
# [2] = The y-coordinate of the begin of an edge starting at the bounding box
#	of startID.
# [3] = The x-coordinate of the end of an edge ending at the bounding box of
#	destID.
# [4] = The y-coordinate of the end of an edge ending at the bounding box of
#	destID.
# --------------------------------------------------------------------------
# proc arr_set_nodeattrs { window nodeID shape mode x y width height}
#
# out: void
#
# Internal procedure which registers the coordinates for the given node.
# If all coordinates are undefined (i.e. empty strings), a default boundary
# box is computed.
# It does also set the node shape and edge attachment mode for this node.
# Defaults are the current default node shape, and the current (at the time of
# usage) default attachment mode.
# --------------------------------------------------------------------------
# proc arr_gridded_rect_Tcl { sx sy widthS heightS dx dy widthD heightD}
#
# For each item, there are four positions, where an edge can start from:
# N, S, E, and W.
# arr_gridded_rect_Tcl selects the position such, that the edge
# does not enter the boundary box of one of the connected items.
# --------------------------------------------------------------------------
# proc arr_continuous_rect_Tcl {sx sy widthS heightS dx dy widthD heightD}
#
# Tcl version of the C function arr_continuous_rect_C.
# --------------------------------------------------------------------------
# arr_continuous_oval_Tcl {sx sy widthS heightS dx dy widthD heightD}
#
# Tcl version of the C function arr_continuous_oval_C.
# --------------------------------------------------------------------------
# proc arr_isct { edgeX edgeY vectorX vectorY centerX centerY}
#
# Auxiliary procedure of arr_Edge_coords_Tcl that computes the intersection
# of two vectors (or lines).
# Result is the factor that is used to get to the point of the intersection by
# multiplying this factor and one of the vectors.
# --------------------------------------------------------------------------
# --------------------------------------------------------------------------
# proc arr_create_bidir { window startID destID edgeID}
#
# out: void
#
# Creates a bidirectional line which is tagged with $edgeID.
# --------------------------------------------------------------------------
# proc arr_move_bidir { window startID destID edgeID}
#
# out: void
#
# Moves bidirectional lines which are tagged with $edgeID.
# --------------------------------------------------------------------------
# proc arr_remove_bidir { window edgeID}
#
# out: void
#
# Removes lines which are tagged with $edgeID.
# --------------------------------------------------------------------------
# proc arr_create_unidir { window startID destID edgeID}
#
# out: void
#
# Creates an unidirectional line which is tagged with $edgeID.
# --------------------------------------------------------------------------
# proc arr_move_unidir { window startID destID edgeID}
#
# out: void
#
# Moves unidirectional lines which are tagged with $edgeID.
# --------------------------------------------------------------------------
# proc arr_remove_unidir { window edgeID}
#
# out: void
#
# Removes lines which are tagged with $edgeID.
# --------------------------------------------------------------------------
# proc arr_create_nodir { window startID destID edgeID}
#
# out: void
#
# Creates an undirected line which is tagged with $edgeID of the current
# edge color and edge width. If any of the latter display parameters is
# undefined, Tk's defaults are used instead.
# --------------------------------------------------------------------------
# proc arr_move_nodir { window startID destID edgeID}
#
# out: void
#
# Moves lines which are tagged with $edgeID.
# --------------------------------------------------------------------------
# proc arr_remove_nodir { window edgeID}
#
# out: void
#
# Removes lines which are tagged with $edgeID.
# --------------------------------------------------------------------------
}


# ---------
proc arr_init { } {
 global arr_

 set arr_(Modes)  {}
 set arr_(Shapes) {}

 arr_set_coordfuncs gridded    ""
 arr_set_coordfuncs continuous ""
 arr_set_coordfuncs ""	       rect

 arr_option color   ""
 arr_option attach  ""
 arr_option shape   ""
 arr_option stipple ""
 arr_option type    ""
 arr_option width   ""
}

# ---------
 proc arr_coordfunc { mode shape} {

 foreach func [list arr_${mode}_${shape}_C arr_${mode}_${shape}_Tcl \
		    arr_${mode}_rect_C	   arr_${mode}_rect_Tcl \
		    arr_continous_rect_C] {
    if {[info commands $func] != ""} {
       return $func
    }
 }
 return arr_continous_rect_Tcl
}

# ---------
 proc arr_set_coordfuncs { mode shape} {
 global arr_

 if {$mode != ""} {
    foreach shape $arr_(Shapes) {
       set arr_(C,$mode,$shape) [arr_coordfunc $mode $shape]
    }
    if {[lsearch -exact $arr_(Modes) $mode] == -1} {
       lappend arr_(Modes) $mode
    }
 } else {
    foreach mode $arr_(Modes) {
       set arr_(C,$mode,$shape) [arr_coordfunc $mode $shape]
    }
    if {[lsearch -exact $arr_(Shapes) $shape] == -1} {
       lappend arr_(Shapes) $shape
    }
 }
}

# ---------
proc arr_Set { setID {mode all} {value "___"}} {
 global arr_

 if {$value == "___" && $mode != "all"} {
    error "arr_Set: missing 'elem' parameter"
 }
 switch $mode \
    insert {
       if {[catch "set arr_(S,$setID)" set]} {
	 set	 arr_(S,$setID) $value
       } elseif {[lsearch $set $value] == -1} {
	 lappend arr_(S,$setID) $value
       }
    }\
    remove {
       if {[catch "set arr_(S,$setID)" set]} {
	  error "arr_Set: '$setID' is not a valid set ID"
       } elseif {[set pos [lsearch $set $value]] != -1} {
	  set arr_(S,$setID) [lreplace $set $pos $pos]
       }
    }\
    all {
       if {$value != "___"} {
	  set arr_(S,$setID) $value
       } elseif {![info exists arr_(S,$setID)]} {
          error "arr_Set: '$setID' is not a valid set ID"
       }
    }\
    default {
       error "arr_Set: unknown mode '$mode' - must be insert, remove or all"
    }
 return $arr_(S,$setID)
}


# ---------
proc arr_option { option {value _%_}} {
 global arr_

 switch $option \
    color   { set field EdgeColor   
	      set dflt	""
	    }\
    attach  { set field AttachMode
	      set dflt	continuous
	    }\
    shape   { set field NodeShape
	      set dflt	rect
	    }\
    stipple { set field EdgeStipple 
	      set dflt	""
	    }\
    type    { set field EdgeType    
	      set dflt	unidir
	    }\
    width   { set field EdgeWidth   
	      set dflt	""
	    }\
    default { error "arr_option: unknown display parameter '$option'" }

 switch $value \
    "_%_" {
       if {[info exists arr_($field)]} {
	  return $arr_($field)
       } else {
	  return ""
       }
    }\
    "" {
       set value $dflt
    }

 if {$option == "attach"} {
    if {![info exists arr_(C,$value,rect)]} {
       arr_set_coordfuncs $value ""
    }
 }
 return [set arr_($field) $value]
}


# ---------
proc arr_Node_add { window nodeID {mode new} {shape ""} {attach ""}
				  {x ""} {y ""} {width ""} {height ""}} {
 global arr_
 
 switch $mode \
    new	    { if {![info exists arr_($nodeID,X)]} {
		 set arr_($nodeID,Nodes) {}
		 arr_set_nodeattrs $window $nodeID \
					   $shape $attach $x $y $width $height
	    }}\
    refresh { if {[info exists arr_($nodeID,X)]} {
		arr_set_nodeattrs $window $nodeID \
					   $shape $attach $x $y $width $height
	    }}\
    default { error "arr_Node_add: unkown mode '$mode'" }
}


# ---------
proc arr_Node_move { window nodeID {goX 0} {goY 0}} {
 global arr_
	
 if {[info exists arr_($nodeID,X)]} {
    incr arr_($nodeID,X) $goX
    incr arr_($nodeID,Y) $goY

    foreach otherID $arr_($nodeID,Nodes) {
       set edge "${otherID}_$nodeID"

       if {[info exists arr_($edge,S)]} {
	  "arr_move_$arr_($edge,S)" $window $otherID $nodeID $edge
       } elseif {[info exists arr_([set edge "${nodeID}_$otherID"],S)]} {
	  "arr_move_$arr_($edge,S)" $window $nodeID $otherID $edge
       }
 }}
}


# ---------
proc arr_Node_remove { window nodeID} {
 global arr_

 if {[info exists arr_($nodeID,X)]} {
    foreach otherID $arr_($nodeID,Nodes) {
       set idx			[lsearch  $arr_($otherID,Nodes) $nodeID]
       set arr_($otherID,Nodes) [lreplace $arr_($otherID,Nodes) $idx $idx]

       set edge1 "${otherID}_$nodeID"
       set edge2 "${nodeID}_$otherID"

       if {[info exists arr_($edge1,S)]} {
	  "arr_remove_$arr_($edge1,S)" $window $edge1
	  unset arr_($edge1,S)

       } elseif {[info exists arr_($edge2,S)]} {
	  "arr_remove_$arr_($edge2,S)" $window $edge2
	  unset arr_($edge2,S)
       }
    }
    unset arr_($nodeID,Nodes)\
	  arr_($nodeID,X) arr_($nodeID,Y)\
	  arr_($nodeID,W) arr_($nodeID,H)\
	  arr_($nodeID,S) arr_($nodeID,M)\
	  arr_($nodeID,Mfield)
 }
}


# ---------
proc arr_Node_list { window} {
 global arr_

 set resultlist {}

 foreach item [$window find all] {
    if {[info exists arr_($item,X)]} {
       lappend resultlist $item
    }
    foreach tag [$window gettags $item] {
      if {[info exists arr_($tag,X)] && [lsearch $resultlist $tag] == -1} {
	 lappend resultlist $tag
      }}}
 return $resultlist
}


# ---------
proc arr_Node_neighbours { window nodeID} {
 global arr_

 if {[info exists arr_($nodeID,Nodes)]} {
    return $arr_($nodeID,Nodes)
 }
 return {}
}


# ---------
proc arr_Edge_create { window startID destID {type ""}} {
 global arr_
       
 set edgeID "${startID}_$destID" 

 if {	[info exists arr_($startID,X)]
     && [info exists arr_($destID,X)]
     && $startID != $destID
     && ![info exists arr_($edgeID,S)]
     && ![info exists arr_(${destID}_$startID,S)]} {

    if {$type == ""} {
       set type $arr_(EdgeType)
    }
    lappend arr_($startID,Nodes) $destID
    lappend arr_($destID,Nodes)	 $startID
    set	    arr_($edgeID,S)	   $type

    "arr_create_$type" $window $startID $destID $edgeID
    return $edgeID
 }
 return ""
}


# ---------
proc arr_Edge_remove { window ID_1 ID_2} {
 global arr_

 if {[info exists arr_($ID_1,X)] && [info exists arr_($ID_2,X)]} {
    set edgeID "${ID_1}_$ID_2"

    if {![info exists arr_($edgeID,S)]} {
       set edgeID "${ID_2}_$ID_1"

       if {![info exists arr_($edgeID,S)]} return
    }
    "arr_remove_$arr_($edgeID,S)" $window $edgeID

    set idx		  [lsearch  $arr_($ID_1,Nodes) $ID_2]
    set arr_($ID_1,Nodes) [lreplace $arr_($ID_1,Nodes) $idx $idx]

    set idx		    [lsearch  $arr_($ID_2,Nodes) $ID_1]
    set arr_($ID_2,Nodes) [lreplace $arr_($ID_2,Nodes) $idx $idx]

    unset arr_($edgeID,S)
 }
}


# ---------
proc arr_Edge_ID { window ID_1 ID_2} {
 global arr_

 if {[info exists arr_($ID_1,X)] && [info exists arr_($ID_2,X)]} {
    set edgeID "${ID_1}_$ID_2"
    if {[info exists arr_($edgeID,S)]} { return $edgeID }

    set edgeID "${ID_2}_$ID_1"
    if {[info exists arr_($edgeID,S)]} { return $edgeID }
 }
 return ""
}


# ---------
proc arr_Edge_type { window ID_1 ID_2 {type ""}} {
 global arr_

 if {[info exists arr_($ID_1,X)] && [info exists arr_($ID_2,X)]} {
    set edgeID "${ID_1}_$ID_2"
    if {![info exists arr_([set edgeID "${ID_1}_$ID_2"],S)]} {
       if {![info exists arr_([set edgeID "${ID_2}_$ID_1"],S)]} { return "" }

       set _id	  $ID_1
       set ID_1	  $ID_2
       set ID_2	  $_id
    }
    if {$type == ""} {
       return $arr_($edgeID,S)

    } elseif {$type != $arr_($edgeID,S)} {
       "arr_remove_$arr_($edgeID,S)" $window $edgeID
       "arr_create_[set arr_($edgeID,S) $type]" $window $ID_1 $ID_2 $edgeID
 }}
 return ""
}


# ---------
proc arr_Edge_coords { startID destID} {
 global arr_

 if {[info exists arr_($startID,X)] && [info exists arr_($destID,X)]} {
    set argS "$arr_($startID,X) $arr_($startID,Y)\
	      $arr_($startID,W) $arr_($startID,H)"
    set argD "$arr_($destID,X)	$arr_($destID,Y)\
	      $arr_($destID,W) $arr_($destID,H)"

    return "[eval "$arr_(C,$arr_($arr_($startID,Mfield)),$arr_($startID,S)) $argS $argD"] [lrange\
		   [eval "$arr_(C,$arr_($arr_($destID,Mfield)),$arr_($destID,S)) $argD $argS"]\
		   1 2]"
 }
 return {}
}


# ---------
proc arr_Drag { mode window {x 0} {y 0} {ID ""}} {
 global arr_

 set x [$window canvasx $x]
 set y [$window canvasy $y]

 switch $mode \
    init {
       # perform 'done'
       $window dtag arr_drag

       # compute "inner" and "outer" edges, define common tag for inner items
       if {[info exists arr_($ID,X)]} {
	  set set $ID
       } elseif {[catch "set arr_(S,$ID)" set]} {
	  error "arr_Drag: '$ID' is neither a valid node ID nor a valid set ID"
       }
       set arr_(dragCalls) ""
       set dragInner	   {}
       set edgeMoves	   ""

       foreach nodeID $set {
	  foreach other $arr_($nodeID,Nodes) {
	     if {[info exists arr_(${nodeID}_$other,S)]} {
		set edgeID "${nodeID}_$other"
		set nodes  "$nodeID $other"
	     } else {
		set edgeID "${other}_$nodeID"
		set nodes  "$other $nodeID"
	     }
	     if {[lsearch $set $other] == -1} {
		append edgeMoves \
		       ";arr_move_$arr_($edgeID,S) $window $nodes $edgeID"
	     } elseif {[lsearch $dragInner $edgeID] == -1} {
		lappend dragInner $edgeID
		$window addtag arr_drag withtag $edgeID
	     }
	  }
	  append arr_(dragCalls) \
		 ";incr arr_($nodeID,X) \$dx;incr arr_($nodeID,Y) \$dy"

	  $window addtag arr_drag withtag $nodeID
       }
       append arr_(dragCalls) $edgeMoves
    }\
    move {
       set dx [expr $x-$arr_(dragX)]
       set dy [expr $y-$arr_(dragY)]

       $window move arr_drag $dx $dy
       eval $arr_(dragCalls)
    }\
    done {
       # remove tags from selected items
       $window dtag arr_drag
       set arr_(dragCalls) ""
    }\
    default {
       error \
	 "arr_Drag: unknown mode '$mode' - must be init, move or done"
    }
 set arr_(dragX) $x
 set arr_(dragY) $y
}


# ---------
 proc arr_set_nodeattrs { window nodeID shape mode x y width height} {
 global arr_

 if {"$height$width$x$y" == ""} {
    set coords [$window bbox $nodeID]
    
    set x1 [lindex $coords 0]
    set y1 [lindex $coords 1]
    set x2 [lindex $coords 2]
    set y2 [lindex $coords 3]

    set x      [expr {$x1 + ($x2-$x1)/2}]
    set width  [expr {$x2-$x1}]
    set y      [expr {$y1 + ($y2-$y1)/2}]
    set height [expr {$y2-$y1}]
 }
 set arr_($nodeID,X)	  $x
 set arr_($nodeID,Y)	  $y
 set arr_($nodeID,H)	  $height
 set arr_($nodeID,W)	  $width
 set arr_($nodeID,M)	  $mode
 set arr_($nodeID,Mfield) [expr {$mode == "" ? "AttachMode" : "$nodeID,M"}]

 if {$shape == ""} {
    set shape $arr_(NodeShape)
 }
 set arr_($nodeID,S) $shape

 if {![info exists arr_(C,continuous,$shape)]} {
    arr_set_coordfuncs "" $shape
 }
}


# ---------
 proc arr_gridded_rect_Tcl { sx sy widthS heightS dx dy widthD heightD} {

 if {[set start_LowY [expr {$sy + ($heightS/2)}]]
		  <= [expr {$dy - ($heightD/2)}]} { # dest_HighY
    set startY	 $start_LowY		
    set startX	 $sx	   
    set pos	 S

 } elseif {[set start_HighY [expr {$sy - ($heightS/2)}]]
			 >= [expr {$dy + ($heightD/2)}]} { # dest_LowY
    set startY	 $start_HighY 
    set startX	 $sx
    set pos	 N

 } elseif {[set start_RightX [expr {$sx + ($widthS/2)}]]
			   < [expr {$dx - ($widthD/2)}]} { # dest_LeftX
    set startY	 $sy 
    set startX	 $start_RightX
    set pos	 E

 } else {
    set startY	 $sy 
    set startX	 [expr {$sx - ($widthS/2)}]
    set pos	 W
 }

 return [list $pos $startX $startY]
}


# ---------
 proc arr_continuous_rect_Tcl {sx sy widthS heightS dx dy widthD heightD} {

 set deltaY [expr {$sy - $dy}]
 set deltaX [expr {$sx - $dx}]

 if {$deltaX == 0} {
    set edgestartX $sx

    if {$deltaY >= 0} {
       set edgestartY [expr {$sy - ($heightS/2)}]
       set edgepos    NE
    } else {
       set edgestartY [expr {$sy + ($heightS/2)}]
       set edgepos    SE
    }

 } elseif {$deltaY == 0} {
    set edgestartY $sy

    if {$deltaX >= 0} {
       set edgestartX [expr {$sx - ($widthS/2)}]
       set edgepos    SW 
    } else {
       set edgestartX [expr {$sx + ($widthS/2)}]
       set edgepos    NW
    }

 } elseif {$deltaX < 0} {
    if {$deltaY >= 0} {
       #-- 1. quadrant :
       set edgeX      [expr {$sx + ($widthS/2)}]
       set edgeY      [expr {$sy - ($heightS/2)}]
       set a	      [arr_isct $edgeX $edgeY $deltaX $deltaY $sx $sy]
       set edgestartX [expr {round($sx - ($a * $deltaX))}]
       set edgestartY [expr {round($sy - ($a * $deltaY))}]
       
       set edgepos   NE

    } else {
       #-- 4. quadrant :
       set edgeX      [expr {$sx + ($widthS/2)}]
       set edgeY      [expr {$sy + ($heightS/2)}]
       set a	      [arr_isct $edgeX $edgeY $deltaX $deltaY $sx $sy]
       set edgestartX [expr {round($sx - ($a * $deltaX))}]
       set edgestartY [expr {round($sy - ($a * $deltaY))}]
       
       set edgepos   SE
    }
 } elseif {$deltaY < 0} {
    #-- 3. quadrant :
    set edgeX	   [expr {$sx - ($widthS/2)}]
    set edgeY	   [expr {$sy + ($heightS/2)}]
    set a	   [arr_isct $edgeX $edgeY $deltaX $deltaY $sx $sy]
    set edgestartX [expr {round($sx - ($a * $deltaX))}]
    set edgestartY [expr {round($sy - ($a * $deltaY))}]

    set edgepos	  SW

 } else {
    #-- 2. quadrant :
    set edgeX	   [expr {$sx - ($widthS/2)}]
    set edgeY	   [expr {$sy - ($heightS/2)}]
    set a	   [arr_isct $edgeX $edgeY $deltaX $deltaY $sx $sy]
    set edgestartX [expr {round($sx - ($a * $deltaX))}]
    set edgestartY [expr {round($sy - ($a * $deltaY))}]

    set edgepos	  NW
 }
 return [list $edgepos $edgestartX $edgestartY]
}


# ---------
 proc arr_continuous_oval_Tcl {sx sy widthS heightS dx dy widthD heightD} {

 set deltaX [expr {$dx - $sx}]
 set deltaY [expr {$dy - $sy}]
 set f	    [expr {2 * sqrt($deltaX*$deltaX + $deltaY*$deltaY)}]

 set edgeX  [expr {$sx + round($widthS	* $deltaX / $f)}]
 set edgeY  [expr {$sy + round($heightS * $deltaY / $f)}]

 if {$deltaX >= 0} {
    set edgepos [expr {($deltaY <= 0) ? "NE" : "SE"}]
 } else {
    set edgepos [expr {($deltaY <= 0) ? "NW" : "SW"}]
 }

 return [list $edgepos $edgeX $edgeY]
}


# ---------
 proc arr_isct { edgeX edgeY vectorX vectorY centerX centerY} {

 set a1 [expr {abs(double(($edgeX - $centerX)) / $vectorX)}]
 set a2 [expr {abs(double(($edgeY - $centerY)) / $vectorY)}]

 if {$a1 > $a2} {return $a2}
 return $a1
}


# ---------
 proc arr_create_bidir { window startID destID edgeID} {

 arr_create_nodir $window $startID $destID $edgeID
 $window itemconfigure $edgeID -arrow both
}


# ---------
 proc arr_move_bidir { window startID destID edgeID} {

 eval "$window coords $edgeID [lrange [arr_Edge_coords $startID $destID] 1 4]"
}


# ---------
 proc arr_remove_bidir { window edgeID} {

 $window delete $edgeID
}


# ---------
 proc arr_create_unidir { window startID destID edgeID} {

 arr_create_nodir $window $startID $destID $edgeID
 $window itemconfigure $edgeID -arrow last
}


# ---------
 proc arr_move_unidir { window startID destID edgeID} {

 eval "$window coords $edgeID [lrange [arr_Edge_coords $startID $destID] 1 4]"
}


# ---------
 proc arr_remove_unidir { window edgeID} {

 $window delete $edgeID
}


# ---------
 proc arr_create_nodir { window startID destID edgeID} {

 global arr_

 set opts ""
 if {$arr_(EdgeColor) != ""} {
    append opts " -fill $arr_(EdgeColor)"
 }
 if {$arr_(EdgeWidth) != ""} {
    append opts " -width $arr_(EdgeWidth)"
 }
 if {$arr_(EdgeStipple) != ""} {
    append opts " -stipple $arr_(EdgeStipple)"
 }

 eval "$window create line [lrange [arr_Edge_coords $startID $destID] 1 4]\
			   -tags $edgeID -arrow none $opts"
}


# ---------
 proc arr_move_nodir { window startID destID edgeID} {

 eval "$window coords $edgeID [lrange [arr_Edge_coords $startID $destID] 1 4]"
}


# ---------
 proc arr_remove_nodir { window edgeID} {

 $window delete $edgeID
}

arr_init


# Internal procedures

# eof
#
