##############################################################################
#
# tcldbm.tcl: A simple wrapper for the gdbm database package.
#             It allows to use tcl arrays as a database, without
#             to write any line for reading, writing, opening
#             and closing the database.
#
#
# VERSION: 0.1
# 
# NOTE: to use this package, you will need either a tclsh which supports
#       gdbm commands or a Tcl version >= 7.5 and a dynamic library 
#       libtclgdbm-0.1.so.
#       It should also work with ndbm, though this is not tested
#
# You can get a package of "tcl+gdbm" from:
#        ftp://ftp.neosoft.com/pub/tcl/alcatel/extensions/tcl+gdbm-0.1.tar.gz
#
# and a package of "tcl+ndbm" from:
#        ftp://ftp.neosoft.com/pub/tcl/alcatel/extensions/tcl+ndbm-0.1.tar.gz
#
#
#
##############################################################################
#
# Copyright (C) 1997  Mario Weilguni <mweilguni@sime.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
#
##############################################################################
#
# KNOWN BUGS AND LIMITATIONS:
#     - all array indices from an attached array must remain in memory. This
#       is because the "array" commands (names, anymore...) do not support
#       tracing. If your database has a lot of entries, all keys to those
#       entries will remain in memory.
#       Not really a bug, but a nasty limitation :-(
#
##############################################################################


# a few constants needed 
set DBM(VERSION) "0.1"
set DBM(DBM_VERSION) "0.1"
set DBM(SHARED_EXTENSION) "so"

# if you want to use ndbm, change the next line!!!
set DBM(DB) "gdbm"


##############################################################################
#
# if the dbm_commands do not exist, try to load them as a shared
# library
#
##############################################################################
proc dbm_init { } {
    global DBM tcl_version tcl_pkgPath

    if {[llength [info commands $DBM(DB)]] == 0} {
	# if we use Tcl 7.5 or higher, we can try to load it as shared library
	if { $tcl_version >= 7.5 } {
	    set searchpath [list [info library] \
				"." \
				"/usr/lib" \
				"/usr/local/lib" \
				"/usr/lib/tcl+$DBM(DB)" \
				"/usr/local/lib/tcl+$DBM(DB)" \
				"/usr/lib/tcl+$DBM(DB)-$DBM(DBM_VERSION)" \
				"/usr/local/lib/tcl+$DBM(DB)-$DBM(DBM_VERSION)"]
	    if {[info exists tcl_pkgPath]} {
		lappend searchpath $tcl_pkgPath
	    }
	    foreach i $searchpath {
		set name [file join $i libtcl$DBM(DB)-$DBM(DBM_VERSION).$DBM(SHARED_EXTENSION)]
		if {[file exists $name]} {
		    set ret [catch { load $name $DBM(DB)}]
		    if {$ret == 0} {
			return 1
		    }
		}
	    }
	    puts stderr "dbm_init: failed to load shared library!"
	    return 0
	} else {
	    puts stderr "dbm_init: $DBM(DB) command not implemented!"
	    return 0
	}
    }
}


##############################################################################
#
# removes a cache entry for an array variable
#
##############################################################################
proc dbm_remove_cache_entry { varname } {
    global DBM $varname

    if {[llength $DBM(CACHE:LIST:$varname)] > 0} {
	set i [lindex $DBM(CACHE:LIST:$varname) 0]
	array set $varname [list $i ""]
	set DBM(CACHE:LIST:$varname) [lrange $DBM(CACHE:LIST:$varname) 1 end]
    }
}    


##############################################################################
#
# Trace function for read access to the array
#
##############################################################################
proc dbm_trace_read { name varname index op } {
    global DBM $varname

    # if the whole array is specified then return
    if {[string length $index] == 0} {
	return
    }

    # if the data is already in the array
    set i [lsearch -exact $DBM(CACHE:LIST:$varname) $index]
    if {$i != -1} {
	# move element to end the front of the cache list
	set DBM(CACHE:LIST:$varname) \
	    [concat \
		 [lrange $DBM(CACHE:LIST:$varname) 0 [expr $i - 1]] \
		 [lrange $DBM(CACHE:LIST:$varname) [expr $i + 1] end] \
		 [lindex $DBM(CACHE:LIST:$varname) $i]]
	return
    }	
    
    # check if we have to remove some entries from the cache
    if {[llength $DBM(CACHE:LIST:$varname)] >= $DBM(CACHE:MAXCACHE:$varname)} {
	for { set n 0 } { $n < [expr ( $DBM(CACHE:MAXCACHE:$varname) + 3 ) / 4]} { incr n } {
	    dbm_remove_cache_entry $varname
	}
    }

    # now load the value from the database
    if {[$DBM(DB) exists $name $index] != 0} {
	array set $varname [list $index [$DBM(DB) fetch $name $index]]
	lappend DBM(CACHE:LIST:$varname) $index
    }
}


##############################################################################
#
# Trace function for write access to the array
#
##############################################################################
proc dbm_trace_write { name varname index op } {
    global DBM $varname

    # if the whole array is specified then return
    if {[string length $index] == 0} {
	return
    }    

    set result [catch { $DBM(DB) store $name $index \
			    [lindex [array get $varname $index] 1] } err]
    if {$result != 0} {
	puts stderr "dbm_trace_write: $err!"
    }
    array set $varname [list $index ""]
}


##############################################################################
#
# Trace function for destroying the array or parts of the array
#
##############################################################################
proc dbm_trace_unset { name varname index op } {
    global DBM $varname


    # if the whole array is specified then close the database
    # and clean up
    if {[string length $index] == 0} {
	catch { $DBM(DB) close $name }
	catch { 
	    foreach i [array names DBM *$varname] {
		unset DBM($i)
	    }
	}
	# paranoia settings
	catch {
	    foreach i [trace info $varname] {
		trace vdelete $varname [lindex $i 0] [lindex $i 1]
	    }
	}	    
    } else {
	# if only a index of the array is specified, remove this index
	# from the database and the internal cache
	set result [catch { $DBM(DB) delete $name $index } err]
	if {$result != 0} {
	    puts stderr "dbm_trace_unset: $err!"
	}
	set i [lsearch -exact -- $DBM(CACHE:LIST:$varname)]
	if { $i != -1 } {
	    set DBM(CACHE:LIST:$varname) \
		[concat \
		     [lrange $DBM(CACHE:LIST:$varname) 0 [expr $i - 1]] \
		     [lrange $DBM(CACHE:LIST:$varname) [expr $i + 1] end]]
	}
    }
}


##############################################################################
#
# Changes the number of cache entries for a given array
#
##############################################################################
proc dbm_setcache { varname cache_entries } {
    global DBM

    if {[info exists DBM(CACHE:LIST:$varname)]} {
	# remove entries if necessary
	while { [llength $DBM(CACHE:LIST:$varname)] > $cache_entries } {
	    dbm_remove_cache_entry $varname
	}
	
	set DBM(CACHE:MAXCACHE:$varname) $cache_entries
    }	    
}    



##############################################################################
#
# Attaches a xDBM-file to an array variable. All array contents are destroyed.
# All keys are loaded, but not data is loaded.
# 
# mode should be at least "r", but may be
# "r" - readable (must be readable)
# "w" - writeable
# "c" - create if not existant
# "n" - create regardless if existant
#
# varname is the name of the array that will be attached.
#         varname MUST be a global variable
#
# cache specifies the maximum number of entries which will
#       be cached (this is necessary to prevent the array
#       from growing to large).
#       For effiancy, cache should be at least 10 (default)
#
# returns 1 on success, 0 on error
#
##############################################################################
proc dbm_attach { filename mode varname } {
    global DBM $varname

    set db $DBM(DB)

    catch { destroy $varname }

    if {[string first "r" $mode] == -1} {
	puts stderr "dbm_attach: mode MUST be readable!"
	return 0
    }
    
    if {[dbm_init] == 0} {
	return 0
    }
    
    set result [catch {$db open $filename $mode} err]
    if {$result != 0} {
	puts stderr "dbm_open: $err!"
	return 0
    } else {
	set name $err
    }
    
    # create all array indices from the database
    set keys [$db list $name]
    foreach i $keys {
	array set $varname [list $i ""]	
    }	

    # create a cache for all keys
    set DBM(CACHE:NAME:$varname) $name
    set DBM(CACHE:LIST:$varname) [list]
    dbm_setcache $varname 10
    
    # attach trace functions to the array
    trace variable $varname "r" "dbm_trace_read $name"
    trace variable $varname "w" "dbm_trace_write $name"
    trace variable $varname "u" "dbm_trace_unset $name"
    
}


##############################################################################
#
# Unattaches a xDBM-file from an array variable.
# The database will be closed.
# 
# varname is the name of the array that will be unattached.
#         varname MUST be a global variable
#
# keepdata: if this argument is "1", then all data from the database is loaded
#           before the database is closed. 
#           BEWARE: if your database is large, this can consume a lot of memory
#                   and result in massive swapping!!!
#           if this argument is "0" which is default, the array will be 
#           destroyed after closing the database.
#
# returns 1 on success, 0 on error
#
##############################################################################
proc dbm_unattach { varname { keepdata 0 }} {
    global DBM $varname
    
    if {[info exists DBM(CACHE:NAME:$varname)]} {
	set name $DBM(CACHE:NAME:$varname)

	# remove tracing
	foreach i [trace vinfo $varname] {
	    trace vdelete $varname [lindex $i 0] [lindex $i 1]
	}
	
	if {$keepdata} {
	    # read in the whole database
	    catch {		
		foreach i [$DBM(DB) list $name] {
		    array set $varname [list $i [$DBM(DB) fetch $name $i]]
		}
	    }
	} else {
	    unset $varname
	}

	# close database
	catch { $DBM(DB) close $name }
	
	# clean up
	catch { 
	    foreach i [array names DBM *$varname] {
		unset DBM($i)
	    }
	}
	return 1
    } else {
	return 0
    }
}
