/*-------------------------------------------------------------------------
 *
 * handlers.c		- Handlers for the PL/pgSQL
 *			  procedural language
 *
 * Portions Copyright (c) 1996-2012, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 *
 * IDENTIFICATION
 *	  src/pl/plpgpsm/handlers.c
 *
 *-------------------------------------------------------------------------
 */

#include "plpgpsm.h"
#include "executor/spi.h"
#include "utils/guc.h"

PG_MODULE_MAGIC;

bool dump_parser;

/*
 * library load-time initialization
 *
 */
void
_PG_init(void)
{
	static bool inited = false;

	if (inited)
		return;

	DefineCustomBoolVariable("plpgpsm.dump_parser",
		 "If true, AST tree is dumped",
		 NULL,
		 &dump_parser,
		 true,
		 PGC_USERSET, 0,
		 NULL, NULL, NULL);

	EmitWarningsOnPlaceholders("plpgpsm");

	inited = true;
}

/*
 * The PostgreSQL function manager and trigger manager
 * call this function for execution of PL/pgPSM procedures.
 *
 */
PG_FUNCTION_INFO_V1(plpgpsm_call_handler);

Datum
plpgpsm_call_handler(PG_FUNCTION_ARGS)
{
	Datum		retval;
	int			rc;
	PLpgPSM_function *func;

	/*
	 * Connect to SPI manager
	 */
	if ((rc = SPI_connect()) != SPI_OK_CONNECT)
		elog(ERROR, "SPI_connect failed: %s", SPI_result_code_string(rc));

	/* Find or compile the function */
	func = plpgpsm_compile(fcinfo, false);

	/*
	 * Determine if called as function or trigger and call appropriate
	 * subhandler
	 */
	retval = plpgpsm_exec_function(func, fcinfo);

	/*
	 * Disconnect from SPI manager
	 */
	if ((rc = SPI_finish()) != SPI_OK_FINISH)
		elog(ERROR, "SPI_finish failed: %s", SPI_result_code_string(rc));

	return retval;
}

/*
 * This function attempts to validate a PL/pgPSM function at
 * CREATE FUNCTION time
 *
 */
PG_FUNCTION_INFO_V1(plpgpsm_validator);

Datum
plpgpsm_validator(PG_FUNCTION_ARGS)
{
	Oid				funcoid = PG_GETARG_OID(0);
	FunctionCallInfoData	fake_fcinfo;
	FmgrInfo		flinfo;

	MemSet(&fake_fcinfo, 0, sizeof(fake_fcinfo));
	MemSet(&flinfo, 0, sizeof(flinfo));
	fake_fcinfo.flinfo = &flinfo;
	flinfo.fn_oid = funcoid;
	flinfo.fn_mcxt = CurrentMemoryContext;

	/* Test-compile the function */
	plpgpsm_compile(&fake_fcinfo, true);

	if (dump_parser)
		plpgpsm_elog_astnode(NOTICE, parser.asttree);

	PG_RETURN_VOID();
}
