################################################################################
##$Namespace: profiler
##$Summary: Provides components for profiling the performance of procedure calls
##$Version: 1.0.0
##$Author: Michael E Allen
##$Copyright: Copyright(C)2001 Michael E Allen
##$License: GNU General Public License
##$Create Date: August 2, 2001
##$Description: 
##$Bugs: 
##+	None
##$To Do: 
##+	Nothing
##$Changes:
##+	None
##$Disclaimer:
##+	This program is free software; you can redistribute it and/or modify
##+	it under the terms of the GNU General Public License as published by
##+	the Free Software Foundation; either version 2 of the License, or
##+	(at your option) any later version.
##+	This program is distributed in the hope that it will be useful,
##+	but WITHOUT ANY WARRANTY; without even the implied warranty of
##+	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##+	GNU General Public License for more details.
##+	
##+	You should have received a copy of the GNU General Public License
##+	along with this program; if not, write to the Free Software
##+	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
################################################################################
namespace eval ::profiler {
	variable mCallers        0
	variable mCompileTime    0
	variable mCallCount      0
	variable mTotalRunTime   0
	variable mDescendantTime 0
	variable mDescendants    0
	variable mStatTime       0

	################################################################################
	##$Procedure: Handler
	##$Summary: Profiles a function
	##$Version: 1.0.0
	##$Author: Michael E Allen
	##$Create Date: August 2, 2001
	##$Description:
	##+	This function works together with Proc, which replaces the proc 
	##+	command.  When a new procedure is defined, it creates an alias to 
	##+	this function; when that procedure is called, it calls this handler 
	##+	first, which gathers profiling information from the call.
	##$Bugs: 
	##+	None
	##$To Do: 
	##+	Nothing
	##$Changes:
	##+	None
	##$Synopsis:
	##+	::profiler::Handler name args
	##$Parameters:
	##+	name:
	##+		The name of the procedure that is being called
	##+	args:
	##+		The arguments to pass to the procedure
	##$Results:
	##+	Returns the body of the procedure
	################################################################################
	proc Handler {pName args} {
		variable mCallers
		variable mCompileTime
		variable mCallCount
		variable mTotalRunTime
		variable mDescendantTime
		variable mDescendants
		variable mStatTime
		
		if {([info level] == 1)} {
			set lCaller GLOBAL
		} else {
			set lCaller [lindex [info level -1] 0]
			set lCaller [string range $lCaller 0 end-4]
		}

		if {([catch {incr mCallers($pName,$lCaller)}])} {
			set mCallers($pName,$lCaller) 1
		}

		StartTimer $pName.$lCaller

		set lBody [uplevel ${pName}ORIG $args]
		set lElapsedTime [MarkTimer $pName.$lCaller]
		
		lappend mStatTime($pName) $lElapsedTime

		if {([incr mCallCount($pName)] == 1)} {
			set mCompileTime($pName) $lElapsedTime
		}

		incr mTotalRunTime($pName) $lElapsedTime

		if { [catch {incr mDescendantTime($lCaller) $lElapsedTime}] } {
			set mDescendantTime($lCaller) $lElapsedTime
		}
		if { [catch {incr mDescendants($lCaller,$pName)}] } {
			set mDescendants($lCaller,$pName) 1
		}
		return $lBody
	}
	
	################################################################################
	##$Procedure: Proc
	##$Summary: Replace for the proc command
	##$Version: 1.0.0
	##$Author: Michael E Allen
	##$Create Date: August 2, 2001
	##$Description:
	##+	This procedure is a replacement for the TCL standard proc command.  
	##+	It provides basic profiling capabilities to TCL.  This procedure
	##+	works by creating an alias in the current interpreter that points to
	##+	::profiler::Handler, and then adding the suffix "ORIG" to the end
	##+	of your procedures 'real' name.
	##$Bugs: 
	##+	None
	##$To Do: 
	##+	Nothing
	##$Changes:
	##+	None
	##$Synopsis:
	##+	::profiler::Handler name args body
	##$Parameters:
	##+	name:
	##+		The name of the procedure that is being called
	##+	args:
	##+		The arguments to pass to the procedure
	##+	body:
	##+		The body of the procedure
	##$Results:
	##+	None
	################################################################################
	proc Proc {pName pArgList pBody} {
		variable mCompileTime
		variable mCallCount
		variable mTotalRunTime
		variable mDescendantTime
	    variable mStatTime

	    set lNameSpace [uplevel [list namespace current]]
	
	    if {(![string equal $lNameSpace "::"])} {
			if {(![regexp "^::" $pName])} {
		    	set pName "${lNameSpace}::${pName}"
			}
	    }
	    

	    if {(![regexp "^::" $pName])} {
			set pName "::$pName"
	    }
	
	    set mCallCount($pName) 0
	    set mCompileTime($pName) 0
	    set mTotalRunTime($pName) 0
	    set mDescendantTime($pName) 0
	    set mStatTime($pName) {}

	    uplevel 1 [list ::RealProc ${pName}ORIG $pArgList $pBody]
	    uplevel 1 [list interp alias {} $pName {} ::profiler::Handler $pName]
	    return
	}
	
	################################################################################
	##$Procedure: Init
	##$Summary: Initialize the profiler
	##$Version: 1.0.0
	##$Author: Michael E Allen
	##$Create Date: August 2, 2001
	##$Description:
	##+	This function replaces the standard TCL proc command with the 
	##+ command ::profiler::Proc by first renaming the standard proc command
	##+	and then creating a new alias called proc to the ::profiler::Proc
	##+	command.  Any procedures declared after this procedure is called
	##+	will automatically contain profiling information.
	##$Bugs: 
	##+	None
	##$To Do: 
	##+	Nothing
	##$Changes:
	##+	None
	##$Synopsis:
	##+	::profiler::Init
	##$Parameters:
	##+	None
	##$Results:
	##+	None
	################################################################################
	proc Init {} {
	    rename ::proc ::RealProc
	    interp alias {} proc {} ::profiler::Proc
	
	    return
	}

	################################################################################
	##$Procedure: Kill
	##$Summary: Reverses the effects of ::profiler::Init
	##$Version: 1.0.0
	##$Author: Michael E Allen
	##$Create Date: August 2, 2001
	##$Description:
	##+	This procedure will put the standard TCL proc command back in place
	##+	and remove the alias for ::profiler::Proc.  It will also rename 
	##+	every procedure that was created after ::profiler::Init was 
	##+	called and remove all profiling information
	##$Bugs: 
	##+	None
	##$To Do: 
	##+	Nothing
	##$Changes:
	##+	None
	##$Synopsis:
	##+	::profiler::Kill
	##$Parameters:
	##+	None
	##$Results:
	##+	None
	################################################################################
	proc Kill {} {
		variable mCallers
		variable mCompileTime
		variable mCallCount
		variable mTotalRunTime
		variable mDescendantTime
		variable mDescendants
		variable mStatTime
	
		interp alias {} proc {}
	    rename ::RealProc ::proc

		RenameAllProcs
		
		catch {unset mCallers}
		catch {unset mCompileTime}
		catch {unset mCallCount}
		catch {unset mTotalRunTime}
		catch {unset mDescendantTime}
		catch {unset mDescendants}
		catch {unset mStatTime}
		
	    return
	}

	################################################################################
	##$Procedure: RenameAllProcs
	##$Summary: Recursively renames all procedures in all namespaces
	##$Version: 1.0.0
	##$Author: Michael E Allen
	##$Create Date: August 2, 2001
	##$Description:
	##+	This procedure will go through every namespace (children included)
	##+	and attempt to rename all the procedures contained within
	##$Bugs: 
	##+	None
	##$To Do: 
	##+	Nothing
	##$Changes:
	##+	None
	##$Synopsis:
	##+	::profiler::RenameAllProcs ?namespace?
	##$Parameters:
	##+	namespace:
	##		The namespace to start searching in.  Defaults to the global namespace
	##$Results:
	##+	None
	################################################################################
	proc RenameAllProcs {{pNameSpace ""}} {
		RenameProcs $pNameSpace
		set lNameSpaces [lsort -dictionary [namespace children $pNameSpace]]
		foreach lNameSpace $lNameSpaces {
			RenameAllProcs $lNameSpace
		}
	}

	################################################################################
	##$Procedure: RenameProcs
	##$Summary: Renames all procedures within a namespace
	##$Version: 1.0.0
	##$Author: Michael E Allen
	##$Create Date: August 2, 2001
	##$Description:
	##+	This procedure will rename all the procedures within a namespace.  It
	##+	It searches for any procedure with the name *ORIG, and removes the 
	##+	ORIG suffix from the name.  It will skip any procedures that already
	##+	exist with the new name.  It also removes the orignal interpreter alias
	##$Bugs: 
	##+	None
	##$To Do: 
	##+	Nothing
	##$Changes:
	##+	None
	##$Synopsis:
	##+	::profiler::RenameProcs ?namespace?
	##$Parameters:
	##+	namespace:
	##		The namespace to start searching in.  Defaults to the global namespace
	##$Results:
	##+	None
	################################################################################
	proc RenameProcs {{pNameSpace ""}} {
		variable mCallCount
		set lProcs [info procs "$pNameSpace\::*ORIG"]
		
		foreach lProc $lProcs {
			interp alias {} [string range $lProc 0 end-4] {}	
			catch {rename $lProc [string range $lProc 0 end-4]}
		}
	}


	################################################################################
	##$Procedure: RenameProcs
	##$Summary: Reset collected data for profiled procedures
	##$Version: 1.0.0
	##$Author: Michael E Allen
	##$Create Date: August 2, 2001
	##$Description:
	##+	This procedure will remove all profiling information for a given 
	##+	procedure by setting all the profiling data to zero
	##$Bugs: 
	##+	None
	##$To Do: 
	##+	Nothing
	##$Changes:
	##+	None
	##$Synopsis:
	##+	::profiler::Reset ?pattern?
	##$Parameters:
	##+	pattern:
	##+		procedure name pattern to search for, asterisk is used for wildcard
	##+		matching.  If the procedure is contained within a namespace, then
	##+		be sure to include the namespace, or a leading wildcard.  The
	##+		default is *, which will match everything
	##$Results:
	##+	None
	################################################################################
	proc Reset {{pPattern *}} {
	    variable mCallCount
	    variable mCompileTime
	    variable mTotalRunTime
	    variable mCallers
	    variable mStatTime
	
	    foreach lName [array names mCallCount $pPattern] {
			set mCallCount($lName) 0
			set mCompileTime($lName) 0
			set mTotalRunTime($lName) 0
			set mStatTime($lName) {}
			foreach lCallerName [array names mCallers "$lName,*"] {
			    unset mCallers($lCallerName)
			}
	    }
	    return
	}


# ::profiler::tZero --
#
#	Start a named timer instance
#
# Arguments:
#	tag	name for the timer instance; if none is given, defaults to ""
#
# Results:
#	None.
	################################################################################
	##$Procedure: StartTimer
	##$Summary: Start a named timer instance
	##$Version: 1.0.0
	##$Author: Michael E Allen
	##$Create Date: August 2, 2001
	##$Description:
	##+	This procedure creates a new timer instance by capturing the current
	##+	time in milliseconds and microseconds and storing them within a 
	##+	global namespace variable
	##$Bugs: 
	##+	None
	##$To Do: 
	##+	Nothing
	##$Changes:
	##+	None
	##$Synopsis:
	##+	::profiler::StartTime ?name?
	##$Parameters:
	##+	name:
	##+		The name of timer to create, defaults to NULL
	##+	args:
	##+		The arguments to pass to the procedure
	##$Results:
	##+	Returns the body of the procedure
	################################################################################
	proc StartTimer {{pTimerName ""}} {
	    set lMilliSeconds [clock clicks -milliseconds]
	    set lMicroSeconds  [clock clicks]
	    
	    regsub -all {:} $pTimerName {} pTimerName
	    
	    set ::profiler::mTimer$pTimerName [list $lMicroSeconds $lMilliSeconds] 
	    return
	}

	################################################################################
	##$Procedure: MarkTimer
	##$Summary: Returns the elapsed time for a named timer
	##$Version: 1.0.0
	##$Author: Michael E Allen
	##$Create Date: August 2, 2001
	##$Description:
	##+	This procedure returns the elapsed time since the start of a named timer
	##$Bugs: 
	##+	None
	##$To Do: 
	##+	Nothing
	##$Changes:
	##+	None
	##$Synopsis:
	##+	::profiler::MarkTimer ?name?
	##$Parameters:
	##+	name:
	##+		The name of timer to create, defaults to NULL
	##$Results:
	##+	Returns the elapsed time between the start of the named timer and the
	##+	current time, in microseconds
	################################################################################
	proc MarkTimer {{pTimerName ""}} {
		set lCurrentMicroSeconds [clock clicks]
		set lCurrentMilliSeconds [clock clicks -milliseconds]
		
		regsub -all {:} $pTimerName {} pTimerName
		
		upvar ::profiler::mTimer$pTimerName uTimer
		
		set lStartMicroSeconds [lindex $uTimer 0 ] 
		set lStartMilliSeconds [lindex $uTimer 1 ]
		
		set lElapsedMicroSeconds [ expr { ($lCurrentMicroSeconds-$lStartMicroSeconds) } ]
		set lElapsedMilliSeconds [ expr { ($lCurrentMilliSeconds-$lStartMilliSeconds) } ]000

		set lElapsedTime $lElapsedMicroSeconds

		if { $lElapsedTime < 0 || $lElapsedTime > 1000000 } {
			set lElapsedTime $lElapsedMilliSeconds
		}
		return $lElapsedTime
	}

	################################################################################
	##$Procedure: Statistics
	##$Summary: Compute statistical information for a set of values
	##$Version: 1.0.0
	##$Author: Michael E Allen
	##$Create Date: August 2, 2001
	##$Description:
	##+	This procedure will compute the mean, standard deviation and the 
	##+	covariance for a list of values.  It will then return the results
	##+	in a list
	##$Bugs: 
	##+	None
	##$To Do: 
	##+	Nothing
	##$Changes:
	##+	None
	##$Synopsis:
	##+	::profiler::Statistics args
	##$Parameters:
	##+	args:
	##+		The list of values for which to compute statistics for
	##+	args:
	##+		The arguments to pass to the procedure
	##$Results:
	##+	Returns a list of three values:
	##+		{mean standard_deviation covariance}
	################################################################################
	proc Statistics {args} {
		set lSum      0
		set lMean     0
		set lSigmaSQ  0
		set lSigma    0
		set lCov      0
		set lArgLength [llength $args]

		if {($args != "") && ($lArgLength > 1)} { 
			foreach lValue $args {
				set lSum [expr {$lSum+$lValue}]
			}

			set lMean [expr {$lSum/$lArgLength}]

			foreach lValue $args {
				set lSigmaSQ [expr {$lSigmaSQ+pow(($lValue-$lMean),2)}]
			}
			
			set lSigmaSQ [expr {$lSigmaSQ/($lArgLength-1)}] 
			set lSigma   [expr {round(sqrt($lSigmaSQ))}]
			set lCov     [expr {(($lSigma*1.0)/$lMean)*100}]
			set lCov     [expr {round($lCov*10)/10.0}]
		}   
		return [list $lMean $lSigma $lCov]
	}

	################################################################################
	##$Procedure: Dump
	##$Summary: Returns a list of profile information for a procedure
	##$Version: 1.0.0
	##$Author: Michael E Allen
	##$Create Date: August 2, 2001
	##$Description:
	##+	This procedure will return a list of values containing all the profiling
	##+	information for any procedure that matches the pattern
	##$Bugs: 
	##+	None
	##$To Do: 
	##+	Nothing
	##$Changes:
	##+	None
	##$Synopsis:
	##+	::profiler::Dump ?pattern?
	##$Parameters:
	##+	pattern:
	##+		procedure name pattern to search for, asterisk is used for wildcard
	##+		matching.  If the procedure is contained within a namespace, then
	##+		be sure to include the namespace, or a leading wildcard.  The
	##+		default is *, which will match everything
	##$Results:
	##+	Returns a list of values in the form {name data}, for each procedure
	##+	that matched the pattern.  The data is itself a list of values
	##+	containing the profiling data
	################################################################################
	proc Dump {{pPattern *}} {
		variable mCallCount
		variable mCompileTime
		variable mTotalRunTime
		variable mCallers
		variable mDescendantTime
		variable mDescendants
		variable mDescendantsList
		variable mStatTime

		set lResult ""
		foreach lName [lsort [array names mCallCount $pPattern]] {
			set lCallerNameIndex [expr {[string length $lName] + 1}]

			catch {unset lCallers}

			foreach lCallerName [lsort [array names mCallers "$lName,*"]] {
				set lCallers([string range $lCallerName $lCallerNameIndex end]) $mCallers($lCallerName)
			}

			set lAverageRunTime 0
			set lSigmaRunTime 0
			set lCovRunTime 0
			set lAverageDescendantTime 0

			if {($mCallCount($lName) > 0)} {
				set lResult [eval Statistics $mStatTime($lName)]

				set lAverageRunTime [lindex $lResult 0]
				set lSigmaRunTime   [lindex $lResult 1]
				set lCovRunTime     [lindex $lResult 2]
				set lAverageDescendantTime \
					[expr {$mDescendantTime($lName)/$mCallCount($lName)}]
			}
			
			set mDescendantList ""
			
			foreach lDescendantName [lsort [array names mDescendants $lName,*]] {
			    lappend mDescendantList [string range $lDescendantName $lCallerNameIndex end]
			}
			
			lappend lResult $lName [list \
				call_count $mCallCount($lName) \
				callers [array get lCallers] \
				compile_time $mCompileTime($lName) \
				total_runtime $mTotalRunTime($lName) \
				average_runtime $lAverageRunTime \
				standard_deviation_runtime  $lSigmaRunTime \
				covariance_percent_runtime $lCovRunTime \
				descendant_time $mDescendantTime($lName) \
				average_descendant_time $lAverageDescendantTime \
				descendants $mDescendantList]
	    }
	    return $lResult
	}

	################################################################################
	##$Procedure: Print
	##$Summary: Print a formatted report of the profiling information
	##$Version: 1.0.0
	##$Author: Michael E Allen
	##$Create Date: August 2, 2001
	##$Description:
	##+	This procedure will print to standard out, a human readable report
	##+	of the profiling information for a procedure
	##$Bugs: 
	##+	None
	##$To Do: 
	##+	Nothing
	##$Changes:
	##+	None
	##$Synopsis:
	##+	::profiler::Handler name args
	##$Parameters:
	##+	pattern:
	##+		procedure name pattern to search for, asterisk is used for wildcard
	##+		matching.  If the procedure is contained within a namespace, then
	##+		be sure to include the namespace, or a leading wildcard.  The
	##+		default is *, which will match everything
	##$Results:
	##+	None
	################################################################################
	proc Print {{pPattern *}} {
		variable mCallCount
		variable mCompileTime
		variable mTotalRunTime
		variable mCallers
		variable mDescendantTime
		variable mDescendants
		variable mStatTime

	    set lResult ""

	    foreach lName [lsort [array names mCallCount $pPattern]] {
			set lAverageRunTime 0
			set lSigmaRunTime 0
			set lCovRunTime 0
			set lAverageDescendantTime 0
	
			if {($mCallCount($lName) > 0)} {
				set lResult [eval Statistics $mStatTime($lName)]
	
				set lAverageRunTime [lindex $lResult 0]
				set lSigmaRunTime   [lindex $lResult 1]
				set lCovRunTime     [lindex $lResult 2]
				set lAverageDescendantTime \
					[expr {$mDescendantTime($lName)/$mCallCount($lName)}]
			}
	
			puts "Profiling information for $lName"
			puts "[string repeat = 60]"
			puts "            Total calls:  $mCallCount($lName)"
			if {(!$mCallCount($lName))} {
			    puts ""
			    continue
			}

			puts "    Caller distribution:"

			set lCallerNameIndex [expr {[string length $lName] + 1}]

			foreach lCallerName [lsort [array names mCallers "$lName,*"]] {
			    puts "                          [string range $lCallerName $lCallerNameIndex end]:  $mCallers($lCallerName)"
			}

			puts "           Compile time:  $mCompileTime($lName)"
			puts "          Total runtime:  $mTotalRunTime($lName)"
			puts "        Average runtime:  $lAverageRunTime"
			puts "          Runtime StDev:  $lSigmaRunTime"
			puts "         Runtime cov(%):  $lCovRunTime"
			puts "  Total descendant time:  $mDescendantTime($lName)"
			puts "Average descendant time:  $lAverageDescendantTime"
			puts "Descendants:"
			if {(!$mDescendantTime($lName))} {
			    puts "  none"
			}
			foreach mDescendantName [lsort [array names mDescendants "$lName,*"]] {
			    puts "  [string range $mDescendantName $lCallerNameIndex end]: $mDescendants($mDescendantName)"
			}
			puts ""
	    }
	}
}

