namespace eval ::exceptions {
	namespace export exception throw assert try

	variable Exceptions
	variable ErrorCode

	proc exception {pExceptionType args} {
		variable Exceptions
	
		if {([llength $args] == 0)} {
			if {([info exists Exceptions($pExceptionType)])} {
				return $Exceptions($pExceptionType)
			} else {
				return ""
			}
		} elseif {([llength $args] == 1)} {
			set Exceptions($pExceptionType) [lindex $args 0]
		} else {
			return -code error \
				"wrong # args: should be \"[lindex [info level 0] 0] exception_type ?parents?\""
		}
	}
	
	proc throw {args} {
		variable ErrorCode
		
		global errorCode
	
	    if {([llength $args] == 0)} {
			if {!([info exists ErrorCode]) || !($ErrorCode)} {
				return -code error "no thrown exceptions"
			}
			return -code error -errorcode RETHROW
		} elseif {([llength $args] == 2)} {
			set lExceptionType [lindex $args 0]
			set lMessage [lindex $args 1]
			return -code error \
				-errorcode [list EXCEPTION $lExceptionType $lMessage] \
				-errorinfo "$lExceptionType exception thrown: $lMessage" \
				"$lExceptionType exception thrown: $lMessage"
		} else {
			return -code error \
				"wrong # args: should be \"[lindex [info level 0] 0] ?type message?\""
		}
	}
	
	proc assert {pConditionalBlockList} {
		foreach lCondition $pConditionalBlockList {
			if {([set lCatchResult [catch {uplevel expr $lCondition} lResultCode]] != 0)} {
				return -code $lCatchResult -errorinfo \
					"$lResultCode\n    (\"[lindex [info level 0] 0]\" condition $lCondition)" \
					$lResultCode
			} else {
				if {!($lResultCode)} {
					set lMessage "[lindex [info level 0] 0] failed: \"$lCondition\""
					return -code error -errorcode \
						[list EXCEPTION Assertion $lMessage] $lMessage
				}
			}
		}
		return ""
	}
	
	proc IsKindOf {pException pExceptionType} {
		variable Exceptions
		
		if {[string compare $pException $pExceptionType] == 0} {
			return 1
		}
		
		if {[info exists Exceptions($pException)]} {
			foreach lException $Exceptions($pException) {
				if {[IsKindOf $lException $pExceptionType]} {
					return 1
				}
			}
		}
		
		return 0
	}
	
	proc ConvertErrorInfo {lErrorInfo ndel sstring rstring} {
	    set l [split $lErrorInfo \n]
	    set ll [llength $l]
	    set i [lrange $l 0 [expr {$ll-$ndel-1}]]
	    regsub $sstring [lindex $l [expr {$ll-$ndel}]] $rstring r
	    lappend i $r
	    join $i \n
	}
	
	proc MatchException {pExceptionType pCatchBlock pCatchScript} {
		upvar $pCatchBlock uCatchBlock 
		upvar $pCatchScript uCatchScript
		
		set lCode ""
		foreach {lCode uCatchScript} $uCatchBlock {
			if {[IsKindOf $pExceptionType $lCode]} {
				return 1
			}
		}
		return [expr {[string compare $pExceptionType "default"] == 0}]
	}
	
	proc GetException {pErrorCode pErrorMessage pReturnValue 
	pReturnErrorCode pReturnErrorMessage} {
	    upvar $pReturnErrorCode uReturnErrorCode 
	    upvar $pReturnErrorMessage uReturnErrorMessage
	
	    switch -glob $pErrorCode {
			{EXCEPTION *} {
			    set uReturnErrorCode [lindex $pErrorCode 1]
			    set uReturnErrorMessage [lindex $pErrorCode 2]
			}
			default {
		    	set uReturnErrorCode Error
		    	set uReturnErrorMessage $pReturnValue
			}
	    }
	}
	
	proc try {pScript args} {
		set _NAME [lindex [info level 0] 0]
		
		set bReturnVar 0
		set bCatch 0
		set bFinally 0
		
		# parse command line
		set l [llength $args]
		set i 0
		while {$l-$i > 0} {
			set option [lindex $args $i]
			switch [expr {$l-$i}] {
				1 {
					upvar $option returnVar
					set bReturnVar 1
					incr i
				}
				2 {
					switch $option {
						catch {
							return -code error -errorinfo {} "syntax error: \"$_NAME ... $option \" takes 2 args"
						}
						finally {
							set finallyScript [lindex $args [expr {$i+1}]]
							set bFinally 1
							incr i 2
						}
						default {
							return -code error -errorinfo {} "unsupported option \"$option\""
						}
					}
				}
				default {
					switch $option {
						catch {
							set catchVars [lindex $args [expr {$i+1}]]
							if {[llength $catchVars] > 1} {
								upvar [lindex $catchVars 0] catchException
								upvar [lindex $catchVars 1] catchMsg
							} else {
								upvar [lindex $catchVars 0] catchMsg
							}
							set catchBlock [lindex $args [expr {$i+2}]]
							set bCatch 1
							incr i 3
						}
						finally {
							set finallyScript [lindex $args [expr {$i+1}]]
							set bFinally 1
							incr i 2
						}
						default {
							return -code error -errorinfo {} "unsupported option \"$option\""
						}
					}
				}
			}
		}
		
		global errorCode errorInfo
		global ErrorCode
		if {[set bThrown [info exists ErrorCode]]} {
			set oldThrown $ErrorCode
		}
		set ErrorCode 0
		
		# eval the main block and catch errors
		set c [catch {uplevel $pScript} returnVar]
		set ec $errorCode; set ei [ConvertErrorInfo $errorInfo 3 uplevel $_NAME]; set rv $returnVar
		switch $c {
			1 {
				set errorInfo $ei
				set ErrorCode 1
			}
			default {
				# normal result, or control flow exception (return, break or continue) caught
				# nothing to do
			}
		} 
		
		set bCaught 0
		if {($c == 1) && $bCatch} { # if an exception occurred and there is a catch block
			# get exception info
			GetException $errorCode $errorInfo $returnVar exceptionCode exceptionMsg
		
			# try to match the exception
			set bCaught [MatchException $exceptionCode catchBlock catchScript]
		
			if {$bCaught} {
				# eval the matched exception handler
				set catchMsg $exceptionMsg
				set catchException $exceptionCode
				set cCatch [catch {uplevel $catchScript} returnVar]
				set catchEi [ConvertErrorInfo $errorInfo 3 uplevel "$_NAME ... catch"]
		
				switch $cCatch {
					0 {
						# normal result, nothing to do
					}
					1 {
						# error caught during a catch
						if {[string compare $errorCode "RETHROW"] == 0} {
							# the exception has been rethrown
							set bCaught 0
						} else {
							if {$bThrown} {
								set ErrorCode $oldThrown
							} else {
								catch {unset ErrorCode}
							}
							return -code $cCatch -errorcode $errorCode -errorinfo $catchEi $returnVar
						}
					}
					default {
						# control flow exception (return, break or continue) caught
						# set the error code and info to the caught values
						set c  $cCatch
						set ei $catchEi
					}
				}
			}
		}
		
		if {$bThrown} {
			set ErrorCode $oldThrown
		} else {
			catch {unset ErrorCode}
		}
		
		if $bFinally { # if there is a finally block
			# eval the finally block
			set cFinally [catch {uplevel $finallyScript} returnVar]
			set finallyEi [ConvertErrorInfo $errorInfo 3 uplevel "$_NAME ... finally"]
			switch $cFinally {
				0 {}
				default {
					# error or control flow exception (return, break or continue) caught
					return -code $cFinally -errorcode $errorCode -errorinfo $finallyEi $returnVar
				}
			}
		}
		
		# set correct error info then return
		set errorCode $ec
		set errorInfo $ei
		switch $c {
			0 {
				# normal result
				return $c
			}
			1 {
				if {!$bCaught} { # if the exception is unhandled
					# propagate exception
					uplevel [list return -code $c -errorcode $ec -errorinfo $ei $rv]
				}
				return $c
			}
			default {
				# control flow exception (return, break or continue) caught
				return -code $c -errorcode $errorCode -errorinfo $ei $returnVar
			}
		}
	}

	exception UnknownException
	exception Exception
	exception SimpleException Exception
	exception MemoryException SimpleException
	exception NotSupportedException SimpleException
	return ""
}
