#include "parser.h"

/*======================= THE GRAMMAR =============================

  BNF for arithmetic expressions (improved unary minus)

    <expression>    ::= <term> [<addop> <term>]*
    <term>          ::= <signed factor> [<mulop> <signed factor>]*
    <signed factor> ::= [<addop>] <factor>
    <factor>        ::= <integer> | <variable> | (<expression>)



  NOT DONE YET !: 
  BNF for boolean algebra

    <b-expression>::= <b-term> [<orop> <b-term>]*
    <b-term>      ::= <not-factor> [AND <not-factor>]*
    <not-factor>  ::= [NOT] <b-factor>
    <b-factor>    ::= <b-literal> | <b-variable> | (<b-expression>)

=================================================================*/


Parser::Parser(istream& in){
  lexer=new Lexer(in);
  tree=new TreeNode();
}


Parser::~Parser(){
  delete lexer;
}


void Parser::Error(const string& s){
  cerr<<"Error: "<<s
      <<" at row "<<row
      <<" , col "<<col
      <<"."<<endl;
  exit(1); // better = throw exception here!
  bNoErrors=false;
}

void Parser::Expected(const string& s){
  Error( "Expected "+s );
}

void Parser::getToken(){
  look=lexer->lex();
  row=lexer->getRow();
  col=lexer->getCol();
}


bool Parser::parse(){
  bNoErrors=true;
  while( symTables.size() > 0 ) symTables.pop();
  getToken();
  tree=Program();
  return bNoErrors;
}


TreeNode* Parser::getTree(){
  return tree;
}

void Parser::Match(int x){
  if(look.type!=x){
    string tokStr="";
    switch(x){
      case tokIf        : tokStr+="if";           break;
      case tokElse      : tokStr+="else";         break;
      case tokWhile     : tokStr+="while";        break;
      case tokFor       : tokStr+="for";          break;
      case tokTo        : tokStr+="to";           break;
      case tokStep      : tokStr+="step";         break;
      case tokNumber    : tokStr+="number";       break;
      case tokString    : tokStr+="string";       break;
      case tokId        : tokStr+="id";           break;
      case tokInputArg  : tokStr+="input arg";    break;
      case tokArgCount  : tokStr+="arg count $$"; break;
      case tokProcId    : tokStr+="procedure id"; break;
      case tokBegin     : tokStr+="begin";        break;
      case tokEnd       : tokStr+="end";          break;
      case tokPrint     : tokStr+="print";        break;
      case tokInput     : tokStr+="input";        break;
    
      case tokOr        : tokStr+="or";      break;
      case tokAnd       : tokStr+="and";     break;
      case tokNot       : tokStr+="not";     break;
  
      case tokGe        : tokStr+=">=";      break;
      case tokGt        : tokStr+=">";       break;
      case tokLe        : tokStr+="<=";      break;
      case tokLt        : tokStr+="<";       break;
      case tokNe        : tokStr+="!=";      break;
      case tokEq        : tokStr+="==";      break;
      case tokAssign    : tokStr+="=";       break;

      case tokReturn    : tokStr+="return";  break;
      case tokBreak     : tokStr+="break";   break;

      case tokForEach   : tokStr+="foreach";   break;
      case tokIn        : tokStr+="in";        break;
      case tokSeperated : tokStr+="seperated"; break;
      case tokBy        : tokStr+="by";        break;
    
      case tokRun       : tokStr+="run";          break;
      case tokWrite     : tokStr+="write";        break;
      
      case tokLeft      : tokStr+="left";     break;
      case tokMid       : tokStr+="mid";      break;
      case tokRight     : tokStr+="right";    break;
      case tokLen       : tokStr+="len";      break;
      case tokAsc       : tokStr+="asc";      break;
      case tokChr       : tokStr+="chr";      break;
      case tokRan       : tokStr+="ran";      break;
      case tokRead      : tokStr+="read";     break;
      case tokRestore   : tokStr+="restore";  break;
      case tokData      : tokStr+="data";     break;
      case tokMod	: tokStr+="mod";      break;
      
      case tokEof       : tokStr+="end of file";  break;
      case tokError     : tokStr+="error token";  break;

      default: tokStr+= (char)x; break;
    }
    Error("Syntax error, expected '"+tokStr+"'");
  }
  else{
    getToken(); 
  }  
}



TreeNode* Parser::getId( const string& idname ){
  TreeNode* n = new TreeNode( idNode, row, col );
  n->value    = idname;
 
  //todo : execute a find...
  if( symTables.top().find( idname ) != symTables.top().end() ){
    n->address = symTables.top()[idname]->address;
  }
  else{
    n->address = symTables.top().size();
    symTables.top()[idname]=n;
  }
  return n;  
}


TreeNode* Parser::getId(){
  TreeNode* id = getId( look.str );
  Match( tokId );
  return id;
}


TreeNode* Parser::runFunction(){
  TreeNode* n=new TreeNode( runNode, row, col );
  Match(tokRun);
  Match('(');
  
  n->appendChild( Expression() );
  
  Match(')');
  return n;
}

TreeNode* Parser::writeFunction(){
  TreeNode* n=new TreeNode( writeNode, row, col );
  Match(tokWrite);
  Match('(');
  
  n->appendChild( Expression() );
  Match(',');
  n->appendChild( Expression() );
  
  Match(')');
  return n;
}

TreeNode* Parser::leftFunction(){
  TreeNode* n = new TreeNode( leftNode, row, col );

  Match(tokLeft);
  Match('(');
  n->appendChild( Expression() );
  Match(',');
  n->appendChild( Expression() );
  Match(')');
  
  return n;
}

TreeNode* Parser::modFunction(){
  TreeNode* n = new TreeNode( modNode, row, col );
  
  Match( tokMod );
  Match('(');
  n->appendChild( Expression() );
  Match(',');
  n->appendChild( Expression() );
  Match(')');
  
  return n;
}

 
TreeNode* Parser::midFunction(){
  TreeNode* n = new TreeNode( midNode, row, col );

  Match(tokMid);
  Match('(');
  n->appendChild( Expression() );
  Match(',');
  n->appendChild( Expression() );
  
  if( look.type == ',' ){  //length is given
    Match(',');
    n->appendChild( Expression() );
  }
  
  Match(')');

  return n;
}

  
TreeNode* Parser::rightFunction(){
  TreeNode* n = new TreeNode( rightNode, row, col );

  Match(tokRight);
  Match('(');
  n->appendChild( Expression() );
  Match(',');
  n->appendChild( Expression() );
  Match(')');

  return n;
}

 
TreeNode* Parser::lenFunction(){
  TreeNode* n = new TreeNode( lenNode, row, col );

  Match(tokLen);
  Match('(');
  n->appendChild( Expression() );
  Match(')');

  return n;
}

   
TreeNode* Parser::ascFunction(){
  TreeNode* n = new TreeNode( ascNode, row, col );

  Match(tokAsc);
  Match('(');
  n->appendChild( Expression() );
  Match(')');

  return n;
}

   
TreeNode* Parser::chrFunction(){
  TreeNode* n = new TreeNode( chrNode, row, col );

  Match(tokChr);
  Match('(');
  n->appendChild( Expression() );
  Match(')');

  return n;
}

   
TreeNode* Parser::ranFunction(){
  TreeNode* n = new TreeNode( ranNode, row, col );

  Match( tokRan );
  Match('(');
  if( look.type !=')' ) n->appendChild( Expression() );
  Match(')');

  return n;
}


TreeNode* Parser::strFunction(){
  TreeNode* n = new TreeNode( strNode, row, col );
  
  Match( tokStr );
  Match( '(' );
  n->appendChild( Expression() );
  Match( ')' );
  
  return n;
}

TreeNode* Parser::valFunction(){
  TreeNode* n = new TreeNode( valNode, row, col );
  
  Match( tokVal );
  Match( '(' );
  n->appendChild( Expression() );
  Match( ')' );
  
  return n;
}


   
TreeNode* Parser::readStatement(){
  TreeNode* n = new TreeNode( readNode, row, col );

  Match( tokRead );
  n->appendChild( getId() );
  while( look.type == ',' ){
    getToken(); //the ,
    n->appendChild( getId() );
  }
  return n;
}

  
TreeNode* Parser::restoreStatement(){
  TreeNode* n = new TreeNode( restoreNode, row, col );

  Match( tokRestore );

  return n;
}


TreeNode* Parser::dataLine(){
  TreeNode* n = new TreeNode( dataNode, row, col );

  Match( tokData );
  
  n->appendChild( Expression() );
  while( look.type == ',' ){
    Match( ',' );
    n->appendChild( Expression() );
  }

  return n;
}

  


TreeNode* Parser::signedFactor(){
  TreeNode* sfac; //used by '-' and tokNot  
  switch( look.type ){
    case '+':     Match('+');
                  return Factor();
                  break;

    case '-':     Match('-');
                  sfac=Factor();
                  if( sfac->getType() == constantNode ){
                    sfac->value = "-" + sfac->value;
                    sfac->const_value = - sfac->const_value;
                    return sfac;
                  }
                  else{
                    TreeNode* minus= new TreeNode( minusNode, row, col );
                    minus->appendChild( sfac );
                    return minus;
                  }
                  break;

    case tokNot:  Match(tokNot);
                  sfac=Factor();
                  if( sfac->getType() == constantNode ){
                    if( sfac->value == "0" ){
                      sfac->value = "1";
                    }
                    else{
                      sfac->value = "0";
                    }
                    
                    return sfac;
                  }
                  else{
                    TreeNode* n = new TreeNode( notNode, row, col );
                    n->appendChild( sfac );
                    return n;
                  }
                  break;

    default:      return Factor();
                  break;
  }
}



/*---------------------------------------------------------------*/
/* Parse and Translate a Math Factor */
TreeNode* Parser::Factor()
{
  TreeNode* n;
  Number d;
  switch( look.type ){
    case '(':         Match('(');
                      n=Expression();
                      Match(')');
                      break;

    case tokId:       n=getId();
                      if( look.type == '(' ){       //is function call
                        string name = n->value;
                        delete n;              
                        n = FunctionCall( name );
                        n->setType( funcReturnNode ); //expect returned value on stack
                      }
                      break;
                  
    case tokString:   n=new TreeNode( stringConstantNode, row, col );
                      n->value = look.str;
                      Match(tokString);
                      break;

    case tokNumber:   n=new TreeNode( constantNode, row, col );
                      d=Number( look.str );
                      d.toDouble();
                      n->const_value = d.val;
                      Match(tokNumber);
                      break;
                      
    case tokInputArg: n = new TreeNode( inputArgNode, row, col );
                      n->value = look.str;
                      Match( tokInputArg );
                      break;
                      
    case tokArgCount: n = new TreeNode( argCountNode, row, col );
                      Match( tokArgCount );
                      break;
    
    case tokRun:      n=runFunction();     break;
    case tokLeft:     n=leftFunction();    break;
    case tokMid:      n=midFunction();     break;
    case tokMod:      n=modFunction();     break;
    case tokRight:    n=rightFunction();   break;
    case tokLen:      n=lenFunction();     break;
    case tokAsc:      n=ascFunction();     break;
    case tokChr:      n=chrFunction();     break;
    case tokRan:      n=ranFunction();     break;
    case tokVal:      n=valFunction();     break;
    case tokStr:      n=strFunction();     break;
         
    default:          Error("Illegal char in expression");
                      n=new TreeNode( Unknown, row, col );
                      getToken();
                      break;
  }
  return n;
}




/*---------------------------------------------------------------*/
/* identify a multiplicative operator */
bool Parser::isMulOp(token t){
  return 
    (t.type == '*') ||
    (t.type == '/') ||
    (t.type == tokAnd)
  ;
}



/*---------------------------------------------------------------*/
/* Parse and Translate a Math Term */
TreeNode* Parser::Term()
{
  TreeNode* termNode=signedFactor();
  TreeNode* pos=termNode;
  TreeNode* left=NULL;
  TreeNode* right=NULL;
  
  while( isMulOp(look) ){
    left=pos;
    pos=new TreeNode( Unknown, row, col );
    pos->appendChild( left );
        
    switch( look.type ){
      case '*':     Match('*');
                    right = signedFactor();
                    pos->setType( mulNode );
                    break;
      
      case '/':     Match('/');
                    right = signedFactor();
                    pos->setType( divNode );
                    break;
      
      case tokAnd:  Match(tokAnd);
                    right = signedFactor();
                    pos->setType( andNode );
                    break;
                    
      default:      Expected( "* or /" ); 
                    getToken();
                    return pos;
                    break;
    }
    
    if(right!=NULL) pos->appendChild( right );
    termNode=pos;
  }//end while
  
  return termNode;
}


bool Parser::isAddOp(token t){
  return 
    (t.type == '+') ||
    (t.type == '-') ||
    (t.type == tokGt)   ||
    (t.type == tokGe)   ||
    (t.type == tokLt)   ||
    (t.type == tokLe)   ||
    (t.type == tokEq)   ||
    (t.type == tokNe)   ||
    
    (t.type == tokOr )  ||
    (t.type == tokGe)
  ;
}


/*---------------------------------------------------------------*/
/* Parse and Translate an Expression */
TreeNode* Parser::Expression()
{
  TreeNode* retExp=Term();
  TreeNode* pos=retExp;
  TreeNode* left=NULL;
  TreeNode* right=NULL;

  while( isAddOp( look ) ){
    left=pos;
    pos=new TreeNode( Unknown, row, col );
    pos->appendChild( left );
      
    switch( look.type ){
      case '+':   Match('+');
                  right = Term() ;
                  pos->setType( addNode );
                  break;

      case '-':   Match('-');
                  right = Term();
                  pos->setType( subNode );
                  break;

      case tokGt: Match(tokGt);
                  right = Term();
                  pos->setType( nodeGT );
                  break;

      case tokLt: Match(tokLt);
                  right = Term();
                  pos->setType( nodeLT );
                  break;

      case tokGe: Match(tokGe);
                  right = Term();
                  pos->setType( nodeGE );
                  break;

      case tokLe: Match(tokLe);
                  right = Term();
                  pos->setType( nodeLE );
                  break;
      
      case tokEq: Match(tokEq);
                  right = Term();
                  pos->setType( nodeEQ );
                  break;

      case tokNe: Match(tokNe);
                  right = Term();
                  pos->setType( nodeNE );
                  break;
      
      case tokOr: Match(tokOr);
                  right = Term();
                  pos->setType( orNode );
                  break;

      default :   Expected("additive operation" );
                  getToken();
                  return pos;
                  break;
    }
    if(right!=NULL) pos->appendChild( right );
    retExp=pos;
  }
    
  return retExp;
}


TreeNode* Parser::Assignment( const string& name ){
  TreeNode* assign=new TreeNode( assignNode, row, col );
  Match(tokAssign);

  //first child of assign is id or lhv of assignment
  TreeNode* left=getId( name );
  assign->appendChild(left);
  
  //next child is expression or rhv of assignment
  TreeNode* right=Expression();
  assign->appendChild(right);

  return assign;
}



/*
  paramlist is like idlist except it does not only
  except id's it accepts expressions seperated by ','!
*/
TreeNode* Parser::ParamList(){
  TreeNode* ilist=new TreeNode( idListNode, row, col );

  //check for empty idlist -> function with no parameters
  if( look.type == ')' ) return ilist;
  
  //get id's seperated by ','
  ilist->appendChild( Expression() ); //aaah KISS (Keep It Simple Sidney)
  while( look.type == ',' ){
    Match(',');
    ilist->appendChild( Expression() ); 
  }

  return ilist;
}



/*
  <functioncall> := tokId '(' <paramlist> ')' 
  //hacked in functionality to do shell replacement by not using ( we return a commandNode 
*/
TreeNode* Parser::FunctionCall( const string& name ){
  TreeNode* fcall=0;
  if( look.type == '(' ){
    fcall=new TreeNode( functionCallNode, row, col );
  
    //first child contains function name
    TreeNode* funcid= new TreeNode( idNode, row, col );
    funcid->value = name;
    fcall->appendChild( funcid );
  
  
    Match( '(' );
    fcall->appendChild( ParamList() );
    Match( ')' );
  }
  else{
    fcall = new TreeNode( commandNode, row, col );
    fcall->value = name; 
    //todo we need to rewrite lots of code because we want \n to end the command sequence
    //we also need to be able to lex something like ls /home/user which will be command ls with arguments /home/user
  }
  return fcall;
}


// this is either an assignment or a function call!
TreeNode* Parser::Other(){
  string idname=look.str; 
  Match(tokId);
  
  if( look.type == tokAssign ){
    return Assignment( idname );
  }
  else{
    return FunctionCall( idname );
  }
}




/*
  <while> ::= tokWhile <expression> ( <statement> | <block> )
*/
TreeNode* Parser::While(){
  TreeNode* wNode=new TreeNode( whileNode, row, col );
  Match(tokWhile);
  wNode->appendChild( Expression() );
  
  if(look.type == tokBegin){ //while followed by a block
    wNode->appendChild( Block() ); 
  } 
  else{ //while followed by single statement
    wNode->appendChild( Statement() );
  }
  
  return wNode;
}






/*
  <for> ::= tokFor <id>'='<expression> tokTo <expression> (<tokStep> <expression>)? ( <statement> | <block> )
  for loops with step have 5 children
  for loops without step have 4 children
*/
TreeNode* Parser::For(){
  TreeNode* fNode=new TreeNode( forNode, row, col );
  Match(tokFor);
  fNode->appendChild( getId() ); //loop id
  Match(tokAssign);
  
  fNode->appendChild( Expression() ); //start value expression
  Match(tokTo);
  fNode->appendChild( Expression() ); //stop value expression

  if( look.type == tokStep ){
    Match(tokStep);
    fNode->appendChild( Expression() ); //step expression
  }
    
  if(look.type == tokBegin){ //for followed by a block
    fNode->appendChild( Block() ); 
  } 
  else{ //while followed by single statement
    fNode->appendChild( Statement() );
  }
  
  return fNode;
}


/*
  tokForEach <id> tokIn <expression> (tokSeperated tokBy <expression>)? (<statement>|<block>)
  4 children 
  default seperator is "\n"
*/
TreeNode* Parser::ForEach(){
  TreeNode* fNode=new TreeNode( forEachNode, row, col );
  Match(tokForEach);
  fNode->appendChild( getId() ); //loop id
  Match(tokIn);
  fNode->appendChild( Expression() ); //expression (string with \n's)
  

  if( look.type == tokSeperated ){
    Match(tokSeperated);
    Match(tokBy);
    fNode->appendChild( Expression() ); //seperator expression
  }
  else{ //default seperator "\n"
    fNode->appendChild( NewLineNode() );
  }
    
  if(look.type == tokBegin){ //for followed by a block
    fNode->appendChild( Block() ); 
  } 
  else{ //while followed by single statement
    fNode->appendChild( Statement() );
  }
  
  return fNode;
}









/*
  <if> ::= tokIf <expression> ( <statement> | <block> )
                ( tokElse ( <statement> | <block> ) )?
 
 the expression is in first child
 the first block or statement is in second child
 the else block or statement is in third child
*/
TreeNode* Parser::If(){
  TreeNode* node=new TreeNode( ifNode, row, col );
  Match(tokIf);
  node->appendChild( Expression() );
  
  if(look.type == tokBegin){  //if followed by a block
    node->appendChild( Block() ); 
  } 
  else{   //if followed by single statement
    node->appendChild( Statement() );
  }
  
  if(look.type == tokElse){ //else part
    Match(tokElse);
  
    if(look.type == tokBegin){  //else is followed by block
      node->appendChild( Block() );
    }
    else{
      node->appendChild( Statement() );
    }
  
  }
  
  return node;
}


TreeNode* Parser::getString(){
  TreeNode* str = new TreeNode( stringConstantNode, row, col );
  
  str->value = look.str;
  Match( tokString );
  
  return str;
}

TreeNode* Parser::NewLineNode(){
  TreeNode* newline = new TreeNode( stringConstantNode, row, col );
  newline->value="\n";
  return newline;        
}

/*
  <print> ::= 'print' ( ( <expression> | <string> ) ',' )+
  if there is trailing semicolon then no newline is added!
*/
TreeNode* Parser::Print(){
  TreeNode* node=new TreeNode( printNode, row, col );
  getToken(); //is tokPrint or tokPrintLn
  
  //first expression
  node->appendChild( Expression() );
  
  //following strings or expressions
  while( look.type == ',' ){
    getToken(); //the comma
    node->appendChild( Expression() );
  }
  
  if( look.type == ';' ){
    Match(';');
  } 
  else{
    node->appendChild( NewLineNode() );
  }
  
  return node;
}



TreeNode* Parser::Input(){
  TreeNode* inp=new TreeNode( inputNode, row, col );
  
  Match( tokInput ); 
  inp->appendChild( getId() ); 
  
  return inp;
}

TreeNode* Parser::Return(){
  TreeNode* ret=new TreeNode( returnNode, row, col );
  
  Match( tokReturn );
  ret->appendChild( Expression() );
  
  return ret;
}

TreeNode* Parser::Break(){
  TreeNode* brk = new TreeNode( breakNode, row, col );
  Match ( tokBreak );
  
  return brk;
}

TreeNode* Parser::Statement(){
  switch(look.type){
    case tokLabel   : Match( tokLabel ); 
                      cerr << "Warning labels not implemented yet!"<<endl; 
                      return Statement();     break;
    case tokIf      : return If();            break;
    case tokFor     : return For();           break;
    case tokForEach : return ForEach();       break;
    case tokWhile   : return While();         break;
    case tokPrint   : return Print();         break;
    case tokInput   : return Input();         break;
    case tokReturn  : return Return();        break;
    case tokBreak   : return Break();         break;
    case tokId      : return Other();         break; //assignment or function call

    case tokWrite   : return writeFunction(); break;
        
    case tokEnd     : break; //caught by Block
    
    case tokBegin   : Error("Begin without matching end");
                      getToken();
                      return new TreeNode(Unknown, row, col );
                      break;
    case tokRead    : return readStatement();           break;
    case tokRestore : return restoreStatement();        break;
    case ';'        : Match( ';' ); return Statement();  break; //statement seperator...
                                            
    default         : break;    
  }

  Error("Incorrect statement :"+look.str); 
  getToken();
  return new TreeNode( Unknown, row, col );
}


TreeNode* Parser::Block(){
  TreeNode* block=new TreeNode( blockNode, row, col );
  
  Match(tokBegin);
  while( (look.type!=tokEnd) && (look.type!=tokEof) ){
    block->appendChild( Statement() );
  }
  Match(tokEnd);
  
  return block;
}


//quick hack, we don't like begin, end for the main block
//we don't really need it if we require that every function
//starts with keyword function.
TreeNode* Parser::MainBlock(){
  symTable vars;
  symTables.push( vars );
  
  TreeNode* mainBlock=new TreeNode( blockNode, row, col );

  bool bPendingEnd=false;
  if( look.type == tokBegin ){ //old style, with begin-end around main block
    Match( tokBegin );
    bPendingEnd=true;  
  }
 
  while( (look.type!=tokEof) && (look.type!=tokEnd) && (look.type != tokData) ){
    mainBlock->appendChild( Statement() );
  }
  
  if( bPendingEnd ){
    Match( tokEnd );
  }
  
  symTables.pop();
  return mainBlock;
}


TreeNode* Parser::IdList(){
  TreeNode* ilist=new TreeNode( idListNode, row, col );

  //check for empty idlist -> function with no parameters
  if( look.type == ')' ) return ilist;
  
  //get id's seperated by ','
  ilist->appendChild( getId() );
  while( look.type == ',' ){
    Match(',');
    ilist->appendChild( getId() );
  }

  return ilist;
}


/*==================================================================
   EBNF for a function 
   <function> := tokId '(' <idlist> ')' <block>

   we can safely use tokId because we require ( .. )
   to be given after the id when it is called, 
   if this were not the case
   we would have to extend the lexer so that it gives a
   tokFunctionId whenever an id has the same name as a function...
===================================================================*/
TreeNode* Parser::Function(){
  symTable vars;
  symTables.push( vars );
  
  TreeNode* func=new TreeNode( functionNode, row, col );
  
  Match(tokFunction);
  TreeNode* idn=getId();
  
  func->appendChild( idn );
  Match('(');
  func->appendChild( IdList() );    
  Match(')');
  func->appendChild( Block() );

  symTables.pop();
  return func;
}


TreeNode* Parser::Program(){
  TreeNode* program=new TreeNode( programNode, row, col );

  //get the functions
  while( look.type == tokFunction ){
    program->appendChild( Function() );
  }

  //the main, or execution starting block
  program->appendChild( MainBlock() );
  
  while( look.type == tokData ){
    program->appendChild( dataLine() );
  }
  
  Match( tokEof );
  return program;
}


