#include "executer.h"

#include <stdlib.h>
#include <ctime>
#include <stdio.h>


Executer::Executer( TreeNode* tree, int argc, char** args ){
  this->tree = tree;
  functionTable.clear();
  bBreak=false;
  bReturn=false;
  
  this->argc = argc;
  this->args = args;
  
  srand( time(0) ); //initialize random numbers with random seed... 
  fData.clear();
}

void Executer::run(){
  bBreak=false;
  bReturn=false;
  fData.clear();
  
  //find program block and execute it
  
  TreeNode *node, *mainNode=0;
  TreeNode::const_iterator i;
  for(i = tree->begin() ; i != tree->end(); ++i ){
    node=*i;
    if( node->getType() == blockNode){
      mainNode = node; 
    }
    else if( node->getType() == functionNode ){
      string funcname=node->firstChild()->value;
      functionTable[ funcname ] = node; //store for later use
    }
    else if( node->getType() == dataNode ){
      TreeNode::const_iterator d;
      for( d=node->begin(); d != node->end(); ++d ){
        execute( *d );
        fData.push_back( Number( *runStack.top() ) );
        delete runStack.top();
        runStack.pop();
      }
    }
  }
  
  fDataPointer = fData.begin();
  
  if( mainNode != 0 ){
    symtable main;
    symbolTables.push( main ); //new symbol table for main block
    execute( mainNode );       //execute main block
    symbolTables.pop();        //free up stack
  }
  
  fData.clear();
}


void Executer::runError( const string& error, TreeNode* node ){
  cerr  << "RUN ERROR: "  << error 
        << " at line "    << node->getRow() 
        << " col "        << node->getCol()
        << endl;
}

void Executer::execute(TreeNode* node){
  switch( node->getType() ){
    case blockNode          : execBlock( node );        break;
    case forNode            : execFor( node );          break;
    case forEachNode        : execForEach( node );      break;
    case whileNode          : execWhile( node );        break;
    case ifNode             : execIf( node );           break;
    case printNode          : execPrint( node );        break;
    case inputNode          : execInput( node );        break;
    case assignNode         : execAssign( node );       break;
    case idNode             : execId( node );           break;
    case inputArgNode       : execInputArg( node );     break;
    case argCountNode       : execArgCount( node );     break;
    case constantNode       : execConstant( node );     break; 
    case stringConstantNode : execStrConstant( node );  break; 
    
    case addNode            : execAdd( node );          break;
    case mulNode            : execMul( node );          break;
    case divNode            : execDiv( node );          break;
    case subNode            : execSub( node );          break;
    case minusNode          : execMinus( node );        break;
        
    case nodeGE             : execGE( node );           break; 
    case nodeGT             : execGT( node );           break;
    case nodeLE             : execLE( node );           break;
    case nodeLT             : execLT( node );           break;
    case nodeNE             : execNE( node );           break;
    case nodeEQ             : execEQ( node );           break;
    
    case andNode            : execAnd( node );          break;
    case orNode             : execOr( node );           break;
    case notNode            : execNot( node );          break;
    
    case functionCallNode   : execFunction( node );     break;
    case funcReturnNode     : execRetFunction( node );  break;
    case returnNode         : execReturn( node );       break;
    case breakNode          : execBreak( node );        break;
    
    case runNode            : execRun( node );          break;
    case writeNode          : execWrite( node );        break;
    
    case leftNode           : execLeft( node );         break;
    case rightNode          : execRight( node );        break;
    case midNode            : execMid( node );          break;
    case modNode            : execMod( node );          break;
    case lenNode            : execLen( node );          break;
    case ascNode            : execAsc( node );          break;
    case chrNode            : execChr( node );          break;
    case ranNode            : execRan( node );          break;
    case readNode           : execRead( node );         break;
    case restoreNode        : execRestore( node );      break;
    case dataNode           : execData( node );         break;
    case strNode            : execStr( node );          break;
    case valNode            : execVal( node );          break;
    
    case commandNode        : execCommand( node );	break;
    
    default                 : runError( "Internal compiler error, found unsupported node with value = "+node->value, node );
                              break;
  }  
}



//execute a function
//first child   = function name
//second child  = parameters
void Executer::execFunction( TreeNode* node ){
  
  string funcname = node->firstChild()->value;

  //locate function node  
  functable::iterator p=functionTable.find( funcname );
  if( p==functionTable.end() ){
    runError( "Call to undefined function : "+ funcname +".", node );
    return;
  }
  
  TreeNode* funcnode    = p->second;
  TreeNode* funcIds     = funcnode->secondChild();
  TreeNode* callparams  = node->secondChild();
    
  //check if number of parameters match
  if( callparams->size() != funcIds->size() ){
    runError( "Call to function "+ funcname + " with wrong number of parameters.", callparams );
    return;
  }

  //pass parameters to function
  //by adding them to it's symboltable and setting the values
  symtable funcSymTable; 
  
  TreeNode::reverse_iterator pfrom;
  for(pfrom=callparams->rbegin(); pfrom!=callparams->rend(); ++pfrom ){
      
    //execute the parameters which can be expressions
    //pfrom may contain id's from higher symbol table
    //pto vars need to be created in current symbol table...
    execute( *pfrom );  
  }
  
  TreeNode::iterator pto;
  symbolTables.push(funcSymTable); //use new symboltable for current function
  for( pto=funcIds->begin(); pto!=funcIds->end(); ++pto ){
    doAssign( *pto, runStack.top() );
    runStack.pop();  
  }
  
  
  //execute function statement block
  bReturn=false; //set to true when return is called
  execute( funcnode->thirdChild() );
  bReturn=false; //function execution done
  
  symbolTables.pop(); //release function symboltable    
}


//execute a function and expect and get return 
//value from stack
//first child   = function name
//second child  = parameters
void Executer::execRetFunction( TreeNode* node ){
  unsigned int sizebefore = runStack.size();
  execFunction( node );
  if( runStack.size() == sizebefore ){
    runError( "function " + node->firstChild()->value + " did not return a value, pushing 0 on stack!", node );
    runStack.push( new Number(0) );
  }
}


void Executer::execReturn( TreeNode* node ){
  execute( node->firstChild() ); //execute return expression
  bReturn=true; //notify blocks of return
}


void Executer::execBreak( TreeNode* node ){
  bBreak=true; //stops loop block execution
}


void Executer::execBlock( TreeNode* node ){
  //execute all statements in block
  TreeNode::iterator i;
  for( i=node->begin(); i!=node->end(); ++i ){
    execute( *i );

    if( bReturn || bBreak){
      break; //jump out of block
    }
    
  }
}


Number* Executer::getId( TreeNode* id ){
  Number* nr;
  
  if( symbolTables.top().find( id->address ) != symbolTables.top().end() ){
    nr = symbolTables.top()[ id->address ];
  }
  else{
    nr = new Number(0.0);
    symbolTables.top()[ id->address ] = nr;
  }
  
  return nr;
}


void Executer::execForEach( TreeNode* node ){
  TreeNode* id         = node->firstChild();
  TreeNode* expr       = node->secondChild();
  TreeNode* seperator  = node->thirdChild();
  TreeNode* statements = node->fourthChild();
  
  execute( expr );
  string expStr = runStack.top()->strVal;
  delete runStack.top();
  runStack.pop();
  
  execute( seperator );
  string sepStr = runStack.top()->strVal;
  delete runStack.top();
  runStack.pop();

  Number *loopId   = getId(id);
  loopId->toString();
  string *loopIter = &loopId->strVal;
  
  bBreak=false;
  string::size_type pos;
  while( expStr.size() > 0 ){
    pos=expStr.find(sepStr);

    if( pos == string::npos ){  //no seperator found
      *loopIter = expStr;       //entire string
      expStr="";
    }
    else{
      *loopIter = expStr.substr(0,pos);
      expStr.erase( 0, pos + sepStr.size() );
    }
    
    execute( statements );
    if( bBreak || bReturn ) break; //jump out loop;
  }
  bBreak=false;
  
}



void Executer::execFor( TreeNode* node ){
  TreeNode* id=node->firstChild();
  TreeNode* startNode=node->secondChild();
  TreeNode* stopNode=node->thirdChild();
  TreeNode* statements=node->fourthChild();
  
  execute( startNode );
  Number* startVal = runStack.top();
  runStack.pop();
  
  double* loopIter = &getId( id )->val;
  
  execute( stopNode );
  Number* stopVal = runStack.top();
  runStack.pop();
  
  if(node->size() == 4 ){ //for loop without step part
    bBreak=false;
    for( double d=startVal->val; d<=stopVal->val; d=d+1 ){
      *loopIter = d;
      execute( statements );
      if( bBreak || bReturn ) break; //jump out loop
    }
  }
  else{ //for loop with step part
    TreeNode* step  = node->fourthChild();
    statements      = node->fifthChild();
    
    execute(step);
    Number* stepVal=runStack.top();
    runStack.pop();
    
    bBreak=false;
    if( (stepVal->val >= 0.0) && (startVal->val <= stopVal->val) ){
      for( double d=startVal->val; d<=stopVal->val; d=d+stepVal->val ){
        *loopIter = d;
        execute( statements );
        if( bBreak || bReturn ) break; //jump out loop
      }
    }
    else if( (stepVal->val < 0.0) && (startVal->val >= stopVal->val) ){
      for( double d=startVal->val; d>=stopVal->val; d=d+stepVal->val ){
        *loopIter= d;
        execute( statements );
        if( bBreak || bReturn ) break; //jump out loop
      }    
    }
    delete stepVal;
  }

  bBreak = false;
  
  delete startVal;
  delete stopVal;
}



void Executer::execWhile( TreeNode* node ){
  TreeNode* condition = node->firstChild();
  TreeNode* statements = node->secondChild();

  bBreak=false;
  execute( condition );
  Number* condVal = runStack.top();
  runStack.pop();
  while( condVal->val != 0 ){
    execute( statements );
    if( bBreak || bReturn ) break; //jump out loop
    
    delete condVal;
    execute( condition );
    condVal = runStack.top();
    runStack.pop();
  }
  bBreak=false;
  delete condVal;
}

     
void Executer::execIf( TreeNode* node ){

  TreeNode* condition = node->firstChild();
  TreeNode* ifblok = node->secondChild();

  execute( condition );
  Number* condVal = runStack.top();
  runStack.pop();


  //determine if there is an else part
  if( node->size() == 2 ){ //no else    
    if( condVal->val != 0 ){
      execute( ifblok );
    }
  }
  else{ //else part given
    TreeNode* elseblok = node->thirdChild();
    if( condVal->val != 0 ){
      execute( ifblok );
    }
    else{
      execute( elseblok );
    }
  }
  
  delete condVal;

}


void Executer::execPrint( TreeNode* node ){
  TreeNode::iterator i;
  for( i=node->begin(); i!=node->end(); ++i ){
    execute( *i );
    
    if( !runStack.top()->bString ){
      printf("%g ", runStack.top()->val );
    }
    else{
      printf("%s", runStack.top()->strVal.c_str() );
    }
    //clean but just not fast enough!
    //cout << *runStack.top();

    delete runStack.top();
    runStack.pop();
  }
  //cout<<flush;
  
}



void Executer::execInput( TreeNode* node ){
  Number* val=new Number();
  
  //ask input from cin
  //cout<<"?"; // basic style , don't like it :)
  cin >> *val;
  
  symbolTables.top()[ node->firstChild()->address ] = val;  
}

void Executer::doAssign( TreeNode* var, Number* val ){
  symtable& table = symbolTables.top();
  symtable::iterator pos = table.find( var->address );

  if( pos != table.end() ){
    //delete pos->second; //free previously allocated value!
    pos->second = val;
  }
  else{
    //symbolTables.top()[ var->address ] = val; //not fast enough, does lookup again :(
    table.insert( pair<unsigned int, Number*>( var->address, val ) );
  }  
}

inline void Executer::execAssign( TreeNode* node ){
  execute( node->secondChild() );
  doAssign( node->firstChild(), runStack.top() );
  runStack.pop(); //we don't delete the pointer because it is now used in symboltable :)
}

void Executer::execId( TreeNode* node ){
  symtable& table = symbolTables.top();
  symtable::iterator pos = table.find( node->address );
  if( pos != table.end() ){
    runStack.push( new Number( *pos->second ) );
  }
  else{ //create new variable or give error, we choose for giving an error
    runStack.push( new Number(0.0) );
    runError( "Undefined function or variable " + node->value, node );
  }
}

void Executer::execInputArg( TreeNode* node ){
  istringstream in( node->value );
  int pos;
  in >> pos;

  if( ( pos >= 0 ) && (pos < argc) ) runStack.push( new Number( string( args[ pos ] ) ) );
  else runStack.push( new Number("") );
}

void Executer::execArgCount( TreeNode* node ){
  runStack.push( new Number( (double) argc ) );  
}

void Executer::execConstant( TreeNode* node ){
  Number* c = new Number( node->const_value );
  runStack.push( c );  
}

void Executer::execStrConstant( TreeNode* node ){
  runStack.push( new Number( node->value )  );  
}


#define GET_BINARY_OPERANDS \
  execute( node->firstChild() );   \
  execute( node->secondChild() );  \
                                   \
  Number* b = runStack.top();      \
  runStack.pop();                  \
  Number* a = runStack.top();      


void Executer::execAdd( TreeNode* node ){
  GET_BINARY_OPERANDS
  *a = *a + *b;
  delete b;
}

void Executer::execMul( TreeNode* node ){
  GET_BINARY_OPERANDS
  *a = *a * (*b);
  delete b;
}

       
void Executer::execDiv( TreeNode* node ){
  GET_BINARY_OPERANDS
  *a = *a / *b;
  delete b;
}

       
void Executer::execSub( TreeNode* node ){
  GET_BINARY_OPERANDS
  *a = *a - *b;
  delete b;
}

       
void Executer::execLT( TreeNode* node ){
  GET_BINARY_OPERANDS
  *a = (double) (*a < *b);
  delete b;
}

void Executer::execLE( TreeNode* node ){
  GET_BINARY_OPERANDS
  *a = *a <= *b;
  delete b;
}

void Executer::execGT( TreeNode* node ){
  GET_BINARY_OPERANDS
  *a = *a > *b;
  delete b;
}

void Executer::execGE( TreeNode* node ){
  GET_BINARY_OPERANDS
  *a = *a >= *b;
  delete b;
}


void Executer::execEQ( TreeNode* node ){
  GET_BINARY_OPERANDS
  *a = *a == *b;
  delete b;
}

void Executer::execNE( TreeNode* node ){
  GET_BINARY_OPERANDS
  *a = *a != *b;
  delete b;
}


  
void Executer::execAnd( TreeNode* node ){
  GET_BINARY_OPERANDS
  bool nl = *a != 0;
  bool nr = *b != 0;
  *a = (double) ( nl && nr );
  
  delete b;
}

       
void Executer::execOr( TreeNode* node ){
  GET_BINARY_OPERANDS
  bool nl = *a != 0;
  bool nr = *b != 0;
  *a = (double) ( nl || nr );
  
  delete b;
}


void Executer::execNot( TreeNode* node ){
  execute( node->firstChild() );
  runStack.top()->notOperation();
}


void Executer::execMinus( TreeNode* node ){
  execute( node->firstChild() );
  runStack.top()->minus();
}


string Executer::runCommand( const string& command ){
  FILE *pstream;
  
  if(  ( pstream = popen( command.c_str(), "r" ) ) == NULL ) return "";
  
  string Line;
  char buf[100];
  
  while( fgets(buf, sizeof(buf), pstream) !=NULL){
    Line += buf;
  }
  pclose(pstream);

  return Line;
}


void Executer::execRun( TreeNode* node ){
  execute( node->firstChild() );
  
  Number* cmdVal = runStack.top();
  string cmd = cmdVal->strVal;
  runStack.pop();
  
  runStack.push( new Number( runCommand(cmd) ) );
  delete cmdVal;
}

void Executer::execCommand( TreeNode* node ){
  cout << runCommand( node->value ) << endl;
}


void Executer::execWrite( TreeNode* node ){
  execute( node->firstChild() );
  
  Number* cmdVal = runStack.top();
  string fileName = cmdVal->strVal;
  runStack.pop();
  
  ofstream out(fileName.c_str());
  if(out.is_open()){
    execute( node->secondChild() );
    out<<*runStack.top();
    delete runStack.top();
    runStack.pop();
  }
  else{
    runError( "Could not open file: " + fileName +" for writing!", node );
  }
  out.close();
  delete cmdVal;
}




void Executer::execLeft( TreeNode* node ){
  GET_BINARY_OPERANDS
  string val  = a->strVal;
  int to      = (int) b->val;
  delete a;
  delete b;
  runStack.pop();

  if( ( to >= 0 ) && ( to < (int)val.size() ) ){
    runStack.push( new Number( val.substr( 0, to ) ) );
  }
  else{
    runError("'left' function has argument that is out of boundaries of given string.", node );
    runStack.push( new Number() );
  }
}


void Executer::execRight( TreeNode* node ){
  GET_BINARY_OPERANDS
  string val  = a->strVal;
  int to      = (int) b->val;
  delete a;
  delete b;
  runStack.pop();

  if( ( to >= 0 ) && ( to < (int)val.size() ) ){
    runStack.push( new Number( val.substr( (int)val.size()-to, (int)val.size() ) ) );
  }
  else{
    runError( "'right' function has argument that is out of boundaries of given string.", node );
    runStack.push( new Number() );
  }
}

void Executer::execMod( TreeNode* node ){
  GET_BINARY_OPERANDS
  long nr = (long) a->val;
  long m  = (long) b->val;
  runStack.push( new Number( (double) (nr%m) ) );
}

void Executer::execMid( TreeNode* node ){
  GET_BINARY_OPERANDS
  string val  = a->strVal;
  int    from = (int) b->val-1;
  delete a;
  delete b;
  runStack.pop();

  int to=0;
  
  if( node->size() == 3 ){
    execute( node->thirdChild() );
    to = (int) runStack.top()->val;
    delete runStack.top();
    runStack.pop();
  }
  else{
    to = val.size()-from;
  }
   
  if( ( from < (int)val.size() ) && ( from >= 0 ) && ( from+to <= (int)val.size() ) ){
    runStack.push( new Number( val.substr( from, to ) ) );
  }
  else{
    runError( "'mid' from, to arguments run out of string boundaries or from pos is not less than to.", node );
    runStack.push( new Number() );
  } 
}        

void Executer::execLen( TreeNode* node ){
  execute( node->firstChild() );
  Number* nr = new Number( (double) runStack.top()->strVal.size() );
  delete runStack.top();
  runStack.pop();
  runStack.push( nr );
}

void Executer::execAsc( TreeNode* node ){
  execute( node->firstChild() );
  string val = runStack.top()->strVal;
  delete runStack.top();
  runStack.pop();

  if( val.size() == 0 ){
    runError( "Empty string given to ASC function!", node );
    runStack.push( new Number() );
  }
  else{
    int asciVal = (int)val[0];
    runStack.push( new Number( (double)asciVal) );
  }
}

void Executer::execChr( TreeNode* node ){
  execute( node->firstChild() );
  int val = (int) runStack.top()->val;
  delete runStack.top();
  runStack.pop();
    
  string retStr="";
  char c = (char) val;
  retStr+=c;
  
  runStack.push( new Number( retStr ) );
}


void Executer::execRan( TreeNode* node ){
  if( node->hasChildren() ){
    execute( node->firstChild() );
    int range = (int) runStack.top()->val;
    delete runStack.top();
    runStack.pop();
    
    if( range < 1 ){ 
      runError( "Argument of ran() function must be larger than 1.", node );
      runStack.push( new Number(0) );
      return;
    }
    runStack.push( new Number( (int)((double)range*rand()/(RAND_MAX+1.0)) ) );
  }
  else{
    runStack.push( new Number( rand()/(RAND_MAX+1.0) ) );
  }
  
}

void Executer::execRead( TreeNode* node ){
  for( TreeNode::const_iterator t = node->begin(); t!= node->end(); ++t ){
    if( fDataPointer == fData.end() ) break;
    
    TreeNode* var  = *t;
    //( symbolTables.top() )[ var->getName() ] = *fDataPointer;
    
    Number* val = new Number( *fDataPointer );
    doAssign( var, val );
    
    ++fDataPointer;
  }
} 

void Executer::execRestore( TreeNode* ){
  fDataPointer = fData.begin();
}

void Executer::execData( TreeNode* node ){
  runError( "Data statement not implemented here, internal error!", node );
}


void Executer::execStr( TreeNode* n ){
  execute( n->firstChild() );
  runStack.top()->toString();
}

void Executer::execVal( TreeNode* n ){
  execute( n->firstChild() );
  runStack.top()->toDouble();
}

