//
// $Id: TextBufferTest.m,v 1.6 2007/04/15 23:33:01 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "TextBufferTest.h"
#import "Random.h"
#import <ObjectiveLib/Text.h>
#import <ObjectiveLib/Character.h>
#import <ObjectiveLib/FileInStream.h>
#if defined(OL_NO_OPENSTEP)
#import <ObjectiveLib/Reaper.h>
#import <ObjectiveLib/Exception.h>
#else
#import <Foundation/NSException.h>
#endif
#include <stdlib.h>
#import <limits.h>
#import <fcntl.h>
#import <errno.h>
#import <string.h>
#import <unistd.h>
#import <sys/types.h>
#import <sys/stat.h>
#if defined(__NEXT_RUNTIME__)
#import <objc/objc-class.h>
#endif

@implementation TextBufferTest

- (void) testAppend
{
    OLTextBuffer* buf = REAP([OLTextBuffer textBuffer]);
    OLTextBuffer* buf2;
    olchar moreChars[] =
    {
        OL_SMALL_S_CHAR,
        OL_SPACE_CHAR,
        OL_SMALL_A_CHAR,
        OL_SMALL_R_CHAR,
        OL_SMALL_E_CHAR,
        OL_SPACE_CHAR,
        OL_SMALL_B_CHAR,
        OL_SMALL_I_CHAR,
        OL_SMALL_G_CHAR
    };
    olchar evenMoreChars[] =
    {
        OL_SPACE_CHAR,
        OL_SMALL_L_CHAR,
        OL_SMALL_O_CHAR,
        OL_SMALL_V_CHAR,
        OL_SMALL_E_CHAR,
        OL_SMALL_R_CHAR,
        OL_SMALL_S_CHAR
    };
    unsigned i;

    [buf appendChar: OL_SMALL_D_CHAR];
    if ([buf size] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %u", [buf size]];
    }
    if ([buf at: 0] != OL_SMALL_D_CHAR)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \\u%0.4X, but got \\u%0.4X",
            OL_SMALL_D_CHAR, [buf at: 0]];
    }
    [[buf appendChar: OL_SMALL_O_CHAR] appendChar: OL_SMALL_G_CHAR];
    if ([buf size] != 3)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 3, but got %u", [buf size]];
    }
    if ([buf at: 1] != OL_SMALL_O_CHAR)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \\u%0.4X, but got \\u%0.4X",
            OL_SMALL_O_CHAR, [buf at: 1]];
    }
    if ([buf at: 2] != OL_SMALL_G_CHAR)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \\u%0.4X, but got \\u%0.4X",
            OL_SMALL_G_CHAR, [buf at: 2]];
    }
    if ([buf appendChars: moreChars fromOffset: 0 count: 9] != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf,
            [buf appendChars: moreChars fromOffset: 0 count: 0]];
    }
    if ([buf size] != 12)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 12, but got %u", [buf size]];
    }
    for (i = 0; i < 9; i++)
    {
        if ([buf at: i + 3] != moreChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                moreChars[i], [buf at: i + 3]];
        }
    }
    if ([buf appendText: REAP([OLText textWithCString: " lovers"])] != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf,
            [buf appendText: REAP([OLText textWithCString: ""])]];
    }
    if ([buf size] != 19)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 19, but got %u", [buf size]];
    }
    for (i = 0; i < 7; i++)
    {
        if ([buf at: i + 12] != evenMoreChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                evenMoreChars[i], [buf at: i + 12]];
        }
    }
    buf2 = REAP([OLTextBuffer textBuffer]);
    for (i = 0; i < 100000; i++)
        [buf2 appendChar: OLRandom() % 0xFFFF];
    if ([buf appendTextBuffer: buf2] != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf,
            [buf appendTextBuffer: REAP([OLTextBuffer textBuffer])]];
    }
    if ([buf size] != 100019)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 100019, but got %u", [buf size]];
    }
    for (i = 0; i < 100000; i++)
    {
        if ([buf at: i + 19] != [buf2 at: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                [buf2 at: i], [buf at: i + 19]];
        }
    }
}

- (void) testAssign
{
    OLTextBuffer* buf = REAP([OLTextBuffer textBuffer]);
    OLTextBuffer* returned;
    olchar i;

    for (i = 0x0500; i < 0x0600; i++)
        [buf appendChar: i];
    returned = [buf assignAt: 0 character: 0x0700];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 0x0100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", 0x0100, [buf size]];
    }
    if ([buf at: 0] != 0x0700)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", 0x0700, [buf at: 0]];
    }
    for (i = 1; i < 0x0100; i++)
    {
        if ([buf at: i] != i + 0x0500)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %u, but got %u", i + 0x0500, [buf at: i]];
        }
    }
    returned = [buf assignAt: 0x0099 character: 0x0800];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 0x0100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", 0x0100, [buf size]];
    }
    if ([buf at: 0] != 0x0700)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", 0x0700, [buf at: 0]];
    }
    for (i = 1; i < 0x0099; i++)
    {
        if ([buf at: i] != i + 0x0500)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %u, but got %u", i + 0x0500, [buf at: i]];
        }
    }
    if ([buf at: 0x0099] != 0x0800)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", 0x0800, [buf at: 0x0099]];
    }
}

- (void) testCompare
{
    OLTextBuffer* b1 = REAP([OLTextBuffer textBuffer]);
    OLTextBuffer* b2 = REAP([OLTextBuffer textBuffer]);
    unsigned i;
    int rc;

    for (i = 0; i < 128; i++)
    {
        [b1 appendChar: 0x0500 + i];
        [b2 appendChar: 0x0500 + i];
    }
    rc = [b1 compare: b2];
    if (rc != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i", rc];
    }
    if (![b1 isEqual: b2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The buffers should be equal"];
    }
    [b2 assignAt: 100 character: [b2 at: 100] - 1];
    rc = [b1 compare: b2];
    if (rc <= 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected > 0, but got %i", rc];
    }
    if ([b1 isEqual: b2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The buffers should not be equal"];
    }
    [b2 assignAt: 100 character: [b2 at: 100] + 2];
    rc = [b1 compare: b2];
    if (rc >= 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected < 0, but got %i", rc];
    }
    if ([b1 isEqual: b2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The buffers should not be equal"];
    }
    [b2 assignAt: 100 character: [b2 at: 100] - 1];
    [b2 resize: 127 filledWith: 0];
    rc = [b1 compare: b2];
    if (rc <= 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected > 0, but got %i", rc];
    }
    if ([b1 isEqual: b2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The buffers should not be equal"];
    }
    rc = [b2 compare: b1];
    if (rc >= 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected < 0, but got %i", rc];
    }
    if ([b1 isEqual: b2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The buffers should not be equal"];
    }
}

- (void) testConvenienceAllocators
{
    OLTextBuffer* buf = REAP([OLTextBuffer textBuffer]);
    OLText* text;
    olchar chars[] =
    {
        OL_SMALL_D_CHAR,
        OL_SMALL_O_CHAR,
        OL_SMALL_G_CHAR,
        OL_SMALL_G_CHAR,
        OL_SMALL_Y_CHAR
    };
    unsigned i;

    if (![buf empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The buffer should be empty"];
    }
    text = [[OLText alloc] initWithChars: chars count: 5];
    buf = REAP([OLTextBuffer textBufferWithText: text]);
    if ([buf size] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %u", [buf size]];
    }
    for (i = 0; i < 5; i++)
    {
        if ([buf at: i] != [text at: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                [text at: i], [buf at: i]];
        }
    }
    [text RELEASE];
}

- (void) testCopy
{
    OLTextBuffer* buf = REAP([OLTextBuffer textBuffer]);
    OLTextBuffer* copy;
    olchar i;

    for (i = 0; i < 0xFFFF; i++)
        [buf appendChar: i];
    copy = [buf copy];
    if (![buf isEqual: copy])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The buffers should be equal"];
    }
    [copy RELEASE];
}

- (void) testErase
{
    OLTextBuffer* buf = REAP([OLTextBuffer textBufferWithText:
        REAP([OLText textWithCString: "dog"])]);
    OLTextBuffer* returned;
    int i;

    returned = [buf eraseAt: 0];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %u", [buf size]];
    }
    if ([buf at: 0] != OL_SMALL_O_CHAR)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \\u%0.4X, but got \\u%0.4X",
            OL_SMALL_O_CHAR, [buf at: 0]];
    }
    if ([buf at: 1] != OL_SMALL_G_CHAR)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \\u%0.4X, but got \\u%0.4X",
            OL_SMALL_G_CHAR, [buf at: 1]];
    }
    returned = [buf eraseAt: 1];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %u", [buf size]];
    }
    if ([buf at: 0] != OL_SMALL_O_CHAR)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \\u%0.4X, but got \\u%0.4X",
            OL_SMALL_O_CHAR, [buf at: 0]];
    }
    [buf clear];
    if ([buf size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %u", [buf size]];
    }
    for (i = 0; i < 1000; i++)
        [buf appendChar: 0x0500 + i];
    returned = [buf eraseFromOffset: 0 count: 100];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 900)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 900, but got %u", [buf size]];
    }
    for (i = 0; i < [buf size]; i++)
    {
        if ([buf at: i] != 0x0500 + 100 + i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                0x0500 + 100 + i, [buf at: i]];
        }
    }
    [buf clear];
    for (i = 0; i < 1000; i++)
        [buf appendChar: 0x0500 + i];
    returned = [buf eraseFromOffset: 500 count: 100];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 900)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 900, but got %u", [buf size]];
    }
    for (i = 0; i < 500; i++)
    {
        if ([buf at: i] != 0x0500 + i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                0x0500 + i, [buf at: i]];
        }
    }
    for (i = 500; i < 900; i++)
    {
        if ([buf at: i] != 0x0500 + 100 + i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                0x0500 + 100 + i, [buf at: i]];
        }
    }
    [buf clear];
    for (i = 0; i < 1000; i++)
        [buf appendChar: 0x0500 + i];
    returned = [buf eraseFromOffset: 900 count: 100];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 900)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 900, but got %u", [buf size]];
    }
    for (i = 0; i < [buf size]; i++)
    {
        if ([buf at: i] != 0x0500 + i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                0x0500 + i, [buf at: i]];
        }
    }
}

- (void) testExtraction
{
    OLTextBuffer* buf = REAP([OLTextBuffer textBuffer]);
    OLText* text;
    olchar chars[50];
    unsigned i;

    for (i = 0; i < 128; i++)
        [buf appendChar: 0x0500 + i];
    for (i = 0; i < 128; i++)
    {
        if ([buf at: i] != 0x0500 + i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                i + 0x0500, [buf at: i]];
            break;
        }
    }
    [buf getCharacters: chars fromOffset: 64 count: 50];
    for (i = 0; i < 50; i++)
    {
        if ([buf at: i + 64] != chars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                [buf at: i + 64], chars[i]];
            break;
        }
    }
    text = REAP([buf substrFromOffset: 100 count: 28]);
    for (i = 0; i < 28; i++)
    {
        if ([buf at: i + 100] != [text at: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                [buf at: i + 100], [text at: i]];
            break;
        }
    }
    text = REAP([buf text]);
    for (i = 0; i < 128; i++)
    {
        if ([text at: i] != 0x0500 + i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                i + 0x0500, [text at: i]];
            break;
        }
    }
}

- (void) testFindChar
{
    OLTextBuffer* buf = REAP([OLTextBuffer textBuffer]);
    unsigned i;
    unsigned rc;

    for (i = 0; i < 128; i++)
        [buf appendChar: 0x0500 + i];
    rc = [buf findChar: 0x0500 + 100 fromOffset: 0];
    if (rc != 100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 100, but got %u", rc];
    }
    rc = [buf findChar: 77 fromOffset: 0];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [buf findChar: 0x0500 + 110 fromOffset: 100];
    if (rc != 110)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 110, but got %u", rc];
    }
    rc = [buf findChar: 0x0510 fromOffset: 100];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
}

- (void) testFindFirstLastNotOf
{
    OLText* text;
    OLTextBuffer* buf;
    unsigned rc;
    int i;

    buf = [[OLTextBuffer alloc] init];
    for (i = 0; i < 10; i++)
        [buf appendChar: 0x0500 + i];
    text = REAP([buf text]);
    [buf clear];
    for (i = 0; i < 5000; i++)
    {
        if (i == 2499)
            [buf appendChar: 0x050A];
        else if (i == 2500)
            [buf appendChar: 0x050B];
        else
            [buf appendChar: 0x0500 + (OLRandom() % 10)];
    }
    rc = [buf findFirstNotOf: text fromOffset: 0];
    if (rc != 2499)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2499, but got %u", rc];
    }
    rc = [buf findFirstNotOf: text fromOffset: 2499];
    if (rc != 2499)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2499, but got %u", rc];
    }
    rc = [buf findFirstNotOf: text fromOffset: 2501];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [buf findLastNotOf: text fromOffset: UINT_MAX];
    if (rc != 2500)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2500, but got %u", rc];
    }
    rc = [buf findLastNotOf: text fromOffset: 2499];
    if (rc != 2499)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2499, but got %u", rc];
    }
    rc = [buf findLastNotOf: text fromOffset: 2498];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    [buf RELEASE];
}

- (void) testFindFirstLastOf
{
    OLText* text;
    OLTextBuffer* buf;
    olchar seq[] = { 0x0010, 0x0020, 0x0550, 0x0551, 0x9000, 0xA000 };
    unsigned rc;
    int i;

    buf = [[OLTextBuffer alloc] init];
    [buf appendChars: seq fromOffset: 0 count: sizeof(seq) / sizeof(olchar)];
    text = REAP([buf text]);
    [buf clear];
    for (i = 0; i < 128; i++)
        [buf appendChar: 0x0500 + i];
    rc = [buf findFirstOf: text fromOffset: 0];
    if (rc != 0x50)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", 0x50, rc];
    }
    rc = [buf findFirstOf: text fromOffset: 0x50];
    if (rc != 0x50)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", 0x50, rc];
    }
    rc = [buf findFirstOf: text fromOffset: 0x52];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [buf findLastOf: text fromOffset: UINT_MAX];
    if (rc != 0x51)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", 0x51, rc];
    }
    rc = [buf findLastOf: text fromOffset: 0x51];
    if (rc != 0x51)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", 0x51, rc];
    }
    rc = [buf findLastOf: text fromOffset: 0x49];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    [buf clear];
    [buf appendChars: seq fromOffset: 0 count: sizeof(seq) / sizeof(olchar)];
    [buf assignAt: 2 character: 0x8000];
    [buf assignAt: 3 character: 0x8001];
    text = REAP([buf text]);
    [buf clear];
    for (i = 0; i < 128; i++)
        [buf appendChar: 0x0500 + i];
    rc = [buf findFirstOf: text fromOffset: 0];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [buf findFirstOf: text fromOffset: 0x50];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [buf findFirstOf: text fromOffset: 0x51];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [buf findLastOf: text fromOffset: UINT_MAX];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [buf findLastOf: text fromOffset: 0x51];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [buf findLastOf: text fromOffset: 0x49];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    [buf RELEASE];
}

- (void) testFindText
{
    uint8_t* mapped;
    OLText* text;
    OLText* target;
    OLTextBuffer* buf;
    struct stat statBuf;
    OLFileInStream* finStream =
        REAP([OLFileInStream streamWithPath: "test/HamletComplete.utf16-le.txt"]);
    unsigned pos;
    olchar charsNotThere[] = { 0x000D, 0x000A, 0xFFFF, 0xFFFE, 0xFFFD, 0xFFFC, 0xFFFB, 0xFFFA };
    unsigned amountRead;

    if (stat("test/HamletComplete.utf16-le.txt", &statBuf) == -1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Count not stat test/HamletComplete.utf16-le.txt"];
        return;
    }
    mapped = (uint8_t*)malloc(statBuf.st_size);
    amountRead = [finStream readBytes: mapped count: statBuf.st_size];
    if (amountRead < statBuf.st_size)
    {
        free(mapped);
        [self errInFile: __FILE__ line: __LINE__
            format: "Couldn't read the entire file. Read %u or %u bytes.",
            amountRead, statBuf.st_size];
        return;
    }
    text = [[OLText alloc] initWithBytes: mapped + 2 count: statBuf.st_size - 2
        encoding: "UTF-16LE"];
    free(mapped);
    buf = REAP([OLTextBuffer textBufferWithText: text]);
    [text RELEASE];
    target = REAP([buf substrFromOffset: 0 count: 20]);
    pos = [buf findText: target fromOffset: 0];
    if (pos != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %u", pos];
    }
    target = REAP([buf substrFromOffset: [buf size] - 100 count: 100]);
    pos = [buf findText: target fromOffset: [buf size] / 2];
    if (pos != ([buf size] - 100))
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", [text size] - 100, pos];
    }
    target = REAP([buf substrFromOffset: 0x25674 / 2 count: 50]);
    pos = [buf findText: target fromOffset: 0];
    if (pos != (0x25674 / 2))
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", 0x25674 / 2, pos];
    }
    target = [[OLText alloc] initWithChars: charsNotThere count: 8];
    pos = [buf findText: target fromOffset: 0];
    if (pos != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, pos];
    }
    [target RELEASE];
}

- (void) testInsert
{
    OLTextBuffer* buf = REAP([OLTextBuffer textBufferWithText:
        REAP([OLText textWithCString: "dog"])]);
    OLTextBuffer* returned;
    OLText* lovely;
    olchar firstInsert[] =
    {
        OL_SMALL_A_CHAR,
        OL_SMALL_D_CHAR,
        OL_SMALL_O_CHAR,
        OL_SMALL_G_CHAR
    };
    olchar secondInsert[] =
    {
        OL_SMALL_A_CHAR,
        OL_SMALL_D_CHAR,
        OL_SMALL_O_CHAR,
        OL_SMALL_G_CHAR,
        OL_SMALL_G_CHAR,
        OL_SMALL_Y_CHAR
    };
    olchar thirdInsert[] =
    {
        OL_SMALL_A_CHAR,
        OL_SMALL_L_CHAR,
        OL_SMALL_O_CHAR,
        OL_SMALL_V_CHAR,
        OL_SMALL_E_CHAR,
        OL_SMALL_L_CHAR,
        OL_SMALL_Y_CHAR,
        OL_SMALL_D_CHAR,
        OL_SMALL_O_CHAR,
        OL_SMALL_G_CHAR,
        OL_SMALL_G_CHAR,
        OL_SMALL_Y_CHAR
    };
    int i;

    returned = [buf insertChar: OL_SMALL_A_CHAR atOffset: 0];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %u", [buf size]];
    }
    for (i = 0; i < 4; i++)
    {
        if ([buf at: i] != firstInsert[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "(%i) Expected \\u%0.4X, but got \\u%0.4X",
                i, firstInsert[i], [buf at: i]];
        }
    }
    returned = [buf insertChars: secondInsert + 4 atOffset: 4 count: 2];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 6)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 6, but got %u", [buf size]];
    }
    for (i = 0; i < 6; i++)
    {
        if ([buf at: i] != secondInsert[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "(%i) Expected \\u%0.4X, but got \\u%0.4X",
                i, secondInsert[i], [buf at: i]];
        }
    }
    lovely = [[OLText alloc] initWithChars: thirdInsert + 1 count: 6];
    returned = [buf insertText: lovely atOffset: 1];
    [lovely RELEASE];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 12)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 12, but got %u", [buf size]];
    }
    for (i = 0; i < 12; i++)
    {
        if ([buf at: i] != thirdInsert[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "(%i) Expected \\u%0.4X, but got \\u%0.4X",
                i, thirdInsert[i], [buf at: i]];
        }
    }
}

- (void) testInitializers
{
    OLTextBuffer* buf;
    OLText* fleas;

    buf = [[OLTextBuffer alloc] init];
    if (![buf IS_KIND_OF: [OLTextBuffer class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"OLTextBuffer\", but got \"%s\"",
            ((Class)[buf class])->name];
    }
    [self logMessage: "The default text buffer capacity is %u", [buf capacity]];
    if (![buf empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "the new buffer should be empty"];
    }
    [buf RELEASE];
    buf = [[OLTextBuffer alloc] initWithCapacity: 702];
    if (![buf IS_KIND_OF: [OLTextBuffer class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"OLTextBuffer\", but got \"%s\"",
            ((Class)[buf class])->name];
    }
    if ([buf capacity] != 702)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 702, but got %u", [buf capacity]];
    }
    [buf RELEASE];
    fleas = REAP([OLText textWithCString: "My dog has fleas"]);
    buf = [[OLTextBuffer alloc] initWithText: fleas];
    if (![buf IS_KIND_OF: [OLTextBuffer class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"OLTextBuffer\", but got \"%s\"",
            ((Class)[buf class])->name];
    }
    if (![REAP([buf text]) isEqual: fleas])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            [fleas cString], [REAP([buf text]) cString]];
    }
    [buf RELEASE];
}

- (void) testProperties
{
    OLTextBuffer* buf = REAP([OLTextBuffer textBuffer]);
    olchar i;

    if (![buf empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The text buffer should be empty"];
    }
    for (i = 0; i < 10000; i++)
        [buf appendChar: i];
    if ([buf length] != 10000)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 10000, but got %u", [buf length]];
    }
    if ([buf size] != 10000)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 10000, but got %u", [buf size]];
    }
}

- (void) testReplace
{
    OLTextBuffer* buf = REAP([OLTextBuffer textBuffer]);
    OLTextBuffer* returned;
    OLText* replacement;
    olchar baseChars[] =
    {
        OL_SMALL_A_CHAR,
        OL_SMALL_F_CHAR,
        OL_SMALL_I_CHAR,
        OL_SMALL_N_CHAR,
        OL_SMALL_E_CHAR,
        OL_SMALL_D_CHAR,
        OL_SMALL_O_CHAR,
        OL_SMALL_G_CHAR
    };
    olchar uglyChars[] =
    {
        OL_SMALL_N_CHAR,
        OL_SMALL_U_CHAR,
        OL_SMALL_G_CHAR,
        OL_SMALL_L_CHAR,
        OL_SMALL_Y_CHAR
    };
    olchar bigChars[] =
    {
        OL_SMALL_B_CHAR,
        OL_SMALL_I_CHAR,
        OL_SMALL_G_CHAR
    };
    olchar godChars[] =
    {
        OL_SMALL_G_CHAR,
        OL_SMALL_O_CHAR,
        OL_SMALL_D_CHAR
    };
    olchar horseChars[] =
    {
        OL_SMALL_H_CHAR,
        OL_SMALL_O_CHAR,
        OL_SMALL_R_CHAR,
        OL_SMALL_S_CHAR,
        OL_SMALL_E_CHAR
    };
    olchar messChars[] =
    {
        OL_SMALL_M_CHAR,
        OL_SMALL_E_CHAR,
        OL_SMALL_S_CHAR,
        OL_SMALL_S_CHAR
    };
    olchar buttChars[] =
    {
        OL_SMALL_B_CHAR,
        OL_SMALL_U_CHAR,
        OL_SMALL_T_CHAR,
        OL_SMALL_T_CHAR
    };
    olchar digglyChars[] =
    {
        OL_SMALL_D_CHAR,
        OL_SMALL_I_CHAR,
        OL_SMALL_G_CHAR,
        OL_SMALL_G_CHAR,
        OL_SMALL_L_CHAR,
        OL_SMALL_Y_CHAR
    };
    int i;

    [buf appendChars: baseChars fromOffset: 0 count: 8];
    [self logMessage: "%s", [REAP([buf text]) cString]];
    replacement = [[OLText alloc] initWithChars: uglyChars count: 5];
    returned = [buf replaceFromOffset: 1 count: 4 withText: replacement];
    [self logMessage: "%s", [REAP([buf text]) cString]];
    [replacement RELEASE];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 9)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 9, but got %u", [buf size]];
    }
    if ([buf at: 0] != baseChars[0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \\u%0.4X, but got \\u%0.4X",
            baseChars[0], [buf at: 0]];
    }
    for (i = 0; i < 5; i++)
    {
        if ([buf at: i + 1] != uglyChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                uglyChars[i], [buf at: i + 1]];
        }
    }
    for (i = 5; i < 8; i++)
    {
        if ([buf at: i + 1] != baseChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                baseChars[i], [buf at: i + 1]];
        }
    }
    replacement = [[OLText alloc] initWithChars: bigChars count: 3];
    returned = [buf replaceFromOffset: 1 count: 5 withText: replacement];
    [self logMessage: "%s", [REAP([buf text]) cString]];
    [replacement RELEASE];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 7)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 7, but got %u", [buf size]];
    }
    if ([buf at: 0] != baseChars[0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \\u%0.4X, but got \\u%0.4X",
            baseChars[0], [buf at: 0]];
    }
    for (i = 0; i < 3; i++)
    {
        if ([buf at: i + 1] != bigChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                bigChars[i], [buf at: i + 1]];
        }
    }
    for (i = 5; i < 8; i++)
    {
        if ([buf at: i - 1] != baseChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                baseChars[i], [buf at: i - 1]];
        }
    }
    replacement = [[OLText alloc] initWithChars: godChars count: 3];
    returned = [buf replaceFromOffset: 1 count: 3 withText: replacement];
    [self logMessage: "%s", [REAP([buf text]) cString]];
    [replacement RELEASE];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 7)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 7, but got %u", [buf size]];
    }
    if ([buf at: 0] != baseChars[0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \\u%0.4X, but got \\u%0.4X",
            baseChars[0], [buf at: 0]];
    }
    for (i = 0; i < 3; i++)
    {
        if ([buf at: i + 1] != godChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                godChars[i], [buf at: i + 1]];
        }
    }
    for (i = 5; i < 8; i++)
    {
        if ([buf at: i - 1] != baseChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                baseChars[i], [buf at: i - 1]];
        }
    }
    [buf clear];
    [buf appendChars: baseChars fromOffset: 0 count: 8];
    [self logMessage: "%s", [REAP([buf text]) cString]];
    replacement = [[OLText alloc] initWithChars: horseChars count: 5];
    returned = [buf replaceFromOffset: 5 count: 3 withText: replacement];
    [self logMessage: "%s", [REAP([buf text]) cString]];
    [replacement RELEASE];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 10)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 10, but got %u", [buf size]];
    }
    for (i = 0; i < 5; i++)
    {
        if ([buf at: i] != baseChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                baseChars[i], [buf at: i]];
        }
    }
    for (i = 0; i < 5; i++)
    {
        if ([buf at: i + 5] != horseChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                horseChars[i], [buf at: i + 5]];
        }
    }
    replacement = [[OLText alloc] initWithChars: messChars count: 4];
    returned = [buf replaceFromOffset: 5 count: 5 withText: replacement];
    [self logMessage: "%s", [REAP([buf text]) cString]];
    [replacement RELEASE];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 9)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 9, but got %u", [buf size]];
    }
    for (i = 0; i < 5; i++)
    {
        if ([buf at: i] != baseChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                baseChars[i], [buf at: i]];
        }
    }
    for (i = 0; i < 4; i++)
    {
        if ([buf at: i + 5] != messChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                messChars[i], [buf at: i + 5]];
        }
    }
    replacement = [[OLText alloc] initWithChars: buttChars count: 4];
    returned = [buf replaceFromOffset: 5 count: 4 withText: replacement];
    [self logMessage: "%s", [REAP([buf text]) cString]];
    [replacement RELEASE];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 9)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 9, but got %u", [buf size]];
    }
    for (i = 0; i < 5; i++)
    {
        if ([buf at: i] != baseChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                baseChars[i], [buf at: i]];
        }
    }
    for (i = 0; i < 4; i++)
    {
        if ([buf at: i + 5] != buttChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                buttChars[i], [buf at: i + 5]];
        }
    }
    replacement = [[OLText alloc] init];
    returned = [buf replaceFromOffset: 0 count: 5 withText: replacement];
    [self logMessage: "%s", [REAP([buf text]) cString]];
    [replacement RELEASE];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %u", [buf size]];
    }
    for (i = 0; i < 4; i++)
    {
        if ([buf at: i] != buttChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                buttChars[i], [buf at: i]];
        }
    }
    replacement = [[OLText alloc] initWithChars: digglyChars count: 6];
    returned = [buf replaceFromOffset: 0 count: 4 withText: replacement];
    [self logMessage: "%s", [REAP([buf text]) cString]];
    [replacement RELEASE];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 6)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 6, but got %u", [buf size]];
    }
    for (i = 0; i < 6; i++)
    {
        if ([buf at: i] != digglyChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                digglyChars[i], [buf at: i]];
        }
    }
}

- (void) testResize
{
    OLTextBuffer* buf = REAP([OLTextBuffer textBuffer]);
    OLTextBuffer* returned;
    olchar i;

    for (i = 0; i < 1000; i++)
        [buf appendChar: i];
    returned = [buf resize: 500 filledWith: 0xFFFF];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 500)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 500, but got %u", [buf size]];
    }
    for (i = 0; i < 500; i++)
    {
        if ([buf at: i] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                i, [buf at: i]];
        }
    }
    returned = [buf resize: 1000 filledWith: 0xFFFF];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    if ([buf size] != 1000)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1000, but got %u", [buf size]];
    }
    for (i = 0; i < 500; i++)
    {
        if ([buf at: i] != i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                i, [buf at: i]];
        }
    }
    for (i = 500; i < 1000; i++)
    {
        if ([buf at: i] != 0xFFFF)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                0xFFFF, [buf at: i]];
        }
    }
}

- (void) testReverse
{
    OLTextBuffer* buf = REAP([OLTextBuffer textBuffer]);
    OLTextBuffer* returned;
    unsigned i;

    for (i = 0; i < 0x10000; i++)
        [buf appendChar: i];
    returned = [buf reverse];
    if (returned != buf)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %p, but got %p", buf, returned];
    }
    for (i = 0; i < 0x10000; i++)
    {
        if ([buf at: i] != 0xFFFF - i)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "(%u) Expected \\u%0.4X, but got \\u%0.4X",
                i, 0xFFFF - i, [buf at: i]];
        }
    }
}

- (void) testRFindChar
{
    OLTextBuffer* buf;
    int i;
    unsigned rc;

    buf = [[OLTextBuffer alloc] init];
    for (i = 0; i < 5000; i++)
    {
        if (i == 2501)
            [buf appendChar: 0x0500 + i - 1];
        else
            [buf appendChar: 0x0500 + i];
    }
    rc = [buf rfindChar: 0x0500 + 2500 fromOffset: UINT_MAX];
    if (rc != 2501)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2501, but got %u", rc];
    }
    rc = [buf rfindChar: 0x0500 + 2500 fromOffset: 2500];
    if (rc != 2500)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2500, but got %u", rc];
    }
    rc = [buf rfindChar: 0x0500 + 2500 fromOffset: 2499];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [buf rfindChar: 0x000A fromOffset: UINT_MAX];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [buf rfindChar: 0x0500 + 2500 fromOffset: 0];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [buf rfindChar: 0x000A fromOffset: 0];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    [buf RELEASE];
}

@end

