//
// $Id: GzipStreamTest.m,v 1.6 2007/03/06 20:42:21 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if defined(HAVE_ZLIB)

#import "GzipStreamTest.h"
#import <ObjectiveLib/GzipOutStream.h>
#import <ObjectiveLib/GzipInStream.h>
#import <ObjectiveLib/DataOutStream.h>
#import <ObjectiveLib/DataInStream.h>
#import <ObjectiveLib/FileOutStream.h>
#import <ObjectiveLib/FileInStream.h>
#if defined(OL_NO_OPENSTEP)
#import <ObjectiveLib/Reaper.h>
#endif
#import <limits.h>
#import <string.h>
#include <stdlib.h>
#import <unistd.h>
#import <sys/stat.h>
#include <time.h>

@implementation GzipStreamTest

- (void) testBasicGzipStreams
{
    uint8_t srcBuf[256];
    uint8_t destBuf[256];
    int i;
    OLDataOutStream* dout;
    OLGzipOutStream* zout;
    OLGzipInStream* zin;
    OLDataInStream* din;
    unsigned streamed;

    for (i = 0; i < 256; i++)
        srcBuf[i] = i;
    dout = [[OLDataOutStream alloc] init];
    zout = [[OLGzipOutStream alloc] initWithOutStream: dout];
    streamed = [zout writeBytes: srcBuf count: 256];
    if (streamed != 256)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 256, but got %u", streamed];
    }
    [zout RELEASE];
    din = [[OLDataInStream alloc] initWithBytes: [dout bytes] count: [dout count]];
    [dout RELEASE];
    zin = [[OLGzipInStream alloc] initWithInStream: din];
    [din RELEASE];
    streamed = [zin readBytes: destBuf count: 256];
    if (streamed != 256)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 256, but got %u", streamed];
    }
    streamed = [zin readBytes: destBuf count: 700];
    if (streamed != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected end-of-stream, but got %u", streamed];
    }
    [zin RELEASE];
    if (memcmp(srcBuf, destBuf, 256) != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The inflated data are wrong"];
    }
}

- (void) testComment
{
    OLGzipOutStream* zout;
    OLGzipInStream* zin;
    OLDataOutStream* dout;
    OLDataInStream* din;

    dout = [[OLDataOutStream alloc] init];
    zout = [[OLGzipOutStream alloc] initWithOutStream: dout];
    [zout setComment: "My dog has fleas"];
    [zout writeByte: 1];
    [zout RELEASE];
    din = [[OLDataInStream alloc] initWithBytes: [dout bytes] count: [dout count]];
    [dout RELEASE];
    zin = [[OLGzipInStream alloc] initWithInStream: din];
    [din RELEASE];
    if ([zin comment] == NULL ||
        strcmp([zin comment], "My dog has fleas"))
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"My dog has fleas\", but got \"%s\"",
            [zin comment]];
    }
    [zin RELEASE];
}

- (void) testCompatibilityIn
{
    OLFileOutStream* fout;
    OLFileInStream* fin;
    OLGzipInStream* zin;
    int i;
    uint8_t b;

    fout = [[OLFileOutStream alloc] initWithPath: "gzipin.testfile"];
    for (i = 0; i < 256; i++)
        [fout writeByte: i];
    [fout RELEASE];
    if (system("gzip -f gzipin.testfile") != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Couldn't gzip the data"];
        unlink("gzipin.testfile");
    }
    else
    {
        fin = [[OLFileInStream alloc] initWithPath: "gzipin.testfile.gz"];
        zin = [[OLGzipInStream alloc] initWithInStream: fin];
        [fin RELEASE];
        for (i = 0; i < 256; i++)
        {
            b = [zin readByte];
            if (b != i)
            {
                [self errInFile: __FILE__ line: __LINE__
                    format: "Expected %i, but got %u", i, b];
            }
        }
        [zin RELEASE];
        unlink("gzipin.testfile.gz");
    }
}

- (void) testCompatibilityOut
{
    OLFileOutStream* fout;
    OLGzipOutStream* zout;
    OLFileInStream* fin;
    int i;
    uint8_t b;

    fout = [[OLFileOutStream alloc] initWithPath: "gzipout.testfile.gz"];
    zout = [[OLGzipOutStream alloc] initWithOutStream: fout];
    [fout RELEASE];
    for (i = 0; i < 256; i++)
        [zout writeByte: i];
    [zout RELEASE];
    fout = [[OLFileOutStream alloc] initWithPath: "gzipout.testfile"];
    for (i = 0; i < 256; i++)
        [fout writeByte: i];
    [fout RELEASE];
    if (system("gzip -d -f gzipout.testfile.gz") != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Couldn't gzip the data"];
        unlink("gzipout.testfile.gz");
    }
    else
    {
        fin = [[OLFileInStream alloc] initWithPath: "gzipout.testfile"];
        for (i = 0; i < 256; i++)
        {
            b = [fin readByte];
            if (b != i)
            {
                [self errInFile: __FILE__ line: __LINE__
                    format: "Expected %i, but got %u", i, b];
            }
        }
        if ([fin readBytes: &b count: 1] != UINT_MAX)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Should be at end of file"];
        }
        [fin RELEASE];
        unlink("gzipout.testfile");
    }
}

- (void) testConvenienceAllocators
{
    OLDataOutStream* dout;
    OLGzipOutStream* zout;
    OLGzipInStream* zin;
    int val;

    dout = REAP([OLDataOutStream stream]);
    zout = REAP([OLGzipOutStream streamWithOutStream: dout]);
    [zout writeInt: 75];
    [zout close];
    zin = REAP([OLGzipInStream streamWithInStream:
        REAP([OLDataInStream streamWithBytes: [dout bytes] count: [dout count]])]);
    val = [zin readInt];
    if (val != 75)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 75, but got %i", val];
    }
}

- (void) testExtraFields
{
    OLGzipOutStream* zout;
    OLGzipInStream* zin;
    OLDataOutStream* dout;
    OLDataInStream* din;
    const uint8_t* mdData = (const uint8_t*)"My dog has fleas";
    const uint8_t* yvData = (const uint8_t*)"I love Yehudi Villa";
    const uint8_t* ef;
    unsigned fieldLength;

    dout = [[OLDataOutStream alloc] init];
    zout = [[OLGzipOutStream alloc] initWithOutStream: dout];
    [zout addExtraField: "MD" withBytes: mdData count: strlen((const char*)mdData)];
    [zout addExtraField: "YV" withBytes: yvData count: strlen((const char*)yvData)];
    [zout writeByte: 1];
    [zout RELEASE];
    din = [[OLDataInStream alloc] initWithBytes: [dout bytes] count: [dout count]];
    [dout RELEASE];
    zin = [[OLGzipInStream alloc] initWithInStream: din];
    [din RELEASE];
    ef = [zin extraField: "MD" count: &fieldLength];
    if (ef == NULL || fieldLength != strlen((const char*)mdData) ||
        memcmp(ef, mdData, fieldLength) != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The extra field was not carried through"];
    }
    ef = [zin extraField: "YV" count: &fieldLength];
    if (ef == NULL || fieldLength != strlen((const char*)yvData) ||
        memcmp(ef, yvData, fieldLength) != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The extra field was not carried through"];
    }
    ef = [zin extraField: "GG" count: &fieldLength];
    if (ef != NULL)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "An extra field that shouldn't have been found was found"];
    }
    [zin RELEASE];
}

- (void) testMarkModificationTime
{
    OLGzipOutStream* zout;
    OLGzipInStream* zin;
    OLDataOutStream* dout;
    OLDataInStream* din;
    time_t mark;
    time_t found;

    dout = [[OLDataOutStream alloc] init];
    zout = [[OLGzipOutStream alloc] initWithOutStream: dout];
    mark = time(NULL);
    [self logMessage: "Marking time at %s", ctime(&mark)];
    [zout markModificationTime];
    [zout writeByte: 1];
    [zout RELEASE];
    din = [[OLDataInStream alloc] initWithBytes: [dout bytes] count: [dout count]];
    [dout RELEASE];
    zin = [[OLGzipInStream alloc] initWithInStream: din];
    [din RELEASE];
    if ([zin modificationTime] != mark)
    {
        found = [zin modificationTime];
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %s, but got %s",
            ctime(&mark), ctime(&found)];
    }
    [zin RELEASE];
}

- (void) testOriginalFileName
{
    OLGzipOutStream* zout;
    OLGzipInStream* zin;
    OLDataOutStream* dout;
    OLDataInStream* din;
    struct stat statBuf;

    if (system("echo howdy > gziporig.testfile") != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Couldn't create the test file"];
    }
    else
    {
        statBuf.st_mtime = 0;
        stat("gziporig.testfile", &statBuf);
        dout = [[OLDataOutStream alloc] init];
        zout = [[OLGzipOutStream alloc] initWithOutStream: dout];
        [zout setOriginalFileName: "gziporig.testfile" includeModificationTime: YES];
        [zout writeByte: 1];
        [zout RELEASE];
        din = [[OLDataInStream alloc] initWithBytes: [dout bytes] count: [dout count]];
        [dout RELEASE];
        zin = [[OLGzipInStream alloc] initWithInStream: din];
        [din RELEASE];
        if ([zin originalFileName] == NULL ||
            strcmp([zin originalFileName], "gziporig.testfile") != 0)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"gziporig.testfile\", but got \"%s\"",
                [zin originalFileName]];
        }
        if ([zin modificationTime] != statBuf.st_mtime)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %u, but got %u",
                [zin modificationTime], statBuf.st_mtime];
        }
        [zin RELEASE];
        unlink("gziporig.testfile");
    }
}

@end

#endif
