//
// $Id: BoolVectorTest.m,v 1.17 2007/03/28 03:16:52 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "BoolVectorTest.h"
#import "Random.h"
#import <ObjectiveLib/Vector.h>
#import <ObjectiveLib/DataOutStream.h>
#import <ObjectiveLib/DataInStream.h>
#import <ObjectiveLib/ObjectOutStream.h>
#import <ObjectiveLib/ObjectInStream.h>
#import "Number.h"
#if defined(OL_NO_OPENSTEP)
#import <ObjectiveLib/Reaper.h>
#else
#import <Foundation/NSData.h>
#import <Foundation/NSArchiver.h>
#if defined(HAVE_KEYED_ARCHIVES)
#import <Foundation/NSKeyedArchiver.h>
#endif
#endif
#include <stdlib.h>
#if defined(__NEXT_RUNTIME__)
#import <objc/objc-class.h>
#else
#import <objc/objc.h>
#endif

@implementation BoolVectorTest

- (void) testAccessors
{
    OLBoolVector* v;

    v = [[OLBoolVector alloc] init];
    [v pushBack: YES];
    [v pushBack: NO];

    if (![v at: 0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected YES, but got NO"];
    }
    if ([v at: 1])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected NO, but got YES"];
    }
    if (![v front])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected YES, but got NO"];
    }
    if ([v back])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected NO, but got YES"];
    }
    [v RELEASE];
}

- (void) testAssign
{
    OLBoolVector* v;
    OLBoolVector* v2;
    int i;

    v = [[OLBoolVector alloc] initWithSize: 5 filledWithBool: NO];
    v2 = [v copy];
    [v assign: 2 filledWith: YES];
    if ([v size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %i", [v size]];
    }
    if (![v at: 0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected YES, but got NO"];
    }
    if (![v at: 1])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected YES, but got NO"];
    }
    [v assignFrom: REAP([v2 begin]) to: REAP([v2 end])];
    [v2 RELEASE];
    if ([v size] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %i", [v size]];
    }
    for (i = 0; i < 5; i++)
    {
        if ([v at: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected NO, but got YES"];
        }
    }
    [v assignAt: 2 value: YES];
    if (![v at: 2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected YES, but got NO"];
    }
    [v RELEASE];
}

#if !defined(OL_NO_OPENSTEP)
- (void) testCoding
{
    OLBoolVector* v1;
    OLBoolVector* v2;
    NSMutableData* data;
    NSArchiver* archiver;
    NSData* archData;
    int i;

    v1 = [[OLBoolVector alloc] init];
    data = [[NSMutableData alloc] initWithCapacity: 1000];
    archiver = [[NSArchiver alloc] initForWritingWithMutableData: data];
    [archiver encodeObject: v1];
    [archiver RELEASE];
    [self logMessage: "A boolean vector of 0 elements creates an archive of %u bytes",
        [data length]];
    v2 = [NSUnarchiver unarchiveObjectWithData: data];
    [data RELEASE];
    if (![v1 isEqual: v2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors should be equal"];
    }
#if defined(HAVE_KEYED_ARCHIVES)
    archData = [NSKeyedArchiver archivedDataWithRootObject: v1];
    [self logMessage: "A boolean vector of 0 elements creates a keyed archive of %u bytes",
        [archData length]];
    v2 = [NSKeyedUnarchiver unarchiveObjectWithData: archData];
    if (![v1 isEqual: v2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors should be equal"];
    }
#endif
    for (i = 0; i < 10000; i++)
        [v1 pushBack: (OLRandom() & 1) ? YES : NO];
    data = [[NSMutableData alloc] initWithCapacity: 1000];
    archiver = [[NSArchiver alloc] initForWritingWithMutableData: data];
    [archiver encodeObject: v1];
    [archiver RELEASE];
    [self logMessage: "A boolean vector of 10,000 elements creates an archive of %u bytes",
        [data length]];
    v2 = [NSUnarchiver unarchiveObjectWithData: data];
    [data RELEASE];
    if (![v1 isEqual: v2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors should be equal"];
    }
#if defined(HAVE_KEYED_ARCHIVES)
    archData = [NSKeyedArchiver archivedDataWithRootObject: v1];
    [self logMessage: "A boolean vector of 10,000 elements creates a keyed archive of %u bytes",
        [archData length]];
    v2 = [NSKeyedUnarchiver unarchiveObjectWithData: archData];
    if (![v1 isEqual: v2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors should be equal"];
    }
#endif
    [v1 RELEASE];
}
#endif

- (void) testConvenienceAllocators
{
    OLBoolVector* bv;
    OLBoolVector* bv2;

    bv = REAP([OLBoolVector boolVector]);
    if (![bv IS_MEMBER_OF: [OLBoolVector class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            ((Class)[OLBoolVector class])->name, ((Class)[bv class])->name];
    }
    if (![bv empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vector should be empty"];
    }
    [bv pushBack: YES];
    [bv pushBack: NO];
    [bv pushBack: YES];
    bv2 = REAP([OLBoolVector boolVectorFrom: REAP([bv begin]) to: REAP([bv end])]);
    if (![bv isEqual: bv2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors should be equal"];
    }
    [bv pushBack: NO];
    [bv pushBack: YES];
    bv2 = REAP([OLBoolVector boolVectorWithBoolVector: bv]);
    if (![bv isEqual: bv2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors should be equal"];
    }
    bv = REAP([OLBoolVector boolVectorWithCapacity: 7000]);
    if (![bv IS_MEMBER_OF: [OLBoolVector class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            ((Class)[OLBoolVector class])->name, ((Class)[bv class])->name];
    }
    if ([bv capacity] < 7000)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected at least 7000, but got %u", [bv capacity]];
    }
    if (![bv empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vector should be empty"];
    }
    bv = REAP([OLBoolVector boolVectorWithSize: 5234 filledWithBool: YES]);
    if (![bv IS_MEMBER_OF: [OLBoolVector class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            ((Class)[OLBoolVector class])->name, ((Class)[bv class])->name];
    }
    if ([bv size] != 5234)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5234, but got %u", [bv size]];
    }
}

- (void) testErase
{
    OLBoolVector* v;
    OLBitIterator* itor;
    OLBitIterator* itor2;
    int i;

    v = [[OLBoolVector alloc] init];
    [v pushBack: YES];
    [v pushBack: NO];
    [v popBack];
    if ([v size] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %i", [v size]];
    }
    if (![v at: 0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected YES, but got NO"];
    }
    itor = REAP([v begin]);
    [v insertAt: itor count: 4 filledWith: NO];
    itor = REAP([v end]);
    [v insertAt: itor count: 5 filledWith: NO];
    itor = REAP([v begin]);
    [itor advanceBy: 4];
    itor = REAP([v erase: itor]);
    if ([v size] != 9)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 9, but got %i", [v size]];
    }
    for (i = 0; i < 9; i++)
    {
        if ([v at: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected NO, but got YES at index %i", i];
        }
    }
    [v insertAt: itor count: 4 filledWith: YES];
    itor = REAP([v begin]);
    [itor advanceBy: 4];
    itor2 = [itor copy];
    [itor2 advanceBy: 4];
    REAP([v eraseFrom: itor to: itor2]);
    [itor2 RELEASE];
    if ([v size] != 9)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 9, but got %i", [v size]];
    }
    for (i = 0; i < 9; i++)
    {
        if ([v at: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected NO, but got YES at index %i", i];
        }
    }
    [v RELEASE];
}

- (void) testInsert
{
    OLBoolVector* v;
    OLBoolVector* v2;
    int i;
    OLBitIterator* itor;

    v = [[OLBoolVector alloc] init];
    itor = REAP([v insertAt: REAP([v begin]) value: YES]);
    if ([v size] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %i", [v size]];
    }
    if (![v at: 0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Wrong value at index 0"];
    }
    if (![itor isEqual: REAP([v begin])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The iterator should be equal to [v begin]"];
    }
    [v RELEASE];

    v = [[OLBoolVector alloc] init];
    [v insertAt: REAP([v begin]) count: 10000 filledWith: YES];
    if ([v size] != 10000)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size should be 10000, got %u", [v size]];
    }
    for (i = 0; i < 10000; i++)
    {
        if (![v at: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Did not get the right value at index %i", i];
        }
    }
    itor = REAP([v begin]);
    [itor advanceBy: 5];
    itor = REAP([v insertAt: itor value: NO]);
    if ([v size] != 10001)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size should be 10001, got %u", [v size]];
    }
    if ([[itor dereference] boolValue])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected NO, but got YES"];
    }
    [itor reverse];
    if (![[itor dereference] boolValue])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected YES, but got NO"];
    }
    [itor advanceBy: 2];
    if (![[itor dereference] boolValue])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected YES, but got NO"];
    }

    v2 = [[OLBoolVector alloc] init];
    [v2 pushBack: YES];
    [v2 pushBack: YES];
    if ([v2 size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, got %u", [v2 size]];
    }
    if (![v2 at: 0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected YES, got NO"];
    }
    if (![v2 at: 1])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected YES, got NO"];
    }

    [v insertAt: itor from: REAP([v2 begin]) to: REAP([v2 end])];
    [v2 RELEASE];
    if ([v size] != 10003)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 10003, got %u", [v size]];
    }
    itor = REAP([v begin]);
    [itor advanceBy: 6];
    if (![[itor dereference] boolValue])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected YES, got NO"];
    }
    [itor advance];
    if (![[itor dereference] boolValue])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected YES, got NO"];
    }
    [v RELEASE];
}

- (void) testInitializers
{
    OLBoolVector* v;
    OLBoolVector* v2;
    int i;

    v = [[OLBoolVector alloc] init];
    if ([v size] != 0 || [v capacity] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size and capacity should be 0."];
    }
    [v RELEASE];

    v = [[OLBoolVector alloc] initWithCapacity: 10];
    if ([v size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size should be 0."];
    }
    [v RELEASE];

    v = [[OLBoolVector alloc] initWithSize: 1 filledWithBool: YES];
    if ([v size] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size should be 1."];
    }
    if (![v at: 0])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Did not get the right value at index 0"];
    }
    [v RELEASE];

    v = [[OLBoolVector alloc] initWithSize: 100 filledWithBool: NO];
    if ([v size] != 100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size should be 100, got %u", [v size]];
    }
    for (i = 0; i < 100; i++)
    {
        if ([v at: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Did not get the right value at index %i", i];
        }
    }

    v2 = [[OLBoolVector alloc] initWithBoolVector: v];
    if ([v2 size] != 100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size should be 100, got %u", [v2 size]];
    }
    for (i = 0; i < 100; i++)
    {
        if ([v at: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Did not get the right value at index %i", i];
        }
    }
    if (![v isEqual: v2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors should be equal"];
    }
    [v2 RELEASE];

    v2 = [[OLBoolVector alloc] initFrom: REAP([v begin]) to: REAP([v end])];
    if ([v2 size] != 100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size should be 100, got %u", [v2 size]];
    }
    for (i = 0; i < 100; i++)
    {
        if ([v at: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Did not get the right value at index %i", i];
        }
    }
    if (![v isEqual: v2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors should be equal"];
    }
    [v2 RELEASE];
    [v RELEASE];
}

- (void) testIterators
{
    OLBoolVector* v;
    OLBitIterator* itor;
    OLBitIterator* begin;
    OLBitIterator* end;
    OLReverseRandomIterator* rbegin;
    OLReverseRandomIterator* rend;
    BOOL vals[] = { YES, NO, YES, NO, YES };
    int i;
    int diff;

    v = [[OLBoolVector alloc] init];
    for (i = 0; i < 5; i++)
        [v pushBack: vals[i]];
    for (itor = REAP([v begin]), end = REAP([v end]), i = 0;
         ![itor isEqual: end]; [itor advance], i++)
    {
        if ([[itor dereference] boolValue] != vals[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                vals[i], [[itor dereference] boolValue]];
        }
    }
    for ([itor reverse], i = 4; i >= 0; [itor reverse], i--)
    {
        if ([[itor dereference] boolValue] != vals[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                vals[i], [[itor dereference] boolValue]];
        }
    }
    [itor advanceBy: 2];
    begin = REAP([v begin]);
    diff = [itor difference: begin];
    if (diff != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %i", diff];
    }
    if ([[itor dereference] boolValue] != NO)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i",
            [[itor dereference] boolValue]];
    }
    [itor advanceBy: 2];
    [itor advanceBy: -3];
    diff = [itor difference: begin];
    if (diff != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i", diff];
    }
    if ([[itor dereference] boolValue] != YES)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %i, but got %i",
            YES, [[itor dereference] boolValue]];
    }
    if (![itor isEqual: begin])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The iterator should be equal to begin"];
    }
    for (rbegin = REAP([v rbegin]), rend = REAP([v rend]), i = 4;
         ![rbegin isEqual: rend];
         [rbegin advance], i--)
    {
        if ([[rbegin dereference] boolValue] != vals[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i",
                vals[i], [[itor dereference] boolValue]];
        }
    }
    [v RELEASE];
}

- (void) testProperties
{
    OLBoolVector* v;
    OLBoolVector* v2;
    BOOL vals[] = { YES, NO, YES, NO, YES };
    int i;

    v = [[OLBoolVector alloc] init];
    if (![v empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vector must report that it is empty"];
    }
    for (i = 0; i < 5; i++)
        [v pushBack: vals[i]];
    v2 = [[OLBoolVector alloc] init];
    for (i = 0; i < 5; i++)
        [v2 pushBack: vals[i]];
    if (![v isEqual: v2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors must be equal"];
    }
    if ([v size] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size must be exactly 5"];
    }
    if ([v empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vector cannot report that it is empty"];
    }
    [v clear];
    if (![v empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vector must report that it is empty"];
    }
    if ([v isEqual: v2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors must not be equal"];
    }
    if ([v maxSize] != [v2 maxSize])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "All vectors must have the same max size"];
    }
    [self logMessage: "The max size of a boolean vector is %u", [v maxSize]];
    [v RELEASE];
    [v2 RELEASE];
}

- (void) testReserveResizeSwap
{
    OLBoolVector* v;
    OLBoolVector* v2;
    OLBoolVector* vCopy;
    OLBoolVector* v2Copy;
    int nums[] = { YES, NO, YES, NO, YES };
    int i;

    v = [[OLBoolVector alloc] init];
    for (i = 0; i < 5; i++)
        [v pushBack: nums[i]];
    [v reserve: 1000];
    if ([v capacity] < 1000)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The capacity must be at least 1000, got %u", [v capacity]];
    }
    for (i = 0; i < 5; i++)
    {
        if ([v at: i] != nums[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i", nums[i], [v at: i]];
        }
    }
    v2 = [v copy];
    [v resize: 100 filledWith: NO];
    if ([v size] != 100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 100, but got %i", [v size]];
    }
    for (i = 5; i < 100; i++)
    {
        if ([v at: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected NO, but got YES"];
        }
    }
    vCopy = [v copy];
    v2Copy = [v2 copy];
    [v swap: v2];
    if (![v isEqual: v2Copy])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected the vectors to be equal"];
    }
    if (![v2 isEqual: vCopy])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected the vectors to be equal"];
    }
    [v RELEASE];
    [v2 RELEASE];
    [vCopy RELEASE];
    [v2Copy RELEASE];
}

- (void) testStreaming
{
    OLDataOutStream* dout;
    OLObjectOutStream* oout;
    OLObjectInStream* oin;
    OLBoolVector* vector;
    OLBoolVector* readVector;
    int i;

    dout = REAP([OLDataOutStream stream]);
    oout = REAP([OLObjectOutStream streamWithOutStream: dout]);
    vector = [[OLBoolVector alloc] init];
    for (i = 0; i< 10000; i++)
        [vector pushBack: (OLRandom() % 2) ? YES : NO];
    [oout writeObject: vector];
    oin = REAP([OLObjectInStream streamWithInStream:
        REAP([OLDataInStream streamWithBytes: [dout bytes] count: [dout count]])]);
    readVector = REAP([oin readObject]);
    if (![readVector IS_MEMBER_OF: [OLBoolVector class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected OLBoolVector class, but got %s", ((Class)[readVector class])->name];
    }
    if (![readVector isEqual: vector])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors should be equal"];
    }
    [vector RELEASE];
}

@end
