//
// $Id: Queue.m,v 1.16 2007/03/06 20:42:20 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "Queue.h"
#import "Algorithm.h"
#import "Macros.h"
#import "Deque.h"
#import "VectorPackage.h"
#import "ObjectInStream.h"
#import "ObjectOutStream.h"
#if !defined(OL_NO_OPENSTEP)
#import <Foundation/NSCoder.h>
#import <Foundation/NSString.h>
#endif

#if !defined(OL_NO_OPENSTEP)
NSString* const PREDICATE_KEY = @"OL_PREDICATE_KEY";
NSString* const VECTOR_KEY = @"OL_VECTOR_KEY";
NSString* const DEQUE_KEY = @"OL_DEQUE_KEY";
#endif

@implementation OLQueue

+ (id) queue
{
    OL_BEGIN_AUTO_CTOR(OLQueue)
        init
    OL_END_AUTO_CTOR;
}

+ (id) queueWithQueue: (OLQueue*)right
{
    OL_BEGIN_AUTO_CTOR(OLQueue)
        initWithQueue: right
    OL_END_AUTO_CTOR;
}

- (id) init
{
    [super init];
    deque = [[OLDeque alloc] init];
    return self;
}


#if !defined(OL_NO_OPENSTEP)
- (id) initWithCoder: (NSCoder*)decoder
{
    [super init];
    if ([decoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [decoder allowsKeyedCoding])
    {
        deque = OBJ_RETAIN_AUTO([decoder decodeObjectForKey: DEQUE_KEY]);
    }
    else
    {
        deque = OBJ_RETAIN_AUTO([decoder decodeObject]);
    }
    return self;
}
#endif

- (id) initWithObjectInStream: (OLObjectInStream*)stream
{
    [super init];
    deque = OBJ_RETAIN_AUTO([stream readObject]);
    return self;
}

- (id) initWithQueue: (OLQueue*)queue
{
    [super init];
    deque = [queue->deque copy];
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    OBJ_RELEASE(deque);
    SUPER_FREE;
}

- (id) back
{
    return [deque back];
}

- (int) compare: (id)other
{
    if (IS_KIND_OF(other, OLQueue))
        return [deque compare: ((OLQueue*)other)->deque];
    return -1;
}

#if defined(OL_NO_OPENSTEP)

- (id) copy
{
    return [[OLQueue alloc] initWithQueue: self];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLQueue allocWithZone: zone] initWithQueue: self];
}

#endif

- (BOOL) empty
{
    return [deque empty];
}

#if !defined(OL_NO_OPENSTEP)
- (void) encodeWithCoder: (NSCoder*)encoder
{
    if ([encoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [encoder allowsKeyedCoding])
    {
        [encoder encodeObject: deque forKey: DEQUE_KEY];
    }
    else
    {
        [encoder encodeObject: deque];
    }
}
#endif

- (id) front
{
    return [deque front];
}

- (BOOL) isEqual: (id)object
{
    return IS_KIND_OF(object, OLQueue) &&
           [((OLQueue*)object)->deque isEqual: deque];
}

- (void) pop
{
    [deque popFront];
}

- (void) push: (id)object
{
    [deque pushBack: object];
}

- (unsigned) size
{
    return [deque size];
}

- (void) writeSelfToStream: (OLObjectOutStream*)stream
{
    [stream writeObject: deque];
}

@end

@implementation OLPriorityQueue

+ (id) priorityQueue
{
    OL_BEGIN_AUTO_CTOR(OLPriorityQueue)
        init
    OL_END_AUTO_CTOR;
}

+ (id) priorityQueueFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OL_BEGIN_AUTO_CTOR(OLPriorityQueue)
        initFrom: first to: last
    OL_END_AUTO_CTOR;
}

+ (id) priorityQueueFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last predicate: (OLStreamableFunctor<OLBoolBinaryFunction>*)pred
{
    OL_BEGIN_AUTO_CTOR(OLPriorityQueue)
        initFrom: first to: last predicate: pred
    OL_END_AUTO_CTOR;
}

+ (id) priorityQueueWithPredicate: (OLStreamableFunctor<OLBoolBinaryFunction>*)pred
{
    OL_BEGIN_AUTO_CTOR(OLPriorityQueue)
        initWithPredicate: pred
    OL_END_AUTO_CTOR;
}

+ (id) priorityQueueWithPriorityQueue: (OLPriorityQueue*)right
{
    OL_BEGIN_AUTO_CTOR(OLPriorityQueue)
        initWithPriorityQueue: right
    OL_END_AUTO_CTOR;
}

- (id) init
{
    OLLess* less = [[OLLess alloc] init];

    [self initWithPredicate: less];
    OBJ_RELEASE(less);
    return self;
}

- (id) initFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OLLess* less = [[OLLess alloc] init];

    [self initFrom: first to: last predicate: less];
    OBJ_RELEASE(less);
    return self;
}

- (id) initFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last predicate: (OLStreamableFunctor<OLBoolBinaryFunction>*)pred
{
    OLArrayIterator* begin;
    OLArrayIterator* end;

    [super init];
    predicate = OBJ_RETAIN(pred);
    vector = [[OLVector alloc] initFrom: first to: last];
    begin = [vector beginImpl];
    end = [vector endImpl];
    [OLAlgorithm makeHeapFrom: begin to: end predicate: predicate];
    OBJ_RELEASE(begin);
    OBJ_RELEASE(end);
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (id) initWithCoder: (NSCoder*)decoder
{
    [super init];
    if ([decoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [decoder allowsKeyedCoding])
    {
        predicate = OBJ_RETAIN_AUTO([decoder decodeObjectForKey: PREDICATE_KEY]);
        vector = OBJ_RETAIN_AUTO([decoder decodeObjectForKey: VECTOR_KEY]);
    }
    else
    {
        predicate = OBJ_RETAIN_AUTO([decoder decodeObject]);
        vector = OBJ_RETAIN_AUTO([decoder decodeObject]);
    }
    return self;
}
#endif

- (id) initWithObjectInStream: (OLObjectInStream*)stream
{
    [super init];
    predicate = OBJ_RETAIN_AUTO([stream readObject]);
    vector = OBJ_RETAIN_AUTO([stream readObject]);
    return self;
}

- (id) initWithPredicate: (OLStreamableFunctor<OLBoolBinaryFunction>*)pred
{
    [super init];
    predicate = OBJ_RETAIN(pred);
    vector = [[OLVector alloc] init];
    return self;
}

- (id) initWithPriorityQueue: (OLPriorityQueue*)right
{
    [super init];
    predicate = OBJ_RETAIN(right->predicate);
    vector = [right->vector copy];
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    OBJ_RELEASE(vector);
    OBJ_RELEASE(predicate);
    SUPER_FREE;
}

- (int) compare: (id)other
{
    if (IS_KIND_OF(other, OLPriorityQueue))
        return [vector compare: ((OLPriorityQueue*)other)->vector];
    return -1;
}

#if defined(OL_NO_OPENSTEP)

- (id) copy
{
    return [[OLPriorityQueue alloc] initWithPriorityQueue: self];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLPriorityQueue allocWithZone: zone] initWithPriorityQueue: self];
}

#endif

- (BOOL) empty
{
    return [vector empty];
}

#if !defined(OL_NO_OPENSTEP)
- (void) encodeWithCoder: (NSCoder*)encoder
{
    if ([encoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [encoder allowsKeyedCoding])
    {
        [encoder encodeObject: predicate forKey: PREDICATE_KEY];
        [encoder encodeObject: vector forKey: VECTOR_KEY];
    }
    else
    {
        [encoder encodeObject: predicate];
        [encoder encodeObject: vector];
    }
}
#endif

- (BOOL) isEqual: (id)object
{
    return IS_KIND_OF(object, OLPriorityQueue) &&
           [((OLPriorityQueue*)object)->vector isEqual: vector];
}

- (void) pop
{
    OLArrayIterator* begin = [vector beginImpl];
    OLArrayIterator* end = [vector endImpl];

    [OLAlgorithm popHeapFrom: begin to: end predicate: predicate];
    OBJ_RELEASE(begin);
    OBJ_RELEASE(end);
    [vector popBack];
}

- (void) push: (id)object
{
    OLArrayIterator* begin;
    OLArrayIterator* end;

    [vector pushBack: object];
    begin = [vector beginImpl];
    end = [vector endImpl];
    [OLAlgorithm pushHeapFrom: begin to: end predicate: predicate];
    OBJ_RELEASE(begin);
    OBJ_RELEASE(end);
}

- (unsigned) size
{
    return [vector size];
}

- (id) top
{
    return [vector front];
}

- (void) writeSelfToStream: (OLObjectOutStream*)stream
{
    [stream writeObject: predicate];
    [stream writeObject: vector];
}

@end
