//
// $Id: ObjectInStream.m,v 1.18 2007/03/25 18:12:15 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "ObjectInStream.h"
#import "InStreamPackage.h"
#import "Macros.h"
#import "HashMap.h"
#import "RunTime.h"
#import "Streamable.h"
#import "SharedPointerTable.h"
#import "HashFunction.h"
#import "Exception.h"
#if defined(OL_NO_OPENSTEP)
#import "Text.h"
#else
#import <Foundation/NSString.h>
#import <Foundation/NSException.h>
#import <Foundation/NSData.h>
#endif
#import <string.h>
#import <stdlib.h>
#import <limits.h>

#if defined(OL_NO_OPENSTEP)
OLConstantString* const OLClassNotFoundException = @"OLClassNotFoundException";
#else
NSString* const OLClassNotFoundException = @"OLClassNotFoundException";
#endif

#define FREE_MY_RESOURCES \
    [self close]; \
    OBJ_RELEASE(sharedPointers); \
    OBJ_RELEASE(classes)

@interface OLClassName :
#if defined(OL_NO_OPENSTEP)
    Object
#else
    NSObject
#endif
{
@private
    char* name;
}

- (id) initWithName: (const char*)nm;
#if defined(OL_NO_OPENSTEP)
- (id) free;
#else
- (void) dealloc;
#endif
- (int) compare: (id)other;
#if defined(OL_NO_OPENSTEP)
- (id) copy;
#else
- (id) copyWithZone: (NSZone*)zone;
#endif
- (unsigned) hash;

@end

@interface OLInteger :
#if defined(OL_NO_OPENSTEP)
    Object
#else
    NSObject
#endif
{
@private
    int value;
}

- (id) initWithInteger: (int)val;
#if defined(OL_NO_OPENSTEP)
- (id) copy;
#else
- (id) copyWithZone: (NSZone*)zone;
#endif
- (int) value;

@end

@interface OLObjectInStream (PrivateMethods)

- (const char*) lookUpTypeName: (uint8_t)type;
- (void) verifyType: (uint8_t)type;

@end

#if defined(OL_NO_OPENSTEP)
@protocol OLObjectInStreamUnknownMethods

- (id) awakeAfterUsingCoder: (id)coder;
- (id) initWithCoder: (id)coder;

@end
#endif

@implementation OLObjectInStream

+ (id) streamWithInStream: (OLInStream*)underStream
{
    OL_BEGIN_AUTO_CTOR(OLObjectInStream)
        initWithInStream: underStream
    OL_END_AUTO_CTOR;
}

- (id) initWithInStream: (OLInStream*)underStream
{
    [super initWithInStream: underStream];
    classes = [[OLHashMap alloc] init];
    sharedPointers = [[OLSharedPointerTable alloc] init];
    [self readHeader];
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (void) dealloc
{
	FREE_MY_RESOURCES;
    SUPER_FREE;
}
#endif

- (unsigned) classVersion: (const char*)className
{
    OLClassName* nameObj = [[OLClassName alloc] initWithName: className];
    id num = [classes valueForKey: nameObj];

    OBJ_RELEASE(nameObj);
    return (num == nil) ? UINT_MAX : [num value];
}

#if !defined(OL_NO_OPENSTEP)
- (void*) decodeBytesWithReturnedLength: (unsigned*)numBytes
{
    [self verifyType: WIRE_TYPE_BYTE_BLOCK];
    return [stream decodeBytesWithReturnedLength: numBytes];
}

- (NSData*) decodeDataObject
{
    [self verifyType: WIRE_TYPE_DATA];
    return [stream decodeDataObject];
}

- (id) decodeObject
{
    return [self readObject];
}

- (void) decodeValueOfObjCType: (const char*)valueType at: (void*)address
{
    uint8_t type;

    switch (*valueType)
    {
    case _C_CHR:
    case _C_UCHR:
        [self verifyType: WIRE_TYPE_CHAR];
        [stream decodeValueOfObjCType: valueType at: address];
        break;
    case _C_SHT:
    case _C_USHT:
        [self verifyType: WIRE_TYPE_SHORT];
        [stream decodeValueOfObjCType: valueType at: address];
        break;
    case _C_INT:
    case _C_UINT:
        *(unsigned*)address = [self readInt];
        break;
    case _C_LNG:
    case _C_ULNG:
        [self verifyType: WIRE_TYPE_LONG];
        [stream decodeValueOfObjCType: valueType at: address];
        break;
    case _C_LNG_LNG:
    case _C_ULNG_LNG:
        [self verifyType: WIRE_TYPE_LONG_LONG];
        [stream decodeValueOfObjCType: valueType at: address];
        break;
    case _C_FLT:
        *(float*)address = [self readFloat];
        break;
    case _C_DBL:
        *(double*)address = [self readDouble];
        break;
    case _C_ID:
        *(id*)address = [self readObject];
        break;
    case _C_CLASS:
        *(Class*)address = [self readClass];
        break;
    case _C_SEL:
        type = [stream readByte];
        if (type == WIRE_TYPE_SELECTOR)
        {
            [stream decodeValueOfObjCType: valueType at: address];
            [sharedPointers addPointer: *(void**)address];
        }
        else if (type == WIRE_TYPE_HANDLE)
        {
            *(SEL*)address = [sharedPointers lookUp: [stream readInt32]];
        }
        else
        {
            RAISE_EXCEPTION(OLInputOutputException,
                @"Found unexpected type in stream - \"%s\"",
                [self lookUpTypeName: type]);
        }
        break;
    case _C_CHARPTR:
        [self verifyType: WIRE_TYPE_CHARPTR];
        [stream decodeValueOfObjCType: valueType at: address];
        break;
    case _C_ARY_B:
        [self verifyType: WIRE_TYPE_ARRAY];
        [stream decodeValueOfObjCType: valueType at: address];
        break;
    case _C_PTR:
        [self verifyType: WIRE_TYPE_POINTER];
        [stream decodeValueOfObjCType: valueType at: address];
        break;
    case _C_STRUCT_B:
        [self verifyType: WIRE_TYPE_STRUCTURE];
        [stream decodeValueOfObjCType: valueType at: address];
        break;
    default:
        [stream decodeValueOfObjCType: valueType at: address];
    }
}
#endif

#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources
{
	FREE_MY_RESOURCES;
	[super freeStreamResources];
}
#endif

- (BOOL) readBool
{
    [self verifyType: WIRE_TYPE_BOOL];
    return [stream readBool];
}

- (Class) readClass
{
    char* nameBuf;
    unsigned nameBufLen = 30;
    uint16_t len;
    int version;
    Class cls;
    Class origCls = nil;
    OLInteger* versionObj;
    OLClassName* name;
    uint8_t type;

    [self verifyType: WIRE_TYPE_CLASS];
    type = [stream readByte];
    if (type == WIRE_TYPE_HANDLE)
    {
        origCls = [sharedPointers lookUp: [stream readInt32]];
    }
    else
    {
        if (type != WIRE_TYPE_CLASS_NAME)
        {
            RAISE_EXCEPTION(OLInputOutputException,
                @"Expected a type of \"%s\" but found \"%s\"",
                [self lookUpTypeName: WIRE_TYPE_CLASS_NAME], [self lookUpTypeName: type]);
        }
        nameBuf = objc_malloc(nameBufLen);
        do
        {
            len = [stream readInt16];
            if (nameBufLen < len + 1)
            {
                nameBufLen = len + 1;
                objc_free(nameBuf);
                nameBuf = objc_malloc(nameBufLen);
            }
            [self completelyReadBytes: (uint8_t*)nameBuf count: len];
            nameBuf[len] = 0;
            cls = objc_get_class(nameBuf);
            if (cls == nil)
            {
                objc_free(nameBuf);
                RAISE_EXCEPTION(OLClassNotFoundException,
                    @"The class \"%s\" is not registered", nameBuf);
            }
            if (origCls == nil)
                origCls = cls;
            [sharedPointers addPointer: cls];
            version = [stream readInt];
            name = [[OLClassName alloc] initWithName: nameBuf];
            versionObj = [[OLInteger alloc] initWithInteger: version];
            [classes assignKey: name value: versionObj];
            OBJ_RELEASE(name);
            OBJ_RELEASE(versionObj);
            type = [stream readByte];
            if (type != WIRE_TYPE_CLASS_NAME && type != WIRE_TYPE_END_CLASS)
            {
                objc_free(nameBuf);
                RAISE_EXCEPTION(OLInputOutputException,
                    @"Found unexpected type in stream - \"%s\"",
                    [self lookUpTypeName: type]);
            }
        } while (type == WIRE_TYPE_CLASS_NAME);
        objc_free(nameBuf);
    }
    return origCls;
}

- (double) readDouble
{
    [self verifyType: WIRE_TYPE_DOUBLE];
    return [stream readDouble];
}

- (float) readFloat
{
    [self verifyType: WIRE_TYPE_FLOAT];
    return [stream readFloat];
}

- (void) readHeader
{
    if ([stream readInt32] != OL_STREAM_MAGIC)
    {
        RAISE_EXCEPTION(OLInputOutputException,
            @"The magic number indentifying the stream is incorrect");
    }
    systemVersion = [stream readInt32];
}

- (unsigned) readInt
{
    [self verifyType: WIRE_TYPE_INT];
    return [stream readInt];
}

- (uint16_t) readInt16
{
    [self verifyType: WIRE_TYPE_INT16];
    return [stream readInt16];
}

- (uint32_t) readInt32
{
    [self verifyType: WIRE_TYPE_INT32];
    return [stream readInt32];
}

- (uint64_t) readInt64
{
    [self verifyType: WIRE_TYPE_INT64];
    return [stream readInt64];
}

- (id) readObject
{
    uint8_t type;
    id object = nil;
    Class cls;
    BOOL initWithStream = YES;

    [self verifyType: WIRE_TYPE_INSTANCE];
    type = [stream readByte];
    if (type == WIRE_TYPE_NIL)
    {
        // object is already nil
    }
    else if (type == WIRE_TYPE_HANDLE)
    {
        object = [sharedPointers lookUp: [stream readInt32]];
    }
    else if (type == WIRE_TYPE_OBJECT_DATA)
    {
        cls = [self readClass];
#if defined(OL_NO_OPENSTEP)
        if ([cls instancesRespondTo: @selector(initWithObjectInStream:)])
#else
        if ([cls instancesRespondToSelector: @selector(initWithObjectInStream:)])
#endif
        {
            initWithStream = YES;
        }
#if defined(OL_NO_OPENSTEP)
        else if ([cls instancesRespondTo: @selector(initWithCoder:)])
#else
        else if ([cls instancesRespondToSelector: @selector(initWithCoder:)])
#endif
        {
            initWithStream = NO;
        }
        else
        {
            RAISE_EXCEPTION(OLInputOutputException,
                @"Instances of the class \"%s\" do not respond to either initWithObjectInStream: or initWithCoder:", cls->name);
        }
        object = [cls alloc];
        object = initWithStream ?
            [object initWithObjectInStream: self] :
            [object initWithCoder: self];
        if (RESPONDS_TO(object, @selector(awakeAfterUsingCoder:)))
            object = [object awakeAfterUsingCoder: self];
        [sharedPointers addObject: object];
        object = OBJ_AUTORELEASE(object);
    }
    else
    {
        RAISE_EXCEPTION(OLInputOutputException,
            @"Found unexpected type in stream - \"%s\"",
            [self lookUpTypeName: type]);
    }
    return object;
}

- (unsigned) systemVersion
{
    return systemVersion;
}

#if !defined(OL_NO_OPENSTEP)
- (unsigned) versionForClassName: (NSString*)className
{
    OLClassName* nameObj = [[OLClassName alloc] initWithName: [className UTF8String]];
    id num = [classes valueForKey: nameObj];

    OBJ_RELEASE(nameObj);
    return (num == nil) ? NSNotFound : [num value];
}
#endif

@end

@implementation OLObjectInStream (PrivateMethods)

#define OL_TYPE_NAME_ENTRY(t,n) \
    if (type == WIRE_TYPE_##t) \
        return #n
        
- (const char*) lookUpTypeName: (uint8_t)type
{
    OL_TYPE_NAME_ENTRY(HANDLE, handle);
    OL_TYPE_NAME_ENTRY(CLASS, class);
    OL_TYPE_NAME_ENTRY(END_CLASS, end class);
    OL_TYPE_NAME_ENTRY(STRING, string);
    OL_TYPE_NAME_ENTRY(INSTANCE, id);
    OL_TYPE_NAME_ENTRY(NIL, nil);
    OL_TYPE_NAME_ENTRY(DATA, data object);
    OL_TYPE_NAME_ENTRY(BOOL, BOOL);
    OL_TYPE_NAME_ENTRY(DOUBLE, double);
    OL_TYPE_NAME_ENTRY(FLOAT, float);
    OL_TYPE_NAME_ENTRY(INT, unsigned int);
    OL_TYPE_NAME_ENTRY(INT16, uint16_t);
    OL_TYPE_NAME_ENTRY(INT32, uint32_t);
    OL_TYPE_NAME_ENTRY(INT64, uint64_t);
    OL_TYPE_NAME_ENTRY(CHAR, unsigned char);
    OL_TYPE_NAME_ENTRY(SHORT, unsigned short);
    OL_TYPE_NAME_ENTRY(LONG, unsigned long);
    OL_TYPE_NAME_ENTRY(LONG_LONG, unsigned long long);
    OL_TYPE_NAME_ENTRY(SELECTOR, SEL);
    OL_TYPE_NAME_ENTRY(CHARPTR, char*);
    OL_TYPE_NAME_ENTRY(ARRAY, array);
    OL_TYPE_NAME_ENTRY(POINTER, void*);
    OL_TYPE_NAME_ENTRY(NULL, NULL);
    OL_TYPE_NAME_ENTRY(STRUCTURE, structure);
    OL_TYPE_NAME_ENTRY(OBJECT_DATA, object data);
    OL_TYPE_NAME_ENTRY(CLASS_NAME, class name);
    return "unknown";
}

- (void) verifyType: (uint8_t)type
{
    uint8_t foundType = [stream readByte];

    if (foundType != type)
    {
        RAISE_EXCEPTION(OLInputOutputException,
            @"Expected to read a value of type \"%s\", but found a type of \"%s\"",
            [self lookUpTypeName: type], [self lookUpTypeName: foundType]);
    }
}

@end

@implementation OLClassName

- (id) initWithName: (const char*)nm
{
    [super init];
    name = objc_malloc(strlen(nm) + 1);
    strcpy(name, nm);
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    objc_free(name);
    SUPER_FREE;
}

- (int) compare: (id)other
{
    if (!IS_KIND_OF(other, OLClassName))
        return -1;
    return strcmp(name, ((OLClassName*)other)->name);
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    return [[OLClassName alloc] initWithName: name];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLClassName allocWithZone: zone] initWithName: name];
}
#endif

- (unsigned) hash
{
    return OLHash((const uint8_t*)name, strlen(name));
}

@end

@implementation OLInteger

- (id) initWithInteger: (int)val
{
    [super init];
    value = val;
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    return [[OLInteger alloc] initWithInteger: value];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLInteger allocWithZone: zone] initWithInteger: value];
}
#endif

- (int) value
{
    return value;
}

@end
