//
// $Id: GzipInStream.m,v 1.10 2007/03/06 20:42:19 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "GzipInStream.h"
#import "Map.h"
#import "Macros.h"
#import "RunTime.h"
#import "ByteOrder.h"
#import "InStreamPackage.h"
#import "Exception.h"
#if defined(OL_NO_OPENSTEP)
#import "Text.h"
#else
#import <Foundation/NSString.h>
#import <Foundation/NSException.h>
#endif
#import <zlib.h>
#import <string.h>
#import <limits.h>
#import <stdlib.h>

#define FREE_MY_RESOURCES \
    objc_free(fileName); \
    objc_free(comment); \
    OBJ_RELEASE(extraFields)

@interface OLExtraFieldID :
#if defined(OL_NO_OPENSTEP)
    Object
#else
    NSObject
#endif
{
@private
    char identifier[2];
}

- (id) initWithID: (const char*)ident;
- (int) compare: (id)other;
#if defined(OL_NO_OPENSTEP)
- (id) copy;
#else
- (id) copyWithZone: (NSZone*)zone;
#endif

@end

@interface OLExtraFieldBuffer :
#if defined(OL_NO_OPENSTEP)
    Object
#else
    NSObject
#endif
{
@private
    uint8_t*    buffer;
    unsigned    count;
}

- (id) initWithBuffer: (uint8_t*)buf count: (unsigned)cnt;
#if defined(OL_NO_OPENSTEP)
- (id) free;
#else
- (void) dealloc;
#endif
- (const uint8_t*) buffer;
#if defined(OL_NO_OPENSTEP)
- (id) copy;
#else
- (id) copyWithZone: (NSZone*)zone;
#endif
- (unsigned) count;

@end

@interface OLGzipInStream (PrivateMethods)

- (void) readExtraFieldsWithCRC: (unsigned long*)lcrc;
- (void) readHeader;
- (char*) readHeaderStringWithCRC: (unsigned long*)lcrc;
- (uint16_t) readLE16WithCRC: (unsigned long*)lcrc;
- (uint32_t) readLE32WithCRC: (unsigned long*)lcrc;
- (void) readTrailer;

@end

@implementation OLGzipInStream

+ (id) streamWithInStream: (OLInStream*)underStream
{
    OL_BEGIN_AUTO_CTOR(OLGzipInStream)
        initWithInStream: underStream
    OL_END_AUTO_CTOR;
}

- (id) initWithInStream: (OLInStream*)underStream bufferSize: (unsigned)size readZlibHeader: (BOOL)zlibHeader
{
    [super initWithInStream: underStream bufferSize: size readZlibHeader: NO];
    fileName = NULL;
    comment = NULL;
    extraFields = nil;
    modTime = 0;
    crc = crc32(0, Z_NULL, 0);
    endOfStream = NO;
    [self readHeader];
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (void) dealloc
{
	FREE_MY_RESOURCES;
	SUPER_FREE;
}
#endif

- (const char*) comment
{
    return comment;
}

- (const uint8_t*) extraField: (const char*)identifier count: (unsigned*)countOut
{
    OLExtraFieldID* idObj;
    OLExtraFieldBuffer* found;

    *countOut = 0;
    if (extraFields == nil)
        return NULL;
    if (strlen(identifier) != 2)
    {
        RAISE_EXCEPTION(OLInputOutputException, @"Illegal extra field identifier");
    }
    idObj = [[OLExtraFieldID alloc] initWithID: identifier];
    found = [extraFields valueForKey: idObj];
    OBJ_RELEASE(idObj);
    if (found != nil)
    {
        *countOut = [found count];
        return [found buffer];
    }
    return NULL;
}

#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources
{
	FREE_MY_RESOURCES;
	[super freeStreamResources];
}
#endif

- (uint32_t) modificationTime
{
    return modTime;
}

- (const char*) originalFileName
{
    return fileName;
}

- (unsigned) readBytes: (uint8_t*)dest count: (unsigned)max
{
    unsigned numRead;

    if (endOfStream)
        return UINT_MAX;
    numRead = [super readBytes: dest count: max];
    if (numRead == UINT_MAX)
    {
        [self readTrailer];
        endOfStream = YES;
    }
    else
    {
        crc = crc32(crc, dest, numRead);
    }
    return numRead;
}

@end

@implementation OLGzipInStream (PrivateMethods)

- (void) readExtraFieldsWithCRC: (unsigned long*)lcrc
{
    uint16_t xlen = [self readLE16WithCRC: lcrc];
    uint16_t total = 0;
    char idBuf[2];
    OLExtraFieldBuffer* data;
    OLExtraFieldID* key;
    uint16_t dataLen;
    uint8_t* dataBuf = objc_malloc(1024);
    unsigned dataBufSize = 1024;

    extraFields = [[OLMap alloc] init];
    while (total < xlen)
    {
        [stream completelyReadBytes: (uint8_t*)idBuf count: 2];
        *lcrc = crc32(*lcrc, (Bytef*)idBuf, 2);
        total += 2;
        dataLen = [self readLE16WithCRC: lcrc];
        total += 2;
        if (dataLen > dataBufSize)
        {
            objc_free(dataBuf);
            dataBuf = objc_malloc(dataLen);
            dataBufSize = dataLen;
        }
        [stream completelyReadBytes: dataBuf count: dataLen];
        *lcrc = crc32(*lcrc, dataBuf, dataLen);
        total += dataLen;
        key = [[OLExtraFieldID alloc] initWithID: idBuf];
        data = [[OLExtraFieldBuffer alloc] initWithBuffer: dataBuf count: dataLen];
        [extraFields assignKey: key value: data];
        OBJ_RELEASE(key);
        OBJ_RELEASE(data);
    }
    objc_free(dataBuf);
}

- (void) readHeader
{
    uint8_t flags;
    unsigned long hcrc = crc32(0, Z_NULL, 0);
    uint8_t hbuf[] = { GZ_ID1, GZ_ID2, Z_DEFLATED };
    uint16_t fcrc;

    if ([stream readByte] != GZ_ID1)
    {
        RAISE_EXCEPTION(OLInputOutputException, @"Data not in gzip format");
    }
    if ([stream readByte] != GZ_ID2)
    {
        RAISE_EXCEPTION(OLInputOutputException, @"Data not in gzip format");
    }
    if ([stream readByte] != Z_DEFLATED)
    {
        RAISE_EXCEPTION(OLInputOutputException, @"Unsupported compression method in gzip data");
    }
    hcrc = crc32(hcrc, hbuf, 3);
    flags = [stream readByte];
    hcrc = crc32(hcrc, &flags, 1);
    modTime = [self readLE32WithCRC: &hcrc];
    hbuf[0] = [stream readByte];
    hbuf[1] = [stream readByte];
    hcrc = crc32(hcrc, hbuf, 2);
    if (flags & GZ_FEXTRA)
        [self readExtraFieldsWithCRC: &hcrc];
    if (flags & GZ_FNAME)
        fileName = [self readHeaderStringWithCRC: &hcrc];
    if (flags & GZ_FCOMMENT)
        comment = [self readHeaderStringWithCRC: &hcrc];
    if (flags & GZ_FHCRC)
    {
        fcrc = [self readLE16WithCRC: NULL];
        if (fcrc != (hcrc & 0xFFFF))
        {
            RAISE_EXCEPTION(OLInputOutputException, @"Corrupt gzip header");
        }
    }
}

- (char*) readHeaderStringWithCRC: (unsigned long*)lcrc
{
    unsigned size = 0;
    unsigned capacity = 100;
    uint8_t* buf = objc_malloc(capacity);
    uint8_t* newBuffer;
    char* str;
    uint8_t b = [stream readByte];

    *lcrc = crc32(*lcrc, &b, 1);
    while (b != 0)
    {
        if (size == capacity)
        {
            capacity = capacity * 2;
            newBuffer = objc_malloc(capacity);
            memcpy(newBuffer, buf, size);
            objc_free(buf);
            buf = newBuffer;
        }
        buf[size++] = b;
        b = [stream readByte];
        *lcrc = crc32(*lcrc, &b, 1);
    }
    str = objc_malloc(size + 1);
    memcpy(str, buf, size);
    str[size] = 0;
    objc_free(buf);
    return str;
}

- (uint16_t) readLE16WithCRC: (unsigned long*)lcrc
{
    uint16_t unswapped;

    [stream completelyReadBytes: (uint8_t*)&unswapped count: 2];
    if (lcrc != NULL)
        *lcrc = crc32(*lcrc, (uint8_t*)&unswapped, 2);
    return L16_TO_H(unswapped);
}

- (uint32_t) readLE32WithCRC: (unsigned long*)lcrc
{
    uint32_t unswapped;

    [stream completelyReadBytes: (uint8_t*)&unswapped count: 4];
    if (lcrc != NULL)
        *lcrc = crc32(*lcrc, (uint8_t*)&unswapped, 4);
    return L32_TO_H(unswapped);
}

- (void) readTrailer
{
    uint8_t trailerBuf[8];
    unsigned remainder = 8 - zstream->avail_in;

    if (remainder < 8)
        memcpy(trailerBuf, zstream->next_in, zstream->avail_in);
    [stream completelyReadBytes: trailerBuf + zstream->avail_in count: remainder];
    if (L32_TO_H(*(uint32_t*)trailerBuf) != crc)
    {
        RAISE_EXCEPTION(OLInputOutputException,
            @"The CRC of the uncompressed data is inconsistent");
    }
    if (L32_TO_H(*(uint32_t*)(trailerBuf + 4)) != zstream->total_out)
    {
        RAISE_EXCEPTION(OLInputOutputException,
            @"The number of uncompressed bytes is inconsistent");
    }
}

@end

@implementation OLExtraFieldID

- (id) initWithID: (const char*)ident
{
    [super init];
    identifier[0] = ident[0];
    identifier[1] = ident[1];
    return self;
}

- (int) compare: (id)other
{
    if (!IS_KIND_OF(other, OLExtraFieldID))
        return -1;
    return strncmp(identifier, ((OLExtraFieldID*)other)->identifier, 2);
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    return [[OLExtraFieldID alloc] initWithID: identifier];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLExtraFieldID allocWithZone: zone] initWithID: identifier];
}
#endif

@end

@implementation OLExtraFieldBuffer

- (id) initWithBuffer: (uint8_t*)buf count: (unsigned)cnt
{
    [super init];
    buffer = objc_malloc(cnt);
    memcpy(buffer, buf, cnt);
    count = cnt;
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    objc_free(buffer);
    SUPER_FREE;
}

- (const uint8_t*) buffer
{
    return buffer;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    return [[OLExtraFieldBuffer alloc] initWithBuffer: buffer count: count];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLExtraFieldBuffer allocWithZone: zone] initWithBuffer: buffer count: count];
}
#endif

- (unsigned) count
{
    return count;
}

@end

