/**
 * @file libcomprex/archive_io.c Archive I/O Functions
 * 
 * $Id: archive_io.c,v 1.12 2003/01/02 02:49:38 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <stdio.h>
#include <libcomprex/internal.h>

CxArchive *
cxOpenArchive(const char *path)
{
	CxArchive *archive;

	if (path == NULL)
		return NULL;

	cxOpenArchiveOrFile(path, CX_MODE_READ_ONLY, NULL, &archive);

	return archive;
}

CxArchive *
cxOpenArchiveStream(FILE *fp)
{
	CxArchive *archive;

	if (fp == NULL)
		return NULL;

	cxInternalOpenStream(fp, CX_MODE_READ_ONLY, NULL, &archive);

	return archive;
}

CxArchive *
cxOpenArchiveFromHandle(CxFile *file, CxAccessMode mode)
{
	CxFP *fp;
	CxArchive *archive;

	if (file == NULL)
		return NULL;

	if ((fp = cxOpenFileHandle(file, mode)) == NULL)
		return NULL;

	cxOpenArchiveOrFile2(fp, CX_MODE_READ_ONLY, NULL, &archive);

	return archive;
}

CxArchive *
cxOpenArchiveBuffer(char *buffer, size_t size)
{
	CxArchive *archive;

	if (buffer == NULL || size == 0)
		return NULL;

	cxInternalOpenBuffer(buffer, size, CX_MODE_READ_ONLY, NULL, &archive);

	return archive;
}

void
cxCloseArchive(CxArchive *archive)
{
	if (archive == NULL)
		return;

	cxDestroyArchive(archive);
}

CxStatus
cxSaveArchive(CxArchive *archive, const char *path)
{
	const char *ext;
	char *basename;
	CxFP *fp;
	CxModule *module;
	CxStatus status;

	if (archive == NULL)
		return CX_ERROR;

	if (path == NULL)
	{
		if (cxGetArchiveFileName(archive) == NULL)
			return CX_ERROR;

		path = cxGetArchiveFileName(archive);
	}

	ext = strrchr(path, '.');

	if (ext == NULL)
		return CX_NOT_SUPPORTED;

	ext++;

	/* Let's find a module that can save this. */
	for (module = cxGetFirstModule(CX_MODULE_ARCHIVE);
		 module != NULL;
		 module = module->next)
	{
		if (module->ops.archive->supportsExtension(ext))
		{
			cxSetArchiveModule(archive, module);
			break;
		}
	}

	if (archive->module == NULL)
		return CX_NOT_SUPPORTED;

	/* Try to open the file for writing. */
	fp = cxOpenFile(path, CX_MODE_READ_WRITE);

	if (fp == NULL)
		return CX_ERROR;

	/* Set info about the archive. */
	basename = cxGetBaseName(path);
	cxSetArchiveFileName(archive, basename);
	free(basename);

	cxSetArchivePhysicalPath(archive, path);
	cxSetArchivePath(archive,         path);

	cxSetArchiveLocal(archive, 1);

	/* Write the archive. */
	status = archive->module->ops.archive->saveArchive(archive, fp);

	/* TODO: Set info about the file size and archive size. */

	/* Close up. */
	cxClose(fp);

	return status;
}

CxStatus
cxExtractArchive(CxArchive *archive)
{
	CxFsIterator *iter;
	CxFsNode *node;
	size_t fileCount, onFile = 0;

	if (archive == NULL)
		return CX_ERROR;

	iter = cxNewFsIterator(archive, CX_FSITER_FILES_DIRS);
	fileCount = cxGetArchiveFileCount(archive);

	if (iter == NULL)
		return CX_ERROR;

	for (node = cxGetFsIterFirst(iter);
		 node != NULL;
		 node = cxGetFsIterNext(iter))
	{
		if (cxGetFsNodeType(node) == CX_FSNODETYPE_DIRECTORY)
		{
			cxMakePhysDirs(cxGetDirPath(node), archive);
		}
		else if (cxGetFsNodeType(node) == CX_FSNODETYPE_FILE)
		{
			/* Do you invoke my wrath?! */
			if (archive->ex_callback != NULL)
				archive->ex_callback(archive, node, ++onFile, fileCount);

			cxInternalExtractFile(node, cxGetFsNodePath(node));
		}
	}

	return CX_SUCCESS;
}

