/*-
 * Copyright (c) 2005 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipastat_limits.c,v 1.2.2.1 2011/11/15 18:12:29 simon Exp $";
#endif /* !lint */

#include <sys/types.h>

#include <limits.h>
#include <stdlib.h>
#include <string.h>

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "memfunc.h"

#include "ipastat_log.h"
#include "ipastat_main.h"
#include "ipastat_rules.h"
#include "ipastat_st.h"

char		has_opt_limits = 0;	/* Set if -q ... -l ... */

#ifdef WITH_LIMITS

signed char	dynamic_limits;		/* dynamic_limits parameter. */

ipa_mzone	*limit_mzone;		/* Mzone for struct limit{}. */

struct limit *
alloc_limit(void)
{
	struct limit *limit;

	limit = mzone_alloc(limit_mzone);
	if (limit == NULL) {
		xlogmsgx(IPA_LOG_ERR, "alloc_limit: mzone_alloc failed");
		return (NULL);
	}
	limit->info = NULL;
	limit->st_list = NULL;
	limit->inited = 0;
	return (limit);
}

/*
 * Return pointer to limit with the given name.
 */
struct limit *
limit_by_name(const struct rule *rule, const char *name)
{
	struct limit *limit;

	STAILQ_FOREACH(limit, &rule->limits, link)
		if (strcmp(name, limit->name) == 0)
			break;
	return (limit);
}

static void
get_min_max_no(const struct opt_limits *opt_limits, unsigned int *minno,
    unsigned int *maxno)
{
	const struct limit *limit;
	const struct opt_limit *opt_limit;

	*minno = UINT_MAX;
	*maxno = 0;
	STAILQ_FOREACH(opt_limit, opt_limits, link) {
		limit = opt_limit->limit;
		if (*minno > limit->no)
			*minno = limit->no;
		if (*maxno < limit->no)
			*maxno = limit->no;
	}
}

/*
 * Initialize all limits from one rule in modules.
 */
int
init_limits(const struct opt_rule *opt_rule)
{
	const struct rule *rule;
	const struct opt_limit *opt_limit;
	const struct opt_limits *opt_limits;
	struct limit *limit;
	unsigned int minno, maxno, no;

	rule = opt_rule->rule;
	opt_limits = &opt_rule->opt_limits;

	get_min_max_no(opt_limits, &minno, &maxno);

	for (no = minno; no <= maxno; ++no)
		STAILQ_FOREACH(opt_limit, &opt_rule->opt_limits, link) {
			limit = opt_limit->limit;
			if (limit->no == no) {
				if (!limit->inited) {
					limit->inited = 1;
					if (st_init_limit(rule, limit) < 0) {
						logbt("init_limits");
						return (-1);
					}
				}
				break;
			}
		}

	return (0);
}

int
deinit_limits(const struct opt_rule *opt_rule)
{
	const struct rule *rule;
	const struct opt_limit *opt_limit;
	const struct opt_limits *opt_limits;
	struct limit *limit;
	unsigned int minno, maxno, no;
	int rv;

	rule = opt_rule->rule;
	opt_limits = &opt_rule->opt_limits;

	get_min_max_no(opt_limits, &minno, &maxno);

	rv = 0;
	for (no = minno; no <= maxno; ++no)
		STAILQ_FOREACH(opt_limit, &opt_rule->opt_limits, link) {
			limit = opt_limit->limit;
			if (limit->no == no) {
				if (limit->inited) {
					limit->inited = 0;
					if (st_deinit_limit(rule, limit) < 0) {
						logbt("deinit_limits");
						rv = -1;
					}
				}
				break;
			}
		}

	return (rv);
}

/*
 * Copy all limits from list_src to rule.
 */
int
copy_limits(struct rule *rule, const struct limits_list *list)
{
	const struct limit *lsrc;
	struct limit *ldst;

	STAILQ_FOREACH(lsrc, list, link) {
		ldst = mzone_alloc(limit_mzone);
		if (ldst == NULL) {
			xlogmsgx(IPA_LOG_ERR, "copy_limits: "
			    "mzone_alloc failed");
			return (-1);
		}

		/* Copy settings from source limit. */
		*ldst = *lsrc;

		ldst->free_mask = 0;

		/* Link just allocated limit to rule. */
		STAILQ_INSERT_TAIL(&rule->limits, ldst, link);
	}
	return (0);
}

/*
 * Release memory used by a list of limits.
 */
void
free_limits(struct limits_list *limits)
{
	struct limit *limit, *limit_next;

	STAILQ_FOREACH_SAFE(limit, limits, link, limit_next) {
		if (limit->free_mask & LIMIT_FREE_NAME)
			mem_free(limit->name, m_anon);
		mem_free(limit->info, m_result);
		mzone_free(limit_mzone, limit);
	}
}

/*
 * Add one optional limit given in the -q -r option.
 */
int
opt_limit_add(const char *name)
{
	struct opt_limit *opt_limit;

	opt_limit = mem_malloc(sizeof(*opt_limit), m_anon);
	if (opt_limit == NULL) {
		logmsgx(IPA_LOG_ERR, "opt_limit_add: mem_malloc failed");
		return (-1);
	}

	opt_limit->name = name;
	opt_limit->opt_st = cur_opt_st;
	opt_limit->data = NULL;

	cur_opt_rule->type = OPT_RULE_LIMIT;

	STAILQ_INSERT_TAIL(&cur_opt_rule->opt_limits, opt_limit, link);

	has_opt_limits = 1;
	return (0);
}

int
opt_limits_parse(const struct opt_rule *opt_rule)
{
	struct opt_limit *opt_limit;
	struct rule *rule;
	struct limit *limit;
	unsigned int limitno;

	if (STAILQ_EMPTY(&opt_rule->opt_limits))
		return (0);

	rule = opt_rule->rule;
	if (STAILQ_EMPTY(&rule->limits))
		limitno = 0;
	else {
		limit = STAILQ_LAST(&rule->limits, limit, link);
		limitno = limit->no + 1;
	}

	STAILQ_FOREACH(opt_limit, &opt_rule->opt_limits, link) {
		limit = limit_by_name(rule, opt_limit->name);
		if (limit == NULL) {
			/* This limit is not given in the configuration file. */
			if (!dynamic_limits) {
				logmsgx(IPA_LOG_ERR, "opt_limits_parse: "
				    "unknown rule %s, limit %s", rule->name,
				    opt_limit->name);
				return (-1);
			}
			limit = alloc_limit();
			if (limit == NULL) {
				logbt("opt_limits_parse");
				return (-1);
			}
			STAILQ_INSERT_TAIL(&rule->limits, limit, link);
			limit->no = limitno++;
			limit->name = (char *)opt_limit->name;
			limit->free_mask = 0;
		}
		opt_limit->limit = limit;
		if (opt_limit->opt_st != NULL)
			limit->st_list = opt_limit->opt_st->st_list;
		if (limit->st_list == NULL)
			limit->st_list = rule->st_list;
	}

	return (0);
}

void
opt_limits_free(const struct opt_limits *opt_limits)
{
	struct opt_limit *opt_limit, *opt_limit_next;

	STAILQ_FOREACH_SAFE(opt_limit, opt_limits, link, opt_limit_next)
		mem_free(opt_limit, m_anon);
}
#endif /* WITH_LIMITS */
