/*-
 * Copyright (c) 2003 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *  @(#)$Id: ipa_time.h,v 1.3 2011/01/23 18:42:35 simon Exp $
 */

#ifndef IPA_TIME_H
#define IPA_TIME_H

/* update_time parameter default value. */
#define UPDATE_TIME_DEF		(1 * SECONDS_IN_MINUTE)

/* wakeup_time parameter default value. */
#define WAKEUP_TIME_DEF		(10 * SECONDS_IN_MINUTE)

/* sensitive_time parameter default value. */
#define SENSITIVE_TIME_DEF	(SECONDS_IN_MINUTE / 2)

#define TIME_TO_SEC(tm) (			\
	(tm)->tm_hour * SECONDS_IN_HOUR +	\
	(tm)->tm_min * SECONDS_IN_MINUTE +	\
	(tm)->tm_sec				\
)

/* Mark event as not scheduled. */
#define EVENT_NOT_SCHEDULED	(1000000)

/*
 * If time of scheduled event is less than current time
 * no more than FAKE_TIME_DELTA, then fake current time.
 * To disable this feature, set this macro to zero.
 */
#ifndef FAKE_TIME_DELTA
# define FAKE_TIME_DELTA	5
#endif

/*
 * One time event.
 */
struct tevent {
	SLIST_ENTRY(tevent) link;
	unsigned int	event_sec;	/* Next event time. */
	unsigned int	event_step;	/* Interval between events. */
};

#define TEVENT_NSIZE	10
#define TEVENT_NALLOC	10

/*
 * List of all time events.
 */
SLIST_HEAD(tevents_list, tevent);

/*
 * One time interval in "worktime" parameter in seconds
 * starting from midnight.
 */
struct tint {
	STAILQ_ENTRY(tint) link;
	unsigned int	sec1;		/* From sec1. */
	unsigned int	sec2;		/* Till sec2. */
};

#define TINT_NSIZE	20
#define TINT_NALLOC	20

/*
 * List of all time intervals for one day.
 */
STAILQ_HEAD(tint_list, tint);

/*
 * List of all tint_lists.
 */
struct tint_set {
	STAILQ_ENTRY(tint_set) link;	/* For list of all tint_sets. */
	struct tint_list list;		/* One tint_list. */
};

STAILQ_HEAD(tint_sets, tint_set);

#define WT_FLAG_ACTIVE		0x1	/* Worktime is active. */

#define WT_IS_ACTIVE(x)		((x)->wt_flags == WT_FLAG_ACTIVE)
#define WT_IS_INACTIVE(x)	(!WT_IS_ACTIVE(x))

#define WT_SET_ACTIVE(x)	((x)->wt_flags = WT_FLAG_ACTIVE)
#define WT_SET_INACTIVE(x)	((x)->wt_flags = 0)

/*
 * One worktime.
 */
struct worktime {
	SLIST_ENTRY(worktime) link;	/* For list building. */
	const struct tint_list *tint_list[DAYS_IN_WEEK];
	const struct tint *curtint;	/* Current time interval. */
	unsigned int	active_sec;	/* Time when to make wt active. */
	unsigned int	inactive_sec;	/* Time when to make wt inactive. */
	unsigned int	wt_flags;	/* 0 or WT_FLAG_ACTIVE. */
};

#define WORKTIME_NSIZE	5
#define WORKTIME_NALLOC	5

/*
 * List of all worktimes.
 */
SLIST_HEAD(worktimes_list, worktime);

#ifdef WITH_LIMITS
/*
 * [+<X>] <time> or <time> +<X> like parameter value.
 */
struct texp {
	unsigned int	seconds;	/* Number of seconds. */
	char		side;		/* 0 if +X <time>, 1 if <time> +X. */
	char		upto;		/* TEXP_UPTO_xxx. */
};

#define TEXP_UPTO_NOTSET	'-'
#define TEXP_UPTO_SIMPLE	' '
#define TEXP_UPTO_MINUTE	'm'
#define TEXP_UPTO_HOUR		'h'
#define TEXP_UPTO_DAY		'D'
#define TEXP_UPTO_WEEK		'W'
#define TEXP_UPTO_MONTH		'M'

#endif /* WITH_LIMITS */

/* Buffer length for representing seconds as time. */
#define SEC_STR_SIZE	(sizeof("xx:xx:xx"))

/* Buffer length for representing uint64_t seconds. */
#define TIME_STR_SIZE	17

/* Buffer length for representing date and time in struct tm{}. */
#define TM_STR_SIZE	(sizeof("xxxx.xx.xx/xx:xx:xx") + 10)

extern unsigned int wakeup_time;
extern unsigned int freeze_time;
extern unsigned int sleep_after_dump;
extern unsigned int sensitive_time;
extern signed char debug_time;
extern signed char debug_worktime;

extern const struct worktime *global_worktime;
extern struct worktime worktime_default;

extern const struct tevent *global_update_tevent;
extern const struct tevent *global_append_tevent;

extern const char *const wdays[];

extern const char *const active_msg[];

extern ipa_mzone *tint_mzone;
extern struct tint_sets tint_sets;

extern ipa_mzone *worktime_mzone;
extern struct worktimes_list worktimes_list;

extern ipa_mzone *tevent_mzone;
extern struct tevents_list tevents_list;

extern time_t	curtime;
extern ipa_tm	curdate, curdate_new;
extern unsigned int cursec, cursec_new;
extern unsigned int curwday;
extern char	newday_flag;

extern void	sec_to_time(unsigned int, ipa_tm *);
extern const char *time_str(unsigned int);
extern const char *sec_str(unsigned int);
extern const char *tint_str(const struct tint *);

extern void	free_tint_set(struct tint_set *);
extern void	free_worktimes(void);
extern void	init_worktime_default(void);
extern const struct worktime *find_worktime(struct worktime *);

extern unsigned int worktimes_check_sec;

extern void	worktimes_newday(int);
extern void	worktimes_check(void);

#ifdef WITH_LIMITS
extern int	set_wday(ipa_tm *);
extern void	ipa_tm_texp(ipa_tm *, const struct texp *);
#endif

#ifdef WITH_THRESHOLDS
extern void	ipa_tm_sub(ipa_tm *, unsigned int);
#endif

#ifdef WITH_ANY_LIMITS
extern void	fix_240000(ipa_tm *);
extern int	cmp_ipa_tm(const ipa_tm *, const ipa_tm *);
extern int	check_ipa_tm(const ipa_tm *);
extern unsigned int ipa_tm_diff(const ipa_tm *, const ipa_tm *);
extern const char *tm_str(const ipa_tm *);
extern const char *tm_str2(const ipa_tm *);
extern int	check_worktime_subset(const struct worktime *,
		    const struct worktime *);
#endif

extern int	init_time_values(void);
extern int	new_time_values(void);

#endif /* !IPA_TIME_H */
