package io

import (
	"bytes"
	"fmt"
	"io"
	"text/template"

	"git.sr.ht/~charles/rq/util"
)

func init() {
	registerOutputHandler("template", func() OutputHandler { return &TemplateOutputHandler{} })
}

// Declare conformance with OutputHandler interface.
var _ OutputHandler = &TemplateOutputHandler{}

// TemplateOutputHandler handles serializing Template data.
type TemplateOutputHandler struct {
	initialized bool
	templ       string
	sep         string
}

func (t *TemplateOutputHandler) init() {
	if t.initialized {
		return
	}

	t.templ = "no template provided (HINT: did you forget the output.template option on your output dataspec?)"
	t.sep = "\n"
	t.initialized = true
}

// Name implements OutputHandler.Name().
func (t *TemplateOutputHandler) Name() string {
	return "template"
}

// SetOption implements OutputHandler.SetOption().
func (t *TemplateOutputHandler) SetOption(name string, value string) error {
	t.init()

	if name == "template.sep" {
		u, err := util.Unescape(value)
		if err == nil {
			value = u
		}
	}

	switch name {
	case "output.template":
		t.templ = value
	case "output.sep":
		t.sep = value
	}

	return nil
}

// Format implements OutputHandler.Format()
func (t *TemplateOutputHandler) Format(writer io.Writer, data interface{}) error {
	var err error

	t.init()

	if dataL, ok := data.([]interface{}); ok {
		for i, v := range dataL {
			err := t.Format(writer, v)
			if err != nil {
				return err
			}
			if i < (len(dataL) - 1) {
				_, err := writer.Write([]byte("\n"))
				if err != nil {
					return err
				}
			}
		}

		_, err := writer.Write([]byte("\n"))
		return err
	}

	if dataL, ok := data.([]map[string]interface{}); ok {
		for i, v := range dataL {
			err := t.Format(writer, v)
			if err != nil {
				return err
			}
			if i < (len(dataL) - 1) {
				_, err := writer.Write([]byte(t.sep))
				if err != nil {
					return err
				}
			}
		}

		_, err := writer.Write([]byte("\n"))
		return err
	}

	dataM, ok := data.(map[string]interface{})
	if !ok {
		return fmt.Errorf("template: data of type %T cannot be used as the data for the template output handler", data)
	}

	templ, err := template.New("").Funcs(templateFuncs).Parse(t.templ)
	if err != nil {
		return err
	}

	buf := &bytes.Buffer{}
	err = templ.Execute(buf, dataM)
	if err != nil {
		return err
	}

	_, err = writer.Write(buf.Bytes())

	return err

}
