package io

import (
	"encoding/csv"
	"io"

	"git.sr.ht/~charles/rq/util"
)

func init() {
	registerOutputHandler("csv", func() OutputHandler { return &CSVOutputHandler{} })

	// aliases
	registerOutputHandler("tsv", func() OutputHandler {
		return &CSVOutputHandler{
			comma:       '\t',
			headers:     true,
			initialized: true,
		}
	})
}

// Declare conformance with OutputHandler interface.
var _ OutputHandler = &CSVOutputHandler{}

// CSVOutputHandler handles serializing CSVs.
//
// The following options are supported:
//
// csv:comma (rune) the comma-character to use (default: ,)
//
// csv:headers (bool) if true, a header row will be generated in situations
// where it is possible to do so (default: true)
type CSVOutputHandler struct {
	comma       rune
	initialized bool
	headers     bool
}

func (c *CSVOutputHandler) init() {
	if c.initialized {
		return
	}

	c.comma = ','
	c.initialized = true
	c.headers = true
}

// Name implements OutputHandler.Name().
func (c *CSVOutputHandler) Name() string {
	return "csv"
}

// SetOption implements OutputHandler.SetOption().
func (c *CSVOutputHandler) SetOption(name string, value string) error {
	c.init()

	if (name == "csv.comma") && (len(value) > 1) {
		u, err := util.Unescape(value)
		if (err == nil) && (len(u) == 1) {
			value = u
		}
	}

	switch name {
	case "csv.comma":
		c.comma = []rune(value)[0]
	case "csv.headers":
		c.headers = util.Truthy(value)
	}

	return nil
}

// Format implements OutputHandler.Format()
func (c *CSVOutputHandler) Format(writer io.Writer, data interface{}) error {
	c.init()

	table, err := util.Tabularize(data, true)
	if err != nil {
		return err
	}

	w := csv.NewWriter(writer)
	w.Comma = c.comma
	return w.WriteAll(table)
}
