package io

import (
	"bytes"
	"regexp"
	"testing"

	"git.sr.ht/~charles/rq/util"
	"github.com/clbanning/mxj/v2"
	"github.com/stretchr/testify/assert"
)

func TestInput(t *testing.T) {

	// To simplify writing the cases, the expected output is expressed
	// in JSON that we'll Unmarshal before comparison.
	cases := []struct {
		input         []byte
		options       map[string]interface{}
		parsed        interface{}
		handler       string
		expectHandler string
		mustError     bool
	}{
		{
			input:   []byte(`{"foo": "bar", "baz": "quux"}`),
			parsed:  map[string]interface{}{"baz": "quux", "foo": "bar"},
			handler: "json",
		},
		{
			input:   []byte(`{"foo": "bar", "baz": "quux"}`),
			parsed:  map[string]interface{}{"baz": "quux", "foo": "bar"},
			handler: "yaml",
		},
		{
			input:   []byte("key1:\n  - one\n  - two\n  - three"),
			parsed:  map[string]interface{}{"key1": []interface{}{"one", "two", "three"}},
			handler: "yaml",
		},
		{
			input:   []byte("FOO=BAR\nBAZ='QUUX'\nSPAM=\"ham\"\nnumeric=1234\n#comment\nhello=world #another comment"),
			parsed:  map[string]string{"FOO": "BAR", "BAZ": "QUUX", "SPAM": "ham", "numeric": "1234", "hello": "world"},
			handler: "dotenv",
		},
		{
			input:   []byte("foo,bar,baz"),
			parsed:  []interface{}{[]interface{}{"foo", "bar", "baz"}},
			handler: "csv",
		},
		{
			input:   []byte("1,1.7,true,0xa"),
			parsed:  []interface{}{[]interface{}{1, 1.7, true, 10}},
			handler: "csv",
		},
		{
			input:   []byte("1,1.7,true,0xa"),
			parsed:  [][]string{[]string{"1", "1.7", "true", "0xa"}},
			handler: "csv",
			options: map[string]interface{}{"csv.infer": false},
		},
		{
			input: []byte("foo,bar,baz\n123,quux,false\n456,spam,true\n"),
			parsed: []interface{}{
				[]interface{}{"foo", "bar", "baz"},
				[]interface{}{123, "quux", false},
				[]interface{}{456, "spam", true},
			},
			handler: "csv",
		},
		{
			input: []byte("foo,bar,baz\n123,quux,false\n456,spam,true\n"),
			parsed: [][]string{
				[]string{"foo", "bar", "baz"},
				[]string{"123", "quux", "false"},
				[]string{"456", "spam", "true"},
			},
			handler: "csv",
			options: map[string]interface{}{"csv.infer": false},
		},
		{
			input: []byte("foo,bar,baz\n123,quux,false\n456,spam,true\n"),
			parsed: []map[string]interface{}{
				map[string]interface{}{"foo": 123, "bar": "quux", "baz": false},
				map[string]interface{}{"foo": 456, "bar": "spam", "baz": true},
			},
			handler: "csv",
			options: map[string]interface{}{"csv.headers": true},
		},
		{
			input: []byte("foo,bar,baz\n123,quux,false\n456,spam,true\n"),
			parsed: []map[string]interface{}{
				map[string]interface{}{"bar": "quux", "baz": "false", "foo": "123"},
				map[string]interface{}{"bar": "spam", "baz": "true", "foo": "456"},
			},
			handler: "csv",
			options: map[string]interface{}{"csv.headers": true, "csv.infer": false},
		},
		{
			input: []byte("foo;bar;baz\n123;quux;false\n456;spam;true\n"),
			parsed: []map[string]interface{}{
				map[string]interface{}{"foo": 123, "bar": "quux", "baz": false},
				map[string]interface{}{"foo": 456, "bar": "spam", "baz": true},
			},
			handler: "csv",
			options: map[string]interface{}{"csv.comma": ';', "csv.headers": true},
		},
		{
			input: []byte("junk\n\"more junk\"\nfoo;bar;baz\n123;quux;false\n456;spam;true\n"),
			parsed: []map[string]interface{}{
				map[string]interface{}{"foo": 123, "bar": "quux", "baz": false},
				map[string]interface{}{"foo": 456, "bar": "spam", "baz": true},
			},
			handler: "csv",
			options: map[string]interface{}{"csv.comma": ';', "csv.headers": true, "csv.skip_lines": 2},
		},
		{
			input: []byte("#junk\n#more junk\nfoo;bar;baz\n123;quux;false\n456;spam;true\n"),
			parsed: []map[string]interface{}{
				map[string]interface{}{"foo": 123, "bar": "quux", "baz": false},
				map[string]interface{}{"foo": 456, "bar": "spam", "baz": true},
			},
			handler: "csv",
			options: map[string]interface{}{"csv.comma": ';', "csv.headers": true, "csv.comment": '#'},
		},
		{
			input:   []byte("foo=\"bar\"\nbaz=27"),
			parsed:  map[string]interface{}{"baz": int64(27), "foo": "bar"},
			handler: "toml",
		},
		{
			input:   []byte(``),
			options: map[string]interface{}{"base64.data": "eyJiYXoiOjI3LCJmb28iOiJiYXIifQ=="},
			parsed:  map[string]interface{}{"baz": float64(27), "foo": "bar"},
			handler: "base64",
		},
		{
			input: []byte("[section1]\nfoo=bar"),
			parsed: map[string]interface{}{
				"section1": map[string]interface{}{
					"foo": "bar",
				},
			},
			handler: "ini",
		},
		{
			input: []byte("[section1]\nfoo=bar\n[section2]\nkey1=7\n# comment\nkey2 = hi"),
			parsed: map[string]interface{}{
				"section1": map[string]interface{}{
					"foo": "bar",
				},
				"section2": map[string]interface{}{
					"key1": float64(7),
					"key2": "hi",
				},
			},
			handler: "ini",
		},
		{
			input: []byte(`
io_mode = "async"

service "http" "web_proxy" {
  listen_addr = "127.0.0.1:8080"

  process "main" {
    command = ["/usr/local/bin/awesome-app", "server"]
  }

  process "mgmt" {
    command = ["/usr/local/bin/awesome-app", "mgmt"]
  }
}`),
			parsed: map[string]interface{}{
				"io_mode": "async",
				"service": map[string]interface{}{
					"http": map[string]interface{}{
						"web_proxy": []interface{}{
							map[string]interface{}{
								"listen_addr": "127.0.0.1:8080",
								"process": map[string]interface{}{
									"main": []interface{}{
										map[string]interface{}{
											"command": []interface{}{
												"/usr/local/bin/awesome-app",
												"server",
											},
										},
									},
									"mgmt": []interface{}{
										map[string]interface{}{
											"command": []interface{}{
												"/usr/local/bin/awesome-app",
												"mgmt",
											},
										},
									},
								},
							},
						},
					},
				},
			},
			handler: "hcl",
		},
		{
			input:   []byte(`a| b|	C |`),
			parsed:  `a| b|	C |`,
			handler: "raw",
		},
		{
			input:   []byte(`a| b|	C |`),
			parsed:  []string{"a", " b", "\tC "},
			handler: "raw",
			options: map[string]interface{}{"raw.rs": regexp.QuoteMeta("|")},
		},
		{
			input:   []byte(`a| b|	C |`),
			parsed:  []string{"a", "b", "\tC "},
			handler: "raw",
			options: map[string]interface{}{"raw.rs": regexp.QuoteMeta("|"), "raw.lcutset": " "},
		},
		{
			input:   []byte(`a| b|	C |`),
			parsed:  []string{"a", " b", "\tC"},
			handler: "raw",
			options: map[string]interface{}{"raw.rs": regexp.QuoteMeta("|"), "raw.rcutset": " "},
		},
		{
			input:   []byte(`a| b|	C |`),
			parsed:  []string{"a", "b", "C"},
			handler: "raw",
			options: map[string]interface{}{"raw.rs": regexp.QuoteMeta("|"), "raw.cutset": "\t\r\n "},
		},
		{
			input:   []byte("a\n| b|	C |"),
			parsed:  []string{"a\n", "b", "C"},
			handler: "raw",
			options: map[string]interface{}{"raw.rs": regexp.QuoteMeta("|"), "raw.cutset": "\t\r "},
		},
		{
			input:   []byte("a\n| b|	C |"),
			parsed:  []string{"a", "b", "C"},
			handler: "raw",
			options: map[string]interface{}{"raw.rs": regexp.QuoteMeta("|"), "raw.cutset": "\t\n\r "},
		},
		{
			input:   []byte("a\n| 1|	C |false"),
			parsed:  []interface{}{"a", 1, "C", false},
			handler: "raw",
			options: map[string]interface{}{"raw.infer": true, "raw.rs": regexp.QuoteMeta("|"), "raw.cutset": "\t\n\r "},
		},
		{
			input: []byte("foo   bar	baz\n1 2 3\n4   5	6\ntrue false  true"),
			parsed: [][]interface{}{
				[]interface{}{"foo", "bar", "baz"},
				[]interface{}{"1", "2", "3"},
				[]interface{}{"4", "5", "6"},
				[]interface{}{"true", "false", "true"},
			},
			handler: "raw",
			options: map[string]interface{}{"raw.fs": "[ \t]+", "raw.rs": "[\n]+", "raw.cutset": "\t\n\r "},
		},
		{
			input: []byte("foo   bar	baz\n1 2 3\n4   5	6\ntrue false  true"),
			parsed: [][]interface{}{
				[]interface{}{"foo", "bar", "baz"},
				[]interface{}{1, 2, 3},
				[]interface{}{4, 5, 6},
				[]interface{}{true, false, true},
			},
			handler: "raw",
			options: map[string]interface{}{"raw.infer": "true", "raw.fs": "[ \t]+", "raw.rs": "[\n]+", "raw.cutset": "\t\n\r "},
		},
		{
			input: []byte("foo   bar	baz\n1 2 3\n4   5	6\ntrue false  true"),
			parsed: []map[string]interface{}{
				map[string]interface{}{"foo": 1, "bar": 2, "baz": 3},
				map[string]interface{}{"foo": 4, "bar": 5, "baz": 6},
				map[string]interface{}{"foo": true, "bar": false, "baz": true},
			},
			handler: "raw",
			options: map[string]interface{}{"raw.infer": "true", "raw.headers": "true", "raw.fs": "[ \t]+", "raw.rs": "[\n]+", "raw.cutset": "\t\n\r "},
		},
		{
			input: []byte("foo   bar	baz\n1 2 3\n4   5	6\ntrue false  true"),
			parsed: []map[string]interface{}{
				map[string]interface{}{"foo": "1", "bar": "2", "baz": "3"},
				map[string]interface{}{"foo": "4", "bar": "5", "baz": "6"},
				map[string]interface{}{"foo": "true", "bar": "false", "baz": "true"},
			},
			handler: "raw",
			options: map[string]interface{}{"raw.headers": "true", "raw.fs": "[ \t]+", "raw.rs": "[\n]+", "raw.cutset": "\t\n\r "},
		},
		{
			input: []byte("foo   bar	baz\n1 2 3\n4   5	6\ntrue false  true"),
			parsed: []map[string]interface{}{
				map[string]interface{}{"foo": 1, "bar": 2, "baz": 3},
				map[string]interface{}{"foo": 4, "bar": 5, "baz": 6},
				map[string]interface{}{"foo": true, "bar": false, "baz": true},
			},
			handler:       "tabular",
			expectHandler: "raw",
		},
		{
			input: []byte("foo   bar	baz\n1 2 3\n4   5	6\ntrue false  true"),
			parsed: [][]interface{}{
				[]interface{}{"foo", "bar", "baz"},
				[]interface{}{"1", "2", "3"},
				[]interface{}{"4", "5", "6"},
				[]interface{}{"true", "false", "true"},
			},
			handler:       "awk",
			expectHandler: "raw",
		},
		{
			input:         []byte("a,b,c,d,e"),
			parsed:        []string{"a", "b", "c,d,e"},
			handler:       "raw",
			expectHandler: "raw",
			options:       map[string]interface{}{"raw.coalesce": "3", "raw.rs": ","},
		},
		{
			input: []byte("a,b,c,d,e\n1,2,3,4,5"),
			parsed: [][]interface{}{
				[]interface{}{"a", "b", "c,d,e"},
				[]interface{}{"1", "2", "3,4,5"},
			},
			handler:       "raw",
			expectHandler: "raw",
			options:       map[string]interface{}{"raw.coalesce": "3", "raw.rs": "\n", "raw.fs": ","},
		},
		{
			input: []byte("foo   bar	baz\n\n1 2 3   \n4   5	6\ntrue false  true"),
			parsed: []string{
				"foo   bar	baz",
				"1 2 3",
				"4   5	6",
				"true false  true",
			},
			handler:       "lines",
			expectHandler: "raw",
		},
		{
			input: []byte(`    PID TTY          TIME CMD
   1747 tty2     00:08:40 Xorg
   1773 tty2     00:00:00 gnome-session-b
  86012 pts/0    00:00:00 tmux: client
  86038 pts/1    00:00:43 nvim
 105087 pts/2    00:00:00 ps`),
			parsed: []map[string]interface{}{
				map[string]interface{}{
					"PID":  1747,
					"TTY":  "tty2",
					"TIME": "00:08:40",
					"CMD":  "Xorg",
				},
				map[string]interface{}{
					"PID":  1773,
					"TTY":  "tty2",
					"TIME": "00:00:00",
					"CMD":  "gnome-session-b",
				},
				map[string]interface{}{
					"PID":     86012,
					"TTY":     "pts/0",
					"TIME":    "00:00:00",
					"CMD":     "tmux:",
					"column4": "client",
				},
				map[string]interface{}{
					"PID":  86038,
					"TTY":  "pts/1",
					"TIME": "00:00:43",
					"CMD":  "nvim",
				},
				map[string]interface{}{
					"PID":  105087,
					"TTY":  "pts/2",
					"TIME": "00:00:00",
					"CMD":  "ps",
				},
			},
			handler:       "tabular",
			expectHandler: "raw",
		},
		{
			input: []byte(`    PID TTY          TIME CMD
   1747 tty2     00:08:40 Xorg
   1773 tty2     00:00:00 gnome-session-b
  86012 pts/0    00:00:00 tmux: client
  86038 pts/1    00:00:43 nvim
 105087 pts/2    00:00:00 ps`),
			parsed: [][]interface{}{
				[]interface{}{"PID", "TTY", "TIME", "CMD"},
				[]interface{}{"1747", "tty2", "00:08:40", "Xorg"},
				[]interface{}{"1773", "tty2", "00:00:00", "gnome-session-b"},
				[]interface{}{"86012", "pts/0", "00:00:00", "tmux:", "client"},
				[]interface{}{"86038", "pts/1", "00:00:43", "nvim"},
				[]interface{}{"105087", "pts/2", "00:00:00", "ps"},
			},
			handler:       "awk",
			expectHandler: "raw",
		},
		{
			input: []byte(`{
"foo": "bar"
"baz": no quotes!
# comments

# blank lines!
quux: 7,
}`),
			parsed: map[string]interface{}{
				"foo":  "bar",
				"baz":  "no quotes!",
				"quux": 7.0,
			},
			handler: "hjson",
		},
		{
			input: []byte(`
{ home       = "/home/bill"
, privateKey = "/home/bill/.ssh/id_ed25519"
, publicKey  = "/home/blil/.ssh/id_ed25519.pub"
}
`),
			parsed: map[string]interface{}{
				"home":       "/home/bill",
				"privateKey": "/home/bill/.ssh/id_ed25519",
				"publicKey":  "/home/blil/.ssh/id_ed25519.pub",
			},
			handler: "dhall",
		},
		{
			input:   []byte(`<doc><list-element0><field1>row1 col1</field1><field2>row1 col2</field2></list-element0><list-element1><field1>row2 col1</field1><field2>row2 col2</field2></list-element1></doc>`),
			handler: "xml",
			parsed: mxj.Map(mxj.Map{
				"doc": map[string]interface{}{
					"list-element0": map[string]interface{}{
						"field1": "row1 col1",
						"field2": "row1 col2",
					},
					"list-element1": map[string]interface{}{
						"field1": "row2 col1",
						"field2": "row2 col2",
					},
				},
			}),
		},
		{
			input:   []byte(`{"foo": "bar", "baz": "quux"}`),
			parsed:  []interface{}{map[string]interface{}{"baz": "quux", "foo": "bar"}},
			handler: "ndjson",
		},
		{
			input: []byte(`{"foo": "bar", "baz": "quux"}
{"aaa": "xxx", "bbb": "yyy"}`),
			parsed: []interface{}{
				map[string]interface{}{"baz": "quux", "foo": "bar"},
				map[string]interface{}{"aaa": "xxx", "bbb": "yyy"},
			},
			handler: "ndjson",
		},
		{
			input: []byte(`{
"foo": "bar",
"baz": "quux"}
`),
			handler:   "ndjson",
			mustError: true,
		},
		{
			input: []byte(`
{"hello": "world"}

{"foo": "bar"}



{"baz": "quux"}
`),
			handler: "ndjson",
			parsed: []interface{}{
				map[string]interface{}{"hello": "world"},
				map[string]interface{}{"foo": "bar"},
				map[string]interface{}{"baz": "quux"},
			},
		},
		{
			input: []byte(`
{"hello": "world"}

{"foo": "bar"}



{"baz": "quux"}
`),
			handler: "ndjson",
			options: map[string]interface{}{
				"ndjson.allowempty": false,
			},
			mustError: true,
		},
		{
			input:         []byte("foo\tbar\tbaz"),
			parsed:        []interface{}{[]interface{}{"foo", "bar", "baz"}},
			handler:       "tsv",
			expectHandler: "csv",
		},
		{
			input:         []byte("1\t1.7\ttrue\t0xa"),
			parsed:        []interface{}{[]interface{}{1, 1.7, true, 10}},
			handler:       "tsv",
			expectHandler: "csv",
		},
		{
			input: []byte(`
// hello, JSONC
{
	/* comment */
	foo: bar
	baz: [quux, aaa], // comment
}
`),
			parsed:        map[string]interface{}(map[string]interface{}{"baz": []interface{}{"quux", "aaa"}, "foo": "bar"}),
			handler:       "jsonc",
			expectHandler: "jsonc",
		},
	}

	for i, c := range cases {
		t.Logf("---- case %d ----------------\ninput: %s\n", i, string(c.input))
		reader := &bytes.Buffer{}
		reader.Write(c.input)

		handler, err := SelectInputHandler(c.handler)

		assert.Nil(t, err)

		if c.expectHandler != "" {
			// this is used when testing aliases
			assert.Equal(t, c.expectHandler, handler.Name())
		} else {
			assert.Equal(t, c.handler, handler.Name())
		}

		for k, v := range c.options {
			err := handler.SetOption(k, util.ValueToString(v))
			assert.Nil(t, err)
		}

		var parsed interface{}
		parsed, err = handler.Parse(reader)

		if c.mustError {
			assert.NotNil(t, err)
			continue
		}

		assert.Nil(t, err)

		assert.Equal(t, c.parsed, parsed)
	}
}

func TestOutput(t *testing.T) {
	cases := []struct {
		input         interface{}
		output        []byte
		handler       string
		options       map[string]interface{}
		expectHandler string
	}{
		{
			input: map[string]interface{}{
				"foo": 7,
				"bar": []string{"one", "two", "three"},
			},
			handler: "json",
			output:  []byte(`{"foo": 7,"bar": ["one", "two", "three"]}`),
			options: map[string]interface{}{
				"output.colorize": false,
			},
		},
		{
			input: map[string]interface{}{
				"foo": 7,
				"bar": []string{"one", "two", "three"},
			},
			handler: "yaml",
			output:  []byte(`{"foo": 7, "bar": ["one", "two", "three"]}`),
			options: map[string]interface{}{
				"output.colorize": false,
			},
		},
		{
			input: map[string]interface{}{
				"foo": 7,
				"bar": []string{"one", "two", "three"},
			},
			handler: "toml",
			output: []byte(`bar = ["one", "two", "three"]
foo = 7
`),
			options: map[string]interface{}{
				"output.colorize": false,
			},
		},
		{
			input: []map[string]interface{}{
				map[string]interface{}{"field1": "row1 col1", "field2": "row1 col2"},
				map[string]interface{}{"field1": "row2 col1", "field2": "row2 col2"},
			},
			handler: "csv",
			output:  []byte("field1,field2\nrow1 col1,row1 col2\nrow2 col1,row2 col2\n"),
			options: map[string]interface{}{
				"csv.comma": ',',
			},
		},
		{
			input: []map[string]interface{}{
				map[string]interface{}{"field1": "row1 col1", "field2": "row1 col2"},
				map[string]interface{}{"field1": "row2 col1", "field2": "row2 col2"},
			},
			handler: "md-table",
			output:  []byte("| field1    | field2    |\n|-----------|-----------|\n| row1 col1 | row1 col2 |\n| row2 col1 | row2 col2 |\n"),
		},
		{
			input: []map[string]interface{}{
				map[string]interface{}{"a": "这是一个测试", "b": "123"},
			},
			handler: "md-table",
			output:  []byte("| a      | b   |\n|--------|-----|\n| 这是一个测试 | 123 |\n"),
		},
		{
			input: []map[string]interface{}{
				map[string]interface{}{"field1": "row1 col1", "field2": "row1 col2"},
				map[string]interface{}{"field1": "row2 col1", "field2": "row2 col2"},
			},
			handler: "md-table",
			output:  []byte("| field1 | field2 |\n|--------|--------|\n| row1 col1 | row1 col2 |\n| row2 col1 | row2 col2 |\n"),
			options: map[string]interface{}{
				"output.pretty": false,
			},
		},
		{
			input:   7,
			handler: "raw",
			output:  []byte("7\n"),
		},
		{
			input:   7.32,
			handler: "raw",
			output:  []byte("7.32\n"),
		},
		{
			input:   "test 123",
			handler: "raw",
			output:  []byte("test 123"),
		},
		{
			input:   []string{"a", "b", "c", "d"},
			handler: "raw",
			output:  []byte("a\nb\nc\nd"),
		},
		{
			input:   []int{1, 2, 3, 4},
			handler: "raw",
			output:  []byte("1\n2\n3\n4"),
		},
		{
			input:   []float64{1.1, 2.2, 3.3, 4.4},
			handler: "raw",
			output:  []byte("1.1\n2.2\n3.3\n4.4"),
		},
		{
			input:   []bool{true, true, false},
			handler: "raw",
			output:  []byte("true\ntrue\nfalse"),
		},
		{
			input:   []interface{}{true, 5, 1.3, "test"},
			handler: "raw",
			output:  []byte("true\n5\n1.3\ntest"),
		},
		{
			input:   map[string]interface{}{"foo": "bar"},
			handler: "raw",
			output:  []byte(`{"foo":"bar"}`),
		},
		{
			input: []interface{}{
				3.14,
				7,
				"hello, world!",
				map[string]interface{}{"foo": "bar"},
			},
			handler: "raw",
			output:  []byte("3.14\n7\nhello, world!\n{\"foo\":\"bar\"}"),
		},
		{
			input: []map[string]string{
				map[string]string{
					"col1": "v1",
					"col2": "v2",
					"col3": "v3",
				},
				map[string]string{
					"col1": "v4",
					"col2": "v5",
					"col3": "v6",
				},
			},
			handler: "raw",
			options: map[string]interface{}{
				"raw.fl": " FL ",
				"raw.fs": " FS ",
				"raw.fr": " FR ",
				"raw.rl": " RL ",
				"raw.rs": " RS ",
				"raw.rr": " RR ",
			},
			output: []byte(" RL  FL v1 FR  FS  FL v2 FR  FS  FL v3 FR  RR  RS  RL  FL v4 FR  FS  FL v5 FR  FS  FL v6 FR  RR  RS "),
		},
		{
			input: []map[string]interface{}{
				map[string]interface{}{"field1": "row1 col1", "field2": "row1 col2"},
				map[string]interface{}{"field1": "row2 col1", "field2": "row2 col2"},
			},
			handler: "xml",
			output:  []byte("<doc><list-element0><field1>row1 col1</field1><field2>row1 col2</field2></list-element0><list-element1><field1>row2 col1</field1><field2>row2 col2</field2></list-element1></doc>"),
			options: map[string]interface{}{
				"output.colorize": false,
				"output.pretty":   false,
			},
		},
		{
			input:   []interface{}{"foo", "bar", 123, 4.56},
			handler: "xml",
			output:  []byte("<doc><list-element0>foo</list-element0><list-element1>bar</list-element1><list-element2>123</list-element2><list-element3>4.56</list-element3></doc>"),
			options: map[string]interface{}{
				"output.colorize": false,
				"output.pretty":   false,
			},
		},
		{
			input:   []interface{}{"foo", "bar", 123, 4.56},
			handler: "xml",
			output:  []byte("<doc><list>foo</list><list>bar</list><list>123</list><list>4.56</list></doc>"),
			options: map[string]interface{}{
				"output.colorize": false,
				"output.pretty":   false,
				"xml.list-style":  "common",
			},
		},
		{
			input:   interface{}("hello, world!"),
			handler: "xml",
			output:  []byte("<doc><object>hello, world!</object></doc>"),
			options: map[string]interface{}{
				"output.colorize": false,
				"output.pretty":   false,
			},
		},
		{
			input:   interface{}("hello, world!"),
			handler: "xml",
			output:  []byte("<foobar><object>hello, world!</object></foobar>"),
			options: map[string]interface{}{
				"output.colorize": false,
				"output.pretty":   false,
				"xml.root-tag":    "foobar",
			},
		},
		{
			input:   map[string]interface{}{"foo": "bar", "baz": 7},
			handler: "sh",
			output:  []byte("baz=7\nfoo=\"bar\""),
		},
		{
			input:   [][]string{[]string{"foo", "bar", "baz"}, []string{"spam", "ham"}},
			handler: "csv",
			output:  []byte("foo,bar,baz\nspam,ham\n"),
		},
		{
			input: []interface{}{
				map[string]interface{}{
					"foo": 7,
					"bar": []string{"one", "two", "three"},
				},
				map[string]interface{}{
					"hello": "world",
				},
			},
			handler: "ndjson",
			output: []byte(`{"bar":["one","two","three"],"foo":7}
{"hello":"world"}
`),
			options: map[string]interface{}{
				"output.colorize": false,
			},
		},
		{
			input:         [][]string{[]string{"foo", "bar", "baz"}, []string{"spam", "ham"}},
			handler:       "tsv",
			output:        []byte("foo\tbar\tbaz\nspam\tham\n"),
			expectHandler: "csv",
		},
		{
			input: []map[string]interface{}{
				map[string]interface{}{"field1": "row1 col1", "field2": "row1 col2"},
				map[string]interface{}{"field1": "row2 col1", "field2": "row2 col2"},
			},
			handler:       "tsv",
			output:        []byte("field1\tfield2\nrow1 col1\trow1 col2\nrow2 col1\trow2 col2\n"),
			expectHandler: "csv",
		},
		{
			input:         map[string]interface{}{"foo": 123, "bar": "quux"},
			handler:       "null",
			output:        []byte{},
			expectHandler: "null",
		},
		{
			input:   map[string]interface{}{"foo": 123, "bar": "quux"},
			handler: "template",
			output:  []byte(`123 | quux`),
			options: map[string]interface{}{
				"output.template": "{{.foo}} | {{.bar}}",
			},
			expectHandler: "template",
		},
		{
			input: []interface{}{
				map[string]interface{}{"foo": 123, "bar": "quux"},
				map[string]interface{}{"foo": 456, "bar": "spam"},
			},
			handler: "template",
			output:  []byte("123 | quux\n456 | spam\n"),
			options: map[string]interface{}{
				"output.template": "{{.foo}} | {{.bar}}",
			},
			expectHandler: "template",
		},
		{
			input: []map[string]interface{}{
				map[string]interface{}{"foo": 123, "bar": "quux"},
				map[string]interface{}{"foo": 456, "bar": "spam"},
			},
			handler: "template",
			output:  []byte("123 | quux#456 | spam\n"),
			options: map[string]interface{}{
				"output.template": "{{.foo}} | {{.bar}}",
				"output.sep":      "#",
			},
			expectHandler: "template",
		},
		{
			input: map[string]interface{}{
				"foo": 7,
				"bar": []any{"one", "two", "three"},
			},
			handler: "hcl",
			output:  []byte("bar = [\"one\", \"two\", \"three\"]\nfoo = 7\n"),
			options: map[string]interface{}{
				"output.colorize": false,
			},
		},
		{
			input: []map[string]interface{}{
				map[string]interface{}{
					"foo": 7,
					"bar": []any{"one", "two", "three"},
				},
				map[string]interface{}{
					"a": "aaa",
					"b": "bbb",
				},
			},
			handler: "hcl",
			output:  []byte("data = [{\n  bar = [\"one\", \"two\", \"three\"]\n  foo = 7\n  }, {\n  a = \"aaa\"\n  b = \"bbb\"\n}]\n"),
			options: map[string]interface{}{
				"output.colorize": false,
			},
		},
		{
			input: map[string]interface{}{
				"bbb": []float64{333333333.33333329, 1e30, 4.50, 2e-3, 0.000000000000000000000000001},
				"aaa": "\u20ac$\u000F",
				"ccc": []any{nil, true, false},
			},
			handler: "json",
			output:  []byte(`{"aaa":"€$\u000f","bbb":[333333333.3333333,1e+30,4.5,0.002,1e-27],"ccc":[null,true,false]}`),
			options: map[string]interface{}{
				"json.canonical": true,
			},
		},
	}

	for i, c := range cases {
		t.Logf("----------[%d] handler=%s\nc.input=%+v\n", i, c.handler, c.input)

		writer := &bytes.Buffer{}

		handler, err := SelectOutputHandler(c.handler)
		assert.Nil(t, err)

		if c.expectHandler == "" {
			assert.Equal(t, c.handler, handler.Name())
		} else {
			assert.Equal(t, c.expectHandler, handler.Name())
		}

		for k, v := range c.options {
			err := handler.SetOption(k, util.ValueToString(v))
			assert.Nil(t, err)
		}

		err = handler.Format(writer, c.input)
		t.Logf("error=%v", err)
		assert.Nil(t, err)

		if c.handler == "json" {
			assert.JSONEq(t, string(c.output), string(writer.Bytes()))
		} else if c.handler == "ndjson" {
			assert.Equal(t, string(c.output), string(writer.Bytes()))
		} else if c.handler == "yaml" {
			assert.YAMLEq(t, string(c.output), string(writer.Bytes()))
		} else if c.handler == "toml" {
			assert.Equal(t, string(c.output), string(writer.Bytes()))
		} else if c.handler == "csv" {
			assert.Equal(t, string(c.output), string(writer.Bytes()))
		} else if c.handler == "tsv" {
			assert.Equal(t, string(c.output), string(writer.Bytes()))
		} else if c.handler == "raw" {
			assert.Equal(t, string(c.output), string(writer.Bytes()))
		} else if c.handler == "md-table" {
			assert.Equal(t, string(c.output), string(writer.Bytes()))
		} else if c.handler == "xml" {
			assert.Equal(t, string(c.output), string(writer.Bytes()))
		} else if c.handler == "sh" {
			assert.Equal(t, string(c.output), string(writer.Bytes()))
		} else if c.handler == "template" {
			assert.Equal(t, string(c.output), string(writer.Bytes()))
		} else if c.handler == "null" {
			assert.Equal(t, string(c.output), string(writer.Bytes()))
		} else if c.handler == "hcl" {
			assert.Equal(t, string(c.output), string(writer.Bytes()))
		} else {
			t.Logf("unknown handler '%s'", c.handler)
			assert.True(t, false)
		}
	}
}
