package io

import (
	"bufio"
	"fmt"
	"io"
	"strings"

	"git.sr.ht/~charles/rq/util"
	json "github.com/json-iterator/go"
)

func init() {
	registerInputHandler("ndjson", func() InputHandler { return &NDJSONInputHandler{} })
}

// Declare conformance with InputHandler interface.
var _ InputHandler = &NDJSONInputHandler{}

// NDJSONInputHandler handles parsing NDJSON data.
//
// The following options are supported:
//
// ndjson.allowempty (bool) if asserted, empty lines are ignored, otherwise they  cause
// errors (default: true)
type NDJSONInputHandler struct {
	initialized bool
	allowempty  bool
}

func (j *NDJSONInputHandler) init() {
	if j.initialized {
		return
	}

	j.initialized = true
	j.allowempty = true
}

// Name implements InputHandler.Name().
func (j *NDJSONInputHandler) Name() string {
	return "ndjson"
}

// Parse implements InputHandler.Parse().
func (j *NDJSONInputHandler) Parse(reader io.Reader) (interface{}, error) {
	j.init()

	scanner := bufio.NewScanner(reader)

	data := []interface{}{}

	lineno := 0
	for scanner.Scan() {
		line := scanner.Text()
		// remove whitespace as defined in RFC8259 pp.6, so our
		// length check will be valid
		line = strings.Trim(line, "\x20\x09\x0a\x0d")

		if len(line) == 0 {
			if !j.allowempty {
				return nil, fmt.Errorf("line %d is empty, which is disallowed because ndjson.allowempty is false", lineno)
			}
			continue
		}

		var parsed interface{}
		err := json.Unmarshal([]byte(line), &parsed)
		if err != nil {
			return nil, fmt.Errorf("failed to parse line '%d': %w", lineno, err)
		}
		data = append(data, parsed)
		lineno++
	}

	return data, nil
}

// SetOption implements InputHandler.SetOption().
func (j *NDJSONInputHandler) SetOption(name string, value string) error {
	j.init()

	switch name {
	case "ndjson.allowempty":
		j.allowempty = util.Truthy(value)
	}

	return nil
}
