# Rego Query

[![builds.sr.ht status](https://builds.sr.ht/~charles/rq.svg)](https://builds.sr.ht/~charles/rq?) [![go.dev reference](https://img.shields.io/badge/go.dev-reference-007d9c?logo=go&logoColor=white&style=flat-square)](https://pkg.go.dev/mod/git.sr.ht/~charles/rq) [![Go Report Card](https://goreportcard.com/badge/git.sr.ht/~charles/rq)](https://goreportcard.com/report/git.sr.ht/~charles/rq) [![documentation](https://img.shields.io/badge/documentation-%F0%9F%93%96-blue.svg)](https://git.sr.ht/~charles/rq/tree/master/item/doc/README.md) [![buy me a coffee](https://img.shields.io/badge/%E2%98%95-buy%20me%20a%20coffee-blueviolet)](https://www.buymeacoffee.com/cdaniels) [![issue tracker](https://img.shields.io/badge/issue%20tracker-%F0%9F%90%9B-blue)](https://todo.sr.ht/~charles/rq) [![mailing list](https://img.shields.io/badge/mailing%20list-%F0%9F%93%A7-blue)](https://lists.sr.ht/~charles/rq)

`rq` (Rego Query) is inspired by [`jq`](https://stedolan.github.io/jq/), and aims to offer a similar set of capabilities oriented around running [Rego](https://www.openpolicyagent.org/docs/latest/policy-language/) queries. In the long run, `rq` is intended to make it as easy as possible to utilize Rego both interactively as a shell command, and within shell scripts. Though it is already possible to use Rego in this way with via `opa eval` using the official [`opa`](https://github.com/open-policy-agent/opa) command, `rq` is intended to be easier and simpler to use specifically in a shell context, and has additional convenience features to support this use case, such as more input and output formats, syntax highlighted output, and so on.

A simple demonstration (see the [Showcase](#showcase) section for more):

```plain
$ data='{"foo": 7, "bar": 3, "baz": 12, "quux": 4, "spam": 6}'
$ # find all keys in data where the value is at least 5
$ echo "$data" | rq '{k | some k; input[k] > 5}'
[
	"baz",
	"foo",
	"spam"
]
```

A short (10 minute) talk providing a brief overview of `rq`: [rq: Datalog for your shell pipelines](https://www.youtube.com/watch?v=CVe0q9cGyA0).

## Features

* Supported input formats
	* CSV (including comments, arbitrary delimiters, headers, and leading line skip)
	* INI
	* [Dhall](https://dhall-lang.org/)
	* [HCL](https://github.com/hashicorp/hcl)
	* [HJSON](https://hjson.github.io/)
	* [JSON](https://www.rfc-editor.org/rfc/rfc7159.html)
	* [NDJSON](https://github.com/ndjson/ndjson-spec)
	* [TOML](https://toml.io/en/)
	* [XML](https://www.w3.org/XML/)
	* [YAML](https://yaml.org/)
	* awk (field and record splitting similar to awk's defaults)
	* dotenv (shell-like variables, via [gotenv](github.com/subosito/gotenv))
	* lines (input is split into non-empty lines)
	* raw (plain UTF-8 string)
	* tabular (input is parsed as a table with header and type inference, uses field and record splitting similar to the awk format)
    * [jsonc](https://github.com/komkom/jsonc)
    * tf (alias for HCL)
* Supported output formats
	* CSV
    * JCS ([RFC8785](https://datatracker.ietf.org/doc/html/rfc8785))
	* JSON
	* NDJSON
	* TOML
	* XML
	* YAML
	* dotenv (POSIX shell variable assignments, suitable for use with `eval`)
	* md-table (markdown compatible tables)
	* null (discards output data without formatting it)
	* raw (primitive types, and lists of primitive types, are printed one per line without quotes, otherwise behaves like the JSON output format)
    * HCL
    * sh (alias for dotenv)
    * template (output is used as the data for a Go string template, see `-t / --template`)
    * tf (alias for HCL)
* Syntax-highlighted output using [chroma](https://github.com/alecthomas/chroma)
* Larger programs can be written as scripts, with shebang support via `rq script`
* Additional [builtins](./doc/builtins.md) to augment Rego's standard library, many with useful side-effects
    * `rq.run()` - run arbitrary shell commands
    * `rq.tree()` - recursively search the local filesystem
    * `rq.env()` - access environment variables
    * `rq.args()` - access CLI arguments (only for `rq script`)
    * `rq.error()` - crash the Rego evaluator with an error
    * `rq.encode()` - serialize a Rego object to a string using one of `rq`'s output formats
    * `rq.decode()` - deserialize a string to a Rego object using one of `rq`'s input formats
    * `rq.write()` - write a Rego object to disk using one of `rq`'s output formats
    * `rq.read()` - read a Rego object from disk using one of `rq`'s input formats
    * `rq.parsedate()` - parse a date string without needing to explicitly provide a layout
    * `rq.version()` - access `rq` version information
    * `rq.convert()` - performs unit conversions
    * `rq.abs()` - convert a path to absolute form
    * `rq.base()`  - extract the final element from a path
    * `rq.ext()` - extract the file extension from a path
    * `rq.dir()` - extract all but the final element of a path
    * `rq.splitpath()` - split a path to individual elements
    * `rq.joinpath()` - join path elements
    * `rq.getwd()` - retrieve `rq`'s working directory
    * `rq.chdir()` - change `rq`'s working directory
    * `rq.scriptpath()` - retrieve the path to the running script (only for `rq script`)
    * `rq.template()` - template a string with data from a Rego object
    * `rq.fake()` - generate fake data using [`jaswdr/faker`](https://github.com/jaswdr/faker)
    * `rq.sfake()` - similar to `rq.fake()`, but allows referencing previously generated data symbolically

## Installation Instructions

With `go install`:

```plain
$ go install git.sr.ht/~charles/rq/cmd/rq@latest
```

With Make:

```
$ git clone https://git.sr.ht/~charles/rq
$ cd rq
$ git checkout <tag name>  # skip if you want to use the latest development version
$ make install
```

With [asdf](https://asdf-vm.com/) (note: you must have a Go compiler installed):

```
$ asdf plugin add rq https://git.sr.ht/~charles/asdf-rq
$ asdf install rq latest
$ asdf global rq latest
```

On Linux or MacOS, you can also install the latest binary version to `/usr/bin` with:

```
$ curl -LSsf 'https://git.sr.ht/~charles/rq/blob/master/scripts/install.sh' | sudo bash
```

The installation script supports overriding the OS, version, platform, and installation directory via environment variables. These are documented in [`script/install.sh`](https://git.sr.ht/~charles/rq/tree/master/item/scripts/install.sh).

## Usage Instructions

See `rq --help`. You might also like the [getting started guide](./doc/starting.md).

## Showcase

For more `rq` usage examples, check out the [cookbook](./doc/cookbook.md).

Sample data:

```plain
$ cat sample_data/books.json
[
    {
        "title": "Writing An Interpreter In Go",
        "authors": ["Thorsten Ball"],
        "isbn": "978-3982016115",
        "year": 2018
    },
    {
        "title": "Writing A Compiler In Go",
        "authors": ["Thorsten Ball"],
        "isbn": "978-3982016108",
        "year": 2018
    },
    {
        "title": "The Go Programming Language",
        "authors": ["Alan A. A. Donovan", "Brian W. Kernighan"],
        "isbn": "978-0134190440",
        "year": 2015
    },
    {
        "title": "Hands-On GUI Application Development in Go",
        "authors": ["Andrew Williams"],
        "isbn": "978-1789138412",
        "year": 2019
    },
    {
        "title": "Building Cross-Platform GUI Applications with Fyne",
        "authors": ["Andrew Williams"],
        "isbn": "1800563167",
        "year": 2021
    }
]
$ cat sample_data/books.csv
title,"first author",isbn,year
Writing An Interpreter In Go,Thorsten Ball,978-3982016115,2018
Writing A Compiler In Go,Thorsten Ball,978-3982016108,2018
"The Go Programming Language","Alan A. A. Donovan",978-0134190440,2015
Hands-On GUI Application Development in Go,Andrew Williams,978-1789138412,2019
Building Cross-Platform GUI Applications with Fyne,Andrew Williams,1800563167,2021
```

Read JSON data:

```plain
$ rq < sample_data/books.json
[
	{
		"authors": [
			"Thorsten Ball"
		],
		"isbn": "978-3982016115",
		"title": "Writing An Interpreter In Go",
		"year": 2018
	},
	{
		"authors": [
			"Thorsten Ball"
		],
		"isbn": "978-3982016108",
		"title": "Writing A Compiler In Go",
		"year": 2018
	},
	{
		"authors": [
			"Alan A. A. Donovan",
			"Brian W. Kernighan"
		],
		"isbn": "978-0134190440",
		"title": "The Go Programming Language",
		"year": 2015
	},
	{
		"authors": [
			"Andrew Williams"
		],
		"isbn": "978-1789138412",
		"title": "Hands-On GUI Application Development in Go",
		"year": 2019
	},
	{
		"authors": [
			"Andrew Williams"
		],
		"isbn": "1800563167",
		"title": "Building Cross-Platform GUI Applications with Fyne",
		"year": 2021
	}
]
```

Read CSV data, with headers:

```plain
$ rq -i csv -H < sample_data/books.csv
[
	{
		"first author": "Thorsten Ball",
		"isbn": "978-3982016115",
		"title": "Writing An Interpreter In Go",
		"year": 2018
	},
	{
		"first author": "Thorsten Ball",
		"isbn": "978-3982016108",
		"title": "Writing A Compiler In Go",
		"year": 2018
	},
	{
		"first author": "Alan A. A. Donovan",
		"isbn": "978-0134190440",
		"title": "The Go Programming Language",
		"year": 2015
	},
	{
		"first author": "Andrew Williams",
		"isbn": "978-1789138412",
		"title": "Hands-On GUI Application Development in Go",
		"year": 2019
	},
	{
		"first author": "Andrew Williams",
		"isbn": 1800563167,
		"title": "Building Cross-Platform GUI Applications with Fyne",
		"year": 2021
	}
]
```

Select all books published after 2018:

```plain
$ rq -i csv -H '{book.title | book := input[_]; book.year > 2018}' < sample_data/books.csv
[
	"Building Cross-Platform GUI Applications with Fyne",
	"Hands-On GUI Application Development in Go"
]
```

As above, but with CSV output:

```plain
authors.0,isbn,title,year
Andrew Williams,1800563167,Building Cross-Platform GUI Applications with Fyne,2021
Andrew Williams,978-1789138412,Hands-On GUI Application Development in Go,2019
```

Select all authors who have published at least 2 books:

```plain
$ rq '{author | book := input[_]; author := book.authors[_]; count({b | b := input[_]; author in b.authors}) > 1}' < sample_data/books.json
[
	"Andrew Williams",
	"Thorsten Ball"
]
```

As above, but with YAML output:

```plain
$ rq -o yaml '{author | book := input[_]; author := book.authors[_]; count({b | b := input[_]; author in b.authors}) > 1}' < sample_data/books.json
- Andrew Williams
- Thorsten Ball
```

As above, but with "raw" output:

```plain
$ rq -o raw '{author | book := input[_]; author := book.authors[_]; count({b | b := input[_]; author in b.authors}) > 1}' < sample_data/books.json
Andrew Williams
Thorsten Ball
```

Run a shell command and parse the output as JSON:

```plain
$ rq 'rq.run(["jq", ".[0]"], {"stdin_spec": {"file_path": "sample_data/books.json"}, "stdout_spec": "json:"})'
{
	"exitcode": 0,
	"stderr": "",
	"stdout": {
		"authors": [
			"Thorsten Ball"
		],
		"isbn": "978-3982016115",
		"title": "Writing An Interpreter In Go",
		"year": 2018
	}
}
```

Generate fake data for unit tests:

```plain
$ rq '{rq.fake("internet.user"): {"name": rq.fake("name"), "email": rq.fake("email")} | i := numbers.range(1, 5)[_]}'
{
	"gulgowski.russel": {
		"email": "pauline@example.org",
		"name": "Carmel Franecki"
	},
	"jacquelyn": {
		"email": "arden.stamm@example.org",
		"name": "Olga Toy V"
	},
	"marcus.larkin": {
		"email": "murazik@example.org",
		"name": "Aniya Graham Jr."
	},
	"rogahn": {
		"email": "cole.magdalen@example.org",
		"name": "Imani Gislason"
	},
	"swaniawski.torey": {
		"email": "abshire@example.org",
		"name": "Ms. Halie Herzog"
	}
}
```

## License

See [`./LICENSE`](./LICENSE).

## Changelog

See [`./CHANGELOG.md`](./CHANGELOG.md).

## Documentation

See [`./doc/README.md`](./doc/README.md).

## Community & Contribution

Feel free to discuss or send patches via [the rq mailing list](https://lists.sr.ht/~charles/rq).

If you have found an issue with `rq`, please report it via [the rq issue tracker](https://todo.sr.ht/~charles/rq).

For instructions on how to properly prepare patches for use with SourceHut projects such as `rq`, see [the git.sr.ht docs](https://man.sr.ht/git.sr.ht/).

## Supporting `rq`

If you would like to support the project financially, donations are accepted via [buy me a coffee](https://www.buymeacoffee.com/cdaniels). Your support is greatly appreciated.

## Maturity

I use `rq` on a regular basis and find it valuable for my own uses. At this stage, I don't predict major backwards-incompatible changes in the foreseeable future. Keep in mind that `rq` is a 1-person hobby project. Bugs will be fixed, dependencies updated, and breaking changes avoided on a "best effort" basis.

## Disclaimer

`rq` is an unofficial community project and is not affiliated with Open Policy Agent.
