{ inputs, ... }:
let
  sharedLib = inputs.self.sharedLib;
in
{
  perSystem =
    {
      self',
      pkgs,
      lib,
      ...
    }:
    let
      testLib = inputs.self.testLib { inherit self' lib pkgs; };
    in
    {
      checks =
        {
          repoCreate1 = {
            withAuth = true;
            test =
              # user info test
              ''
                repo = ${
                  testLib.berg_run_json {
                    cmd = ''repo create --name "${sharedLib.repo}" --description "asdf" --private "private" --default-branch "${sharedLib.main}"'';
                  }
                }
                printj(repo)

                assert repo["name"] == "${sharedLib.repo}", f"repo name incorrect\ngot:{repo}"
                assert repo["owner"]["login"] == "${sharedLib.user}", f"repo owner incorrect\ngot:{repo}"
                assert repo["description"] == "asdf", f"repo description incorrect\ngot:{repo}"
                assert repo["private"] == True, f"repo private incorrect\ngot:{repo}"
                assert repo["default_branch"] == "${sharedLib.main}", f"repo default branch incorrect\ngot:{repo}"
              '';
          };
          repoCreateAlreadyExists = {
            withAuth = true;
            test =
              # user info test
              ''
                ${testLib.berg_run {
                  cmd = ''repo create --name "${sharedLib.repo}" --description "asdf" --private "private" --default-branch "${sharedLib.main}"'';
                }}
                error = ${
                  testLib.berg_run_json {
                    success = false;
                    cmd = ''repo create --name "${sharedLib.repo}" --description "asdf" --private "private" --default-branch "${sharedLib.main}"'';
                  }
                }
                printj(error)

                assert "409" in error["error_message"], f"expected error when recreating existing repo\ngot:{error}"
                assert "Repository with the same name already exists!" in error["error_message"], f"expected error when recreating existing repo\ngot:{error}"
              '';
          };
          repoCreate2 = {
            withAuth = true;
            test =
              # user info test
              ''
                repo = ${
                  testLib.berg_run_json {
                    cmd = ''repo create --name "${sharedLib.otherRepo}" --description "asdf" --private "public" --default-branch "mega"'';
                  }
                }
                printj(repo)

                assert repo["name"] == "${sharedLib.otherRepo}", f"repo name incorrect\ngot:{repo}"
                assert repo["owner"]["login"] == "${sharedLib.user}", f"repo owner incorrect\ngot:{repo}"
                assert repo["description"] == "asdf", f"repo description incorrect\ngot:{repo}"
                assert repo["private"] == False, f"repo private incorrect\ngot:{repo}"
                assert repo["default_branch"] == "mega", f"repo default branch incorrect\ngot:{repo}"
              '';
          };
          repoInfo = {
            withRepoSetup = true;
            withAuth = true;
            test = ''
              repo = ${
                testLib.berg_run_json {
                  cmd = ''repo info'';
                  cdTestRepo = true;
                }
              }

              printj(repo)

              assert repo["name"] == "${sharedLib.repo}", f"repo name incorrect\ngot:{repo}"
              assert repo["owner"]["login"] == "${sharedLib.user}", f"repo owner incorrect\ngot:{repo}"
              assert repo["description"] == "", f"repo description incorrect\ngot:{repo}"
              assert repo["private"] == False, f"repo private incorrect\ngot:{repo}"
              assert repo["default_branch"] == "${sharedLib.main}", f"repo default branch incorrect\ngot:{repo}"
            '';
          };
          repoInfoWithoutGitRepo = {
            withAuth = true;
            withRepoSetup = true;
            test = ''
              # Test repo info with OWNER/REPO argument from outside a git repository
              repo = ${
                testLib.berg_run_json {
                  cmd = ''repo info --owner-repo ${sharedLib.user}/${sharedLib.repo}'';
                  cdTestRepo = false;
                }
              }

              printj(repo)

              assert repo["name"] == "${sharedLib.repo}", f"repo name incorrect\ngot:{repo}"
              assert repo["owner"]["login"] == "${sharedLib.user}", f"repo owner incorrect\ngot:{repo}"
              assert repo["description"] == "", f"repo description incorrect\ngot:{repo}"
              assert repo["private"] == False, f"repo private incorrect\ngot:{repo}"
              assert repo["default_branch"] == "${sharedLib.main}", f"repo default branch incorrect\ngot:{repo}"
            '';
          };
          repoBranchList = {
            withAuth = true;
            withRepoSetup = true;
            withChangeBranches = true;
            test = ''
              branches = ${
                testLib.berg_run_json {
                  cmd = ''repo branch list'';
                  cdTestRepo = true;
                }
              }

              printj(branches)

              assert len(branches) == 2, f"branches amount incorrect\ngot:{branches}"

              branch_names = [b["name"] for b in branches] 
              assert "${sharedLib.main}" in branch_names, f"${sharedLib.main} wasn't listed\ngot:{branch_names}"
              assert "${sharedLib.branch}" in branch_names, f"${sharedLib.branch} wasn't listed\ngot:{branch_names}"
            '';
          };
          repoBranchDelete = {
            withAuth = true;
            withRepoSetup = true;
            withChangeBranches = true;
            test = ''
              ${testLib.berg_run {
                cmd = ''repo branch delete ${sharedLib.branch}'';
                cdTestRepo = true;
              }}

              branches = ${
                testLib.berg_run_json {
                  cmd = ''repo branch list'';
                  cdTestRepo = true;
                }
              }

              printj(branches)

              assert len(branches) == 1, f"branches amount incorrect\ngot:{branches}"

              branch_names = [b["name"] for b in branches] 
              assert "${sharedLib.main}" in branch_names, f"${sharedLib.main} wasn't listed\ngot:{branch_names}"
              assert "${sharedLib.branch}" not in branch_names, f"${sharedLib.branch} was listed\ngot:{branch_names}"
            '';
          };
        }
        |> testLib.embeddInFlake;
    };
}
