{ inputs, ... }:
{

  flake.homeModules = rec {
    default = berg;
    berg =
      {
        config,
        lib,
        pkgs,
        ...
      }:
      let
        tomlFormat = pkgs.formats.toml { };
        cfg = config.programs.berg;
      in
      {

        options.programs.berg = {
          enable = lib.mkEnableOption "codeberg-cli";
          package = lib.mkPackageOption inputs.self.packages.${pkgs.stdenv.system} "berg" { };
          settings = lib.mkOption {
            description = ''
              Configuration values for the cli.

              See `berg config info` for all possible values.
            '';
            example = {
              editor = "nvim";
              max_width = 80;
            };
            inherit (tomlFormat) type;
            default = { };
          };

        };

        config = lib.mkIf cfg.enable {

          home.packages = [
            cfg.package
          ];

          home.file.".config/berg-cli/berg.toml" = {
            enable = cfg.settings != { };
            source = tomlFormat.generate "berg.toml" cfg.settings;
          };

        };

      };
  };

  perSystem =
    {
      pkgs,
      ...
    }:
    let
      home-manager = builtins.getFlake "github:nix-community/home-manager/929535c3082afdf0b18afec5ea1ef14d7689ff1c";
    in
    {
      checks = {
        home-manager-basic = pkgs.testers.runNixOSTest {
          name = "home-manager-tests-work";
          nodes.machine =
            { ... }:
            {
              imports = [ home-manager.nixosModules.home-manager ];
              users.users.alice = {
                isNormalUser = true;
                initialPassword = "alice";
              };
              home-manager.users.alice = {
                home = {
                  file.".config/example/config.txt".text = "Hello, Nix!";
                  stateVersion = "25.05";
                };
              };
            };
          testScript = ''
            machine.start()
            machine.wait_for_unit("multi-user.target")
            content = machine.succeed("su - alice -c 'cat ~/.config/example/config.txt'")

            assert content == "Hello, Nix!", "home manager file content not as expected"
          '';
        };
        home-manager-berg = pkgs.testers.runNixOSTest {
          name = "home-manager-tests-work";
          nodes.machine =
            { ... }:
            {
              imports = [ home-manager.nixosModules.home-manager ];

              users.users.alice = {
                isNormalUser = true;
                initialPassword = "alice";
              };

              home-manager.users.alice = {
                imports = [ inputs.self.homeModules.berg ];

                programs.berg = {
                  enable = true;
                  settings = {
                    editor = "nvim";
                    max_width = 100;
                  };
                };

                home.stateVersion = "25.05";
              };
            };
          testScript = ''
            machine.start()
            machine.wait_for_unit("multi-user.target")
            machine.succeed("su - alice -c 'berg --help'")
            config = machine.succeed("su - alice -c 'berg config info'")

            print(config)
            assert ".config/berg-cli/berg.toml" in config, "home manager config isn't in use"
            assert "nvim" in config, "home manager sets nvim as editor"
            assert "100" in config, "home manager sets 100 as max width"
          '';
        };
      };
    };

}
