use anyhow::Context;

/// Given a list of `items`, this function allows the user to interactively select a subset of the
/// given list.
///
/// - To guide the user, you have to provide a `prompt`
/// - The `is_selected` closure can help to preselect a certain set of items that pass the
///   predicate
/// - Some types are not natively displayable. In these cases you can decide what to show to the
///   user for each item with `f_display`
pub fn multi_fuzzy_select_with_key<T>(
    items: &[T],
    prompt: impl AsRef<str>,
    is_selected: impl Fn(&T) -> bool,
    f_display: impl Fn(&T) -> String,
) -> anyhow::Result<Vec<&T>> {
    // collect pre-selected items
    let already_selected = items
        .iter()
        .enumerate()
        .filter(|(_, elem)| is_selected(elem))
        .map(|(idx, _)| idx)
        .collect::<Vec<_>>();

    // collect what's shown to the user
    let displayed_items = items.iter().map(f_display).collect::<Vec<_>>();

    // do the interactive selection
    let selected_indices = inquire::MultiSelect::new(prompt.as_ref(), displayed_items)
        .with_default(&already_selected)
        .raw_prompt()
        .context("There's nothing to select from")?;

    // get the items for the selected indices
    let selected_items = selected_indices
        .into_iter()
        .map(|raw| raw.index)
        .filter_map(|idx| items.get(idx))
        .collect::<Vec<_>>();

    Ok(selected_items)
}

/// Basically the same as [`fuzzy_select_with_key_with_default`] without a default value
pub fn fuzzy_select_with_key<T>(
    items: &[T],
    prompt: impl AsRef<str>,
    f_display: impl Fn(&T) -> String,
) -> anyhow::Result<&T> {
    fuzzy_select_with_key_with_default(items, prompt, f_display, None)
}

/// Given a list of `items`, this function allows the user to interactively select a *exactly one*
/// item of the given list.
///
/// - To guide the user, you have to provide a `prompt`
/// - Some types are not natively displayable. In these cases you can decide what to show to the
///   user for each item with `f_display`
/// - The `default_index` optional index value can pre-select one item
pub fn fuzzy_select_with_key_with_default<T>(
    items: &[T],
    prompt: impl AsRef<str>,
    f_display: impl Fn(&T) -> String,
    default_index: Option<usize>,
) -> anyhow::Result<&T> {
    // return `None` if we have nothing to select from
    if items.is_empty() {
        anyhow::bail!("Nothing to select from. Aborting.")
    }

    let displayed_items = items.iter().map(f_display).collect::<Vec<_>>();

    // build standard dialogue
    let mut dialogue = inquire::Select::new(prompt.as_ref(), displayed_items);

    // optionally add default selection
    if let Some(index) = default_index {
        dialogue = dialogue.with_starting_cursor(index);
    }

    // select an item by key
    let selected_index = dialogue.raw_prompt().map_err(anyhow::Error::from)?.index;

    Ok(&items[selected_index])
}

/// Common confimation prompt (y/n) which maps
///
/// - `y` -> `true`
/// - `n` -> `false`
pub fn confirm_with_prompt(prompt: &str) -> anyhow::Result<bool> {
    inquire::Confirm::new(prompt)
        .with_help_message("(y/n)?")
        .prompt()
        .map_err(anyhow::Error::from)
}
