use crate::actions::GeneralArgs;
use crate::render::option::option_display;
use crate::render::spinner::spin_until_ready;
use crate::types::context::BergContext;
use crate::types::git::OwnerRepo;
use clap::Parser;
use forgejo_api::structs::{IssueGetMilestonesListQuery, Milestone};

/// List all milestones in the current repository
#[derive(Parser, Debug)]
pub struct ListMilestonesArgs {
    /// Number of milestones to be displayed
    #[arg(short, long, value_name = "N", default_value_t = 5)]
    pub count: usize,
}

impl ListMilestonesArgs {
    pub async fn run(self, general_args: GeneralArgs) -> anyhow::Result<()> {
        let _ = general_args;
        let ctx = BergContext::new(self, general_args).await?;

        let OwnerRepo { repo, owner } = ctx.owner_repo()?;
        let milestones_list = spin_until_ready(ctx.client.issue_get_milestones_list(
            owner.as_str(),
            repo.as_str(),
            IssueGetMilestonesListQuery::default(),
        ))
        .await?;

        present_milestones_list(&ctx, milestones_list);

        Ok(())
    }
}

fn present_milestones_list(ctx: &BergContext<ListMilestonesArgs>, milestones: Vec<Milestone>) {
    let milestones_empty = milestones.is_empty();

    let mut table = ctx.make_table();

    table
        .set_header(vec![format!(
            "Milestones{}",
            milestones_empty.then_some(" (empty)").unwrap_or_default()
        )])
        .add_rows(
            milestones
                .into_iter()
                .map(|milestone| vec![option_display(&milestone.title)]),
        );

    println!("{table}", table = table.show());
}
