/* GIMP - The GNU Image Manipulation Program
 * Copyright (C) 1995-2003 Spencer Kimball and Peter Mattis
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

/* NOTE: This file is auto-generated by pdbgen.pl. */

#include "config.h"

#include "stamp-pdbgen.h"

#include <cairo.h>

#include <gegl.h>

#include <gdk-pixbuf/gdk-pixbuf.h>

#include "libgimpbase/gimpbase.h"
#include "libgimpcolor/gimpcolor.h"
#include "libgimpconfig/gimpconfig.h"
#include "libgimpmath/gimpmath.h"

#include "libgimpbase/gimpbase.h"

#include "pdb-types.h"

#include "config/gimpcoreconfig.h"
#include "core/gimp.h"
#include "core/gimpchannel.h"
#include "core/gimpcontext.h"
#include "core/gimpdrawable-operation.h"
#include "core/gimpdrawable.h"
#include "core/gimpimage-crop.h"
#include "core/gimpimage-resize.h"
#include "core/gimpimage-rotate.h"
#include "core/gimpimage-undo.h"
#include "core/gimpimage.h"
#include "core/gimpparamspecs.h"
#include "core/gimppickable-auto-shrink.h"
#include "core/gimppickable.h"
#include "gegl/gimp-babl.h"
#include "gegl/gimp-gegl-utils.h"

#include "gimppdb.h"
#include "gimppdberror.h"
#include "gimppdb-utils.h"
#include "gimpprocedure.h"
#include "internal-procs.h"

#include "gimp-intl.h"


static GeglNode *
wrap_in_graph (GeglNode *node)
{
  GeglNode *new_node;
  GeglNode *input;
  GeglNode *output;

  new_node = gegl_node_new ();

  gegl_node_add_child (new_node, node);
  g_object_unref (node);

  gimp_gegl_node_set_underlying_operation (new_node, node);

  input  = gegl_node_get_input_proxy  (new_node, "input");
  output = gegl_node_get_output_proxy (new_node, "output");

  gegl_node_link_many (input,
                       node,
                       output,
                       NULL);

  return new_node;
}

static GeglNode *
wrap_in_selection_bounds (GeglNode     *node,
                          GimpDrawable *drawable)
{
  gint x, y;
  gint width, height;

  if (gimp_item_mask_intersect (GIMP_ITEM (drawable),
                                &x, &y, &width, &height))
    {
      GeglNode *new_node;
      GeglNode *input;
      GeglNode *output;
      GeglNode *translate_before;
      GeglNode *crop;
      GeglNode *translate_after;

      new_node = gegl_node_new ();

      gegl_node_add_child (new_node, node);
      g_object_unref (node);

      gimp_gegl_node_set_underlying_operation (new_node, node);

      input  = gegl_node_get_input_proxy  (new_node, "input");
      output = gegl_node_get_output_proxy (new_node, "output");

      translate_before = gegl_node_new_child (new_node,
                                              "operation", "gegl:translate",
                                              "x",         (gdouble) -x,
                                              "y",         (gdouble) -y,
                                              NULL);
      crop = gegl_node_new_child (new_node,
                                  "operation", "gegl:crop",
                                  "width",     (gdouble) width,
                                  "height",    (gdouble) height,
                                  NULL);
      translate_after = gegl_node_new_child (new_node,
                                             "operation", "gegl:translate",
                                             "x",         (gdouble) x,
                                             "y",         (gdouble) y,
                                             NULL);

      gegl_node_link_many (input,
                           translate_before,
                           crop,
                           node,
                           translate_after,
                           output,
                           NULL);

      return new_node;
    }
  else
    {
      return node;
    }
}

static GeglNode *
wrap_in_gamma_cast (GeglNode     *node,
                    GimpDrawable *drawable)
{
  if (gimp_drawable_get_trc (drawable) != GIMP_TRC_LINEAR)
    {
      const Babl *drawable_format;
      const Babl *cast_format;
      GeglNode   *new_node;
      GeglNode   *input;
      GeglNode   *output;
      GeglNode   *cast_before;
      GeglNode   *cast_after;

      drawable_format = gimp_drawable_get_format (drawable);

      cast_format =
        gimp_babl_format (gimp_babl_format_get_base_type (drawable_format),
                          gimp_babl_precision (gimp_babl_format_get_component_type (drawable_format),
                                               GIMP_TRC_LINEAR),
                          babl_format_has_alpha (drawable_format),
                          babl_format_get_space (drawable_format));

      new_node = gegl_node_new ();

      gegl_node_add_child (new_node, node);
      g_object_unref (node);

      gimp_gegl_node_set_underlying_operation (new_node, node);

      input  = gegl_node_get_input_proxy  (new_node, "input");
      output = gegl_node_get_output_proxy (new_node, "output");

      cast_before = gegl_node_new_child (new_node,
                                         "operation",     "gegl:cast-format",
                                         "input-format",  drawable_format,
                                         "output-format", cast_format,
                                         NULL);
      cast_after  = gegl_node_new_child (new_node,
                                         "operation",     "gegl:cast-format",
                                         "input-format",  cast_format,
                                         "output-format", drawable_format,
                                         NULL);

      gegl_node_link_many (input,
                           cast_before,
                           node,
                           cast_after,
                           output,
                           NULL);

      return new_node;
    }
  else
    {
      return node;
    }
}

static GeglNode *
create_buffer_source_node (GeglNode     *parent,
                           GimpDrawable *drawable)
{
  GeglNode   *new_node;
  GeglBuffer *buffer;

  buffer = gimp_drawable_get_buffer (drawable);
  g_object_ref (buffer);
  new_node = gegl_node_new_child (parent,
                                  "operation", "gegl:buffer-source",
                                  "buffer", buffer,
                                  NULL);
  g_object_unref (buffer);
  return new_node;
}

static gboolean
bump_map (GimpDrawable *drawable,
          GimpDrawable *bump_map,
          gdouble       azimuth,
          gdouble       elevation,
          gint          depth,
          gint          offset_x,
          gint          offset_y,
          gdouble       waterlevel,
          gdouble       ambient,
          gboolean      compensate,
          gboolean      invert,
          gint          type,
          gboolean      tiled,
          GimpProgress  *progress,
          GError       **error)
{
  if (gimp_pdb_item_is_attached (GIMP_ITEM (drawable), NULL,
                                 GIMP_PDB_ITEM_CONTENT, error) &&
      gimp_pdb_item_is_not_group (GIMP_ITEM (drawable), error))
    {
      GeglNode *graph;
      GeglNode *node;
      GeglNode *src_node;

      node = gegl_node_new_child (NULL,
                                  "operation", "gegl:bump-map",
                                  "tiled",      tiled,
                                  "type",       type,
                                  "compensate", compensate,
                                  "invert",     invert,
                                  "azimuth",    azimuth,
                                  "elevation",  elevation,
                                  "depth",      depth,
                                  "offset_x",   offset_x,
                                  "offset_y",   offset_y,
                                  "waterlevel", waterlevel,
                                  "ambient",    ambient,
                                  NULL);

      graph = wrap_in_graph (node);

      src_node = create_buffer_source_node (graph, bump_map);

      gegl_node_connect (src_node, "output", node, "aux");

      gimp_drawable_apply_operation (drawable, progress,
                                     C_("undo-type", "Bump Map"),
                                     graph);
      g_object_unref (graph);

      return TRUE;
    }
    else
      return FALSE;
}

static gboolean
displace (GimpDrawable  *drawable,
          gdouble        amount_x,
          gdouble        amount_y,
          gboolean       do_x,
          gboolean       do_y,
          GimpDrawable  *displace_map_x,
          GimpDrawable  *displace_map_y,
          gint           displace_type,
          gint           displace_mode,
          GimpProgress  *progress,
          GError       **error)
{
  if (gimp_pdb_item_is_attached (GIMP_ITEM (drawable), NULL,
                                 GIMP_PDB_ITEM_CONTENT, error) &&
      gimp_pdb_item_is_not_group (GIMP_ITEM (drawable), error))
    {
      if (do_x || do_y)
        {
          GeglNode *graph;
          GeglNode *node;
          GeglAbyssPolicy abyss_policy = GEGL_ABYSS_NONE;

          switch (displace_type)
            {
              case 1:
                abyss_policy = GEGL_ABYSS_LOOP;
                break;
              case 2:
                abyss_policy = GEGL_ABYSS_CLAMP;
                break;
              case 3:
                abyss_policy = GEGL_ABYSS_BLACK;
                break;
            }

          node = gegl_node_new_child (NULL,
                                      "operation",     "gegl:displace",
                                      "displace_mode", displace_mode,
                                      "sampler_type",  GEGL_SAMPLER_CUBIC,
                                      "abyss_policy",  abyss_policy,
                                      "amount_x",      amount_x,
                                      "amount_y",      amount_y,
                                      NULL);

          graph = wrap_in_graph (node);

          if (do_x)
            {
              GeglNode *src_node;
              src_node = create_buffer_source_node (graph, displace_map_x);
              gegl_node_connect (src_node, "output", node, "aux");
            }

          if (do_y)
            {
              GeglNode *src_node;
              src_node = create_buffer_source_node (graph, displace_map_y);
              gegl_node_connect (src_node, "output", node, "aux2");
            }

          gimp_drawable_apply_operation (drawable, progress,
                                         C_("undo-type", "Displace"),
                                         graph);
          g_object_unref (graph);
        }

      return TRUE;
    }
  else
    return FALSE;
}

static gboolean
gaussian_blur (GimpDrawable  *drawable,
               gdouble        horizontal,
               gdouble        vertical,
               gint           method,
               GimpProgress  *progress,
               GError       **error)
{
  if (gimp_pdb_item_is_attached (GIMP_ITEM (drawable), NULL,
                                 GIMP_PDB_ITEM_CONTENT, error) &&
      gimp_pdb_item_is_not_group (GIMP_ITEM (drawable), error))
    {
      GeglNode *node;

      node = gegl_node_new_child (NULL,
                                  "operation",    "gegl:gaussian-blur",
                                  "std-dev-x",    horizontal,
                                  "std-dev-y",    vertical,
                                  "filter",       method,
                                  "abyss-policy", 1,
                                  NULL);

      node = wrap_in_gamma_cast (node, drawable);

      gimp_drawable_apply_operation (drawable, progress,
                                     C_("undo-type", "Gaussian Blur"),
                                     node);
      g_object_unref (node);

      return TRUE;
    }

  return FALSE;
}

static gint
newsprint_color_model (gint colorspace)
{
  switch (colorspace)
    {
    case 0: return 1; /* black on white */
    case 1: return 2; /* rgb */
    case 2: return 3; /* cmyk */
    case 3: return 1; /* black on white */
    }

  return 2;
}

static gint
newsprint_pattern (gint spotfn)
{
  switch (spotfn)
    {
    case 0: return 1; /* circle */
    case 1: return 0; /* line */
    case 2: return 2; /* diamond */
    case 3: return 4; /* ps circle */
    case 4: return 2; /* FIXME postscript diamond */
    }

  return 1;
}

static gdouble
newsprint_angle (gdouble angle)
{
  while (angle > 180.0)
    angle -= 360.0;

  while (angle < -180.0)
    angle += 360.0;

  return angle;
}

static GimpValueArray *
plug_in_autocrop_invoker (GimpProcedure         *procedure,
                          Gimp                  *gimp,
                          GimpContext           *context,
                          GimpProgress          *progress,
                          const GimpValueArray  *args,
                          GError               **error)
{
  gboolean success = TRUE;
  GimpImage *image;
  GimpDrawable *drawable;

  image = g_value_get_object (gimp_value_array_index (args, 1));
  drawable = g_value_get_object (gimp_value_array_index (args, 2));

  if (success)
    {
      if (gimp_pdb_item_is_attached (GIMP_ITEM (drawable), NULL,
                                     GIMP_PDB_ITEM_CONTENT, error))
        {
          gint x, y, width, height;
          gint off_x, off_y;

          gimp_pickable_auto_shrink (GIMP_PICKABLE (drawable),
                                     0, 0,
                                     gimp_item_get_width  (GIMP_ITEM (drawable)),
                                     gimp_item_get_height (GIMP_ITEM (drawable)),
                                     &x, &y, &width, &height);

          gimp_item_get_offset (GIMP_ITEM (drawable), &off_x, &off_y);
          x += off_x;
          y += off_y;

          gimp_image_undo_group_start (image, GIMP_UNDO_GROUP_ITEM_RESIZE,
                                       _("Autocrop image"));

          if (x          < 0                             ||
              y          < 0                             ||
              x + width  > gimp_image_get_width  (image) ||
              y + height > gimp_image_get_height (image))
            {
              /*
               * partially outside the image area, we need to
               * resize the image to be able to crop properly.
               */
              gimp_image_resize (image, context, width, height, -x, -y, NULL);

              x = y = 0;
            }

          gimp_image_crop (image, context, GIMP_FILL_TRANSPARENT,
                           x, y, width, height, TRUE);

          gimp_image_undo_group_end (image);
        }
      else
        success = FALSE;
    }

  return gimp_procedure_get_return_values (procedure, success,
                                           error ? *error : NULL);
}

static GimpValueArray *
plug_in_autocrop_layer_invoker (GimpProcedure         *procedure,
                                Gimp                  *gimp,
                                GimpContext           *context,
                                GimpProgress          *progress,
                                const GimpValueArray  *args,
                                GError               **error)
{
  gboolean success = TRUE;
  GimpImage *image;
  GimpDrawable *drawable;

  image = g_value_get_object (gimp_value_array_index (args, 1));
  drawable = g_value_get_object (gimp_value_array_index (args, 2));

  if (success)
    {
      if (gimp_pdb_item_is_attached (GIMP_ITEM (drawable), NULL,
                                     GIMP_PDB_ITEM_CONTENT, error))
        {
          GList *layers = gimp_image_get_selected_layers (image);
          GList *iter;
          gint   x, y, width, height;

          if (layers)
            {
              switch (gimp_pickable_auto_shrink (GIMP_PICKABLE (drawable),
                                                 0, 0,
                                                 gimp_item_get_width  (GIMP_ITEM (drawable)),
                                                 gimp_item_get_height (GIMP_ITEM (drawable)),
                                                 &x, &y, &width, &height))
                {
                case GIMP_AUTO_SHRINK_SHRINK:
                  gimp_image_undo_group_start (image, GIMP_UNDO_GROUP_ITEM_RESIZE,
                                               _("Autocrop layer"));

                  for (iter = layers; iter; iter = iter->next)
                      gimp_item_resize (GIMP_ITEM (iter->data),
                                        context, GIMP_FILL_TRANSPARENT,
                                        width, height, -x, -y);

                  gimp_image_undo_group_end (image);
                  break;

                default:
                  break;
                }
            }
          else
            {
              success = FALSE;
            }
        }
      else
        {
          success = FALSE;
        }
    }

  return gimp_procedure_get_return_values (procedure, success,
                                           error ? *error : NULL);
}

static GimpValueArray *
plug_in_bump_map_invoker (GimpProcedure         *procedure,
                          Gimp                  *gimp,
                          GimpContext           *context,
                          GimpProgress          *progress,
                          const GimpValueArray  *args,
                          GError               **error)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  GimpDrawable *bumpmap;
  gdouble azimuth;
  gdouble elevation;
  gint depth;
  gint xofs;
  gint yofs;
  gdouble waterlevel;
  gdouble ambient;
  gboolean compensate;
  gboolean invert;
  gint type;

  drawable = g_value_get_object (gimp_value_array_index (args, 2));
  bumpmap = g_value_get_object (gimp_value_array_index (args, 3));
  azimuth = g_value_get_double (gimp_value_array_index (args, 4));
  elevation = g_value_get_double (gimp_value_array_index (args, 5));
  depth = g_value_get_int (gimp_value_array_index (args, 6));
  xofs = g_value_get_int (gimp_value_array_index (args, 7));
  yofs = g_value_get_int (gimp_value_array_index (args, 8));
  waterlevel = g_value_get_double (gimp_value_array_index (args, 9));
  ambient = g_value_get_double (gimp_value_array_index (args, 10));
  compensate = g_value_get_boolean (gimp_value_array_index (args, 11));
  invert = g_value_get_boolean (gimp_value_array_index (args, 12));
  type = g_value_get_int (gimp_value_array_index (args, 13));

  if (success)
    {
      success = bump_map (drawable,
                          bumpmap,
                          azimuth,
                          elevation,
                          depth,
                          xofs,
                          yofs,
                          waterlevel,
                          ambient,
                          compensate,
                          invert,
                          type,
                          FALSE,
                          progress,
                          error);
    }

  return gimp_procedure_get_return_values (procedure, success,
                                           error ? *error : NULL);
}

static GimpValueArray *
plug_in_c_astretch_invoker (GimpProcedure         *procedure,
                            Gimp                  *gimp,
                            GimpContext           *context,
                            GimpProgress          *progress,
                            const GimpValueArray  *args,
                            GError               **error)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;

  drawable = g_value_get_object (gimp_value_array_index (args, 2));

  if (success)
    {
      if (gimp_pdb_item_is_attached (GIMP_ITEM (drawable), NULL,
                                     GIMP_PDB_ITEM_CONTENT, error) &&
          gimp_pdb_item_is_not_group (GIMP_ITEM (drawable), error))
        {
          GeglNode *node =
            gegl_node_new_child (NULL,
                                 "operation",   "gegl:stretch-contrast",
                                 "keep-colors", (gboolean) FALSE,
                                 NULL);

          gimp_drawable_apply_operation (drawable, progress,
                                         C_("undo-type", "Stretch Contrast"),
                                         node);
          g_object_unref (node);
        }
      else
        success = FALSE;
    }

  return gimp_procedure_get_return_values (procedure, success,
                                           error ? *error : NULL);
}

static GimpValueArray *
plug_in_cubism_invoker (GimpProcedure         *procedure,
                        Gimp                  *gimp,
                        GimpContext           *context,
                        GimpProgress          *progress,
                        const GimpValueArray  *args,
                        GError               **error)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gdouble tile_size;
  gdouble tile_saturation;
  gint bg_color;

  drawable = g_value_get_object (gimp_value_array_index (args, 2));
  tile_size = g_value_get_double (gimp_value_array_index (args, 3));
  tile_saturation = g_value_get_double (gimp_value_array_index (args, 4));
  bg_color = g_value_get_int (gimp_value_array_index (args, 5));

  if (success)
    {
      if (gimp_pdb_item_is_attached (GIMP_ITEM (drawable), NULL,
                                     GIMP_PDB_ITEM_CONTENT, error) &&
          gimp_pdb_item_is_not_group (GIMP_ITEM (drawable), error))
        {
          GeglColor *color;
          GeglNode  *node;

          if (bg_color)
            {
              color = gegl_color_duplicate (gimp_context_get_background (context));
              gimp_color_set_alpha (color, 0.0);
            }
          else
            {
              color = gegl_color_new ("black");
              gegl_color_set_rgba_with_space (color, 0.0, 0.0, 0.0, 0.0, NULL);
            }

          node = gegl_node_new_child (NULL,
                                      "operation",       "gegl:cubism",
                                      "tile-size",       tile_size,
                                      "tile-saturation", tile_saturation,
                                      "bg-color",        color,
                                      NULL);
          g_object_unref (color);

          gimp_drawable_apply_operation (drawable, progress,
                                         C_("undo-type", "Cubism"),
                                         node);
          g_object_unref (node);
        }
      else
        success = FALSE;
    }

  return gimp_procedure_get_return_values (procedure, success,
                                           error ? *error : NULL);
}

static GimpValueArray *
plug_in_displace_invoker (GimpProcedure         *procedure,
                          Gimp                  *gimp,
                          GimpContext           *context,
                          GimpProgress          *progress,
                          const GimpValueArray  *args,
                          GError               **error)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gdouble amount_x;
  gdouble amount_y;
  gboolean do_x;
  gboolean do_y;
  GimpDrawable *displace_map_x;
  GimpDrawable *displace_map_y;
  gint displace_type;

  drawable = g_value_get_object (gimp_value_array_index (args, 2));
  amount_x = g_value_get_double (gimp_value_array_index (args, 3));
  amount_y = g_value_get_double (gimp_value_array_index (args, 4));
  do_x = g_value_get_boolean (gimp_value_array_index (args, 5));
  do_y = g_value_get_boolean (gimp_value_array_index (args, 6));
  displace_map_x = g_value_get_object (gimp_value_array_index (args, 7));
  displace_map_y = g_value_get_object (gimp_value_array_index (args, 8));
  displace_type = g_value_get_int (gimp_value_array_index (args, 9));

  if (success)
    {
      success = displace (drawable,
                          amount_x,
                          amount_y,
                          do_x,
                          do_y,
                          displace_map_x,
                          displace_map_y,
                          displace_type,
                          0,
                          progress,
                          error);
    }

  return gimp_procedure_get_return_values (procedure, success,
                                           error ? *error : NULL);
}

static GimpValueArray *
plug_in_edge_invoker (GimpProcedure         *procedure,
                      Gimp                  *gimp,
                      GimpContext           *context,
                      GimpProgress          *progress,
                      const GimpValueArray  *args,
                      GError               **error)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gdouble amount;
  gint warpmode;
  gint edgemode;

  drawable = g_value_get_object (gimp_value_array_index (args, 2));
  amount = g_value_get_double (gimp_value_array_index (args, 3));
  warpmode = g_value_get_int (gimp_value_array_index (args, 4));
  edgemode = g_value_get_int (gimp_value_array_index (args, 5));

  if (success)
    {
      if (gimp_pdb_item_is_attached (GIMP_ITEM (drawable), NULL,
                                     GIMP_PDB_ITEM_CONTENT, error) &&
          gimp_pdb_item_is_not_group (GIMP_ITEM (drawable), error))
        {
          GeglNode        *node;
          GeglAbyssPolicy  border_behavior = GEGL_ABYSS_NONE;

          switch (warpmode)
            {
            case 0:
              border_behavior = GEGL_ABYSS_NONE;
              break;

            case 1:
              border_behavior = GEGL_ABYSS_LOOP;
              break;

            case 2:
              border_behavior = GEGL_ABYSS_CLAMP;
              break;

            case 3:
              border_behavior = GEGL_ABYSS_BLACK;
              break;
            }

          node = gegl_node_new_child (NULL,
                                      "operation",       "gegl:edge",
                                      "algorithm",       edgemode,
                                      "amount",          amount,
                                      "border-behavior", border_behavior,
                                      NULL);

          gimp_drawable_apply_operation (drawable, progress,
                                         C_("undo-type", "Edge"),
                                         node);
          g_object_unref (node);
        }
      else
        success = FALSE;
    }

  return gimp_procedure_get_return_values (procedure, success,
                                           error ? *error : NULL);
}

static GimpValueArray *
plug_in_emboss_invoker (GimpProcedure         *procedure,
                        Gimp                  *gimp,
                        GimpContext           *context,
                        GimpProgress          *progress,
                        const GimpValueArray  *args,
                        GError               **error)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gdouble azimuth;
  gdouble elevation;
  gint depth;
  gboolean emboss;

  drawable = g_value_get_object (gimp_value_array_index (args, 2));
  azimuth = g_value_get_double (gimp_value_array_index (args, 3));
  elevation = g_value_get_double (gimp_value_array_index (args, 4));
  depth = g_value_get_int (gimp_value_array_index (args, 5));
  emboss = g_value_get_boolean (gimp_value_array_index (args, 6));

  if (success)
    {
      if (gimp_pdb_item_is_attached (GIMP_ITEM (drawable), NULL,
                                     GIMP_PDB_ITEM_CONTENT, error) &&
          gimp_pdb_item_is_not_group (GIMP_ITEM (drawable), error))
        {
          GeglNode *node;

          node = gegl_node_new_child (NULL,
                                      "operation",  "gegl:emboss",
                                      "type",       emboss ? 0 : 1,
                                      "azimuth",    azimuth,
                                      "elevation",  elevation,
                                      "depth",      depth,
                                      NULL);

          node = wrap_in_gamma_cast (node, drawable);

          gimp_drawable_apply_operation (drawable, progress,
                                         C_("undo-type", "Emboss"),
                                         node);
        }
      else
        success = FALSE;
    }

  return gimp_procedure_get_return_values (procedure, success,
                                           error ? *error : NULL);
}

static GimpValueArray *
plug_in_gauss_invoker (GimpProcedure         *procedure,
                       Gimp                  *gimp,
                       GimpContext           *context,
                       GimpProgress          *progress,
                       const GimpValueArray  *args,
                       GError               **error)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gdouble horizontal;
  gdouble vertical;
  gint method;

  drawable = g_value_get_object (gimp_value_array_index (args, 2));
  horizontal = g_value_get_double (gimp_value_array_index (args, 3));
  vertical = g_value_get_double (gimp_value_array_index (args, 4));
  method = g_value_get_int (gimp_value_array_index (args, 5));

  if (success)
    {
      success = gaussian_blur (drawable, horizontal, vertical, method, progress,
                               error);
    }

  return gimp_procedure_get_return_values (procedure, success,
                                           error ? *error : NULL);
}

static GimpValueArray *
plug_in_maze_invoker (GimpProcedure         *procedure,
                      Gimp                  *gimp,
                      GimpContext           *context,
                      GimpProgress          *progress,
                      const GimpValueArray  *args,
                      GError               **error)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gint width;
  gint height;
  gboolean tileable;
  gint algorithm;
  gint seed;

  drawable = g_value_get_object (gimp_value_array_index (args, 2));
  width = g_value_get_int (gimp_value_array_index (args, 3));
  height = g_value_get_int (gimp_value_array_index (args, 4));
  tileable = g_value_get_boolean (gimp_value_array_index (args, 5));
  algorithm = g_value_get_int (gimp_value_array_index (args, 6));
  seed = g_value_get_int (gimp_value_array_index (args, 7));

  if (success)
    {
      if (gimp_pdb_item_is_attached (GIMP_ITEM (drawable), NULL,
                                     GIMP_PDB_ITEM_CONTENT, error) &&
          gimp_pdb_item_is_not_group (GIMP_ITEM (drawable), error))
        {
          GeglNode  *node;
          GeglColor *fg_color;
          GeglColor *bg_color;

          fg_color = gimp_context_get_foreground (context);
          bg_color = gimp_context_get_background (context);

          node =  gegl_node_new_child (NULL,
                                       "operation",      "gegl:maze",
                                       "x",              width,
                                       "y",              height,
                                       "algorithm-type", algorithm,
                                       "tileable",       tileable,
                                       "seed",           seed,
                                       "fg-color",       fg_color,
                                       "bg-color",       bg_color,
                                       NULL);

          gimp_drawable_apply_operation (drawable, progress,
                                         C_("undo-type", "Maze"),
                                        node);
          g_object_unref (node);
        }
      else
        success = FALSE;
    }

  return gimp_procedure_get_return_values (procedure, success,
                                           error ? *error : NULL);
}

static GimpValueArray *
plug_in_oilify_invoker (GimpProcedure         *procedure,
                        Gimp                  *gimp,
                        GimpContext           *context,
                        GimpProgress          *progress,
                        const GimpValueArray  *args,
                        GError               **error)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gint mask_size;
  gint mode;

  drawable = g_value_get_object (gimp_value_array_index (args, 2));
  mask_size = g_value_get_int (gimp_value_array_index (args, 3));
  mode = g_value_get_int (gimp_value_array_index (args, 4));

  if (success)
    {
      if (gimp_pdb_item_is_attached (GIMP_ITEM (drawable), NULL,
                                     GIMP_PDB_ITEM_CONTENT, error) &&
          gimp_pdb_item_is_not_group (GIMP_ITEM (drawable), error))
        {
          GeglNode *node;

          node = gegl_node_new_child (NULL,
                                      "operation",       "gegl:oilify",
                                      "mask-radius",     MAX (1, mask_size / 2),
                                      "use-inten",       mode ? TRUE : FALSE,
                                      NULL);

          gimp_drawable_apply_operation (drawable, progress,
                                         C_("undo-type", "Oilify"),
                                         node);
          g_object_unref (node);
        }
      else
        success = FALSE;
    }

  return gimp_procedure_get_return_values (procedure, success,
                                           error ? *error : NULL);
}

static GimpValueArray *
plug_in_plasma_invoker (GimpProcedure         *procedure,
                        Gimp                  *gimp,
                        GimpContext           *context,
                        GimpProgress          *progress,
                        const GimpValueArray  *args,
                        GError               **error)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gint seed;
  gdouble turbulence;

  drawable = g_value_get_object (gimp_value_array_index (args, 2));
  seed = g_value_get_int (gimp_value_array_index (args, 3));
  turbulence = g_value_get_double (gimp_value_array_index (args, 4));

  if (success)
    {
      if (gimp_pdb_item_is_attached (GIMP_ITEM (drawable), NULL,
                                     GIMP_PDB_ITEM_CONTENT, error) &&
          gimp_pdb_item_is_not_group (GIMP_ITEM (drawable), error))
        {
          GimpImage *image = gimp_item_get_image (GIMP_ITEM (drawable));
          GeglNode  *node;
          gint       x, y, width, height;

          gimp_item_mask_intersect (GIMP_ITEM (drawable), &x, &y, &width, &height);

          if (! gimp_channel_is_empty (gimp_image_get_mask (image)))
            x = y = 0;

          node = gegl_node_new_child (NULL,
                                      "operation",  "gegl:plasma",
                                      "seed",       seed,
                                      "turbulence", turbulence,
                                      "x",          x,
                                      "y",          y,
                                      "width",      width,
                                      "height",     height,
                                      NULL);

          gimp_drawable_apply_operation (drawable, progress,
                                         C_("undo-type", "Plasma"),
                                         node);
          g_object_unref (node);
        }
      else
        success = FALSE;
    }

  return gimp_procedure_get_return_values (procedure, success,
                                           error ? *error : NULL);
}

static GimpValueArray *
plug_in_rotate_invoker (GimpProcedure         *procedure,
                        Gimp                  *gimp,
                        GimpContext           *context,
                        GimpProgress          *progress,
                        const GimpValueArray  *args,
                        GError               **error)
{
  gboolean success = TRUE;
  GimpImage *image;
  GimpDrawable *drawable;
  gint angle;
  gboolean everything;

  image = g_value_get_object (gimp_value_array_index (args, 1));
  drawable = g_value_get_object (gimp_value_array_index (args, 2));
  angle = g_value_get_int (gimp_value_array_index (args, 3));
  everything = g_value_get_boolean (gimp_value_array_index (args, 4));

  if (success)
    {
      GimpRotationType rotate_type = angle - 1;

      if (everything)
        {
          gimp_image_rotate (image, context, rotate_type, progress);
        }
      else if (gimp_pdb_item_is_attached (GIMP_ITEM (drawable), NULL,
                                          GIMP_PDB_ITEM_CONTENT, error))
        {
          GimpItem *item = GIMP_ITEM (drawable);
          gint      off_x, off_y;
          gdouble   center_x, center_y;

          gimp_item_get_offset (item, &off_x, &off_y);

          center_x = ((gdouble) off_x + (gdouble) gimp_item_get_width  (item) / 2.0);
          center_y = ((gdouble) off_y + (gdouble) gimp_item_get_height (item) / 2.0);

          gimp_item_rotate (item, context, rotate_type, center_x, center_y,
                            GIMP_IS_CHANNEL (drawable));
        }
      else
        success = FALSE;
    }

  return gimp_procedure_get_return_values (procedure, success,
                                           error ? *error : NULL);
}

static GimpValueArray *
plug_in_noisify_invoker (GimpProcedure         *procedure,
                         Gimp                  *gimp,
                         GimpContext           *context,
                         GimpProgress          *progress,
                         const GimpValueArray  *args,
                         GError               **error)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gboolean independent;
  gdouble noise_1;
  gdouble noise_2;
  gdouble noise_3;
  gdouble noise_4;

  drawable = g_value_get_object (gimp_value_array_index (args, 2));
  independent = g_value_get_boolean (gimp_value_array_index (args, 3));
  noise_1 = g_value_get_double (gimp_value_array_index (args, 4));
  noise_2 = g_value_get_double (gimp_value_array_index (args, 5));
  noise_3 = g_value_get_double (gimp_value_array_index (args, 6));
  noise_4 = g_value_get_double (gimp_value_array_index (args, 7));

  if (success)
    {
      if (gimp_pdb_item_is_attached (GIMP_ITEM (drawable), NULL,
                                     GIMP_PDB_ITEM_CONTENT, error) &&
          gimp_pdb_item_is_not_group (GIMP_ITEM (drawable), error))
        {
          GeglNode *node;
          gdouble   r, g, b, a;

          if (gimp_drawable_is_gray (drawable))
            {
              r = noise_1;
              g = noise_1;
              b = noise_1;
              a = noise_2;
            }
          else
            {
              r = noise_1;
              g = noise_2;
              b = noise_3;
              a = noise_4;
            }

          node = gegl_node_new_child (NULL,
                                      "operation",   "gegl:noise-rgb",
                                      "correlated",  FALSE,
                                      "independent", independent,
                                      "red",         r,
                                      "green",       g,
                                      "blue",        b,
                                      "alpha",       a,
                                      "seed",        g_random_int (),
                                      NULL);

          node = wrap_in_gamma_cast (node, drawable);

          gimp_drawable_apply_operation (drawable, progress,
                                         C_("undo-type", "Noisify"),
                                         node);
          g_object_unref (node);
        }
      else
        success = FALSE;
    }

  return gimp_procedure_get_return_values (procedure, success,
                                           error ? *error : NULL);
}

static GimpValueArray *
plug_in_solid_noise_invoker (GimpProcedure         *procedure,
                             Gimp                  *gimp,
                             GimpContext           *context,
                             GimpProgress          *progress,
                             const GimpValueArray  *args,
                             GError               **error)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gboolean tileable;
  gboolean turbulent;
  gint seed;
  gint detail;
  gdouble xsize;
  gdouble ysize;

  drawable = g_value_get_object (gimp_value_array_index (args, 2));
  tileable = g_value_get_boolean (gimp_value_array_index (args, 3));
  turbulent = g_value_get_boolean (gimp_value_array_index (args, 4));
  seed = g_value_get_int (gimp_value_array_index (args, 5));
  detail = g_value_get_int (gimp_value_array_index (args, 6));
  xsize = g_value_get_double (gimp_value_array_index (args, 7));
  ysize = g_value_get_double (gimp_value_array_index (args, 8));

  if (success)
    {
      if (gimp_pdb_item_is_attached (GIMP_ITEM (drawable), NULL,
                                     GIMP_PDB_ITEM_CONTENT, error) &&
          gimp_pdb_item_is_not_group (GIMP_ITEM (drawable), error))
        {
          GeglNode *node;
          gint      x, y, width, height;

          gimp_item_mask_intersect (GIMP_ITEM (drawable), &x, &y, &width, &height);

          node = gegl_node_new_child (NULL,
                                      "operation", "gegl:noise-solid",
                                      "x-size",    xsize,
                                      "y-size",    ysize,
                                      "detail",    detail,
                                      "tileable",  tileable,
                                      "turbulent", turbulent,
                                      "seed",      seed,
                                      "width",     width,
                                      "height",    height,
                                      NULL);

          gimp_drawable_apply_operation (drawable, progress,
                                         C_("undo-type", "Solid Noise"),
                                         node);
          g_object_unref (node);
        }
      else
        success = FALSE;
    }

  return gimp_procedure_get_return_values (procedure, success,
                                           error ? *error : NULL);
}

static GimpValueArray *
plug_in_spread_invoker (GimpProcedure         *procedure,
                        Gimp                  *gimp,
                        GimpContext           *context,
                        GimpProgress          *progress,
                        const GimpValueArray  *args,
                        GError               **error)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gdouble spread_amount_x;
  gdouble spread_amount_y;

  drawable = g_value_get_object (gimp_value_array_index (args, 2));
  spread_amount_x = g_value_get_double (gimp_value_array_index (args, 3));
  spread_amount_y = g_value_get_double (gimp_value_array_index (args, 4));

  if (success)
    {
      if (gimp_pdb_item_is_attached (GIMP_ITEM (drawable), NULL,
                                     GIMP_PDB_ITEM_CONTENT, error) &&
          gimp_pdb_item_is_not_group (GIMP_ITEM (drawable), error))
        {
          GeglNode *node =
            gegl_node_new_child (NULL,
                                 "operation", "gegl:noise-spread",
                                 "amount-x",  (gint) spread_amount_x,
                                 "amount-y",  (gint) spread_amount_y,
                                 "seed",      g_random_int (),
                                 NULL);

          gimp_drawable_apply_operation (drawable, progress,
                                         C_("undo-type", "Spread"),
                                         node);
          g_object_unref (node);
        }
      else
        success = FALSE;
    }

  return gimp_procedure_get_return_values (procedure, success,
                                           error ? *error : NULL);
}

static GimpValueArray *
plug_in_threshold_alpha_invoker (GimpProcedure         *procedure,
                                 Gimp                  *gimp,
                                 GimpContext           *context,
                                 GimpProgress          *progress,
                                 const GimpValueArray  *args,
                                 GError               **error)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gint threshold;

  drawable = g_value_get_object (gimp_value_array_index (args, 2));
  threshold = g_value_get_int (gimp_value_array_index (args, 3));

  if (success)
    {
      if (gimp_pdb_item_is_attached (GIMP_ITEM (drawable), NULL,
                                     GIMP_PDB_ITEM_CONTENT, error) &&
          gimp_pdb_item_is_not_group (GIMP_ITEM (drawable), error) &&
          gimp_drawable_has_alpha (drawable))
        {
          GeglNode *node =
            gegl_node_new_child (NULL,
                                 "operation", "gimp:threshold-alpha",
                                 "value",     threshold / 255.0,
                                 NULL);

          gimp_drawable_apply_operation (drawable, progress,
                                         C_("undo-type", "Threshold Alpha"),
                                         node);
          g_object_unref (node);
        }
      else
        success = FALSE;
    }

  return gimp_procedure_get_return_values (procedure, success,
                                           error ? *error : NULL);
}

static GimpValueArray *
plug_in_waves_invoker (GimpProcedure         *procedure,
                       Gimp                  *gimp,
                       GimpContext           *context,
                       GimpProgress          *progress,
                       const GimpValueArray  *args,
                       GError               **error)
{
  gboolean success = TRUE;
  GimpDrawable *drawable;
  gdouble amplitude;
  gdouble phase;
  gdouble wavelength;
  gboolean type;

  drawable = g_value_get_object (gimp_value_array_index (args, 2));
  amplitude = g_value_get_double (gimp_value_array_index (args, 3));
  phase = g_value_get_double (gimp_value_array_index (args, 4));
  wavelength = g_value_get_double (gimp_value_array_index (args, 5));
  type = g_value_get_boolean (gimp_value_array_index (args, 6));

  if (success)
    {
      if (gimp_pdb_item_is_attached (GIMP_ITEM (drawable), NULL,
                                     GIMP_PDB_ITEM_CONTENT, error) &&
          gimp_pdb_item_is_not_group (GIMP_ITEM (drawable), error))
        {
          GeglNode *node;
          gdouble   width  = gimp_item_get_width  (GIMP_ITEM (drawable));
          gdouble   height = gimp_item_get_height (GIMP_ITEM (drawable));
          gdouble   aspect;

          while (phase < 0)
            phase += 360.0;

          phase = fmod (phase, 360.0);

          aspect = CLAMP (width / height, 0.1, 10.0);

          node = gegl_node_new_child (NULL,
                                     "operation", "gegl:waves",
                                     "x",         0.5,
                                     "y",         0.5,
                                     "amplitude", amplitude,
                                     "phi",       (phase - 180.0) / 180.0,
                                     "period",    wavelength * 2.0,
                                     "aspect",    aspect,
                                     "clamp",     ! type,
                                     NULL);

          gimp_drawable_apply_operation (drawable, progress,
                                         C_("undo-type", "Waves"),
                                         node);
          g_object_unref (node);
        }
      else
        success = FALSE;
    }

  return gimp_procedure_get_return_values (procedure, success,
                                           error ? *error : NULL);
}

void
register_plug_in_compat_procs (GimpPDB *pdb)
{
  GimpProcedure *procedure;

  /*
   * gimp-plug-in-autocrop
   */
  procedure = gimp_procedure_new (plug_in_autocrop_invoker);
  gimp_object_set_static_name (GIMP_OBJECT (procedure),
                               "plug-in-autocrop");
  gimp_procedure_set_static_help (procedure,
                                  "Remove empty borders from the image",
                                  "Remove empty borders from the image.",
                                  NULL);
  gimp_procedure_set_static_attribution (procedure,
                                         "Spencer Kimball & Peter Mattis",
                                         "Spencer Kimball & Peter Mattis",
                                         "1997");
  gimp_procedure_add_argument (procedure,
                               g_param_spec_enum ("run-mode",
                                                  "run mode",
                                                  "The run mode",
                                                  GIMP_TYPE_RUN_MODE,
                                                  GIMP_RUN_INTERACTIVE,
                                                  GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_image ("image",
                                                      "image",
                                                      "Input image)",
                                                      FALSE,
                                                      GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("drawable",
                                                         "drawable",
                                                         "Input drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_pdb_register_procedure (pdb, procedure);
  g_object_unref (procedure);

  /*
   * gimp-plug-in-autocrop-layer
   */
  procedure = gimp_procedure_new (plug_in_autocrop_layer_invoker);
  gimp_object_set_static_name (GIMP_OBJECT (procedure),
                               "plug-in-autocrop-layer");
  gimp_procedure_set_static_help (procedure,
                                  "Crop the selected layers based on empty borders of the input drawable",
                                  "Crop the selected layers of the input \"image\" based on empty borders of the input \"drawable\". \n\nThe input drawable serves as a base for detecting cropping extents (transparency or background color), and is not necessarily among the cropped layers (the current selected layers).",
                                  NULL);
  gimp_procedure_set_static_attribution (procedure,
                                         "Spencer Kimball & Peter Mattis",
                                         "Spencer Kimball & Peter Mattis",
                                         "1997");
  gimp_procedure_add_argument (procedure,
                               g_param_spec_enum ("run-mode",
                                                  "run mode",
                                                  "The run mode",
                                                  GIMP_TYPE_RUN_MODE,
                                                  GIMP_RUN_INTERACTIVE,
                                                  GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_image ("image",
                                                      "image",
                                                      "Input image)",
                                                      FALSE,
                                                      GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("drawable",
                                                         "drawable",
                                                         "Input drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_pdb_register_procedure (pdb, procedure);
  g_object_unref (procedure);

  /*
   * gimp-plug-in-bump-map
   */
  procedure = gimp_procedure_new (plug_in_bump_map_invoker);
  gimp_object_set_static_name (GIMP_OBJECT (procedure),
                               "plug-in-bump-map");
  gimp_procedure_set_static_help (procedure,
                                  "Create an embossing effect using a bump map",
                                  "This plug-in uses the algorithm described by John Schlag, \"Fast Embossing Effects on Raster Image Data\" in Graphics GEMS IV (ISBN 0-12-336155-9). It takes a drawable to be applied as a bump map to another image and produces a nice embossing effect.",
                                  NULL);
  gimp_procedure_set_static_attribution (procedure,
                                         "Compatibility procedure. Please see 'gegl:bump-map' for credits.",
                                         "Compatibility procedure. Please see 'gegl:bump-map' for credits.",
                                         "2015");
  gimp_procedure_add_argument (procedure,
                               g_param_spec_enum ("run-mode",
                                                  "run mode",
                                                  "The run mode",
                                                  GIMP_TYPE_RUN_MODE,
                                                  GIMP_RUN_INTERACTIVE,
                                                  GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_image ("image",
                                                      "image",
                                                      "Input image (unused)",
                                                      FALSE,
                                                      GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("drawable",
                                                         "drawable",
                                                         "Input drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("bumpmap",
                                                         "bumpmap",
                                                         "Bump map drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("azimuth",
                                                    "azimuth",
                                                    "Azimuth",
                                                    0.0, 360.0, 0.0,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("elevation",
                                                    "elevation",
                                                    "Elevation",
                                                    0.5, 90.0, 0.5,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("depth",
                                                 "depth",
                                                 "Depth",
                                                 1, 65, 1,
                                                 GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("xofs",
                                                 "xofs",
                                                 "X offset",
                                                 G_MININT32, G_MAXINT32, 0,
                                                 GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("yofs",
                                                 "yofs",
                                                 "Y offset",
                                                 G_MININT32, G_MAXINT32, 0,
                                                 GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("waterlevel",
                                                    "waterlevel",
                                                    "Level that full transparency should represent",
                                                    0.0, 1.0, 0.0,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("ambient",
                                                    "ambient",
                                                    "Ambient lighting factor",
                                                    0.0, 1.0, 0.0,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_boolean ("compensate",
                                                     "compensate",
                                                     "Compensate for darkening",
                                                     FALSE,
                                                     GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_boolean ("invert",
                                                     "invert",
                                                     "Invert bumpmap",
                                                     FALSE,
                                                     GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("type",
                                                 "type",
                                                 "Type of map { LINEAR (0), SPHERICAL (1), SINUSOIDAL (2) }",
                                                 0, 3, 0,
                                                 GIMP_PARAM_READWRITE));
  gimp_pdb_register_procedure (pdb, procedure);
  g_object_unref (procedure);

  /*
   * gimp-plug-in-c-astretch
   */
  procedure = gimp_procedure_new (plug_in_c_astretch_invoker);
  gimp_object_set_static_name (GIMP_OBJECT (procedure),
                               "plug-in-c-astretch");
  gimp_procedure_set_static_help (procedure,
                                  "Stretch contrast to cover the maximum possible range",
                                  "This simple plug-in does an automatic contrast stretch. For each channel in the image, it finds the minimum and maximum values... it uses those values to stretch the individual histograms to the full contrast range. For some images it may do just what you want; for others it may not work that well.",
                                  NULL);
  gimp_procedure_set_static_attribution (procedure,
                                         "Compatibility procedure. Please see 'gegl:stretch-contrast' for credits.",
                                         "Compatibility procedure. Please see 'gegl:stretch-contrast' for credits.",
                                         "2013");
  gimp_procedure_add_argument (procedure,
                               g_param_spec_enum ("run-mode",
                                                  "run mode",
                                                  "The run mode",
                                                  GIMP_TYPE_RUN_MODE,
                                                  GIMP_RUN_INTERACTIVE,
                                                  GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_image ("image",
                                                      "image",
                                                      "Input image (unused)",
                                                      FALSE,
                                                      GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("drawable",
                                                         "drawable",
                                                         "Input drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_pdb_register_procedure (pdb, procedure);
  g_object_unref (procedure);

  /*
   * gimp-plug-in-cubism
   */
  procedure = gimp_procedure_new (plug_in_cubism_invoker);
  gimp_object_set_static_name (GIMP_OBJECT (procedure),
                               "plug-in-cubism");
  gimp_procedure_set_static_help (procedure,
                                  "Convert the image into randomly rotated square blobs",
                                  "Convert the image into randomly rotated square blobs.",
                                  NULL);
  gimp_procedure_set_static_attribution (procedure,
                                         "Compatibility procedure. Please see 'gegl:cubism' for credits.",
                                         "Compatibility procedure. Please see 'gegl:cubism' for credits.",
                                         "2013");
  gimp_procedure_add_argument (procedure,
                               g_param_spec_enum ("run-mode",
                                                  "run mode",
                                                  "The run mode",
                                                  GIMP_TYPE_RUN_MODE,
                                                  GIMP_RUN_INTERACTIVE,
                                                  GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_image ("image",
                                                      "image",
                                                      "Input image (unused)",
                                                      FALSE,
                                                      GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("drawable",
                                                         "drawable",
                                                         "Input drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("tile-size",
                                                    "tile size",
                                                    "Average diameter of each tile (in pixels)",
                                                    0.0, 100.0, 0.0,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("tile-saturation",
                                                    "tile saturation",
                                                    "Expand tiles by this amount",
                                                    0.0, 10.0, 0.0,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("bg-color",
                                                 "bg color",
                                                 "Background color { BLACK (0), BG (1) }",
                                                 0, 1, 0,
                                                 GIMP_PARAM_READWRITE));
  gimp_pdb_register_procedure (pdb, procedure);
  g_object_unref (procedure);

  /*
   * gimp-plug-in-displace
   */
  procedure = gimp_procedure_new (plug_in_displace_invoker);
  gimp_object_set_static_name (GIMP_OBJECT (procedure),
                               "plug-in-displace");
  gimp_procedure_set_static_help (procedure,
                                  "Displace pixels as indicated by displacement maps",
                                  "Displaces the contents of the specified drawable by the amounts specified by 'amount-x' and 'amount-y' multiplied by the luminance of corresponding pixels in the 'displace-map' drawables.",
                                  NULL);
  gimp_procedure_set_static_attribution (procedure,
                                         "Compatibility procedure. Please see 'gegl:displace' for credits.",
                                         "Compatibility procedure. Please see 'gegl:displace' for credits.",
                                         "2015");
  gimp_procedure_add_argument (procedure,
                               g_param_spec_enum ("run-mode",
                                                  "run mode",
                                                  "The run mode",
                                                  GIMP_TYPE_RUN_MODE,
                                                  GIMP_RUN_INTERACTIVE,
                                                  GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_image ("image",
                                                      "image",
                                                      "Input image (unused)",
                                                      FALSE,
                                                      GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("drawable",
                                                         "drawable",
                                                         "Input drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("amount-x",
                                                    "amount x",
                                                    "Displace multiplier for x direction",
                                                    -500.0, 500.0, -500.0,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("amount-y",
                                                    "amount y",
                                                    "Displace multiplier for y direction",
                                                    -500.0, 500.0, -500.0,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_boolean ("do-x",
                                                     "do x",
                                                     "Displace in x direction ?",
                                                     FALSE,
                                                     GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_boolean ("do-y",
                                                     "do y",
                                                     "Displace in y direction ?",
                                                     FALSE,
                                                     GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("displace-map-x",
                                                         "displace map x",
                                                         "Displacement map for x direction",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("displace-map-y",
                                                         "displace map y",
                                                         "Displacement map for y direction",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("displace-type",
                                                 "displace type",
                                                 "Edge behavior { WRAP (1), SMEAR (2), BLACK (3) }",
                                                 1, 3, 1,
                                                 GIMP_PARAM_READWRITE));
  gimp_pdb_register_procedure (pdb, procedure);
  g_object_unref (procedure);

  /*
   * gimp-plug-in-edge
   */
  procedure = gimp_procedure_new (plug_in_edge_invoker);
  gimp_object_set_static_name (GIMP_OBJECT (procedure),
                               "plug-in-edge");
  gimp_procedure_set_static_help (procedure,
                                  "Several simple methods for detecting edges",
                                  "Perform edge detection on the contents of the specified drawable. AMOUNT is an arbitrary constant, WRAPMODE is like displace plug-in (useful for tileable image). EDGEMODE sets the kind of matrix transform applied to the pixels, SOBEL was the method used in older versions.",
                                  NULL);
  gimp_procedure_set_static_attribution (procedure,
                                         "Compatibility procedure. Please see 'gegl:edge' for credits.",
                                         "Compatibility procedure. Please see 'gegl:edge' for credits.",
                                         "2015");
  gimp_procedure_add_argument (procedure,
                               g_param_spec_enum ("run-mode",
                                                  "run mode",
                                                  "The run mode",
                                                  GIMP_TYPE_RUN_MODE,
                                                  GIMP_RUN_INTERACTIVE,
                                                  GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_image ("image",
                                                      "image",
                                                      "Input image (unused)",
                                                      FALSE,
                                                      GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("drawable",
                                                         "drawable",
                                                         "Input drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("amount",
                                                    "amount",
                                                    "Edge detection amount",
                                                    1.0, 10.0, 1.0,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("warpmode",
                                                 "warpmode",
                                                 "Edge detection behavior { NONE (0), WRAP (1), SMEAR (2), BLACK (3) }",
                                                 0, 3, 0,
                                                 GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("edgemode",
                                                 "edgemode",
                                                 "Edge detection algorithm { SOBEL (0), PREWITT (1), GRADIENT (2), ROBERTS (3), DIFFERENTIAL (4), LAPLACE (5) }",
                                                 0, 5, 0,
                                                 GIMP_PARAM_READWRITE));
  gimp_pdb_register_procedure (pdb, procedure);
  g_object_unref (procedure);

  /*
   * gimp-plug-in-emboss
   */
  procedure = gimp_procedure_new (plug_in_emboss_invoker);
  gimp_object_set_static_name (GIMP_OBJECT (procedure),
                               "plug-in-emboss");
  gimp_procedure_set_static_help (procedure,
                                  "Simulate an image created by embossing",
                                  "Emboss or Bumpmap the given drawable, specifying the angle and elevation for the light source.",
                                  NULL);
  gimp_procedure_set_static_attribution (procedure,
                                         "Compatibility procedure. Please see 'gegl:emboss' for credits.",
                                         "Compatibility procedure. Please see 'gegl:emboss' for credits.",
                                         "2019");
  gimp_procedure_add_argument (procedure,
                               g_param_spec_enum ("run-mode",
                                                  "run mode",
                                                  "The run mode",
                                                  GIMP_TYPE_RUN_MODE,
                                                  GIMP_RUN_INTERACTIVE,
                                                  GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_image ("image",
                                                      "image",
                                                      "Input image (unused)",
                                                      FALSE,
                                                      GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("drawable",
                                                         "drawable",
                                                         "Input drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("azimuth",
                                                    "azimuth",
                                                    "The Light Angle (degrees)",
                                                    0.0, 360.0, 0.0,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("elevation",
                                                    "elevation",
                                                    "The Elevation Angle (degrees)",
                                                    0.0, 180, 0.0,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("depth",
                                                 "depth",
                                                 "The Filter Width",
                                                 1, 99, 1,
                                                 GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_boolean ("emboss",
                                                     "emboss",
                                                     "Emboss (TRUE), Bumpmap (FALSE)",
                                                     FALSE,
                                                     GIMP_PARAM_READWRITE));
  gimp_pdb_register_procedure (pdb, procedure);
  g_object_unref (procedure);

  /*
   * gimp-plug-in-gauss
   */
  procedure = gimp_procedure_new (plug_in_gauss_invoker);
  gimp_object_set_static_name (GIMP_OBJECT (procedure),
                               "plug-in-gauss");
  gimp_procedure_set_static_help (procedure,
                                  "Simplest, most commonly used way of blurring",
                                  "Applies a gaussian blur to the drawable, with specified radius of affect. The standard deviation of the normal distribution used to modify pixel values is calculated based on the supplied radius. Horizontal and vertical blurring can be independently invoked by specifying only one to run. The 'method' parameter is ignored.",
                                  NULL);
  gimp_procedure_set_static_attribution (procedure,
                                         "Compatibility procedure. Please see 'gegl:gaussian-blur' for credits.",
                                         "Compatibility procedure. Please see 'gegl:gaussian-blur' for credits.",
                                         "2014");
  gimp_procedure_add_argument (procedure,
                               g_param_spec_enum ("run-mode",
                                                  "run mode",
                                                  "The run mode",
                                                  GIMP_TYPE_RUN_MODE,
                                                  GIMP_RUN_INTERACTIVE,
                                                  GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_image ("image",
                                                      "image",
                                                      "Input image (unused)",
                                                      FALSE,
                                                      GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("drawable",
                                                         "drawable",
                                                         "Input drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("horizontal",
                                                    "horizontal",
                                                    "Horizontal radius of gaussian blur (in pixels)",
                                                    0.0, 1500.0, 0.0,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("vertical",
                                                    "vertical",
                                                    "Vertical radius of gaussian blur (in pixels)",
                                                    0.0, 1500.0, 0.0,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("method",
                                                 "method",
                                                 "Blur method { AUTO (0), FIR (1), IIR (2) }",
                                                 0, 1, 0,
                                                 GIMP_PARAM_READWRITE));
  gimp_pdb_register_procedure (pdb, procedure);
  g_object_unref (procedure);

  /*
   * gimp-plug-in-maze
   */
  procedure = gimp_procedure_new (plug_in_maze_invoker);
  gimp_object_set_static_name (GIMP_OBJECT (procedure),
                               "plug-in-maze");
  gimp_procedure_set_static_help (procedure,
                                  "Draw a labyrinth",
                                  "Generates a maze using either the depth-first search method or Prim's algorithm. Can make tileable mazes too.",
                                  NULL);
  gimp_procedure_set_static_attribution (procedure,
                                         "Compatibility procedure. Please see 'gegl:maze' for credits.",
                                         "Compatibility procedure. Please see 'gegl:maze' for credits.",
                                         "2015");
  gimp_procedure_add_argument (procedure,
                               g_param_spec_enum ("run-mode",
                                                  "run mode",
                                                  "The run mode",
                                                  GIMP_TYPE_RUN_MODE,
                                                  GIMP_RUN_INTERACTIVE,
                                                  GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_image ("image",
                                                      "image",
                                                      "Input image (unused)",
                                                      FALSE,
                                                      GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("drawable",
                                                         "drawable",
                                                         "Input drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("width",
                                                 "width",
                                                 "Width of the passages",
                                                 1, 1024, 1,
                                                 GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("height",
                                                 "height",
                                                 "Height of the passages",
                                                 1, 1024, 1,
                                                 GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_boolean ("tileable",
                                                     "tileable",
                                                     "Tileable maze?",
                                                     FALSE,
                                                     GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("algorithm",
                                                 "algorithm",
                                                 "Generation algorithm (0 = DEPTH FIRST, 1 = PRIM'S ALGORITHM)",
                                                 0, 1, 0,
                                                 GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("seed",
                                                 "seed",
                                                 "Random Seed",
                                                 G_MININT32, G_MAXINT32, 0,
                                                 GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("multiple",
                                                 "multiple",
                                                 "Multiple (use 57)",
                                                 G_MININT32, G_MAXINT32, 0,
                                                 GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("offset",
                                                 "offset",
                                                 "Offset (use 1)",
                                                 G_MININT32, G_MAXINT32, 0,
                                                 GIMP_PARAM_READWRITE));
  gimp_pdb_register_procedure (pdb, procedure);
  g_object_unref (procedure);

  /*
   * gimp-plug-in-oilify
   */
  procedure = gimp_procedure_new (plug_in_oilify_invoker);
  gimp_object_set_static_name (GIMP_OBJECT (procedure),
                               "plug-in-oilify");
  gimp_procedure_set_static_help (procedure,
                                  "Smear colors to simulate an oil painting",
                                  "This function performs the well-known oil-paint effect on the specified drawable.",
                                  NULL);
  gimp_procedure_set_static_attribution (procedure,
                                         "Compatibility procedure. Please see 'gegl:oilify' for credits.",
                                         "Compatibility procedure. Please see 'gegl:oilify' for credits.",
                                         "2019");
  gimp_procedure_add_argument (procedure,
                               g_param_spec_enum ("run-mode",
                                                  "run mode",
                                                  "The run mode",
                                                  GIMP_TYPE_RUN_MODE,
                                                  GIMP_RUN_INTERACTIVE,
                                                  GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_image ("image",
                                                      "image",
                                                      "Input image (unused)",
                                                      FALSE,
                                                      GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("drawable",
                                                         "drawable",
                                                         "Input drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("mask-size",
                                                 "mask size",
                                                 "Oil paint mask size",
                                                 1, 200, 1,
                                                 GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("mode",
                                                 "mode",
                                                 "Algorithm { RGB (0), INTENSITY (1) }",
                                                 0, 1, 0,
                                                 GIMP_PARAM_READWRITE));
  gimp_pdb_register_procedure (pdb, procedure);
  g_object_unref (procedure);

  /*
   * gimp-plug-in-plasma
   */
  procedure = gimp_procedure_new (plug_in_plasma_invoker);
  gimp_object_set_static_name (GIMP_OBJECT (procedure),
                               "plug-in-plasma");
  gimp_procedure_set_static_help (procedure,
                                  "Create a random plasma texture",
                                  "This plug-in produces plasma fractal images.",
                                  NULL);
  gimp_procedure_set_static_attribution (procedure,
                                         "Compatibility procedure. Please see 'gegl:plasma' for credits.",
                                         "Compatibility procedure. Please see 'gegl:plasma' for credits.",
                                         "2013");
  gimp_procedure_add_argument (procedure,
                               g_param_spec_enum ("run-mode",
                                                  "run mode",
                                                  "The run mode",
                                                  GIMP_TYPE_RUN_MODE,
                                                  GIMP_RUN_INTERACTIVE,
                                                  GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_image ("image",
                                                      "image",
                                                      "Input image (unused)",
                                                      FALSE,
                                                      GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("drawable",
                                                         "drawable",
                                                         "Input drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("seed",
                                                 "seed",
                                                 "Random seed",
                                                 -1, G_MAXINT, -1,
                                                 GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("turbulence",
                                                    "turbulence",
                                                    "The value of the turbulence",
                                                    0.0, 7.0, 0.0,
                                                    GIMP_PARAM_READWRITE));
  gimp_pdb_register_procedure (pdb, procedure);
  g_object_unref (procedure);

  /*
   * gimp-plug-in-rotate
   */
  procedure = gimp_procedure_new (plug_in_rotate_invoker);
  gimp_object_set_static_name (GIMP_OBJECT (procedure),
                               "plug-in-rotate");
  gimp_procedure_set_static_help (procedure,
                                  "Rotates a layer or the whole image by 90, 180 or 270 degrees",
                                  "This plug-in does rotate the active layer or the whole image clockwise by multiples of 90 degrees. When the whole image is chosen, the image is resized if necessary.",
                                  NULL);
  gimp_procedure_set_static_attribution (procedure,
                                         "Sven Neumann <sven@gimp.org>",
                                         "Sven Neumann",
                                         "2014");
  gimp_procedure_add_argument (procedure,
                               g_param_spec_enum ("run-mode",
                                                  "run mode",
                                                  "The run mode",
                                                  GIMP_TYPE_RUN_MODE,
                                                  GIMP_RUN_INTERACTIVE,
                                                  GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_image ("image",
                                                      "image",
                                                      "Input image (unused)",
                                                      FALSE,
                                                      GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("drawable",
                                                         "drawable",
                                                         "Input drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("angle",
                                                 "angle",
                                                 "Angle { 90 (1), 180 (2), 270 (3) } degrees",
                                                 1, 3, 1,
                                                 GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_boolean ("everything",
                                                     "everything",
                                                     "Rotate the whole image",
                                                     FALSE,
                                                     GIMP_PARAM_READWRITE));
  gimp_pdb_register_procedure (pdb, procedure);
  g_object_unref (procedure);

  /*
   * gimp-plug-in-noisify
   */
  procedure = gimp_procedure_new (plug_in_noisify_invoker);
  gimp_object_set_static_name (GIMP_OBJECT (procedure),
                               "plug-in-noisify");
  gimp_procedure_set_static_help (procedure,
                                  "Adds random noise to image channels",
                                  "Add normally distributed random values to image channels. For color images each color channel may be treated together or independently.",
                                  NULL);
  gimp_procedure_set_static_attribution (procedure,
                                         "Compatibility procedure. Please see 'gegl:noise-rgb' for credits.",
                                         "Compatibility procedure. Please see 'gegl:noise-rgb' for credits.",
                                         "2013");
  gimp_procedure_add_argument (procedure,
                               g_param_spec_enum ("run-mode",
                                                  "run mode",
                                                  "The run mode",
                                                  GIMP_TYPE_RUN_MODE,
                                                  GIMP_RUN_INTERACTIVE,
                                                  GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_image ("image",
                                                      "image",
                                                      "Input image (unused)",
                                                      FALSE,
                                                      GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("drawable",
                                                         "drawable",
                                                         "Input drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_boolean ("independent",
                                                     "independent",
                                                     "Noise in channels independent",
                                                     FALSE,
                                                     GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("noise-1",
                                                    "noise 1",
                                                    "Noise in the first channel (red, gray)",
                                                    0.0, 1.0, 0.0,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("noise-2",
                                                    "noise 2",
                                                    "Noise in the second channel (green, gray_alpha)",
                                                    0.0, 1.0, 0.0,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("noise-3",
                                                    "noise 3",
                                                    "Noise in the third channel (blue)",
                                                    0.0, 1.0, 0.0,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("noise-4",
                                                    "noise 4",
                                                    "Noise in the fourth channel (alpha)",
                                                    0.0, 1.0, 0.0,
                                                    GIMP_PARAM_READWRITE));
  gimp_pdb_register_procedure (pdb, procedure);
  g_object_unref (procedure);

  /*
   * gimp-plug-in-solid-noise
   */
  procedure = gimp_procedure_new (plug_in_solid_noise_invoker);
  gimp_object_set_static_name (GIMP_OBJECT (procedure),
                               "plug-in-solid-noise");
  gimp_procedure_set_static_help (procedure,
                                  "Create a random cloud-like texture",
                                  "Generates 2D textures using Perlin's classic solid noise function.",
                                  NULL);
  gimp_procedure_set_static_attribution (procedure,
                                         "Compatibility procedure. Please see 'gegl:noise-solid' for credits.",
                                         "Compatibility procedure. Please see 'gegl:noise-solid' for credits.",
                                         "2014");
  gimp_procedure_add_argument (procedure,
                               g_param_spec_enum ("run-mode",
                                                  "run mode",
                                                  "The run mode",
                                                  GIMP_TYPE_RUN_MODE,
                                                  GIMP_RUN_INTERACTIVE,
                                                  GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_image ("image",
                                                      "image",
                                                      "Input image (unused)",
                                                      FALSE,
                                                      GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("drawable",
                                                         "drawable",
                                                         "Input drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_boolean ("tileable",
                                                     "tileable",
                                                     "Create a tileable output",
                                                     FALSE,
                                                     GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_boolean ("turbulent",
                                                     "turbulent",
                                                     "Make a turbulent noise",
                                                     FALSE,
                                                     GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("seed",
                                                 "seed",
                                                 "Random seed",
                                                 G_MININT32, G_MAXINT32, 0,
                                                 GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("detail",
                                                 "detail",
                                                 "Detail level",
                                                 0, 15, 0,
                                                 GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("xsize",
                                                    "xsize",
                                                    "Horizontal texture size",
                                                    0.1, 16.0, 0.1,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("ysize",
                                                    "ysize",
                                                    "Vertical texture size",
                                                    0.1, 16.0, 0.1,
                                                    GIMP_PARAM_READWRITE));
  gimp_pdb_register_procedure (pdb, procedure);
  g_object_unref (procedure);

  /*
   * gimp-plug-in-spread
   */
  procedure = gimp_procedure_new (plug_in_spread_invoker);
  gimp_object_set_static_name (GIMP_OBJECT (procedure),
                               "plug-in-spread");
  gimp_procedure_set_static_help (procedure,
                                  "Move pixels around randomly",
                                  "Spreads the pixels of the specified drawable. Pixels are randomly moved to another location whose distance varies from the original by the horizontal and vertical spread amounts.",
                                  NULL);
  gimp_procedure_set_static_attribution (procedure,
                                         "Compatibility procedure. Please see 'gegl:noise-spread' for credits.",
                                         "Compatibility procedure. Please see 'gegl:noise-spread' for credits.",
                                         "2013");
  gimp_procedure_add_argument (procedure,
                               g_param_spec_enum ("run-mode",
                                                  "run mode",
                                                  "The run mode",
                                                  GIMP_TYPE_RUN_MODE,
                                                  GIMP_RUN_INTERACTIVE,
                                                  GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_image ("image",
                                                      "image",
                                                      "Input image (unused)",
                                                      FALSE,
                                                      GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("drawable",
                                                         "drawable",
                                                         "Input drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("spread-amount-x",
                                                    "spread amount x",
                                                    "Horizontal spread amount",
                                                    0, 512, 0,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("spread-amount-y",
                                                    "spread amount y",
                                                    "Vertical spread amount",
                                                    0, 512, 0,
                                                    GIMP_PARAM_READWRITE));
  gimp_pdb_register_procedure (pdb, procedure);
  g_object_unref (procedure);

  /*
   * gimp-plug-in-threshold-alpha
   */
  procedure = gimp_procedure_new (plug_in_threshold_alpha_invoker);
  gimp_object_set_static_name (GIMP_OBJECT (procedure),
                               "plug-in-threshold-alpha");
  gimp_procedure_set_static_help (procedure,
                                  "Make transparency all-or-nothing",
                                  "Make transparency all-or-nothing.",
                                  NULL);
  gimp_procedure_set_static_attribution (procedure,
                                         "Spencer Kimball & Peter Mattis",
                                         "Spencer Kimball & Peter Mattis",
                                         "1997");
  gimp_procedure_add_argument (procedure,
                               g_param_spec_enum ("run-mode",
                                                  "run mode",
                                                  "The run mode",
                                                  GIMP_TYPE_RUN_MODE,
                                                  GIMP_RUN_INTERACTIVE,
                                                  GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_image ("image",
                                                      "image",
                                                      "Input image (unused)",
                                                      FALSE,
                                                      GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("drawable",
                                                         "drawable",
                                                         "Input drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_int ("threshold",
                                                 "threshold",
                                                 "Threshold",
                                                 0, 255, 0,
                                                 GIMP_PARAM_READWRITE));
  gimp_pdb_register_procedure (pdb, procedure);
  g_object_unref (procedure);

  /*
   * gimp-plug-in-waves
   */
  procedure = gimp_procedure_new (plug_in_waves_invoker);
  gimp_object_set_static_name (GIMP_OBJECT (procedure),
                               "plug-in-waves");
  gimp_procedure_set_static_help (procedure,
                                  "Distort the image with waves",
                                  "Distort the image with waves.",
                                  NULL);
  gimp_procedure_set_static_attribution (procedure,
                                         "Compatibility procedure. Please see 'gegl:waves' for credits.",
                                         "Compatibility procedure. Please see 'gegl:waves' for credits.",
                                         "2013");
  gimp_procedure_add_argument (procedure,
                               g_param_spec_enum ("run-mode",
                                                  "run mode",
                                                  "The run mode",
                                                  GIMP_TYPE_RUN_MODE,
                                                  GIMP_RUN_INTERACTIVE,
                                                  GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_image ("image",
                                                      "image",
                                                      "Input image (unused)",
                                                      FALSE,
                                                      GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               gimp_param_spec_drawable ("drawable",
                                                         "drawable",
                                                         "Input drawable",
                                                         FALSE,
                                                         GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("amplitude",
                                                    "amplitude",
                                                    "The Amplitude of the Waves",
                                                    0, 101, 0,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("phase",
                                                    "phase",
                                                    "The Phase of the Waves",
                                                    -360, 360, -360,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_double ("wavelength",
                                                    "wavelength",
                                                    "The Wavelength of the Waves",
                                                    0.1, 100, 0.1,
                                                    GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_boolean ("type",
                                                     "type",
                                                     "Type of waves: { 0 = smeared, 1 = black }",
                                                     FALSE,
                                                     GIMP_PARAM_READWRITE));
  gimp_procedure_add_argument (procedure,
                               g_param_spec_boolean ("reflective",
                                                     "reflective",
                                                     "Use Reflection (not implemented)",
                                                     FALSE,
                                                     GIMP_PARAM_READWRITE));
  gimp_pdb_register_procedure (pdb, procedure);
  g_object_unref (procedure);
}
