/*
 * Data for approximation of erf.
 *
 * Copyright (c) 2023, Arm Limited.
 * SPDX-License-Identifier: MIT OR Apache-2.0 WITH LLVM-exception
 */

#include "math_config.h"

/* Lookup table used in vector erf.
   For each possible rounded input r (multiples of 1/128), between
   r = 0.0 and r = 6.0 (769 values):
   - the first entry __erf_data.tab.erf contains the values of erf(r),
   - the second entry __erf_data.tab.scale contains the values of
   2/sqrt(pi)*exp(-r^2). Note that indices 0 and 1 are never hit by the
   algorithm, since lookup is performed only for x >= 1/64-1/512.  */
const struct sv_erf_data __sv_erf_data = {
  .erf = { 0x0.0000000000000p+0,
	   0x1.20dbf3deb1340p-7,
	   0x1.20d77083f17a0p-6,
	   0x1.b137e0cf584dcp-6,
	   0x1.20c5645dd2538p-5,
	   0x1.68e5d3bbc9526p-5,
	   0x1.b0fafef135745p-5,
	   0x1.f902a77bd3821p-5,
	   0x1.207d480e90658p-4,
	   0x1.44703e87e8593p-4,
	   0x1.68591a1e83b5dp-4,
	   0x1.8c36beb8a8d23p-4,
	   0x1.b0081148a873ap-4,
	   0x1.d3cbf7e70a4b3p-4,
	   0x1.f78159ec8bb50p-4,
	   0x1.0d939005f65e5p-3,
	   0x1.1f5e1a35c3b89p-3,
	   0x1.311fc15f56d14p-3,
	   0x1.42d7fc2f64959p-3,
	   0x1.548642321d7c6p-3,
	   0x1.662a0bdf7a89fp-3,
	   0x1.77c2d2a765f9ep-3,
	   0x1.895010fdbdbfdp-3,
	   0x1.9ad142662e14dp-3,
	   0x1.ac45e37fe2526p-3,
	   0x1.bdad72110a648p-3,
	   0x1.cf076d1233237p-3,
	   0x1.e05354b96ff36p-3,
	   0x1.f190aa85540e2p-3,
	   0x1.015f78a3dcf3dp-2,
	   0x1.09eed6982b948p-2,
	   0x1.127631eb8de32p-2,
	   0x1.1af54e232d609p-2,
	   0x1.236bef825d9a2p-2,
	   0x1.2bd9db0f7827fp-2,
	   0x1.343ed6989b7d9p-2,
	   0x1.3c9aa8b84bedap-2,
	   0x1.44ed18d9f6462p-2,
	   0x1.4d35ef3e5372ep-2,
	   0x1.5574f4ffac98ep-2,
	   0x1.5da9f415ff23fp-2,
	   0x1.65d4b75b00471p-2,
	   0x1.6df50a8dff772p-2,
	   0x1.760aba57a76bfp-2,
	   0x1.7e15944d9d3e4p-2,
	   0x1.861566f5fd3c0p-2,
	   0x1.8e0a01cab516bp-2,
	   0x1.95f3353cbb146p-2,
	   0x1.9dd0d2b721f39p-2,
	   0x1.a5a2aca209394p-2,
	   0x1.ad68966569a87p-2,
	   0x1.b522646bbda68p-2,
	   0x1.bccfec24855b8p-2,
	   0x1.c4710406a65fcp-2,
	   0x1.cc058392a6d2dp-2,
	   0x1.d38d4354c3bd0p-2,
	   0x1.db081ce6e2a48p-2,
	   0x1.e275eaf25e458p-2,
	   0x1.e9d68931ae650p-2,
	   0x1.f129d471eabb1p-2,
	   0x1.f86faa9428f9dp-2,
	   0x1.ffa7ea8eb5fd0p-2,
	   0x1.03693a371519cp-1,
	   0x1.06f794ab2cae7p-1,
	   0x1.0a7ef5c18edd2p-1,
	   0x1.0dff4f247f6c6p-1,
	   0x1.1178930ada115p-1,
	   0x1.14eab43841b55p-1,
	   0x1.1855a5fd3dd50p-1,
	   0x1.1bb95c3746199p-1,
	   0x1.1f15cb50bc4dep-1,
	   0x1.226ae840d4d70p-1,
	   0x1.25b8a88b6dd7fp-1,
	   0x1.28ff0240d52cdp-1,
	   0x1.2c3debfd7d6c1p-1,
	   0x1.2f755ce9a21f4p-1,
	   0x1.32a54cb8db67bp-1,
	   0x1.35cdb3a9a144dp-1,
	   0x1.38ee8a84beb71p-1,
	   0x1.3c07ca9cb4f9ep-1,
	   0x1.3f196dcd0f135p-1,
	   0x1.42236e79a5fa6p-1,
	   0x1.4525c78dd5966p-1,
	   0x1.4820747ba2dc2p-1,
	   0x1.4b13713ad3513p-1,
	   0x1.4dfeba47f63ccp-1,
	   0x1.50e24ca35fd2cp-1,
	   0x1.53be25d016a4fp-1,
	   0x1.569243d2b3a9bp-1,
	   0x1.595ea53035283p-1,
	   0x1.5c2348ecc4dc3p-1,
	   0x1.5ee02e8a71a53p-1,
	   0x1.61955607dd15dp-1,
	   0x1.6442bfdedd397p-1,
	   0x1.66e86d0312e82p-1,
	   0x1.69865ee075011p-1,
	   0x1.6c1c9759d0e5fp-1,
	   0x1.6eab18c74091bp-1,
	   0x1.7131e5f496a5ap-1,
	   0x1.73b1021fc0cb8p-1,
	   0x1.762870f720c6fp-1,
	   0x1.78983697dc96fp-1,
	   0x1.7b00578c26037p-1,
	   0x1.7d60d8c979f7bp-1,
	   0x1.7fb9bfaed8078p-1,
	   0x1.820b1202f27fbp-1,
	   0x1.8454d5f25760dp-1,
	   0x1.8697120d92a4ap-1,
	   0x1.88d1cd474a2e0p-1,
	   0x1.8b050ef253c37p-1,
	   0x1.8d30debfc572ep-1,
	   0x1.8f5544bd00c04p-1,
	   0x1.91724951b8fc6p-1,
	   0x1.9387f53df5238p-1,
	   0x1.959651980da31p-1,
	   0x1.979d67caa6631p-1,
	   0x1.999d4192a5715p-1,
	   0x1.9b95e8fd26abap-1,
	   0x1.9d8768656cc42p-1,
	   0x1.9f71ca72cffb6p-1,
	   0x1.a1551a16aaeafp-1,
	   0x1.a331628a45b92p-1,
	   0x1.a506af4cc00f4p-1,
	   0x1.a6d50c20fa293p-1,
	   0x1.a89c850b7d54dp-1,
	   0x1.aa5d265064366p-1,
	   0x1.ac16fc7143263p-1,
	   0x1.adca142b10f98p-1,
	   0x1.af767a741088bp-1,
	   0x1.b11c3c79bb424p-1,
	   0x1.b2bb679ead19cp-1,
	   0x1.b4540978921eep-1,
	   0x1.b5e62fce16095p-1,
	   0x1.b771e894d602ep-1,
	   0x1.b8f741ef54f83p-1,
	   0x1.ba764a2af2b78p-1,
	   0x1.bbef0fbde6221p-1,
	   0x1.bd61a1453ab44p-1,
	   0x1.bece0d82d1a5cp-1,
	   0x1.c034635b66e23p-1,
	   0x1.c194b1d49a184p-1,
	   0x1.c2ef0812fc1bdp-1,
	   0x1.c443755820d64p-1,
	   0x1.c5920900b5fd1p-1,
	   0x1.c6dad2829ec62p-1,
	   0x1.c81de16b14cefp-1,
	   0x1.c95b455cce69dp-1,
	   0x1.ca930e0e2a825p-1,
	   0x1.cbc54b476248dp-1,
	   0x1.ccf20ce0c0d27p-1,
	   0x1.ce1962c0e0d8bp-1,
	   0x1.cf3b5cdaf0c39p-1,
	   0x1.d0580b2cfd249p-1,
	   0x1.d16f7dbe41ca0p-1,
	   0x1.d281c49d818d0p-1,
	   0x1.d38eefdf64fddp-1,
	   0x1.d4970f9ce00d9p-1,
	   0x1.d59a33f19ed42p-1,
	   0x1.d6986cfa798e7p-1,
	   0x1.d791cad3eff01p-1,
	   0x1.d8865d98abe01p-1,
	   0x1.d97635600bb89p-1,
	   0x1.da61623cb41e0p-1,
	   0x1.db47f43b2980dp-1,
	   0x1.dc29fb60715afp-1,
	   0x1.dd0787a8bb39dp-1,
	   0x1.dde0a90611a0dp-1,
	   0x1.deb56f5f12d28p-1,
	   0x1.df85ea8db188ep-1,
	   0x1.e0522a5dfda73p-1,
	   0x1.e11a3e8cf4eb8p-1,
	   0x1.e1de36c75ba58p-1,
	   0x1.e29e22a89d766p-1,
	   0x1.e35a11b9b61cep-1,
	   0x1.e4121370224ccp-1,
	   0x1.e4c6372cd8927p-1,
	   0x1.e5768c3b4a3fcp-1,
	   0x1.e62321d06c5e0p-1,
	   0x1.e6cc0709c8a0dp-1,
	   0x1.e7714aec96534p-1,
	   0x1.e812fc64db369p-1,
	   0x1.e8b12a44944a8p-1,
	   0x1.e94be342e6743p-1,
	   0x1.e9e335fb56f87p-1,
	   0x1.ea7730ed0bbb9p-1,
	   0x1.eb07e27a133aap-1,
	   0x1.eb9558e6b42cep-1,
	   0x1.ec1fa258c4beap-1,
	   0x1.eca6ccd709544p-1,
	   0x1.ed2ae6489ac1ep-1,
	   0x1.edabfc7453e63p-1,
	   0x1.ee2a1d004692cp-1,
	   0x1.eea5557137ae0p-1,
	   0x1.ef1db32a2277cp-1,
	   0x1.ef93436bc2daap-1,
	   0x1.f006135426b26p-1,
	   0x1.f0762fde45ee6p-1,
	   0x1.f0e3a5e1a1788p-1,
	   0x1.f14e8211e8c55p-1,
	   0x1.f1b6d0fea5f4dp-1,
	   0x1.f21c9f12f0677p-1,
	   0x1.f27ff89525acfp-1,
	   0x1.f2e0e9a6a8b09p-1,
	   0x1.f33f7e43a706bp-1,
	   0x1.f39bc242e43e6p-1,
	   0x1.f3f5c1558b19ep-1,
	   0x1.f44d870704911p-1,
	   0x1.f4a31ebcd47dfp-1,
	   0x1.f4f693b67bd77p-1,
	   0x1.f547f10d60597p-1,
	   0x1.f59741b4b97cfp-1,
	   0x1.f5e4907982a07p-1,
	   0x1.f62fe80272419p-1,
	   0x1.f67952cff6282p-1,
	   0x1.f6c0db3c34641p-1,
	   0x1.f7068b7b10fd9p-1,
	   0x1.f74a6d9a38383p-1,
	   0x1.f78c8b812d498p-1,
	   0x1.f7cceef15d631p-1,
	   0x1.f80ba18636f07p-1,
	   0x1.f848acb544e95p-1,
	   0x1.f88419ce4e184p-1,
	   0x1.f8bdf1fb78370p-1,
	   0x1.f8f63e416ebffp-1,
	   0x1.f92d077f8d56dp-1,
	   0x1.f96256700da8ep-1,
	   0x1.f99633a838a57p-1,
	   0x1.f9c8a7989af0dp-1,
	   0x1.f9f9ba8d3c733p-1,
	   0x1.fa2974addae45p-1,
	   0x1.fa57ddfe27376p-1,
	   0x1.fa84fe5e05c8dp-1,
	   0x1.fab0dd89d1309p-1,
	   0x1.fadb831a9f9c3p-1,
	   0x1.fb04f6868a944p-1,
	   0x1.fb2d3f20f9101p-1,
	   0x1.fb54641aebbc9p-1,
	   0x1.fb7a6c834b5a2p-1,
	   0x1.fb9f5f4739170p-1,
	   0x1.fbc3433260ca5p-1,
	   0x1.fbe61eef4cf6ap-1,
	   0x1.fc07f907bc794p-1,
	   0x1.fc28d7e4f9cd0p-1,
	   0x1.fc48c1d033c7ap-1,
	   0x1.fc67bcf2d7b8fp-1,
	   0x1.fc85cf56ecd38p-1,
	   0x1.fca2fee770c79p-1,
	   0x1.fcbf5170b578bp-1,
	   0x1.fcdacca0bfb73p-1,
	   0x1.fcf57607a6e7cp-1,
	   0x1.fd0f5317f582fp-1,
	   0x1.fd2869270a56fp-1,
	   0x1.fd40bd6d7a785p-1,
	   0x1.fd58550773cb5p-1,
	   0x1.fd6f34f52013ap-1,
	   0x1.fd85621b0876dp-1,
	   0x1.fd9ae142795e3p-1,
	   0x1.fdafb719e6a69p-1,
	   0x1.fdc3e835500b3p-1,
	   0x1.fdd7790ea5bc0p-1,
	   0x1.fdea6e062d0c9p-1,
	   0x1.fdfccb62e52d3p-1,
	   0x1.fe0e9552ebdd6p-1,
	   0x1.fe1fcfebe2083p-1,
	   0x1.fe307f2b503d0p-1,
	   0x1.fe40a6f70af4bp-1,
	   0x1.fe504b1d9696cp-1,
	   0x1.fe5f6f568b301p-1,
	   0x1.fe6e1742f7cf6p-1,
	   0x1.fe7c466dc57a1p-1,
	   0x1.fe8a004c19ae6p-1,
	   0x1.fe97483db8670p-1,
	   0x1.fea4218d6594ap-1,
	   0x1.feb08f7146046p-1,
	   0x1.febc950b3fa75p-1,
	   0x1.fec835695932ep-1,
	   0x1.fed37386190fbp-1,
	   0x1.fede5248e38f4p-1,
	   0x1.fee8d486585eep-1,
	   0x1.fef2fd00af31ap-1,
	   0x1.fefcce6813974p-1,
	   0x1.ff064b5afffbep-1,
	   0x1.ff0f766697c76p-1,
	   0x1.ff18520700971p-1,
	   0x1.ff20e0a7ba8c2p-1,
	   0x1.ff2924a3f7a83p-1,
	   0x1.ff312046f2339p-1,
	   0x1.ff38d5cc4227fp-1,
	   0x1.ff404760319b4p-1,
	   0x1.ff47772010262p-1,
	   0x1.ff4e671a85425p-1,
	   0x1.ff55194fe19dfp-1,
	   0x1.ff5b8fb26f5f6p-1,
	   0x1.ff61cc26c1578p-1,
	   0x1.ff67d08401202p-1,
	   0x1.ff6d9e943c231p-1,
	   0x1.ff733814af88cp-1,
	   0x1.ff789eb6130c9p-1,
	   0x1.ff7dd41ce2b4dp-1,
	   0x1.ff82d9e1a76d8p-1,
	   0x1.ff87b1913e853p-1,
	   0x1.ff8c5cad200a5p-1,
	   0x1.ff90dcaba4096p-1,
	   0x1.ff9532f846ab0p-1,
	   0x1.ff9960f3eb327p-1,
	   0x1.ff9d67f51ddbap-1,
	   0x1.ffa14948549a7p-1,
	   0x1.ffa506302ebaep-1,
	   0x1.ffa89fe5b3625p-1,
	   0x1.ffac17988ef4bp-1,
	   0x1.ffaf6e6f4f5c0p-1,
	   0x1.ffb2a5879f35ep-1,
	   0x1.ffb5bdf67fe6fp-1,
	   0x1.ffb8b8c88295fp-1,
	   0x1.ffbb970200110p-1,
	   0x1.ffbe599f4f9d9p-1,
	   0x1.ffc10194fcb64p-1,
	   0x1.ffc38fcffbb7cp-1,
	   0x1.ffc60535dd7f5p-1,
	   0x1.ffc862a501fd7p-1,
	   0x1.ffcaa8f4c9beap-1,
	   0x1.ffccd8f5c66d1p-1,
	   0x1.ffcef371ea4d7p-1,
	   0x1.ffd0f92cb6ba7p-1,
	   0x1.ffd2eae369a07p-1,
	   0x1.ffd4c94d29fdbp-1,
	   0x1.ffd6951b33686p-1,
	   0x1.ffd84ef9009eep-1,
	   0x1.ffd9f78c7524ap-1,
	   0x1.ffdb8f7605ee7p-1,
	   0x1.ffdd1750e1220p-1,
	   0x1.ffde8fb314ebfp-1,
	   0x1.ffdff92db56e5p-1,
	   0x1.ffe1544d01ccbp-1,
	   0x1.ffe2a1988857cp-1,
	   0x1.ffe3e19349dc7p-1,
	   0x1.ffe514bbdc197p-1,
	   0x1.ffe63b8c8b5f7p-1,
	   0x1.ffe7567b7b5e1p-1,
	   0x1.ffe865fac722bp-1,
	   0x1.ffe96a78a04a9p-1,
	   0x1.ffea645f6d6dap-1,
	   0x1.ffeb5415e7c44p-1,
	   0x1.ffec39ff380b9p-1,
	   0x1.ffed167b12ac2p-1,
	   0x1.ffede9e5d3262p-1,
	   0x1.ffeeb49896c6dp-1,
	   0x1.ffef76e956a9fp-1,
	   0x1.fff0312b010b5p-1,
	   0x1.fff0e3ad91ec2p-1,
	   0x1.fff18ebe2b0e1p-1,
	   0x1.fff232a72b48ep-1,
	   0x1.fff2cfb0453d9p-1,
	   0x1.fff3661e9569dp-1,
	   0x1.fff3f634b79f9p-1,
	   0x1.fff48032dbe40p-1,
	   0x1.fff50456dab8cp-1,
	   0x1.fff582dc48d30p-1,
	   0x1.fff5fbfc8a439p-1,
	   0x1.fff66feee5129p-1,
	   0x1.fff6dee89352ep-1,
	   0x1.fff7491cd4af6p-1,
	   0x1.fff7aebcff755p-1,
	   0x1.fff80ff8911fdp-1,
	   0x1.fff86cfd3e657p-1,
	   0x1.fff8c5f702ccfp-1,
	   0x1.fff91b102fca8p-1,
	   0x1.fff96c717b695p-1,
	   0x1.fff9ba420e834p-1,
	   0x1.fffa04a7928b1p-1,
	   0x1.fffa4bc63ee9ap-1,
	   0x1.fffa8fc0e5f33p-1,
	   0x1.fffad0b901755p-1,
	   0x1.fffb0ecebee1bp-1,
	   0x1.fffb4a210b172p-1,
	   0x1.fffb82cd9dcbfp-1,
	   0x1.fffbb8f1049c6p-1,
	   0x1.fffbeca6adbe9p-1,
	   0x1.fffc1e08f25f5p-1,
	   0x1.fffc4d3120aa1p-1,
	   0x1.fffc7a37857d2p-1,
	   0x1.fffca53375ce3p-1,
	   0x1.fffcce3b57bffp-1,
	   0x1.fffcf564ab6b7p-1,
	   0x1.fffd1ac4135f9p-1,
	   0x1.fffd3e6d5cd87p-1,
	   0x1.fffd607387b07p-1,
	   0x1.fffd80e8ce0dap-1,
	   0x1.fffd9fdeabccep-1,
	   0x1.fffdbd65e5ad0p-1,
	   0x1.fffdd98e903b2p-1,
	   0x1.fffdf46816833p-1,
	   0x1.fffe0e0140857p-1,
	   0x1.fffe26683972ap-1,
	   0x1.fffe3daa95b18p-1,
	   0x1.fffe53d558ae9p-1,
	   0x1.fffe68f4fa777p-1,
	   0x1.fffe7d156d244p-1,
	   0x1.fffe904222101p-1,
	   0x1.fffea2860ee1ep-1,
	   0x1.fffeb3ebb267bp-1,
	   0x1.fffec47d19457p-1,
	   0x1.fffed443e2787p-1,
	   0x1.fffee34943b15p-1,
	   0x1.fffef1960d85dp-1,
	   0x1.fffeff32af7afp-1,
	   0x1.ffff0c273bea2p-1,
	   0x1.ffff187b6bc0ep-1,
	   0x1.ffff2436a21dcp-1,
	   0x1.ffff2f5fefcaap-1,
	   0x1.ffff39fe16963p-1,
	   0x1.ffff44178c8d2p-1,
	   0x1.ffff4db27f146p-1,
	   0x1.ffff56d4d5e5ep-1,
	   0x1.ffff5f8435efcp-1,
	   0x1.ffff67c604180p-1,
	   0x1.ffff6f9f67e55p-1,
	   0x1.ffff77154e0d6p-1,
	   0x1.ffff7e2c6aea2p-1,
	   0x1.ffff84e93cd75p-1,
	   0x1.ffff8b500e77cp-1,
	   0x1.ffff9164f8e46p-1,
	   0x1.ffff972be5c59p-1,
	   0x1.ffff9ca891572p-1,
	   0x1.ffffa1de8c582p-1,
	   0x1.ffffa6d13de73p-1,
	   0x1.ffffab83e54b8p-1,
	   0x1.ffffaff99bac4p-1,
	   0x1.ffffb43555b5fp-1,
	   0x1.ffffb839e52f3p-1,
	   0x1.ffffbc09fa7cdp-1,
	   0x1.ffffbfa82616bp-1,
	   0x1.ffffc316d9ed0p-1,
	   0x1.ffffc6586abf6p-1,
	   0x1.ffffc96f1165ep-1,
	   0x1.ffffcc5cec0c1p-1,
	   0x1.ffffcf23ff5fcp-1,
	   0x1.ffffd1c637b2bp-1,
	   0x1.ffffd4456a10dp-1,
	   0x1.ffffd6a3554a1p-1,
	   0x1.ffffd8e1a2f22p-1,
	   0x1.ffffdb01e8546p-1,
	   0x1.ffffdd05a75eap-1,
	   0x1.ffffdeee4f810p-1,
	   0x1.ffffe0bd3e852p-1,
	   0x1.ffffe273c15b7p-1,
	   0x1.ffffe41314e06p-1,
	   0x1.ffffe59c6698bp-1,
	   0x1.ffffe710d565ep-1,
	   0x1.ffffe8717232dp-1,
	   0x1.ffffe9bf4098cp-1,
	   0x1.ffffeafb377d5p-1,
	   0x1.ffffec2641a9ep-1,
	   0x1.ffffed413e5b7p-1,
	   0x1.ffffee4d01cd6p-1,
	   0x1.ffffef4a55bd4p-1,
	   0x1.fffff039f9e8fp-1,
	   0x1.fffff11ca4876p-1,
	   0x1.fffff1f302bc1p-1,
	   0x1.fffff2bdb904dp-1,
	   0x1.fffff37d63a36p-1,
	   0x1.fffff43297019p-1,
	   0x1.fffff4dde0118p-1,
	   0x1.fffff57fc4a95p-1,
	   0x1.fffff618c3da6p-1,
	   0x1.fffff6a956450p-1,
	   0x1.fffff731ee681p-1,
	   0x1.fffff7b2f8ed6p-1,
	   0x1.fffff82cdcf1bp-1,
	   0x1.fffff89ffc4aap-1,
	   0x1.fffff90cb3c81p-1,
	   0x1.fffff9735b73bp-1,
	   0x1.fffff9d446cccp-1,
	   0x1.fffffa2fc5015p-1,
	   0x1.fffffa8621251p-1,
	   0x1.fffffad7a2652p-1,
	   0x1.fffffb248c39dp-1,
	   0x1.fffffb6d1e95dp-1,
	   0x1.fffffbb196132p-1,
	   0x1.fffffbf22c1e2p-1,
	   0x1.fffffc2f171e3p-1,
	   0x1.fffffc688a9cfp-1,
	   0x1.fffffc9eb76acp-1,
	   0x1.fffffcd1cbc28p-1,
	   0x1.fffffd01f36afp-1,
	   0x1.fffffd2f57d68p-1,
	   0x1.fffffd5a2041fp-1,
	   0x1.fffffd8271d12p-1,
	   0x1.fffffda86faa9p-1,
	   0x1.fffffdcc3b117p-1,
	   0x1.fffffdedf37edp-1,
	   0x1.fffffe0db6b91p-1,
	   0x1.fffffe2ba0ea5p-1,
	   0x1.fffffe47ccb60p-1,
	   0x1.fffffe62534d4p-1,
	   0x1.fffffe7b4c81ep-1,
	   0x1.fffffe92ced93p-1,
	   0x1.fffffea8ef9cfp-1,
	   0x1.fffffebdc2ec6p-1,
	   0x1.fffffed15bcbap-1,
	   0x1.fffffee3cc32cp-1,
	   0x1.fffffef5251c2p-1,
	   0x1.ffffff0576917p-1,
	   0x1.ffffff14cfb92p-1,
	   0x1.ffffff233ee1dp-1,
	   0x1.ffffff30d18e8p-1,
	   0x1.ffffff3d9480fp-1,
	   0x1.ffffff4993c46p-1,
	   0x1.ffffff54dab72p-1,
	   0x1.ffffff5f74141p-1,
	   0x1.ffffff6969fb8p-1,
	   0x1.ffffff72c5fb6p-1,
	   0x1.ffffff7b91176p-1,
	   0x1.ffffff83d3d07p-1,
	   0x1.ffffff8b962bep-1,
	   0x1.ffffff92dfba2p-1,
	   0x1.ffffff99b79d2p-1,
	   0x1.ffffffa0248e8p-1,
	   0x1.ffffffa62ce54p-1,
	   0x1.ffffffabd69b4p-1,
	   0x1.ffffffb127525p-1,
	   0x1.ffffffb624592p-1,
	   0x1.ffffffbad2affp-1,
	   0x1.ffffffbf370cdp-1,
	   0x1.ffffffc355dfdp-1,
	   0x1.ffffffc733572p-1,
	   0x1.ffffffcad3626p-1,
	   0x1.ffffffce39b67p-1,
	   0x1.ffffffd169d0cp-1,
	   0x1.ffffffd466fa5p-1,
	   0x1.ffffffd7344aap-1,
	   0x1.ffffffd9d4aabp-1,
	   0x1.ffffffdc4ad7ap-1,
	   0x1.ffffffde9964ep-1,
	   0x1.ffffffe0c2bf0p-1,
	   0x1.ffffffe2c92dbp-1,
	   0x1.ffffffe4aed5ep-1,
	   0x1.ffffffe675bbdp-1,
	   0x1.ffffffe81fc4ep-1,
	   0x1.ffffffe9aeb97p-1,
	   0x1.ffffffeb24467p-1,
	   0x1.ffffffec81ff2p-1,
	   0x1.ffffffedc95e7p-1,
	   0x1.ffffffeefbc85p-1,
	   0x1.fffffff01a8b6p-1,
	   0x1.fffffff126e1ep-1,
	   0x1.fffffff221f30p-1,
	   0x1.fffffff30cd3fp-1,
	   0x1.fffffff3e8892p-1,
	   0x1.fffffff4b606fp-1,
	   0x1.fffffff57632dp-1,
	   0x1.fffffff629e44p-1,
	   0x1.fffffff6d1e56p-1,
	   0x1.fffffff76ef3fp-1,
	   0x1.fffffff801c1fp-1,
	   0x1.fffffff88af67p-1,
	   0x1.fffffff90b2e3p-1,
	   0x1.fffffff982fc1p-1,
	   0x1.fffffff9f2e9fp-1,
	   0x1.fffffffa5b790p-1,
	   0x1.fffffffabd229p-1,
	   0x1.fffffffb18582p-1,
	   0x1.fffffffb6d844p-1,
	   0x1.fffffffbbd0aap-1,
	   0x1.fffffffc0748fp-1,
	   0x1.fffffffc4c96cp-1,
	   0x1.fffffffc8d462p-1,
	   0x1.fffffffcc9a41p-1,
	   0x1.fffffffd01f89p-1,
	   0x1.fffffffd36871p-1,
	   0x1.fffffffd678edp-1,
	   0x1.fffffffd954aep-1,
	   0x1.fffffffdbff2ap-1,
	   0x1.fffffffde7ba0p-1,
	   0x1.fffffffe0cd16p-1,
	   0x1.fffffffe2f664p-1,
	   0x1.fffffffe4fa30p-1,
	   0x1.fffffffe6daf7p-1,
	   0x1.fffffffe89b0cp-1,
	   0x1.fffffffea3c9ap-1,
	   0x1.fffffffebc1a9p-1,
	   0x1.fffffffed2c21p-1,
	   0x1.fffffffee7dc8p-1,
	   0x1.fffffffefb847p-1,
	   0x1.ffffffff0dd2bp-1,
	   0x1.ffffffff1ede9p-1,
	   0x1.ffffffff2ebdap-1,
	   0x1.ffffffff3d843p-1,
	   0x1.ffffffff4b453p-1,
	   0x1.ffffffff58126p-1,
	   0x1.ffffffff63fc3p-1,
	   0x1.ffffffff6f121p-1,
	   0x1.ffffffff79626p-1,
	   0x1.ffffffff82fabp-1,
	   0x1.ffffffff8be77p-1,
	   0x1.ffffffff94346p-1,
	   0x1.ffffffff9bec8p-1,
	   0x1.ffffffffa319fp-1,
	   0x1.ffffffffa9c63p-1,
	   0x1.ffffffffaffa4p-1,
	   0x1.ffffffffb5be5p-1,
	   0x1.ffffffffbb1a2p-1,
	   0x1.ffffffffc014ep-1,
	   0x1.ffffffffc4b56p-1,
	   0x1.ffffffffc901cp-1,
	   0x1.ffffffffccfffp-1,
	   0x1.ffffffffd0b56p-1,
	   0x1.ffffffffd4271p-1,
	   0x1.ffffffffd759dp-1,
	   0x1.ffffffffda520p-1,
	   0x1.ffffffffdd13cp-1,
	   0x1.ffffffffdfa2dp-1,
	   0x1.ffffffffe202dp-1,
	   0x1.ffffffffe4371p-1,
	   0x1.ffffffffe642ap-1,
	   0x1.ffffffffe8286p-1,
	   0x1.ffffffffe9eb0p-1,
	   0x1.ffffffffeb8d0p-1,
	   0x1.ffffffffed10ap-1,
	   0x1.ffffffffee782p-1,
	   0x1.ffffffffefc57p-1,
	   0x1.fffffffff0fa7p-1,
	   0x1.fffffffff218fp-1,
	   0x1.fffffffff3227p-1,
	   0x1.fffffffff4188p-1,
	   0x1.fffffffff4fc9p-1,
	   0x1.fffffffff5cfdp-1,
	   0x1.fffffffff6939p-1,
	   0x1.fffffffff748ep-1,
	   0x1.fffffffff7f0dp-1,
	   0x1.fffffffff88c5p-1,
	   0x1.fffffffff91c6p-1,
	   0x1.fffffffff9a1bp-1,
	   0x1.fffffffffa1d2p-1,
	   0x1.fffffffffa8f6p-1,
	   0x1.fffffffffaf92p-1,
	   0x1.fffffffffb5b0p-1,
	   0x1.fffffffffbb58p-1,
	   0x1.fffffffffc095p-1,
	   0x1.fffffffffc56dp-1,
	   0x1.fffffffffc9e8p-1,
	   0x1.fffffffffce0dp-1,
	   0x1.fffffffffd1e1p-1,
	   0x1.fffffffffd56cp-1,
	   0x1.fffffffffd8b3p-1,
	   0x1.fffffffffdbbap-1,
	   0x1.fffffffffde86p-1,
	   0x1.fffffffffe11dp-1,
	   0x1.fffffffffe380p-1,
	   0x1.fffffffffe5b6p-1,
	   0x1.fffffffffe7c0p-1,
	   0x1.fffffffffe9a2p-1,
	   0x1.fffffffffeb60p-1,
	   0x1.fffffffffecfbp-1,
	   0x1.fffffffffee77p-1,
	   0x1.fffffffffefd6p-1,
	   0x1.ffffffffff11ap-1,
	   0x1.ffffffffff245p-1,
	   0x1.ffffffffff359p-1,
	   0x1.ffffffffff457p-1,
	   0x1.ffffffffff542p-1,
	   0x1.ffffffffff61bp-1,
	   0x1.ffffffffff6e3p-1,
	   0x1.ffffffffff79bp-1,
	   0x1.ffffffffff845p-1,
	   0x1.ffffffffff8e2p-1,
	   0x1.ffffffffff973p-1,
	   0x1.ffffffffff9f8p-1,
	   0x1.ffffffffffa73p-1,
	   0x1.ffffffffffae4p-1,
	   0x1.ffffffffffb4cp-1,
	   0x1.ffffffffffbadp-1,
	   0x1.ffffffffffc05p-1,
	   0x1.ffffffffffc57p-1,
	   0x1.ffffffffffca2p-1,
	   0x1.ffffffffffce7p-1,
	   0x1.ffffffffffd27p-1,
	   0x1.ffffffffffd62p-1,
	   0x1.ffffffffffd98p-1,
	   0x1.ffffffffffdcap-1,
	   0x1.ffffffffffdf8p-1,
	   0x1.ffffffffffe22p-1,
	   0x1.ffffffffffe49p-1,
	   0x1.ffffffffffe6cp-1,
	   0x1.ffffffffffe8dp-1,
	   0x1.ffffffffffeabp-1,
	   0x1.ffffffffffec7p-1,
	   0x1.ffffffffffee1p-1,
	   0x1.ffffffffffef8p-1,
	   0x1.fffffffffff0ep-1,
	   0x1.fffffffffff22p-1,
	   0x1.fffffffffff34p-1,
	   0x1.fffffffffff45p-1,
	   0x1.fffffffffff54p-1,
	   0x1.fffffffffff62p-1,
	   0x1.fffffffffff6fp-1,
	   0x1.fffffffffff7bp-1,
	   0x1.fffffffffff86p-1,
	   0x1.fffffffffff90p-1,
	   0x1.fffffffffff9ap-1,
	   0x1.fffffffffffa2p-1,
	   0x1.fffffffffffaap-1,
	   0x1.fffffffffffb1p-1,
	   0x1.fffffffffffb8p-1,
	   0x1.fffffffffffbep-1,
	   0x1.fffffffffffc3p-1,
	   0x1.fffffffffffc8p-1,
	   0x1.fffffffffffcdp-1,
	   0x1.fffffffffffd1p-1,
	   0x1.fffffffffffd5p-1,
	   0x1.fffffffffffd9p-1,
	   0x1.fffffffffffdcp-1,
	   0x1.fffffffffffdfp-1,
	   0x1.fffffffffffe2p-1,
	   0x1.fffffffffffe4p-1,
	   0x1.fffffffffffe7p-1,
	   0x1.fffffffffffe9p-1,
	   0x1.fffffffffffebp-1,
	   0x1.fffffffffffedp-1,
	   0x1.fffffffffffeep-1,
	   0x1.ffffffffffff0p-1,
	   0x1.ffffffffffff1p-1,
	   0x1.ffffffffffff3p-1,
	   0x1.ffffffffffff4p-1,
	   0x1.ffffffffffff5p-1,
	   0x1.ffffffffffff6p-1,
	   0x1.ffffffffffff7p-1,
	   0x1.ffffffffffff7p-1,
	   0x1.ffffffffffff8p-1,
	   0x1.ffffffffffff9p-1,
	   0x1.ffffffffffff9p-1,
	   0x1.ffffffffffffap-1,
	   0x1.ffffffffffffbp-1,
	   0x1.ffffffffffffbp-1,
	   0x1.ffffffffffffbp-1,
	   0x1.ffffffffffffcp-1,
	   0x1.ffffffffffffcp-1,
	   0x1.ffffffffffffdp-1,
	   0x1.ffffffffffffdp-1,
	   0x1.ffffffffffffdp-1,
	   0x1.ffffffffffffdp-1,
	   0x1.ffffffffffffep-1,
	   0x1.ffffffffffffep-1,
	   0x1.ffffffffffffep-1,
	   0x1.ffffffffffffep-1,
	   0x1.ffffffffffffep-1,
	   0x1.ffffffffffffep-1,
	   0x1.fffffffffffffp-1,
	   0x1.fffffffffffffp-1,
	   0x1.fffffffffffffp-1,
	   0x1.fffffffffffffp-1,
	   0x1.fffffffffffffp-1,
	   0x1.fffffffffffffp-1,
	   0x1.fffffffffffffp-1,
	   0x1.fffffffffffffp-1,
	   0x1.fffffffffffffp-1,
	   0x1.fffffffffffffp-1,
	   0x1.fffffffffffffp-1,
	   0x1.0000000000000p+0,
	   0x1.0000000000000p+0,
	   0x1.0000000000000p+0,
	   0x1.0000000000000p+0,
	   0x1.0000000000000p+0,
	   0x1.0000000000000p+0,
	   0x1.0000000000000p+0,
	   0x1.0000000000000p+0,
	   0x1.0000000000000p+0,
	   0x1.0000000000000p+0,
	   0x1.0000000000000p+0,
  },
  .scale = { 0x1.20dd750429b6dp+0,
	     0x1.20d8f1975c85dp+0,
	     0x1.20cb67bd452c7p+0,
	     0x1.20b4d8bac36c1p+0,
	     0x1.209546ad13ccfp+0,
	     0x1.206cb4897b148p+0,
	     0x1.203b261cd0052p+0,
	     0x1.2000a00ae3804p+0,
	     0x1.1fbd27cdc72d3p+0,
	     0x1.1f70c3b4f2cc7p+0,
	     0x1.1f1b7ae44867fp+0,
	     0x1.1ebd5552f795bp+0,
	     0x1.1e565bca400d4p+0,
	     0x1.1de697e413d28p+0,
	     0x1.1d6e14099944ap+0,
	     0x1.1cecdb718d61cp+0,
	     0x1.1c62fa1e869b6p+0,
	     0x1.1bd07cdd189acp+0,
	     0x1.1b357141d95d5p+0,
	     0x1.1a91e5a748165p+0,
	     0x1.19e5e92b964abp+0,
	     0x1.19318bae53a04p+0,
	     0x1.1874ddcdfce24p+0,
	     0x1.17aff0e56ec10p+0,
	     0x1.16e2d7093cd8cp+0,
	     0x1.160da304ed92fp+0,
	     0x1.153068581b781p+0,
	     0x1.144b3b337c90cp+0,
	     0x1.135e3075d076bp+0,
	     0x1.12695da8b5bdep+0,
	     0x1.116cd8fd67618p+0,
	     0x1.1068b94962e5ep+0,
	     0x1.0f5d1602f7e41p+0,
	     0x1.0e4a073dc1b91p+0,
	     0x1.0d2fa5a70c168p+0,
	     0x1.0c0e0a8223359p+0,
	     0x1.0ae54fa490722p+0,
	     0x1.09b58f724416bp+0,
	     0x1.087ee4d9ad247p+0,
	     0x1.07416b4fbfe7cp+0,
	     0x1.05fd3ecbec297p+0,
	     0x1.04b27bc403d30p+0,
	     0x1.03613f2812dafp+0,
	     0x1.0209a65e29545p+0,
	     0x1.00abcf3e187a9p+0,
	     0x1.fe8fb01a47307p-1,
	     0x1.fbbbbef34b4b2p-1,
	     0x1.f8dc092d58ff8p-1,
	     0x1.f5f0cdaf15313p-1,
	     0x1.f2fa4c16c0019p-1,
	     0x1.eff8c4b1375dbp-1,
	     0x1.ecec7870ebca7p-1,
	     0x1.e9d5a8e4c934ep-1,
	     0x1.e6b4982f158b9p-1,
	     0x1.e38988fc46e72p-1,
	     0x1.e054be79d3042p-1,
	     0x1.dd167c4cf9d2ap-1,
	     0x1.d9cf06898cdafp-1,
	     0x1.d67ea1a8b5368p-1,
	     0x1.d325927fb9d89p-1,
	     0x1.cfc41e36c7df9p-1,
	     0x1.cc5a8a3fbea40p-1,
	     0x1.c8e91c4d01368p-1,
	     0x1.c5701a484ef9dp-1,
	     0x1.c1efca49a5011p-1,
	     0x1.be68728e29d5dp-1,
	     0x1.bada596f25436p-1,
	     0x1.b745c55905bf8p-1,
	     0x1.b3aafcc27502ep-1,
	     0x1.b00a46237d5bep-1,
	     0x1.ac63e7ecc1411p-1,
	     0x1.a8b8287ec6a09p-1,
	     0x1.a5074e2157620p-1,
	     0x1.a1519efaf889ep-1,
	     0x1.9d97610879642p-1,
	     0x1.99d8da149c13fp-1,
	     0x1.96164fafd8de3p-1,
	     0x1.925007283d7aap-1,
	     0x1.8e86458169af8p-1,
	     0x1.8ab94f6caa71dp-1,
	     0x1.86e9694134b9ep-1,
	     0x1.8316d6f48133dp-1,
	     0x1.7f41dc12c9e89p-1,
	     0x1.7b6abbb7aaf19p-1,
	     0x1.7791b886e7403p-1,
	     0x1.73b714a552763p-1,
	     0x1.6fdb11b1e0c34p-1,
	     0x1.6bfdf0beddaf5p-1,
	     0x1.681ff24b4ab04p-1,
	     0x1.6441563c665d4p-1,
	     0x1.60625bd75d07bp-1,
	     0x1.5c8341bb23767p-1,
	     0x1.58a445da7c74cp-1,
	     0x1.54c5a57629db0p-1,
	     0x1.50e79d1749ac9p-1,
	     0x1.4d0a6889dfd9fp-1,
	     0x1.492e42d78d2c5p-1,
	     0x1.4553664273d24p-1,
	     0x1.417a0c4049fd0p-1,
	     0x1.3da26d759aef5p-1,
	     0x1.39ccc1b136d5ap-1,
	     0x1.35f93fe7d1b3dp-1,
	     0x1.32281e2fd1a92p-1,
	     0x1.2e5991bd4cbfcp-1,
	     0x1.2a8dcede3673bp-1,
	     0x1.26c508f6bd0ffp-1,
	     0x1.22ff727dd6f7bp-1,
	     0x1.1f3d3cf9ffe5ap-1,
	     0x1.1b7e98fe26217p-1,
	     0x1.17c3b626c7a11p-1,
	     0x1.140cc3173f007p-1,
	     0x1.1059ed7740313p-1,
	     0x1.0cab61f084b93p-1,
	     0x1.09014c2ca74dap-1,
	     0x1.055bd6d32e8d7p-1,
	     0x1.01bb2b87c6968p-1,
	     0x1.fc3ee5d1524b0p-2,
	     0x1.f511a91a67d2ap-2,
	     0x1.edeeee0959518p-2,
	     0x1.e6d6ffaa65a25p-2,
	     0x1.dfca26f5bbf88p-2,
	     0x1.d8c8aace11e63p-2,
	     0x1.d1d2cfff91594p-2,
	     0x1.cae8d93f1d7b6p-2,
	     0x1.c40b0729ed547p-2,
	     0x1.bd3998457afdap-2,
	     0x1.b674c8ffc6283p-2,
	     0x1.afbcd3afe8ab6p-2,
	     0x1.a911f096fbc26p-2,
	     0x1.a27455e14c93cp-2,
	     0x1.9be437a7de946p-2,
	     0x1.9561c7f23a47bp-2,
	     0x1.8eed36b886d93p-2,
	     0x1.8886b1e5ecfd1p-2,
	     0x1.822e655b417e6p-2,
	     0x1.7be47af1f5d89p-2,
	     0x1.75a91a7f4d2edp-2,
	     0x1.6f7c69d7d3ef8p-2,
	     0x1.695e8cd31867ep-2,
	     0x1.634fa54fa285fp-2,
	     0x1.5d4fd33729015p-2,
	     0x1.575f3483021c3p-2,
	     0x1.517de540ce2a3p-2,
	     0x1.4babff975a04cp-2,
	     0x1.45e99bcbb7915p-2,
	     0x1.4036d0468a7a2p-2,
	     0x1.3a93b1998736cp-2,
	     0x1.35005285227f1p-2,
	     0x1.2f7cc3fe6f423p-2,
	     0x1.2a09153529381p-2,
	     0x1.24a55399ea239p-2,
	     0x1.1f518ae487dc8p-2,
	     0x1.1a0dc51a9934dp-2,
	     0x1.14da0a961fd14p-2,
	     0x1.0fb6620c550afp-2,
	     0x1.0aa2d09497f2bp-2,
	     0x1.059f59af7a906p-2,
	     0x1.00abff4dec7a3p-2,
	     0x1.f79183b101c5bp-3,
	     0x1.edeb406d9c824p-3,
	     0x1.e4652fadcb6b2p-3,
	     0x1.daff4969c0b04p-3,
	     0x1.d1b982c501370p-3,
	     0x1.c893ce1dcbef7p-3,
	     0x1.bf8e1b1ca2279p-3,
	     0x1.b6a856c3ed54fp-3,
	     0x1.ade26b7fbed95p-3,
	     0x1.a53c4135a6526p-3,
	     0x1.9cb5bd549b111p-3,
	     0x1.944ec2e4f5630p-3,
	     0x1.8c07329874652p-3,
	     0x1.83deeada4d25ap-3,
	     0x1.7bd5c7df3fe9cp-3,
	     0x1.73eba3b5b07b7p-3,
	     0x1.6c205655be71fp-3,
	     0x1.6473b5b15a7a1p-3,
	     0x1.5ce595c455b0ap-3,
	     0x1.5575c8a468361p-3,
	     0x1.4e241e912c305p-3,
	     0x1.46f066040a832p-3,
	     0x1.3fda6bc016994p-3,
	     0x1.38e1fae1d6a9dp-3,
	     0x1.3206dceef5f87p-3,
	     0x1.2b48d9e5dea1cp-3,
	     0x1.24a7b84d38971p-3,
	     0x1.1e233d434b813p-3,
	     0x1.17bb2c8d41535p-3,
	     0x1.116f48a6476ccp-3,
	     0x1.0b3f52ce8c383p-3,
	     0x1.052b0b1a174eap-3,
	     0x1.fe6460fef4680p-4,
	     0x1.f2a901ccafb37p-4,
	     0x1.e723726b824a9p-4,
	     0x1.dbd32ac4c99b0p-4,
	     0x1.d0b7a0f921e7cp-4,
	     0x1.c5d0497c09e74p-4,
	     0x1.bb1c972f23e50p-4,
	     0x1.b09bfb7d11a83p-4,
	     0x1.a64de673e8837p-4,
	     0x1.9c31c6df3b1b8p-4,
	     0x1.92470a61b6965p-4,
	     0x1.888d1d8e510a3p-4,
	     0x1.7f036c0107294p-4,
	     0x1.75a96077274bap-4,
	     0x1.6c7e64e7281cbp-4,
	     0x1.6381e2980956bp-4,
	     0x1.5ab342383d177p-4,
	     0x1.5211ebf41880bp-4,
	     0x1.499d478bca735p-4,
	     0x1.4154bc68d75c3p-4,
	     0x1.3937b1b319259p-4,
	     0x1.31458e6542847p-4,
	     0x1.297db960e4f63p-4,
	     0x1.21df9981f8e53p-4,
	     0x1.1a6a95b1e786fp-4,
	     0x1.131e14fa1625dp-4,
	     0x1.0bf97e95f2a64p-4,
	     0x1.04fc3a0481321p-4,
	     0x1.fc4b5e32d6259p-5,
	     0x1.eeea8c1b1db93p-5,
	     0x1.e1d4cf1e2450ap-5,
	     0x1.d508f9a1ea64ep-5,
	     0x1.c885df3451a07p-5,
	     0x1.bc4a54a84e834p-5,
	     0x1.b055303221015p-5,
	     0x1.a4a549829587ep-5,
	     0x1.993979e14fffdp-5,
	     0x1.8e109c4622913p-5,
	     0x1.83298d717210ep-5,
	     0x1.78832c03aa2b1p-5,
	     0x1.6e1c5893c380bp-5,
	     0x1.63f3f5c4de13bp-5,
	     0x1.5a08e85af27e0p-5,
	     0x1.505a174e9c929p-5,
	     0x1.46e66be002240p-5,
	     0x1.3dacd1a8d8ccdp-5,
	     0x1.34ac36ad8dafep-5,
	     0x1.2be38b6d92415p-5,
	     0x1.2351c2f2d1449p-5,
	     0x1.1af5d2e04f3f6p-5,
	     0x1.12ceb37ff9bc3p-5,
	     0x1.0adb5fcfa8c75p-5,
	     0x1.031ad58d56279p-5,
	     0x1.f7182a851bca2p-6,
	     0x1.e85c449e377f2p-6,
	     0x1.da0005e5f28dfp-6,
	     0x1.cc0180af00a8bp-6,
	     0x1.be5ecd2fcb5f9p-6,
	     0x1.b1160991ff737p-6,
	     0x1.a4255a00b9f03p-6,
	     0x1.978ae8b55ce1bp-6,
	     0x1.8b44e6031383ep-6,
	     0x1.7f5188610ddc8p-6,
	     0x1.73af0c737bb45p-6,
	     0x1.685bb5134ef13p-6,
	     0x1.5d55cb54cd53ap-6,
	     0x1.529b9e8cf9a1ep-6,
	     0x1.482b8455dc491p-6,
	     0x1.3e03d891b37dep-6,
	     0x1.3422fd6d12e2bp-6,
	     0x1.2a875b5ffab56p-6,
	     0x1.212f612dee7fbp-6,
	     0x1.181983e5133ddp-6,
	     0x1.0f443edc5ce49p-6,
	     0x1.06ae13b0d3255p-6,
	     0x1.fcab1483ea7fcp-7,
	     0x1.ec72615a894c4p-7,
	     0x1.dcaf3691fc448p-7,
	     0x1.cd5ec93c12431p-7,
	     0x1.be7e5ac24963bp-7,
	     0x1.b00b38d6b3575p-7,
	     0x1.a202bd6372dcep-7,
	     0x1.94624e78e0fafp-7,
	     0x1.87275e3a6869dp-7,
	     0x1.7a4f6aca256cbp-7,
	     0x1.6dd7fe3358230p-7,
	     0x1.61beae53b72b7p-7,
	     0x1.56011cc3b036dp-7,
	     0x1.4a9cf6bda3f4cp-7,
	     0x1.3f8ff5042a88ep-7,
	     0x1.34d7dbc76d7e5p-7,
	     0x1.2a727a89a3f14p-7,
	     0x1.205dac02bd6b9p-7,
	     0x1.1697560347b25p-7,
	     0x1.0d1d69569b82dp-7,
	     0x1.03ede1a45bfeep-7,
	     0x1.f60d8aa2a88f2p-8,
	     0x1.e4cc4abf7d065p-8,
	     0x1.d4143a9dfe965p-8,
	     0x1.c3e1a5f5c077cp-8,
	     0x1.b430ecf4a83a8p-8,
	     0x1.a4fe83fb9db25p-8,
	     0x1.9646f35a76623p-8,
	     0x1.8806d70b2fc36p-8,
	     0x1.7a3ade6c8b3e4p-8,
	     0x1.6cdfcbfc1e263p-8,
	     0x1.5ff2750fe7820p-8,
	     0x1.536fc18f7ce5cp-8,
	     0x1.4754abacdf1dcp-8,
	     0x1.3b9e3f9d06e3fp-8,
	     0x1.30499b503957fp-8,
	     0x1.2553ee2a336bfp-8,
	     0x1.1aba78ba3af89p-8,
	     0x1.107a8c7323a6ep-8,
	     0x1.06918b6355624p-8,
	     0x1.f9f9cfd9c3035p-9,
	     0x1.e77448fb66bb9p-9,
	     0x1.d58da68fd1170p-9,
	     0x1.c4412bf4b8f0bp-9,
	     0x1.b38a3af2e55b4p-9,
	     0x1.a3645330550ffp-9,
	     0x1.93cb11a30d765p-9,
	     0x1.84ba3004a50d0p-9,
	     0x1.762d84469c18fp-9,
	     0x1.6821000795a03p-9,
	     0x1.5a90b00981d93p-9,
	     0x1.4d78bba8ca5fdp-9,
	     0x1.40d564548fad7p-9,
	     0x1.34a305080681fp-9,
	     0x1.28de11c5031ebp-9,
	     0x1.1d83170fbf6fbp-9,
	     0x1.128eb96be8798p-9,
	     0x1.07fdb4dafea5fp-9,
	     0x1.fb99b8b8279e1p-10,
	     0x1.e7f232d9e2630p-10,
	     0x1.d4fed7195d7e8p-10,
	     0x1.c2b9cf7f893bfp-10,
	     0x1.b11d702b3deb1p-10,
	     0x1.a024365f771bdp-10,
	     0x1.8fc8c794b03b5p-10,
	     0x1.8005f08d6f1efp-10,
	     0x1.70d6a46e07ddap-10,
	     0x1.6235fbd7a4345p-10,
	     0x1.541f340697987p-10,
	     0x1.468dadf4080abp-10,
	     0x1.397ced7af2b15p-10,
	     0x1.2ce898809244ep-10,
	     0x1.20cc76202c5fap-10,
	     0x1.15246dda49d47p-10,
	     0x1.09ec86c75d497p-10,
	     0x1.fe41cd9bb4eeep-11,
	     0x1.e97ba3b77f306p-11,
	     0x1.d57f524723822p-11,
	     0x1.c245d4b998479p-11,
	     0x1.afc85e0f82e12p-11,
	     0x1.9e005769dbc1dp-11,
	     0x1.8ce75e9f6f8a0p-11,
	     0x1.7c7744d9378f7p-11,
	     0x1.6caa0d3582fe9p-11,
	     0x1.5d79eb71e893bp-11,
	     0x1.4ee1429bf7cc0p-11,
	     0x1.40daa3c89f5b6p-11,
	     0x1.3360ccd23db3ap-11,
	     0x1.266ea71d4f71ap-11,
	     0x1.19ff4663ae9dfp-11,
	     0x1.0e0de78654d1ep-11,
	     0x1.0295ef6591848p-11,
	     0x1.ef25d37f49fe1p-12,
	     0x1.da01102b5f851p-12,
	     0x1.c5b5412dcafadp-12,
	     0x1.b23a5a23e4210p-12,
	     0x1.9f8893d8fd1c1p-12,
	     0x1.8d986a4187285p-12,
	     0x1.7c629a822bc9ep-12,
	     0x1.6be02102b3520p-12,
	     0x1.5c0a378c90bcap-12,
	     0x1.4cda5374ea275p-12,
	     0x1.3e4a23d1f4702p-12,
	     0x1.30538fbb77ecdp-12,
	     0x1.22f0b496539bdp-12,
	     0x1.161be46ad3b50p-12,
	     0x1.09cfa445b00ffp-12,
	     0x1.fc0d55470cf51p-13,
	     0x1.e577bbcd49935p-13,
	     0x1.cfd4a5adec5bfp-13,
	     0x1.bb1a9657ce465p-13,
	     0x1.a740684026555p-13,
	     0x1.943d4a1d1ed39p-13,
	     0x1.8208bc334a6a5p-13,
	     0x1.709a8db59f25cp-13,
	     0x1.5feada379d8b7p-13,
	     0x1.4ff207314a102p-13,
	     0x1.40a8c1949f75ep-13,
	     0x1.3207fb7420eb9p-13,
	     0x1.2408e9ba3327fp-13,
	     0x1.16a501f0e42cap-13,
	     0x1.09d5f819c9e29p-13,
	     0x1.fb2b792b40a22p-14,
	     0x1.e3bcf436a1a95p-14,
	     0x1.cd55277c18d05p-14,
	     0x1.b7e94604479dcp-14,
	     0x1.a36eec00926ddp-14,
	     0x1.8fdc1b2dcf7b9p-14,
	     0x1.7d2737527c3f9p-14,
	     0x1.6b4702d7d5849p-14,
	     0x1.5a329b7d30748p-14,
	     0x1.49e17724f4d41p-14,
	     0x1.3a4b60ba9aa4dp-14,
	     0x1.2b6875310f785p-14,
	     0x1.1d312098e9dbap-14,
	     0x1.0f9e1b4dd36dfp-14,
	     0x1.02a8673a94691p-14,
	     0x1.ec929a665b449p-15,
	     0x1.d4f4b4c8e09edp-15,
	     0x1.be6abbb10a5aap-15,
	     0x1.a8e8cc1fadef6p-15,
	     0x1.94637d5bacfdbp-15,
	     0x1.80cfdc72220cfp-15,
	     0x1.6e2367dc27f95p-15,
	     0x1.5c540b4936fd2p-15,
	     0x1.4b581b8d170fcp-15,
	     0x1.3b2652b06c2b2p-15,
	     0x1.2bb5cc22e5db6p-15,
	     0x1.1cfe010e2052dp-15,
	     0x1.0ef6c4c84a0fep-15,
	     0x1.01984165a5f36p-15,
	     0x1.e9b5e8d00ce76p-16,
	     0x1.d16f5716c6c1ap-16,
	     0x1.ba4f035d60e02p-16,
	     0x1.a447b7b03f045p-16,
	     0x1.8f4ccca7fc90dp-16,
	     0x1.7b5223dac7336p-16,
	     0x1.684c227fcacefp-16,
	     0x1.562fac4329b48p-16,
	     0x1.44f21e49054f2p-16,
	     0x1.34894a5e24657p-16,
	     0x1.24eb7254ccf83p-16,
	     0x1.160f438c70913p-16,
	     0x1.07ebd2a2d2844p-16,
	     0x1.f4f12e9ab070ap-17,
	     0x1.db5ad0b27805cp-17,
	     0x1.c304efa2c6f4ep-17,
	     0x1.abe09e9144b5ep-17,
	     0x1.95df988e76644p-17,
	     0x1.80f439b4ee04bp-17,
	     0x1.6d11788a69c64p-17,
	     0x1.5a2adfa0b4bc4p-17,
	     0x1.4834877429b8fp-17,
	     0x1.37231085c7d9ap-17,
	     0x1.26eb9daed6f7ep-17,
	     0x1.1783ceac28910p-17,
	     0x1.08e1badf0fcedp-17,
	     0x1.f5f7d88472604p-18,
	     0x1.db92b5212fb8dp-18,
	     0x1.c282cd3957edap-18,
	     0x1.aab7abace48dcp-18,
	     0x1.94219bfcb4928p-18,
	     0x1.7eb1a2075864dp-18,
	     0x1.6a597219a93d9p-18,
	     0x1.570b69502f313p-18,
	     0x1.44ba864670882p-18,
	     0x1.335a62115bce2p-18,
	     0x1.22df298214423p-18,
	     0x1.133d96ae7e0ddp-18,
	     0x1.046aeabcfcdecp-18,
	     0x1.ecb9cfe1d8642p-19,
	     0x1.d21397ead99cbp-19,
	     0x1.b8d094c86d374p-19,
	     0x1.a0df0f0c626dcp-19,
	     0x1.8a2e269750a39p-19,
	     0x1.74adc8f4064d3p-19,
	     0x1.604ea819f007cp-19,
	     0x1.4d0231928c6f9p-19,
	     0x1.3aba85fe22e1fp-19,
	     0x1.296a70f414053p-19,
	     0x1.1905613b3abf2p-19,
	     0x1.097f6156f32c5p-19,
	     0x1.f59a20caf6695p-20,
	     0x1.d9c73698fb1dcp-20,
	     0x1.bf716c6168baep-20,
	     0x1.a6852c6b58392p-20,
	     0x1.8eefd70594a88p-20,
	     0x1.789fb715aae95p-20,
	     0x1.6383f726a8e04p-20,
	     0x1.4f8c96f26a26ap-20,
	     0x1.3caa61607f920p-20,
	     0x1.2acee2f5ecdb8p-20,
	     0x1.19ec60b1242edp-20,
	     0x1.09f5cf4dd2877p-20,
	     0x1.f5bd95d8730d8p-21,
	     0x1.d9371e2ff7c35p-21,
	     0x1.be41de54d155ap-21,
	     0x1.a4c89e08ef4f3p-21,
	     0x1.8cb738399b12cp-21,
	     0x1.75fa8dbc84becp-21,
	     0x1.608078a70dcbcp-21,
	     0x1.4c37c0394d094p-21,
	     0x1.39100d5687bfep-21,
	     0x1.26f9df8519bd6p-21,
	     0x1.15e6827001f18p-21,
	     0x1.05c803e4831c1p-21,
	     0x1.ed22548cffd35p-22,
	     0x1.d06ad6ecdf971p-22,
	     0x1.b551c847fbc96p-22,
	     0x1.9bc09f112b494p-22,
	     0x1.83a1ff0aa239dp-22,
	     0x1.6ce1aa3fd7bddp-22,
	     0x1.576c72b514859p-22,
	     0x1.43302cc4a0da8p-22,
	     0x1.301ba221dc9bbp-22,
	     0x1.1e1e857adc568p-22,
	     0x1.0d2966b1746f7p-22,
	     0x1.fa5b4f49cc6b2p-23,
	     0x1.dc3ae30b55c16p-23,
	     0x1.bfd7555a3bd68p-23,
	     0x1.a517d9e61628ap-23,
	     0x1.8be4f8f6c951fp-23,
	     0x1.74287ded49339p-23,
	     0x1.5dcd669f2cd34p-23,
	     0x1.48bfd38302870p-23,
	     0x1.34ecf8a3c124ap-23,
	     0x1.22430f521cbcfp-23,
	     0x1.10b1488aeb235p-23,
	     0x1.0027c00a263a6p-23,
	     0x1.e12ee004efc37p-24,
	     0x1.c3e44ae32b16bp-24,
	     0x1.a854ea14102a8p-24,
	     0x1.8e6761569f45dp-24,
	     0x1.7603bac345f65p-24,
	     0x1.5f1353cdad001p-24,
	     0x1.4980cb3c80949p-24,
	     0x1.3537f00b6ad4dp-24,
	     0x1.2225b12bffc68p-24,
	     0x1.10380e1adb7e9p-24,
	     0x1.febc107d5efaap-25,
	     0x1.df0f2a0ee6946p-25,
	     0x1.c14b2188bcee4p-25,
	     0x1.a553644f7f07dp-25,
	     0x1.8b0cfce0579dfp-25,
	     0x1.725e7c5dd20f7p-25,
	     0x1.5b2fe547a1340p-25,
	     0x1.456a974e92e93p-25,
	     0x1.30f93c3699078p-25,
	     0x1.1dc7b5b978cf8p-25,
	     0x1.0bc30c5d52f15p-25,
	     0x1.f5b2be65a0c7fp-26,
	     0x1.d5f3a8dea7357p-26,
	     0x1.b82915b03515bp-26,
	     0x1.9c3517e789488p-26,
	     0x1.81fb7df06136ep-26,
	     0x1.6961b8d641d06p-26,
	     0x1.524ec4d916caep-26,
	     0x1.3cab1343d18d1p-26,
	     0x1.2860757487a01p-26,
	     0x1.155a09065d4f7p-26,
	     0x1.0384250e4c9fcp-26,
	     0x1.e59890b926c78p-27,
	     0x1.c642116a8a9e3p-27,
	     0x1.a8e405e651ab6p-27,
	     0x1.8d5f98114f872p-27,
	     0x1.7397c5a66e307p-27,
	     0x1.5b71456c5a4c4p-27,
	     0x1.44d26de513197p-27,
	     0x1.2fa31d6371537p-27,
	     0x1.1bcca373b7b43p-27,
	     0x1.0939ab853339fp-27,
	     0x1.efac5187b2863p-28,
	     0x1.cf1e86235d0e6p-28,
	     0x1.b0a68a2128babp-28,
	     0x1.9423165bc4444p-28,
	     0x1.7974e743dea3cp-28,
	     0x1.607e9eacd1050p-28,
	     0x1.4924a74dec728p-28,
	     0x1.334d19e0c2160p-28,
	     0x1.1edfa3c5f5ccap-28,
	     0x1.0bc56f1b54701p-28,
	     0x1.f3d2185e047d9p-29,
	     0x1.d26cb87945e87p-29,
	     0x1.b334fac4b9f99p-29,
	     0x1.96076f7918d1cp-29,
	     0x1.7ac2d72fc2c63p-29,
	     0x1.614801550319ep-29,
	     0x1.4979ac8b28926p-29,
	     0x1.333c68e2d0548p-29,
	     0x1.1e767bce37dd7p-29,
	     0x1.0b0fc5b6d05a0p-29,
	     0x1.f1e3523b41d7dp-30,
	     0x1.d00de6608effep-30,
	     0x1.b0778b7b3301ap-30,
	     0x1.92fb04ec0f6cfp-30,
	     0x1.77756ec9f78fap-30,
	     0x1.5dc61922d5a06p-30,
	     0x1.45ce65699ff6dp-30,
	     0x1.2f71a5f159970p-30,
	     0x1.1a94ff571654fp-30,
	     0x1.071f4bbea09ecp-30,
	     0x1.e9f1ff8ddd774p-31,
	     0x1.c818223a202c7p-31,
	     0x1.a887bd2b4404dp-31,
	     0x1.8b1a336c5eb6bp-31,
	     0x1.6fab63324088ap-31,
	     0x1.56197e30205bap-31,
	     0x1.3e44e45301b92p-31,
	     0x1.281000bfe4c3fp-31,
	     0x1.135f28f2d50b4p-31,
	     0x1.00187dded5975p-31,
	     0x1.dc479de0ef001p-32,
	     0x1.bad4fdad3caa1p-32,
	     0x1.9baed3ed27ab8p-32,
	     0x1.7ead9ce4285bbp-32,
	     0x1.63ac6b4edc88ep-32,
	     0x1.4a88be2a6390cp-32,
	     0x1.332259185f1a0p-32,
	     0x1.1d5b1f3793044p-32,
	     0x1.0916f04b6e18bp-32,
	     0x1.ec77101de6926p-33,
	     0x1.c960bf23153e0p-33,
	     0x1.a8bd20fc65ef7p-33,
	     0x1.8a61745ec7d1dp-33,
	     0x1.6e25d0e756261p-33,
	     0x1.53e4f7d1666cbp-33,
	     0x1.3b7c27a7ddb0ep-33,
	     0x1.24caf2c32af14p-33,
	     0x1.0fb3186804d0fp-33,
	     0x1.f830c0bb41fd7p-34,
	     0x1.d3c0f1a91c846p-34,
	     0x1.b1e5acf351d87p-34,
	     0x1.92712d259ce66p-34,
	     0x1.7538c60a04476p-34,
	     0x1.5a14b04b47879p-34,
	     0x1.40dfd87456f4cp-34,
	     0x1.2977b1172b9d5p-34,
	     0x1.13bc07e891491p-34,
	     0x1.ff1dbb4300811p-35,
	     0x1.d9a880f306bd8p-35,
	     0x1.b6e45220b55e0p-35,
	     0x1.96a0b33f2c4dap-35,
	     0x1.78b07e9e924acp-35,
	     0x1.5ce9ab1670dd2p-35,
	     0x1.4325167006bb0p-35,
	     0x1.2b3e53538ff3fp-35,
	     0x1.15137a7f44864p-35,
	     0x1.0084ff125639dp-35,
	     0x1.daeb0b7311ec7p-36,
	     0x1.b7937d1c40c52p-36,
	     0x1.96d082f59ab06p-36,
	     0x1.7872d9fa10aadp-36,
	     0x1.5c4e8e37bc7d0p-36,
	     0x1.423ac0df49a40p-36,
	     0x1.2a117230ad284p-36,
	     0x1.13af4f04f9998p-36,
	     0x1.fde703724e560p-37,
	     0x1.d77f0c82e7641p-37,
	     0x1.b3ee02611d7ddp-37,
	     0x1.92ff33023d5bdp-37,
	     0x1.7481a9e69f53fp-37,
	     0x1.5847eda620959p-37,
	     0x1.3e27c1fcc74bdp-37,
	     0x1.25f9ee0b923dcp-37,
	     0x1.0f9a0686531ffp-37,
	     0x1.f5cc7718082afp-38,
	     0x1.cf7e53d6a2ca5p-38,
	     0x1.ac0f5f3229372p-38,
	     0x1.8b498644847eap-38,
	     0x1.6cfa9bcca59dcp-38,
	     0x1.50f411d4fd2cdp-38,
	     0x1.370ab8327af5ep-38,
	     0x1.1f167f88c6b6ep-38,
	     0x1.08f24085d4597p-38,
	     0x1.e8f70e181d619p-39,
	     0x1.c324c20e337dcp-39,
	     0x1.a03261574b54ep-39,
	     0x1.7fe903cdf5855p-39,
	     0x1.6215c58da3450p-39,
	     0x1.46897d4b69fc6p-39,
	     0x1.2d1877d731b7bp-39,
	     0x1.159a386b11517p-39,
	     0x1.ffd27ae9393cep-40,
	     0x1.d7c593130dd0bp-40,
	     0x1.b2cd607c79bcfp-40,
	     0x1.90ae4d3405651p-40,
	     0x1.71312dd1759e2p-40,
	     0x1.5422ef5d8949dp-40,
	     0x1.39544b0ecc957p-40,
	     0x1.20997f73e73ddp-40,
	     0x1.09ca0eaacd277p-40,
	     0x1.e9810295890ecp-41,
	     0x1.c2b45b5aa4a1dp-41,
	     0x1.9eee068fa7596p-41,
	     0x1.7df2b399c10a8p-41,
	     0x1.5f8b87a31bd85p-41,
	     0x1.4385c96e9a2d9p-41,
	     0x1.29b2933ef4cbcp-41,
	     0x1.11e68a6378f8ap-41,
	     0x1.f7f338086a86bp-42,
	     0x1.cf8d7d9ce040ap-42,
	     0x1.aa577251ae484p-42,
	     0x1.8811d739efb5ep-42,
	     0x1.68823e52970bep-42,
	     0x1.4b72ae68e8b4cp-42,
	     0x1.30b14dbe876bcp-42,
	     0x1.181012ef86610p-42,
	     0x1.01647ba798744p-42,
	     0x1.d90e917701675p-43,
	     0x1.b2a87e86d0c8ap-43,
	     0x1.8f53dcb377293p-43,
	     0x1.6ed2f2515e933p-43,
	     0x1.50ecc9ed47f19p-43,
	     0x1.356cd5ce7799ep-43,
	     0x1.1c229a587ab78p-43,
	     0x1.04e15ecc7f3f6p-43,
	     0x1.deffc7e6a6017p-44,
	     0x1.b7b040832f310p-44,
	     0x1.938e021f36d76p-44,
	     0x1.7258610b3b233p-44,
	     0x1.53d3bfc82a909p-44,
	     0x1.37c92babdc2fdp-44,
	     0x1.1e06010120f6ap-44,
	     0x1.065b9616170d4p-44,
	     0x1.e13dd96b3753ap-45,
	     0x1.b950d32467392p-45,
	     0x1.94a72263259a5p-45,
	     0x1.72fd93e036cdcp-45,
	     0x1.54164576929abp-45,
	     0x1.37b83c521fe96p-45,
	     0x1.1daf033182e96p-45,
	     0x1.05ca50205d26ap-45,
	     0x1.dfbb6235639fap-46,
	     0x1.b7807e294781fp-46,
	     0x1.9298add70a734p-46,
	     0x1.70beaf9c7ffb6p-46,
	     0x1.51b2cd6709222p-46,
	     0x1.353a6cf7f7fffp-46,
	     0x1.1b1fa8cbe84a7p-46,
	     0x1.0330f0fd69921p-46,
	     0x1.da81670f96f9bp-47,
	     0x1.b24a16b4d09aap-47,
	     0x1.8d6eeb6efdbd6p-47,
	     0x1.6ba91ac734785p-47,
	     0x1.4cb7966770ab5p-47,
	     0x1.305e9721d0981p-47,
	     0x1.1667311fff70ap-47,
	     0x1.fd3de10d62855p-48,
	     0x1.d1aefbcd48d0cp-48,
	     0x1.a9cc93c25aca9p-48,
	     0x1.85487ee3ea735p-48,
	     0x1.63daf8b4b1e0cp-48,
	     0x1.45421e69a6ca1p-48,
	     0x1.294175802d99ap-48,
	     0x1.0fa17bf41068fp-48,
	     0x1.f05e82aae2bb9p-49,
	     0x1.c578101b29058p-49,
	     0x1.9e39dc5dd2f7cp-49,
	     0x1.7a553a728bbf2p-49,
	     0x1.5982008db1304p-49,
	     0x1.3b7e00422e51bp-49,
	     0x1.200c898d9ee3ep-49,
	     0x1.06f5f7eb65a56p-49,
	     0x1.e00e9148a1d25p-50,
	     0x1.b623734024e92p-50,
	     0x1.8fd4e01891bf8p-50,
	     0x1.6cd44c7470d89p-50,
	     0x1.4cd9c04158cd7p-50,
	     0x1.2fa34bf5c8344p-50,
	     0x1.14f4890ff2461p-50,
	     0x1.f92c49dfa4df5p-51,
	     0x1.ccaaea71ab0dfp-51,
	     0x1.a40829f001197p-51,
	     0x1.7eef13b59e96cp-51,
	     0x1.5d11e1a252bf5p-51,
	     0x1.3e296303b2297p-51,
	     0x1.21f47009f43cep-51,
	     0x1.083768c5e4541p-51,
	     0x1.e1777d831265ep-52,
	     0x1.b69f10b0191b5p-52,
	     0x1.8f8a3a05b5b52p-52,
	     0x1.6be573c40c8e7p-52,
	     0x1.4b645ba991fdbp-52,
	     0x1.2dc119095729fp-52,
  },
};
