/*-----------------------------------------------------------------------*/
/*   Mach64 library test program  M64TEST.C  version  0.1                */
/*                                                                       */
/*   This program is written for DJGPP version >= 2.0                    */
/*                                                                       */
/*   Andreas Leipelt,  Hamburg April 1997                                */
/*   Please send comments, suggestions, bug reports etc. to              */
/*   leipelt@math.uni-hamburg.de                                         */
/*-----------------------------------------------------------------------*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <dpmi.h>
#include <go32.h>
#include <dos.h>
#include <sys/nearptr.h>
#include <sys/farptr.h>
#include <crt0.h>
#include "mach64.h"
#include "pal.inc"
#include "font16x8.inc"

int _crt0_startup_flags = _CRT0_FLAG_NEARPTR | _CRT0_FLAG_NONMOVE_SBRK;

#define WHITE  0xFFFFFFFF

typedef void (*pixel_func)(int,int,int);
typedef int  (*color_func)(int,int,int);

int            width, height, bpp, bpl;
unsigned char *frame_buffer;
pixel_func     pixel;
color_func     color;

void draw_string(int x,int y,int c,char *s);
void pixel8(int x,int y,int c);
void pixel16(int x,int y,int c);
void pixel24(int x,int y,int c);
void pixel32(int x,int y,int c);
int  color8(int r,int g,int b);
int  color15(int r,int g,int b);
int  color16(int r,int g,int b);
int  color24(int r,int g,int b);
int  init_graphics(unsigned short mode);
void box8(int x1,int y1,int x2,int y2,int c);
void strip(int r1,int g1,int b1,int r2,int g2,int b2,int y,int h);

int main()
{
  __dpmi_regs              regs;
  int                      b, c, h, i, j, k, w, tbl_ofs;
  int                      bits[7] = {0, 4, 8, 15, 16, 24, 32};
  unsigned short           size;
  char                     *buf, str[128];
  M64_query_structure      *MQ;
  M64_mode_table_structure **MT, *M_ptr;

  if (M64_detect() != M64_OK) {
    fprintf(stderr,"No Mach64 found.\n");
    return 0;
  }

  if (M64_query_device_data_structure_size(M64_HEADER_AND_TABLES, &size) != M64_OK) {
    fprintf(stderr, "Mach64 error\n");
    return 1;
  }

  if ((buf = (char*)malloc(size)) == NULL) {
    fprintf(stderr, "out of memory\n");
    return 1;
  }

  if (M64_query_device(M64_HEADER_AND_TABLES, size, buf) != M64_OK) {
    free(buf);
    fprintf(stderr, "Mach64 error\n");
    return 1;
  }

  MQ = (M64_query_structure*)buf;
  MT = (M64_mode_table_structure**)malloc((int)MQ->number_of_mode_tables *
					   sizeof(M64_mode_table_structure*));
  if (MT == NULL) {
    free(buf);
    fprintf(stderr,"out of memory\n");
    return 1;
  }

  tbl_ofs = (int)MQ->mode_table_offset;
  for (i=0; i < MQ->number_of_mode_tables; i++) {
    MT[i] = (M64_mode_table_structure*)(buf + tbl_ofs);
    tbl_ofs += (int)MQ->mode_table_size;
  }
  /* check for really different modes */
  i = 0;
  while (i < MQ->number_of_mode_tables) {
    j=i+1;
    while (j < MQ->number_of_mode_tables) {
      if (MT[i]->mode_number == MT[j]->mode_number) {
	if (MT[i]->max_pixel_depth < MT[j]->max_pixel_depth) MT[i] = MT[j];
	MQ->number_of_mode_tables--;
	for (k=j; k < MQ->number_of_mode_tables; k++) MT[k] = MT[k+1];
      }
      else j++;
    }
    i++;
  }

  for (i=0; i < MQ->number_of_mode_tables-1; i++) {
    for (j=i+1; j < MQ->number_of_mode_tables; j++) {
      if ((MT[j]->width < MT[i]->width) ||
	  ((MT[j]->width == MT[i]->width) && (MT[j]->height < MT[i]->height))) {
	M_ptr = MT[i];  MT[i] = MT[j];  MT[j] = M_ptr;
      }
    }
  }

  for (;;) {
    printf("\nAvailable modes:\n\n");
    for (i=0; i < MQ->number_of_mode_tables; i++) {
      printf("  %4d x %4d (", MT[i]->width, MT[i]->height);
      for (j=2; j < MT[i]->max_pixel_depth; j++) printf("%dbit ", bits[j]);
      printf("%dbit)\n", bits[j]);
    }

    printf("\nchoose mode (width height bpp) end program with 0 0 0: ");
    scanf("%d %d %d", &width, &height, &bpp);
    i = -1;
    j = 0;
    switch (bpp) {
      case 8 : bpp = 2;  break;
      case 15: bpp = 3;  break;
      case 16: bpp = 4;  break;
      case 24: bpp = 5;  break;
      case 32: bpp = 6;  break;
      default: free(buf);  free(MT);
	       fprintf(stderr,"invalid bit depth: %d\n", bpp);
	       return 1;
    }

    while ((i == -1) && (j < MQ->number_of_mode_tables)) {
      if ((MT[j]->width == width) && (MT[j]->height == height) &&
	  (bpp > 1) && (bpp <= MT[j]->max_pixel_depth)) i = j;
      j++;
    }

    if (i == -1) {
      free(buf);  free(MT);
      fprintf(stderr, "desired mode is not available!");
      return 1;
    }

    if (!init_graphics(MT[i]->mode_number)) {
      free(buf);  free(MT);
      return 1;
    }

    /* draw frame */
    for (i=0; i < width; i++) {
      pixel(i, 0, WHITE);
      pixel(i, height-1, WHITE);
    }
    for (i=1; i < height-1; i++) {
      pixel(0, i, WHITE);
      pixel(width-1, i, WHITE);
    }

    if (bpp == 8) {
      sprintf(str,"%d x %d x 8bit (8bit DAC mode)", width, height);
    }
    else sprintf(str,"%d x %d x %dbit", width, height, bpp);
    draw_string((width - strlen(str)*8)/2, 5, WHITE, str);

    if (bpp == 8) {
      c = 0;
      b = (width - 17 * 2) / 16;
      h = ((height - 26)/ 16) - 2;
      w = (width - b * 16 - 30) / 2;
      j = 26;  i = w;
      while (c < 256) {
	if (i+b >= width) {
	  i = w;  j += h+2;
	}
	box8(i, j, i+b, j+h, c);
	c++;
	i += b+2;
      }
    }
    else {
      h = 26;
      w = (height - 30) / 7 - 2;
      strip(0, 0, 0, 255, 0, 0, h, w);
      h += w+2;
      strip(0, 0, 0, 0, 255, 0, h, w);
      h += w+2;
      strip(0, 0, 0, 0, 0, 255, h, w);
      h += w+2;
      strip(0, 0, 0, 255, 255, 255, h, w);
      h += w+2;
      strip(255, 0, 0, 0, 255, 0, h, w);
      h += w+2;
      strip(255, 0, 0, 0, 0, 255, h, w);
      h += w+2;
      strip(0, 255, 0, 0, 0, 255, h, w);
    }

    getkey();
    regs.x.ax = 3;
    __dpmi_int(0x10, &regs);
    free(buf);  free(MT);
  }
  return 0;
}

void draw_string(int x, int y, int c, char *s)
{
  char          *ptr = s;
  unsigned char *fptr, row;
  int i, j;

  while (*ptr) {
    fptr = &font16x8[((int)*ptr & 0x7F)* 16];
    for (i=0; i < 16; i++) {
      row = *fptr;
      for (j=0; j < 8; j++) {
	if (row & 128) pixel(x+j, y+i, c);
	row <<= 1;
      }
      fptr++;
    }
    ptr++;
    x+=8;
  }
}

void pixel8(int x, int y, int c)
{
  *(frame_buffer + y*bpl + x) = (unsigned char)(c & 0xFF);
}

void pixel16(int x, int y, int c)
{
  unsigned short *scr_ptr = (unsigned short*)(frame_buffer+y*bpl+2*x);
  *scr_ptr = (unsigned short)(c & 0xFFFF);
}

void pixel24(int x, int y, int c)
{
  unsigned char *scr_ptr = frame_buffer + y*bpl+3*x;

  *scr_ptr++ = (unsigned char)(c & 0xFF);
  *scr_ptr++ = (unsigned char)((c >> 8) & 0xFF);
  *scr_ptr   = (unsigned char)((c >> 16) & 0xFF);
}

void pixel32(int x, int y, int c)
{
  unsigned char *scr_ptr = frame_buffer + y*bpl+4*x+1;

  *scr_ptr++ = (unsigned char)(c & 0xFF);
  *scr_ptr++ = (unsigned char)((c >> 8) & 0xFF);
  *scr_ptr   = (unsigned char)((c >> 16) & 0xFF);
}

int color8(int r, int g, int b) { return r; }

int color15(int r, int g, int b)
{
  return (b >> 3) | ((g >> 3) << 5) | ((r >> 3) << 10);
}

int color16(int r, int g, int b)
{
  return (b >> 3) | ((g >> 2) << 5) | ((r >> 3) << 11);
}

int color24(int r, int g, int b)
{
  return b | (g << 8) | (r << 16);
}

int init_graphics(unsigned short mode)
{
  __dpmi_meminfo info;
  unsigned char aperture_cfg;
  unsigned short aperture_adr, memory_size_and_color_depth, v_params,
      ASIC_ident;

  if (M64_short_query(&aperture_cfg, &aperture_adr,
   &memory_size_and_color_depth, &ASIC_ident) != M64_OK) {
    fprintf(stderr,"could not get aperture info.\n");
    return 0;
  }
  if ((aperture_cfg & 0x1f) == 0) {
    if (M64_memory_aperture_service(M64_ENABLE_APERTURE) != M64_OK) {
      fprintf(stderr, "aperture is disabled and enabling failed!\n");
      return 0;
    }
  }

  if ((aperture_cfg & 0x1f) == 1) info.size = 4 * 1024 * 1024;
  else info.size = 8 * 1024 * 1024;

  info.address = aperture_adr * 1024 * 1024;
  if(__dpmi_physical_address_mapping(&info) == -1) {
    printf("physical mapping of address 0x%x failed!\n", aperture_adr * 1024 * 1024);
    return 0;
  }

  frame_buffer = (unsigned char *)(info.address + __djgpp_conventional_base);

  v_params = (mode << 8) | M64_DISPLAYED_PITCH | bpp;
  if (bpp == 2) v_params |= M64_ENABLE_8BIT_DAC;
  if (M64_load_CRTC_params_and_set_mode(v_params) != M64_OK) {
    fprintf(stderr,"initializing of display mode failed!\n");
    return 0;
  }
  M64_set_palette(0, 254, vga_pal);
  M64_set_palette_entry(255, 255, 255, 255);
  M64_get_mode_info(&width, &height, &bpp, &bpl);

  memset(frame_buffer, 0, bpl * height);

  switch (bpp) {
    case 8 : pixel = pixel8;
	     color = color8;
	     break;
    case 15: pixel = pixel16;
	     color = color15;
	     break;
    case 16: pixel = pixel16;
	     color = color16;
	     break;
    case 24: pixel = pixel24;
	     color = color24;
	     break;
    case 32: pixel = pixel32;
	     color = color24;
	     break;
  }
  return 1;
}

void box8(int x1, int y1, int x2, int y2, int c)
{
  int l = x2 - x1 + 1;
  int i;
  unsigned char *scr_ptr = frame_buffer + y1*bpl + x1;

  for (i=y1; i <= y2; i++) {
    memset(scr_ptr, c, l);
    scr_ptr += bpl;
  }
}

void interpolate(int r1, int g1, int b1, int r2, int g2, int b2, int i,
		 int *r, int *g, int *b)
{
   *r = r1 + i * (r2 - r1) / (width-5);
   *g = g1 + i * (g2 - g1) / (width-5);
   *b = b1 + i * (b2 - b1) / (width-5);
}

void strip(int r1, int g1, int b1, int r2, int g2, int b2, int y, int h)
{
  int b, c, g, i, j, r;

  for (i=0; i < width-4; i++) {
    interpolate(r1,g1,b1,r2,g2,b2,i,&r, &g, &b);
    c = color(r, g, b);
    for (j=0; j < h; j++) pixel(i+2, y+j, c);
  }
}