/*
     _______                     ___                          ________
    /       \         /\        |   |\             /\        |        \
   /         >       /  \       |   ||            /  \       |         \
  /   ______/ >     /    \      |   ||           /    \      |    __    \
 <   <_______/     /      \     |   ||          /      \     |   |\_\    \
  \        \      /   /\   \    |   ||         /   /\   \    |   ||  \    \
   \        \    |   /_L\   |   |   ||        |   /_L\   |   |   ||   >   |\
    \_____   \   |          |\  |   ||        |          |\  |   ||  /    /|
   __L____>   >  |          ||  |   |L____    |          ||  |   |L_/    / /
  /          / > |   ____   ||  |         |\  |   ____   ||  |          / /
 <          / /  |   |\_|   ||  |         ||  |   |\_|   ||  |         / /
  \________/ /   |___|| |___||  |_________||  |___|| |___||  |________/ /
   \________/     \___\  \___\   \_________\   \___\  \___\   \_______\/


                an Addon Package for Allegro by Sven Sandberg


This file contains a function for drawing a 3d sphere. Note that there are
some macros that are defined in s_sfere2 that are necessary for the routine
to work, so you'd better be careful if you change anything here. This file
is inteded to be included from s_gfx.c only.
*/
#ifndef s_sphere_c
#define s_sphere_c

//#include "s_sphere.h"
#include "s_defs.h"
#include "s_gfx.h"
#include <allegro.h>
#include <math.h>



/********************
****             ****
**** draw_sphere ****
****             ****
*********************
Draws an enlightened sphere on the bitmap. The sphere's center is at position
(`sphere_x',`sphere_y',0) and it has a radius of `sphere_radius'. The light
is at position (`light_x',`light_y',`light_z') (relative to the bitmap, not
the sphere). The darkest enlightened place on the sphere has palette index
`first_color_index'. The brightest place on the sphere has palette index
`last_color_index'. Between these, the darkness is interpolated. The color of
places on the sphere that aren't enlightened at all have the palette index
`darkness_color', which should normally be the same as `first_color_index'.

This function can draw to any kind of bitmaps (except RLE and compiled, of
course), including screen. The drawing is a lot faster to 8-bit memory
bitmaps than to other, though. For 15-, 16-, 24- and 32-bit bitmaps, the
color parameters mean the real value of the color rather than the index to
the palette.

How the code works: (Explained in very bad English, I think. Probably noone
else that me can read it. :o)
I don't want to explain all the code in detail here, but I could tell some
general things about it. The function does one x-loop and one y-loop. It
calculates z from the x and y values. Only 1/8 of the z-values are
calculated, the others are mirrored by negating and swapping x and y. z must
be calculated using a `sqrt()', which is bad. It probably takes most of the
time.
The x-loop is inside the y-loop. The x-loop continues until a place outside
the sphere is reached. This is detected by checking x2+y2<radius2. The y-loop
continues until it detects that x never changed inside the y-loop. x-index
starts at the value y, which means the loops cover only the area from 0
degrees to 45 degrees. All other places can be reached by mirroring the
position.
At many places squares are needed. All squares that increase in the loops are
calculated only by adding a number to them. This is possible since the number
added also increases. The derivative then changes linear, and gives a correct
second grade increament.
The brightness `b' is calculated like this: x*lightx + y*lighty + z*lightz,
where the light position is scaled so that it is on the surface of the
sphere. I can't explain why that works and gives a good-looking shadow. I've
only read that it works.

To avoid the slow `sqrt()', it may be possible to implement another
algorithm that uses the classical circle algorithm (is it called Bressenham
too?). That doesn't work with the normal code, however. Maybe it works after
adding some extra cx or d_e or something before doing the z-hemisphere.
However, I don't know how that algorithm works either, so I can't figure out
what to do about it. A beginning of that algorithm is below the real
function.
*/
void draw_sphere_function_name(BITMAP *bmp,
 int sphere_x,int sphere_y,int sphere_radius,
 int light_x,int light_y,int light_z,
 int first_color_index,int last_color_index,int darkness_color)
{
int x,y;
int b;
int light_dist;
int min_x,max_x,min_y,max_y;
int num_colors=last_color_index-first_color_index;//Don't add 1, since the
										//brightest color then  would be one too much.
//For optimization.
int dist2,
 y2,y2add,
 x2,x2add, //x2_init,x2add_init,
 light_y_mul_y, light_x_mul_y,
 light_x_mul_x, light_y_mul_x, light_x_mul_x_init, light_y_mul_x_init,
 light_x_mul_x_mul, light_x_mul_y_mul, light_y_mul_x_mul, light_y_mul_y_mul,
 light_z_mul_z_plus_offset,
 //Almost the same as the square of (sphere_radius + 1.5). Adding 1 makes the
 //sphere bigger: The diameter now is (radius * 2 + 1) instead of
 //(radius * 2 - 1). Adding .5 gives a much better rounding.
 sphere_radius2=(sphere_radius+1)*(sphere_radius+2);
#ifdef draw_sphere_is_linear
int dew=0;//Init `dew' here to avoid `not initialised'-warning.
uchar *de1, *de2, *de3, *de4, *de1_init, *de4_init;
#endif

//Create loop ranges (=clipping).
min_x = -sphere_radius;
max_x = sphere_radius;
min_y = -sphere_radius;
max_y = sphere_radius;
if(min_x+sphere_x<0)
	min_x=-sphere_x;
if(max_x+sphere_x>bmp->w)
	max_x=bmp->w-sphere_x;
if(min_y+sphere_y<0)
	min_y=-sphere_y;
if(max_y+sphere_y>bmp->h)
	max_y=bmp->h-sphere_y;
if((min_x>bmp->w) || (max_x<0) || (min_y>bmp->h) || (max_y<0) || (num_colors<1))
	return;

//light_x and light_y are now relative to the sphere's centrum.
light_x -= sphere_x;
light_y -= sphere_y;
//Scale light position so that it is on the surface of the sphere.
light_dist = sqrt(light_x*light_x + light_y*light_y + light_z*light_z);
light_x = (light_x * sphere_radius) / light_dist;
light_y = (light_y * sphere_radius) / light_dist;
light_z = (light_z * sphere_radius) / light_dist;

//Some values that we can pre-calculate to optimize.
y2=0;//min_y*min_y
y2add=1;//min_y*2+1
light_x_mul_y=0;
light_y_mul_y=0;
//Store these so that we don't need to calculate them in the loop.
light_x_mul_x_init= -light_x;//Decrease now so that we can increase first
light_y_mul_x_init= -light_y;//time.
//Make x!=y.
x=1;
#ifdef draw_sphere_is_linear
if(bmp->h > 1)
	dew = (&bmp->line[1][0]) - (&bmp->line[0][0]);
de1=de2=de3=de4=&bmp->line[sphere_y][sphere_x];
de1_init=de1-dew;//Decrease now so that we can increase first time.
de4_init=de4+dew;
#endif
//When y becomes so big that there are no pixels left to draw, then x==y.
for(y=0; x!=y; y++){
	x2=y2;
	x2add=y2add;
	light_x_mul_x=(light_x_mul_x_init+=light_x);
	light_y_mul_x=(light_y_mul_x_init+=light_y);
#ifdef draw_sphere_is_linear
	de1=(de1_init+=dew);
	de4=(de4_init-=dew);
#endif
	//Loop till we come outside the sphere.
	for(x=y; (dist2=x2+y2)<sphere_radius2; x++){
		//Calculate z position from x and y.
		light_z_mul_z_plus_offset = first_color_index +
		 (num_colors * light_z * sqrt(sphere_radius2-dist2)) /sphere_radius2;

		//Pre-calculate some values for brightness in the 8 mirrored positions.
		light_x_mul_x_mul = (num_colors * light_x_mul_x) / sphere_radius2;
		light_x_mul_y_mul = (num_colors * light_x_mul_y) / sphere_radius2;
		light_y_mul_x_mul = (num_colors * light_y_mul_x) / sphere_radius2;
		light_y_mul_y_mul = (num_colors * light_y_mul_y) / sphere_radius2;
		//Point #1: Right/up/down.
		b=light_z_mul_z_plus_offset + light_x_mul_x_mul+light_y_mul_y_mul;
		if(b>=first_color_index)
			draw_sphere_putpixel(*(de2+x), x, y, b);
		else
			draw_sphere_putpixel(*(de2+x), x, y, darkness_color);
		//Point #2: Left/up/down.
		b=light_z_mul_z_plus_offset -light_x_mul_x_mul+light_y_mul_y_mul;
		if(b>=first_color_index)
			draw_sphere_putpixel(*(de2-x), x, y, b);
		else
			draw_sphere_putpixel(*(de2-x), x, y, darkness_color);
		//Point #3: Right/down/up.
		b=light_z_mul_z_plus_offset +light_x_mul_x_mul-light_y_mul_y_mul;
		if(b>=first_color_index)
			draw_sphere_putpixel(*(de3+x), x, y, b);
		else
			draw_sphere_putpixel(*(de3+x), x, y, darkness_color);
		//Point #4: Left/down/up.
		b=light_z_mul_z_plus_offset -light_x_mul_x_mul-light_y_mul_y_mul;
		if(b>=first_color_index)
			draw_sphere_putpixel(*(de3-x), x, y, b);
		else
			draw_sphere_putpixel(*(de3-x), x, y, darkness_color);
		//Point #5: Right/up/up.
		b=light_z_mul_z_plus_offset +light_x_mul_y_mul+light_y_mul_x_mul;
		if(b>=first_color_index)
			draw_sphere_putpixel(*(de1+y), x, y, b);
		else
			draw_sphere_putpixel(*(de1+y), x, y, darkness_color);
		//Point #6: Left/up/up.
		b=light_z_mul_z_plus_offset -light_x_mul_y_mul+light_y_mul_x_mul;
		if(b>=first_color_index)
			draw_sphere_putpixel(*(de1-y), x, y, b);
		else
			draw_sphere_putpixel(*(de1-y), x, y, darkness_color);
		//Point #7: Right/down/down.
		b=light_z_mul_z_plus_offset +light_x_mul_y_mul-light_y_mul_x_mul;
		if(b>=first_color_index)
			draw_sphere_putpixel(*(de4+y), x, y, b);
		else
			draw_sphere_putpixel(*(de4+y), x, y, darkness_color);
		//Point #7: Left/down/down.
		b=light_z_mul_z_plus_offset -light_x_mul_y_mul-light_y_mul_x_mul;
		if(b>=first_color_index)
			draw_sphere_putpixel(*(de4-y), x, y, b);
		else
			draw_sphere_putpixel(*(de4-y), x, y, darkness_color);

		x2+=(x2add+=2);
		light_x_mul_x+=light_x;
		light_y_mul_x+=light_y;
#ifdef draw_sphere_is_linear
		de1+=dew;
		de4-=dew;
#endif
	}
	y2+=(y2add+=2);
	light_x_mul_y+=light_x;
	light_y_mul_y+=light_y;
	x++;
#ifdef draw_sphere_is_linear
	de2+=dew;
	de3-=dew;
#endif
}

}



//This is the beginning of an optimized `draw_sphere()'-function using the
//cool circle-drawing algorithm. This avoids the `sqrt()' call that probably
//takes most of the time. However, it looks much worse and I have to figure
//out how to fix that (if it is possible) before implementing it.

#if 0==1


BITMAP *sp_bmp;
int sp_x, sp_y, sp_radius;
int sp_light_x, sp_light_y, sp_light_z;
int sp_first_color_index, sp_last_color_index, sp_darkness_color;
int sp_num_colors;
int sp_radius2;

/**************************
****                   ****
**** draw_sphere_pixel ****
****                   ****
***************************
*/
void draw_sphere_pixel(int x, int y, int z)
{
//Calc brightness.
int b=sp_first_color_index+
 (sp_num_colors*(sp_light_x*x + sp_light_y*y + sp_light_z*z)) / sp_radius2;
//If pixel was enlightened.
if(b>=sp_first_color_index)
	sp_bmp->line[sp_y+y][sp_x+x] = b;
else
	sp_bmp->line[sp_y+y][sp_x+x] = sp_darkness_color;
}

/*************************
****                  ****
**** draw_sphere_line ****
****                  ****
**************************
Helper function that draws one horizontal line on the sphere.
*/
void draw_sphere_line(int x,int y)
{
//x is the radius.
int cx = 0;
int cz = x;
int df = 1 - x;
int d_e = 3;
int d_se = -2 * x + 5;

do {
	draw_sphere_pixel(cx,y,cz);
	draw_sphere_pixel(-cx,y,cz);

	if (df < 0)  {
		df += d_e;
		d_e += 2;
		d_se += 2;
	} else {
		df += d_se;
		d_e += 2;
		d_se += 4;
		cz--;
	} 

	cx++; 

} while (cx <= cz);
}

/********************
****             ****
**** draw_sphere ****
****             ****
*********************
*/
void draw_sphere_linear(BITMAP *bmp, int sphere_x,int sphere_y,int sphere_radius,
 int light_x,int light_y,int light_z,
 int first_color_index,int last_color_index,int darkness_color)
{
int cx = 0;
int cy = sphere_radius;
int df = 1 - sphere_radius;
int d_e = 3;
int d_se = -2 * sphere_radius + 5;

sp_radius2=sphere_radius*sphere_radius;
sp_bmp=bmp;
sp_x=sphere_x;
sp_y=sphere_y;
sp_light_x=light_x;
sp_light_y=light_y;
sp_light_z=light_z;
sp_first_color_index=first_color_index;
sp_num_colors=last_color_index-first_color_index+1;
sp_radius=sphere_radius;

do {
	draw_sphere_line(cx, cy);
	draw_sphere_line(cx, -cy);
	draw_sphere_line(cy, cx);
	draw_sphere_line(cy, -cx);

	if (df < 0)  {
		df += d_e;
		d_e += 2;
		d_se += 2;
	} else {
		df += d_se;
		d_e += 2;
		d_se += 4;
		cy--;
	} 

	cx++; 

} while (cx <= cy);

}

#endif   //if 0==1




#endif
