/*
     _______                     ___                          ________
    /       \         /\        |   |\             /\        |        \
   /         >       /  \       |   ||            /  \       |         \
  /   ______/ >     /    \      |   ||           /    \      |    __    \
 <   <_______/     /      \     |   ||          /      \     |   |\_\    \
  \        \      /   /\   \    |   ||         /   /\   \    |   ||  \    \
   \        \    |   /_L\   |   |   ||        |   /_L\   |   |   ||   >   |\
    \_____   \   |          |\  |   ||        |          |\  |   ||  /    /|
   __L____>   >  |          ||  |   |L____    |          ||  |   |L_/    / /
  /          / > |   ____   ||  |         |\  |   ____   ||  |          / /
 <          / /  |   |\_|   ||  |         ||  |   |\_|   ||  |         / /
  \________/ /   |___|| |___||  |_________||  |___|| |___||  |________/ /
   \________/     \___\  \___\   \_________\   \___\  \___\   \_______\/


                an Addon Package for Allegro by Sven Sandberg


This file contains some math functions (different kinds of rounding mostly).

*/
#ifndef s_math_h
#define s_math_h

//For `ulong' etc.
#include "s_defs.h"
//For `_cos_tbl[]'.
#include <allegro.h>


#define _get_bit(buf,bitno)   ((((buf)[(bitno) >> 3]) >> ((bitno) & 7)) & 1)

extern inline int get_bit(uchar *buf, int bitno);
extern inline ulonglong get_bits(uchar *buf, int firstbit, int numbits);
extern void copy_bits(uchar *so, uchar *de,
 int sostart, int destart, int numbits);

//Round a double to an integer.
#define d2l(r)       ((long)        ( (r) + .5 ))
#define d2ul(r)      ((ulong)       ( (r) + .5 ))
#define d2ll(r)      ((longlong)    ( (r) + .5 ))
#define d2ull(r)     ((ulonglong)   ( (r) + .5 ))

//Round a double to the closest whole number, but still in the form of a
//double.
#define round(r)     (floor((r) + .5))


//Divide two integers. Instead of truncating the result (removing eventual
//decimals), the result is rounded to the closest number.
#define _rounddiv(n,d)  ( ((n) + ((d) >> 1)) / (d) )
extern inline long rounddiv(long n,long d);
extern inline ulong urounddiv(ulong n,ulong d);
extern inline slonglong llrounddiv(slonglong n,slonglong d);
extern inline ulonglong ullrounddiv(ulonglong n,ulonglong d);
#define _ceildiv(n,d)   ( ((n) + (d) - 1) / (d) )
extern inline long ceildiv(long n,long d);
extern inline ulong uceildiv(slong n,slong d);
extern inline slonglong llceildiv(slonglong n,slonglong d);
extern inline ulonglong ullceildiv(ulonglong n,ulonglong d);

//Round the integer `n' upwards/downwards to the closest number divisible by
//`d'.
#define _roundup(n,d)   ((n) - (((n) - 1) % (d)) - 1 + (d))
extern inline long roundup(long n,long d);
extern inline ulong uroundup(ulong n,ulong d);
extern inline slonglong llroundup(slonglong n,slonglong d);
extern inline ulonglong ullroundup(ulonglong n,ulonglong d);
#define _rounddown(n,d) ((n) - ((n) % (d)))
extern inline long rounddown(long n,long d);
extern inline ulong urounddown(ulong n,ulong d);
extern inline slonglong llrounddown(slonglong n,slonglong d);
extern inline ulonglong ullrounddown(ulonglong n,ulonglong d);

//Returns the smallest number that you need to add to n to make it divisible
//by d.
#define _upmod(n,d)     ((d) - (((n) - 1) % (d)) - 1)
extern inline long upmod(long n,long d);
extern inline ulong uupmod(ulong n,ulong d);
extern inline slonglong llupmod(slonglong n,slonglong d);
extern inline ulonglong ullupmod(ulonglong n,ulonglong d);

/* This is similar to upmod(). It returns the smallest number >=1 that you
	must add to `n' to make it divisible by `d'. A way to realize what this
	means is to think that you're placed at column `n' in a text editor that
	has tab width `d'. This function returns the number of columns that you
	jump to the right when you press tab.
*/
#define _nextmod(n,d)   ((d) - ((n) % (d)))
extern inline long nextmod(long n,long d);
extern inline ulong unextmod(ulong n,ulong d);
extern inline slonglong llnextmod(slonglong n,slonglong d);
extern inline ulonglong ullnextmod(ulonglong n,ulonglong d);

#define _abs(r)         (((r)<0) ? -(r) : (r))
#define _sign(r)        (((r)>0) ? (1) : (((r)<0) ? (-1) : (0)))
#define _min(a,b)       (((a)<(b)) ? (a) : (b))
#define _max(a,b)       (((a)>(b)) ? (a) : (b))
#define _bound(a,b,c)   (_max(_min(b,c), a))
#define _sq(r)          ((r) * (r))
#define _swap(a,b)                                                         \
	{                                                                       \
		(a) ^= (b);                                                          \
		(b) ^= (a);                                                          \
		(a) ^= (b);                                                          \
	}
extern slong ssign(slong x);
extern slong ssq(slong x);
extern slong smin(slong x1,slong x2);
extern slong smax(slong x1,slong x2);
extern slong sbound(slong min, slong x, slong max);
extern ulong uabs(ulong x);
extern ulong usign(ulong x);
extern ulong usq(ulong x);
extern ulong umin(ulong x1,ulong x2);
extern ulong umax(ulong x1,ulong x2);
extern ulong ubound(ulong min, ulong x, ulong max);
extern slonglong llabs(slonglong x);
extern slonglong llsign(slonglong x);
extern slonglong llsq(slonglong x);
extern slonglong llmin(slonglong x1,slonglong x2);
extern slonglong llmax(slonglong x1,slonglong x2);
extern slonglong llbound(slonglong min, slonglong x, slonglong max);
extern ulonglong ullabs(ulonglong x);
extern ulonglong ullsign(ulonglong x);
extern ulonglong ullsq(ulonglong x);
extern ulonglong ullmin(ulonglong x1,ulonglong x2);
extern ulonglong ullmax(ulonglong x1,ulonglong x2);
extern ulonglong ullbound(ulonglong min, ulonglong x, ulonglong max);


//Char trigonometry (a full circle is 256, the max value of sin(x) is 256).
#define csin(a)   (_cos_tbl[(((a)>>1) - 128) & 0x1ff])
#define ccos(a)   (_cos_tbl[(((a)>>1) & 0x1ff])
#define ctan(a)   (_tan_tbl[(a) & 0xff])
#define casin(a)  (_acos_tbl[ (a) << 1 ])
//Long trigonometry (a full circle is the maximum for a 32-bit unsigned long,
//and the max value of sin(x) is the same).
#define lsin(a)   (_cos_tbl[(( (a) - (128<<23) + (1<<22) ) >> 23) & 0x1FF])
#define lcos(a)   (_cos_tbl[( (a) + (1<<22) ) >> 23])
#define ltan(a)   (_tan_tbl[( (a) + (1<<23) ) >> 24])


//Faster multiplication.
#define m2(num)      ( (num)+(num) )
#define m4(num)      ( (num)<<2 )
#define m8(num)      ( (num)<<3 )
#define m16(num)     ( (num)<<4 )
#define m32(num)     ( (num)<<5 )
#define m64(num)     ( (num)<<6 )
#define m128(num)    ( (num)<<7 )
#define m256(num)    ( (num)<<8 )
#define m512(num)    ( (num)<<9 )
#define m1024(num)   ( (num)<<10 )
#define m2048(num)   ( (num)<<11 )
#define m4096(num)   ( (num)<<12 )

#define m3(num)      ( (num) + (num) + (num) )
#define m6(num)      ( ((num) + (num) + (num)) <<1 )
#define m12(num)     ( ((num) + (num) + (num)) <<2 )
#define m24(num)     ( ((num) + (num) + (num)) <<3 )
#define m48(num)     ( ((num) + (num) + (num)) <<4 )
#define m96(num)     ( ((num) + (num) + (num)) <<5 )
#define m192(num)    ( ((num) + (num) + (num)) <<6 )
#define m384(num)    ( ((num) + (num) + (num)) <<7 )
#define m768(num)    ( ((num) + (num) + (num)) <<8 )
#define m1536(num)   ( ((num) + (num) + (num)) <<9 )

#define m5(num)      ( ((num)<<2) + (num) )
#define m10(num)     ( ((num)<<3) + ((num) + (num)) )
#define m20(num)     ( ((num)<<4) + ((num)<<2) )
#define m40(num)     ( ((num)<<5) + ((num)<<3) )
#define m80(num)     ( ((num)<<6) + ((num)<<4) )
#define m160(num)    ( ((num)<<7) + ((num)<<5) )
#define m320(num)    ( ((num)<<8) + ((num)<<6) )
#define m640(num)    ( ((num)<<9) + ((num)<<7) )
#define m1280(num)   ( ((num)<<10)+ ((num)<<8) )
#define m2560(num)   ( ((num)<<11)+ ((num)<<9) )

#define m9(num)      ( ((num)<<3) + (num) )
#define m18(num)     ( ((num)<<4) + ((num) + (num)) )
#define m36(num)     ( ((num)<<5) + ((num)<<2) )
#define m72(num)     ( ((num)<<6) + ((num)<<3) )
#define m144(num)    ( ((num)<<7) + ((num)<<4) )
#define m288(num)    ( ((num)<<8) + ((num)<<5) )
#define m576(num)    ( ((num)<<9) + ((num)<<6) )
#define m1152(num)   ( ((num)<<10)+ ((num)<<7) )

//I'm not sure the m50 and m70 macros will be faster than multiplication.
//That may depend on the processor.
#define m30(num)     ( ((num)<<5) - (num) - (num) )
#define m50(num)     ( ((num)<<5) + ((num)<<4) + (num) + (num) )
#define m60(num)     ( ((num)<<6) - ((num)<<2) )
#define m70(num)     ( ((num)<<5) + (((num) + (num) + (num)) << 2) )

/*The cm* functions initialize the number to the new value, they don't only
return it like the corresponding m* functions.*/
#define cm2(num)     ( (num)+=(num) )
#define cm4(num)     ( (num)<<=2 )
#define cm8(num)     ( (num)<<=3 )
#define cm16(num)    ( (num)<<=4 )
#define cm32(num)    ( (num)<<=5 )
#define cm64(num)    ( (num)<<=6 )
#define cm128(num)   ( (num)<<=7 )
#define cm256(num)   ( (num)<<=8 )
#define cm512(num)   ( (num)<<=9 )
#define cm1024(num)  ( (num)<<=10 )
#define cm2048(num)  ( (num)<<=11 )
#define cm4096(num)  ( (num)<<=12 )

#define cm3(num)     ( (num)+= ((num) + (num)) )
#define cm6(num)     ( (num) = (((num) + (num) + (num)) <<1) )
#define cm12(num)    ( (num) = (((num) + (num) + (num)) <<2) )
#define cm24(num)    ( (num) = (((num) + (num) + (num)) <<3) )
#define cm48(num)    ( (num) = (((num) + (num) + (num)) <<4) )
#define cm96(num)    ( (num) = (((num) + (num) + (num)) <<5) )
#define cm192(num)   ( (num) = (((num) + (num) + (num)) <<6) )
#define cm384(num)   ( (num) = (((num) + (num) + (num)) <<7) )
#define cm768(num)   ( (num) = (((num) + (num) + (num)) <<8) )
#define cm1536(num)  ( (num) = (((num) + (num) + (num)) <<9) )

#define cm5(num)     ( (num)+= ((num)<<2) )
#define cm10(num)    ( (num) = ((num)<<3) + ((num) + (num)) )
#define cm20(num)    ( (num) = ((num)<<4) + ((num)<<2) )
#define cm40(num)    ( (num) = ((num)<<5) + ((num)<<3) )
#define cm80(num)    ( (num) = ((num)<<6) + ((num)<<4) )
#define cm160(num)   ( (num) = ((num)<<7) + ((num)<<5) )
#define cm320(num)   ( (num) = ((num)<<8) + ((num)<<6) )
#define cm640(num)   ( (num) = ((num)<<9) + ((num)<<7) )
#define cm1280(num)  ( (num) = ((num)<<10)+ ((num)<<8) )
#define cm2560(num)  ( (num) = ((num)<<11)+ ((num)<<9) )

#define cm9(num)     ( (num)+= ((num)<<3) )
#define cm18(num)    ( (num) = ((num)<<4) + ((num) + (num)) )
#define cm36(num)    ( (num) = ((num)<<5) + ((num)<<2) )
#define cm72(num)    ( (num) = ((num)<<6) + ((num)<<3) )
#define cm144(num)   ( (num) = ((num)<<7) + ((num)<<4) )
#define cm288(num)   ( (num) = ((num)<<8) + ((num)<<5) )
#define cm576(num)   ( (num) = ((num)<<9) + ((num)<<6) )
#define cm1152(num)  ( (num) = ((num)<<10)+ ((num)<<7) )


//Faster division.
#define d2(num)      ( (num)>>1 )
#define d4(num)      ( (num)>>2 )
#define d8(num)      ( (num)>>3 )
#define d16(num)     ( (num)>>4 )
#define d32(num)     ( (num)>>5 )
#define d64(num)     ( (num)>>6 )
#define d128(num)    ( (num)>>7 )
#define d256(num)    ( (num)>>8 )
#define d512(num)    ( (num)>>9 )
#define d1024(num)   ( (num)>>10 )
#define d2048(num)   ( (num)>>11 )
#define d4096(num)   ( (num)>>12 )
//These also change the number.
#define cd2(num)     ( (num)>>=1 )
#define cd4(num)     ( (num)>>=2 )
#define cd8(num)     ( (num)>>=3 )
#define cd16(num)    ( (num)>>=4 )
#define cd32(num)    ( (num)>>=5 )
#define cd64(num)    ( (num)>>=6 )
#define cd128(num)   ( (num)>>=7 )
#define cd256(num)   ( (num)>>=8 )
#define cd512(num)   ( (num)>>=9 )
#define cd1024(num)  ( (num)>>=10 )
#define cd2048(num)  ( (num)>>=11 )
#define cd4096(num)  ( (num)>>=12 )

#endif
