/*
     _______                     ___                          ________
    /       \         /\        |   |\             /\        |        \
   /         >       /  \       |   ||            /  \       |         \
  /   ______/ >     /    \      |   ||           /    \      |    __    \
 <   <_______/     /      \     |   ||          /      \     |   |\_\    \
  \        \      /   /\   \    |   ||         /   /\   \    |   ||  \    \
   \        \    |   /_L\   |   |   ||        |   /_L\   |   |   ||   >   |\
    \_____   \   |          |\  |   ||        |          |\  |   ||  /    /|
   __L____>   >  |          ||  |   |L____    |          ||  |   |L_/    / /
  /          / > |   ____   ||  |         |\  |   ____   ||  |          / /
 <          / /  |   |\_|   ||  |         ||  |   |\_|   ||  |         / /
  \________/ /   |___|| |___||  |_________||  |___|| |___||  |________/ /
   \________/     \___\  \___\   \_________\   \___\  \___\   \_______\/


                an Addon Package for Allegro by Sven Sandberg


This file contains some math functions (different kinds of rounding mostly).

*/
#ifndef s_math_c
#define s_math_c

#include "s_math.h"
#include "s_common.h"

/************************
****                 ****
**** sign,min,sq etc ****
****                 ****
*************************
Functions instead of macros.
*/
slong ssign(slong x){return _sign(x);}
slong ssq(slong x){return _sq(x);}
slong smin(slong x1,slong x2){return _min(x1,x2);}
slong smax(slong x1,slong x2){return _max(x1,x2);}
slong bound(slong min, slong x, slong max){return _bound(min,x,max);}
ulong uabs(ulong x){return _abs(x);}
ulong usign(ulong x){return _sign(x);}
ulong usq(ulong x){return _sq(x);}
ulong umin(ulong x1,ulong x2){return _min(x1,x2);}
ulong umax(ulong x1,ulong x2){return _max(x1,x2);}
ulong ubound(ulong min, ulong x, ulong max){return _bound(min,x,max);}
slonglong llabs(slonglong x){return _abs(x);}
slonglong llsign(slonglong x){return _sign(x);}
slonglong llsq(slonglong x){return _sq(x);}
slonglong llmin(slonglong x1,slonglong x2){return _min(x1,x2);}
slonglong llmax(slonglong x1,slonglong x2){return _max(x1,x2);}
slonglong llbound(slonglong min, slonglong x, slonglong max){return _bound(min,x,max);}
ulonglong ullabs(ulonglong x){return _abs(x);}
ulonglong ullsign(ulonglong x){return _sign(x);}
ulonglong ullsq(ulonglong x){return _sq(x);}
ulonglong ullmin(ulonglong x1,ulonglong x2){return _min(x1,x2);}
ulonglong ullmax(ulonglong x1,ulonglong x2){return _max(x1,x2);}
ulonglong ullbound(ulonglong min, ulonglong x, ulonglong max){return _bound(min,x,max);}




/*****************
****          ****
**** rounddiv ****
****          ****
******************
Divide and round the result to the closest.
*/
inline long rounddiv(long n,long d)
{
return (n + (d >> 1)) / d;
}
inline ulong urounddiv(ulong n,ulong d)
{
return (n + (d >> 1)) / d;
}
inline ulonglong ullrounddiv(ulonglong n,ulonglong d)
{
return (n + (d >> 1)) / d;
}
inline slonglong llrounddiv(slonglong n,slonglong d)
{
return (n + (d >> 1)) / d;
}
/****************
****         ****
**** ceildiv ****
****         ****
*****************
Divide and round the result upwards.
*/
inline long ceildiv(long n,long d)
{
return (n + d - 1) / d;
}
inline ulong uceildiv(slong n,slong d)
{
return (n + d - 1) / d;
}
inline slonglong llceildiv(slonglong n,slonglong d)
{
return (n + d - 1) / d;
}
inline ulonglong ullceildiv(ulonglong n,ulonglong d)
{
return (n + d - 1) / d;
}


/****************
****         ****
**** roundup ****
****         ****
*****************
Find the smallest number divisible by d that is bigger than or equal to n.
*/
inline long roundup(long n,long d)
{
return n + d + ((n-1)%d) - 1;
}
inline ulong uroundup(ulong n,ulong d)
{
return n + d + ((n-1)%d) - 1;
}
inline slonglong llroundup(slonglong n,slonglong d)
{
return n + d + ((n-1)%d) - 1;
}
inline ulonglong ullroundup(ulonglong n,ulonglong d)
{
return n + d + ((n-1)%d) - 1;
}
/******************
****           ****
**** rounddown ****
****           ****
*******************
Find the biggest number divisible by d that is smaller than or equal to n.
*/
inline long rounddown(long n,long d)
{
return n - (n%d);
}
inline ulong urounddown(ulong n,ulong d)
{
return n - (n%d);
}
inline slonglong llrounddown(slonglong n,slonglong d)
{
return n - (n%d);
}
inline ulonglong ullrounddown(ulonglong n,ulonglong d)
{
return n - (n%d);
}

/**************
****       ****
**** upmod ****
****       ****
***************
Return the smallest number that you must add to n to make it divisible by d.
*/
inline long upmod(long n,long d)
{
return _upmod(n,d);
}
inline ulong uupmod(ulong n,ulong d)
{
return _upmod(n,d);
}
inline slonglong llupmod(slonglong n,slonglong d)
{
return _upmod(n,d);
}
inline ulonglong ullupmod(ulonglong n,ulonglong d)
{
return _upmod(n,d);
}

/****************
****         ****
**** nextmod ****
****         ****
*****************
This is similar to upmod(). It returns the smallest number >=1 that you must
add to n to make it divisible by d. A way to realize what this means is to
think that you're placed at column n in a text editor that has tab width d.
This function returns the number of columns that you jump to the right when
you press tab.
*/
inline long nextmod(long n,long d)
{
return _nextmod(n,d);
}
inline ulong unextmod(ulong n,ulong d)
{
return _nextmod(n,d);
}
inline slonglong llnextmod(slonglong n,slonglong d)
{
return _nextmod(n,d);
}
inline ulonglong ullnextmod(ulonglong n,ulonglong d)
{
return _nextmod(n,d);
}



/****************
****         ****
**** get_bit ****
****         ****
*****************
Returns the `bitno'th bit in `buf'.
*/
inline int get_bit(uchar *buf, int bitno)
{
return _get_bit(buf,bitno);
}


/*****************
****          ****
**** get_bits ****
****          ****
******************
Returns some bits in `buf'. The first bit is `bitno' and the number of bits
returned is `numbits'. `buf' should contain so many bytes that it could hold
the number even when `numbits' is 64.
*/
ulonglong get_bits(uchar *buf, int firstbit, int numbits)
{
//Get the bytes that are going to be used.
return (   (  *( (ulonglong*)(buf + (firstbit>>3)) )
//Shift them right in case first bit isn't on an even byte.
 >> (firstbit & 7)  )

//Add the extra byte that we may have lost if `firstbit' isn't on an even
//byte (64 bits would then occupy 9 bytes, so the last byte would get lost).
 + ( (((ulonglong)buf[(firstbit>>3) + 8])
//Move the extra byte to its proper position. (Two <<32 instead of one <<64
//is needed, because otherwise nothing will be shifted at all when firstbit&7
//is 0. << is undefined when the number of bits to shift is more than 64.)
 << 32) << (32 - (firstbit & 7)) )   )

//Finally, we need to `&' the result so far with a number that contains 1-s
//on the `numbits' rightmost bits, in case too many bits were copied.
 & (max_ulonglong  >> (64 - numbits));
}



#define _get_byte(buf, startbit)                                           \
	(uchar)((*(ushort*)((buf) + ((startbit) >> 3))) >> ((startbit) & 7))
/******************
****           ****
**** copy_bits ****
****           ****
*******************
Copies `numbits' bits from `so' to `de', starting in `so' with `firstbit'.
`de' is always filled from the beginning.
*/
void copy_bits_tostart(uchar *so, uchar *de, int firstbit, int numbits)
{
//If we're copying from somewhere inside a byte.
if(firstbit & 7){
	int soi, dei;
	int max_soi;

	//Always copy something after the loop. Not always copy something in
	//the loop.
	max_soi = firstbit + numbits - 8;
	for(soi = firstbit, dei = 0; soi < max_soi; soi += 8, dei++){
		de[dei] = _get_byte(so, soi);
	}
	//If end of range in `so' is placed so that last byte in `de' is taken
	//from two bytes in `so' instead of from one.
	if(((firstbit + numbits - 1) & 7) + 1 <= (firstbit & 7)){
		//Remove right part of de byte. (shift 1 to 8 steps, inclusive)
		de[dei] &= (int)max_uchar << (((numbits - 1) & 7) + 1);
		//Remove left part of so byte, then copy it to de.
		//(shift 0 to 7 steps, inclusive)
		de[dei] |= _get_byte(so, soi) & (max_uchar >> ((8 - numbits) & 7));
	//If end of range in `so' is placed so that last byte in `de' is taken
	//from one byte in `so' instead of from two.
	}else{
		//`numbits' can't divisible by 8: in that case we wouldn't come here,
		//but to the block before `else' above.
		de[dei] &= (int)max_uchar << (numbits & 7);
		//Instead of `_get_byte()': doesn't read more than one byte.
		de[dei] |= (so[soi >> 3] >> (firstbit & 7)) &
		//Remove left part of byte.
		 (max_uchar >> ((8 - numbits) & 7));
	}
//If we're copying from an even byte, no shifts are needed, and we can use
//`memcopy()'.
}else{
	//Always copy something after the `memcopy()'.
	memcopy(de, so + (firstbit >> 3), (numbits - 1) >> 3);
	//Remove bits in `de'.
	de[(numbits - 1) >> 3] &= max_uchar << (numbits & 7);
	//Copy bits from `so' to `de'.
	de[(numbits - 1) >> 3] |= so[(firstbit + numbits - 1) >> 3] &
	//Remove the extra bits in the `so' byte.
	 (max_uchar >> ((8 - numbits) & 7));
}
}

//Returns 8 bits that are not on an even byte.
void copy_bits(uchar *so, uchar *de, int sostart, int destart, int numbits)
{
if(numbits == 0)
	return;
if(destart & 7){
	//If `de' will start and end within one byte.
	if(((destart + numbits - 1) >> 3) == (destart >> 3)){
		//Clear the bits we will copy to.
		de[destart >> 3] &= ~((max_uchar >> (8 - numbits)) << (destart & 7));
		//If we need to read from two bytes in `so'.
		if(((sostart + numbits) >> 3) > (sostart >> 3)){
			//Init `de'.
			de[destart >> 3] |= (_get_byte(so, sostart) &
			 //Remove the leftmost bits read.
			 (max_uchar >> (8 - numbits))) <<
			 //Shift the value to its proper position.
			 (destart & 7);
		//If we only need to read from one byte.
		}else{
			//Copy from only one byte.
			de[destart >> 3] |= ((so[sostart >> 3] >> (sostart & 7)) &
			 //`and' away overflowing bits.
			 (max_uchar >> (8 - numbits))) <<
			 //`shift' bits to the right position.
			 (destart & 7);
		}
	//If more than one byte will be required in `de'.
	}else{
		//Number of bits copied first time.
		int startbits = (8 - destart) & 7;
		//Clear bits we are about to copy to.
		de[destart >> 3] &= (int)max_uchar >> startbits;
		//If first `de' byte is copied from two `so' bytes.
		if((sostart & 7) > (destart & 7)){
			de[destart >> 3] |= (_get_byte(so, sostart) &
			 //Remove overflowing bits.
			 (max_uchar >> startbits)) <<
			 //Move bits to their position.
			 (sostart & 7);
		//If first `de' byte is copied from only one `so' byte.
		}else{
			de[destart >> 3] |= (so[sostart >> 3] >>
			 //Remove least significant overflowing bits.
			 (sostart & 7)) <<
			 //Move bits to their position.
			 (destart & 7);
		}
		copy_bits_tostart(so, de + (destart >> 3) + 1,
		 sostart + startbits, numbits - startbits);
	}
}else{
	copy_bits_tostart(so, de + (destart >> 3), sostart, numbits);
}
}
#undef get_byte


#endif
