/*
     _______                     ___                          ________
    /       \         /\        |   |\             /\        |        \
   /         >       /  \       |   ||            /  \       |         \
  /   ______/ >     /    \      |   ||           /    \      |    __    \
 <   <_______/     /      \     |   ||          /      \     |   |\_\    \
  \        \      /   /\   \    |   ||         /   /\   \    |   ||  \    \
   \        \    |   /_L\   |   |   ||        |   /_L\   |   |   ||   >   |\
    \_____   \   |          |\  |   ||        |          |\  |   ||  /    /|
   __L____>   >  |          ||  |   |L____    |          ||  |   |L_/    / /
  /          / > |   ____   ||  |         |\  |   ____   ||  |          / /
 <          / /  |   |\_|   ||  |         ||  |   |\_|   ||  |         / /
  \________/ /   |___|| |___||  |_________||  |___|| |___||  |________/ /
   \________/     \___\  \___\   \_________\   \___\  \___\   \_______\/


                an Addon Package for Allegro by Sven Sandberg


Example program that shows the `draw_sphere()' function, that draws an
enlightened sphere.

*/
#include <allegro.h>
#include <salad.h>


//A struct holding the position and size of a sphere. The position tells
//where the centre of the sphere is.
typedef struct {
	int x,y,z;
	int radius;
}TSPHERE;


//Variables containing the position and size of the sphere and the light.
TSPHERE sphere, light;
//Variables containing the last position for the sphere and the light.
TSPHERE last_sphere, last_light;


//The size of screen, and the card. These are automatically initialised by
//`gfx_mode_select()', that is called in `init()'.
int card, screen_width, screen_height;



/*************
****      ****
**** init ****
****      ****
**************
Installs Allegro and Salad, lets the user select graphics mode and inits
some variables.
*/
void init(void)
{
//Loop counter for palette init.
int i;
//The palette.
PALETTE pal;

//Install a lot of things.
allegro_init();
install_timer();
salad_init();
if(install_mouse() == -1)
	exitfail("Unable to install mouse!");
if(install_keyboard())
	exitfail("Unable to install keyboard!");

//Let user select graphics mode.
gfxmode(320, 200);
for(i=0;i<256;i++)
	pal[i]=orange;
pal[0]=orangelight;
pal[255]=black;
pal[16]=white;
set_palette(pal);
if(!gfx_mode_select(&card, &screen_width, &screen_height))
	//Exit if user chose "Cancel".
	exit(0);
if(set_gfx_mode(card, screen_width, screen_height, 0, 0))
	exitfail("Unable to set graphics mode!");

//Init position of sphere.
sphere.radius=SCREEN_H/8;
sphere.x = SCREEN_W / 2;
sphere.y = SCREEN_H / 2;
sphere.z = 0;//z is not used anyway.

//Init position of light.
light.radius = 4;
light.x = SCREEN_W / 2;
light.y = SCREEN_H / 2;
light.z = SCREEN_W / 4;

//Set limits for mouse.
set_mouse_range(sphere.radius, sphere.radius,
 SCREEN_W - sphere.radius, SCREEN_H - sphere.radius);

//Fill palette with colors ranging from orange of 25% brightness to bright
//white.
pal[0]=black;
for(i=1; i<256; i++)
	pal[i] = luminate(orange, ((i + 128) * max_luminance) / (256+128));
set_palette(pal);
}





/*************
****      ****
**** play ****
****      ****
**************
Does the game.
*/
int play(void)
{
//The bitmap that the sphere is drawn to.
BITMAP *bmp;
//Frame counter.
int numframes = 0;
//Does the mouse control the sphere or the light? Initial state=light.
enum{we_are_moving_sphere, we_are_moving_light}what_are_we_moving =
 we_are_moving_sphere;
//Last state of mouse buttons. Initial state = no button pressed.
int last_mouse_b = 0;

//Create the bitmap.
bmp = create_bitmap(sphere.radius * 2 + 1, sphere.radius * 2 + 1);
clear(bmp);
clear(screen);

while(!key[KEY_ESC]){
	//Frame counter.
	numframes++;

	//Update positions according to mouse.
	if(what_are_we_moving==we_are_moving_sphere){
		sphere.x=mouse_x;
		sphere.y=mouse_y;
	}else{
		light.x=mouse_x;
		light.y=mouse_y;
	}

	/* Draw a new sphere to a memory bitmap.
		Parameters (in the same layout as the call below):
		bitmap, sphere x, sphere y, sphere radius,
		light x, light y,
		light z,
		first color index, last color index, unenlightened color.
	*/
	draw_sphere(bmp, sphere.radius, sphere.radius, sphere.radius,
	 light.x - sphere.x + sphere.radius, light.y - sphere.y + sphere.radius,
	 light.z,
	 1,255,1);

	//We only clear old sphere and light if their position has changed. This
	//avoids some of the flickering.
	if((last_light.x != light.x) || (last_light.y != light.y) ||
	 (last_sphere.x != sphere.x) || (last_sphere.y != sphere.y)){
		//Clear old light.
		rel_rectfill(screen,
		 last_light.x - light.radius, last_light.y - light.radius,
		 light.radius * 2 + 1, light.radius * 2 + 1, 0);
		//Clear old sphere.
		rectfill(screen,
		 last_sphere.x - sphere.radius, last_sphere.y - sphere.radius,
		 last_sphere.x + sphere.radius, last_sphere.y + sphere.radius,0);
	}

	//Draw new sphere.
	position_blit(bmp, screen,
	 sphere.x - sphere.radius, sphere.y - sphere.radius);
	//Draw new light.
	circlefill(screen, light.x, light.y, light.radius, 128);

	//Check if mouse button has been pressed. If it has, mouse should change
	//from the sphere to the light or vice versa.
	if(mouse_b&&!last_mouse_b){
		if(what_are_we_moving==we_are_moving_sphere){
			what_are_we_moving=we_are_moving_light;
			position_mouse(light.x,light.y);
		}else{
			what_are_we_moving=we_are_moving_sphere;
			position_mouse(sphere.x,sphere.y);
		}
	}

	//Update last positions.
	last_light=light;
	last_sphere=sphere;
	last_mouse_b=mouse_b;
}

//Clean up.
destroy_bitmap(bmp);

//Return frame count.
return numframes;
}





/*************
****      ****
**** main ****
****      ****
**************
Calls `init()' to initialise things, then calls `play()' to play the game,
and finally prints a speed report.
*/
int main(int argc, char *argv[])
{
//Frame and time counter.
int num_frames,milliseconds;

//Sets up some things.
init();

//Start stop-watch.
timer_start(1000);
//Play the game. `play()' returns number of frames.
num_frames=play();
//Stop stop-watch and get current time.
milliseconds=timer_stop();

allegro_exit();
setscreenheight(50);
//Print results.
printf(
"Salad Example\n"
"Example of the function `draw_sphere()' in Salad by Sven Sandberg.\n"
"(Don't worry - the flickering is not caused by the sphere function. You could\n"
"easily remove it using `vsync()', but then the frame counter would not be\n"
"correct\n"
"\n"
"Function syntax:\n"
"   draw_sphere(                                  //Function name.\n"
"    bmp,                                         //The bitmap.\n"
"    sphere_x, sphere_y, sphere_radius,           //Position of the sphere.\n"
"    light_x, light_y, light_z,                   //Position of the light.\n"
"    first_color, last_color, darkness_color);    //Color indexes to use.\n"
"\n"
"Results:\n"
"Sphere size:       %dx%d pixels\n"
"Screen size:       %dx%d pixels\n"
"Time:              %f seconds\n"
"Number of frames:  %d frames\n"
"Frames per second: %f fps\n"
"Seconds per frame: %f spf\n",
 sphere.radius * 2 + 1, sphere.radius * 2 + 1,
 screen_width, screen_height,
 (double)milliseconds/1000.0,
 num_frames,
 (double)num_frames/((double)milliseconds/1000.0),
 ((double)milliseconds/1000.0)/(double)num_frames);

return 0;
}
