/*
 * DZcomm : serial communication add-on for Allegro.
 * Copyright (c) 1997 Dim Zegebart, Moscow Russia.
 * zager@post.comstar.ru
 * version : 0.6
 * RTS/CTS flow control and IRQ8-IRQ15 fixed by Salvador Eduardo Tropea (SET)
 * e-mail: salvador@inti.gov.ar or set-soft@usa.net
 * file : dzcomm.c 
 */

#include <stdio.h>
#include <sys/movedata.h>
#include <stdlib.h>
#include <dos.h>
#include <conio.h>
#include <string.h>
#include <io.h>
#include <ctype.h>

#include "palantir.h"
#include "internal.h"

#define com_(N) device *com##N

com_(1);
com_(2);
com_(3);
com_(4);
com_(5);
com_(6);
com_(7);
com_(8);

#define com_wrapper_(N) int com##N##_wrapper(void) \
                        { int tmp; \
                          DISABLE();\
                          lwp_disable; \
                          dz_comm_port_interrupt_handler(com##N##); \
                          lwp_enable; \
                          return(0); \
                        } \
                        END_OF_FUNCTION(com##N##_wrapper);

com_wrapper_(1);
com_wrapper_(2);
com_wrapper_(3);
com_wrapper_(4);
com_wrapper_(5);
com_wrapper_(6);
com_wrapper_(7);
com_wrapper_(8);

static comm_port *irq_bot_com_port[16]={0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};

int (*com_wrapper[8])();
device **com_port[8];

//#include "d:/djgpp/work/xfault/xfault.h"

char szDZCommErr[50];
inline void dz_make_comm_err(char *szErr) {strcpy(szDZCommErr,szErr);}
//comm port functions
inline void comm_port_rdr(comm_port *port);
inline void comm_port_trx(comm_port *port);
inline void comm_port_rtmout(comm_port *port);
inline void interrupt_on(comm_port *port,byte _interrupt_);
inline void interrupt_off(comm_port *port,byte _interrupt_);
void comm_port_fifo_init(comm_port *port);
void comm_port_uninstall(comm_port *port);

#ifdef __DEVICE__
void comm_empty_handler(device *dev);
// #ifdef __LWP__
//void commd (void);
//lwp *comm_app;
// #endif
#endif

//const byte THREOFF=0xfd;
const byte THREINT=0x02;
const byte RDAINT=0x04;


#ifdef __LWP__
int port_kill_hnd(a_message code,int p1,int p2)
{ //int tmp; //variable used by lwp_enable(disable) macros
  dz_app *app=cur_app;

  switch(code)
   { case APP_START:
      break;
     case APP_STOP:
      if (app!=NULL&&(device*)app->app_usrdata!=NULL)
       { comm_port_delete((device*)app->app_usrdata);
       }
//      printf("Comm\n");
      return(0);
      break;
     default:
      break;
   }

 return(1);
}
#endif

//-------------------- COMM_PORT_FIFO_INIT ----------------------

void comm_port_fifo_init(comm_port *port)
{
//setting up FIFO if possible
  int c=inportb(port->SCR);
  outportb(port->SCR,0x55);
  if(inportb(port->SCR)==0x55)  // greater then 8250
   { outportb(port->SCR,c);
     outportb(port->FCR,0x87);//try to enable FIFO with 8 byte trigger level
     if ((c=inportb(port->IIR)&0xc0)!=0xc0) //If not 16550A
       outportb(port->FCR,c&0xfe); //Disable FIFO
//     else printf("\nFIFO OK\n");
   }
}

//-------------------------- PRINT BIN --------------------------

void print_bin_s(int c,char *s)
{ int i,j;
  int mask=0x01;
  sprintf (s,"%c : ",c);
  for (i=0;i<4;i++)
   { for(j=0;j<8;j++)
      if (c&(mask<<(j+i*8))) strcat(s,"1");
      else strcat(s,"0");
      strcat(s,".");
   }
}

//------------------------- INTERRUPT ON -----------------------------

inline void interrupt_on(comm_port *port,byte _interrupt_)
{ byte i;
  i=inportb(port->IER);
  if (!(i&_interrupt_))
   { //putch('x');
     outportb(port->IER,i|_interrupt_);
   }

}
END_OF_FUNCTION(interrupt_on);

//------------------------- INTERRUPT OFF -----------------------------

inline void interrupt_off(comm_port *port,byte _interrupt_)
{ byte i;
  i=inportb(port->IER);
  if (i&_interrupt_) outportb(port->IER,i&~_interrupt_);
}
END_OF_FUNCTION(interrupt_off);

//------------------------- COMM PORT INIT --------------------------

device *comm_port_init(comm com)
{ comm_port *port;
  int tmp;
  device *dev;
  odlist *list;
  szDZCommErr[0]=0;

  if (com>7||com<0)
    { dz_make_comm_err("Comm number out of range (0-7) !");
      return(NULL);
    }

  dev=device_new_("",4096*3,4096*3,1,1);
  if (dev==NULL)
    { dz_make_comm_err("Out of memory !");
      return(NULL);
    }

  lwp_disable;
  port=(comm_port*)malloc(sizeof(comm_port));
  lwp_enable;

  if (port==NULL)
   { dz_make_comm_err("Out of memory !");
     if (dev!=NULL) device_delete(dev);
     return(NULL);
   }

  port->InBuf=dev->read_queue;

//  (port->InBuf)->empty_handler=comm_empty_handler;
  port->OutBuf=dev->write_queue;
  dev->device_empty_handler=comm_empty_handler;
  if ((list=odlist_new())==NULL)
   { dz_make_comm_err("Out of memory !");
     return(NULL);
   }
 if ((port->lwp_OutBuf=queue_new((dev->write_queue)->size))==NULL)
  { dz_make_comm_err("Out of memory !");
    return(NULL);
  }
  dev->clients_write_queue=port->lwp_OutBuf;
//  dev->read_queue->fill_level=2048;
//  dev->write_queue->fill_level=2048;
  port->nPort=0; //just dumb number
  //getting comport base address
  //0x400+com*2 - memory address where stored default comm base address
  dosmemget(0x400+com*2,2,&port->nPort);
  if (port->nPort==0)
   { //dz_make_comm_err("BIOS data not found !");
     //return(NULL);
     if (com==_com1||com==_com3||com==_com5||com==_com7) port->nPort=0x3f8;
     else port->nPort=0x2f8; //com2 or com4
   }

//set default values (you may change it later)

  if (com==_com1||com==_com3||com==_com5||com==_com7) port->nIRQ=4;
  else port->nIRQ=3; //com2 or com4

  strcpy(dev->szName,"COM ");
  dev->szName[3]=com+49;
  port->installed=NO;
  port->nComm=com;
  port->nBaud=_2400;
  port->nData=BITS_8;
  port->nStop=STOP_1;
  port->nParity=NO_PARITY;
  port->comm_handler=com_wrapper[com];
  port->control_type=XON_XOFF;
  port->msr_handler=NULL; //#d#pointer to modem status register handler.
  port->lsr_handler=NULL; //#d#pointer to line status register handler.
  port->in_cnt=0;
  port->out_cnt=0;

  dev->device_user_data=port;
  dev->read_code=H_COMM_IN;
  dev->write_code=H_COMM_OUT;
  port->dev=dev;
  _go32_dpmi_lock_data(dev,sizeof(device));
  _go32_dpmi_lock_data(port,sizeof(comm_port));
  *(com_port[com])=dev;
  return(dev);
}

//----------------------- COMM PORT INSTALL HANDLER ------------------
//modifyed by Neil Townsend to support IRQ sharing
int comm_port_install_handler(device *dev)
{ comm_port *port=dev->device_user_data;

  if (port==NULL)
   { dz_make_comm_err("Set up comm parametrs first ! Call comm_port_new() first.");
     return(0);
   }
  if (port->installed==YES)
   { dz_make_comm_err("Comm already installed !");
     return(0);
   }
  if (port->nIRQ>15)
   { dz_make_comm_err("IRQ number out of range ! Must be 0 ... 15 .");
     return(0);
   }
  if (port->comm_handler==NULL)
   { dz_make_comm_err("Specify comm's handler (see 'comm_handler' member's description) !");
     return(0);
   }
  if (port->nBaud==0)
   { dz_make_comm_err("Invalid baud rate !");
     return(0);
   }

  switch(port->control_type)
   { case XON_XOFF :
       port->xon=0;
       port->xoff=0;
       port->xonxoff_rcvd=XON_RCVD;
       port->xonxoff_send=XON_SENT;
       break;
     case RTS_CTS :
       port->cts=CTS_ON;
       port->rts=RTS_ON; //fixed by SET (was RTS_OFF)
       break;
     default :
       break;
   }

  port->THR=port->nPort;
  port->RDR=port->nPort;
  port->BRDL=port->nPort;
  port->BRDH=1+port->nPort;
  port->IER=1+port->nPort;
  port->IIR=2+port->nPort;
  port->FCR=2+port->nPort;
  port->LCR=3+port->nPort;
  port->MCR=4+port->nPort;
  port->LSR=5+port->nPort;
  port->MSR=6+port->nPort;
  port->SCR=7+port->nPort;
  { byte i;
    for(i=0;i<8;i++) outportb(port->RDR+i,0);
  }

  comm_port_fifo_init(port);

  { unsigned char temp;
    do
     { temp=inportb(port->RDR);
       temp=inportb(port->LSR);
       temp=inportb(port->MSR);
       temp=inportb(port->IIR);
     }
    while(!(temp & 1));
  }

//set communication parametrs
//MOVED forwards so done before interupts enabled
  { int divisor=115200/port->nBaud;
    byte divlow,divhigh;
    byte comm_param=port->nData|(port->nStop<<2)|(port->nParity<<3);
    // Set Port Toggle to BRDL/BRDH registers
    outportb(port->LCR,comm_param|0x80);
    divlow=divisor&0x000000ff;
    divhigh=(divisor>>8)&0x000000ff;
    outportb(port->BRDL,divlow); // Set Baud Rate
    outportb(port->BRDH,divhigh);
    outportb(port->LCR,comm_param&0x7F); // Set LCR and Port Toggle
  }
  
//set interrupt parametrs
  { const byte IMR_8259_0=0x21;
    const byte IMR_8259_1=0xa1;
    const byte ISR_8259_0=0x20;
    const byte ISR_8259_1=0xa0;
    const byte IERALL = 0x0f; //mask for all communications interrupts
    const byte MCRALL = 0x0f; //DTR,RTS,OUT1=OUT2=1 is ON
    //calculating mask for 8259 controller's IMR
    //and number of interrupt hadler for given irq level
    if (port->nIRQ<=7) //if 0<=irq<=7 first IMR address used
     { port->interrupt_enable_mask=~(0x01<<port->nIRQ);
       port->nIRQVector=port->nIRQ+8;
       port->IMR_8259=IMR_8259_0;
       port->ISR_8259=ISR_8259_0;
     }
    else
     { port->interrupt_enable_mask=~(0x01<<(port->nIRQ%8));
       port->nIRQVector=0x70+(port->nIRQ-8);
       port->IMR_8259=IMR_8259_1;
       port->ISR_8259=ISR_8259_1;
     }

    port->next_port = NULL;
    port->last_port = irq_bot_com_port[port->nIRQ];
    if (irq_bot_com_port[port->nIRQ]!=NULL) irq_bot_com_port[port->nIRQ]->next_port=port;
    irq_bot_com_port[port->nIRQ]=port;
    DISABLE();
    if (port->last_port == NULL)
     { // DZ orig lines in here
       _install_irq(port->nIRQVector,port->comm_handler);
       //enable interrupt port->nIRQ level
       outportb(port->IMR_8259,inportb(port->IMR_8259)&
                port->interrupt_enable_mask);
     }
       
    // DZ again:
    outportb(port->MCR,MCRALL); //setup modem
    outportb(port->IER,IERALL); //enable all communication's interrupts
    // to here
    ENABLE();
  }

  #ifdef __LWP__
  { lwp *commr_app;
    lwp *commw_app;
    dz_app *port_kill;

    commr_app=lwp_spawn(hwrd,4096,1,NULL,(int*)dev);
    commw_app=lwp_spawn(hwwd,4096,1,NULL,(int*)dev);
    if (commr_app==NULL||commw_app==NULL) return(0);
//    lwp_disable;
    port_kill=dz_app_new("Port kill hnd",4096,1,port_kill_hnd);
    if (port_kill==NULL) return(0);
    port_kill->app_usrdata=(int*)dev;
//    lwp_enable;
  }
  #endif

  port->installed=YES;
  return(1);
}
/*
//------------------ COMM PORT INSTALL HANDLER ------------------
int comm_port_install_handler(device *dev)
{ int tmp;
  comm_port *port=dev->device_user_data;;
  if (dev==NULL)
   { dz_make_comm_err("Set up device parametrs first ! Call device_new() first.");
     return(0);
   }

  if (port==NULL)
   { dz_make_comm_err("Set up comm parametrs first ! Call comm_port_new() first.");
     return(0);
   }
  if (port->installed==YES)
   { dz_make_comm_err("Comm already installed !");
     return(0);
   }
  if (port->nIRQ>15)
   { dz_make_comm_err("IRQ number out of range ! Must be 0 ... 15 .");
     return(0);
   }
  if (port->comm_handler==NULL)
   { dz_make_comm_err("Specify comm's handler (see 'comm_handler' member's description) !");
     return(0);
   }
  if (port->nBaud==0)
   { dz_make_comm_err("Invalid boud rate !");
     return(0);
   }

  switch(port->control_type)
   { case XON_XOFF :
       port->xon=0;
       port->xoff=0;
       port->xonxoff_rcvd=XON_RCVD;
       port->xonxoff_send=XON_SENT;
       break;
     case RTS_CTS :
       port->cts=CTS_ON;
       port->rts=RTS_ON; //fixed by SET (was RTS_OFF)
     default :
       break;
   }

  port->THR=port->nPort;
  port->RDR=port->nPort;
  port->BRDL=port->nPort;
  port->BRDH=1+port->nPort;
  port->IER=1+port->nPort;
  port->IIR=2+port->nPort;
  port->FCR=2+port->nPort;
  port->LCR=3+port->nPort;
  port->MCR=4+port->nPort;
  port->LSR=5+port->nPort;
  port->MSR=6+port->nPort;
  port->SCR=7+port->nPort;
  { byte i;
    for(i=0;i<8;i++) outportb(port->RDR+i,0);
  }

  comm_port_fifo_init(port);

  //set interrupt parametrs
  { const byte IMR_8259_0=0x21;
    const byte IMR_8259_1=0xa1;
    const byte ISR_8259_0=0x20;
    const byte ISR_8259_1=0xa0;
    const byte IERALL=0x0f; //mask for all communications interrupts
    const byte MCRALL = 0x0f; //DTR,RTS,OUT1=OUT2=1 is ON
    //calculating mask for 8259 controller's IMR
    //and number of interrupt hadler for given irq level
    if (port->nIRQ<=7) //if 0<=irq<=7 first IMR address used
     { port->interrupt_enable_mask=~(0x01<<port->nIRQ);
       port->nIRQVector=port->nIRQ+8;
       port->IMR_8259=IMR_8259_0;
       port->ISR_8259=ISR_8259_0;
     }
    else
     { port->interrupt_enable_mask=~(0x01<<(port->nIRQ%8));
       port->nIRQVector=0x70+(port->nIRQ-8);
       port->IMR_8259=IMR_8259_1;
       port->ISR_8259=ISR_8259_1;
     }

    #ifdef __LWP__
    { lwp *commr_app;
      lwp *commw_app;
      dz_app *port_kill;

      commr_app=lwp_spawn(hwrd,4096,1,NULL,(int*)dev);
      commw_app=lwp_spawn(hwwd,4096,1,NULL,(int*)dev);
      if (commr_app==NULL||commw_app==NULL) return(0);
      lwp_disable;
      port_kill=dz_app_new("Port kill hnd",4096,1,port_kill_hnd);
      if (port_kill==NULL) return(0);
      port_kill->app_usrdata=(int*)dev;
      lwp_enable;
    }
    #endif

    lwp_disable;
    disable();
    _install_irq(port->nIRQVector,port->comm_handler);
    //enable interrupt port->nIRQ level
    outportb(port->IMR_8259,inportb(port->IMR_8259)&
             port->interrupt_enable_mask);

    outportb(port->MCR,MCRALL); //setup modem
    outportb(port->IER,IERALL); //enable all communication's interrupts
    lwp_enable;
    enable();

    //set communication parametrs
    { int divisor=115200/port->nBaud;
      byte divlow,divhigh;
      byte comm_param=port->nData|(port->nStop<<2)|(port->nParity<<3);
      // Set Port Toggle to BRDL/BRDH registers
      outportb(port->LCR,comm_param|0x80);
      divlow=divisor&0x000000ff;
      divhigh=(divisor>>8)&0x000000ff;
      outportb(port->BRDL,divlow); // Set Baud Rate
      outportb(port->BRDH,divhigh);
      outportb(port->LCR,comm_param&0x7F); // Set LCR and Port Toggle
    }
  }
  port->installed=YES;
//  if (inportb(port->MSR)&0x10) port->cts=CTS_ON;
//  else port->cts=CTS_OFF;
  return(1);
}
*/

//------------------ DZ COMM PORT INTERRUPT HANDLER ---------------
// always return zero
inline int dz_comm_port_interrupt_handler(device *dev)
{ int int_id;
//  static int am_i_here=0;
  int c;
  comm_port *port=dev->device_user_data;

  while (1) //loop while !end of all interrupts
   {
     int_id=inportb(port->IIR); //get interrupt id

     //next loop added by Neil Townsend
     while ((int_id&0x01)==1) //no interrupts (left) to handle on this port
      { if (port->next_port==NULL)
         { outportb(0x20,0x20);
           if (port->nIRQ>7) //Thanks for SET for this fix
             outportb(0xA0,0x20); //now IRQ8-IRQ15 handled properly.

           return(0);
         }
        port=(comm_port*)port->next_port;
        int_id=inportb(port->IIR);
      }

      switch (int_id)
        {
          case 0xcc: //Character Timeout Indication N.Lohmann 6.3.98
                     // only possible if FiFo's are enabled.
//putch('T');
            comm_port_rtmout(port);
            break;
          case 0xc6: //FiFo LSINT
          case 0x06: //LSINT:
            c=inportb(port->LSR);
//            printf("L%d\n",c);
            if (port->lsr_handler!=NULL) port->lsr_handler(c);
            break;
          case 0xc4: //FiFo RDAINT
          case 0x04: //RDAINT
            comm_port_rdr(port);
            break;
          case 0xc2: //FiFo THREINT
          case 0x02: //THREINT
//            putch('o');
            comm_port_trx(port);
            break;
          case 0xc0: //FiFo MSINT
          case 0x00: //MSINT
            c=inportb(port->MSR);
//            printf("M%d\n",c);
//            putch('a');
            if (port->msr_handler!=NULL) port->msr_handler(c);
            if (port->control_type==RTS_CTS)
             { if (port->cts==CTS_OFF && c&0x10)
                { port->cts=CTS_ON;
                  if (port->rts==RTS_ON)
                    interrupt_on(port,THREINT);
                }
               else if (port->cts==CTS_ON && !c&0x10)
                  { port->cts=CTS_OFF;
//                  putch('x');
                    interrupt_off(port,THREINT);
                  }
             }

            break;
         }//end case
   } //end while 1

  return(0);
}
END_OF_FUNCTION(dz_comm_port_interrupt_handler);

//----------------------- COMM PORT UNINSTALL ---------------------
// New routine to allow uninstalling without deleting
// and to simplyfy _delete and _reinstall
// by Neil Townsend
void comm_port_uninstall(comm_port *port)
{ comm_port *i_port;

  if (port->installed==NO) return;
  i_port = irq_bot_com_port[port->nIRQ];

  DISABLE();
  if (i_port == port)
   { irq_bot_com_port[port->nIRQ] = (comm_port *) port->last_port;
     if (irq_bot_com_port[port->nIRQ]) irq_bot_com_port[port->nIRQ]->next_port = NULL;
     i_port = NULL;
   }
  else i_port = (comm_port *) i_port->last_port;
  while (i_port)
   { if (i_port == port)
      { if (i_port->last_port) ((comm_port *)i_port->last_port)->next_port = i_port->next_port;
        if (i_port->next_port) ((comm_port *)i_port->next_port)->last_port = i_port->last_port;
        i_port = NULL;
      }
     if (i_port) i_port = i_port->last_port;
   }

  if (irq_bot_com_port[port->nIRQ] == NULL) _remove_irq(port->nIRQVector);

  { const byte IEROFF=0x00;
    const byte MCROFF=0x00;
    outportb(port->IER,IEROFF);
    outportb(port->MCR,MCROFF);
    outportb(port->IMR_8259,inportb(port->IMR_8259)&~(port->interrupt_enable_mask));
  }
  ENABLE();
  port->installed = NO;

}

//----------------------- COMM PORT DELETE ----------------------------

void comm_port_delete(device *dev)
{
 comm_port *port=dev->device_user_data;

 if (port->installed==NO) return;

 comm_port_uninstall(port);
 free(port);

 #ifdef __DZ_DEBUG__
 printf("Port %s deinstalled\n",dev->szName);
 #endif
 return;

}

//------------------------- COMM PORT OUT -----------------------------

inline int comm_port_out(device *dev,byte c)
{ //int tmp;
  int i;

  comm_port *port=dev->device_user_data;
//  putch('O');
//  putch(c);

  i=queue_put(port->OutBuf,c);
  queue_put(port->lwp_OutBuf,c);
  switch (port->control_type)
   { case NO_CONTROL :
      interrupt_on(port,THREINT);
      break;
     case XON_XOFF : // XON/XOFF
      if (port->xonxoff_rcvd!=XOFF_RCVD)
        { //putch('X');
          interrupt_on(port,THREINT);
        }
      break;
     case RTS_CTS : // RTS_CTR
      if (port->cts==CTS_ON) //modem is 'clear to send'
       { if (port->rts==RTS_ON) //Fixed by SET (was RTS_OFF)
          { //putch(c);
            interrupt_on(port,THREINT);
          }
       }
      break;
     default:
      break;
   }
  return(i);
}

//------------------------- COMM PORT TEST -----------------------------

inline int comm_port_test(device *dev)
{ int c;
  comm_port *port=dev->device_user_data;

  switch (port->control_type)
   { case NO_CONTROL :
      break;
     case XON_XOFF : // XON/XOFF
      if (((port->InBuf)->tail<(port->InBuf)->fill_level)&&(port->xonxoff_send!=XON_SENT))
        { port->xon=1;
          interrupt_on(port,THREINT);
        }
      break;
     case RTS_CTS :
      if (((port->InBuf)->tail<(port->InBuf)->fill_level)&&(port->rts==RTS_OFF))
        { outportb(port->MCR,inportb(port->MCR)|0x02); //setting RTS
          port->rts=RTS_ON;
//          interrupt_on(port,THREINT);
        }
     default:
      break;
   }
  if (queue_empty(port->InBuf))
   {
     return(-1);
   }
  c=queue_get(port->InBuf);
  return(c);
}

//----------------------- COMM PORT STRING SEND -------------------------

void comm_port_string_send(device *port,char *s)
{
  int i;

  if (s==NULL) return;
  for (i=0;s[i]!=0;i++) comm_port_out(port,s[i]);
}

//---------------------- COMM PORT COMMAND SEND -------------------------

void comm_port_command_send(device *port,char *s)
{ //comm_port *port_=port->device_user_data;
  int i;
  int tmp;

  if (s==NULL) return;
  lwp_disable;
  for (i=0;s[i]!=0;i++) comm_port_out(port,s[i]);
  comm_port_out(port,13);
//  comm_port_out(port,'\n');
  lwp_enable;
}

/*
//------------------- COMM PORT BREAK SEND -----------------

#ifdef __DEVICE__
void comm_port_break_send(device *dev)
#else
void comm_port_break_send(comm_port *port)
#endif
{ int c;
  #ifdef __DEVICE__
  comm_port *port=dev->device_user_data;
  #endif
  disable();
  c=inportb(port->LCR);
  outportb(port->LCR,c|(0x1<<6));
//  outportb(port->LCR,c);
  enable();
}
*/

//------------------- COMM PORT HANG UP -----------------

void modem_hangup(device *dev)
{
//  comm_port_out(dev,'\e');
//  rest(3000);
//  lwp_sleep(2,0);
  comm_port_string_send(dev,"+++");
//  rest(3000);
  lwp_sleep(3,0);
  comm_port_string_send(dev,"ATH0\r");
  lwp_sleep(3,0);

}

//---------------------- COMM PORT HAND ---------------------------

void comm_port_hand(device *dev,int m)
{ int c;
  comm_port *port=dev->device_user_data;
  c=inportb(port->MCR);
  outportb(port->MCR,c&m);
}

inline int comm_port_send_xoff(device *dev)
{
  comm_port *port=dev->device_user_data;
  if (port->xonxoff_send!=XOFF_SENT)
   { port->xoff=1;
     interrupt_on(port,THREINT);
   }
  return(1);
}

inline int comm_port_send_xon(device *dev)
{
  comm_port *port=dev->device_user_data;
  if (port->xonxoff_send!=XON_SENT)
   { port->xon=1;
     interrupt_on(port,THREINT);
   }
  return(1);
}

//-------------------- COMM PORT LOAD SETTINGS ---------------------

int comm_port_load_settings(device *dev,char *ini_name)
{
  comm_port *port=cport_(dev->device_user_data);
  FILE *ini;
  int l,i;
  char *buf,*buf_end,*s,v[64];
  comm com=port->nComm;
  char com_name[7]={'[','c','o','m',com+49,']',0};
  char com_end[11]={'[','c','o','m',com+49,' ','e','n','d',']',0};

  if ((ini=fopen(ini_name,"rt"))==NULL) return(0);
  l=filelength(fileno(ini));
  if ((buf=(char*)alloca(l))==NULL)
    { fclose(ini);
      return(0);
    }

  l=fread(buf,1,l,ini);
  fclose(ini);
  buf[l]=0;
  strlwr(buf);

  buf=strstr(buf,com_name);
  if (buf==NULL) return -1;

  buf_end=strstr(buf,com_end);
  if (buf_end==NULL) return(-1);
  *buf_end=0;
//  printf("%s\n",s);

  //get boud rate
  if ((s=strstr(buf,"baud"))!=NULL)
   { if (not_commented(s)&&(s=strchr(s,'='))!=NULL)
      { s++;i=0;
        while ((*s!=';')&&(*s!='\n')&&(*s!=0)&&(i<9))
         if (!isspace(*s)) v[i++]=*s++;

        v[i]=0;
        if ((i=atoi(v))!=0) port->nBaud=i;
      }
   }

  //get irq number
  if ((s=strstr(buf,"irq"))!=NULL)
   { if (not_commented(s)&&(s=strchr(s,'='))!=NULL)
      { s++;i=0;
        while ((*s!=';')&&(*s!='\n')&&(*s!=0)&&(i<9))
         if (!isspace(*s)) v[i++]=*s++;

        v[i]=0;
        if ((i=atoi(v))!=0) port->nIRQ=i;
      }
   }

  //get comm address
  if ((s=strstr(buf,"address"))!=NULL)
   { if (not_commented(s)&&(s=strchr(s,'='))!=NULL)
      { s++;i=0;
        while ((*s!=';')&&(*s!='\n')&&(*s!=0)&&(i<9))
         if (!isspace(*s)) v[i++]=*s++;

        v[i]=0;
        if ((i=strtol(v,NULL,0))!=0) port->nPort=strtol(v,NULL,0);
      }
   }

  //get data bits
  if ((s=strstr(buf,"data"))!=NULL)
   { if (not_commented(s)&&(s=strchr(s,'='))!=NULL)
       { s++;i=0;
         while ((*s!=';')&&(*s!='\n')&&(*s!=0)&&(i<9))
          if (!isspace(*s)) v[i++]=*s++;

         v[i]=0;
         if ((i=atoi(v))!=0)
          switch (i)
           { case 8 :
              port->nData=BITS_8;
              break;
             case 7 :
              port->nData=BITS_7;
              break;
             case 6 :
              port->nData=BITS_6;
              break;
             case 5 :
              port->nData=BITS_5;
              break;
           }
       }
   }

  //get parity bits
  if ((s=strstr(buf,"parity"))!=NULL)
   { if (not_commented(s)&&(s=strchr(s,'='))!=NULL)
       { s++;i=0;
         while ((*s!=';')&&(*s!='\n')&&(*s!=0)&&(i<9))
          if (!isspace(*s)) v[i++]=*s++;
         v[i]=0;

         if (strcmp(v,"even")==0) port->nParity=EVEN_PARITY;
         else
         if (strcmp(v,"odd")==0) port->nParity=ODD_PARITY;
         else
         if (strcmp(v,"no")==0||strcmp(v,"none")==0) port->nParity=NO_PARITY;
       }
   }

  //get control type
  if ((s=strstr(buf,"control"))!=NULL)
   { if (not_commented(s)&&(s=strchr(s,'='))!=NULL)
       { s++;i=0;
         while ((*s!=';')&&(*s!='\n')&&(*s!=0)&&(i<9))
          if (!isspace(*s)) v[i++]=*s++;
         v[i]=0;

         if (strcmp(v,"xon_xoff")==0||strcmp(v,"xon/xoff")==0) port->control_type=XON_XOFF;
         else
          if (strcmp(v,"no")==0||strcmp(v,"none")==0) port->control_type=NO_CONTROL;
          else
           if (strcmp(v,"rts_cts")==0||strcmp(v,"rts/cts")==0) port->control_type=RTS_CTS;
       }
   }

  if ((s=strstr(buf,"stop"))!=NULL) //get boud rate
   { if (not_commented(s)&&(s=strchr(s,'='))!=NULL)
       { s++;i=0;
         while ((*s!=';')&&(*s!='\n')&&(*s!=0)&&(i<9))
          if (!isspace(*s)) v[i++]=*s++;

         v[i]=0;
         if ((i=atoi(v))!=0)
          switch (i)
           { case 1 :
              port->nStop=STOP_1;
              break;
             case 2 :
              port->nStop=STOP_2;
              break;
           }
       }
   }

  if ((s=strstr(buf,"name"))!=NULL)
   { if (not_commented(s)&&(s=strchr(s,'='))!=NULL)
      { s++;i=0;
        while ((*s!=';')&&(*s!='\n')&&(*s!=0)&&(i<64))
         v[i++]=*s++;

        v[i]=0;
        strcpy((port->dev)->szName,v);
      }
   }

  return(1);
}

//----------------------- COMM PORT RTMOUT  ------------------------
//Nils Lohmann 8.3.98
//Time Out Indication, service routine  triggered after 4 Char-times
//if there is at least one Byte in FiFo and no read-Cykle happened.

inline void comm_port_rtmout(comm_port *port)
{
  comm_port_rdr(port);
}
END_OF_FUNCTION(comm_port_rtmout);

//----------------------- COMM PORT RDR ------------------------

inline void comm_port_rdr(comm_port *port)
{ int c=inportb(port->RDR);
  int n;
//  putch(c);
  #ifdef __DZ_DEBUG__
  if ((port->InBuf)->status==LWP_LOCKED) putch('L');
  #endif
  switch (port->control_type)
   { case NO_CONTROL :
      queue_put(port->InBuf,c);
      port->in_cnt++;
      break;
     case XON_XOFF : // XON/XOFF
      if (c==XON_ASCII)
       { port->xonxoff_rcvd=XON_RCVD;
         //queue_put(port->InBuf,c);
         interrupt_on(port,THREINT);
         return;
       }
      if (c==XOFF_ASCII)
       { port->xonxoff_rcvd=XOFF_RCVD;
         interrupt_off(port,THREINT);
         return;
       }
//      putch('R');
//      putch(c);
      n=queue_put(port->InBuf,c);
//      n=0;
      port->in_cnt++;
      if (n||port->dev->client_apps_status||port->dev->stop_request)//buffer is near full
       { //outportb(port->THR,XOFF_ASCII);
//         putch('d');
         if (port->xonxoff_send!=XOFF_SENT)
          { port->xoff=1;
            interrupt_on(port,THREINT);
//            putch('D');
          }
       }
      break;
     case RTS_CTS:
      n=queue_put(port->InBuf,c);
      port->in_cnt++;
//      putch(c);
      if (n||port->dev->client_apps_status) //buffer is near full
       { outportb(port->MCR,inportb(port->MCR)&(~0x02)); //dropping RTS
//         putch('F');
         port->rts=RTS_OFF;
       }
      break;
     default :
      break;
   }
}
END_OF_FUNCTION(comm_port_rdr)

//----------------------- COMM PORT TRX ------------------------

inline void comm_port_trx(comm_port *port)
{ int c;
  int tmp;

  switch (port->control_type)
   { case NO_CONTROL :
      if ((port->OutBuf)->empt==EMPTY)//queue empty, nothing to send
       { interrupt_off(port,THREINT);
         return;
       }
      c=queue_get(port->OutBuf);
      port->out_cnt++;
      outportb(port->THR,c);
      break;
     case XON_XOFF : // XON/XOFF
      if (port->xoff==1)
       { outportb(port->THR,XOFF_ASCII);
         port->xoff=0;
         port->xonxoff_send=XOFF_SENT;
       }
      else if (port->xon==1)
       { outportb(port->THR,XON_ASCII);
         port->xon=0;
         port->xonxoff_send=XON_SENT;
       }
      if ((port->OutBuf)->empt==EMPTY)//queue empty, nothing to send
       { interrupt_off(port,THREINT);
         return;
       }
      lwp_disable;
      #ifdef __DZ_DEBUG__
      if ((port->OutBuf)->status==LWP_LOCKED) putch('L');
      #endif
      c=queue_get(port->OutBuf);
      outportb(port->THR,c);
      lwp_enable;
      port->out_cnt++;
      break;
     case RTS_CTS:
      if (port->cts==CTS_OFF)
       { return;
       }
//      if (port->rts==RTS_OFF)
//      { port->rts=RTS_ON;
//        outportb(port->MCR,inportb(port->MCR)|0x02); //setting RTS
//       }
      if ((port->OutBuf)->empt==EMPTY)//queue empty, nothig to send
       { //outportb(port->MCR,inportb(port->MCR)&(~0x02)); //dropping RTS
         //port->rts=RTS_OFF;
         interrupt_off(port,THREINT);
         return;
       }

      c=queue_get(port->OutBuf);
      outportb(port->THR,c);
      port->out_cnt++;
      break;
     default :
      break;
   }
}
END_OF_FUNCTION(comm_port_trx);

#ifdef __DEVICE__
//---------------------- COMM EMPTY HANDLER -----------------------

void comm_empty_handler(device *dev)
{
  comm_port *port=cdport_(dev);
//  if (dev->status==LWP_LOCKED) return;
  if (dev->client_apps_status||dev->stop_request) return;

  switch (port->control_type)
   { case NO_CONTROL :
      break;
     case XON_XOFF : // XON/XOFF
      if (port->xonxoff_send!=XON_SENT)
        { port->xon=1;
        }
      interrupt_on(port,THREINT);
      break;
     case RTS_CTS:
      if (port->rts==RTS_OFF)
        { outportb(port->MCR,inportb(port->MCR)|0x02);
          port->rts=RTS_ON;
          interrupt_on(port,THREINT);
        }
      break;
     default:
      break;
   }
//  return(0);
}
#endif __DEVICE__

//------------------------- COMM PORT INIT -----------------------------

void dzcomm_init(void)
{
  LOCK_FUNCTION(dz_comm_port_interrupt_handler);
  LOCK_FUNCTION(comm_port_rdr);
  LOCK_FUNCTION(comm_port_trx);
  LOCK_FUNCTION(comm_port_rtmout);
  LOCK_FUNCTION(interrupt_on);
  LOCK_FUNCTION(interrupt_off);
  LOCK_FUNCTION(com1_wrapper);
  LOCK_FUNCTION(com2_wrapper);
  LOCK_FUNCTION(com3_wrapper);
  LOCK_FUNCTION(com4_wrapper);
  LOCK_FUNCTION(com5_wrapper);
  LOCK_FUNCTION(com6_wrapper);
  LOCK_FUNCTION(com7_wrapper);
  LOCK_FUNCTION(com8_wrapper);
  LOCK_VARIABLE(com1);
  LOCK_VARIABLE(com2);
  LOCK_VARIABLE(com3);
  LOCK_VARIABLE(com4);
  LOCK_VARIABLE(com5);
  LOCK_VARIABLE(com6);
  LOCK_VARIABLE(com7);
  LOCK_VARIABLE(com8);
  com1=NULL;
  com2=NULL;
  com3=NULL;
  com4=NULL;
  com5=NULL;
  com6=NULL;
  com7=NULL;
  com8=NULL;
  com_wrapper[0]=com1_wrapper;
  com_wrapper[1]=com2_wrapper;
  com_wrapper[2]=com3_wrapper;
  com_wrapper[3]=com4_wrapper;
  com_wrapper[4]=com5_wrapper;
  com_wrapper[5]=com6_wrapper;
  com_wrapper[6]=com7_wrapper;
  com_wrapper[7]=com8_wrapper;
  com_port[0]=&com1;
  com_port[1]=&com2;
  com_port[2]=&com3;
  com_port[3]=&com4;
  com_port[4]=&com5;
  com_port[5]=&com6;
  com_port[6]=&com7;
  com_port[7]=&com8;

//  port_monitor=NULL;

}

//----------------------- COMM PORT REINSTALL ---------------------------

int comm_port_reinstall(device *dev)
{
  comm_port *port=dev->device_user_data;

  if (port->installed==NO) return(0);

  comm_port_uninstall(port);
  return (comm_port_install_handler(dev));

}
