/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.ranger.plugin.model;

import org.apache.ranger.authorization.utils.StringUtil;
import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;

import java.util.Map;

@JsonAutoDetect(fieldVisibility=JsonAutoDetect.Visibility.ANY)
@JsonInclude(JsonInclude.Include.NON_EMPTY)
@JsonIgnoreProperties(ignoreUnknown=true)
public class RangerServiceResource extends RangerBaseModelObject {
	private static final long serialVersionUID = 1L;

	private String                                         serviceName;
	private Map<String, RangerPolicy.RangerPolicyResource> resourceElements;
	private String                                         ownerUser;
	private String                                         resourceSignature;
	private Map<String, String>							   additionalInfo;

	public RangerServiceResource(String guid, String serviceName, Map<String, RangerPolicy.RangerPolicyResource> resourceElements, String resourceSignature, String ownerUser, Map<String, String> additionalInfo) {
		super();
		setGuid(guid);
		setServiceName(serviceName);
		setResourceElements(resourceElements);
		setResourceSignature(resourceSignature);
		setOwnerUser(ownerUser);
		setAdditionalInfo(additionalInfo);
	}
	public RangerServiceResource(String guid, String serviceName, Map<String, RangerPolicy.RangerPolicyResource> resourceElements, String resourceSignature, String ownerUser) {
		this(guid, serviceName, resourceElements, resourceSignature,ownerUser, null);
	}
	public RangerServiceResource(String guid, String serviceName, Map<String, RangerPolicy.RangerPolicyResource> resourceElements, String resourceSignature) {
		this(guid, serviceName, resourceElements, resourceSignature, null);

	}

	public RangerServiceResource(String guid, String serviceName, Map<String, RangerPolicy.RangerPolicyResource> resourceElements) {
		this(guid, serviceName, resourceElements, null, null);
	}
	public RangerServiceResource(String serviceName, Map<String, RangerPolicy.RangerPolicyResource> resourceElements) {
		this(null, serviceName, resourceElements, null, null);
	}

	public RangerServiceResource() {
		this(null, null, null, null, null);
	}

	public String getServiceName() { return serviceName; }

	public Map<String, RangerPolicy.RangerPolicyResource> getResourceElements() { return resourceElements; }

	public String getResourceSignature() {
		return resourceSignature;
	}

	public String getOwnerUser() {
		return ownerUser;
	}

	public Map<String, String> getAdditionalInfo() {
		return additionalInfo;
	}

	public void setServiceName(String serviceName) {
		this.serviceName = serviceName;
	}

	public void setResourceElements(Map<String, RangerPolicy.RangerPolicyResource> resource) {
		this.resourceElements = resource;
	}

	public void setResourceSignature(String resourceSignature) {
		this.resourceSignature = resourceSignature;
	}

	public void setOwnerUser(String ownerUser) {
		this.ownerUser = ownerUser;
	}

	public void setAdditionalInfo(Map<String, String> additionalInfo) {
		this.additionalInfo = additionalInfo;
	}

	public void dedupStrings(Map<String, String> strTbl) {
		super.dedupStrings(strTbl);

		serviceName      = StringUtil.dedupString(serviceName, strTbl);
		resourceElements = StringUtil.dedupStringsMapOfPolicyResource(resourceElements, strTbl);
		ownerUser        = StringUtil.dedupString(ownerUser, strTbl);
		additionalInfo   = StringUtil.dedupStringsMap(additionalInfo, strTbl);
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();

		toString(sb);

		return sb.toString();
	}

	public StringBuilder toString(StringBuilder sb) {

		sb.append("RangerServiceResource={ ");

		super.toString(sb);

		sb.append("guid={").append(getGuid()).append("} ");
		sb.append("serviceName={").append(serviceName).append("} ");

		sb.append("resourceElements={");
		if(resourceElements != null) {
			for(Map.Entry<String, RangerPolicy.RangerPolicyResource> e : resourceElements.entrySet()) {
				sb.append(e.getKey()).append("={");
				e.getValue().toString(sb);
				sb.append("} ");
			}
		}
		sb.append("} ");

		sb.append("ownerUser={").append(ownerUser).append("} ");

		sb.append("additionalInfo={");
		if(additionalInfo != null) {
			for(Map.Entry<String, String> e : additionalInfo.entrySet()) {
				sb.append(e.getKey()).append("={").append(e.getValue()).append("} ");
			}
		}
		sb.append("} ");

		sb.append("resourceSignature={").append(resourceSignature).append("} ");

		sb.append(" }");

		return sb;
	}
}

