/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.service.schema;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;
import com.google.protobuf.ByteString;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.validation.constraints.NotNull;
import org.apache.bookkeeper.client.AsyncCallback;
import org.apache.bookkeeper.client.BookKeeper;
import org.apache.bookkeeper.client.LedgerEntry;
import org.apache.bookkeeper.client.LedgerHandle;
import org.apache.bookkeeper.client.api.BKException;
import org.apache.bookkeeper.client.api.DigestType;
import org.apache.bookkeeper.mledger.impl.LedgerMetadataUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.pulsar.broker.PulsarService;
import org.apache.pulsar.broker.ServiceConfiguration;
import org.apache.pulsar.broker.service.schema.SchemaStorageFormat;
import org.apache.pulsar.broker.service.schema.exceptions.SchemaException;
import org.apache.pulsar.common.protocol.schema.SchemaStorage;
import org.apache.pulsar.common.protocol.schema.SchemaVersion;
import org.apache.pulsar.common.protocol.schema.StoredSchema;
import org.apache.pulsar.common.schema.LongSchemaVersion;
import org.apache.pulsar.common.util.FutureUtil;
import org.apache.pulsar.metadata.api.MetadataCache;
import org.apache.pulsar.metadata.api.MetadataSerde;
import org.apache.pulsar.metadata.api.MetadataStoreException;
import org.apache.pulsar.metadata.api.Stat;
import org.apache.pulsar.metadata.api.extended.MetadataStoreExtended;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BookkeeperSchemaStorage
implements SchemaStorage {
    private static final Logger log = LoggerFactory.getLogger(BookkeeperSchemaStorage.class);
    private static final String SchemaPath = "/schemas";
    private static final byte[] LedgerPassword = "".getBytes();
    private final MetadataStoreExtended store;
    private final PulsarService pulsar;
    private final MetadataCache<SchemaStorageFormat.SchemaLocator> locatorEntryCache;
    private final ServiceConfiguration config;
    private BookKeeper bookKeeper;
    private final ConcurrentMap<String, CompletableFuture<StoredSchema>> readSchemaOperations = new ConcurrentHashMap<String, CompletableFuture<StoredSchema>>();

    @VisibleForTesting
    BookkeeperSchemaStorage(PulsarService pulsar) {
        this.pulsar = pulsar;
        this.store = pulsar.getLocalMetadataStore();
        this.config = pulsar.getConfiguration();
        this.locatorEntryCache = this.store.getMetadataCache((MetadataSerde)new MetadataSerde<SchemaStorageFormat.SchemaLocator>(){

            public byte[] serialize(String path, SchemaStorageFormat.SchemaLocator value) {
                return value.toByteArray();
            }

            public SchemaStorageFormat.SchemaLocator deserialize(String path, byte[] content, Stat stat) throws IOException {
                return SchemaStorageFormat.SchemaLocator.parseFrom(content);
            }
        });
    }

    public void start() throws IOException {
        this.bookKeeper = this.pulsar.getBookKeeperClientFactory().create(this.pulsar.getConfiguration(), this.store, this.pulsar.getIoEventLoopGroup(), Optional.empty(), null).join();
    }

    public CompletableFuture<SchemaVersion> put(String key, byte[] value, byte[] hash) {
        return this.putSchema(key, value, hash).thenApply(LongSchemaVersion::new);
    }

    public CompletableFuture<SchemaVersion> put(String key, Function<CompletableFuture<List<CompletableFuture<StoredSchema>>>, CompletableFuture<Pair<byte[], byte[]>>> fn) {
        CompletableFuture<SchemaVersion> promise = new CompletableFuture<SchemaVersion>();
        this.put(key, fn, promise);
        return promise;
    }

    private void put(String key, Function<CompletableFuture<List<CompletableFuture<StoredSchema>>>, CompletableFuture<Pair<byte[], byte[]>>> fn, CompletableFuture<SchemaVersion> promise) {
        CompletableFuture<Pair<Optional<LocatorEntry>, List<CompletableFuture<StoredSchema>>>> schemasWithLocator = this.getAllWithLocator(key);
        ((CompletableFuture)schemasWithLocator.thenCompose(pair -> ((CompletableFuture)((CompletableFuture)fn.apply(CompletableFuture.completedFuture((List)pair.getRight()))).thenCompose(p -> {
            if (p == null) {
                return CompletableFuture.completedFuture(null);
            }
            return this.putSchema(key, (byte[])p.getLeft(), (byte[])p.getRight(), (Optional)pair.getLeft());
        })).thenApply(version -> version != null ? new LongSchemaVersion(version.longValue()) : null))).whenComplete((v, ex) -> {
            if (ex == null) {
                promise.complete((SchemaVersion)v);
            } else {
                Throwable cause = FutureUtil.unwrapCompletionException((Throwable)ex);
                if (cause instanceof MetadataStoreException.AlreadyExistsException || cause instanceof MetadataStoreException.BadVersionException) {
                    this.put(key, fn, promise);
                } else {
                    promise.completeExceptionally((Throwable)ex);
                }
            }
        });
    }

    public CompletableFuture<StoredSchema> get(String key, SchemaVersion version) {
        if (version == SchemaVersion.Latest) {
            return this.getSchema(key);
        }
        LongSchemaVersion longVersion = (LongSchemaVersion)version;
        return this.getSchema(key, longVersion.getVersion());
    }

    public CompletableFuture<List<CompletableFuture<StoredSchema>>> getAll(String key) {
        return this.getAllWithLocator(key).thenApply(Pair::getRight);
    }

    private CompletableFuture<Pair<Optional<LocatorEntry>, List<CompletableFuture<StoredSchema>>>> getAllWithLocator(String key) {
        return this.getLocator(key).thenApply(locator -> {
            if (log.isDebugEnabled()) {
                log.debug("[{}] Get all schemas - locator: {}", (Object)key, locator);
            }
            if (locator.isEmpty()) {
                return Pair.of((Object)locator, Collections.emptyList());
            }
            SchemaStorageFormat.SchemaLocator schemaLocator = ((LocatorEntry)locator.get()).locator;
            ArrayList list = new ArrayList();
            schemaLocator.getIndexList().forEach(indexEntry -> list.add(this.readSchemaEntry(indexEntry.getPosition()).thenApply(entry -> new StoredSchema(entry.getSchemaData().toByteArray(), (SchemaVersion)new LongSchemaVersion(indexEntry.getVersion())))));
            return Pair.of((Object)locator, list);
        });
    }

    CompletableFuture<Optional<LocatorEntry>> getLocator(String key) {
        return this.getSchemaLocator(BookkeeperSchemaStorage.getSchemaPath(key));
    }

    public List<Long> getSchemaLedgerList(String key) throws IOException {
        Optional<LocatorEntry> locatorEntry = null;
        try {
            locatorEntry = this.getLocator(key).get();
        }
        catch (Exception e) {
            log.warn("Failed to get list of schema-storage ledger for {}, the exception as follow: \n {}", (Object)key, (Object)(e instanceof ExecutionException ? e.getCause() : e));
            throw new IOException("Failed to get schema ledger for" + key);
        }
        LocatorEntry entry = locatorEntry.orElse(null);
        return entry != null ? entry.locator.getIndexList().stream().map(i -> i.getPosition().getLedgerId()).collect(Collectors.toList()) : null;
    }

    @VisibleForTesting
    BookKeeper getBookKeeper() {
        return this.bookKeeper;
    }

    public CompletableFuture<SchemaVersion> delete(String key, boolean forcefully) {
        return this.deleteSchema(key, forcefully).thenApply(version -> {
            if (version == null) {
                return null;
            }
            return new LongSchemaVersion(version.longValue());
        });
    }

    public CompletableFuture<SchemaVersion> delete(String key) {
        return this.delete(key, false);
    }

    @NotNull
    private CompletableFuture<StoredSchema> getSchema(String schemaId) {
        return this.readSchemaOperations.computeIfAbsent(schemaId, key -> {
            if (log.isDebugEnabled()) {
                log.debug("[{}] Fetching schema from store", (Object)schemaId);
            }
            return this.getSchemaLocator(BookkeeperSchemaStorage.getSchemaPath(schemaId)).thenCompose(locator -> {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Got schema locator {}", (Object)schemaId, locator);
                }
                if (!locator.isPresent()) {
                    return CompletableFuture.completedFuture(null);
                }
                SchemaStorageFormat.SchemaLocator schemaLocator = ((LocatorEntry)locator.get()).locator;
                return this.readSchemaEntry(schemaLocator.getInfo().getPosition()).thenApply(entry -> new StoredSchema(entry.getSchemaData().toByteArray(), (SchemaVersion)new LongSchemaVersion(schemaLocator.getInfo().getVersion())));
            });
        }).whenComplete((res, ex) -> {
            if (log.isDebugEnabled()) {
                log.debug("[{}] Get operation completed. res={} -- ex={}", new Object[]{schemaId, res, ex});
            }
            this.readSchemaOperations.remove(schemaId);
        });
    }

    public SchemaVersion versionFromBytes(byte[] version) {
        ByteBuffer bb = ByteBuffer.wrap(version);
        return new LongSchemaVersion(bb.getLong());
    }

    public void close() throws Exception {
        if (this.bookKeeper != null) {
            this.bookKeeper.close();
        }
    }

    @NotNull
    private CompletableFuture<StoredSchema> getSchema(String schemaId, long version) {
        if (log.isDebugEnabled()) {
            log.debug("[{}] Get schema - version: {}", (Object)schemaId, (Object)version);
        }
        return this.getSchemaLocator(BookkeeperSchemaStorage.getSchemaPath(schemaId)).thenCompose(locator -> {
            if (log.isDebugEnabled()) {
                log.debug("[{}] Get schema - version: {} - locator: {}", new Object[]{schemaId, version, locator});
            }
            if (!locator.isPresent()) {
                return CompletableFuture.completedFuture(null);
            }
            SchemaStorageFormat.SchemaLocator schemaLocator = ((LocatorEntry)locator.get()).locator;
            if (version > schemaLocator.getInfo().getVersion()) {
                return CompletableFuture.completedFuture(null);
            }
            return this.findSchemaEntryByVersion(schemaLocator.getIndexList(), version).thenApply(entry -> new StoredSchema(entry.getSchemaData().toByteArray(), (SchemaVersion)new LongSchemaVersion(version)));
        });
    }

    @NotNull
    private CompletableFuture<Long> putSchema(String schemaId, byte[] data, byte[] hash) {
        return this.getSchemaLocator(BookkeeperSchemaStorage.getSchemaPath(schemaId)).thenCompose(optLocatorEntry -> this.putSchema(schemaId, data, hash, (Optional<LocatorEntry>)optLocatorEntry));
    }

    private CompletableFuture<Long> putSchema(String schemaId, byte[] data, byte[] hash, Optional<LocatorEntry> optLocatorEntry) {
        if (optLocatorEntry.isPresent()) {
            SchemaStorageFormat.SchemaLocator locator = optLocatorEntry.get().locator;
            if (log.isDebugEnabled()) {
                log.debug("[{}] findSchemaEntryByHash - hash={}", (Object)schemaId, (Object)hash);
            }
            return this.readSchemaEntry(locator.getIndexList().get(0).getPosition()).thenCompose(schemaEntry -> this.addNewSchemaEntryToStore(schemaId, locator.getIndexList(), data).thenCompose(position -> this.updateSchemaLocator(schemaId, (LocatorEntry)optLocatorEntry.get(), (SchemaStorageFormat.PositionInfo)position, hash)));
        }
        return this.createNewSchema(schemaId, data, hash);
    }

    private CompletableFuture<Long> createNewSchema(String schemaId, byte[] data, byte[] hash) {
        SchemaStorageFormat.IndexEntry emptyIndex = SchemaStorageFormat.IndexEntry.newBuilder().setVersion(0L).setHash(ByteString.copyFrom((byte[])hash)).setPosition(SchemaStorageFormat.PositionInfo.newBuilder().setEntryId(-1L).setLedgerId(-1L)).build();
        return this.addNewSchemaEntryToStore(schemaId, Collections.singletonList(emptyIndex), data).thenCompose(position -> {
            SchemaStorageFormat.IndexEntry info = SchemaStorageFormat.IndexEntry.newBuilder().setVersion(0L).setPosition((SchemaStorageFormat.PositionInfo)position).setHash(ByteString.copyFrom((byte[])hash)).build();
            return this.createSchemaLocator(BookkeeperSchemaStorage.getSchemaPath(schemaId), SchemaStorageFormat.SchemaLocator.newBuilder().setInfo(info).addAllIndex(Lists.newArrayList((Object[])new SchemaStorageFormat.IndexEntry[]{info})).build()).thenApply(ignore -> 0L);
        });
    }

    @NotNull
    private CompletableFuture<Long> deleteSchema(String schemaId, boolean forcefully) {
        return (forcefully ? CompletableFuture.completedFuture(null) : BookkeeperSchemaStorage.ignoreUnrecoverableBKException(this.getSchema(schemaId))).thenCompose(schemaAndVersion -> {
            if (!forcefully && Objects.isNull(schemaAndVersion)) {
                return CompletableFuture.completedFuture(null);
            }
            long version = -1L;
            CompletableFuture future = new CompletableFuture();
            this.getLocator(schemaId).whenComplete((locator, ex) -> {
                if (ex != null) {
                    future.completeExceptionally((Throwable)ex);
                } else {
                    if (!locator.isPresent()) {
                        future.complete(null);
                        return;
                    }
                    List<SchemaStorageFormat.IndexEntry> indexEntryList = ((LocatorEntry)locator.get()).locator.getIndexList();
                    ArrayList deleteFutures = new ArrayList(indexEntryList.size());
                    indexEntryList.forEach(indexEntry -> {
                        long ledgerId = indexEntry.getPosition().getLedgerId();
                        CompletableFuture deleteFuture = new CompletableFuture();
                        deleteFutures.add(deleteFuture);
                        this.bookKeeper.asyncDeleteLedger(ledgerId, (rc, cnx) -> {
                            if (rc != 0) {
                                log.warn("Failed to delete ledger {} of {}: {}", new Object[]{ledgerId, schemaId, rc});
                            }
                            deleteFuture.complete(null);
                        }, null);
                    });
                    FutureUtil.waitForAll(deleteFutures).whenComplete((v, e) -> {
                        String path = BookkeeperSchemaStorage.getSchemaPath(schemaId);
                        ((CompletableFuture)this.store.delete(path, Optional.empty()).thenRun(() -> future.complete(-1L))).exceptionally(zkException -> {
                            if (zkException.getCause() instanceof MetadataStoreException.NotFoundException) {
                                if (log.isDebugEnabled()) {
                                    log.debug("No node for schema path: {}", (Object)path);
                                }
                                future.complete(null);
                            } else {
                                future.completeExceptionally((Throwable)zkException);
                            }
                            return null;
                        });
                    });
                }
            });
            return future;
        });
    }

    @NotNull
    private static String getSchemaPath(String schemaId) {
        return "/schemas/" + schemaId;
    }

    @NotNull
    private CompletableFuture<SchemaStorageFormat.PositionInfo> addNewSchemaEntryToStore(String schemaId, List<SchemaStorageFormat.IndexEntry> index, byte[] data) {
        SchemaStorageFormat.SchemaEntry schemaEntry = Functions.newSchemaEntry(index, data);
        return this.createLedger(schemaId).thenCompose(ledgerHandle -> {
            long ledgerId = ledgerHandle.getId();
            return this.addEntry((LedgerHandle)ledgerHandle, schemaEntry).thenApply(entryId -> {
                ledgerHandle.closeAsync();
                return Functions.newPositionInfo(ledgerId, entryId);
            });
        });
    }

    @NotNull
    private CompletableFuture<Long> updateSchemaLocator(final String schemaId, LocatorEntry locatorEntry, final SchemaStorageFormat.PositionInfo position, byte[] hash) {
        long nextVersion = locatorEntry.locator.getInfo().getVersion() + 1L;
        SchemaStorageFormat.SchemaLocator locator = locatorEntry.locator;
        SchemaStorageFormat.IndexEntry info = SchemaStorageFormat.IndexEntry.newBuilder().setVersion(nextVersion).setPosition(position).setHash(ByteString.copyFrom((byte[])hash)).build();
        ArrayList<SchemaStorageFormat.IndexEntry> indexList = new ArrayList<SchemaStorageFormat.IndexEntry>();
        indexList.addAll(locator.getIndexList());
        indexList.add(info);
        return ((CompletableFuture)this.updateSchemaLocator(BookkeeperSchemaStorage.getSchemaPath(schemaId), SchemaStorageFormat.SchemaLocator.newBuilder().setInfo(info).addAllIndex(indexList).build(), locatorEntry.version).thenApply(ignore -> nextVersion)).whenComplete((__, ex) -> {
            if (ex != null) {
                Throwable cause = FutureUtil.unwrapCompletionException((Throwable)ex);
                log.warn("[{}] Failed to update schema locator with position {}", new Object[]{schemaId, position, cause});
                if (cause instanceof MetadataStoreException.AlreadyExistsException || cause instanceof MetadataStoreException.BadVersionException) {
                    this.bookKeeper.asyncDeleteLedger(position.getLedgerId(), new AsyncCallback.DeleteCallback(){

                        public void deleteComplete(int rc, Object ctx) {
                            if (rc != 0) {
                                log.warn("[{}] Failed to delete ledger {} after updating schema locator failed, rc: {}", new Object[]{schemaId, position.getLedgerId(), rc});
                            }
                        }
                    }, null);
                }
            }
        });
    }

    @NotNull
    private CompletableFuture<SchemaStorageFormat.SchemaEntry> findSchemaEntryByVersion(List<SchemaStorageFormat.IndexEntry> index, long version) {
        if (index.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        SchemaStorageFormat.IndexEntry lowest = index.get(0);
        if (version < lowest.getVersion()) {
            return this.readSchemaEntry(lowest.getPosition()).thenCompose(entry -> this.findSchemaEntryByVersion(entry.getIndexList(), version));
        }
        for (SchemaStorageFormat.IndexEntry entry2 : index) {
            if (entry2.getVersion() == version) {
                return this.readSchemaEntry(entry2.getPosition());
            }
            if (entry2.getVersion() <= version) continue;
            break;
        }
        return CompletableFuture.completedFuture(null);
    }

    @NotNull
    private CompletableFuture<SchemaStorageFormat.SchemaEntry> readSchemaEntry(SchemaStorageFormat.PositionInfo position) {
        if (log.isDebugEnabled()) {
            log.debug("Reading schema entry from {}", (Object)position);
        }
        return ((CompletableFuture)this.openLedger(position.getLedgerId()).thenCompose(ledger -> Functions.getLedgerEntry(ledger, position.getEntryId()).thenCompose(entry -> this.closeLedger((LedgerHandle)ledger).thenApply(ignore -> entry)))).thenCompose(Functions::parseSchemaEntry);
    }

    @NotNull
    private CompletableFuture<Void> updateSchemaLocator(String id, SchemaStorageFormat.SchemaLocator schema, long version) {
        return this.store.put(id, schema.toByteArray(), Optional.of(version)).thenApply(__ -> null);
    }

    @NotNull
    private CompletableFuture<LocatorEntry> createSchemaLocator(String id, SchemaStorageFormat.SchemaLocator locator) {
        return this.store.put(id, locator.toByteArray(), Optional.of(-1L)).thenApply(stat -> new LocatorEntry(locator, stat.getVersion()));
    }

    @NotNull
    private CompletableFuture<Optional<LocatorEntry>> getSchemaLocator(String schema) {
        return this.locatorEntryCache.getWithStats(schema).thenApply(o -> o.map(r -> new LocatorEntry((SchemaStorageFormat.SchemaLocator)r.getValue(), r.getStat().getVersion())));
    }

    @NotNull
    private CompletableFuture<Long> addEntry(LedgerHandle ledgerHandle, SchemaStorageFormat.SchemaEntry entry) {
        CompletableFuture<Long> future = new CompletableFuture<Long>();
        ledgerHandle.asyncAddEntry(entry.toByteArray(), (rc, handle, entryId, ctx) -> {
            if (rc != 0) {
                future.completeExceptionally(BookkeeperSchemaStorage.bkException("Failed to add entry", rc, ledgerHandle.getId(), -1L));
            } else {
                future.complete(entryId);
            }
        }, null);
        return future;
    }

    @NotNull
    private CompletableFuture<LedgerHandle> createLedger(String schemaId) {
        Map metadata = LedgerMetadataUtils.buildMetadataForSchema((String)schemaId);
        CompletableFuture<LedgerHandle> future = new CompletableFuture<LedgerHandle>();
        try {
            this.bookKeeper.asyncCreateLedger(this.config.getManagedLedgerDefaultEnsembleSize(), this.config.getManagedLedgerDefaultWriteQuorum(), this.config.getManagedLedgerDefaultAckQuorum(), BookKeeper.DigestType.fromApiDigestType((DigestType)this.config.getManagedLedgerDigestType()), LedgerPassword, (rc, handle, ctx) -> {
                if (rc != 0) {
                    future.completeExceptionally(BookkeeperSchemaStorage.bkException("Failed to create ledger", rc, -1L, -1L));
                } else {
                    future.complete(handle);
                }
            }, null, metadata);
        }
        catch (Throwable t) {
            log.error("[{}] Encountered unexpected error when creating schema ledger", (Object)schemaId, (Object)t);
            return FutureUtil.failedFuture((Throwable)t);
        }
        return future;
    }

    @NotNull
    private CompletableFuture<LedgerHandle> openLedger(Long ledgerId) {
        CompletableFuture<LedgerHandle> future = new CompletableFuture<LedgerHandle>();
        this.bookKeeper.asyncOpenLedger(ledgerId.longValue(), BookKeeper.DigestType.fromApiDigestType((DigestType)this.config.getManagedLedgerDigestType()), LedgerPassword, (rc, handle, ctx) -> {
            if (rc != 0) {
                future.completeExceptionally(BookkeeperSchemaStorage.bkException("Failed to open ledger", rc, ledgerId, -1L));
            } else {
                future.complete(handle);
            }
        }, null);
        return future;
    }

    @NotNull
    private CompletableFuture<Void> closeLedger(LedgerHandle ledgerHandle) {
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        ledgerHandle.asyncClose((rc, handle, ctx) -> {
            if (rc != 0) {
                future.completeExceptionally(BookkeeperSchemaStorage.bkException("Failed to close ledger", rc, ledgerHandle.getId(), -1L));
            } else {
                future.complete(null);
            }
        }, null);
        return future;
    }

    public CompletableFuture<List<Long>> getStoreLedgerIdsBySchemaId(String schemaId) {
        CompletableFuture<List<Long>> ledgerIdsFuture = new CompletableFuture<List<Long>>();
        ((CompletableFuture)this.getSchemaLocator(BookkeeperSchemaStorage.getSchemaPath(schemaId)).thenAccept(locator -> {
            if (log.isDebugEnabled()) {
                log.debug("[{}] Get all store schema ledgerIds - locator: {}", (Object)schemaId, locator);
            }
            if (!locator.isPresent()) {
                ledgerIdsFuture.complete(Collections.emptyList());
                return;
            }
            HashSet ledgerIds = new HashSet();
            SchemaStorageFormat.SchemaLocator schemaLocator = ((LocatorEntry)locator.get()).locator;
            schemaLocator.getIndexList().forEach(indexEntry -> ledgerIds.add(indexEntry.getPosition().getLedgerId()));
            ledgerIdsFuture.complete(new ArrayList(ledgerIds));
        })).exceptionally(e -> {
            ledgerIdsFuture.completeExceptionally((Throwable)e);
            return null;
        });
        return ledgerIdsFuture;
    }

    public static Exception bkException(String operation, int rc, long ledgerId, long entryId) {
        String message = BKException.getMessage((int)rc) + " -  ledger=" + ledgerId + " - operation=" + operation;
        if (entryId != -1L) {
            message = message + " - entry=" + entryId;
        }
        boolean recoverable = rc != -7 && rc != -13 && rc != -25;
        return new SchemaException(recoverable, message);
    }

    public static <T> CompletableFuture<T> ignoreUnrecoverableBKException(CompletableFuture<T> source) {
        return source.exceptionally(t -> {
            if (t.getCause() != null && t.getCause() instanceof SchemaException && !((SchemaException)t.getCause()).isRecoverable()) {
                if (log.isDebugEnabled()) {
                    log.debug("Schema data in bookkeeper may be deleted by other operations.", t);
                }
                return null;
            }
            throw t instanceof CompletionException ? (CompletionException)t : new CompletionException((Throwable)t);
        });
    }

    static class LocatorEntry {
        final SchemaStorageFormat.SchemaLocator locator;
        final long version;

        LocatorEntry(SchemaStorageFormat.SchemaLocator locator, long version) {
            this.locator = locator;
            this.version = version;
        }
    }

    static interface Functions {
        public static CompletableFuture<LedgerEntry> getLedgerEntry(LedgerHandle ledger, long entry) {
            CompletableFuture<LedgerEntry> future = new CompletableFuture<LedgerEntry>();
            ledger.asyncReadEntries(entry, entry, (rc, handle, entries, ctx) -> {
                if (rc != 0) {
                    future.completeExceptionally(BookkeeperSchemaStorage.bkException("Failed to read entry", rc, ledger.getId(), entry));
                } else {
                    future.complete((LedgerEntry)entries.nextElement());
                }
            }, null);
            return future;
        }

        public static CompletableFuture<SchemaStorageFormat.SchemaEntry> parseSchemaEntry(LedgerEntry ledgerEntry) {
            CompletableFuture<SchemaStorageFormat.SchemaEntry> result = new CompletableFuture<SchemaStorageFormat.SchemaEntry>();
            try {
                result.complete(SchemaStorageFormat.SchemaEntry.parseFrom(ledgerEntry.getEntry()));
            }
            catch (IOException e) {
                result.completeExceptionally(e);
            }
            return result;
        }

        public static SchemaStorageFormat.SchemaEntry newSchemaEntry(List<SchemaStorageFormat.IndexEntry> index, byte[] data) {
            return SchemaStorageFormat.SchemaEntry.newBuilder().setSchemaData(ByteString.copyFrom((byte[])data)).addAllIndex(index).build();
        }

        public static SchemaStorageFormat.PositionInfo newPositionInfo(long ledgerId, long entryId) {
            return SchemaStorageFormat.PositionInfo.newBuilder().setLedgerId(ledgerId).setEntryId(entryId).build();
        }
    }
}

