/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.connect.mirror;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.kafka.clients.admin.Admin;
import org.apache.kafka.clients.admin.AlterConsumerGroupOffsetsResult;
import org.apache.kafka.clients.admin.ConsumerGroupDescription;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.common.ConsumerGroupState;
import org.apache.kafka.common.KafkaFuture;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.UnknownMemberIdException;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.mirror.Checkpoint;
import org.apache.kafka.connect.mirror.CheckpointStore;
import org.apache.kafka.connect.mirror.MirrorCheckpointConnector;
import org.apache.kafka.connect.mirror.MirrorCheckpointMetrics;
import org.apache.kafka.connect.mirror.MirrorCheckpointTaskConfig;
import org.apache.kafka.connect.mirror.MirrorUtils;
import org.apache.kafka.connect.mirror.OffsetSyncStore;
import org.apache.kafka.connect.mirror.ReplicationPolicy;
import org.apache.kafka.connect.mirror.Scheduler;
import org.apache.kafka.connect.mirror.TopicFilter;
import org.apache.kafka.connect.source.SourceRecord;
import org.apache.kafka.connect.source.SourceTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MirrorCheckpointTask
extends SourceTask {
    private static final Logger log = LoggerFactory.getLogger(MirrorCheckpointTask.class);
    private Admin sourceAdminClient;
    private Admin targetAdminClient;
    private String sourceClusterAlias;
    private String targetClusterAlias;
    private String checkpointsTopic;
    private Duration interval;
    private Duration pollTimeout;
    private TopicFilter topicFilter;
    private Set<String> consumerGroups;
    private ReplicationPolicy replicationPolicy;
    private OffsetSyncStore offsetSyncStore;
    private boolean stopping;
    private MirrorCheckpointMetrics metrics;
    private Scheduler scheduler;
    private Map<String, Map<TopicPartition, OffsetAndMetadata>> idleConsumerGroupsOffset;
    private CheckpointStore checkpointStore;

    public MirrorCheckpointTask() {
    }

    MirrorCheckpointTask(String sourceClusterAlias, String targetClusterAlias, ReplicationPolicy replicationPolicy, OffsetSyncStore offsetSyncStore, Set<String> consumerGroups, Map<String, Map<TopicPartition, OffsetAndMetadata>> idleConsumerGroupsOffset, CheckpointStore checkpointStore) {
        this.sourceClusterAlias = sourceClusterAlias;
        this.targetClusterAlias = targetClusterAlias;
        this.replicationPolicy = replicationPolicy;
        this.offsetSyncStore = offsetSyncStore;
        this.consumerGroups = consumerGroups;
        this.idleConsumerGroupsOffset = idleConsumerGroupsOffset;
        this.checkpointStore = checkpointStore;
        this.topicFilter = topic -> true;
        this.interval = Duration.ofNanos(1L);
        this.pollTimeout = Duration.ofNanos(1L);
    }

    public void start(Map<String, String> props) {
        MirrorCheckpointTaskConfig config = new MirrorCheckpointTaskConfig(props);
        this.stopping = false;
        this.sourceClusterAlias = config.sourceClusterAlias();
        this.targetClusterAlias = config.targetClusterAlias();
        this.consumerGroups = config.taskConsumerGroups();
        this.checkpointsTopic = config.checkpointsTopic();
        this.topicFilter = config.topicFilter();
        this.replicationPolicy = config.replicationPolicy();
        this.interval = config.emitCheckpointsInterval();
        this.pollTimeout = config.consumerPollTimeout();
        this.offsetSyncStore = new OffsetSyncStore(config);
        this.sourceAdminClient = config.forwardingAdmin(config.sourceAdminConfig("checkpoint-source-admin"));
        this.targetAdminClient = config.forwardingAdmin(config.targetAdminConfig("checkpoint-target-admin"));
        this.metrics = config.metrics();
        this.idleConsumerGroupsOffset = new HashMap<String, Map<TopicPartition, OffsetAndMetadata>>();
        this.checkpointStore = new CheckpointStore(config, this.consumerGroups);
        this.scheduler = new Scheduler(((Object)((Object)this)).getClass(), config.entityLabel(), config.adminTimeout());
        this.scheduler.execute(() -> {
            boolean checkpointsReadOk = this.checkpointStore.start();
            this.offsetSyncStore.start(!checkpointsReadOk);
            this.scheduler.scheduleRepeating(this::refreshIdleConsumerGroupOffset, config.syncGroupOffsetsInterval(), "refreshing idle consumers group offsets at target cluster");
            this.scheduler.scheduleRepeatingDelayed(this::syncGroupOffset, config.syncGroupOffsetsInterval(), "sync idle consumer group offset from source to target");
        }, "starting checkpoint and offset sync stores");
        log.info("{} checkpointing {} consumer groups {}->{}: {}.", new Object[]{Thread.currentThread().getName(), this.consumerGroups.size(), this.sourceClusterAlias, config.targetClusterAlias(), this.consumerGroups});
    }

    public void commit() {
    }

    public void stop() {
        long start = System.currentTimeMillis();
        this.stopping = true;
        Utils.closeQuietly((AutoCloseable)this.topicFilter, (String)"topic filter");
        Utils.closeQuietly((AutoCloseable)this.checkpointStore, (String)"checkpoints store");
        Utils.closeQuietly((AutoCloseable)this.offsetSyncStore, (String)"offset sync store");
        Utils.closeQuietly((AutoCloseable)this.sourceAdminClient, (String)"source admin client");
        Utils.closeQuietly((AutoCloseable)this.targetAdminClient, (String)"target admin client");
        Utils.closeQuietly((AutoCloseable)this.metrics, (String)"metrics");
        Utils.closeQuietly((AutoCloseable)this.scheduler, (String)"scheduler");
        log.info("Stopping {} took {} ms.", (Object)Thread.currentThread().getName(), (Object)(System.currentTimeMillis() - start));
    }

    public String version() {
        return new MirrorCheckpointConnector().version();
    }

    public List<SourceRecord> poll() throws InterruptedException {
        try {
            long deadline = System.currentTimeMillis() + this.interval.toMillis();
            while (!this.stopping && System.currentTimeMillis() < deadline) {
                Thread.sleep(this.pollTimeout.toMillis());
            }
            if (this.stopping || !this.checkpointStore.isInitialized()) {
                return null;
            }
            ArrayList<SourceRecord> records = new ArrayList<SourceRecord>();
            for (String group : this.consumerGroups) {
                records.addAll(this.sourceRecordsForGroup(group));
            }
            if (records.isEmpty()) {
                return null;
            }
            return records;
        }
        catch (Throwable e) {
            log.warn("Failure polling consumer state for checkpoints.", e);
            return null;
        }
    }

    List<SourceRecord> sourceRecordsForGroup(String group) throws InterruptedException {
        try {
            long timestamp = System.currentTimeMillis();
            Map<TopicPartition, OffsetAndMetadata> upstreamGroupOffsets = this.listConsumerGroupOffsets(group);
            Map<TopicPartition, Checkpoint> newCheckpoints = this.checkpointsForGroup(upstreamGroupOffsets, group);
            this.checkpointStore.update(group, newCheckpoints);
            return newCheckpoints.values().stream().map(x -> this.checkpointRecord((Checkpoint)x, timestamp)).collect(Collectors.toList());
        }
        catch (ExecutionException e) {
            log.error("Error querying offsets for consumer group {} on cluster {}.", new Object[]{group, this.sourceClusterAlias, e});
            return Collections.emptyList();
        }
    }

    Map<TopicPartition, Checkpoint> checkpointsForGroup(Map<TopicPartition, OffsetAndMetadata> upstreamGroupOffsets, String group) {
        return upstreamGroupOffsets.entrySet().stream().filter(x -> this.shouldCheckpointTopic(((TopicPartition)x.getKey()).topic())).map(x -> this.checkpoint(group, (TopicPartition)x.getKey(), (OffsetAndMetadata)x.getValue())).flatMap(o -> o.map(Stream::of).orElseGet(Stream::empty)).filter(x -> x.downstreamOffset() >= 0L).filter(this::checkpointIsMoreRecent).collect(Collectors.toMap(Checkpoint::topicPartition, Function.identity()));
    }

    private boolean checkpointIsMoreRecent(Checkpoint checkpoint) {
        Map<TopicPartition, Checkpoint> checkpoints = this.checkpointStore.get(checkpoint.consumerGroupId());
        if (checkpoints == null) {
            log.trace("Emitting {} (first for this group)", (Object)checkpoint);
            return true;
        }
        Checkpoint lastCheckpoint = checkpoints.get(checkpoint.topicPartition());
        if (lastCheckpoint == null) {
            log.trace("Emitting {} (first for this partition)", (Object)checkpoint);
            return true;
        }
        if (checkpoint.upstreamOffset() < lastCheckpoint.upstreamOffset()) {
            log.trace("Emitting {} (upstream offset rewind)", (Object)checkpoint);
            return true;
        }
        if (checkpoint.downstreamOffset() > lastCheckpoint.downstreamOffset()) {
            log.trace("Emitting {} (downstream offset advanced)", (Object)checkpoint);
            return true;
        }
        if (checkpoint.downstreamOffset() != lastCheckpoint.downstreamOffset()) {
            log.trace("Skipping {} (preventing downstream rewind)", (Object)checkpoint);
        } else {
            log.trace("Skipping {} (repeated checkpoint)", (Object)checkpoint);
        }
        return false;
    }

    private Map<TopicPartition, OffsetAndMetadata> listConsumerGroupOffsets(String group) throws InterruptedException, ExecutionException {
        if (this.stopping) {
            return Collections.emptyMap();
        }
        return (Map)this.sourceAdminClient.listConsumerGroupOffsets(group).partitionsToOffsetAndMetadata().get();
    }

    Optional<Checkpoint> checkpoint(String group, TopicPartition topicPartition, OffsetAndMetadata offsetAndMetadata) {
        long upstreamOffset;
        OptionalLong downstreamOffset;
        if (offsetAndMetadata != null && (downstreamOffset = this.offsetSyncStore.translateDownstream(group, topicPartition, upstreamOffset = offsetAndMetadata.offset())).isPresent()) {
            return Optional.of(new Checkpoint(group, this.renameTopicPartition(topicPartition), upstreamOffset, downstreamOffset.getAsLong(), offsetAndMetadata.metadata()));
        }
        return Optional.empty();
    }

    SourceRecord checkpointRecord(Checkpoint checkpoint, long timestamp) {
        return new SourceRecord(checkpoint.connectPartition(), MirrorUtils.wrapOffset(0L), this.checkpointsTopic, Integer.valueOf(0), Schema.BYTES_SCHEMA, (Object)checkpoint.recordKey(), Schema.BYTES_SCHEMA, (Object)checkpoint.recordValue(), Long.valueOf(timestamp));
    }

    TopicPartition renameTopicPartition(TopicPartition upstreamTopicPartition) {
        if (this.targetClusterAlias.equals(this.replicationPolicy.topicSource(upstreamTopicPartition.topic()))) {
            return new TopicPartition(this.replicationPolicy.originalTopic(upstreamTopicPartition.topic()), upstreamTopicPartition.partition());
        }
        return new TopicPartition(this.replicationPolicy.formatRemoteTopic(this.sourceClusterAlias, upstreamTopicPartition.topic()), upstreamTopicPartition.partition());
    }

    boolean shouldCheckpointTopic(String topic) {
        return this.topicFilter.shouldReplicateTopic(topic);
    }

    public void commitRecord(SourceRecord record, RecordMetadata metadata) {
        this.metrics.checkpointLatency(MirrorUtils.unwrapPartition(record.sourcePartition()), Checkpoint.unwrapGroup((Map)record.sourcePartition()), System.currentTimeMillis() - record.timestamp());
    }

    private void refreshIdleConsumerGroupOffset() {
        Map consumerGroupsDesc = this.targetAdminClient.describeConsumerGroups(this.consumerGroups).describedGroups();
        for (String group : this.consumerGroups) {
            try {
                ConsumerGroupDescription consumerGroupDesc = (ConsumerGroupDescription)((KafkaFuture)consumerGroupsDesc.get(group)).get();
                ConsumerGroupState consumerGroupState = consumerGroupDesc.state();
                if (consumerGroupState != ConsumerGroupState.EMPTY) continue;
                this.idleConsumerGroupsOffset.put(group, (Map<TopicPartition, OffsetAndMetadata>)this.targetAdminClient.listConsumerGroupOffsets(group).partitionsToOffsetAndMetadata().get());
            }
            catch (InterruptedException | ExecutionException e) {
                log.error("Error querying for consumer group {} on cluster {}.", new Object[]{group, this.targetClusterAlias, e});
            }
        }
    }

    Map<String, Map<TopicPartition, OffsetAndMetadata>> syncGroupOffset() {
        HashMap<String, Map<TopicPartition, OffsetAndMetadata>> offsetToSyncAll = new HashMap<String, Map<TopicPartition, OffsetAndMetadata>>();
        for (Map.Entry<String, Map<TopicPartition, OffsetAndMetadata>> group : this.checkpointStore.computeConvertedUpstreamOffset().entrySet()) {
            String consumerGroupId = group.getKey();
            Map<TopicPartition, OffsetAndMetadata> convertedUpstreamOffset = group.getValue();
            HashMap<TopicPartition, OffsetAndMetadata> offsetToSync = new HashMap<TopicPartition, OffsetAndMetadata>();
            Map<TopicPartition, OffsetAndMetadata> targetConsumerOffset = this.idleConsumerGroupsOffset.get(consumerGroupId);
            if (targetConsumerOffset == null) {
                this.syncGroupOffset(consumerGroupId, convertedUpstreamOffset);
                offsetToSyncAll.put(consumerGroupId, convertedUpstreamOffset);
                continue;
            }
            for (Map.Entry<TopicPartition, OffsetAndMetadata> convertedEntry : convertedUpstreamOffset.entrySet()) {
                TopicPartition topicPartition = convertedEntry.getKey();
                OffsetAndMetadata convertedOffset = convertedUpstreamOffset.get(topicPartition);
                if (!targetConsumerOffset.containsKey(topicPartition)) {
                    offsetToSync.put(topicPartition, convertedOffset);
                    continue;
                }
                OffsetAndMetadata targetOffsetAndMetadata = targetConsumerOffset.get(topicPartition);
                if (targetOffsetAndMetadata != null) {
                    long latestDownstreamOffset = targetOffsetAndMetadata.offset();
                    if (latestDownstreamOffset >= convertedOffset.offset()) {
                        log.trace("latestDownstreamOffset {} is larger than or equal to convertedUpstreamOffset {} for TopicPartition {}", new Object[]{latestDownstreamOffset, convertedOffset.offset(), topicPartition});
                        continue;
                    }
                } else {
                    log.warn("Group {} offset for partition {} may has been reset to a negative offset, just sync the offset to target.", (Object)consumerGroupId, (Object)topicPartition);
                }
                offsetToSync.put(topicPartition, convertedOffset);
            }
            if (offsetToSync.size() == 0) {
                log.trace("skip syncing the offset for consumer group: {}", (Object)consumerGroupId);
                continue;
            }
            this.syncGroupOffset(consumerGroupId, offsetToSync);
            offsetToSyncAll.put(consumerGroupId, offsetToSync);
        }
        this.idleConsumerGroupsOffset.clear();
        return offsetToSyncAll;
    }

    void syncGroupOffset(String consumerGroupId, Map<TopicPartition, OffsetAndMetadata> offsetToSync) {
        if (this.targetAdminClient != null) {
            AlterConsumerGroupOffsetsResult result = this.targetAdminClient.alterConsumerGroupOffsets(consumerGroupId, offsetToSync);
            result.all().whenComplete((v, throwable) -> {
                if (throwable != null) {
                    if (throwable.getCause() instanceof UnknownMemberIdException) {
                        log.warn("Unable to sync offsets for consumer group {}. This is likely caused by consumers currently using this group in the target cluster.", (Object)consumerGroupId);
                    } else {
                        log.error("Unable to sync offsets for consumer group {}.", (Object)consumerGroupId, throwable);
                    }
                } else {
                    log.trace("Sync-ed {} offsets for consumer group {}.", (Object)offsetToSync.size(), (Object)consumerGroupId);
                }
            });
        }
    }
}

