/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.util;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.function.Function;
import org.apache.iceberg.DataFile;
import org.apache.iceberg.HistoryEntry;
import org.apache.iceberg.Schema;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.SnapshotRef;
import org.apache.iceberg.Table;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.exceptions.ValidationException;
import org.apache.iceberg.io.FileIO;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableList;
import org.apache.iceberg.relocated.com.google.common.collect.Iterables;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.util.DateTimeUtil;

public class SnapshotUtil {
    private SnapshotUtil() {
    }

    public static boolean isAncestorOf(Table table, long snapshotId, long ancestorSnapshotId) {
        for (Snapshot snapshot : SnapshotUtil.ancestorsOf(snapshotId, table::snapshot)) {
            if (snapshot.snapshotId() != ancestorSnapshotId) continue;
            return true;
        }
        return false;
    }

    public static boolean isAncestorOf(long snapshotId, long ancestorSnapshotId, Function<Long, Snapshot> lookup) {
        for (Snapshot snapshot : SnapshotUtil.ancestorsOf(snapshotId, lookup)) {
            if (snapshot.snapshotId() != ancestorSnapshotId) continue;
            return true;
        }
        return false;
    }

    public static boolean isAncestorOf(Table table, long ancestorSnapshotId) {
        return SnapshotUtil.isAncestorOf(table, table.currentSnapshot().snapshotId(), ancestorSnapshotId);
    }

    public static boolean isParentAncestorOf(Table table, long snapshotId, long ancestorParentSnapshotId) {
        for (Snapshot snapshot : SnapshotUtil.ancestorsOf(snapshotId, table::snapshot)) {
            if (snapshot.parentId() == null || snapshot.parentId() != ancestorParentSnapshotId) continue;
            return true;
        }
        return false;
    }

    public static Iterable<Snapshot> currentAncestors(Table table) {
        return SnapshotUtil.ancestorsOf(table.currentSnapshot(), table::snapshot);
    }

    public static List<Long> currentAncestorIds(Table table) {
        return SnapshotUtil.ancestorIds(table.currentSnapshot(), table::snapshot);
    }

    public static Snapshot oldestAncestor(Table table) {
        Snapshot lastSnapshot = null;
        Iterator<Snapshot> iterator = SnapshotUtil.currentAncestors(table).iterator();
        while (iterator.hasNext()) {
            Snapshot snapshot;
            lastSnapshot = snapshot = iterator.next();
        }
        return lastSnapshot;
    }

    public static Snapshot oldestAncestorOf(Table table, long snapshotId) {
        return SnapshotUtil.oldestAncestorOf(snapshotId, table::snapshot);
    }

    public static Snapshot oldestAncestorOf(long snapshotId, Function<Long, Snapshot> lookup) {
        Snapshot lastSnapshot = null;
        Iterator<Snapshot> iterator = SnapshotUtil.ancestorsOf(snapshotId, lookup).iterator();
        while (iterator.hasNext()) {
            Snapshot snapshot;
            lastSnapshot = snapshot = iterator.next();
        }
        return lastSnapshot;
    }

    public static Iterable<Snapshot> ancestorsOf(long snapshotId, Function<Long, Snapshot> lookup) {
        Snapshot start = lookup.apply(snapshotId);
        Preconditions.checkArgument(start != null, "Cannot find snapshot: %s", snapshotId);
        return SnapshotUtil.ancestorsOf(start, lookup);
    }

    public static Snapshot oldestAncestorAfter(Table table, long timestampMillis) {
        if (table.currentSnapshot() == null) {
            return null;
        }
        Snapshot lastSnapshot = null;
        for (Snapshot snapshot : SnapshotUtil.currentAncestors(table)) {
            if (snapshot.timestampMillis() < timestampMillis) {
                return lastSnapshot;
            }
            if (snapshot.timestampMillis() == timestampMillis) {
                return snapshot;
            }
            lastSnapshot = snapshot;
        }
        if (lastSnapshot != null && lastSnapshot.parentId() == null) {
            return lastSnapshot;
        }
        throw new IllegalStateException("Cannot find snapshot older than " + DateTimeUtil.formatTimestampMillis(timestampMillis));
    }

    public static List<Long> snapshotIdsBetween(Table table, long fromSnapshotId, long toSnapshotId) {
        ArrayList<Long> snapshotIds = Lists.newArrayList(SnapshotUtil.ancestorIds(table.snapshot(toSnapshotId), snapshotId -> snapshotId != fromSnapshotId ? table.snapshot((long)snapshotId) : null));
        return snapshotIds;
    }

    public static Iterable<Long> ancestorIdsBetween(long latestSnapshotId, Long oldestSnapshotId, Function<Long, Snapshot> lookup) {
        return SnapshotUtil.toIds(SnapshotUtil.ancestorsBetween(latestSnapshotId, oldestSnapshotId, lookup));
    }

    public static Iterable<Snapshot> ancestorsBetween(Table table, long latestSnapshotId, Long oldestSnapshotId) {
        return SnapshotUtil.ancestorsBetween(latestSnapshotId, oldestSnapshotId, table::snapshot);
    }

    public static Iterable<Snapshot> ancestorsBetween(long latestSnapshotId, Long oldestSnapshotId, Function<Long, Snapshot> lookup) {
        if (oldestSnapshotId != null) {
            if (latestSnapshotId == oldestSnapshotId) {
                return ImmutableList.of();
            }
            return SnapshotUtil.ancestorsOf(latestSnapshotId, (Long snapshotId) -> !oldestSnapshotId.equals(snapshotId) ? (Snapshot)lookup.apply((Long)snapshotId) : null);
        }
        return SnapshotUtil.ancestorsOf(latestSnapshotId, lookup);
    }

    private static Iterable<Snapshot> ancestorsOf(final Snapshot snapshot, final Function<Long, Snapshot> lookup) {
        if (snapshot != null) {
            return () -> new Iterator<Snapshot>(){
                private Snapshot next;
                private boolean consumed;
                {
                    this.next = snapshot;
                    this.consumed = false;
                }

                @Override
                public boolean hasNext() {
                    if (!this.consumed) {
                        return true;
                    }
                    if (this.next == null) {
                        return false;
                    }
                    Long parentId = this.next.parentId();
                    if (parentId == null) {
                        return false;
                    }
                    this.next = (Snapshot)lookup.apply(parentId);
                    if (this.next != null) {
                        this.consumed = false;
                        return true;
                    }
                    return false;
                }

                @Override
                public Snapshot next() {
                    if (this.hasNext()) {
                        this.consumed = true;
                        return this.next;
                    }
                    throw new NoSuchElementException();
                }
            };
        }
        return ImmutableList.of();
    }

    public static List<Long> ancestorIds(Snapshot snapshot, Function<Long, Snapshot> lookup) {
        return Lists.newArrayList(SnapshotUtil.toIds(SnapshotUtil.ancestorsOf(snapshot, lookup)));
    }

    private static Iterable<Long> toIds(Iterable<Snapshot> snapshots) {
        return Iterables.transform(snapshots, Snapshot::snapshotId);
    }

    public static List<DataFile> newFiles(Long baseSnapshotId, long latestSnapshotId, Function<Long, Snapshot> lookup, FileIO io) {
        ArrayList<DataFile> newFiles = Lists.newArrayList();
        Snapshot lastSnapshot = null;
        Iterator<Snapshot> iterator = SnapshotUtil.ancestorsOf(latestSnapshotId, lookup).iterator();
        while (iterator.hasNext()) {
            Snapshot currentSnapshot;
            lastSnapshot = currentSnapshot = iterator.next();
            if (Objects.equals(currentSnapshot.snapshotId(), baseSnapshotId)) {
                return newFiles;
            }
            Iterables.addAll(newFiles, currentSnapshot.addedDataFiles(io));
        }
        ValidationException.check(Objects.equals(lastSnapshot.parentId(), baseSnapshotId), "Cannot determine history between read snapshot %s and the last known ancestor %s", baseSnapshotId, lastSnapshot.snapshotId());
        return newFiles;
    }

    public static Snapshot snapshotAfter(Table table, long snapshotId) {
        Preconditions.checkArgument(table.snapshot(snapshotId) != null, "Cannot find parent snapshot: %s", snapshotId);
        for (Snapshot current : SnapshotUtil.currentAncestors(table)) {
            if (current.parentId() != snapshotId) continue;
            return current;
        }
        throw new IllegalStateException(String.format("Cannot find snapshot after %s: not an ancestor of table's current snapshot", snapshotId));
    }

    public static long snapshotIdAsOfTime(Table table, long timestampMillis) {
        Long snapshotId = SnapshotUtil.nullableSnapshotIdAsOfTime(table, timestampMillis);
        Preconditions.checkArgument(snapshotId != null, "Cannot find a snapshot older than %s", (Object)DateTimeUtil.formatTimestampMillis(timestampMillis));
        return snapshotId;
    }

    public static Long nullableSnapshotIdAsOfTime(Table table, long timestampMillis) {
        Long snapshotId = null;
        for (HistoryEntry logEntry : table.history()) {
            if (logEntry.timestampMillis() > timestampMillis) continue;
            snapshotId = logEntry.snapshotId();
        }
        return snapshotId;
    }

    public static Schema schemaFor(Table table, long snapshotId) {
        Snapshot snapshot = table.snapshot(snapshotId);
        Preconditions.checkArgument(snapshot != null, "Cannot find snapshot with ID %s", snapshotId);
        Integer schemaId = snapshot.schemaId();
        if (schemaId != null) {
            Schema schema = table.schemas().get(schemaId);
            Preconditions.checkState(schema != null, "Cannot find schema with schema id %s", (Object)schemaId);
            return schema;
        }
        return table.schema();
    }

    public static Schema schemaFor(Table table, Long snapshotId, Long timestampMillis) {
        Preconditions.checkArgument(snapshotId == null || timestampMillis == null, "Cannot use both snapshot id and timestamp to find a schema");
        if (snapshotId != null) {
            return SnapshotUtil.schemaFor(table, snapshotId);
        }
        if (timestampMillis != null) {
            return SnapshotUtil.schemaFor(table, SnapshotUtil.snapshotIdAsOfTime(table, timestampMillis));
        }
        return table.schema();
    }

    public static Schema schemaFor(Table table, String ref) {
        if (ref == null || ref.equals("main")) {
            return table.schema();
        }
        SnapshotRef snapshotRef = table.refs().get(ref);
        if (null == snapshotRef || snapshotRef.isBranch()) {
            return table.schema();
        }
        return SnapshotUtil.schemaFor(table, snapshotRef.snapshotId());
    }

    public static Schema schemaFor(TableMetadata metadata, String ref) {
        if (ref == null || ref.equals("main")) {
            return metadata.schema();
        }
        SnapshotRef snapshotRef = metadata.ref(ref);
        if (snapshotRef == null || snapshotRef.isBranch()) {
            return metadata.schema();
        }
        Snapshot snapshot = metadata.snapshot(snapshotRef.snapshotId());
        return metadata.schemas().get(snapshot.schemaId());
    }

    public static Snapshot latestSnapshot(Table table, String branch) {
        if (branch == null || branch.equals("main")) {
            return table.currentSnapshot();
        }
        return table.snapshot(branch);
    }

    public static Snapshot latestSnapshot(TableMetadata metadata, String branch) {
        if (branch == null || branch.equals("main")) {
            return metadata.currentSnapshot();
        }
        SnapshotRef ref = metadata.ref(branch);
        if (ref == null) {
            return metadata.currentSnapshot();
        }
        return metadata.snapshot(ref.snapshotId());
    }
}

