/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.druid.sql.calcite.table;

import com.google.common.base.Preconditions;
import org.apache.calcite.config.CalciteConnectionConfig;
import org.apache.calcite.rel.type.RelDataType;
import org.apache.calcite.rel.type.RelDataTypeFactory;
import org.apache.calcite.schema.Schema;
import org.apache.calcite.schema.Statistic;
import org.apache.calcite.schema.Statistics;
import org.apache.calcite.schema.TranslatableTable;
import org.apache.calcite.sql.SqlCall;
import org.apache.calcite.sql.SqlNode;
import org.apache.druid.query.DataSource;
import org.apache.druid.segment.column.RowSignature;

import java.util.Objects;

/**
 * Abstract base class for the various kinds of tables which Druid supports.
 */
public abstract class DruidTable implements TranslatableTable
{
  private final RowSignature rowSignature;

  public DruidTable(
      final RowSignature rowSignature
  )
  {
    this.rowSignature = Preconditions.checkNotNull(rowSignature, "rowSignature");
  }

  public RowSignature getRowSignature()
  {
    return rowSignature;
  }

  public abstract DataSource getDataSource();

  public abstract boolean isJoinable();

  public abstract boolean isBroadcast();

  @Override
  public Schema.TableType getJdbcTableType()
  {
    return Schema.TableType.TABLE;
  }

  @Override
  public Statistic getStatistic()
  {
    return Statistics.UNKNOWN;
  }

  @Override
  public RelDataType getRowType(final RelDataTypeFactory typeFactory)
  {
    return RowSignatures.toRelDataType(getRowSignature(), typeFactory);
  }

  @Override
  public boolean isRolledUp(final String column)
  {
    return false;
  }

  @Override
  public boolean rolledUpColumnValidInsideAgg(
      final String column,
      final SqlCall call,
      final SqlNode parent,
      final CalciteConnectionConfig config
  )
  {
    return true;
  }

  @Override
  public boolean equals(Object o)
  {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    DruidTable that = (DruidTable) o;

    if (!Objects.equals(getDataSource(), that.getDataSource())) {
      return false;
    }
    return Objects.equals(rowSignature, that.rowSignature);
  }

  @Override
  public int hashCode()
  {
    return Objects.hash(getDataSource(), rowSignature);
  }

  @Override
  public String toString()
  {
    return "DruidTable{" +
           "dataSource=" + getDataSource() +
           ", rowSignature=" + rowSignature +
           '}';
  }
}
