//--------------------------------------------------------------------------;
//
//  File: BUFOBJ.C
//
//  Description:
//      The C implemtation of the DirectSound Buffer object.
//      This code is the interface and methods for a buffer object.
//      The buffer is created by a driver object (see DRVOBJ.C)
//      This driver can only create 1 buffer object for the primary
//      buffer. (DMA buffer). Other cards will create multiple buffer
//      objects for each secondary hardware buffer.
//
//
//  THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
//  KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
//  IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
//  PURPOSE.
//
//---------------------------------------------------------------------------
//
//  Copyright (c) 1994 - 1995 Microsoft Corporation.  All Rights Reserved.
//
//---------------------------------------------------------------------------
#define  WANTVXDWRAPS

#include <windows.h>
#include "vmm.h"
#include <vmmreg.h>
#include <vxdldr.h>
#include <vxdwraps.h>

#include <configmg.h>
#include <dsound.h>
#include "dsdriver.h"
#include <debug.h>

#include "dsnd.h"
#include "Dos_xfer.h"

#undef CURSEG
#define CURSEG() PCODE

#pragma VxD_PAGEABLE_CODE_SEG
#pragma VxD_PAGEABLE_DATA_SEG

// drvobj.c
extern WORD FreqNom;
extern DWORD dwVoiceAllocated;
extern DWORD dwSizeAlloc;
extern BOOL VoiceFree[];
extern short int BufferRef[256];
extern WORD ModNumber;		//  Number of static buffer allowed (to be dynamically computed 
extern	WORD nVoices;		// Number of static voices that can be played
extern	DWORD dwBufferOpenned;
extern PDSDRIVERBUFFER pPrimaryBufCtx;

//propset
extern PDS3DLIST Buffer3DList;

// Voices Manager table;
extern VOICESMAN VoicesManager[64];

extern	int BlockNb;
extern MemBlDefTyp MemoryBlock[256];

int nre=0;

HRESULT __stdcall
IDsDriverBuffer_Stop(
    IN PIDSDRIVERBUFFER pIDsDriverBuffer
    );
HRESULT __stdcall
IDsDriverBuffer_GetPosition(
    IN PIDSDRIVERBUFFER pIDsDriverBuffer,
    OUT LPDWORD pCurrentPlayCursor,
    OUT LPDWORD pCurrentWriteCursor
    );

//--------------------------------------------------------------------------
//
//  AllocatePcBuffer
//  Description:
//      This code allocates memory in PC (buffer see by application)
//  Parameters:
//      pBufferObjContext: Our Buffer object context (*this)
//      *ppBuffer: Return a valid virtual pointer to the
//          allocated buffer here.
//      pRequestedSize: This is both an input and output var.
//          input: The size of buffer the app DirectSound is
//              requesting.
//          output: The size of the buffer it got.
//  Return:
//      none
//  
//  
//--------------------------------------------------------------------------
BOOL
AllocatePcBuffer(
    IN PDSDRIVERBUFFER pBufferObjContext,
    OUT LPBYTE *ppBuffer,
    IN OUT LPDWORD pRequestedSize
    )
{
    DP(("AllocateBuffer\n"));
	//*ppBuffer=(LPBYTE) (pBufferObjContext->PcLinAddr=(DWORD) _PageAllocate((*pRequestedSize)/0x1000 + 1,PG_SYS,0,0,0,0xfff,0,PAGEFIXED | PAGECONTIG |  PAGEUSEALIGN));//&(pBufferObjContext->PcPhysAddr),PAGECONTIG + PAGEFIXED));
	*ppBuffer=(LPBYTE) (pBufferObjContext->PcLinAddr=(DWORD) _PageAllocate((*pRequestedSize)/0x1000 + 1,PG_SYS,0,0,0,0xfff,0,PAGEFIXED));
	//	*ppBuffer=Buff;
	if (!(*ppBuffer))
	{
		Trace_Out("Can't allocate buffer");
		return FALSE;
	}

    // remember the size for later
    pBufferObjContext->PCBufferSize = *pRequestedSize;

    return TRUE;
}

//--------------------------------------------------------------------------
//
//  IDsDriverBuffer_Release
//  Description:
//      This functions is the destructor of our buffer object.
//  Parameters:
//      pIDsDriverBuffer: Our buffer object context
//  Return:
//      Obect ref count.
//  
//  
//--------------------------------------------------------------------------
ULONG __stdcall
IDsDriverBuffer_Release(
    IN PIDSDRIVERBUFFER pIDsDriverBuffer
    )
{
    PDSDRIVERBUFFER pBufferObjContext = (PDSDRIVERBUFFER)pIDsDriverBuffer;
    ULONG ReturnCount;

    DP(("IDsDriverBuffer_Release\n"));

	nre++;

    // this instance is going away, check our ref count
    pBufferObjContext->ReferenceCounter--;

    ReturnCount = pBufferObjContext->ReferenceCounter;

    
   // ASSERT( !pBufferObjContext->ReferenceCounter );
    if( !pBufferObjContext->ReferenceCounter )
    {
		ASSERT( pBufferObjContext->DacState == DS_DAC_STATE_STOPPED )
        if( pBufferObjContext->DacState == DS_DAC_STATE_STOPPED )
        {
            // We kep a pointer to the driver object (our parent).
            // The driver object keeps a count of allocated objects.
            // Decrement that count.
            pBufferObjContext->pDriverContext->ObjRefCount--;

            // kill this buffer...if we can (let's take care of duplicating buffer)
			if (!(--BufferRef[pBufferObjContext->PhysBufferNb]))	
			{
				FreeBoardRessources(pBufferObjContext);
				if (!_PageFree((PVOID)pBufferObjContext->PcLinAddr,0))
				{
					Trace_Out("error in freeing pc buffer");
				}
			}

			// kill this buffer
			if (pBufferObjContext->p3DProperty!=NULL)		// 3D structure allocated
			{
				_HeapFree(pBufferObjContext->p3DProperty, 0 );
			/*	if (!pBufferObjContext->Primary)
					Remove3DBufferFromList(pBufferObjContext);*/
			}
            _HeapFree( pBufferObjContext, 0 );
		}
    }
    return ReturnCount;
}

//--------------------------------------------------------------------------
//
//  IDsDriverBuffer_QueryInterface
//  Description:
//      We want to be COM, but we're not. This interface would be used
//      to extend functionality to the buffer object interface.
//  Parameters:
//  Return:
//      vtable
//  
//  
//--------------------------------------------------------------------------
HRESULT __stdcall
IDsDriverBuffer_QueryInterface(
    IN PIDSDRIVERBUFFER pIDsDriverBuffer,
    IN REFIID riid,
    IN LPLPVOID ppv
    )
{
    PDSDRIVERBUFFER pBufferObjContext = (PDSDRIVERBUFFER)pIDsDriverBuffer;

    DP(("IDsDriverBuffer_QueryInterface\n"));


    *ppv = NULL;

	if (IsEqualGUID(riid,&IID_IDsDriverPropertySet))
	{		// interface on PropertySet requested
		PDSPROPERTYSET pIPropSet = _HeapAllocate(sizeof(DSPROPERTYSET), HEAPZEROINIT | HEAPSWAP);
		// setup our V-table
        pIPropSet->lpVtbl = &vtDsPropertySet;
		pIPropSet->pIBuffer=pBufferObjContext;

		pIPropSet->ReferenceCounter++;
		*ppv=pIPropSet;

		return DS_OK;
	}

    if(( !IsEqualGUID(riid,&IID_IDsDriverBuffer)) &&
	    (!IsEqualGUID(riid,&IID_IUnknown)  ))
    {
        return DSERR_NOINTERFACE;
    }

    *ppv = pIDsDriverBuffer;
    ((LPUNKNOWN)*ppv)->lpVtbl->AddRef(*ppv);

    return DS_OK;
}

//--------------------------------------------------------------------------
//
//  IDsDriverBuffer_AddRef
//  Description:
//      Add anohter reference count to the buffer object interface
//  Parameters:
//      pIDsDriverBuffer: Buffer object context.
//  Return:
//      Current reference count;
//  
//  
//--------------------------------------------------------------------------
ULONG __stdcall
IDsDriverBuffer_AddRef(
    IN PIDSDRIVERBUFFER pIDsDriverBuffer
)
{
    PDSDRIVERBUFFER pBufferObjContext = (PDSDRIVERBUFFER)pIDsDriverBuffer;
    DP(("IDsDriverBuffer_AddRef\n"));


    // add one.
    pBufferObjContext->ReferenceCounter++;
    // return the CURRENT count.
    return pBufferObjContext->ReferenceCounter;
}

//--------------------------------------------------------------------------
//
//  IDsDriverBuffer_Lock
//  Description:
//      Lock a section of the buffer for reading/writing.
//      The app passes in the postion of the start of the
//      lock, and the number of bytes to lock.
//      Because the buffer is a circular buffer the driver
//      returns:
//          A pointer into the bufffer and size. If the
//          requested start-pos/size does not wrap the circular buffer
//          then the second pointer and size should be set to
//          NULL. If the requesed start-pos/size DOES wrap the circular
//          buffer then the second pointer should be set to the beginning
//          of the circular buffer.
//      NOTE: Becase the MSSNDSYS is DMA based card and only has
//          one primary buffer (the DMA buffer) this call is not
//          needed.
//  Parameters:
//      pIDsDriverBuffer: Buffer object context.
//      ppvAudio1: Pointer to first section of buffer.
//      pdwLen1: Length of first section (bytes).
//      ppvAudio2: Pointer to second section of buffer.
//      pdwLen2: Length of section secton (bytes).
//      dwWritePosition: Requeted write pos.
//      dwWriteLen: Length of write.
//      dwFlags: Reserved.  This parameter should be ignored.
//  Return:
//      Error code
//  
//  
//--------------------------------------------------------------------------
HRESULT __stdcall
IDsDriverBuffer_Lock(
    IN PIDSDRIVERBUFFER pIDsDriverBuffer,
    OUT LPLPVOID ppvAudio1,
    OUT LPDWORD pdwLen1,
    OUT LPLPVOID ppvAudio2,
    OUT LPDWORD pdwLen2,
    IN DWORD dwWritePosition,
    IN DWORD dwWriteLen,
    IN DWORD dwFlags
    )
{
	PDSDRIVERBUFFER pBufferObjContext = (PDSDRIVERBUFFER)pIDsDriverBuffer;
    DP(("Lock\n"));
	


	if  ((dwWritePosition>=pBufferObjContext->PCBufferSize) ||
			(dwWriteLen>pBufferObjContext->PCBufferSize))
		return DSERR_INVALIDPARAM;

	*ppvAudio1= (LPVOID) (pBufferObjContext->PcLinAddr + dwWritePosition);

	if (dwWriteLen + dwWritePosition <= pBufferObjContext->PCBufferSize)
	{							// not wrap
		*pdwLen1=dwWriteLen;
		*pdwLen2=0;
		*ppvAudio2=NULL;
	}
	else
	{							// wrap around the end (last datas in the buffer start)
		*pdwLen1=pBufferObjContext->PCBufferSize-dwWritePosition;
		*pdwLen2=dwWriteLen-*pdwLen1;
		*ppvAudio2=(LPVOID) (pBufferObjContext->PcLinAddr);
	}
	
	pBufferObjContext->BufferWritePosition=dwWritePosition/2;

	if (pBufferObjContext->wBitsPerSample==8)	//8bits
		pBufferObjContext->BufferWritePosition*=2;

	if (pBufferObjContext->nChannels==2)		//stereo
		pBufferObjContext->BufferWritePosition/=2;

    return DS_OK;
}

//--------------------------------------------------------------------------
//
//  IDsDriverBuffer_Unlock
//  Description:
//      Unlock a section of buffer prevously locked.
//      The Unlocked section might be a sub-set of the locked
//      section. The exact data accessed is passed to the un-lock
//      call to help drivers who have virtualized the primary buffer.
//      (i.e. only copy the nessessary data).
//      This call is not implemented on a MSSNDSYS card.
//  Parameters:
//      pIDsDriverBuffer: Buffer object context
//      pvAudio1: Pointer to first section unlocked
//      dwLen1: size of first section unlocked
//      pvAudio2: Pointer to second section unlocked
//      dwLen2: size of second section unlocked
//          NOTE: pvAudio2 and dwLen2 can be NULL.
//              Be sure to check that the unlocked region
//              is continious.
//  Return:
//      Error code
//  
//  
//--------------------------------------------------------------------------
HRESULT __stdcall
IDsDriverBuffer_Unlock(
    IN PIDSDRIVERBUFFER pIDsDriverBuffer,
    IN LPVOID pvAudio1,
    IN DWORD dwLen1,
    IN LPVOID pvAudio2,
    IN DWORD dwLen2
    )
{
	PDSDRIVERBUFFER pBufferObjContext = (PDSDRIVERBUFFER)pIDsDriverBuffer;
	DWORD BuffSize;
    DP(("UnLock\n"));
	

	// Protection
	if ((char *) pvAudio1- (char *) (pBufferObjContext->PcLinAddr) + dwLen1 > pBufferObjContext->PCBufferSize)
	{
		dwLen1=pBufferObjContext->PCBufferSize - ( (char *) pvAudio1-(char *) (pBufferObjContext->PcLinAddr));
		if (dwLen1>pBufferObjContext->PCBufferSize)
			return DSERR_INVALIDPARAM;
	}

	BuffSize=dwLen1;
	if (pBufferObjContext->wBitsPerSample==8)	// 8 bits
		BuffSize*=2;

	if (pBufferObjContext->nChannels!=1)		//stereo
		BuffSize/=2;
	BuffSize/=2;		//size in 16b WORD


	if (!TransferBlock(pBufferObjContext,pBufferObjContext->BufferWritePosition,pvAudio1,BuffSize))
		return DSERR_INVALIDCALL;
	if ((pvAudio2) && (dwLen2))
	{
	// Protection
		if ((char *) pvAudio2- (char *) (pBufferObjContext->PcLinAddr) + dwLen2 > pBufferObjContext->PCBufferSize)
		{
			dwLen2=pBufferObjContext->PCBufferSize- ( (char *)pvAudio2-(char *) (pBufferObjContext->PcLinAddr));
			if (dwLen2>pBufferObjContext->PCBufferSize)
				return DSERR_INVALIDPARAM;
		}


		BuffSize=dwLen2;
		if (pBufferObjContext->wBitsPerSample==8)	// 8 bits
			BuffSize*=2;

		if (pBufferObjContext->nChannels!=1)		//stereo
			BuffSize/=2;
		BuffSize/=2;		//size in 16b WORD

		if (!TransferBlock(pBufferObjContext,0,pvAudio2,BuffSize))
			return DSERR_INVALIDCALL;
	}

    return DS_OK;
}

//--------------------------------------------------------------------------
//
//  IDsDriverBuffer_SetFormat
//  Description:
//      Set the format of the buffer. (This is not the rate/freq)
//      This call is not used on the MSSNDSYS. On DMA based cards
//      DirectSound should use MMSYSTEM to set the rate.
//  Parameters:
//      pIDsDriverBuffer: buffer object context
//      pwfx: pointer to requested wave format. (set the format, not the rate/freq)
//  Return:
//      Error code
//  
//  
//--------------------------------------------------------------------------
HRESULT __stdcall
IDsDriverBuffer_SetFormat(
    IN PIDSDRIVERBUFFER pIDsDriverBuffer,
    IN LPWAVEFORMATEX pwfx
    )
{
	PDSDRIVERBUFFER pBufferObjContext = (PDSDRIVERBUFFER)pIDsDriverBuffer;

    DP(("IDsDriverBuffer_SetFormat \n"));

	FreeBoardRessources(pBufferObjContext);

	pBufferObjContext->wBitsPerSample = pwfx->wBitsPerSample;
	pBufferObjContext->nChannels = pwfx->nChannels;

	// Set the Pitch  (is it really required? (Yes for dsshow3d)
	pBufferObjContext->nSamplesPerSec=(DWORD) (pwfx->nSamplesPerSec*0x400/FreqNom);


	if (AllocateBoardRessources(pBufferObjContext, pBufferObjContext->PCBufferSize))
		return DS_OK;
	else
		return DSERR_INVALIDPARAM;

}

//--------------------------------------------------------------------------
//
//  IDsDriverBuffer_SetFrequency
//  Description:
//      Set the Frequency of the buffer. (not the format)
//      This call is not used on the MSSNDSYS. On DMA based cards
//      DirectSound should use MMSYSTEM to set the Frequency.
//  Parameters:
//      pIDsDriverBuffer: buffer object context
//      pwfx: pointer to requested wave format. (set the Frequency, not the format)
//  Return:
//      Error code
//  
//  
//--------------------------------------------------------------------------
HRESULT __stdcall
IDsDriverBuffer_SetFrequency(
    IN PIDSDRIVERBUFFER pIDsDriverBuffer,
    IN DWORD dwFrequency
    )
{
	PDSDRIVERBUFFER pBufferObjContext = (PDSDRIVERBUFFER)pIDsDriverBuffer;
    //DP(("IDsDriverBuffer_SetFrequency\n"));

	DWORD pbs= (DWORD) pBufferObjContext;
	__asm mov eax,pbs
	Trace_Out("IDsDriverBuffer_SetFrequency #eax");


	if (dwFrequency>0xffff)
		//dwFrequency=0xffff;
		return DSERR_INVALIDPARAM;
	pBufferObjContext->nSamplesPerSec=(DWORD) (dwFrequency*0x400/FreqNom);

	if (pBufferObjContext->DacState != DS_DAC_STATE_STOPPED)
	{
		if (pBufferObjContext->Enabled3D==0)
		{
			bufSetFreq(pBufferObjContext,pBufferObjContext->nSamplesPerSec);
			Trace_Out("P=#eax,  D=#ebx,  V=#dx \n");
		}
		else
			//3D->Doppler
			bufSetFreq(pBufferObjContext,(DWORD) (pBufferObjContext->Doppler*pBufferObjContext->nSamplesPerSec));
	}

	return DS_OK;
}

//--------------------------------------------------------------------------
//
//  IDsDriverBuffer_SetVolumePan
//  Description:
//      Set the volume and pan on a sound buffer.
//      If the buffer format is mono then setting the
//      pan is expected to work. (Steroize the mono source).
//      This interface is not susported on the MSSNDSYS.
//  Parameters:
//      pIDsDriverBuffer: buffer object context
//      pVolPan: Volume and pan information.
//  Return:
//      Error code
//  
//  
//--------------------------------------------------------------------------
HRESULT __stdcall
IDsDriverBuffer_SetVolumePan(
    IN PIDSDRIVERBUFFER pIDsDriverBuffer,
    IN PDSVOLUMEPAN pVolPan
    )
{
	DWORD MainVol, LVol, RVol;
	PDSDRIVERBUFFER pBufferObjContext = (PDSDRIVERBUFFER)pIDsDriverBuffer;
    DP(("IDsDriverBuffer_SetVolume\n"));

		MainVol=(pVolPan->dwVolAmpFactor) >> 8;
		if (MainVol>0xff)
			MainVol=0xff;
		RVol=(pVolPan->dwPanRightAmpFactor) >> 8;
		if (RVol>0xff)
			MainVol=0xff;
		LVol=(pVolPan->dwPanLeftAmpFactor) >> 8;
		if (LVol>0xff)
			LVol=0xff;
	pBufferObjContext->MainVol=MainVol;
	pBufferObjContext->RVol=RVol;
	pBufferObjContext->LVol=LVol;
	
	if (pBufferObjContext->DacState != DS_DAC_STATE_STOPPED)
	{
		// if not a 3D buffer, we must set new volumes
		if (pBufferObjContext->Enabled3D==0)
		{
			if (!pBufferObjContext->Primary)
				bufSetVol(pBufferObjContext, MainVol*(((PDSDRIVERBUFFER)(pBufferObjContext->pDriverContext->pPrimaryBuffer))->MainVol)/0xff);
			else
			{
				UpdateAll3DSound(pBufferObjContext, MainVol);
			}

		
		}
		else
			PosSound( (DS3DLISTENER*) (((PDSDRIVERBUFFER)(pBufferObjContext->pDriverContext->pPrimaryBuffer))->p3DProperty),
					(PDSDRIVERBUFFER) pBufferObjContext);
	}
    return DS_OK;
}


//*******************************************************************
//
// Update all sounds 3D, when mainvol of primary buffer is modified
//
//*******************************************************************
void UpdateAll3DSound(PDSDRIVERBUFFER pBufferObjContext, DWORD MainVol)
{
	// let's modify the position for all 3D buffers associated with the listener
	PDS3DLIST Cur3DBuffer=Buffer3DList;
	if (pBufferObjContext->DacState != DS_DAC_STATE_STOPPED)
		bufSetVol(pBufferObjContext, MainVol);
	while (Cur3DBuffer)
	{
		if (Cur3DBuffer->pBuffer3D->Instance3DId==pBufferObjContext->Instance3DId)
		{
			if (Cur3DBuffer->pBuffer3D->Enabled3D==1)
				modChangeParam(Cur3DBuffer->pBuffer3D, 0x56, MainVol);
			else
				//2D buffer
				if (Cur3DBuffer->pBuffer3D->Primary==0)
					modChangeParam(Cur3DBuffer->pBuffer3D, 0x56, (Cur3DBuffer->pBuffer3D->MainVol*MainVol)/0xff);
		}
		Cur3DBuffer=Cur3DBuffer->pNext;
	}
}


//*******************************************************************
//
// Call by API_DS_Vol (api.asm). Modify the MainVol of primary buffer
//	when waveOutSetVolume is called while DS is running 
//
//*******************************************************************
void	Change3DPrimaryVolume(DWORD PrimaryVolume)
{
	PDSDRIVERBUFFER pBufferObjContext=pPrimaryBufCtx;

	pBufferObjContext->MainVol=PrimaryVolume;
	UpdateAll3DSound(pBufferObjContext, PrimaryVolume);


}
//--------------------------------------------------------------------------
//
//  IDsDriverBuffer_SetPosition
//  Description:
//      Set the current playing position on a sound buffer.
//      This interface should be used by cards that support
//      hardware secondary buffers.
//      This function is not suported by the MSSNDSYS
//  Parameters:
//      pIDsDriverBuffer: buffer object context
//      dwNewPosition: requested new position
//  Return:
//      Error code
//  
//  
//--------------------------------------------------------------------------
HRESULT __stdcall
IDsDriverBuffer_SetPosition(
    IN PIDSDRIVERBUFFER pIDsDriverBuffer,
    IN DWORD dwNewPosition )
{
	DWORD Format;
	DWORD Start, Loop, End;		//relative offset

//	ULONG CurPos, rPage, rOffset, OffsetPos;
	PDSDRIVERBUFFER pBufferObjContext = (PDSDRIVERBUFFER)pIDsDriverBuffer;
    DP(("IDsDriverBuffer_SetPosition\n"));

    if(pBufferObjContext->DacState != DS_DAC_STATE_STOPPED)
	{
		/*CurPos = bufGetPos(pBufferObjContext); 
		// relative page
		rPage=(CurPos & 0xff)-((pBufferObjContext->BoardAddr)>>18);
		rOffset=(CurPos>>8) - (pBufferObjContext->BoardAddr & 0x3ffff);
        CurPos=(rPage<<18) + rOffset;		// Relative position
		if (dwNewPosition>=CurPos)
			OffsetPos=dwNewPosition-CurPos;
		else
			//OffsetPos=dwNewPosition-CurPos;*/
		
		// FIRST, we stop the voice
		bufCommand(pBufferObjContext,BUF_STOP);

		//then, we compute new start position
			Format=0;
		// if 8b stereo, mod are formated on 8bits
		if ((pBufferObjContext->nChannels==2) &&
			(pBufferObjContext->wBitsPerSample==8))
			Format=0x80;
		if (pBufferObjContext->DacState == DS_DAC_STATE_LOOPING)
		{
			End=pBufferObjContext->BufferSize-2;	// end of buffer -1
			Loop=pBufferObjContext->LoopPoint;		// beginning of buffer
		}
		else
		{
			End=pBufferObjContext->BufferSize+6;	// end 8 zero added at the end
			Loop=pBufferObjContext->BufferSize;		// beginning of buffer
		}
		
		if (dwNewPosition>=pBufferObjContext->PCBufferSize)
			dwNewPosition=0;
		Start=dwNewPosition;
		if (((pBufferObjContext->nChannels==1) && (pBufferObjContext->wBitsPerSample==16)) ||
			((pBufferObjContext->nChannels==2) && (pBufferObjContext->wBitsPerSample==8)))
		{			//16m or 8s
			Start/=2;		// position in words
			if (pBufferObjContext->DacState == DS_DAC_STATE_LOOPING)
				Loop/=2;
		}
		else if ((pBufferObjContext->nChannels==2) && (pBufferObjContext->wBitsPerSample==16))
		{
			//16s
			Start/=4;
			if (pBufferObjContext->DacState == DS_DAC_STATE_LOOPING)
				Loop/=4;
		}

		bufDefineMem(pBufferObjContext, Format, Start,Loop,End);
		modChangeParam(pBufferObjContext, 0x59, 0);		//reset volume on back speakers
		bufSetFreq(pBufferObjContext,pBufferObjContext->nSamplesPerSec);
		// if it is a 3D (secondary) buffer, let's make the positionning
		if ((pBufferObjContext->Enabled3D) && (!pBufferObjContext->Primary))
		{
			pBufferObjContext->VFront=0;
			pBufferObjContext->VBack=0;
			pBufferObjContext->VolMaster=0;
			pBufferObjContext->SamPitch=0;
			PosSound( (DS3DLISTENER*) (((PDSDRIVERBUFFER)(pBufferObjContext->pDriverContext->pPrimaryBuffer))->p3DProperty),
					pBufferObjContext );
		}
		else
			bufSetVol(pBufferObjContext,(pBufferObjContext->MainVol*(((PDSDRIVERBUFFER)(pBufferObjContext->pDriverContext->pPrimaryBuffer))->MainVol)/0xff));

		//then, restart to play
		bufCommand(pBufferObjContext,BUF_START);


		//bufSetPos(pBufferObjContext, OffsetPos>>18, OffsetPos & 0x3ffff);
	}
		
		pBufferObjContext->CurrentPos=dwNewPosition;
    return DS_OK;
}

//--------------------------------------------------------------------------
//
//  IDsDriverBuffer_GetPosition
//  Description:
//      Get the current playing position of a sound buffer.
//      The interface provides for sound devices that are not random
//      access. The device may return the next possible write possition
//      in pCurrentWriteCursor. It is most desirable to have the
//      WriteCursor = PlayCursor + 1. If you return a sample accurate
//      WriteCursor then you MUST return a sample accurate PlayCursor.
//      The MSSNDSYS card has no on-board buffering, so the current DMA
//      position is the PlayCursor. If a cards hardware has on-bord memory
//      then it must query the hardware for the PlayCursor.
//  Parameters:
//      pIDsDriverBuffer: buffer object context
//      pCurrentPlayCursor: The sample curretnly playing.
//      pCurrentWriteCursor: The next vaid writable location.
//  Return:
//      Error code
//  
//  
//--------------------------------------------------------------------------
HRESULT __stdcall
IDsDriverBuffer_GetPosition(
    IN PIDSDRIVERBUFFER pIDsDriverBuffer,
    OUT LPDWORD pCurrentPlayCursor,
    OUT LPDWORD pCurrentWriteCursor
    )
{
    PDSDRIVERBUFFER pBufferObjContext = (PDSDRIVERBUFFER)pIDsDriverBuffer;
    HRESULT success = DSERR_GENERIC;
    ULONG count, rPage, rOffset;

	    DP(("GetPos\n"));


    // are we playing? (are we doing a DMA transfer)
    if(pBufferObjContext->DacState != DS_DAC_STATE_STOPPED)
    {
        // get the raw count from the DMA controller.  this is a 0-based
        // count of the remaining dma transfers.
			count = bufGetPos(pBufferObjContext); 
				// byte 0=page;  bytes 1-3offset 

		// relative page
		rPage=(count & 0xff)-((pBufferObjContext->BoardAddr)>>18);
		rOffset=(count>>8) - (pBufferObjContext->BoardAddr & 0x3ffff);
        count=(rPage<<18) + rOffset;		// Relative position

		if (count>pBufferObjContext->BufferSize) 
			count=pBufferObjContext->BufferSize;

		if (((pBufferObjContext->nChannels==1) && (pBufferObjContext->wBitsPerSample==16)) ||
			((pBufferObjContext->nChannels==2) && (pBufferObjContext->wBitsPerSample==8)))
		{			//16m or 8s
			*pCurrentPlayCursor = count*2;
			*pCurrentWriteCursor = count*2;
		}
		else if ((pBufferObjContext->nChannels==2) && (pBufferObjContext->wBitsPerSample==16))
        {
			//16s
			*pCurrentPlayCursor = count*4;
			*pCurrentWriteCursor = count*4;
		}
		else
		{		//8m
			*pCurrentPlayCursor = count;
			*pCurrentWriteCursor = count;
		}

        success = DS_OK;
    }
    else
    {
        // If we're not playing, we should report a position of 0 and succeed
        *pCurrentPlayCursor = 0;
        *pCurrentWriteCursor = 0;
        success = DS_OK;
    }
    return success;
}

//--------------------------------------------------------------------------
//
//  IDsDriverBuffer_Play
//  Description:
//      Start playing the sound buffer
//  Parameters:
//      pIDsDriverBuffer: buffer object context
//      dwFlags: Flags.  Currently, the only defined flag is DSBPLAY_LOOPING.
//	    DSBPLAY_LOOPING: indicates that the buffer should continue playing
//		from the beginning of the buffer after reaching the end.  If
//		this flag is not specified, then play should stop at the end
//		of the buffer.  DSBPLAY_LOOPING will always be set for
//		primary buffers.
//  Return:
//      Error code
//  
//  
//--------------------------------------------------------------------------
HRESULT __stdcall
IDsDriverBuffer_Play(
    IN PIDSDRIVERBUFFER pIDsDriverBuffer,
    IN DWORD dwReserved1,
    IN DWORD dwReserved2,
    IN DWORD dwFlags
    )
{
    PDSDRIVERBUFFER pBufferObjContext = (PDSDRIVERBUFFER)pIDsDriverBuffer;
    HRESULT success = DSERR_GENERIC;
	DWORD CurrentPos;
    DP(("IDsDriverBuffer_Play\n"));

    // since this driver supports only primary buffers, DSBPLAY_LOOPING
    // should always be set in dwFlags.
  
	
	// if we're playing (and not looping), check if the buffer isn't finished
    if(pBufferObjContext->DacState == DS_DAC_STATE_PLAYING)
	{
		IDsDriverBuffer_GetPosition(pIDsDriverBuffer,&CurrentPos, &CurrentPos);
		if (CurrentPos >= pBufferObjContext->PCBufferSize)
		{
			IDsDriverBuffer_Stop(pIDsDriverBuffer);
			pBufferObjContext->CurrentPos=0;
		}
	}

    // make sure we are stopped or paused
    if(pBufferObjContext->DacState == DS_DAC_STATE_STOPPED)
    {
		if (dwFlags & DSBPLAY_LOOPING)
		{
		   pBufferObjContext->DacState = DS_DAC_STATE_LOOPING;
		}
		else
			pBufferObjContext->DacState = DS_DAC_STATE_PLAYING;
		if (StartPlay(pBufferObjContext, TRUE))
			success = DS_OK;
    
    }

    return success;
}

BOOL StartPlay(PDSDRIVERBUFFER pBufferObjContext, BOOL DsStart)
{
	DWORD Format;
	DWORD Start, Loop, End;		//relative offset

	DP(("go to alloc"));
	if (!AllocateSAMVoices(pBufferObjContext))
		return FALSE;
	DP(("ret from alloc"));
	Format=0;
	// if 8b stereo, mod are formated on 8bits
	if ((pBufferObjContext->nChannels==2) &&
		(pBufferObjContext->wBitsPerSample==8))
		Format=0x80;
	if (pBufferObjContext->DacState == DS_DAC_STATE_LOOPING)
	{
		End=pBufferObjContext->BufferSize;	// end of buffer +1
		Loop=pBufferObjContext->LoopPoint;		// beginning of buffer
	}
	else
	{
		int n;
		End=pBufferObjContext->BufferSize+6;	// end 8 zero added at the end
		Loop=pBufferObjContext->BufferSize;		// beginning of buffer

		// fill VoicesManager
		n=0;
		while ((VoicesManager[n].Alloc!=0) && (n<64))
			n++;
		if (n<64)		// must be always the case
		{
			VoicesManager[n].Alloc=1;
			VoicesManager[n].pDsDriverBuffer=pBufferObjContext;
			pBufferObjContext->VmNb=n;
		}
	}
	
	if (pBufferObjContext->CurrentPos>=pBufferObjContext->PCBufferSize)
		pBufferObjContext->CurrentPos=0;
	Start=pBufferObjContext->CurrentPos;
	if (((pBufferObjContext->nChannels==1) && (pBufferObjContext->wBitsPerSample==16)) ||
		((pBufferObjContext->nChannels==2) && (pBufferObjContext->wBitsPerSample==8)))
	{			//16m or 8s
		Start/=2;		// position in words
		if (pBufferObjContext->DacState == DS_DAC_STATE_LOOPING)
			Loop/=2;
	}
	else if ((pBufferObjContext->nChannels==2) && (pBufferObjContext->wBitsPerSample==16))
    {
		//16s
		Start/=4;
		if (pBufferObjContext->DacState == DS_DAC_STATE_LOOPING)
			Loop/=4;
	}

	if (DsStart)
		AddBufferInList(pBufferObjContext);			// add to the playing buffers list
	
    bufDefineMem(pBufferObjContext, Format, Start,Loop,End);
	modChangeParam(pBufferObjContext, 0x59, 0);		//reset volume on back speakers
	bufSetFreq(pBufferObjContext,pBufferObjContext->nSamplesPerSec);
	// if it is a 3D (secondary) buffer, let's make the positionning
	if ((pBufferObjContext->Enabled3D) && (!pBufferObjContext->Primary))
	{
		pBufferObjContext->VFront=0;
		pBufferObjContext->VBack=0;
		pBufferObjContext->VolMaster=0;
		pBufferObjContext->SamPitch=0;
		PosSound( (DS3DLISTENER*) (((PDSDRIVERBUFFER)(pBufferObjContext->pDriverContext->pPrimaryBuffer))->p3DProperty),
				pBufferObjContext );
	}
	else
		bufSetVol(pBufferObjContext,(pBufferObjContext->MainVol*(((PDSDRIVERBUFFER)(pBufferObjContext->pDriverContext->pPrimaryBuffer))->MainVol)/0xff));

	// start to play
	bufCommand(pBufferObjContext,BUF_START);

	DP(("Start"));

	return TRUE;
}


//--------------------------------------------------------------------------
//
//  IDsDriverBuffer_Stop
//  Description:
//      Stop the current sound buffer
//      The current position should be reset to 0.
//  Parameters:
//      pIDsDriverBuffer: buffer object context
//  Return:
//      Error code
//  
//  
//--------------------------------------------------------------------------
HRESULT __stdcall
IDsDriverBuffer_Stop(
    IN PIDSDRIVERBUFFER pIDsDriverBuffer
    )
{
    PDSDRIVERBUFFER pBufferObjContext = (PDSDRIVERBUFFER)pIDsDriverBuffer;
    HRESULT success = DSERR_GENERIC;

    DP(("IDsDriverBuffer_Stop\n"));

    // make sure we are playing or paused
    ASSERT( pBufferObjContext->DacState != DS_DAC_STATE_STOPPED );
    if( pBufferObjContext->DacState != DS_DAC_STATE_STOPPED )
    {
        //bufCommand(pBufferObjContext,BUF_STOP);

		if (pBufferObjContext->Primary)
			pBufferObjContext->CurrentPos=0;
		else
		{
			IDsDriverBuffer_GetPosition(pIDsDriverBuffer,&(pBufferObjContext->CurrentPos),
				&(pBufferObjContext->CurrentPos));
			if (pBufferObjContext->CurrentPos >= pBufferObjContext->PCBufferSize)
				pBufferObjContext->CurrentPos=pBufferObjContext->PCBufferSize;
		}

		// let's close the voice
		FreeRessources(pBufferObjContext);

		RemoveBufferFromList(pBufferObjContext);

		   // set the state of this buffer
	    pBufferObjContext->DacState = DS_DAC_STATE_STOPPED;

		
        success = DS_OK;
    }

    return success;
}

//****************************************************************************
//
//	BOOL TransferBlock(PDSDRIVERBUFFER pBufferObjContext,DWORD BoardBuffOffset, LPVOID PcSrcAddress, DWORD Size)
//		
//	This recursive function transfert a block from PC to board buffer.
//	This function call itself until having in page 64kW blocks.
//
//	Entry :	BoardDestAddress	: Address in SAM 9407 addressing space
//			PcSrcAddress		: Pointer on PC buffer to download
//			Size				: Size to transfer in word (board buffer size)
//*****************************************************************************
BOOL TransferBlock(PDSDRIVERBUFFER pBufferObjContext,DWORD BoardBuffOffset, LPVOID PcSrcAddress, DWORD BuffSize)
{
	LPBYTE SecBuf;
	WORD BaseAddress=pBufferObjContext->BaseAddress;
	BOOL Stereo=TRUE, s8Bits=FALSE;
	DWORD	PcBuffUsed;
	
	DWORD BoardDestAddress=pBufferObjContext->BoardAddr+BoardBuffOffset;
	DWORD BoardDestAddress2=pBufferObjContext->BoardAddr2+BoardBuffOffset;


	if (BuffSize==0)
		return TRUE;
	
	if (pBufferObjContext->nChannels==1)
		Stereo=FALSE;
	if (pBufferObjContext->wBitsPerSample==8)
		s8Bits=TRUE;


	// Is there a page change? 
	if ( ((BoardDestAddress + BuffSize -1) >> 16)!=(BoardDestAddress>>16) )
	{		// page change
		//sub block in page
		if (!TransferBlock(pBufferObjContext,BoardBuffOffset, PcSrcAddress, 0x10000-(BoardDestAddress & 0x0ffff)))
			return FALSE; 
		//complementary sub block
		PcBuffUsed=(0x10000-(BoardDestAddress & 0x0ffff))*2;
		if (s8Bits)	// 8 bits
			PcBuffUsed/=2;

		if (Stereo)		//stereo
			PcBuffUsed*=2;

		SecBuf=(LPBYTE) PcSrcAddress + PcBuffUsed;
		if (!TransferBlock(pBufferObjContext,(BoardDestAddress & 0xffff0000)+0x10000-pBufferObjContext->BoardAddr,
			SecBuf,BuffSize-(0x10000-(BoardDestAddress & 0x0ffff))))
			return FALSE;
		return TRUE;
	}
	// Is there a page change? (for the Right channel buffer if 16b stereo)
	if (Stereo && !s8Bits)
	{
		if ( ((BoardDestAddress2 + BuffSize -1) >> 16)!=(BoardDestAddress2>>16) )
		{		// page change
			//sub block in page
			if (!TransferBlock(pBufferObjContext,BoardBuffOffset, PcSrcAddress, 0x10000-(BoardDestAddress2 & 0x0ffff)))
				return FALSE; 
			//complementary sub block
			PcBuffUsed=(0x10000-(BoardDestAddress2 & 0x0ffff))*4;

			SecBuf=(LPBYTE) PcSrcAddress + PcBuffUsed;
			if (!TransferBlock(pBufferObjContext,(BoardDestAddress2 & 0xffff0000)+0x10000-pBufferObjContext->BoardAddr2,
				SecBuf,BuffSize-(0x10000-(BoardDestAddress2 & 0x0ffff))))
				return FALSE;
			return TRUE;
		}
	}
	// In page 
	if (BuffSize > 0x10000)
	{
		Trace_Out("Bad size!");
		return FALSE;
	}
	if (BuffSize > 0xf000)
	{
		/*if (DownLoad(BaseAddress,PcSrcAddress,(WORD) (BoardDestAddress>>16),(WORD) (BoardDestAddress & 0xffff), 0x7fff ))
		{
			Trace_Out("Erreur in download");
			return FALSE;
		}*/
		if(!MemWrFmt(BaseAddress,BoardDestAddress, PcSrcAddress,0xf000, 
		  Stereo, s8Bits, BoardDestAddress2))
		{
			Trace_Out("Erreur in download");
			return FALSE;
		}
		BuffSize-=0xf000;
		(BYTE *) PcSrcAddress+=2*0xf000;
		if (Stereo && !s8Bits)		//16b stereo
			(BYTE *) PcSrcAddress+=2*0xf000;
		if (!Stereo && s8Bits)
			(BYTE *) PcSrcAddress-=0xf000;

		BoardDestAddress+=0xf000;
		BoardDestAddress2+=0xf000;
	}

	if(!MemWrFmt(BaseAddress,BoardDestAddress, PcSrcAddress,(WORD)BuffSize, 
		  Stereo, s8Bits, BoardDestAddress2))
	{
		Trace_Out("Erreur in download");
		return FALSE;
	}

	/*if (DownLoad(BaseAddress,PcSrcAddress,(WORD) (BoardDestAddress>>16),(WORD) (BoardDestAddress & 0xffff), (WORD) (Size/2) ))
	{
		Trace_Out("Erreur in download");
		return FALSE;
	}*/
	return TRUE;
}


//****************************************************************************
//
//	BOOL FreeRessources(PDSDRIVERBUFFER pBufferObjContext)
//	
//	free on board buffers allocated for a voice
//
//*****************************************************************************

BOOL FreeRessources(PDSDRIVERBUFFER pBufferObjContext)
{
// we better be stopped if we are killing this buffer
        
	// close the voice
	bufCommand(pBufferObjContext,BUF_CLOSE);

	// if stereo, close the 2nd channel
	if (pBufferObjContext->nChannels==2)
	{
		dwVoiceAllocated--;
		if (pBufferObjContext->BufferSize>1024*256)
			dwVoiceAllocated--;
		VoiceFree[pBufferObjContext->wBufferNb2]=TRUE;
	}
	// refresh memory variable
	dwVoiceAllocated--;
	if (pBufferObjContext->BufferSize>1024*256)
		dwVoiceAllocated--;
	// ...and context buffer			
	VoiceFree[pBufferObjContext->wBufferNb]=TRUE;

	//Update Voices Manager
	if (pBufferObjContext->DacState != DS_DAC_STATE_LOOPING)
	{
		VoicesManager[pBufferObjContext->VmNb].Alloc=0;
		VoicesManager[pBufferObjContext->VmNb].pDsDriverBuffer=0;
	}
	return TRUE;
}

BOOL FreeBoardRessources(PDSDRIVERBUFFER pBufferObjContext)
{
/*	if (!DSvxdFree(pBufferObjContext, pBufferObjContext->BoardAddr -8))
	{
		Trace_Out("error in freeing board buffer");
	}*/
	BlockNb=256;
	if (memorFree(MemoryBlock, pBufferObjContext->BoardAddr -8))
	{
		Trace_Out("error in freeing board buffer");
	}
	else 
		dwSizeAlloc-=pBufferObjContext->BufferSize*2;

	// if 16b stereo, free the Right buffer
	if ((pBufferObjContext->nChannels==2) &&
		(pBufferObjContext->wBitsPerSample==16))
	{
		/*if (!DSvxdFree(pBufferObjContext, pBufferObjContext->BoardAddr2 - 8))
		{
			Trace_Out("error in freeing Right board buffer");
		}*/
		if (memorFree(MemoryBlock, pBufferObjContext->BoardAddr2 - 8))
		{
			Trace_Out("error in freeing Right board buffer");
		}
		else
			dwSizeAlloc-=pBufferObjContext->BufferSize*2;
	}	
	BlockNb=64;
	dwBufferOpenned--;
	return TRUE;
}


BOOL KillVoice()
{
	int n, Select=-1;
	DWORD CurrentPos, AlmostFinish=0xffffff;
	BOOL Found=FALSE;

	    DP(("KillVoice\n"));

	for (n=0; n<64; n++)
	{
		if (VoicesManager[n].Alloc)
		{
			IDsDriverBuffer_GetPosition((PIDSDRIVERBUFFER) (VoicesManager[n].pDsDriverBuffer),&CurrentPos, &CurrentPos);
			if (CurrentPos>=VoicesManager[n].pDsDriverBuffer->PCBufferSize)
			{
				IDsDriverBuffer_Stop((PIDSDRIVERBUFFER) VoicesManager[n].pDsDriverBuffer);
				VoicesManager[n].pDsDriverBuffer->CurrentPos=0;
				Found=TRUE;
			}
			else
			{
				if ( (VoicesManager[n].pDsDriverBuffer->PCBufferSize-CurrentPos) < AlmostFinish)
				{
					AlmostFinish=VoicesManager[n].pDsDriverBuffer->PCBufferSize-CurrentPos;
					Select=n;
				}
			}
		}
	}

	if (!Found)
	{
		if (Select!=-1)
		{
			IDsDriverBuffer_Stop((PIDSDRIVERBUFFER) VoicesManager[Select].pDsDriverBuffer);
			VoicesManager[Select].pDsDriverBuffer->CurrentPos=0;
		}
		else
			return FALSE;
	}
	return TRUE;
}

BOOL AllocateSAMVoices(PDSDRIVERBUFFER pBufferObjContext)
{
	WORD BufferNb, BufferNb2;
	DWORD VoiceNeeded=1;

	DP(("AllocateSAMVoices"));

	// How many voices are needed?
	if (pBufferObjContext->BufferSize>1024*256)
		VoiceNeeded*=2;
	if (pBufferObjContext->nChannels==2)			//########
		VoiceNeeded*=2;

		dwVoiceAllocated += VoiceNeeded;

			DP(("res?"));

	// Have enough sam ressource?
	while(dwVoiceAllocated + VoiceNeeded > nVoices)
	{
		// not enough ressources. We have to kill an old buffer.
		if (!KillVoice())
			return FALSE;
	}
	DP(("Y\n"));

	//  Let'allocate a buffer number
	BufferNb=0;
	while ((!VoiceFree[BufferNb]) && (BufferNb<ModNumber))
		BufferNb++;
	if (BufferNb==ModNumber)
	{
		DP(("Too many buffers allocated"));
		return FALSE;
	}
	pBufferObjContext->wBufferNb=BufferNb;

	VoiceFree[BufferNb]=FALSE;					// allocated
	BufferNb2=0;
	if (pBufferObjContext->nChannels==2)
	{				
		while ((!VoiceFree[BufferNb2]) && (BufferNb2<ModNumber))
			BufferNb2++;
		if (BufferNb2==ModNumber)
		{
			Trace_Out("Too many buffers allocated");
			return FALSE;		// error Out of Mem
		}
		VoiceFree[BufferNb2]=FALSE;					// allocated
	}
	pBufferObjContext->wBufferNb2=BufferNb2;

		DP(("Open"));

	bufCommand(pBufferObjContext, BUF_OPEN);

	return TRUE;
}

		
/*BOOL DSvxdFree(PDSDRIVERBUFFER pBufferObjContext, DWORD BufferStartAddr)		
{
	long MemMapAdd;
	MemBlDefTyp MemoryBlock[64];
	int n, DsBlock, BlockFree;
	NewBloc ToAdd;

	DWORD BufferEndAddr;

	BufferEndAddr=BufferStartAddr+pBufferObjContext->BufferSize+16;

	DP(("DSvxdFree"));

	__asm int 3

	// upload the mmt
	if ((MemMapAdd=GetMemMapAddress(pBufferObjContext->BaseAddress))==0xffffffff)				//API Dream called to Get Memory Mapping Table address
	{
		return FALSE;           
	}
	if (UpLoad(pBufferObjContext->BaseAddress,MemoryBlock,(WORD) (MemMapAdd>>16),(WORD) (MemMapAdd & 0xffff),BlockNb*3))	//API Dream called to Get Memory block definition
	{
		return FALSE;
	}		

	n=1;
	//find the block that contain the buffer to delete
	while ((n<BlockNb) && ((DWORD) MemoryBlock[n].Address<=BufferStartAddr))
			n++;
	if (n==BlockNb)
		return FALSE;

	DsBlock=n-1;		//block that contain the buffer to delete

	while (MemoryBlock[n].Type!=0xffff)
		n++;

	BlockFree=BlockNb-n-1;		// nb of remaining blocks in mmt

	//
	if ((BufferStartAddr!=(DWORD) MemoryBlock[DsBlock].Address) && 
		(BufferEndAddr!=(DWORD) MemoryBlock[DsBlock+1].Address))
	{
		// the buffer to delete is inside the block
		if (BlockFree<2)
			return FALSE;

		ToAdd.Address=BufferStartAddr;
		ToAdd.Type=1;	//free block
		if (!InsertMBlock(DsBlock+1,ToAdd,MemoryBlock))   // reserve midi mapping ext space
		{                  
			return FALSE;
		}
		ToAdd.Address=BufferEndAddr;
		ToAdd.Type=MemoryBlock[DsBlock].Type;	//remaining block
		if (!InsertMBlock(DsBlock+2,ToAdd,MemoryBlock))   // reserve midi mapping ext space
		{                  
			return FALSE;
		}
		goto exitFree;
	}

	if ((BufferStartAddr==(DWORD) MemoryBlock[DsBlock].Address) && 
		(BufferEndAddr==(DWORD) MemoryBlock[DsBlock+1].Address))
	{	// no more block 0x31
			//remove this block
		__asm int 3
		if (MemoryBlock[DsBlock-1].Type==1)
		{
			DeleteMBlock(DsBlock,MemoryBlock);
			if (MemoryBlock[DsBlock].Type==1)
				DeleteMBlock(DsBlock,MemoryBlock);
		}
		else if (MemoryBlock[DsBlock+1].Type==1)
		{
			DeleteMBlock(DsBlock+1,MemoryBlock);
			MemoryBlock[DsBlock].Type=1;
		}
		else
			MemoryBlock[DsBlock].Type=1;
		goto exitFree;
		
	}

	if (BufferStartAddr==(DWORD) MemoryBlock[DsBlock].Address)
	{
		MemoryBlock[DsBlock].Address=(long) BufferEndAddr;
		// in the begining of the block
		if (MemoryBlock[DsBlock-1].Type==1)
		{
			goto exitFree;
		}
		if (BlockFree<1)
			return FALSE;

		ToAdd.Address=BufferStartAddr;
		ToAdd.Type=1;	//free block
		if (!InsertMBlock(DsBlock,ToAdd,MemoryBlock))   // reserve midi mapping ext space
		{                  
			return FALSE;
		}
		goto exitFree;
	}

	// in the end of the block
	// in the begining of the block
	if (MemoryBlock[DsBlock+1].Type==1)
	{
		MemoryBlock[DsBlock+1].Address=(long) BufferStartAddr;
		goto exitFree;
	}
	if (BlockFree<1)
		return FALSE;

	ToAdd.Address=BufferStartAddr;
	ToAdd.Type=1;	//free block
	if (!InsertMBlock(DsBlock+1,ToAdd,MemoryBlock))   // reserve midi mapping ext space
	{                  
		return FALSE;
	}
exitFree:
	if (DownLoad(pBufferObjContext->BaseAddress,MemoryBlock,(WORD) (MemMapAdd>>16),(WORD) (MemMapAdd & 0xffff),BlockNb*3))	//API Dream called to Get Memory block definition
	{
		return FALSE;
	}
	
	return TRUE;
}*/


const IDSDRIVERBUFFERVTBL vtDsDriverBuffer = {
    IDsDriverBuffer_QueryInterface,
    IDsDriverBuffer_AddRef,
    IDsDriverBuffer_Release,
    IDsDriverBuffer_Lock,
    IDsDriverBuffer_Unlock,
    IDsDriverBuffer_SetFormat,
    IDsDriverBuffer_SetFrequency,
    IDsDriverBuffer_SetVolumePan,
    IDsDriverBuffer_SetPosition,
    IDsDriverBuffer_GetPosition,
    IDsDriverBuffer_Play,
    IDsDriverBuffer_Stop
};

