/**
 * \file asoundlib.h
 * \brief Application interface library for the ALSA driver
 * \author Jaroslav Kysela <perex@suse.cz>
 * \author Abramo Bagnara <abramo@alsa-project.org>
 * \author Takashi Iwai <tiwai@suse.de>
 * \date 1998-2001
 *
 * Application interface library for the ALSA driver
 *
 *
 *   This library is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License as
 *   published by the Free Software Foundation; either version 2 of
 *   the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifndef __ASOUNDLIB_H
#define __ASOUNDLIB_H

#include <sound/asound.h>
#include <sound/asoundef.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <assert.h>
#include <sys/poll.h>
#include <errno.h>

#ifndef ATTRIBUTE_UNUSED
#define ATTRIBUTE_UNUSED __attribute__ ((__unused__)) /**< don't print warning when attribute is not used */
#endif

#if __GNUC__ > 2 || (__GNUC__ == 2 && __GNUC_MINOR__ > 95)
#define SNDERR(...) snd_lib_error(__FILE__, __LINE__, __FUNCTION__, 0, __VA_ARGS__) /**< show sound error */
#define SYSERR(...) snd_lib_error(__FILE__, __LINE__, __FUNCTION__, errno, __VA_ARGS__) /**< show system error */
#else
#define SNDERR(args...) snd_lib_error(__FILE__, __LINE__, __FUNCTION__, 0, ##args) /**< show sound error */
#define SYSERR(args...) snd_lib_error(__FILE__, __LINE__, __FUNCTION__, errno, ##args) /**< show system error */
#endif

/*
 *  version.h
 */

#define SND_LIB_MAJOR		0 /**< major number of library version */
#define SND_LIB_MINOR		9 /**< minor number of library version */
#define SND_LIB_SUBMINOR	06 /**< subminor number of library version */
/** library version */
#define SND_LIB_VERSION		((SND_LIB_MAJOR<<16)|\
				 (SND_LIB_MINOR<<8)|\
				  SND_LIB_SUBMINOR)
/** library version (string) */
#define SND_LIB_VERSION_STR	"0.9.0beta6"

/**
 *  \defgroup Global Global defines
 *  Global defines
 *  \{
 */

#ifdef SNDRV_LITTLE_ENDIAN
#define SND_LITTLE_ENDIAN SNDRV_LITTLE_ENDIAN
#endif

#ifdef SNDRV_BIG_ENDIAN
#define SND_BIG_ENDIAN SNDRV_BIG_ENDIAN
#endif

/** \} */

/** Async notification client handler */
typedef struct _snd_async_handler snd_async_handler_t;

/** Async notification callback */
typedef void (*snd_async_callback_t)(snd_async_handler_t *handler);

int snd_async_add_handler(snd_async_handler_t **handler, int fd, 
			  snd_async_callback_t callback, void *private_data);
int snd_async_del_handler(snd_async_handler_t *handler);
int snd_async_handler_get_fd(snd_async_handler_t *handler);
void *snd_async_handler_get_callback_private(snd_async_handler_t *handler);
/**
 *  \defgroup Input Input Interface
 *  Input Interface
 *  \{
 */

/** Input handle */
typedef struct _snd_input snd_input_t;

/** Input type */
typedef enum _snd_input_type {
	/** Input from a stdio stream */
	SND_INPUT_STDIO,
	/** Input from a memory buffer */
	SND_INPUT_BUFFER,
} snd_input_type_t;

#ifdef __cplusplus
extern "C" {
#endif

int snd_input_stdio_open(snd_input_t **inputp, const char *file, const char *mode);
int snd_input_stdio_attach(snd_input_t **inputp, FILE *fp, int _close);
int snd_input_buffer_open(snd_input_t **inputp, const char *buffer, ssize_t size);
int snd_input_close(snd_input_t *input);
int snd_input_scanf(snd_input_t *input, const char *format, ...) __attribute__ ((format (scanf, 2, 3)));
char *snd_input_gets(snd_input_t *input, char *str, size_t size);
int snd_input_getc(snd_input_t *input);
int snd_input_ungetc(snd_input_t *input, int c);

#ifdef __cplusplus
}
#endif

/** \} */

/**
 *  \defgroup Output Output Interface
 *  Output Interface
 *  \{
 */

/** Output handle */
typedef struct _snd_output snd_output_t;

/** Output type */
typedef enum _snd_output_type {
	/** Output to a stdio stream */
	SND_OUTPUT_STDIO,
	/** Output to a memory buffer */
	SND_OUTPUT_BUFFER,
} snd_output_type_t;

#ifdef __cplusplus
extern "C" {
#endif

int snd_output_stdio_open(snd_output_t **outputp, const char *file, const char *mode);
int snd_output_stdio_attach(snd_output_t **outputp, FILE *fp, int _close);
int snd_output_buffer_open(snd_output_t **outputp);
size_t snd_output_buffer_string(snd_output_t *output, char **buf);
int snd_output_close(snd_output_t *output);
int snd_output_printf(snd_output_t *output, const char *format, ...) __attribute__ ((format (printf, 2, 3)));
int snd_output_puts(snd_output_t *output, const char *str);
int snd_output_putc(snd_output_t *output, int c);
int snd_output_flush(snd_output_t *output);

#ifdef __cplusplus
}
#endif

/** \} */

/**
 *  \defgroup Error Error handling
 *  Error handling
 *  \{
 */

#define SND_ERROR_BEGIN				500000			/**< begin boundary of sound error codes */
#define SND_ERROR_INCOMPATIBLE_VERSION		(SND_ERROR_BEGIN+0)	/**< protocol is not compatible */

#ifdef __cplusplus
extern "C" {
#endif

const char *snd_strerror(int errnum);

/**
 * \brief Error handler
 * \param file File name
 * \param line Line number
 * \param function Function name
 * \param err errno value (or 0 if not relevant)
 * \param fmt printf(3) format
 * \param ... printf(3) arguments
 */
typedef void (snd_lib_error_handler_t)(const char *file, int line, const char *function, int err, const char *fmt, ...) /* __attribute__ ((format (printf, 5, 6))) */;
extern snd_lib_error_handler_t *snd_lib_error;
extern int snd_lib_error_set_handler(snd_lib_error_handler_t *handler);

#ifdef __cplusplus
}
#endif

/** \} */

/**
 *  \defgroup Config Configuration Interface
 *  Configuration Interface
 *  \{
 */

/** Config node type */
typedef enum _snd_config_type {
	/** Integer number */
        SND_CONFIG_TYPE_INTEGER,
	/** Real number */
        SND_CONFIG_TYPE_REAL,
	/** Character string */
        SND_CONFIG_TYPE_STRING,
	/** Compound */
	SND_CONFIG_TYPE_COMPOUND,
} snd_config_type_t;

/** Config node handle */
typedef struct _snd_config snd_config_t;
/** Config compound iterator */
typedef struct _snd_config_iterator *snd_config_iterator_t;

#ifdef __cplusplus
extern "C" {
#endif

int snd_config_top(snd_config_t **config);

int snd_config_load(snd_config_t *config, snd_input_t *in);
int snd_config_save(snd_config_t *config, snd_output_t *out);

int snd_config_search(snd_config_t *config, const char *key,
		      snd_config_t **result);
int snd_config_searchv(snd_config_t *config, 
		       snd_config_t **result, ...);
int snd_config_search_definition(snd_config_t *config,
				 const char *base, const char *key,
				 snd_config_t **result);

int snd_config_expand(snd_config_t *config, snd_config_t *root,
		      const char *args, void *private_data,
		      snd_config_t **result);
int snd_config_evaluate(snd_config_t *config, snd_config_t *root,
			void *private_data, snd_config_t **result);

int snd_config_add(snd_config_t *config, snd_config_t *leaf);
int snd_config_delete(snd_config_t *config);
int snd_config_copy(snd_config_t **dst, snd_config_t *src);

int snd_config_make(snd_config_t **config, const char *key,
		    snd_config_type_t type);
int snd_config_make_integer(snd_config_t **config, const char *key);
int snd_config_make_real(snd_config_t **config, const char *key);
int snd_config_make_string(snd_config_t **config, const char *key);
int snd_config_make_compound(snd_config_t **config, const char *key, int join);

int snd_config_set_id(snd_config_t *config, const char *id);
int snd_config_set_integer(snd_config_t *config, long value);
int snd_config_set_real(snd_config_t *config, double value);
int snd_config_set_string(snd_config_t *config, const char *value);
int snd_config_set_ascii(snd_config_t *config, const char *ascii);
int snd_config_get_integer(snd_config_t *config, long *value);
int snd_config_get_real(snd_config_t *config, double *value);
int snd_config_get_string(snd_config_t *config, const char **value);
int snd_config_get_ascii(snd_config_t *config, char **value);

snd_config_iterator_t snd_config_iterator_first(snd_config_t *node);
snd_config_iterator_t snd_config_iterator_next(snd_config_iterator_t iterator);
snd_config_iterator_t snd_config_iterator_end(snd_config_t *node);
snd_config_t *snd_config_iterator_entry(snd_config_iterator_t iterator);

/** Helper for compound config node leaves traversal
 * \param pos Current node iterator
 * \param next Next node iterator
 * \param node Compound config node
 *
 * This macro is designed to permit the removal of current node.
 */
#define snd_config_for_each(pos, next, node) \
	for (pos = snd_config_iterator_first(node), next = snd_config_iterator_next(pos); pos != snd_config_iterator_end(node); pos = next, next = snd_config_iterator_next(pos))

snd_config_type_t snd_config_get_type(snd_config_t *config);
const char *snd_config_get_id(snd_config_t *config);

extern snd_config_t *snd_config;
int snd_config_update(void);

/* Misc functions */

int snd_config_get_bool_ascii(const char *ascii);
int snd_config_get_bool(snd_config_t *conf);
int snd_config_get_ctl_iface_ascii(const char *ascii);
int snd_config_get_ctl_iface(snd_config_t *conf);

int snd_config_refer_load(snd_config_t **dst, char **name,
			  snd_config_t *root, snd_config_t *config);

#ifdef __cplusplus
}
#endif

/** \} */

/****************************************************************************
 *                                                                          *
 *                                pcm.h                                     *
 *                        Digital Audio Interface                           *
 *                                                                          *
 ****************************************************************************/

/**
 *  \defgroup PCM PCM Interface
 *  The PCM Interface.
 *  \{
 */

/** PCM generic info container */
typedef struct _snd_pcm_info snd_pcm_info_t;
/** PCM hardware configuration space container */
typedef struct _snd_pcm_hw_params snd_pcm_hw_params_t;
/** PCM software configuration container */
typedef struct _snd_pcm_sw_params snd_pcm_sw_params_t;
/** PCM status container */
 typedef struct _snd_pcm_status snd_pcm_status_t;
/** PCM access types mask */
typedef struct _snd_pcm_access_mask snd_pcm_access_mask_t;
/** PCM formats mask */
typedef struct _snd_pcm_format_mask snd_pcm_format_mask_t;
/** PCM subformats mask */
typedef struct _snd_pcm_subformat_mask snd_pcm_subformat_mask_t;

/** PCM class */
typedef enum _snd_pcm_class {
	/** standard device */
	SND_PCM_CLASS_GENERIC = SNDRV_PCM_CLASS_GENERIC,
	/** multichannel device */
	SND_PCM_CLASS_MULTI = SNDRV_PCM_CLASS_MULTI,
	/** software modem device */
	SND_PCM_CLASS_MODEM = SNDRV_PCM_CLASS_MODEM,
	/** digitizer device */
	SND_PCM_CLASS_DIGITIZER = SNDRV_PCM_CLASS_DIGITIZER,
	SND_PCM_CLASS_LAST = SNDRV_PCM_CLASS_LAST,
} snd_pcm_class_t;

/** PCM subclass */
typedef enum _snd_pcm_subclass {
	/** subdevices are mixed together */
	SND_PCM_SUBCLASS_GENERIC_MIX = SNDRV_PCM_SUBCLASS_GENERIC_MIX,
	/** multichannel subdevices are mixed together */
	SND_PCM_SUBCLASS_MULTI_MIX = SNDRV_PCM_SUBCLASS_MULTI_MIX,
	SND_PCM_SUBCLASS_LAST = SNDRV_PCM_SUBCLASS_LAST,
} snd_pcm_subclass_t;

/** PCM stream (direction) */
typedef enum _snd_pcm_stream {
	/** Playback stream */
	SND_PCM_STREAM_PLAYBACK = SNDRV_PCM_STREAM_PLAYBACK,
	/** Capture stream */
	SND_PCM_STREAM_CAPTURE = SNDRV_PCM_STREAM_CAPTURE,
	SND_PCM_STREAM_LAST = SNDRV_PCM_STREAM_LAST,
} snd_pcm_stream_t;

/** PCM access type */
typedef enum _snd_pcm_access {
	/** mmap access with simple interleaved channels */
	SND_PCM_ACCESS_MMAP_INTERLEAVED = SNDRV_PCM_ACCESS_MMAP_INTERLEAVED,
	/** mmap access with simple non interleaved channels */
	SND_PCM_ACCESS_MMAP_NONINTERLEAVED = SNDRV_PCM_ACCESS_MMAP_NONINTERLEAVED,
	/** mmap access with complex placement */
	SND_PCM_ACCESS_MMAP_COMPLEX = SNDRV_PCM_ACCESS_MMAP_COMPLEX,
	/** snd_pcm_readi/snd_pcm_writei access */
	SND_PCM_ACCESS_RW_INTERLEAVED = SNDRV_PCM_ACCESS_RW_INTERLEAVED,
	/** snd_pcm_readn/snd_pcm_writen access */
	SND_PCM_ACCESS_RW_NONINTERLEAVED = SNDRV_PCM_ACCESS_RW_NONINTERLEAVED,
	SND_PCM_ACCESS_LAST = SNDRV_PCM_ACCESS_LAST,
} snd_pcm_access_t;

/** PCM sample format */
typedef enum _snd_pcm_format {
	/** Unknown */
	SND_PCM_FORMAT_UNKNOWN = -1,
	/** Signed 8 bit */
	SND_PCM_FORMAT_S8 = SNDRV_PCM_FORMAT_S8,
	/** Unsigned 8 bit */
	SND_PCM_FORMAT_U8 = SNDRV_PCM_FORMAT_U8,
	/** Signed 16 bit Little Endian */
	SND_PCM_FORMAT_S16_LE = SNDRV_PCM_FORMAT_S16_LE,
	/** Signed 16 bit Big Endian */
	SND_PCM_FORMAT_S16_BE = SNDRV_PCM_FORMAT_S16_BE,
	/** Unsigned 16 bit Little Endian */
	SND_PCM_FORMAT_U16_LE = SNDRV_PCM_FORMAT_U16_LE,
	/** Unsigned 16 bit Big Endian */
	SND_PCM_FORMAT_U16_BE = SNDRV_PCM_FORMAT_U16_BE,
	/** Signed 24 bit Little Endian */
	SND_PCM_FORMAT_S24_LE = SNDRV_PCM_FORMAT_S24_LE,
	/** Signed 24 bit Big Endian */
	SND_PCM_FORMAT_S24_BE = SNDRV_PCM_FORMAT_S24_BE,
	/** Unsigned 24 bit Little Endian */
	SND_PCM_FORMAT_U24_LE = SNDRV_PCM_FORMAT_U24_LE,
	/** Unsigned 24 bit Big Endian */
	SND_PCM_FORMAT_U24_BE = SNDRV_PCM_FORMAT_U24_BE,
	/** Signed 32 bit Little Endian */
	SND_PCM_FORMAT_S32_LE = SNDRV_PCM_FORMAT_S32_LE,
	/** Signed 32 bit Big Endian */
	SND_PCM_FORMAT_S32_BE = SNDRV_PCM_FORMAT_S32_BE,
	/** Unsigned 32 bit Little Endian */
	SND_PCM_FORMAT_U32_LE = SNDRV_PCM_FORMAT_U32_LE,
	/** Unsigned 32 bit Big Endian */
	SND_PCM_FORMAT_U32_BE = SNDRV_PCM_FORMAT_U32_BE,
	/** Float 32 bit Little Endian */
	SND_PCM_FORMAT_FLOAT_LE = SNDRV_PCM_FORMAT_FLOAT_LE,
	/** Float 32 bit Big Endian */
	SND_PCM_FORMAT_FLOAT_BE = SNDRV_PCM_FORMAT_FLOAT_BE,
	/** Float 64 bit Little Endian */
	SND_PCM_FORMAT_FLOAT64_LE = SNDRV_PCM_FORMAT_FLOAT64_LE,
	/** Float 64 bit Big Endian */
	SND_PCM_FORMAT_FLOAT64_BE = SNDRV_PCM_FORMAT_FLOAT64_BE,
	/** IEC-958 Little Endian */
	SND_PCM_FORMAT_IEC958_SUBFRAME_LE = SNDRV_PCM_FORMAT_IEC958_SUBFRAME_LE,
	/** IEC-958 Big Endian */
	SND_PCM_FORMAT_IEC958_SUBFRAME_BE = SNDRV_PCM_FORMAT_IEC958_SUBFRAME_BE,
	/** Mu-Law */
	SND_PCM_FORMAT_MU_LAW = SNDRV_PCM_FORMAT_MU_LAW,
	/** A-Law */
	SND_PCM_FORMAT_A_LAW = SNDRV_PCM_FORMAT_A_LAW,
	/** Ima-ADPCM */
	SND_PCM_FORMAT_IMA_ADPCM = SNDRV_PCM_FORMAT_IMA_ADPCM,
	/** MPEG */
	SND_PCM_FORMAT_MPEG = SNDRV_PCM_FORMAT_MPEG,
	/** GSM */
	SND_PCM_FORMAT_GSM = SNDRV_PCM_FORMAT_GSM,
	/** Special */
	SND_PCM_FORMAT_SPECIAL = SNDRV_PCM_FORMAT_SPECIAL,
	SND_PCM_FORMAT_LAST = SNDRV_PCM_FORMAT_LAST,
	/** Signed 16 bit CPU endian */
	SND_PCM_FORMAT_S16 = SNDRV_PCM_FORMAT_S16,
	/** Unsigned 16 bit CPU endian */
	SND_PCM_FORMAT_U16 = SNDRV_PCM_FORMAT_U16,
	/** Signed 24 bit CPU endian */
	SND_PCM_FORMAT_S24 = SNDRV_PCM_FORMAT_S24,
	/** Unsigned 24 bit CPU endian */
	SND_PCM_FORMAT_U24 = SNDRV_PCM_FORMAT_U24,
	/** Signed 32 bit CPU endian */
	SND_PCM_FORMAT_S32 = SNDRV_PCM_FORMAT_S32,
	/** Unsigned 32 bit CPU endian */
	SND_PCM_FORMAT_U32 = SNDRV_PCM_FORMAT_U32,
	/** Float 32 bit CPU endian */
	SND_PCM_FORMAT_FLOAT = SNDRV_PCM_FORMAT_FLOAT,
	/** Float 64 bit CPU endian */
	SND_PCM_FORMAT_FLOAT64 = SNDRV_PCM_FORMAT_FLOAT64,
	/** IEC-958 CPU Endian */
	SND_PCM_FORMAT_IEC958_SUBFRAME = SNDRV_PCM_FORMAT_IEC958_SUBFRAME,
} snd_pcm_format_t;

/** PCM sample subformat */
typedef enum _snd_pcm_subformat {
	/** Standard */
	SND_PCM_SUBFORMAT_STD = SNDRV_PCM_SUBFORMAT_STD,
	SND_PCM_SUBFORMAT_LAST = SNDRV_PCM_SUBFORMAT_LAST,
} snd_pcm_subformat_t;

/** PCM state */
typedef enum _snd_pcm_state {
	/** Open */
	SND_PCM_STATE_OPEN = SNDRV_PCM_STATE_OPEN,
	/** Setup installed */ 
	SND_PCM_STATE_SETUP = SNDRV_PCM_STATE_SETUP,
	/** Ready to start */
	SND_PCM_STATE_PREPARED = SNDRV_PCM_STATE_PREPARED,
	/** Running */
	SND_PCM_STATE_RUNNING = SNDRV_PCM_STATE_RUNNING,
	/** Stopped: underrun (playback) or overrun (capture) detected */
	SND_PCM_STATE_XRUN = SNDRV_PCM_STATE_XRUN,
	/** Draining: running (playback) or stopped (capture) */
	SND_PCM_STATE_DRAINING = SNDRV_PCM_STATE_DRAINING,
	/** Paused */
	SND_PCM_STATE_PAUSED = SNDRV_PCM_STATE_PAUSED,
	SND_PCM_STATE_LAST = SNDRV_PCM_STATE_LAST,
} snd_pcm_state_t;

/** PCM start mode */
typedef enum _snd_pcm_start {
	/** Automatic start on data read/write */
	SND_PCM_START_DATA,
	/** Explicit start */
	SND_PCM_START_EXPLICIT,
	SND_PCM_START_LAST,
} snd_pcm_start_t;

/** PCM xrun mode */
typedef enum _snd_pcm_xrun {
	/** Xrun detection disabled */
	SND_PCM_XRUN_NONE,
	/** Stop on xrun detection */
	SND_PCM_XRUN_STOP,
	SND_PCM_XRUN_LAST,
} snd_pcm_xrun_t;

/** PCM timestamp mode */
typedef enum _snd_pcm_tstamp {
	/** No timestamp */
	SND_PCM_TSTAMP_NONE = SNDRV_PCM_TSTAMP_NONE,
	/** Update mmap'ed timestamp */
	SND_PCM_TSTAMP_MMAP = SNDRV_PCM_TSTAMP_MMAP,
	SND_PCM_TSTAMP_LAST = SNDRV_PCM_TSTAMP_LAST,
} snd_pcm_tstamp_t;

/** Unsigned frames quantity */
typedef sndrv_pcm_uframes_t snd_pcm_uframes_t;
/** Signed frames quantity */
typedef sndrv_pcm_sframes_t snd_pcm_sframes_t;
/** Timestamp */
typedef struct timeval snd_timestamp_t;

/** device accepts mmaped access */
#define SND_PCM_INFO_MMAP SNDRV_PCM_INFO_MMAP
/** device accepts  mmaped access with sample resolution */
#define SND_PCM_INFO_MMAP_VALID SNDRV_PCM_INFO_MMAP_VALID
/** device is doing double buffering */
#define SND_PCM_INFO_DOUBLE SNDRV_PCM_INFO_DOUBLE
/** device transfers samples in batch */
#define SND_PCM_INFO_BATCH SNDRV_PCM_INFO_BATCH
/** device accepts interleaved samples */
#define SND_PCM_INFO_INTERLEAVED SNDRV_PCM_INFO_INTERLEAVED
/** device accepts non-interleaved samples */
#define SND_PCM_INFO_NONINTERLEAVED SNDRV_PCM_INFO_NONINTERLEAVED
/** device accepts complex sample organization */
#define SND_PCM_INFO_COMPLEX SNDRV_PCM_INFO_COMPLEX
/** device is capable block transfers */
#define SND_PCM_INFO_BLOCK_TRANSFER SNDRV_PCM_INFO_BLOCK_TRANSFER
/** device can detect DAC/ADC overrange */
#define SND_PCM_INFO_OVERRANGE SNDRV_PCM_INFO_OVERRANGE
/** device is capable to pause */
#define SND_PCM_INFO_PAUSE SNDRV_PCM_INFO_PAUSE
/** device can do only half duplex */
#define SND_PCM_INFO_HALF_DUPLEX SNDRV_PCM_INFO_HALF_DUPLEX
/** device can do only joint duplex (same parameters) */
#define SND_PCM_INFO_JOINT_DUPLEX SNDRV_PCM_INFO_JOINT_DUPLEX
/** device can do a kind of synchronized start */
#define SND_PCM_INFO_SYNC_START SNDRV_PCM_INFO_SYNC_START

/** Non blocking mode \hideinitializer */
#define SND_PCM_NONBLOCK		0x0001
/** Async notification \hideinitializer */
#define SND_PCM_ASYNC			0x0002

/** PCM handle */
typedef struct _snd_pcm snd_pcm_t;

/** PCM type */
enum _snd_pcm_type {
	/** Kernel level PCM */
	SND_PCM_TYPE_HW,
	/** Hooked PCM */
	SND_PCM_TYPE_HOOKS,
	/** One ore more linked PCM with exclusive access to selected
	    channels */
	SND_PCM_TYPE_MULTI,
	/** File writing plugin */
	SND_PCM_TYPE_FILE,
	/** Null endpoint PCM */
	SND_PCM_TYPE_NULL,
	/** Shared memory client PCM */
	SND_PCM_TYPE_SHM,
	/** INET client PCM (not yet implemented) */
	SND_PCM_TYPE_INET,
	/** Copying plugin */
	SND_PCM_TYPE_COPY,
	/** Linear format conversion PCM */
	SND_PCM_TYPE_LINEAR,
	/** A-Law format conversion PCM */
	SND_PCM_TYPE_ALAW,
	/** Mu-Law format conversion PCM */
	SND_PCM_TYPE_MULAW,
	/** IMA-ADPCM format conversion PCM */
	SND_PCM_TYPE_ADPCM,
	/** Rate conversion PCM */
	SND_PCM_TYPE_RATE,
	/** Attenuated static route PCM */
	SND_PCM_TYPE_ROUTE,
	/** Format adjusted PCM */
	SND_PCM_TYPE_PLUG,
	/** Sharing PCM */
	SND_PCM_TYPE_SHARE,
	/** Meter plugin */
	SND_PCM_TYPE_METER,
	/** Mixing PCM */
	SND_PCM_TYPE_MIX,
	/** Attenuated dynamic route PCM (not yet implemented) */
	SND_PCM_TYPE_DROUTE,
	/** Loopback server plugin (not yet implemented) */
	SND_PCM_TYPE_LBSERVER,
	/** Surround plugin */
	SND_PCM_TYPE_SURROUND,
};

/** PCM type */
typedef enum _snd_pcm_type snd_pcm_type_t;

/** PCM area specification */
typedef struct _snd_pcm_channel_area {
	/** base address of channel samples */
	void *addr;
	/** offset to first sample in bits */
	unsigned int first;
	/** samples distance in bits */
	unsigned int step;
} snd_pcm_channel_area_t;

/** #SND_PCM_TYPE_METER scope handle */
typedef struct _snd_pcm_scope snd_pcm_scope_t;

#ifdef __cplusplus
extern "C" {
#endif

int snd_pcm_open(snd_pcm_t **pcm, const char *name, 
		 snd_pcm_stream_t stream, int mode);

int snd_pcm_close(snd_pcm_t *pcm);
int snd_pcm_poll_descriptors_count(snd_pcm_t *pcm);
int snd_pcm_poll_descriptors(snd_pcm_t *pcm, struct pollfd *pfds, unsigned int space);
int snd_pcm_nonblock(snd_pcm_t *pcm, int nonblock);
int snd_async_add_pcm_handler(snd_async_handler_t **handler, snd_pcm_t *pcm, 
			      snd_async_callback_t callback, void *private_data);
snd_pcm_t *snd_async_handler_get_pcm(snd_async_handler_t *handler);
int snd_pcm_info(snd_pcm_t *pcm, snd_pcm_info_t *info);
int snd_pcm_hw_params(snd_pcm_t *pcm, snd_pcm_hw_params_t *params);
int snd_pcm_hw_free(snd_pcm_t *pcm);
int snd_pcm_sw_params(snd_pcm_t *pcm, snd_pcm_sw_params_t *params);
int snd_pcm_prepare(snd_pcm_t *pcm);
int snd_pcm_reset(snd_pcm_t *pcm);
int snd_pcm_status(snd_pcm_t *pcm, snd_pcm_status_t *status);
int snd_pcm_start(snd_pcm_t *pcm);
int snd_pcm_drop(snd_pcm_t *pcm);
int snd_pcm_drain(snd_pcm_t *pcm);
int snd_pcm_pause(snd_pcm_t *pcm, int enable);
snd_pcm_state_t snd_pcm_state(snd_pcm_t *pcm);
int snd_pcm_delay(snd_pcm_t *pcm, snd_pcm_sframes_t *delayp);
snd_pcm_sframes_t snd_pcm_rewind(snd_pcm_t *pcm, snd_pcm_uframes_t frames);
snd_pcm_sframes_t snd_pcm_writei(snd_pcm_t *pcm, const void *buffer, snd_pcm_uframes_t size);
snd_pcm_sframes_t snd_pcm_readi(snd_pcm_t *pcm, void *buffer, snd_pcm_uframes_t size);
snd_pcm_sframes_t snd_pcm_writen(snd_pcm_t *pcm, void **bufs, snd_pcm_uframes_t size);
snd_pcm_sframes_t snd_pcm_readn(snd_pcm_t *pcm, void **bufs, snd_pcm_uframes_t size);

int snd_pcm_dump_hw_setup(snd_pcm_t *pcm, snd_output_t *out);
int snd_pcm_dump_sw_setup(snd_pcm_t *pcm, snd_output_t *out);
int snd_pcm_dump_setup(snd_pcm_t *pcm, snd_output_t *out);
int snd_pcm_dump(snd_pcm_t *pcm, snd_output_t *out);
int snd_pcm_link(snd_pcm_t *pcm1, snd_pcm_t *pcm2);
int snd_pcm_unlink(snd_pcm_t *pcm);

int snd_pcm_wait(snd_pcm_t *pcm, int timeout);
snd_pcm_sframes_t snd_pcm_avail_update(snd_pcm_t *pcm);
const char *snd_pcm_name(snd_pcm_t *pcm);
snd_pcm_type_t snd_pcm_type(snd_pcm_t *pcm);
snd_pcm_stream_t snd_pcm_stream(snd_pcm_t *pcm);

/* HW params */
int snd_pcm_hw_params_any(snd_pcm_t *pcm, snd_pcm_hw_params_t *params);

int snd_pcm_hw_params_get_rate_numden(const snd_pcm_hw_params_t *params,
				      unsigned int *rate_num,
				      unsigned int *rate_den);
int snd_pcm_hw_params_get_sbits(const snd_pcm_hw_params_t *params);
int snd_pcm_hw_params_get_fifo_size(const snd_pcm_hw_params_t *params);
int snd_pcm_hw_params_dump(snd_pcm_hw_params_t *params, snd_output_t *out);

#if 0
typedef struct _snd_pcm_hw_strategy snd_pcm_hw_strategy_t;

/* choices need to be sorted on ascending badness */
typedef struct _snd_pcm_hw_strategy_simple_choices_list {
	unsigned int value;
	unsigned int badness;
} snd_pcm_hw_strategy_simple_choices_list_t;

int snd_pcm_hw_params_strategy(snd_pcm_t *pcm, snd_pcm_hw_params_t *params,
			       const snd_pcm_hw_strategy_t *strategy,
			       unsigned int badness_min,
			       unsigned int badness_max);

void snd_pcm_hw_strategy_free(snd_pcm_hw_strategy_t *strategy);
int snd_pcm_hw_strategy_simple(snd_pcm_hw_strategy_t **strategyp,
			       unsigned int badness_min,
			       unsigned int badness_max);
int snd_pcm_hw_params_try_explain_failure(snd_pcm_t *pcm,
					  snd_pcm_hw_params_t *fail,
					  snd_pcm_hw_params_t *success,
					  unsigned int depth,
					  snd_output_t *out);

#endif

int snd_pcm_sw_params_current(snd_pcm_t *pcm, snd_pcm_sw_params_t *params);
int snd_pcm_sw_params_dump(snd_pcm_sw_params_t *params, snd_output_t *out);

int snd_pcm_status_dump(snd_pcm_status_t *status, snd_output_t *out);

/* mmap */
int snd_pcm_mmap_begin(snd_pcm_t *pcm,
		       const snd_pcm_channel_area_t **areas,
		       snd_pcm_uframes_t *offset,
		       snd_pcm_uframes_t *frames);
int snd_pcm_mmap_commit(snd_pcm_t *pcm, snd_pcm_uframes_t offset,
			snd_pcm_uframes_t frames);

const char *snd_pcm_stream_name(snd_pcm_stream_t stream);
const char *snd_pcm_access_name(snd_pcm_access_t _access);
const char *snd_pcm_format_name(snd_pcm_format_t format);
const char *snd_pcm_format_description(snd_pcm_format_t format);
const char *snd_pcm_subformat_name(snd_pcm_subformat_t subformat);
const char *snd_pcm_subformat_description(snd_pcm_subformat_t subformat);
snd_pcm_format_t snd_pcm_format_value(const char* name);
const char *snd_pcm_start_mode_name(snd_pcm_start_t mode);
const char *snd_pcm_xrun_mode_name(snd_pcm_xrun_t mode);
const char *snd_pcm_tstamp_mode_name(snd_pcm_tstamp_t mode);
const char *snd_pcm_state_name(snd_pcm_state_t state);

int snd_pcm_area_silence(const snd_pcm_channel_area_t *dst_channel, snd_pcm_uframes_t dst_offset,
			 unsigned int samples, snd_pcm_format_t format);
int snd_pcm_areas_silence(const snd_pcm_channel_area_t *dst_channels, snd_pcm_uframes_t dst_offset,
			  unsigned int channels, snd_pcm_uframes_t frames, snd_pcm_format_t format);
int snd_pcm_area_copy(const snd_pcm_channel_area_t *dst_channel, snd_pcm_uframes_t dst_offset,
		      const snd_pcm_channel_area_t *src_channel, snd_pcm_uframes_t src_offset,
		      unsigned int samples, snd_pcm_format_t format);
int snd_pcm_areas_copy(const snd_pcm_channel_area_t *dst_channels, snd_pcm_uframes_t dst_offset,
		       const snd_pcm_channel_area_t *src_channels, snd_pcm_uframes_t src_offset,
		       unsigned int channels, snd_pcm_uframes_t frames, snd_pcm_format_t format);

snd_pcm_sframes_t snd_pcm_bytes_to_frames(snd_pcm_t *pcm, ssize_t bytes);
ssize_t snd_pcm_frames_to_bytes(snd_pcm_t *pcm, snd_pcm_sframes_t frames);
int snd_pcm_bytes_to_samples(snd_pcm_t *pcm, ssize_t bytes);
ssize_t snd_pcm_samples_to_bytes(snd_pcm_t *pcm, int samples);

/** #SND_PCM_TYPE_METER scope functions */
typedef struct _snd_pcm_scope_ops {
	/** \brief Enable and prepare it using current params
	 * \param scope scope handle
	 */
	int (*enable)(snd_pcm_scope_t *scope);
	/** \brief Disable
	 * \param scope scope handle
	 */
	void (*disable)(snd_pcm_scope_t *scope);
	/** \brief PCM has been started
	 * \param scope scope handle
	 */
	void (*start)(snd_pcm_scope_t *scope);
	/** \brief PCM has been stopped
	 * \param scope scope handle
	 */
	void (*stop)(snd_pcm_scope_t *scope);
	/** \brief New frames are present
	 * \param scope scope handle
	 */
	void (*update)(snd_pcm_scope_t *scope);
	/** \brief Reset status
	 * \param scope scope handle
	 */
	void (*reset)(snd_pcm_scope_t *scope);
	/** \brief PCM is closing
	 * \param scope scope handle
	 */
	void (*close)(snd_pcm_scope_t *scope);
} snd_pcm_scope_ops_t;

snd_pcm_uframes_t snd_pcm_meter_get_bufsize(snd_pcm_t *pcm);
unsigned int snd_pcm_meter_get_channels(snd_pcm_t *pcm);
unsigned int snd_pcm_meter_get_rate(snd_pcm_t *pcm);
snd_pcm_uframes_t snd_pcm_meter_get_now(snd_pcm_t *pcm);
snd_pcm_uframes_t snd_pcm_meter_get_boundary(snd_pcm_t *pcm);
int snd_pcm_meter_add_scope(snd_pcm_t *pcm, snd_pcm_scope_t *scope);
snd_pcm_scope_t *snd_pcm_meter_search_scope(snd_pcm_t *pcm, const char *name);
int snd_pcm_scope_malloc(snd_pcm_scope_t **ptr);
void snd_pcm_scope_set_ops(snd_pcm_scope_t *scope, snd_pcm_scope_ops_t *val);
void snd_pcm_scope_set_name(snd_pcm_scope_t *scope, const char *val);
const char *snd_pcm_scope_get_name(snd_pcm_scope_t *scope);
void *snd_pcm_scope_get_callback_private(snd_pcm_scope_t *scope);
void snd_pcm_scope_set_callback_private(snd_pcm_scope_t *scope, void *val);
int snd_pcm_scope_s16_open(snd_pcm_t *pcm, const char *name,
			   snd_pcm_scope_t **scopep);
int16_t *snd_pcm_scope_s16_get_channel_buffer(snd_pcm_scope_t *scope,
					      unsigned int channel);

/* misc */

int snd_pcm_format_signed(snd_pcm_format_t format);
int snd_pcm_format_unsigned(snd_pcm_format_t format);
int snd_pcm_format_linear(snd_pcm_format_t format);
int snd_pcm_format_little_endian(snd_pcm_format_t format);
int snd_pcm_format_big_endian(snd_pcm_format_t format);
int snd_pcm_format_cpu_endian(snd_pcm_format_t format);
int snd_pcm_format_width(snd_pcm_format_t format);			/* in bits */
int snd_pcm_format_physical_width(snd_pcm_format_t format);		/* in bits */
snd_pcm_format_t snd_pcm_build_linear_format(int width, int unsignd, int big_endian);
ssize_t snd_pcm_format_size(snd_pcm_format_t format, size_t samples);
u_int8_t snd_pcm_format_silence(snd_pcm_format_t format);
u_int16_t snd_pcm_format_silence_16(snd_pcm_format_t format);
u_int32_t snd_pcm_format_silence_32(snd_pcm_format_t format);
u_int64_t snd_pcm_format_silence_64(snd_pcm_format_t format);
int snd_pcm_format_set_silence(snd_pcm_format_t format, void *buf, unsigned int samples);

size_t snd_pcm_access_mask_sizeof(void);
/** \hideinitializer
 * \brief allocate an empty #snd_pcm_access_mask_t using standard alloca
 * \param ptr returned pointer
 */
#define snd_pcm_access_mask_alloca(ptr) do { assert(ptr); *ptr = (snd_pcm_access_mask_t *) alloca(snd_pcm_access_mask_sizeof()); memset(*ptr, 0, snd_pcm_access_mask_sizeof()); } while (0)
int snd_pcm_access_mask_malloc(snd_pcm_access_mask_t **ptr);
void snd_pcm_access_mask_free(snd_pcm_access_mask_t *obj);
void snd_pcm_access_mask_copy(snd_pcm_access_mask_t *dst, const snd_pcm_access_mask_t *src);
void snd_pcm_access_mask_none(snd_pcm_access_mask_t *mask);
void snd_pcm_access_mask_any(snd_pcm_access_mask_t *mask);
int snd_pcm_access_mask_test(const snd_pcm_access_mask_t *mask, snd_pcm_access_t val);
void snd_pcm_access_mask_set(snd_pcm_access_mask_t *mask, snd_pcm_access_t val);
void snd_pcm_access_mask_reset(snd_pcm_access_mask_t *mask, snd_pcm_access_t val);

size_t snd_pcm_format_mask_sizeof(void);
/** \hideinitializer
 * \brief allocate an empty #snd_pcm_format_mask_t using standard alloca
 * \param ptr returned pointer
 */
#define snd_pcm_format_mask_alloca(ptr) do { assert(ptr); *ptr = (snd_pcm_format_mask_t *) alloca(snd_pcm_format_mask_sizeof()); memset(*ptr, 0, snd_pcm_format_mask_sizeof()); } while (0)
int snd_pcm_format_mask_malloc(snd_pcm_format_mask_t **ptr);
void snd_pcm_format_mask_free(snd_pcm_format_mask_t *obj);
void snd_pcm_format_mask_copy(snd_pcm_format_mask_t *dst, const snd_pcm_format_mask_t *src);
void snd_pcm_format_mask_none(snd_pcm_format_mask_t *mask);
void snd_pcm_format_mask_any(snd_pcm_format_mask_t *mask);
int snd_pcm_format_mask_test(const snd_pcm_format_mask_t *mask, snd_pcm_format_t val);
void snd_pcm_format_mask_set(snd_pcm_format_mask_t *mask, snd_pcm_format_t val);
void snd_pcm_format_mask_reset(snd_pcm_format_mask_t *mask, snd_pcm_format_t val);

size_t snd_pcm_subformat_mask_sizeof(void);
/** \hideinitializer
 * \brief allocate an empty #snd_pcm_subformat_mask_t using standard alloca
 * \param ptr returned pointer
 */
#define snd_pcm_subformat_mask_alloca(ptr) do { assert(ptr); *ptr = (snd_pcm_subformat_mask_t *) alloca(snd_pcm_subformat_mask_sizeof()); memset(*ptr, 0, snd_pcm_subformat_mask_sizeof()); } while (0)
int snd_pcm_subformat_mask_malloc(snd_pcm_subformat_mask_t **ptr);
void snd_pcm_subformat_mask_free(snd_pcm_subformat_mask_t *obj);
void snd_pcm_subformat_mask_copy(snd_pcm_subformat_mask_t *dst, const snd_pcm_subformat_mask_t *src);
void snd_pcm_subformat_mask_none(snd_pcm_subformat_mask_t *mask);
void snd_pcm_subformat_mask_any(snd_pcm_subformat_mask_t *mask);
int snd_pcm_subformat_mask_test(const snd_pcm_subformat_mask_t *mask, snd_pcm_subformat_t val);
void snd_pcm_subformat_mask_set(snd_pcm_subformat_mask_t *mask, snd_pcm_subformat_t val);
void snd_pcm_subformat_mask_reset(snd_pcm_subformat_mask_t *mask, snd_pcm_subformat_t val);

size_t snd_pcm_hw_params_sizeof(void);
/** \hideinitializer
 * \brief allocate an invalid #snd_pcm_hw_params_t using standard alloca
 * \param ptr returned pointer
 */
#define snd_pcm_hw_params_alloca(ptr) do { assert(ptr); *ptr = (snd_pcm_hw_params_t *) alloca(snd_pcm_hw_params_sizeof()); memset(*ptr, 0, snd_pcm_hw_params_sizeof()); } while (0)
int snd_pcm_hw_params_malloc(snd_pcm_hw_params_t **ptr);
void snd_pcm_hw_params_free(snd_pcm_hw_params_t *obj);
void snd_pcm_hw_params_copy(snd_pcm_hw_params_t *dst, const snd_pcm_hw_params_t *src);
int snd_pcm_hw_params_get_access(const snd_pcm_hw_params_t *params);
int snd_pcm_hw_params_test_access(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_access_t val);
int snd_pcm_hw_params_set_access(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_access_t val);
snd_pcm_access_t snd_pcm_hw_params_set_access_first(snd_pcm_t *pcm, snd_pcm_hw_params_t *params);
snd_pcm_access_t snd_pcm_hw_params_set_access_last(snd_pcm_t *pcm, snd_pcm_hw_params_t *params);
int snd_pcm_hw_params_set_access_mask(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_access_mask_t *mask);
void snd_pcm_hw_params_get_access_mask(snd_pcm_hw_params_t *params, snd_pcm_access_mask_t *mask);
int snd_pcm_hw_params_get_format(const snd_pcm_hw_params_t *params);
int snd_pcm_hw_params_test_format(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_format_t val);
int snd_pcm_hw_params_set_format(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_format_t val);
snd_pcm_format_t snd_pcm_hw_params_set_format_first(snd_pcm_t *pcm, snd_pcm_hw_params_t *params);
snd_pcm_format_t snd_pcm_hw_params_set_format_last(snd_pcm_t *pcm, snd_pcm_hw_params_t *params);
int snd_pcm_hw_params_set_format_mask(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_format_mask_t *mask);
void snd_pcm_hw_params_get_format_mask(snd_pcm_hw_params_t *params, snd_pcm_format_mask_t *mask);
int snd_pcm_hw_params_test_subformat(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_subformat_t val);
int snd_pcm_hw_params_get_subformat(const snd_pcm_hw_params_t *params);
int snd_pcm_hw_params_set_subformat(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_subformat_t val);
snd_pcm_subformat_t snd_pcm_hw_params_set_subformat_first(snd_pcm_t *pcm, snd_pcm_hw_params_t *params);
snd_pcm_subformat_t snd_pcm_hw_params_set_subformat_last(snd_pcm_t *pcm, snd_pcm_hw_params_t *params);
int snd_pcm_hw_params_set_subformat_mask(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_subformat_mask_t *mask);
void snd_pcm_hw_params_get_subformat_mask(snd_pcm_hw_params_t *params, snd_pcm_subformat_mask_t *mask);
int snd_pcm_hw_params_get_channels(const snd_pcm_hw_params_t *params);
unsigned int snd_pcm_hw_params_get_channels_min(const snd_pcm_hw_params_t *params);
unsigned int snd_pcm_hw_params_get_channels_max(const snd_pcm_hw_params_t *params);
int snd_pcm_hw_params_test_channels(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val);
int snd_pcm_hw_params_set_channels(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val);
int snd_pcm_hw_params_set_channels_min(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val);
int snd_pcm_hw_params_set_channels_max(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val);
int snd_pcm_hw_params_set_channels_minmax(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *min, unsigned int *max);
unsigned int snd_pcm_hw_params_set_channels_near(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val);
unsigned int snd_pcm_hw_params_set_channels_first(snd_pcm_t *pcm, snd_pcm_hw_params_t *params);
unsigned int snd_pcm_hw_params_set_channels_last(snd_pcm_t *pcm, snd_pcm_hw_params_t *params);
int snd_pcm_hw_params_get_rate(const snd_pcm_hw_params_t *params, int *dir);
unsigned int snd_pcm_hw_params_get_rate_min(const snd_pcm_hw_params_t *params, int *dir);
unsigned int snd_pcm_hw_params_get_rate_max(const snd_pcm_hw_params_t *params, int *dir);
int snd_pcm_hw_params_test_rate(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val, int dir);
int snd_pcm_hw_params_set_rate(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val, int dir);
int snd_pcm_hw_params_set_rate_min(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
int snd_pcm_hw_params_set_rate_max(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
int snd_pcm_hw_params_set_rate_minmax(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *min, int *mindir, unsigned int *max, int *maxdir);
unsigned int snd_pcm_hw_params_set_rate_near(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val, int *dir);
unsigned int snd_pcm_hw_params_set_rate_first(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, int *dir);
unsigned int snd_pcm_hw_params_set_rate_last(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, int *dir);
int snd_pcm_hw_params_get_period_time(const snd_pcm_hw_params_t *params, int *dir);
unsigned int snd_pcm_hw_params_get_period_time_min(const snd_pcm_hw_params_t *params, int *dir);
unsigned int snd_pcm_hw_params_get_period_time_max(const snd_pcm_hw_params_t *params, int *dir);
int snd_pcm_hw_params_test_period_time(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val, int dir);
int snd_pcm_hw_params_set_period_time(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val, int dir);
int snd_pcm_hw_params_set_period_time_min(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
int snd_pcm_hw_params_set_period_time_max(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
int snd_pcm_hw_params_set_period_time_minmax(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *min, int *mindir, unsigned int *max, int *maxdir);
unsigned int snd_pcm_hw_params_set_period_time_near(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val, int *dir);
unsigned int snd_pcm_hw_params_set_period_time_first(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, int *dir);
unsigned int snd_pcm_hw_params_set_period_time_last(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, int *dir);
snd_pcm_sframes_t snd_pcm_hw_params_get_period_size(const snd_pcm_hw_params_t *params, int *dir);
snd_pcm_uframes_t snd_pcm_hw_params_get_period_size_min(const snd_pcm_hw_params_t *params, int *dir);
snd_pcm_uframes_t snd_pcm_hw_params_get_period_size_max(const snd_pcm_hw_params_t *params, int *dir);
int snd_pcm_hw_params_test_period_size(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_uframes_t val, int dir);
int snd_pcm_hw_params_set_period_size(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_uframes_t val, int dir);
int snd_pcm_hw_params_set_period_size_min(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_uframes_t *val, int *dir);
int snd_pcm_hw_params_set_period_size_max(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_uframes_t *val, int *dir);
int snd_pcm_hw_params_set_period_size_minmax(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_uframes_t *min, int *mindir, snd_pcm_uframes_t *max, int *maxdir);
snd_pcm_uframes_t snd_pcm_hw_params_set_period_size_near(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_uframes_t val, int *dir);
snd_pcm_uframes_t snd_pcm_hw_params_set_period_size_first(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, int *dir);
snd_pcm_uframes_t snd_pcm_hw_params_set_period_size_last(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, int *dir);
int snd_pcm_hw_params_set_period_size_integer(snd_pcm_t *pcm, snd_pcm_hw_params_t *params);
int snd_pcm_hw_params_get_periods(const snd_pcm_hw_params_t *params, int *dir);
unsigned int snd_pcm_hw_params_get_periods_min(const snd_pcm_hw_params_t *params, int *dir);
unsigned int snd_pcm_hw_params_get_periods_max(const snd_pcm_hw_params_t *params, int *dir);
int snd_pcm_hw_params_test_periods(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val, int dir);
int snd_pcm_hw_params_set_periods(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val, int dir);
int snd_pcm_hw_params_set_periods_min(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
int snd_pcm_hw_params_set_periods_max(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
int snd_pcm_hw_params_set_periods_minmax(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *min, int *mindir, unsigned int *max, int *maxdir);
unsigned int snd_pcm_hw_params_set_periods_near(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val, int *dir);
unsigned int snd_pcm_hw_params_set_periods_first(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, int *dir);
unsigned int snd_pcm_hw_params_set_periods_last(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, int *dir);
int snd_pcm_hw_params_set_periods_integer(snd_pcm_t *pcm, snd_pcm_hw_params_t *params);
int snd_pcm_hw_params_get_buffer_time(const snd_pcm_hw_params_t *params, int *dir);
unsigned int snd_pcm_hw_params_get_buffer_time_min(const snd_pcm_hw_params_t *params, int *dir);
unsigned int snd_pcm_hw_params_get_buffer_time_max(const snd_pcm_hw_params_t *params, int *dir);
int snd_pcm_hw_params_test_buffer_time(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val, int dir);
int snd_pcm_hw_params_set_buffer_time(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val, int dir);
int snd_pcm_hw_params_set_buffer_time_min(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
int snd_pcm_hw_params_set_buffer_time_max(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
int snd_pcm_hw_params_set_buffer_time_minmax(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *min, int *mindir, unsigned int *max, int *maxdir);
unsigned int snd_pcm_hw_params_set_buffer_time_near(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val, int *dir);
unsigned int snd_pcm_hw_params_set_buffer_time_first(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, int *dir);
unsigned int snd_pcm_hw_params_set_buffer_time_last(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, int *dir);
snd_pcm_sframes_t snd_pcm_hw_params_get_buffer_size(const snd_pcm_hw_params_t *params);
snd_pcm_uframes_t snd_pcm_hw_params_get_buffer_size_min(const snd_pcm_hw_params_t *params);
snd_pcm_uframes_t snd_pcm_hw_params_get_buffer_size_max(const snd_pcm_hw_params_t *params);
int snd_pcm_hw_params_test_buffer_size(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_uframes_t val);
int snd_pcm_hw_params_set_buffer_size(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_uframes_t val);
int snd_pcm_hw_params_set_buffer_size_min(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_uframes_t *val);
int snd_pcm_hw_params_set_buffer_size_max(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_uframes_t *val);
int snd_pcm_hw_params_set_buffer_size_minmax(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_uframes_t *min, snd_pcm_uframes_t *max);
snd_pcm_uframes_t snd_pcm_hw_params_set_buffer_size_near(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, snd_pcm_uframes_t val);
snd_pcm_uframes_t snd_pcm_hw_params_set_buffer_size_first(snd_pcm_t *pcm, snd_pcm_hw_params_t *params);
snd_pcm_uframes_t snd_pcm_hw_params_set_buffer_size_last(snd_pcm_t *pcm, snd_pcm_hw_params_t *params);
int snd_pcm_hw_params_get_tick_time(const snd_pcm_hw_params_t *params, int *dir);
unsigned int snd_pcm_hw_params_get_tick_time_min(const snd_pcm_hw_params_t *params, int *dir);
unsigned int snd_pcm_hw_params_get_tick_time_max(const snd_pcm_hw_params_t *params, int *dir);
int snd_pcm_hw_params_test_tick_time(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val, int dir);
int snd_pcm_hw_params_set_tick_time(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val, int dir);
int snd_pcm_hw_params_set_tick_time_min(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
int snd_pcm_hw_params_set_tick_time_max(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
int snd_pcm_hw_params_set_tick_time_minmax(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int *min, int *mindir, unsigned int *max, int *maxdir);
unsigned int snd_pcm_hw_params_set_tick_time_near(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val, int *dir);
unsigned int snd_pcm_hw_params_set_tick_time_first(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, int *dir);
unsigned int snd_pcm_hw_params_set_tick_time_last(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, int *dir);

size_t snd_pcm_sw_params_sizeof(void);
/** \hideinitializer
 * \brief allocate an invalid #snd_pcm_sw_params_t using standard alloca
 * \param ptr returned pointer
 */
#define snd_pcm_sw_params_alloca(ptr) do { assert(ptr); *ptr = (snd_pcm_sw_params_t *) alloca(snd_pcm_sw_params_sizeof()); memset(*ptr, 0, snd_pcm_sw_params_sizeof()); } while (0)
int snd_pcm_sw_params_malloc(snd_pcm_sw_params_t **ptr);
void snd_pcm_sw_params_free(snd_pcm_sw_params_t *obj);
void snd_pcm_sw_params_copy(snd_pcm_sw_params_t *dst, const snd_pcm_sw_params_t *src);
int snd_pcm_sw_params_set_start_mode(snd_pcm_t *pcm, snd_pcm_sw_params_t *params, snd_pcm_start_t val);
snd_pcm_start_t snd_pcm_sw_params_get_start_mode(const snd_pcm_sw_params_t *params);
int snd_pcm_sw_params_set_xrun_mode(snd_pcm_t *pcm, snd_pcm_sw_params_t *params, snd_pcm_xrun_t val);
snd_pcm_xrun_t snd_pcm_sw_params_get_xrun_mode(const snd_pcm_sw_params_t *params);
int snd_pcm_sw_params_set_tstamp_mode(snd_pcm_t *pcm, snd_pcm_sw_params_t *params, snd_pcm_tstamp_t val);
snd_pcm_tstamp_t snd_pcm_sw_params_get_tstamp_mode(const snd_pcm_sw_params_t *params);
#if 0
int snd_pcm_sw_params_set_period_step(snd_pcm_t *pcm, snd_pcm_sw_params_t *params, unsigned int val);
unsigned int snd_pcm_sw_params_get_period_step(const snd_pcm_sw_params_t *params);
#endif
int snd_pcm_sw_params_set_sleep_min(snd_pcm_t *pcm, snd_pcm_sw_params_t *params, unsigned int val);
unsigned int snd_pcm_sw_params_get_sleep_min(const snd_pcm_sw_params_t *params);
int snd_pcm_sw_params_set_avail_min(snd_pcm_t *pcm, snd_pcm_sw_params_t *params, snd_pcm_uframes_t val);
snd_pcm_uframes_t snd_pcm_sw_params_get_avail_min(const snd_pcm_sw_params_t *params);
int snd_pcm_sw_params_set_xfer_align(snd_pcm_t *pcm, snd_pcm_sw_params_t *params, snd_pcm_uframes_t val);
snd_pcm_uframes_t snd_pcm_sw_params_get_xfer_align(const snd_pcm_sw_params_t *params);
int snd_pcm_sw_params_set_start_threshold(snd_pcm_t *pcm, snd_pcm_sw_params_t *params, snd_pcm_uframes_t val);
snd_pcm_uframes_t snd_pcm_sw_params_get_start_threshold(const snd_pcm_sw_params_t *params);
int snd_pcm_sw_params_set_stop_threshold(snd_pcm_t *pcm, snd_pcm_sw_params_t *params, snd_pcm_uframes_t val);
snd_pcm_uframes_t snd_pcm_sw_params_get_stop_threshold(const snd_pcm_sw_params_t *params);
int snd_pcm_sw_params_set_silence_threshold(snd_pcm_t *pcm, snd_pcm_sw_params_t *params, snd_pcm_uframes_t val);
snd_pcm_uframes_t snd_pcm_sw_params_get_silence_threshold(const snd_pcm_sw_params_t *params);
int snd_pcm_sw_params_set_silence_size(snd_pcm_t *pcm, snd_pcm_sw_params_t *params, snd_pcm_uframes_t val);
snd_pcm_uframes_t snd_pcm_sw_params_get_silence_size(const snd_pcm_sw_params_t *params);

size_t snd_pcm_status_sizeof(void);
/** \hideinitializer
 * \brief allocate an invalid #snd_pcm_status_t using standard alloca
 * \param ptr returned pointer
 */
#define snd_pcm_status_alloca(ptr) do { assert(ptr); *ptr = (snd_pcm_status_t *) alloca(snd_pcm_status_sizeof()); memset(*ptr, 0, snd_pcm_status_sizeof()); } while (0)
int snd_pcm_status_malloc(snd_pcm_status_t **ptr);
void snd_pcm_status_free(snd_pcm_status_t *obj);
void snd_pcm_status_copy(snd_pcm_status_t *dst, const snd_pcm_status_t *src);
snd_pcm_state_t snd_pcm_status_get_state(const snd_pcm_status_t *obj);
void snd_pcm_status_get_trigger_tstamp(const snd_pcm_status_t *obj, snd_timestamp_t *ptr);
void snd_pcm_status_get_tstamp(const snd_pcm_status_t *obj, snd_timestamp_t *ptr);
snd_pcm_sframes_t snd_pcm_status_get_delay(const snd_pcm_status_t *obj);
snd_pcm_uframes_t snd_pcm_status_get_avail(const snd_pcm_status_t *obj);
snd_pcm_uframes_t snd_pcm_status_get_avail_max(const snd_pcm_status_t *obj);

size_t snd_pcm_info_sizeof(void);
/** \hideinitializer
 * \brief allocate an invalid #snd_pcm_info_t using standard alloca
 * \param ptr returned pointer
 */
#define snd_pcm_info_alloca(ptr) do { assert(ptr); *ptr = (snd_pcm_info_t *) alloca(snd_pcm_info_sizeof()); memset(*ptr, 0, snd_pcm_info_sizeof()); } while (0)
int snd_pcm_info_malloc(snd_pcm_info_t **ptr);
void snd_pcm_info_free(snd_pcm_info_t *obj);
void snd_pcm_info_copy(snd_pcm_info_t *dst, const snd_pcm_info_t *src);
unsigned int snd_pcm_info_get_device(const snd_pcm_info_t *obj);
unsigned int snd_pcm_info_get_subdevice(const snd_pcm_info_t *obj);
snd_pcm_stream_t snd_pcm_info_get_stream(const snd_pcm_info_t *obj);
int snd_pcm_info_get_card(const snd_pcm_info_t *obj);
const char *snd_pcm_info_get_id(const snd_pcm_info_t *obj);
const char *snd_pcm_info_get_name(const snd_pcm_info_t *obj);
const char *snd_pcm_info_get_subdevice_name(const snd_pcm_info_t *obj);
snd_pcm_class_t snd_pcm_info_get_class(const snd_pcm_info_t *obj);
snd_pcm_subclass_t snd_pcm_info_get_subclass(const snd_pcm_info_t *obj);
unsigned int snd_pcm_info_get_subdevices_count(const snd_pcm_info_t *obj);
unsigned int snd_pcm_info_get_subdevices_avail(const snd_pcm_info_t *obj);
void snd_pcm_info_set_device(snd_pcm_info_t *obj, unsigned int val);
void snd_pcm_info_set_subdevice(snd_pcm_info_t *obj, unsigned int val);
void snd_pcm_info_set_stream(snd_pcm_info_t *obj, snd_pcm_stream_t val);

/** type of pcm hook */
typedef enum _snd_pcm_hook_type {
	SND_PCM_HOOK_TYPE_HW_PARAMS,
	SND_PCM_HOOK_TYPE_HW_FREE,
	SND_PCM_HOOK_TYPE_CLOSE,
	SND_PCM_HOOK_TYPE_LAST = SND_PCM_HOOK_TYPE_CLOSE,
} snd_pcm_hook_type_t;

/** PCM hook container */
typedef struct _snd_pcm_hook snd_pcm_hook_t;
/** PCM hook callback function */
typedef int (*snd_pcm_hook_func_t)(snd_pcm_hook_t *hook);
snd_pcm_t *snd_pcm_hook_get_pcm(snd_pcm_hook_t *hook);
void *snd_pcm_hook_get_private(snd_pcm_hook_t *hook);
void snd_pcm_hook_set_private(snd_pcm_hook_t *hook, void *private_data);
int snd_pcm_hook_add(snd_pcm_hook_t **hookp, snd_pcm_t *pcm,
		     snd_pcm_hook_type_t type,
		     snd_pcm_hook_func_t func, void *private_data);
int snd_pcm_hook_remove(snd_pcm_hook_t *hook);

#ifdef __cplusplus
}
#endif

/** \} */

/****************************************************************************
 *                                                                          *
 *                              rawmidi.h                                   *
 *                          RawMIDI interface                               *
 *                                                                          *
 ****************************************************************************/

/**
 *  \defgroup RawMidi RawMidi Interface
 *  The RawMidi Interface.
 *  \{
 */

/** RawMidi information container */
typedef struct _snd_rawmidi_info snd_rawmidi_info_t;
/** RawMidi settings container */
typedef struct _snd_rawmidi_params snd_rawmidi_params_t;
/** RawMidi status container */
typedef struct _snd_rawmidi_status snd_rawmidi_status_t;

/** RawMidi stream (direction) */
typedef enum _snd_rawmidi_stream {
	/** Output stream */
	SND_RAWMIDI_STREAM_OUTPUT = SNDRV_RAWMIDI_STREAM_OUTPUT,
	/** Input stream */
	SND_RAWMIDI_STREAM_INPUT = SNDRV_RAWMIDI_STREAM_INPUT,
	SND_RAWMIDI_STREAM_LAST = SNDRV_RAWMIDI_STREAM_LAST,
} snd_rawmidi_stream_t;

/** Append \hideinitializer */
#define SND_RAWMIDI_APPEND	1
/** Non blocking mode \hideinitializer */
#define SND_RAWMIDI_NONBLOCK	2
/** Write sync mode \hideinitializer */
#define SND_RAWMIDI_SYNC	4

/** RawMidi handle */
typedef struct _snd_rawmidi snd_rawmidi_t;

/** RawMidi type */
typedef enum _snd_rawmidi_type {
	/** Kernel level RawMidi */
	SND_RAWMIDI_TYPE_HW,
	/** Shared memory client RawMidi (not yet implemented) */
	SND_RAWMIDI_TYPE_SHM,
	/** INET client RawMidi (not yet implemented) */
	SND_RAWMIDI_TYPE_INET,
} snd_rawmidi_type_t;

#ifdef __cplusplus
extern "C" {
#endif

int snd_rawmidi_open(snd_rawmidi_t **in_rmidi, snd_rawmidi_t **out_rmidi,
		     const char *name, int mode);
int snd_rawmidi_close(snd_rawmidi_t *rmidi);
int snd_rawmidi_poll_descriptors_count(snd_rawmidi_t *rmidi);
int snd_rawmidi_poll_descriptors(snd_rawmidi_t *rmidi, struct pollfd *pfds, unsigned int space);
int snd_rawmidi_nonblock(snd_rawmidi_t *rmidi, int nonblock);
size_t snd_rawmidi_info_sizeof(void);
/** \hideinitializer
 * \brief allocate an invalid #snd_rawmidi_info_t using standard alloca
 * \param ptr returned pointer
 */
#define snd_rawmidi_info_alloca(ptr) do { assert(ptr); *ptr = (snd_rawmidi_info_t *) alloca(snd_rawmidi_info_sizeof()); memset(*ptr, 0, snd_rawmidi_info_sizeof()); } while (0)
int snd_rawmidi_info_malloc(snd_rawmidi_info_t **ptr);
void snd_rawmidi_info_free(snd_rawmidi_info_t *obj);
void snd_rawmidi_info_copy(snd_rawmidi_info_t *dst, const snd_rawmidi_info_t *src);
unsigned int snd_rawmidi_info_get_device(const snd_rawmidi_info_t *obj);
unsigned int snd_rawmidi_info_get_subdevice(const snd_rawmidi_info_t *obj);
snd_rawmidi_stream_t snd_rawmidi_info_get_stream(const snd_rawmidi_info_t *obj);
int snd_rawmidi_info_get_card(const snd_rawmidi_info_t *obj);
unsigned int snd_rawmidi_info_get_flags(const snd_rawmidi_info_t *obj);
const char *snd_rawmidi_info_get_id(const snd_rawmidi_info_t *obj);
const char *snd_rawmidi_info_get_name(const snd_rawmidi_info_t *obj);
const char *snd_rawmidi_info_get_subdevice_name(const snd_rawmidi_info_t *obj);
unsigned int snd_rawmidi_info_get_subdevices_count(const snd_rawmidi_info_t *obj);
unsigned int snd_rawmidi_info_get_subdevices_avail(const snd_rawmidi_info_t *obj);
void snd_rawmidi_info_set_device(snd_rawmidi_info_t *obj, unsigned int val);
void snd_rawmidi_info_set_subdevice(snd_rawmidi_info_t *obj, unsigned int val);
void snd_rawmidi_info_set_stream(snd_rawmidi_info_t *obj, snd_rawmidi_stream_t val);
int snd_rawmidi_info(snd_rawmidi_t *rmidi, snd_rawmidi_info_t * info);
size_t snd_rawmidi_params_sizeof(void);
/** \hideinitializer
 * \brief allocate an invalid #snd_rawmidi_params_t using standard alloca
 * \param ptr returned pointer
 */
#define snd_rawmidi_params_alloca(ptr) do { assert(ptr); *ptr = (snd_rawmidi_params_t *) alloca(snd_rawmidi_params_sizeof()); memset(*ptr, 0, snd_rawmidi_params_sizeof()); } while (0)
int snd_rawmidi_params_malloc(snd_rawmidi_params_t **ptr);
void snd_rawmidi_params_free(snd_rawmidi_params_t *obj);
void snd_rawmidi_params_copy(snd_rawmidi_params_t *dst, const snd_rawmidi_params_t *src);
int snd_rawmidi_params_set_buffer_size(snd_rawmidi_t *rmidi, snd_rawmidi_params_t *params, size_t val);
size_t snd_rawmidi_params_get_buffer_size(const snd_rawmidi_params_t *params);
int snd_rawmidi_params_set_avail_min(snd_rawmidi_t *rmidi, snd_rawmidi_params_t *params, size_t val);
size_t snd_rawmidi_params_get_avail_min(const snd_rawmidi_params_t *params);
int snd_rawmidi_params_set_no_active_sensing(snd_rawmidi_t *rmidi, snd_rawmidi_params_t *params, int val);
int snd_rawmidi_params_get_no_active_sensing(const snd_rawmidi_params_t *params);
int snd_rawmidi_params(snd_rawmidi_t *rmidi, snd_rawmidi_params_t * params);
int snd_rawmidi_params_current(snd_rawmidi_t *rmidi, snd_rawmidi_params_t *params);
size_t snd_rawmidi_status_sizeof(void);
/** \hideinitializer
 * \brief allocate an invalid #snd_rawmidi_status_t using standard alloca
 * \param ptr returned pointer
 */
#define snd_rawmidi_status_alloca(ptr) do { assert(ptr); *ptr = (snd_rawmidi_status_t *) alloca(snd_rawmidi_status_sizeof()); memset(*ptr, 0, snd_rawmidi_status_sizeof()); } while (0)
int snd_rawmidi_status_malloc(snd_rawmidi_status_t **ptr);
void snd_rawmidi_status_free(snd_rawmidi_status_t *obj);
void snd_rawmidi_status_copy(snd_rawmidi_status_t *dst, const snd_rawmidi_status_t *src);
void snd_rawmidi_status_get_tstamp(const snd_rawmidi_status_t *obj, snd_timestamp_t *ptr);
size_t snd_rawmidi_status_get_avail(const snd_rawmidi_status_t *obj);
size_t snd_rawmidi_status_get_xruns(const snd_rawmidi_status_t *obj);
int snd_rawmidi_status(snd_rawmidi_t *rmidi, snd_rawmidi_status_t * status);
int snd_rawmidi_drain(snd_rawmidi_t *rmidi);
int snd_rawmidi_drop(snd_rawmidi_t *rmidi);
ssize_t snd_rawmidi_write(snd_rawmidi_t *rmidi, const void *buffer, size_t size);
ssize_t snd_rawmidi_read(snd_rawmidi_t *rmidi, void *buffer, size_t size);
const char *snd_rawmidi_name(snd_rawmidi_t *rmidi);
snd_rawmidi_type_t snd_rawmidi_type(snd_rawmidi_t *rmidi);
snd_rawmidi_stream_t snd_rawmidi_stream(snd_rawmidi_t *rawmidi);

#ifdef __cplusplus
}
#endif

/** \} */

/****************************************************************************
 *                                                                          *
 *                               timer.h                                    *
 *                           Timer interface                                *
 *                                                                          *
 ****************************************************************************/

/**
 *  \defgroup Timer Timer Interface
 *  Timer Interface
 *  \{
 */

/** timer identification structure */
typedef struct _snd_timer_id snd_timer_id_t;
/** timer info structure */
typedef struct _snd_timer_info snd_timer_info_t;
/** timer params structure */
typedef struct _snd_timer_params snd_timer_params_t;
/** timer status structure */
typedef struct _snd_timer_status snd_timer_status_t;
/** timer read structure */
typedef struct sndrv_timer_read snd_timer_read_t;

/** timer master class */
typedef enum _snd_timer_class {
	SND_TIMER_CLASS_NONE = SNDRV_TIMER_CLASS_NONE,		/**< invalid */
	SND_TIMER_CLASS_SLAVE = SNDRV_TIMER_CLASS_SLAVE,	/**< slave timer */
	SND_TIMER_CLASS_GLOBAL = SNDRV_TIMER_CLASS_GLOBAL,	/**< global timer */
	SND_TIMER_CLASS_CARD = SNDRV_TIMER_CLASS_CARD,		/**< card timer */
	SND_TIMER_CLASS_PCM = SNDRV_TIMER_CLASS_PCM,		/**< PCM timer */
	SND_TIMER_CLASS_LAST = SNDRV_TIMER_CLASS_LAST,		/**< last timer */
} snd_timer_class_t;

/** timer slave class */
typedef enum _snd_timer_slave_class {
	SND_TIMER_SCLASS_NONE = SNDRV_TIMER_SCLASS_NONE,		/**< none */
	SND_TIMER_SCLASS_APPLICATION = SNDRV_TIMER_SCLASS_APPLICATION,	/**< for internal use */
	SND_TIMER_SCLASS_SEQUENCER = SNDRV_TIMER_SCLASS_SEQUENCER,	/**< sequencer timer */
	SND_TIMER_SCLASS_OSS_SEQUENCER = SNDRV_TIMER_SCLASS_OSS_SEQUENCER, /**< OSS sequencer timer */
	SND_TIMER_SCLASS_LAST = SNDRV_TIMER_SCLASS_LAST,		/**< last slave timer */
} snd_timer_slave_class_t;

/** global timer - system */
#define SND_TIMER_GLOBAL_SYSTEM SNDRV_TIMER_GLOBAL_SYSTEM
/** global timer - RTC */
#define SND_TIMER_GLOBAL_RTC SNDRV_TIMER_GLOBAL_RTC

/** timer open mode flag - nonblock */
#define SND_TIMER_OPEN_NONBLOCK		1

/** timer handle type */
typedef enum _snd_timer_type {
	/** Kernel level HwDep */
	SND_TIMER_TYPE_HW,
	/** Shared memory client timer (not yet implemented) */
	SND_TIMER_TYPE_SHM,
	/** INET client timer (not yet implemented) */
	SND_TIMER_TYPE_INET,
} snd_timer_type_t;

/** timer query handle */
typedef struct _snd_timer_query snd_timer_query_t;
/** timer handle */
typedef struct _snd_timer snd_timer_t;

#ifdef __cplusplus
extern "C" {
#endif

int snd_timer_query_open(snd_timer_query_t **handle, const char *name, int mode);
int snd_timer_query_close(snd_timer_query_t *handle);
int snd_timer_query_next_device(snd_timer_query_t *handle, snd_timer_id_t *tid);

int snd_timer_open(snd_timer_t **handle, const char *name, int mode);
int snd_timer_close(snd_timer_t *handle);
int snd_timer_poll_descriptors_count(snd_timer_t *handle);
int snd_timer_poll_descriptors(snd_timer_t *handle, struct pollfd *pfds, unsigned int space);
int snd_timer_info(snd_timer_t *handle, snd_timer_info_t *timer);
int snd_timer_params(snd_timer_t *handle, snd_timer_params_t *params);
int snd_timer_status(snd_timer_t *handle, snd_timer_status_t *status);
int snd_timer_start(snd_timer_t *handle);
int snd_timer_stop(snd_timer_t *handle);
int snd_timer_continue(snd_timer_t *handle);
ssize_t snd_timer_read(snd_timer_t *handle, void *buffer, size_t size);

size_t snd_timer_id_sizeof(void);
/** allocate #snd_timer_id_t container on stack */
#define snd_timer_id_alloca(ptr) do { assert(ptr); *ptr = (snd_timer_id_t *) alloca(snd_timer_id_sizeof()); memset(*ptr, 0, snd_timer_id_sizeof()); } while (0)
int snd_timer_id_malloc(snd_timer_id_t **ptr);
void snd_timer_id_free(snd_timer_id_t *obj);
void snd_timer_id_copy(snd_timer_id_t *dst, const snd_timer_id_t *src);

void snd_timer_id_set_class(snd_timer_id_t *id, int dev_class);
int snd_timer_id_get_class(snd_timer_id_t *id);
void snd_timer_id_set_sclass(snd_timer_id_t *id, int dev_sclass);
int snd_timer_id_get_sclass(snd_timer_id_t *id);
void snd_timer_id_set_card(snd_timer_id_t *id, int card);
int snd_timer_id_get_card(snd_timer_id_t *id);
void snd_timer_id_set_device(snd_timer_id_t *id, int device);
int snd_timer_id_get_device(snd_timer_id_t *id);
void snd_timer_id_set_subdevice(snd_timer_id_t *id, int subdevice);
int snd_timer_id_get_subdevice(snd_timer_id_t *id);

size_t snd_timer_info_sizeof(void);
/** allocate #snd_timer_info_t container on stack */
#define snd_timer_info_alloca(ptr) do { assert(ptr); *ptr = (snd_timer_info_t *) alloca(snd_timer_info_sizeof()); memset(*ptr, 0, snd_timer_info_sizeof()); } while (0)
int snd_timer_info_malloc(snd_timer_info_t **ptr);
void snd_timer_info_free(snd_timer_info_t *obj);
void snd_timer_info_copy(snd_timer_info_t *dst, const snd_timer_info_t *src);

int snd_timer_info_is_slave(snd_timer_info_t * info);
int snd_timer_info_get_card(snd_timer_info_t * info);
const char *snd_timer_info_get_id(snd_timer_info_t * info);
const char *snd_timer_info_get_name(snd_timer_info_t * info);
long snd_timer_info_get_ticks(snd_timer_info_t * info);
long snd_timer_info_get_resolution(snd_timer_info_t * info);

size_t snd_timer_params_sizeof(void);
/** allocate #snd_timer_params_t container on stack */
#define snd_timer_params_alloca(ptr) do { assert(ptr); *ptr = (snd_timer_params_t *) alloca(snd_timer_params_sizeof()); memset(*ptr, 0, snd_timer_params_sizeof()); } while (0)
int snd_timer_params_malloc(snd_timer_params_t **ptr);
void snd_timer_params_free(snd_timer_params_t *obj);
void snd_timer_params_copy(snd_timer_params_t *dst, const snd_timer_params_t *src);

void snd_timer_params_set_auto_start(snd_timer_params_t * params, int auto_start);
void snd_timer_params_set_ticks(snd_timer_params_t * params, long ticks);
int snd_timer_params_get_ticks(snd_timer_params_t * params);
void snd_timer_params_set_queue_size(snd_timer_params_t * params, long queue_size);
long snd_timer_params_get_queue_size(snd_timer_params_t * params);

size_t snd_timer_status_sizeof(void);
/** allocate #snd_timer_status_t container on stack */
#define snd_timer_status_alloca(ptr) do { assert(ptr); *ptr = (snd_timer_status_t *) alloca(snd_timer_status_sizeof()); memset(*ptr, 0, snd_timer_status_sizeof()); } while (0)
int snd_timer_status_malloc(snd_timer_status_t **ptr);
void snd_timer_status_free(snd_timer_status_t *obj);
void snd_timer_status_copy(snd_timer_status_t *dst, const snd_timer_status_t *src);

struct timeval snd_timer_status_get_timestamp(snd_timer_status_t * status);
long snd_timer_status_get_resolution(snd_timer_status_t * status);
long snd_timer_status_get_lost(snd_timer_status_t * status);
long snd_timer_status_get_overrun(snd_timer_status_t * status);
long snd_timer_status_get_queue(snd_timer_status_t * status);

#ifdef __cplusplus
}
#endif

/** \} */

/****************************************************************************
 *                                                                          *
 *                               hwdep.h                                    *
 *                     Hardware depedent interface                          *
 *                                                                          *
 ****************************************************************************/

/**
 *  \defgroup HwDep Hardware Dependant Interface
 *  The Hardware Dependant Interface.
 *  \{
 */

/** HwDep information container */
typedef struct _snd_hwdep_info snd_hwdep_info_t;

/** HwDep interface */
typedef enum _snd_hwdep_iface {
	SND_HWDEP_IFACE_OPL2 = SNDRV_HWDEP_IFACE_OPL2,		/**< OPL2 raw driver */
	SND_HWDEP_IFACE_OPL3 = SNDRV_HWDEP_IFACE_OPL3,		/**< OPL3 raw driver */
	SND_HWDEP_IFACE_OPL4 = SNDRV_HWDEP_IFACE_OPL4,		/**< OPL4 raw driver */
	SND_HWDEP_IFACE_SB16CSP = SNDRV_HWDEP_IFACE_SB16CSP,	/**< SB16CSP driver */
	SND_HWDEP_IFACE_EMU10K1 = SNDRV_HWDEP_IFACE_EMU10K1,	/**< EMU10K1 driver */
	SND_HWDEP_IFACE_YSS225 = SNDRV_HWDEP_IFACE_YSS225,	/**< YSS225 driver */
	SND_HWDEP_IFACE_ICS2115 = SNDRV_HWDEP_IFACE_ICS2115,	/**< ICS2115 driver */
	SND_HWDEP_IFACE_LAST = SNDRV_HWDEP_IFACE_LAST,		/**< last know hwdep interface */
} snd_hwdep_iface_t;

/** open for reading */
#define SND_HWDEP_OPEN_READ		(O_RDONLY)
/** open for writing */
#define SND_HWDEP_OPEN_WRITE		(O_WRONLY)
/** open for reading and writing */
#define SND_HWDEP_OPEN_DUPLEX		(O_RDWR)
/** flag: open in nonblock mode */
#define SND_HWDEP_OPEN_NONBLOCK		(O_NONBLOCK)

/** HwDep handle type */
typedef enum _snd_hwdep_type {
	/** Kernel level HwDep */
	SND_HWDEP_TYPE_HW,
	/** Shared memory client HwDep (not yet implemented) */
	SND_HWDEP_TYPE_SHM,
	/** INET client HwDep (not yet implemented) */
	SND_HWDEP_TYPE_INET,
} snd_hwdep_type_t;

/** HwDep handle */
typedef struct _snd_hwdep snd_hwdep_t;

#ifdef __cplusplus
extern "C" {
#endif

int snd_hwdep_open(snd_hwdep_t **hwdep, const char *name, int mode);
int snd_hwdep_close(snd_hwdep_t *hwdep);
int snd_hwdep_poll_descriptors(snd_hwdep_t *hwdep, struct pollfd *pfds, unsigned int space);
int snd_hwdep_nonblock(snd_hwdep_t *hwdep, int nonblock);
int snd_hwdep_info(snd_hwdep_t *hwdep, snd_hwdep_info_t * info);
int snd_hwdep_ioctl(snd_hwdep_t *hwdep, unsigned int request, void * arg);
ssize_t snd_hwdep_write(snd_hwdep_t *hwdep, const void *buffer, size_t size);
ssize_t snd_hwdep_read(snd_hwdep_t *hwdep, void *buffer, size_t size);

size_t snd_hwdep_info_sizeof(void);
/** allocate #snd_hwdep_info_t container on stack */
#define snd_hwdep_info_alloca(ptr) do { assert(ptr); *ptr = (snd_hwdep_info_t *) alloca(snd_hwdep_info_sizeof()); memset(*ptr, 0, snd_hwdep_info_sizeof()); } while (0)
int snd_hwdep_info_malloc(snd_hwdep_info_t **ptr);
void snd_hwdep_info_free(snd_hwdep_info_t *obj);
void snd_hwdep_info_copy(snd_hwdep_info_t *dst, const snd_hwdep_info_t *src);

unsigned int snd_hwdep_info_get_device(const snd_hwdep_info_t *obj);
int snd_hwdep_info_get_card(const snd_hwdep_info_t *obj);
const char *snd_hwdep_info_get_id(const snd_hwdep_info_t *obj);
const char *snd_hwdep_info_get_name(const snd_hwdep_info_t *obj);
snd_hwdep_iface_t snd_hwdep_info_get_iface(const snd_hwdep_info_t *obj);
void snd_hwdep_info_set_device(snd_hwdep_info_t *obj, unsigned int val);

#ifdef __cplusplus
}
#endif

/** \} */

/****************************************************************************
 *                                                                          *
 *                              control.h                                   *
 *                          Control Interface                               *
 *                                                                          *
 ****************************************************************************/

/**
 *  \defgroup Control Control Interface
 *  The control interface.
 *  \{
 */

/** IEC958 structure */
typedef struct sndrv_aes_iec958 snd_aes_iec958_t;

/** CTL card info container */
typedef struct _snd_ctl_card_info snd_ctl_card_info_t;

/** CTL element identificator container */
typedef struct _snd_ctl_elem_id snd_ctl_elem_id_t;

/** CTL element identificator list container */
typedef struct _snd_ctl_elem_list snd_ctl_elem_list_t;

/** CTL element info container */
typedef struct _snd_ctl_elem_info snd_ctl_elem_info_t;

/** CTL element value container */
typedef struct _snd_ctl_elem_value snd_ctl_elem_value_t;

/** CTL event container */
typedef struct _snd_ctl_event snd_ctl_event_t;

/** CTL element type */
typedef enum _snd_ctl_elem_type {
	/** Invalid type */
	SND_CTL_ELEM_TYPE_NONE = SNDRV_CTL_ELEM_TYPE_NONE,
	/** Boolean contents */
	SND_CTL_ELEM_TYPE_BOOLEAN = SNDRV_CTL_ELEM_TYPE_BOOLEAN,
	/** Integer contents */
	SND_CTL_ELEM_TYPE_INTEGER = SNDRV_CTL_ELEM_TYPE_INTEGER,
	/** Enumerated contents */
	SND_CTL_ELEM_TYPE_ENUMERATED = SNDRV_CTL_ELEM_TYPE_ENUMERATED,
	/** Bytes contents */
	SND_CTL_ELEM_TYPE_BYTES = SNDRV_CTL_ELEM_TYPE_BYTES,
	/** IEC958 (S/PDIF) setting content */
	SND_CTL_ELEM_TYPE_IEC958 = SNDRV_CTL_ELEM_TYPE_IEC958,
	SND_CTL_ELEM_TYPE_LAST = SNDRV_CTL_ELEM_TYPE_LAST,
} snd_ctl_elem_type_t;

/** CTL related interface */
typedef enum _snd_ctl_elem_iface {
	/** Card level */
	SND_CTL_ELEM_IFACE_CARD = SNDRV_CTL_ELEM_IFACE_CARD,
	/** Hardware dependent device */
	SND_CTL_ELEM_IFACE_HWDEP = SNDRV_CTL_ELEM_IFACE_HWDEP,
	/** Mixer */
	SND_CTL_ELEM_IFACE_MIXER = SNDRV_CTL_ELEM_IFACE_MIXER,
	/** PCM */
	SND_CTL_ELEM_IFACE_PCM = SNDRV_CTL_ELEM_IFACE_PCM,
	/** RawMidi */
	SND_CTL_ELEM_IFACE_RAWMIDI = SNDRV_CTL_ELEM_IFACE_RAWMIDI,
	/** Timer */
	SND_CTL_ELEM_IFACE_TIMER = SNDRV_CTL_ELEM_IFACE_TIMER,
	/** Sequencer */
	SND_CTL_ELEM_IFACE_SEQUENCER = SNDRV_CTL_ELEM_IFACE_SEQUENCER,
	SND_CTL_ELEM_IFACE_LAST = SNDRV_CTL_ELEM_IFACE_LAST,
} snd_ctl_elem_iface_t;

/** Event class */
typedef enum _snd_ctl_event_type {
	/** Elements related event */
	SND_CTL_EVENT_ELEM = SNDRV_CTL_EVENT_ELEM,
	SND_CTL_EVENT_LAST = SNDRV_CTL_EVENT_LAST,
}snd_ctl_event_type_t;

/** Element has been removed (Warning: test this first and if set don't
 * test the other masks) \hideinitializer */
#define SND_CTL_EVENT_MASK_REMOVE SNDRV_CTL_EVENT_MASK_REMOVE
/** Element has been added \hideinitializer */
#define SND_CTL_EVENT_MASK_ADD SNDRV_CTL_EVENT_MASK_ADD
/** Element info has been changed \hideinitializer */
#define SND_CTL_EVENT_MASK_INFO SNDRV_CTL_EVENT_MASK_INFO
/** Element value has been changed \hideinitializer */
#define SND_CTL_EVENT_MASK_VALUE SNDRV_CTL_EVENT_MASK_VALUE

/** Element name for IEC958 (S/PDIF) */
#define SND_CTL_NAME_IEC958 SNDRV_CTL_NAME_IEC958

/** CTL type */
typedef enum _snd_ctl_type {
	/** Kernel level CTL */
	SND_CTL_TYPE_HW,
	/** Shared memory client CTL */
	SND_CTL_TYPE_SHM,
	/** INET client CTL (not yet implemented) */
	SND_CTL_TYPE_INET
} snd_ctl_type_t;

/** Non blocking mode \hideinitializer */
#define SND_CTL_NONBLOCK		0x0001

/** Async notification \hideinitializer */
#define SND_CTL_ASYNC			0x0002

/** CTL handle */
typedef struct _snd_ctl snd_ctl_t;

/** Don't destroy the ctl handle when close */
#define SND_SCTL_NOFREE			0x0001

/** SCTL type */
typedef struct _snd_sctl snd_sctl_t;

#ifdef __cplusplus
extern "C" {
#endif

int snd_card_load(int card);
int snd_card_next(int *card);
int snd_card_get_index(const char *name);
int snd_card_get_name(int card, char **name);
int snd_card_get_longname(int card, char **name);

int snd_sctl_build(snd_sctl_t **ctl, snd_ctl_t *handle, snd_config_t *config,
		   void *private_data, int mode);
int snd_sctl_free(snd_sctl_t *handle);
int snd_sctl_install(snd_sctl_t *handle);
int snd_sctl_remove(snd_sctl_t *handle);

int snd_ctl_open(snd_ctl_t **ctl, const char *name, int mode);
int snd_ctl_close(snd_ctl_t *ctl);
int snd_ctl_nonblock(snd_ctl_t *ctl, int nonblock);
int snd_async_add_ctl_handler(snd_async_handler_t **handler, snd_ctl_t *ctl, 
			      snd_async_callback_t callback, void *private_data);
snd_ctl_t *snd_async_handler_get_ctl(snd_async_handler_t *handler);
int snd_ctl_poll_descriptors_count(snd_ctl_t *ctl);
int snd_ctl_poll_descriptors(snd_ctl_t *ctl, struct pollfd *pfds, unsigned int space);
int snd_ctl_subscribe_events(snd_ctl_t *ctl, int subscribe);
int snd_ctl_card_info(snd_ctl_t *ctl, snd_ctl_card_info_t *info);
int snd_ctl_elem_list(snd_ctl_t *ctl, snd_ctl_elem_list_t * list);
int snd_ctl_elem_info(snd_ctl_t *ctl, snd_ctl_elem_info_t *info);
int snd_ctl_elem_read(snd_ctl_t *ctl, snd_ctl_elem_value_t *value);
int snd_ctl_elem_write(snd_ctl_t *ctl, snd_ctl_elem_value_t *value);
int snd_ctl_elem_lock(snd_ctl_t *ctl, snd_ctl_elem_id_t *id);
int snd_ctl_elem_unlock(snd_ctl_t *ctl, snd_ctl_elem_id_t *id);
int snd_ctl_hwdep_next_device(snd_ctl_t *ctl, int * device);
int snd_ctl_hwdep_info(snd_ctl_t *ctl, snd_hwdep_info_t * info);
int snd_ctl_pcm_next_device(snd_ctl_t *ctl, int *device);
int snd_ctl_pcm_info(snd_ctl_t *ctl, snd_pcm_info_t * info);
int snd_ctl_pcm_prefer_subdevice(snd_ctl_t *ctl, int subdev);
int snd_ctl_rawmidi_next_device(snd_ctl_t *ctl, int * device);
int snd_ctl_rawmidi_info(snd_ctl_t *ctl, snd_rawmidi_info_t * info);
int snd_ctl_rawmidi_prefer_subdevice(snd_ctl_t *ctl, int subdev);

int snd_ctl_read(snd_ctl_t *ctl, snd_ctl_event_t *event);
int snd_ctl_wait(snd_ctl_t *ctl, int timeout);
const char *snd_ctl_name(snd_ctl_t *ctl);
snd_ctl_type_t snd_ctl_type(snd_ctl_t *ctl);

void snd_ctl_elem_set_bytes(snd_ctl_elem_value_t *obj, void *data, size_t size);
const char *snd_ctl_elem_type_name(snd_ctl_elem_type_t type);
const char *snd_ctl_elem_iface_name(snd_ctl_elem_iface_t iface);
const char *snd_ctl_event_type_name(snd_ctl_event_type_t type);

unsigned int snd_ctl_event_elem_get_mask(const snd_ctl_event_t *obj);
unsigned int snd_ctl_event_elem_get_numid(const snd_ctl_event_t *obj);
void snd_ctl_event_elem_get_id(const snd_ctl_event_t *obj, snd_ctl_elem_id_t *ptr);
snd_ctl_elem_iface_t snd_ctl_event_elem_get_interface(const snd_ctl_event_t *obj);
unsigned int snd_ctl_event_elem_get_device(const snd_ctl_event_t *obj);
unsigned int snd_ctl_event_elem_get_subdevice(const snd_ctl_event_t *obj);
const char *snd_ctl_event_elem_get_name(const snd_ctl_event_t *obj);
unsigned int snd_ctl_event_elem_get_index(const snd_ctl_event_t *obj);

int snd_ctl_elem_list_alloc_space(snd_ctl_elem_list_t *obj, unsigned int entries);
void snd_ctl_elem_list_free_space(snd_ctl_elem_list_t *obj);

size_t snd_ctl_elem_id_sizeof(void);
/** \hideinitializer
 * \brief allocate an invalid #snd_ctl_elem_id_t using standard alloca
 * \param ptr returned pointer
 */
#define snd_ctl_elem_id_alloca(ptr) do { assert(ptr); *ptr = (snd_ctl_elem_id_t *) alloca(snd_ctl_elem_id_sizeof()); memset(*ptr, 0, snd_ctl_elem_id_sizeof()); } while (0)
int snd_ctl_elem_id_malloc(snd_ctl_elem_id_t **ptr);
void snd_ctl_elem_id_free(snd_ctl_elem_id_t *obj);
void snd_ctl_elem_id_clear(snd_ctl_elem_id_t *obj);
void snd_ctl_elem_id_copy(snd_ctl_elem_id_t *dst, const snd_ctl_elem_id_t *src);
unsigned int snd_ctl_elem_id_get_numid(const snd_ctl_elem_id_t *obj);
snd_ctl_elem_iface_t snd_ctl_elem_id_get_interface(const snd_ctl_elem_id_t *obj);
unsigned int snd_ctl_elem_id_get_device(const snd_ctl_elem_id_t *obj);
unsigned int snd_ctl_elem_id_get_subdevice(const snd_ctl_elem_id_t *obj);
const char *snd_ctl_elem_id_get_name(const snd_ctl_elem_id_t *obj);
unsigned int snd_ctl_elem_id_get_index(const snd_ctl_elem_id_t *obj);
void snd_ctl_elem_id_set_numid(snd_ctl_elem_id_t *obj, unsigned int val);
void snd_ctl_elem_id_set_interface(snd_ctl_elem_id_t *obj, snd_ctl_elem_iface_t val);
void snd_ctl_elem_id_set_device(snd_ctl_elem_id_t *obj, unsigned int val);
void snd_ctl_elem_id_set_subdevice(snd_ctl_elem_id_t *obj, unsigned int val);
void snd_ctl_elem_id_set_name(snd_ctl_elem_id_t *obj, const char *val);
void snd_ctl_elem_id_set_index(snd_ctl_elem_id_t *obj, unsigned int val);

size_t snd_ctl_card_info_sizeof(void);
/** \hideinitializer
 * \brief allocate an invalid #snd_ctl_card_info_t using standard alloca
 * \param ptr returned pointer
 */
#define snd_ctl_card_info_alloca(ptr) do { assert(ptr); *ptr = (snd_ctl_card_info_t *) alloca(snd_ctl_card_info_sizeof()); memset(*ptr, 0, snd_ctl_card_info_sizeof()); } while (0)
int snd_ctl_card_info_malloc(snd_ctl_card_info_t **ptr);
void snd_ctl_card_info_free(snd_ctl_card_info_t *obj);
void snd_ctl_card_info_clear(snd_ctl_card_info_t *obj);
void snd_ctl_card_info_copy(snd_ctl_card_info_t *dst, const snd_ctl_card_info_t *src);
int snd_ctl_card_info_get_card(const snd_ctl_card_info_t *obj);
const char *snd_ctl_card_info_get_id(const snd_ctl_card_info_t *obj);
const char *snd_ctl_card_info_get_driver(const snd_ctl_card_info_t *obj);
const char *snd_ctl_card_info_get_name(const snd_ctl_card_info_t *obj);
const char *snd_ctl_card_info_get_longname(const snd_ctl_card_info_t *obj);
const char *snd_ctl_card_info_get_mixername(const snd_ctl_card_info_t *obj);
const char *snd_ctl_card_info_get_components(const snd_ctl_card_info_t *obj);

size_t snd_ctl_event_sizeof(void);
/** \hideinitializer
 * \brief allocate an invalid #snd_ctl_event_t using standard alloca
 * \param ptr returned pointer
 */
#define snd_ctl_event_alloca(ptr) do { assert(ptr); *ptr = (snd_ctl_event_t *) alloca(snd_ctl_event_sizeof()); memset(*ptr, 0, snd_ctl_event_sizeof()); } while (0)
int snd_ctl_event_malloc(snd_ctl_event_t **ptr);
void snd_ctl_event_free(snd_ctl_event_t *obj);
void snd_ctl_event_clear(snd_ctl_event_t *obj);
void snd_ctl_event_copy(snd_ctl_event_t *dst, const snd_ctl_event_t *src);
snd_ctl_event_type_t snd_ctl_event_get_type(const snd_ctl_event_t *obj);

size_t snd_ctl_elem_list_sizeof(void);
/** \hideinitializer
 * \brief allocate an invalid #snd_ctl_elem_list_t using standard alloca
 * \param ptr returned pointer
 */
#define snd_ctl_elem_list_alloca(ptr) do { assert(ptr); *ptr = (snd_ctl_elem_list_t *) alloca(snd_ctl_elem_list_sizeof()); memset(*ptr, 0, snd_ctl_elem_list_sizeof()); } while (0)
int snd_ctl_elem_list_malloc(snd_ctl_elem_list_t **ptr);
void snd_ctl_elem_list_free(snd_ctl_elem_list_t *obj);
void snd_ctl_elem_list_clear(snd_ctl_elem_list_t *obj);
void snd_ctl_elem_list_copy(snd_ctl_elem_list_t *dst, const snd_ctl_elem_list_t *src);
void snd_ctl_elem_list_set_offset(snd_ctl_elem_list_t *obj, unsigned int val);
unsigned int snd_ctl_elem_list_get_used(const snd_ctl_elem_list_t *obj);
unsigned int snd_ctl_elem_list_get_count(const snd_ctl_elem_list_t *obj);
void snd_ctl_elem_list_get_id(const snd_ctl_elem_list_t *obj, unsigned int idx, snd_ctl_elem_id_t *ptr);
unsigned int snd_ctl_elem_list_get_numid(const snd_ctl_elem_list_t *obj, unsigned int idx);
snd_ctl_elem_iface_t snd_ctl_elem_list_get_interface(const snd_ctl_elem_list_t *obj, unsigned int idx);
unsigned int snd_ctl_elem_list_get_device(const snd_ctl_elem_list_t *obj, unsigned int idx);
unsigned int snd_ctl_elem_list_get_subdevice(const snd_ctl_elem_list_t *obj, unsigned int idx);
const char *snd_ctl_elem_list_get_name(const snd_ctl_elem_list_t *obj, unsigned int idx);
unsigned int snd_ctl_elem_list_get_index(const snd_ctl_elem_list_t *obj, unsigned int idx);

size_t snd_ctl_elem_info_sizeof(void);
/** \hideinitializer
 * \brief allocate an invalid #snd_ctl_elem_info_t using standard alloca
 * \param ptr returned pointer
 */
#define snd_ctl_elem_info_alloca(ptr) do { assert(ptr); *ptr = (snd_ctl_elem_info_t *) alloca(snd_ctl_elem_info_sizeof()); memset(*ptr, 0, snd_ctl_elem_info_sizeof()); } while (0)
int snd_ctl_elem_info_malloc(snd_ctl_elem_info_t **ptr);
void snd_ctl_elem_info_free(snd_ctl_elem_info_t *obj);
void snd_ctl_elem_info_clear(snd_ctl_elem_info_t *obj);
void snd_ctl_elem_info_copy(snd_ctl_elem_info_t *dst, const snd_ctl_elem_info_t *src);
snd_ctl_elem_type_t snd_ctl_elem_info_get_type(const snd_ctl_elem_info_t *obj);
int snd_ctl_elem_info_is_readable(const snd_ctl_elem_info_t *obj);
int snd_ctl_elem_info_is_writable(const snd_ctl_elem_info_t *obj);
int snd_ctl_elem_info_is_volatile(const snd_ctl_elem_info_t *obj);
int snd_ctl_elem_info_is_inactive(const snd_ctl_elem_info_t *obj);
int snd_ctl_elem_info_is_locked(const snd_ctl_elem_info_t *obj);
int snd_ctl_elem_info_is_owner(const snd_ctl_elem_info_t *obj);
int snd_ctl_elem_info_is_indirect(const snd_ctl_elem_info_t *obj);
pid_t snd_ctl_elem_info_get_owner(const snd_ctl_elem_info_t *obj);
unsigned int snd_ctl_elem_info_get_count(const snd_ctl_elem_info_t *obj);
long snd_ctl_elem_info_get_min(const snd_ctl_elem_info_t *obj);
long snd_ctl_elem_info_get_max(const snd_ctl_elem_info_t *obj);
long snd_ctl_elem_info_get_step(const snd_ctl_elem_info_t *obj);
unsigned int snd_ctl_elem_info_get_items(const snd_ctl_elem_info_t *obj);
void snd_ctl_elem_info_set_item(snd_ctl_elem_info_t *obj, unsigned int val);
const char *snd_ctl_elem_info_get_item_name(const snd_ctl_elem_info_t *obj);
void snd_ctl_elem_info_get_id(const snd_ctl_elem_info_t *obj, snd_ctl_elem_id_t *ptr);
unsigned int snd_ctl_elem_info_get_numid(const snd_ctl_elem_info_t *obj);
snd_ctl_elem_iface_t snd_ctl_elem_info_get_interface(const snd_ctl_elem_info_t *obj);
unsigned int snd_ctl_elem_info_get_device(const snd_ctl_elem_info_t *obj);
unsigned int snd_ctl_elem_info_get_subdevice(const snd_ctl_elem_info_t *obj);
const char *snd_ctl_elem_info_get_name(const snd_ctl_elem_info_t *obj);
unsigned int snd_ctl_elem_info_get_index(const snd_ctl_elem_info_t *obj);
void snd_ctl_elem_info_set_id(snd_ctl_elem_info_t *obj, const snd_ctl_elem_id_t *ptr);
void snd_ctl_elem_info_set_numid(snd_ctl_elem_info_t *obj, unsigned int val);
void snd_ctl_elem_info_set_interface(snd_ctl_elem_info_t *obj, snd_ctl_elem_iface_t val);
void snd_ctl_elem_info_set_device(snd_ctl_elem_info_t *obj, unsigned int val);
void snd_ctl_elem_info_set_subdevice(snd_ctl_elem_info_t *obj, unsigned int val);
void snd_ctl_elem_info_set_name(snd_ctl_elem_info_t *obj, const char *val);
void snd_ctl_elem_info_set_index(snd_ctl_elem_info_t *obj, unsigned int val);

size_t snd_ctl_elem_value_sizeof(void);
/** \hideinitializer
 * \brief allocate an invalid #snd_ctl_elem_value_t using standard alloca
 * \param ptr returned pointer
 */
#define snd_ctl_elem_value_alloca(ptr) do { assert(ptr); *ptr = (snd_ctl_elem_value_t *) alloca(snd_ctl_elem_value_sizeof()); memset(*ptr, 0, snd_ctl_elem_value_sizeof()); } while (0)
int snd_ctl_elem_value_malloc(snd_ctl_elem_value_t **ptr);
void snd_ctl_elem_value_free(snd_ctl_elem_value_t *obj);
void snd_ctl_elem_value_clear(snd_ctl_elem_value_t *obj);
void snd_ctl_elem_value_copy(snd_ctl_elem_value_t *dst, const snd_ctl_elem_value_t *src);
void snd_ctl_elem_value_get_id(const snd_ctl_elem_value_t *obj, snd_ctl_elem_id_t *ptr);
unsigned int snd_ctl_elem_value_get_numid(const snd_ctl_elem_value_t *obj);
snd_ctl_elem_iface_t snd_ctl_elem_value_get_interface(const snd_ctl_elem_value_t *obj);
unsigned int snd_ctl_elem_value_get_device(const snd_ctl_elem_value_t *obj);
unsigned int snd_ctl_elem_value_get_subdevice(const snd_ctl_elem_value_t *obj);
const char *snd_ctl_elem_value_get_name(const snd_ctl_elem_value_t *obj);
unsigned int snd_ctl_elem_value_get_index(const snd_ctl_elem_value_t *obj);
void snd_ctl_elem_value_set_id(snd_ctl_elem_value_t *obj, const snd_ctl_elem_id_t *ptr);
void snd_ctl_elem_value_set_numid(snd_ctl_elem_value_t *obj, unsigned int val);
void snd_ctl_elem_value_set_interface(snd_ctl_elem_value_t *obj, snd_ctl_elem_iface_t val);
void snd_ctl_elem_value_set_device(snd_ctl_elem_value_t *obj, unsigned int val);
void snd_ctl_elem_value_set_subdevice(snd_ctl_elem_value_t *obj, unsigned int val);
void snd_ctl_elem_value_set_name(snd_ctl_elem_value_t *obj, const char *val);
void snd_ctl_elem_value_set_index(snd_ctl_elem_value_t *obj, unsigned int val);
int snd_ctl_elem_value_get_boolean(const snd_ctl_elem_value_t *obj, unsigned int idx);
long snd_ctl_elem_value_get_integer(const snd_ctl_elem_value_t *obj, unsigned int idx);
unsigned int snd_ctl_elem_value_get_enumerated(const snd_ctl_elem_value_t *obj, unsigned int idx);
unsigned char snd_ctl_elem_value_get_byte(const snd_ctl_elem_value_t *obj, unsigned int idx);
void snd_ctl_elem_value_set_boolean(snd_ctl_elem_value_t *obj, unsigned int idx, long val);
void snd_ctl_elem_value_set_integer(snd_ctl_elem_value_t *obj, unsigned int idx, long val);
void snd_ctl_elem_value_set_enumerated(snd_ctl_elem_value_t *obj, unsigned int idx, unsigned int val);
void snd_ctl_elem_value_set_byte(snd_ctl_elem_value_t *obj, unsigned int idx, unsigned char val);
const void * snd_ctl_elem_value_get_bytes(const snd_ctl_elem_value_t *obj);
void snd_ctl_elem_value_get_iec958(const snd_ctl_elem_value_t *obj, snd_aes_iec958_t *ptr);
void snd_ctl_elem_value_set_iec958(snd_ctl_elem_value_t *obj, const snd_aes_iec958_t *ptr);

#ifdef __cplusplus
}
#endif

/**
 *  \defgroup HControl Highlevel Control Interface
 *  \ingroup Control
 *  The highlevel control interface.
 *  \{
 */

/** HCTL element handle */
typedef struct _snd_hctl_elem snd_hctl_elem_t;

/** HCTL handle */
typedef struct _snd_hctl snd_hctl_t;

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \brief Compare function for sorting HCTL elements
 * \param e1 First element
 * \param e2 Second element
 * \return -1 if e1 < e2, 0 if e1 == e2, 1 if e1 > e2
 */
typedef int (*snd_hctl_compare_t)(const snd_hctl_elem_t *e1,
				  const snd_hctl_elem_t *e2);
int snd_hctl_compare_fast(const snd_hctl_elem_t *c1,
			  const snd_hctl_elem_t *c2);
/** 
 * \brief HCTL callback function
 * \param hctl HCTL handle
 * \param mask event mask
 * \param elem related HCTL element (if any)
 * \return 0 on success otherwise a negative error code
 */
typedef int (*snd_hctl_callback_t)(snd_hctl_t *hctl,
				   unsigned int mask,
				   snd_hctl_elem_t *elem);
/** 
 * \brief HCTL element callback function
 * \param elem HCTL element
 * \param mask event mask
 * \return 0 on success otherwise a negative error code
 */
typedef int (*snd_hctl_elem_callback_t)(snd_hctl_elem_t *elem,
					unsigned int mask);

int snd_hctl_open(snd_hctl_t **hctl, const char *name, int mode);
int snd_hctl_close(snd_hctl_t *hctl);
int snd_hctl_nonblock(snd_hctl_t *hctl, int nonblock);
int snd_hctl_poll_descriptors_count(snd_hctl_t *hctl);
int snd_hctl_poll_descriptors(snd_hctl_t *hctl, struct pollfd *pfds, unsigned int space);
unsigned int snd_hctl_get_count(snd_hctl_t *hctl);
int snd_hctl_set_compare(snd_hctl_t *hctl, snd_hctl_compare_t hsort);
snd_hctl_elem_t *snd_hctl_first_elem(snd_hctl_t *hctl);
snd_hctl_elem_t *snd_hctl_last_elem(snd_hctl_t *hctl);
snd_hctl_elem_t *snd_hctl_find_elem(snd_hctl_t *hctl, const snd_ctl_elem_id_t *id);
void snd_hctl_set_callback(snd_hctl_t *hctl, snd_hctl_callback_t callback);
void snd_hctl_set_callback_private(snd_hctl_t *hctl, void *data);
void *snd_hctl_get_callback_private(snd_hctl_t *hctl);
int snd_hctl_load(snd_hctl_t *hctl);
int snd_hctl_free(snd_hctl_t *hctl);
int snd_hctl_handle_events(snd_hctl_t *hctl);
const char *snd_hctl_name(snd_hctl_t *hctl);
int snd_hctl_wait(snd_hctl_t *hctl, int timeout);

snd_hctl_elem_t *snd_hctl_elem_next(snd_hctl_elem_t *elem);
snd_hctl_elem_t *snd_hctl_elem_prev(snd_hctl_elem_t *elem);
int snd_hctl_elem_info(snd_hctl_elem_t *elem, snd_ctl_elem_info_t * info);
int snd_hctl_elem_read(snd_hctl_elem_t *elem, snd_ctl_elem_value_t * value);
int snd_hctl_elem_write(snd_hctl_elem_t *elem, snd_ctl_elem_value_t * value);

snd_hctl_t *snd_hctl_elem_get_hctl(snd_hctl_elem_t *elem);

void snd_hctl_elem_get_id(const snd_hctl_elem_t *obj, snd_ctl_elem_id_t *ptr);
unsigned int snd_hctl_elem_get_numid(const snd_hctl_elem_t *obj);
snd_ctl_elem_iface_t snd_hctl_elem_get_interface(const snd_hctl_elem_t *obj);
unsigned int snd_hctl_elem_get_device(const snd_hctl_elem_t *obj);
unsigned int snd_hctl_elem_get_subdevice(const snd_hctl_elem_t *obj);
const char *snd_hctl_elem_get_name(const snd_hctl_elem_t *obj);
unsigned int snd_hctl_elem_get_index(const snd_hctl_elem_t *obj);
void snd_hctl_elem_set_callback(snd_hctl_elem_t *obj, snd_hctl_elem_callback_t val);
void * snd_hctl_elem_get_callback_private(const snd_hctl_elem_t *obj);
void snd_hctl_elem_set_callback_private(snd_hctl_elem_t *obj, void * val);

#ifdef __cplusplus
}
#endif

/** \} */
/** \} */

/****************************************************************************
 *                                                                          *
 *                               mixer.h                                    *
 *                           Mixer Interface                                *
 *                                                                          *
 ****************************************************************************/

/**
 *  \defgroup Mixer Mixer Interface
 *  The mixer interface.
 *  \{
 */

/** Mixer handle */
typedef struct _snd_mixer snd_mixer_t;
/** Mixer elements class handle */
typedef struct _snd_mixer_class snd_mixer_class_t;
/** Mixer element handle */
typedef struct _snd_mixer_elem snd_mixer_elem_t;

/** 
 * \brief Mixer callback function
 * \param mixer Mixer handle
 * \param mask event mask
 * \param elem related mixer element (if any)
 * \return 0 on success otherwise a negative error code
 */
typedef int (*snd_mixer_callback_t)(snd_mixer_t *ctl,
				    unsigned int mask,
				    snd_mixer_elem_t *elem);

/** 
 * \brief Mixer element callback function
 * \param elem Mixer element
 * \param mask event mask
 * \return 0 on success otherwise a negative error code
 */
typedef int (*snd_mixer_elem_callback_t)(snd_mixer_elem_t *elem,
					 unsigned int mask);

/**
 * \brief Compare function for sorting mixer elements
 * \param e1 First element
 * \param e2 Second element
 * \return -1 if e1 < e2, 0 if e1 == e2, 1 if e1 > e2
 */
typedef int (*snd_mixer_compare_t)(const snd_mixer_elem_t *e1,
				   const snd_mixer_elem_t *e2);

/** Mixer element type */
typedef enum _snd_mixer_elem_type {
	/* Simple (legacy) mixer elements */
	SND_MIXER_ELEM_SIMPLE,
	SND_MIXER_ELEM_LAST = SND_MIXER_ELEM_SIMPLE,
} snd_mixer_elem_type_t;

#ifdef __cplusplus
extern "C" {
#endif

int snd_mixer_open(snd_mixer_t **mixer, int mode);
int snd_mixer_close(snd_mixer_t *mixer);
snd_mixer_elem_t *snd_mixer_first_elem(snd_mixer_t *mixer);
snd_mixer_elem_t *snd_mixer_last_elem(snd_mixer_t *mixer);
int snd_mixer_handle_events(snd_mixer_t *mixer);
int snd_mixer_attach(snd_mixer_t *mixer, const char *name);
int snd_mixer_detach(snd_mixer_t *mixer, const char *name);
int snd_mixer_poll_descriptors_count(snd_mixer_t *mixer);
int snd_mixer_poll_descriptors(snd_mixer_t *mixer, struct pollfd *pfds, unsigned int space);
int snd_mixer_load(snd_mixer_t *mixer);
void snd_mixer_free(snd_mixer_t *mixer);
int snd_mixer_wait(snd_mixer_t *mixer, int timeout);
int snd_mixer_set_compare(snd_mixer_t *mixer, snd_mixer_compare_t msort);

snd_mixer_elem_t *snd_mixer_elem_next(snd_mixer_elem_t *elem);
snd_mixer_elem_t *snd_mixer_elem_prev(snd_mixer_elem_t *elem);

int snd_mixer_class_unregister(snd_mixer_class_t *clss);

#ifdef __cplusplus
}
#endif

/**
 *  \defgroup SimpleMixer Simple Mixer Interface
 *  \ingroup Mixer
 *  The simple mixer interface.
 *  \{
 */

/* Simple (legacy) mixer elements API */

/** Mixer simple element channel identificator */
typedef enum _snd_mixer_selem_channel_id {
	/** Unknown */
	SND_MIXER_SCHN_UNKNOWN = -1,
	/** Front left */
	SND_MIXER_SCHN_FRONT_LEFT = 0,
	/** Front right */
	SND_MIXER_SCHN_FRONT_RIGHT,
	/** Front center */
	SND_MIXER_SCHN_FRONT_CENTER,
	/** Rear left */
	SND_MIXER_SCHN_REAR_LEFT,
	/** Rear right */
	SND_MIXER_SCHN_REAR_RIGHT,
	/** Woofer */
	SND_MIXER_SCHN_WOOFER,
	SND_MIXER_SCHN_LAST = 31,
	/** Mono (Front left alias) */
	SND_MIXER_SCHN_MONO = SND_MIXER_SCHN_FRONT_LEFT
} snd_mixer_selem_channel_id_t;

/** Mixer simple element identificator */
typedef struct _snd_mixer_selem_id snd_mixer_selem_id_t;

#ifdef __cplusplus
extern "C" {
#endif

const char *snd_mixer_selem_channel_name(snd_mixer_selem_channel_id_t channel);

int snd_mixer_selem_register(snd_mixer_t *mixer, void *arg,
			     snd_mixer_class_t **classp);
void snd_mixer_selem_get_id(snd_mixer_elem_t *element,
			    snd_mixer_selem_id_t *id);
const char *snd_mixer_selem_get_name(snd_mixer_elem_t *elem);
unsigned int snd_mixer_selem_get_index(snd_mixer_elem_t *elem);
snd_mixer_elem_t *snd_mixer_find_selem(snd_mixer_t *mixer,
				       const snd_mixer_selem_id_t *id);

int snd_mixer_selem_is_playback_mono(snd_mixer_elem_t *elem);
int snd_mixer_selem_has_playback_channel(snd_mixer_elem_t *obj, snd_mixer_selem_channel_id_t channel);
int snd_mixer_selem_is_capture_mono(snd_mixer_elem_t *elem);
int snd_mixer_selem_has_capture_channel(snd_mixer_elem_t *obj, snd_mixer_selem_channel_id_t channel);
int snd_mixer_selem_get_capture_group(snd_mixer_elem_t *elem);
int snd_mixer_selem_has_common_volume(snd_mixer_elem_t *elem);
int snd_mixer_selem_has_playback_volume(snd_mixer_elem_t *elem);
int snd_mixer_selem_has_playback_volume_joined(snd_mixer_elem_t *elem);
int snd_mixer_selem_has_capture_volume(snd_mixer_elem_t *elem);
int snd_mixer_selem_has_capture_volume_joined(snd_mixer_elem_t *elem);
int snd_mixer_selem_has_common_switch(snd_mixer_elem_t *elem);
int snd_mixer_selem_has_playback_switch(snd_mixer_elem_t *elem);
int snd_mixer_selem_has_playback_switch_joined(snd_mixer_elem_t *elem);
int snd_mixer_selem_has_capture_switch(snd_mixer_elem_t *elem);
int snd_mixer_selem_has_capture_switch_joined(snd_mixer_elem_t *elem);
int snd_mixer_selem_has_capture_switch_exclusive(snd_mixer_elem_t *elem);

int snd_mixer_selem_get_playback_volume(snd_mixer_elem_t *elem, snd_mixer_selem_channel_id_t channel, long *value);
int snd_mixer_selem_get_capture_volume(snd_mixer_elem_t *elem, snd_mixer_selem_channel_id_t channel, long *value);
int snd_mixer_selem_get_playback_switch(snd_mixer_elem_t *elem, snd_mixer_selem_channel_id_t channel, int *value);
int snd_mixer_selem_get_capture_switch(snd_mixer_elem_t *elem, snd_mixer_selem_channel_id_t channel, int *value);
int snd_mixer_selem_set_playback_volume(snd_mixer_elem_t *elem, snd_mixer_selem_channel_id_t channel, long value);
int snd_mixer_selem_set_capture_volume(snd_mixer_elem_t *elem, snd_mixer_selem_channel_id_t channel, long value);
int snd_mixer_selem_set_playback_volume_all(snd_mixer_elem_t *elem, long value);
int snd_mixer_selem_set_capture_volume_all(snd_mixer_elem_t *elem, long value);
int snd_mixer_selem_set_playback_switch(snd_mixer_elem_t *elem, snd_mixer_selem_channel_id_t channel, int value);
int snd_mixer_selem_set_capture_switch(snd_mixer_elem_t *elem, snd_mixer_selem_channel_id_t channel, int value);
int snd_mixer_selem_set_playback_switch_all(snd_mixer_elem_t *elem, int value);
int snd_mixer_selem_set_capture_switch_all(snd_mixer_elem_t *elem, int value);
void snd_mixer_selem_get_playback_volume_range(snd_mixer_elem_t *elem, 
					       long *min, long *max);
void snd_mixer_selem_set_playback_volume_range(snd_mixer_elem_t *elem, 
					       long min, long max);
void snd_mixer_selem_get_capture_volume_range(snd_mixer_elem_t *elem, 
					      long *min, long *max);
void snd_mixer_selem_set_capture_volume_range(snd_mixer_elem_t *elem, 
					      long min, long max);

size_t snd_mixer_selem_id_sizeof(void);
/** \hideinitializer
 * \brief allocate an invalid #snd_mixer_selem_id_t using standard alloca
 * \param ptr returned pointer
 */
#define snd_mixer_selem_id_alloca(ptr) do { assert(ptr); *ptr = (snd_mixer_selem_id_t *) alloca(snd_mixer_selem_id_sizeof()); memset(*ptr, 0, snd_mixer_selem_id_sizeof()); } while (0)
int snd_mixer_selem_id_malloc(snd_mixer_selem_id_t **ptr);
void snd_mixer_selem_id_free(snd_mixer_selem_id_t *obj);
void snd_mixer_selem_id_copy(snd_mixer_selem_id_t *dst, const snd_mixer_selem_id_t *src);

const char *snd_mixer_selem_id_get_name(const snd_mixer_selem_id_t *obj);

unsigned int snd_mixer_selem_id_get_index(const snd_mixer_selem_id_t *obj);

void snd_mixer_selem_id_set_name(snd_mixer_selem_id_t *obj, const char *val);

void snd_mixer_selem_id_set_index(snd_mixer_selem_id_t *obj, unsigned int val);

void snd_mixer_set_callback(snd_mixer_t *obj, snd_mixer_callback_t val);

void * snd_mixer_get_callback_private(const snd_mixer_t *obj);

void snd_mixer_set_callback_private(snd_mixer_t *obj, void * val);

unsigned int snd_mixer_get_count(const snd_mixer_t *obj);

void snd_mixer_elem_set_callback(snd_mixer_elem_t *obj, snd_mixer_elem_callback_t val);

void * snd_mixer_elem_get_callback_private(const snd_mixer_elem_t *obj);

void snd_mixer_elem_set_callback_private(snd_mixer_elem_t *obj, void * val);

snd_mixer_elem_type_t snd_mixer_elem_get_type(const snd_mixer_elem_t *obj);


#ifdef __cplusplus
}
#endif

/** \} */
/** \} */

/**
 *  \defgroup Sequencer Sequencer Interface
 *  Sequencer Interface
 *  \{
 */

/** \} */

/**
 *  \defgroup SeqEvents Sequencer Event Definitions
 *  Sequencer Event Definitions
 *  \ingroup Sequencer
 *  \{
 */

/**
 * Sequencer event data type
 */
typedef unsigned char snd_seq_event_type_t;

/** Sequencer event type */
enum snd_seq_event_type {
	/** system status; event data type = #snd_seq_result_t */
	SND_SEQ_EVENT_SYSTEM = 0,
	/** returned result status; event data type = #snd_seq_result_t */
	SND_SEQ_EVENT_RESULT,

	/** note on and off with duration; event data type = #snd_seq_ev_note_t */
	SND_SEQ_EVENT_NOTE = 5,
	/** note on; event data type = #snd_seq_ev_note_t */
	SND_SEQ_EVENT_NOTEON,
	/** note off; event data type = #snd_seq_ev_note_t */
	SND_SEQ_EVENT_NOTEOFF,
	/** key pressure change (aftertouch); event data type = #snd_seq_ev_note_t */
	SND_SEQ_EVENT_KEYPRESS,
	
	/** controller; event data type = #snd_seq_ev_ctrl_t */
	SND_SEQ_EVENT_CONTROLLER = 10,
	/** program change; event data type = #snd_seq_ev_ctrl_t */
	SND_SEQ_EVENT_PGMCHANGE,
	/** channel pressure; event data type = #snd_seq_ev_ctrl_t */
	SND_SEQ_EVENT_CHANPRESS,
	/** pitchwheel; event data type = #snd_seq_ev_ctrl_t; data is from -8192 to 8191) */
	SND_SEQ_EVENT_PITCHBEND,
	/** 14 bit controller value; event data type = #snd_seq_ev_ctrl_t */
	SND_SEQ_EVENT_CONTROL14,
	/** 14 bit NRPN;  event data type = #snd_seq_ev_ctrl_t */
	SND_SEQ_EVENT_NONREGPARAM,
	/** 14 bit RPN; event data type = #snd_seq_ev_ctrl_t */
	SND_SEQ_EVENT_REGPARAM,

	/** SPP with LSB and MSB values; event data type = #snd_seq_ev_ctrl_t */
	SND_SEQ_EVENT_SONGPOS = 20,
	/** Song Select with song ID number; event data type = #snd_seq_ev_ctrl_t */
	SND_SEQ_EVENT_SONGSEL,
	/** midi time code quarter frame; event data type = #snd_seq_ev_ctrl_t */
	SND_SEQ_EVENT_QFRAME,
	/** SMF Time Signature event; event data type = #snd_seq_ev_ctrl_t */
	SND_SEQ_EVENT_TIMESIGN,
	/** SMF Key Signature event; event data type = #snd_seq_ev_ctrl_t */
	SND_SEQ_EVENT_KEYSIGN,
	        
	/** MIDI Real Time Start message; event data type = #snd_seq_ev_queue_control_t */
	SND_SEQ_EVENT_START = 30,
	/** MIDI Real Time Continue message; event data type = #snd_seq_ev_queue_control_t */
	SND_SEQ_EVENT_CONTINUE,
	/** MIDI Real Time Stop message; event data type = #snd_seq_ev_queue_control_t */
	SND_SEQ_EVENT_STOP,
	/** Set tick queue position; event data type = #snd_seq_ev_queue_control_t */
	SND_SEQ_EVENT_SETPOS_TICK,
	/** Set realtime queue position; event data type = #snd_seq_ev_queue_control_t */
	SND_SEQ_EVENT_SETPOS_TIME,
	/** (SMF) Tempo event; event data type = #snd_seq_ev_queue_control_t */
	SND_SEQ_EVENT_TEMPO,
	/** MIDI Real Time Clock message; event data type = #snd_seq_ev_queue_control_t */
	SND_SEQ_EVENT_CLOCK,
	/** MIDI Real Time Tick message; event data type = #snd_seq_ev_queue_control_t */
	SND_SEQ_EVENT_TICK,
	/** Sync signal; event data type = #snd_seq_ev_queue_control_t */
	SND_SEQ_EVENT_SYNC,
	/** Sync position changed; event data type = #snd_seq_ev_queue_control_t */
	SND_SEQ_EVENT_SYNC_POS,

	/** Tune request; event data type = none */
	SND_SEQ_EVENT_TUNE_REQUEST = 40,
	/** Reset to power-on state; event data type = none */
	SND_SEQ_EVENT_RESET,
	/** Active sensing event; event data type = none */
	SND_SEQ_EVENT_SENSING,

	/** Echo-back event; event data type = any type */
	SND_SEQ_EVENT_ECHO = 50,
	/** OSS emulation raw event; event data type = any type */
	SND_SEQ_EVENT_OSS,

	/** New client has connected; event data type = #snd_seq_addr_t */
	SND_SEQ_EVENT_CLIENT_START = 60,
	/** Client has left the system; event data type = #snd_seq_addr_t */
	SND_SEQ_EVENT_CLIENT_EXIT,
	/** Client status/info has changed; event data type = #snd_seq_addr_t */
	SND_SEQ_EVENT_CLIENT_CHANGE,
	/** New port was created; event data type = #snd_seq_addr_t */
	SND_SEQ_EVENT_PORT_START,
	/** Port was deleted from system; event data type = #snd_seq_addr_t */
	SND_SEQ_EVENT_PORT_EXIT,
	/** Port status/info has changed; event data type = #snd_seq_addr_t */
	SND_SEQ_EVENT_PORT_CHANGE,

	/** Ports connected; event data type = #snd_seq_connect_t */
	SND_SEQ_EVENT_PORT_SUBSCRIBED,
	/** Ports disconnected; event data type = #snd_seq_connect_t */
	SND_SEQ_EVENT_PORT_UNSUBSCRIBED,

	/** Sample select; event data type = #snd_seq_ev_sample_control_t */
	SND_SEQ_EVENT_SAMPLE = 70,
	/** Sample cluster select; event data type = #snd_seq_ev_sample_control_t */
	SND_SEQ_EVENT_SAMPLE_CLUSTER,
	/** voice start */
	SND_SEQ_EVENT_SAMPLE_START,
	/** voice stop */
	SND_SEQ_EVENT_SAMPLE_STOP,
	/** playback frequency */
	SND_SEQ_EVENT_SAMPLE_FREQ,
	/** volume and balance */
	SND_SEQ_EVENT_SAMPLE_VOLUME,
	/** sample loop */
	SND_SEQ_EVENT_SAMPLE_LOOP,
	/** sample position */
	SND_SEQ_EVENT_SAMPLE_POSITION,
	/** private (hardware dependent) event */
	SND_SEQ_EVENT_SAMPLE_PRIVATE1,

	/** user-defined event; event data type = any (fixed size) */
	SND_SEQ_EVENT_USR0 = 90,
	/** user-defined event; event data type = any (fixed size) */
	SND_SEQ_EVENT_USR1,
	/** user-defined event; event data type = any (fixed size) */
	SND_SEQ_EVENT_USR2,
	/** user-defined event; event data type = any (fixed size) */
	SND_SEQ_EVENT_USR3,
	/** user-defined event; event data type = any (fixed size) */
	SND_SEQ_EVENT_USR4,
	/** user-defined event; event data type = any (fixed size) */
	SND_SEQ_EVENT_USR5,
	/** user-defined event; event data type = any (fixed size) */
	SND_SEQ_EVENT_USR6,
	/** user-defined event; event data type = any (fixed size) */
	SND_SEQ_EVENT_USR7,
	/** user-defined event; event data type = any (fixed size) */
	SND_SEQ_EVENT_USR8,
	/** user-defined event; event data type = any (fixed size) */
	SND_SEQ_EVENT_USR9,

	/** begin of instrument management */
	SND_SEQ_EVENT_INSTR_BEGIN = 100,
	/** end of instrument management */
	SND_SEQ_EVENT_INSTR_END,
	/** query instrument interface info */
	SND_SEQ_EVENT_INSTR_INFO,
	/** result of instrument interface info */
	SND_SEQ_EVENT_INSTR_INFO_RESULT,
	/** query instrument format info */
	SND_SEQ_EVENT_INSTR_FINFO,
	/** result of instrument format info */
	SND_SEQ_EVENT_INSTR_FINFO_RESULT,
	/** reset instrument instrument memory */
	SND_SEQ_EVENT_INSTR_RESET,
	/** get instrument interface status */
	SND_SEQ_EVENT_INSTR_STATUS,
	/** result of instrument interface status */
	SND_SEQ_EVENT_INSTR_STATUS_RESULT,
	/** put an instrument to port */
	SND_SEQ_EVENT_INSTR_PUT,
	/** get an instrument from port */
	SND_SEQ_EVENT_INSTR_GET,
	/** result of instrument query */
	SND_SEQ_EVENT_INSTR_GET_RESULT,
	/** free instrument(s) */
	SND_SEQ_EVENT_INSTR_FREE,
	/** get instrument list */
	SND_SEQ_EVENT_INSTR_LIST,
	/** result of instrument list */
	SND_SEQ_EVENT_INSTR_LIST_RESULT,
	/** set cluster parameters */
	SND_SEQ_EVENT_INSTR_CLUSTER,
	/** get cluster parameters */
	SND_SEQ_EVENT_INSTR_CLUSTER_GET,
	/** result of cluster parameters */
	SND_SEQ_EVENT_INSTR_CLUSTER_RESULT,
	/** instrument change */
	SND_SEQ_EVENT_INSTR_CHANGE,

	/** system exclusive data (variable length);  event data type = #snd_seq_ev_ext_t */
	SND_SEQ_EVENT_SYSEX = 130,
	/** error event;  event data type = #snd_seq_ev_ext_t */
	SND_SEQ_EVENT_BOUNCE,
	/** reserved for user apps;  event data type = #snd_seq_ev_ext_t */
	SND_SEQ_EVENT_USR_VAR0 = 135,
	/** reserved for user apps; event data type = #snd_seq_ev_ext_t */
	SND_SEQ_EVENT_USR_VAR1,
	/** reserved for user apps; event data type = #snd_seq_ev_ext_t */
	SND_SEQ_EVENT_USR_VAR2,
	/** reserved for user apps; event data type = #snd_seq_ev_ext_t */
	SND_SEQ_EVENT_USR_VAR3,
	/** reserved for user apps; event data type = #snd_seq_ev_ext_t */
	SND_SEQ_EVENT_USR_VAR4,

	/** NOP; ignored in any case */
	SND_SEQ_EVENT_NONE = 255
};


#ifdef _SND_SEQ_IN_LOCAL
#include <sound/asequencer.h>
#else

/** Sequencer event address */
typedef struct snd_seq_addr {
	unsigned char client;	/**< Client id */
	unsigned char port;	/**< Port id */
} snd_seq_addr_t;

/** Connection (subscription) between ports */
typedef struct snd_seq_connect {
	snd_seq_addr_t sender;	/**< sender address */
	snd_seq_addr_t dest;	/**< destination address */
} snd_seq_connect_t;


/** Real-time data record */
typedef struct snd_seq_real_time {
	unsigned int tv_sec;		/**< seconds */
	unsigned int tv_nsec;		/**< nanoseconds */
} snd_seq_real_time_t;

/** (MIDI) Tick-time data record */
typedef unsigned int snd_seq_tick_time_t;

/** unioned time stamp */
typedef union snd_seq_timestamp {
	snd_seq_tick_time_t tick;	/**< tick-time */
	struct snd_seq_real_time time;	/**< real-time */
} snd_seq_timestamp_t;


/**
 * Event mode flags
 *
 * NOTE: only 8 bits available!
 */
#define SND_SEQ_TIME_STAMP_TICK		(0<<0)	/**< timestamp in clock ticks */
#define SND_SEQ_TIME_STAMP_REAL		(1<<0)	/**< timestamp in real time */
#define SND_SEQ_TIME_STAMP_MASK		(1<<0)	/**< mask for timestamp bits */

#define SND_SEQ_TIME_MODE_ABS		(0<<1)	/**< absolute timestamp */
#define SND_SEQ_TIME_MODE_REL		(1<<1)	/**< relative to current time */
#define SND_SEQ_TIME_MODE_MASK		(1<<1)	/**< mask for time mode bits */

#define SND_SEQ_EVENT_LENGTH_FIXED	(0<<2)	/**< fixed event size */
#define SND_SEQ_EVENT_LENGTH_VARIABLE	(1<<2)	/**< variable event size */
#define SND_SEQ_EVENT_LENGTH_VARUSR	(2<<2)	/**< variable event size - user memory space */
#define SND_SEQ_EVENT_LENGTH_VARIPC	(3<<2)	/**< variable event size - IPC */
#define SND_SEQ_EVENT_LENGTH_MASK	(3<<2)	/**< mask for event length bits */

#define SND_SEQ_PRIORITY_NORMAL		(0<<4)	/**< normal priority */
#define SND_SEQ_PRIORITY_HIGH		(1<<4)	/**< event should be processed before others */
#define SND_SEQ_PRIORITY_MASK		(1<<4)	/**< mask for priority bits */


/** Note event */
typedef struct snd_seq_ev_note {
	unsigned char channel;		/**< channel number */
	unsigned char note;		/**< note */
	unsigned char velocity;		/**< velocity */
	unsigned char off_velocity;	/**< note-off velocity; only for #SND_SEQ_EVENT_NOTE */
	unsigned int duration;		/**< duration until note-off; only for #SND_SEQ_EVENT_NOTE */
} snd_seq_ev_note_t;

/** Controller event */
typedef struct snd_seq_ev_ctrl {
	unsigned char channel;		/**< channel number */
	unsigned char unused[3];	/**< reserved */
	unsigned int param;		/**< control parameter */
	signed int value;		/**< control value */
} snd_seq_ev_ctrl_t;

/** generic set of bytes (12x8 bit) */
typedef struct snd_seq_ev_raw8 {
	unsigned char d[12];		/**< 8 bit value */
} snd_seq_ev_raw8_t;

/** generic set of integers (3x32 bit) */
typedef struct snd_seq_ev_raw32 {
	unsigned int d[3];		/**< 32 bit value */
} snd_seq_ev_raw32_t;

/** external stored data */
typedef struct snd_seq_ev_ext {
	size_t len;			/**< length of data */
	void *ptr;			/**< pointer to data (note: can be 64-bit) */
} snd_seq_ev_ext_t;

/** external stored data - IPC shared memory */
typedef struct snd_seq_ev_ipcshm {
	size_t len;			/**< length of data */
	key_t ipc;			/**< IPC key */
} snd_seq_ev_ipcshm_t;

/** Instrument cluster type */
typedef unsigned int snd_seq_instr_cluster_t;

/** Instrument type */
typedef struct snd_seq_instr {
	snd_seq_instr_cluster_t cluster;	/**< cluster id */
	unsigned int std;	/**< instrument standard id; the upper byte means a private instrument (owner - client id) */
	unsigned short bank;	/**< instrument bank id */
	unsigned short prg;	/**< instrument program id */
} snd_seq_instr_t;

/** sample number */
typedef struct snd_seq_ev_sample {
	unsigned int std;	/**< sample standard id */
	unsigned short bank;	/**< sample bank id */
	unsigned short prg;	/**< sample program id */
} snd_seq_ev_sample_t;

/** sample cluster */
typedef struct snd_seq_ev_cluster {
	snd_seq_instr_cluster_t cluster;	/**< cluster id */
} snd_seq_ev_cluster_t;

/** sample position */
typedef unsigned int snd_seq_position_t; /**< playback position (in samples) * 16 */

/** sample stop mode */
typedef enum snd_seq_stop_mode {
	SND_SEQ_SAMPLE_STOP_IMMEDIATELY = 0,	/**< terminate playing immediately */
	SND_SEQ_SAMPLE_STOP_VENVELOPE = 1,	/**< finish volume envelope */
	SDN_SEQ_SAMPLE_STOP_LOOP = 2		/**< terminate loop and finish wave */
} snd_seq_stop_mode_t;

/** sample frequency */
typedef int snd_seq_frequency_t; /**< playback frequency in HZ * 16 */

/** sample volume control; if any value is set to -1 == do not change */
typedef struct snd_seq_ev_volume {
	signed short volume;	/**< range: 0-16383 */
	signed short lr;	/**< left-right balance; range: 0-16383 */
	signed short fr;	/**< front-rear balance; range: 0-16383 */
	signed short du;	/**< down-up balance; range: 0-16383 */
} snd_seq_ev_volume_t;

/** simple loop redefinition */
typedef struct snd_seq_ev_loop {
	unsigned int start;	/**< loop start (in samples) * 16 */
	unsigned int end;	/**< loop end (in samples) * 16 */
} snd_seq_ev_loop_t;

/** Sample control events */
typedef struct snd_seq_ev_sample_control {
	unsigned char channel;		/**< channel */
	unsigned char unused[3];	/**< reserved */
	union {
		snd_seq_ev_sample_t sample;
		snd_seq_ev_cluster_t cluster;
		snd_seq_position_t position;
		snd_seq_stop_mode_t stop_mode;
		snd_seq_frequency_t frequency;
		snd_seq_ev_volume_t volume;
		snd_seq_ev_loop_t loop;
		unsigned char raw8[8];
	} param;		/**< control parameters */
} snd_seq_ev_sample_control_t;



/** INSTR_BEGIN event */
typedef struct snd_seq_ev_instr_begin {
	int timeout;		/**< zero = forever, otherwise timeout in ms */
} snd_seq_ev_instr_begin_t;

/** Result events */
typedef struct snd_seq_result {
	int event;		/**< processed event type */
	int result;		/**< status */
} snd_seq_result_t;


/** queue timer control */
typedef struct snd_seq_ev_queue_control {
	unsigned char queue;			/**< affected queue */
	unsigned char unused[3];		/**< reserved */
	union {
		signed int value;		/**< affected value (e.g. tempo) */
		snd_seq_timestamp_t time;	/**< time */
		unsigned int position;		/**< sync position */
		unsigned int d32[2];		/**< any data */
		unsigned char d8[8];		/**< any data */
	} param;				/**< data value union */
} snd_seq_ev_queue_control_t;


/** Sequencer event */
typedef struct snd_seq_event {
	snd_seq_event_type_t type;	/**< event type */
	unsigned char flags;		/**< event flags */
	unsigned char tag;		/**< tag */
	
	unsigned char queue;		/**< schedule queue */
	snd_seq_timestamp_t time;	/**< schedule time */

	snd_seq_addr_t source;		/**< source address */
	snd_seq_addr_t dest;		/**< destination address */

	union {
		snd_seq_ev_note_t note;
		snd_seq_ev_ctrl_t control;
		snd_seq_ev_raw8_t raw8;
		snd_seq_ev_raw32_t raw32;
		snd_seq_ev_ext_t ext;
		snd_seq_ev_ipcshm_t ipcshm;
		snd_seq_ev_queue_control_t queue;
		snd_seq_timestamp_t time;
		snd_seq_addr_t addr;
		snd_seq_connect_t connect;
		snd_seq_result_t result;
		snd_seq_ev_instr_begin_t instr_begin;
		snd_seq_ev_sample_control_t sample;
	} data;				/**< event data... */
} snd_seq_event_t;

#endif /* _SND_SEQ_IN_LOCAL */


/** \} */
#ifdef __cplusplus
extern "C" {
#endif

/**
 *  \defgroup SeqGlobal Sequencer System
 *  Global System Interface
 *  \ingroup Sequencer
 *  \{
 */

/** Sequencer handle */
typedef struct _snd_seq snd_seq_t;

/** \internal */
#define SND_ALLOCA(type,ptr) \
do {\
	assert(ptr);\
	*ptr = (type##_t *)alloca(type##_sizeof());\
	memset(*ptr, 0, type##_sizeof());\
} while (0)

/**
 * sequencer opening stream types
 */
#define SND_SEQ_OPEN_OUTPUT	1	/**< open for output (write) */
#define SND_SEQ_OPEN_INPUT	2	/**< open for input (read) */
#define SND_SEQ_OPEN_DUPLEX	(SND_SEQ_OPEN_OUTPUT|SND_SEQ_OPEN_INPUT)	/**< open for both input and output (read/write) */

/**
 * sequencer opening mode
 */
#define SND_SEQ_NONBLOCK	1	/**< non-blocking mode */

/** sequencer handle type */
typedef enum _snd_seq_type {
	SND_SEQ_TYPE_HW,		/**< hardware */
	SND_SEQ_TYPE_SHM,		/**< shared memory (NYI) */
	SND_SEQ_TYPE_INET,		/**< network (NYI) */
} snd_seq_type_t;

/** special client (port) ids */
#define SND_SEQ_ADDRESS_UNKNOWN		253	/**< unknown source */
#define SND_SEQ_ADDRESS_SUBSCRIBERS	254	/**< send event to all subscribed ports */
#define SND_SEQ_ADDRESS_BROADCAST	255	/**< send event to all queues/clients/ports/channels */

/** known client numbers */
#define SND_SEQ_CLIENT_SYSTEM		0	/**< system client */
#define SND_SEQ_CLIENT_DUMMY		62	/**< dummy ports */
#define SND_SEQ_CLIENT_OSS		63	/**< oss sequencer emulator */

/*
 */
int snd_seq_open(snd_seq_t **handle, const char *name, int streams, int mode);
const char *snd_seq_name(snd_seq_t *seq);
snd_seq_type_t snd_seq_type(snd_seq_t *seq);
int snd_seq_close(snd_seq_t *handle);
int snd_seq_poll_descriptors_count(snd_seq_t *handle, short events);
int snd_seq_poll_descriptors(snd_seq_t *handle, struct pollfd *pfds, unsigned int space, short events);
int snd_seq_nonblock(snd_seq_t *handle, int nonblock);
int snd_seq_client_id(snd_seq_t *handle);

size_t snd_seq_get_output_buffer_size(snd_seq_t *handle);
size_t snd_seq_get_input_buffer_size(snd_seq_t *handle);
int snd_seq_set_output_buffer_size(snd_seq_t *handle, size_t size);
int snd_seq_set_input_buffer_size(snd_seq_t *handle, size_t size);

/** system information container */
typedef struct _snd_seq_system_info snd_seq_system_info_t;

size_t snd_seq_system_info_sizeof(void);
/** allocate a #snd_seq_system_info_t container on stack */
#define snd_seq_system_info_alloca(ptr) \
	SND_ALLOCA(snd_seq_system_info, ptr)
int snd_seq_system_info_malloc(snd_seq_system_info_t **ptr);
void snd_seq_system_info_free(snd_seq_system_info_t *ptr);
void snd_seq_system_info_copy(snd_seq_system_info_t *dst, const snd_seq_system_info_t *src);

int snd_seq_system_info_get_queues(const snd_seq_system_info_t *info);
int snd_seq_system_info_get_clients(const snd_seq_system_info_t *info);
int snd_seq_system_info_get_ports(const snd_seq_system_info_t *info);
int snd_seq_system_info_get_channels(const snd_seq_system_info_t *info);

int snd_seq_system_info(snd_seq_t *handle, snd_seq_system_info_t *info);

/** \} */


/**
 *  \defgroup SeqClient Sequencer Client Interface
 *  Sequencer Client Interface
 *  \ingroup Sequencer
 *  \{
 */

/** client information container */
typedef struct _snd_seq_client_info snd_seq_client_info_t;

/** client types */
typedef enum snd_seq_client_type {
	SND_SEQ_USER_CLIENT     = 1,	/**< user client */
	SND_SEQ_KERNEL_CLIENT   = 2	/**< kernel client */
} snd_seq_client_type_t;
                        
size_t snd_seq_client_info_sizeof(void);
/** allocate a #snd_seq_client_info_t container on stack */
#define snd_seq_client_info_alloca(ptr) \
	SND_ALLOCA(snd_seq_client_info, ptr)
int snd_seq_client_info_malloc(snd_seq_client_info_t **ptr);
void snd_seq_client_info_free(snd_seq_client_info_t *ptr);
void snd_seq_client_info_copy(snd_seq_client_info_t *dst, const snd_seq_client_info_t *src);

int snd_seq_client_info_get_client(const snd_seq_client_info_t *info);
snd_seq_client_type_t snd_seq_client_info_get_type(const snd_seq_client_info_t *info);
const char *snd_seq_client_info_get_name(snd_seq_client_info_t *info);
int snd_seq_client_info_get_broadcast_filter(const snd_seq_client_info_t *info);
int snd_seq_client_info_get_error_bounce(const snd_seq_client_info_t *info);
const unsigned char *snd_seq_client_info_get_event_filter(const snd_seq_client_info_t *info);
int snd_seq_client_info_get_num_ports(const snd_seq_client_info_t *info);
int snd_seq_client_info_get_event_lost(const snd_seq_client_info_t *info);

void snd_seq_client_info_set_client(snd_seq_client_info_t *info, int client);
void snd_seq_client_info_set_name(snd_seq_client_info_t *info, const char *name);
void snd_seq_client_info_set_broadcast_filter(snd_seq_client_info_t *info, int val);
void snd_seq_client_info_set_error_bounce(snd_seq_client_info_t *info, int val);
void snd_seq_client_info_set_event_filter(snd_seq_client_info_t *info, unsigned char *filter);

int snd_seq_get_client_info(snd_seq_t *handle, snd_seq_client_info_t *info);
int snd_seq_get_any_client_info(snd_seq_t *handle, int client, snd_seq_client_info_t *info);
int snd_seq_set_client_info(snd_seq_t *handle, snd_seq_client_info_t *info);
int snd_seq_query_next_client(snd_seq_t *handle, snd_seq_client_info_t *info);

/*
 */

/** client pool information container */
typedef struct _snd_seq_client_pool snd_seq_client_pool_t;

size_t snd_seq_client_pool_sizeof(void);
/** allocate a #snd_seq_client_pool_t container on stack */
#define snd_seq_client_pool_alloca(ptr) \
	SND_ALLOCA(snd_seq_client_pool, ptr)
int snd_seq_client_pool_malloc(snd_seq_client_pool_t **ptr);
void snd_seq_client_pool_free(snd_seq_client_pool_t *ptr);
void snd_seq_client_pool_copy(snd_seq_client_pool_t *dst, const snd_seq_client_pool_t *src);

int snd_seq_client_pool_get_client(const snd_seq_client_pool_t *info);
size_t snd_seq_client_pool_get_output_pool(const snd_seq_client_pool_t *info);
size_t snd_seq_client_pool_get_input_pool(const snd_seq_client_pool_t *info);
size_t snd_seq_client_pool_get_output_room(const snd_seq_client_pool_t *info);
size_t snd_seq_client_pool_get_output_free(const snd_seq_client_pool_t *info);
size_t snd_seq_client_pool_get_input_free(const snd_seq_client_pool_t *info);
void snd_seq_client_pool_set_output_pool(snd_seq_client_pool_t *info, size_t size);
void snd_seq_client_pool_set_input_pool(snd_seq_client_pool_t *info, size_t size);
void snd_seq_client_pool_set_output_room(snd_seq_client_pool_t *info, size_t size);

int snd_seq_get_client_pool(snd_seq_t *handle, snd_seq_client_pool_t *info);
int snd_seq_set_client_pool(snd_seq_t *handle, snd_seq_client_pool_t *info);


/** \} */


/**
 *  \defgroup SeqPort Sequencer Port Interface
 *  Sequencer Port Interface
 *  \ingroup Sequencer
 *  \{
 */

/** port information container */
typedef struct _snd_seq_port_info snd_seq_port_info_t;

/** known port numbers */
#define SND_SEQ_PORT_SYSTEM_TIMER	0	/**< system timer port */
#define SND_SEQ_PORT_SYSTEM_ANNOUNCE	1	/**< sysem announce port */

/** port capabilities (32 bits) */
#define SND_SEQ_PORT_CAP_READ		(1<<0)	/**< readable from this port */
#define SND_SEQ_PORT_CAP_WRITE		(1<<1)	/**< writable to this port */

#define SND_SEQ_PORT_CAP_SYNC_READ	(1<<2)	/**< allow read subscriptions */
#define SND_SEQ_PORT_CAP_SYNC_WRITE	(1<<3)	/**< allow write subscriptions */

#define SND_SEQ_PORT_CAP_DUPLEX		(1<<4)	/**< allow read/write duplex */

#define SND_SEQ_PORT_CAP_SUBS_READ	(1<<5)	/**< allow read subscription */
#define SND_SEQ_PORT_CAP_SUBS_WRITE	(1<<6)	/**< allow write subscription */
#define SND_SEQ_PORT_CAP_NO_EXPORT	(1<<7)	/**< routing not allowed */

/** port type */
#define SND_SEQ_PORT_TYPE_SPECIFIC	(1<<0)	/**< hardware specific */
#define SND_SEQ_PORT_TYPE_MIDI_GENERIC	(1<<1)	/**< generic MIDI device */
#define SND_SEQ_PORT_TYPE_MIDI_GM	(1<<2)	/**< General MIDI compatible device */
#define SND_SEQ_PORT_TYPE_MIDI_GS	(1<<3)	/**< GS compatible device */
#define SND_SEQ_PORT_TYPE_MIDI_XG	(1<<4)	/**< XG compatible device */
#define SND_SEQ_PORT_TYPE_MIDI_MT32	(1<<5)	/**< MT-32 compatible device */
#define SND_SEQ_PORT_TYPE_SYNTH		(1<<10)	/**< Synth device */
#define SND_SEQ_PORT_TYPE_DIRECT_SAMPLE (1<<11)	/**< Sampling device (support sample download) */
#define SND_SEQ_PORT_TYPE_SAMPLE	(1<<12)	/**< Sampling device (sample can be downloaded at any time) */
#define SND_SEQ_PORT_TYPE_APPLICATION	(1<<20)	/**< application (sequencer/editor) */


size_t snd_seq_port_info_sizeof(void);
/** allocate a #snd_seq_port_info_t container on stack */
#define snd_seq_port_info_alloca(ptr) \
	SND_ALLOCA(snd_seq_port_info, ptr)
int snd_seq_port_info_malloc(snd_seq_port_info_t **ptr);
void snd_seq_port_info_free(snd_seq_port_info_t *ptr);
void snd_seq_port_info_copy(snd_seq_port_info_t *dst, const snd_seq_port_info_t *src);

int snd_seq_port_info_get_client(const snd_seq_port_info_t *info);
int snd_seq_port_info_get_port(const snd_seq_port_info_t *info);
const snd_seq_addr_t *snd_seq_port_info_get_addr(const snd_seq_port_info_t *info);
const char *snd_seq_port_info_get_name(const snd_seq_port_info_t *info);
unsigned int snd_seq_port_info_get_capability(const snd_seq_port_info_t *info);
unsigned int snd_seq_port_info_get_type(const snd_seq_port_info_t *info);
int snd_seq_port_info_get_midi_channels(const snd_seq_port_info_t *info);
int snd_seq_port_info_get_midi_voices(const snd_seq_port_info_t *info);
int snd_seq_port_info_get_synth_voices(const snd_seq_port_info_t *info);
int snd_seq_port_info_get_read_use(const snd_seq_port_info_t *info);
int snd_seq_port_info_get_write_use(const snd_seq_port_info_t *info);
int snd_seq_port_info_get_port_specified(const snd_seq_port_info_t *info);

void snd_seq_port_info_set_client(snd_seq_port_info_t *info, int client);
void snd_seq_port_info_set_port(snd_seq_port_info_t *info, int port);
void snd_seq_port_info_set_addr(snd_seq_port_info_t *info, const snd_seq_addr_t *addr);
void snd_seq_port_info_set_name(snd_seq_port_info_t *info, const char *name);
void snd_seq_port_info_set_capability(snd_seq_port_info_t *info, unsigned int capability);
void snd_seq_port_info_set_type(snd_seq_port_info_t *info, unsigned int type);
void snd_seq_port_info_set_midi_channels(snd_seq_port_info_t *info, int channels);
void snd_seq_port_info_set_midi_voices(snd_seq_port_info_t *info, int voices);
void snd_seq_port_info_set_synth_voices(snd_seq_port_info_t *info, int voices);
void snd_seq_port_info_set_port_specified(snd_seq_port_info_t *info, int val);

int snd_seq_create_port(snd_seq_t *handle, snd_seq_port_info_t *info);
int snd_seq_delete_port(snd_seq_t *handle, int port);
int snd_seq_get_port_info(snd_seq_t *handle, int port, snd_seq_port_info_t *info);
int snd_seq_get_any_port_info(snd_seq_t *handle, int client, int port, snd_seq_port_info_t *info);
int snd_seq_set_port_info(snd_seq_t *handle, int port, snd_seq_port_info_t *info);
int snd_seq_query_next_port(snd_seq_t *handle, snd_seq_port_info_t *info);

/** \} */


/**
 *  \defgroup SeqSubscribe Sequencer Port Subscription
 *  Sequencer Port Subscription
 *  \ingroup Sequencer
 *  \{
 */

/** port subscription container */
typedef struct _snd_seq_port_subscribe snd_seq_port_subscribe_t;

size_t snd_seq_port_subscribe_sizeof(void);
/** allocate a #snd_seq_port_subscribe_t container on stack */
#define snd_seq_port_subscribe_alloca(ptr) \
	SND_ALLOCA(snd_seq_port_subscribe, ptr)
int snd_seq_port_subscribe_malloc(snd_seq_port_subscribe_t **ptr);
void snd_seq_port_subscribe_free(snd_seq_port_subscribe_t *ptr);
void snd_seq_port_subscribe_copy(snd_seq_port_subscribe_t *dst, const snd_seq_port_subscribe_t *src);

const snd_seq_addr_t *snd_seq_port_subscribe_get_sender(const snd_seq_port_subscribe_t *info);
const snd_seq_addr_t *snd_seq_port_subscribe_get_dest(const snd_seq_port_subscribe_t *info);
int snd_seq_port_subscribe_get_queue(const snd_seq_port_subscribe_t *info);
int snd_seq_port_subscribe_get_exclusive(const snd_seq_port_subscribe_t *info);
int snd_seq_port_subscribe_get_time_update(const snd_seq_port_subscribe_t *info);
int snd_seq_port_subscribe_get_time_real(const snd_seq_port_subscribe_t *info);

void snd_seq_port_subscribe_set_sender(snd_seq_port_subscribe_t *info, const snd_seq_addr_t *addr);
void snd_seq_port_subscribe_set_dest(snd_seq_port_subscribe_t *info, const snd_seq_addr_t *addr);
void snd_seq_port_subscribe_set_queue(snd_seq_port_subscribe_t *info, int q);
void snd_seq_port_subscribe_set_exclusive(snd_seq_port_subscribe_t *info, int val);
void snd_seq_port_subscribe_set_time_update(snd_seq_port_subscribe_t *info, int val);
void snd_seq_port_subscribe_set_time_real(snd_seq_port_subscribe_t *info, int val);

int snd_seq_get_port_subscription(snd_seq_t *handle, snd_seq_port_subscribe_t *sub);
int snd_seq_subscribe_port(snd_seq_t *handle, snd_seq_port_subscribe_t *sub);
int snd_seq_unsubscribe_port(snd_seq_t *handle, snd_seq_port_subscribe_t *sub);

/*
 */

/** subscription query container */
typedef struct _snd_seq_query_subscribe snd_seq_query_subscribe_t;

/** type of query subscription */
typedef enum {
	SND_SEQ_QUERY_SUBS_READ,	/**< query read subscriptions */
	SND_SEQ_QUERY_SUBS_WRITE	/**< query write subscriptions */
} snd_seq_query_subs_type_t;

size_t snd_seq_query_subscribe_sizeof(void);
/** allocate a #snd_seq_query_subscribe_t container on stack */
#define snd_seq_query_subscribe_alloca(ptr) \
	SND_ALLOCA(snd_seq_query_subscribe, ptr)
int snd_seq_query_subscribe_malloc(snd_seq_query_subscribe_t **ptr);
void snd_seq_query_subscribe_free(snd_seq_query_subscribe_t *ptr);
void snd_seq_query_subscribe_copy(snd_seq_query_subscribe_t *dst, const snd_seq_query_subscribe_t *src);

int snd_seq_query_subscribe_get_client(const snd_seq_query_subscribe_t *info);
int snd_seq_query_subscribe_get_port(const snd_seq_query_subscribe_t *info);
const snd_seq_addr_t *snd_seq_query_subscribe_get_root(const snd_seq_query_subscribe_t *info);
snd_seq_query_subs_type_t snd_seq_query_subscribe_get_type(const snd_seq_query_subscribe_t *info);
int snd_seq_query_subscribe_get_index(const snd_seq_query_subscribe_t *info);
int snd_seq_query_subscribe_get_num_subs(const snd_seq_query_subscribe_t *info);
const snd_seq_addr_t *snd_seq_query_subscribe_get_addr(const snd_seq_query_subscribe_t *info);
int snd_seq_query_subscribe_get_queue(const snd_seq_query_subscribe_t *info);
int snd_seq_query_subscribe_get_exclusive(const snd_seq_query_subscribe_t *info);
int snd_seq_query_subscribe_get_time_update(const snd_seq_query_subscribe_t *info);
int snd_seq_query_subscribe_get_time_real(const snd_seq_query_subscribe_t *info);

void snd_seq_query_subscribe_set_client(snd_seq_query_subscribe_t *info, int client);
void snd_seq_query_subscribe_set_port(snd_seq_query_subscribe_t *info, int port);
void snd_seq_query_subscribe_set_root(snd_seq_query_subscribe_t *info, const snd_seq_addr_t *addr);
void snd_seq_query_subscribe_set_type(snd_seq_query_subscribe_t *info, snd_seq_query_subs_type_t type);
void snd_seq_query_subscribe_set_index(snd_seq_query_subscribe_t *info, int index);

int snd_seq_query_port_subscribers(snd_seq_t *seq, snd_seq_query_subscribe_t * subs);

/** \} */


/**
 *  \defgroup SeqQueue Sequencer Queue Interface
 *  Sequencer Queue Interface
 *  \ingroup Sequencer
 *  \{
 */

/** queue information container */
typedef struct _snd_seq_queue_info snd_seq_queue_info_t;
/** queue status container */
typedef struct _snd_seq_queue_status snd_seq_queue_status_t;
/** queue tempo container */
typedef struct _snd_seq_queue_tempo snd_seq_queue_tempo_t;
/** queue timer information container */
typedef struct _snd_seq_queue_timer snd_seq_queue_timer_t;

/** special queue ids */
#define SND_SEQ_QUEUE_DIRECT		253	/**< direct dispatch */

size_t snd_seq_queue_info_sizeof(void);
/** allocate a #snd_seq_queue_info_t container on stack */
#define snd_seq_queue_info_alloca(ptr) \
	SND_ALLOCA(snd_seq_queue_info, ptr)
int snd_seq_queue_info_malloc(snd_seq_queue_info_t **ptr);
void snd_seq_queue_info_free(snd_seq_queue_info_t *ptr);
void snd_seq_queue_info_copy(snd_seq_queue_info_t *dst, const snd_seq_queue_info_t *src);

int snd_seq_queue_info_get_queue(const snd_seq_queue_info_t *info);
const char *snd_seq_queue_info_get_name(const snd_seq_queue_info_t *info);
int snd_seq_queue_info_get_owner(const snd_seq_queue_info_t *info);
int snd_seq_queue_info_get_locked(const snd_seq_queue_info_t *info);
unsigned int snd_seq_queue_info_get_flags(const snd_seq_queue_info_t *info);

void snd_seq_queue_info_set_name(snd_seq_queue_info_t *info, const char *name);
void snd_seq_queue_info_set_owner(snd_seq_queue_info_t *info, int owner);
void snd_seq_queue_info_set_locked(snd_seq_queue_info_t *info, int locked);
void snd_seq_queue_info_set_flags(snd_seq_queue_info_t *info, unsigned int flags);

int snd_seq_create_queue(snd_seq_t *seq, snd_seq_queue_info_t *info);
int snd_seq_alloc_named_queue(snd_seq_t *seq, const char *name);
int snd_seq_alloc_queue(snd_seq_t *handle);
int snd_seq_free_queue(snd_seq_t *handle, int q);
int snd_seq_get_queue_info(snd_seq_t *seq, int q, snd_seq_queue_info_t *info);
int snd_seq_set_queue_info(snd_seq_t *seq, int q, snd_seq_queue_info_t *info);
int snd_seq_query_named_queue(snd_seq_t *seq, const char *name);

int snd_seq_get_queue_usage(snd_seq_t *handle, int q);
int snd_seq_set_queue_usage(snd_seq_t *handle, int q, int used);

/*
 */
size_t snd_seq_queue_status_sizeof(void);
/** allocate a #snd_seq_queue_status_t container on stack */
#define snd_seq_queue_status_alloca(ptr) \
	SND_ALLOCA(snd_seq_queue_status, ptr)
int snd_seq_queue_status_malloc(snd_seq_queue_status_t **ptr);
void snd_seq_queue_status_free(snd_seq_queue_status_t *ptr);
void snd_seq_queue_status_copy(snd_seq_queue_status_t *dst, const snd_seq_queue_status_t *src);

int snd_seq_queue_status_get_queue(const snd_seq_queue_status_t *info);
int snd_seq_queue_status_get_events(const snd_seq_queue_status_t *info);
snd_seq_tick_time_t snd_seq_queue_status_get_tick_time(const snd_seq_queue_status_t *info);
const snd_seq_real_time_t *snd_seq_queue_status_get_real_time(const snd_seq_queue_status_t *info);
unsigned int snd_seq_queue_status_get_status(const snd_seq_queue_status_t *info);

int snd_seq_get_queue_status(snd_seq_t *handle, int q, snd_seq_queue_status_t *status);

/*
 */
size_t snd_seq_queue_tempo_sizeof(void);
/** allocate a #snd_seq_queue_tempo_t container on stack */
#define snd_seq_queue_tempo_alloca(ptr) \
	SND_ALLOCA(snd_seq_queue_tempo, ptr)
int snd_seq_queue_tempo_malloc(snd_seq_queue_tempo_t **ptr);
void snd_seq_queue_tempo_free(snd_seq_queue_tempo_t *ptr);
void snd_seq_queue_tempo_copy(snd_seq_queue_tempo_t *dst, const snd_seq_queue_tempo_t *src);

int snd_seq_queue_tempo_get_queue(const snd_seq_queue_tempo_t *info);
unsigned int snd_seq_queue_tempo_get_tempo(const snd_seq_queue_tempo_t *info);
int snd_seq_queue_tempo_get_ppq(const snd_seq_queue_tempo_t *info);
void snd_seq_queue_tempo_set_tempo(snd_seq_queue_tempo_t *info, unsigned int tempo);
void snd_seq_queue_tempo_set_ppq(snd_seq_queue_tempo_t *info, int ppq);

int snd_seq_get_queue_tempo(snd_seq_t *handle, int q, snd_seq_queue_tempo_t *tempo);
int snd_seq_set_queue_tempo(snd_seq_t *handle, int q, snd_seq_queue_tempo_t *tempo);

/*
 */
size_t snd_seq_queue_timer_sizeof(void);
/** allocate a #snd_seq_queue_timer_t container on stack */
#define snd_seq_queue_timer_alloca(ptr) \
	SND_ALLOCA(snd_seq_queue_timer, ptr)
int snd_seq_queue_timer_malloc(snd_seq_queue_timer_t **ptr);
void snd_seq_queue_timer_free(snd_seq_queue_timer_t *ptr);
void snd_seq_queue_timer_copy(snd_seq_queue_timer_t *dst, const snd_seq_queue_timer_t *src);

int snd_seq_queue_timer_get_queue(const snd_seq_queue_timer_t *info);
int snd_seq_queue_timer_get_type(const snd_seq_queue_timer_t *info);
const snd_timer_id_t *snd_seq_queue_timer_get_id(const snd_seq_queue_timer_t *info);
unsigned int snd_seq_queue_timer_get_resolution(const snd_seq_queue_timer_t *info);

void snd_seq_queue_timer_set_type(snd_seq_queue_timer_t *info, int type);
void snd_seq_queue_timer_set_id(snd_seq_queue_timer_t *info, const snd_timer_id_t *id);
void snd_seq_queue_timer_set_resolution(snd_seq_queue_timer_t *info, unsigned int resolution);

int snd_seq_get_queue_timer(snd_seq_t *handle, int q, snd_seq_queue_timer_t *timer);
int snd_seq_set_queue_timer(snd_seq_t *handle, int q, snd_seq_queue_timer_t *timer);

/** \} */

/**
 *  \defgroup SeqEvent Sequencer Event API
 *  Sequencer Event API
 *  \ingroup Sequencer
 *  \{
 */

snd_seq_event_t *snd_seq_create_event(void);
int snd_seq_free_event(snd_seq_event_t *ev);
ssize_t snd_seq_event_length(snd_seq_event_t *ev);
int snd_seq_event_output(snd_seq_t *handle, snd_seq_event_t *ev);
int snd_seq_event_output_buffer(snd_seq_t *handle, snd_seq_event_t *ev);
int snd_seq_event_output_direct(snd_seq_t *handle, snd_seq_event_t *ev);
int snd_seq_event_input(snd_seq_t *handle, snd_seq_event_t **ev);
int snd_seq_event_input_pending(snd_seq_t *seq, int fetch_sequencer);
int snd_seq_drain_output(snd_seq_t *handle);
int snd_seq_event_output_pending(snd_seq_t *seq);
int snd_seq_extract_output(snd_seq_t *handle, snd_seq_event_t **ev);
int snd_seq_drop_output(snd_seq_t *handle);
int snd_seq_drop_output_buffer(snd_seq_t *handle);
int snd_seq_drop_input(snd_seq_t *handle);
int snd_seq_drop_input_buffer(snd_seq_t *handle);

/** event removal conditionals */
typedef struct _snd_seq_remove_events snd_seq_remove_events_t;

/** Remove conditional flags */
#define SND_SEQ_REMOVE_INPUT		(1<<0)	/**< Flush input queues */
#define SND_SEQ_REMOVE_OUTPUT		(1<<1)	/**< Flush output queues */
#define SND_SEQ_REMOVE_DEST		(1<<2)	/**< Restrict by destination q:client:port */
#define SND_SEQ_REMOVE_DEST_CHANNEL	(1<<3)	/**< Restrict by channel */
#define SND_SEQ_REMOVE_TIME_BEFORE	(1<<4)	/**< Restrict to before time */
#define SND_SEQ_REMOVE_TIME_AFTER	(1<<5)	/**< Restrict to time or after */
#define SND_SEQ_REMOVE_TIME_TICK	(1<<6)	/**< Time is in ticks */
#define SND_SEQ_REMOVE_EVENT_TYPE	(1<<7)	/**< Restrict to event type */
#define SND_SEQ_REMOVE_IGNORE_OFF 	(1<<8)	/**< Do not flush off events */
#define SND_SEQ_REMOVE_TAG_MATCH 	(1<<9)	/**< Restrict to events with given tag */

size_t snd_seq_remove_events_sizeof(void);
/** allocate a #snd_seq_remove_events_t container on stack */
#define snd_seq_remove_events_alloca(ptr) \
	SND_ALLOCA(snd_seq_remove_events, ptr)
int snd_seq_remove_events_malloc(snd_seq_remove_events_t **ptr);
void snd_seq_remove_events_free(snd_seq_remove_events_t *ptr);
void snd_seq_remove_events_copy(snd_seq_remove_events_t *dst, const snd_seq_remove_events_t *src);

unsigned int snd_seq_remove_events_get_condition(const snd_seq_remove_events_t *info);
int snd_seq_remove_events_get_queue(const snd_seq_remove_events_t *info);
const snd_seq_timestamp_t *snd_seq_remove_events_get_time(const snd_seq_remove_events_t *info);
const snd_seq_addr_t *snd_seq_remove_events_get_dest(const snd_seq_remove_events_t *info);
int snd_seq_remove_events_get_channel(const snd_seq_remove_events_t *info);
int snd_seq_remove_events_get_event_type(const snd_seq_remove_events_t *info);
int snd_seq_remove_events_get_tag(const snd_seq_remove_events_t *info);

void snd_seq_remove_events_set_condition(snd_seq_remove_events_t *info, unsigned int flags);
void snd_seq_remove_events_set_queue(snd_seq_remove_events_t *info, int queue);
void snd_seq_remove_events_set_time(snd_seq_remove_events_t *info, const snd_seq_timestamp_t *time);
void snd_seq_remove_events_set_dest(snd_seq_remove_events_t *info, const snd_seq_addr_t *addr);
void snd_seq_remove_events_set_channel(snd_seq_remove_events_t *info, int channel);
void snd_seq_remove_events_set_event_type(snd_seq_remove_events_t *info, int type);
void snd_seq_remove_events_set_tag(snd_seq_remove_events_t *info, int tag);

int snd_seq_remove_events(snd_seq_t *handle, snd_seq_remove_events_t *info);

/** \} */

/**
 *  \defgroup SeqMisc Sequencer Miscellaneous
 *  Sequencer Mescellaneous
 *  \ingroup Sequencer
 *  \{
 */

void snd_seq_set_bit(int nr, void *array);
int snd_seq_change_bit(int nr, void *array);
int snd_seq_get_bit(int nr, void *array);

/** \} */


/**
 *  \defgroup SeqEvType Sequencer Event Type Checks
 *  Sequencer Event Type Checks
 *  \ingroup Sequencer
 *  \{
 */

/* event type macros */
enum {
	SND_SEQ_EVFLG_RESULT,
	SND_SEQ_EVFLG_NOTE,
	SND_SEQ_EVFLG_CONTROL,
	SND_SEQ_EVFLG_QUEUE,
	SND_SEQ_EVFLG_SYSTEM,
	SND_SEQ_EVFLG_MESSAGE,
	SND_SEQ_EVFLG_CONNECTION,
	SND_SEQ_EVFLG_SAMPLE,
	SND_SEQ_EVFLG_USERS,
	SND_SEQ_EVFLG_INSTR,
	SND_SEQ_EVFLG_QUOTE,
	SND_SEQ_EVFLG_NONE,
	SND_SEQ_EVFLG_RAW,
	SND_SEQ_EVFLG_FIXED,
	SND_SEQ_EVFLG_VARIABLE,
	SND_SEQ_EVFLG_VARUSR,
	SND_SEQ_EVFLG_IPC
};

enum {
	SND_SEQ_EVFLG_NOTE_ONEARG,
	SND_SEQ_EVFLG_NOTE_TWOARG
};

enum {
	SND_SEQ_EVFLG_QUEUE_NOARG,
	SND_SEQ_EVFLG_QUEUE_TICK,
	SND_SEQ_EVFLG_QUEUE_TIME,
	SND_SEQ_EVFLG_QUEUE_VALUE
};

/**
 * Exported event type table
 *
 * This table is referred by snd_seq_ev_is_xxx.
 */
extern const unsigned int snd_seq_event_types[];

#define _SND_SEQ_TYPE(x)	(1<<(x))	/**< master type - 24bit */
#define _SND_SEQ_TYPE_OPT(x)	((x)<<24)	/**< optional type - 8bit */

/** check the event type */
#define snd_seq_type_check(ev,x) (snd_seq_event_types[(ev)->type] & _SND_SEQ_TYPE(x))

/** event type check: result events */
#define snd_seq_ev_is_result_type(ev) \
	snd_seq_type_check(ev, SND_SEQ_EVFLG_RESULT)
/** event type check: note events */
#define snd_seq_ev_is_note_type(ev) \
	snd_seq_type_check(ev, SND_SEQ_EVFLG_NOTE)
/** event type check: control events */
#define snd_seq_ev_is_control_type(ev) \
	snd_seq_type_check(ev, SND_SEQ_EVFLG_CONTROL)
/** event type check: channel specific events */
#define snd_seq_ev_is_channel_type(ev) \
	(snd_seq_event_types[(ev)->type] & (_SND_SEQ_TYPE(SND_SEQ_EVFLG_NOTE) | _SND_SEQ_TYPE(SND_SEQ_EVFLG_CONTROL)))

/** event type check: queue control events */
#define snd_seq_ev_is_queue_type(ev) \
	snd_seq_type_check(ev, SND_SEQ_EVFLG_QUEUE)
/** event type check: system status messages */
#define snd_seq_ev_is_message_type(ev) \
	snd_seq_type_check(ev, SND_SEQ_EVFLG_MESSAGE)
/** event type check: system status messages */
#define snd_seq_ev_is_subscribe_type(ev) \
	snd_seq_type_check(ev, SND_SEQ_EVFLG_CONNECTION)
/** event type check: sample messages */
#define snd_seq_ev_is_sample_type(ev) \
	snd_seq_type_check(ev, SND_SEQ_EVFLG_SAMPLE)
/** event type check: user-defined messages */
#define snd_seq_ev_is_user_type(ev) \
	snd_seq_type_check(ev, SND_SEQ_EVFLG_USERS)
/** event type check: instrument layer events */
#define snd_seq_ev_is_instr_type(ev) \
	snd_seq_type_check(ev, SND_SEQ_EVFLG_INSTR)
/** event type check: fixed length events */
#define snd_seq_ev_is_fixed_type(ev) \
	snd_seq_type_check(ev, SND_SEQ_EVFLG_FIXED)
/** event type check: variable length events */
#define snd_seq_ev_is_variable_type(ev)	\
	snd_seq_type_check(ev, SND_SEQ_EVFLG_VARIABLE)
/** event type check: user pointer events */
#define snd_seq_ev_is_varusr_type(ev) \
	snd_seq_type_check(ev, SND_SEQ_EVFLG_VARUSR)
/** event type check: ipc events */
#define snd_seq_ev_is_varipc_type(ev) \
	snd_seq_type_check(ev, SND_SEQ_EVFLG_IPC)
/** event type check: reserved for kernel */
#define snd_seq_ev_is_reserved(ev) \
	(! snd_seq_event_types[(ev)->type])

/**
 * macros to check event flags
 */
/** prior events */
#define snd_seq_ev_is_prior(ev)	\
	(((ev)->flags & SND_SEQ_PRIORITY_MASK) == SND_SEQ_PRIORITY_HIGH)

/** get the data length type */
#define snd_seq_ev_length_type(ev) \
	((ev)->flags & SND_SEQ_EVENT_LENGTH_MASK)
/** fixed length events */
#define snd_seq_ev_is_fixed(ev)	\
	(snd_seq_ev_length_type(ev) == SND_SEQ_EVENT_LENGTH_FIXED)
/** variable length events */
#define snd_seq_ev_is_variable(ev) \
	(snd_seq_ev_length_type(ev) == SND_SEQ_EVENT_LENGTH_VARIABLE)
/** variable length on user-space */
#define snd_seq_ev_is_varusr(ev) \
	(snd_seq_ev_length_type(ev) == SND_SEQ_EVENT_LENGTH_VARUSR)
/** variable length on IPC shm */
#define snd_seq_ev_is_varipc(ev) \
	(snd_seq_ev_length_type(ev) == SND_SEQ_EVENT_LENGTH_VARIPC)

/** time-stamp type */
#define snd_seq_ev_timestamp_type(ev) \
	((ev)->flags & SND_SEQ_TIME_STAMP_MASK)
/** event is in tick time */
#define snd_seq_ev_is_tick(ev) \
	(snd_seq_ev_timestamp_type(ev) == SND_SEQ_TIME_STAMP_TICK)
/** event is in real-time */
#define snd_seq_ev_is_real(ev) \
	(snd_seq_ev_timestamp_type(ev) == SND_SEQ_TIME_STAMP_REAL)

/** time-mode type */
#define snd_seq_ev_timemode_type(ev) \
	((ev)->flags & SND_SEQ_TIME_MODE_MASK)
/** scheduled in absolute time */
#define snd_seq_ev_is_abstime(ev) \
	(snd_seq_ev_timemode_type(ev) == SND_SEQ_TIME_MODE_ABS)
/** scheduled in relative time */
#define snd_seq_ev_is_reltime(ev) \
	(snd_seq_ev_timemode_type(ev) == SND_SEQ_TIME_MODE_REL)

/** direct dispatched events */
#define snd_seq_ev_is_direct(ev) \
	((ev)->queue == SND_SEQ_QUEUE_DIRECT)

/** \} */

#ifdef __cplusplus
}
#endif
/****************************************************************************
 *                                                                          *
 *                      Sequencer Middle Level                              *
 *                                                                          *
 ****************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

/**
 *  \defgroup SeqMiddle Sequencer Middle Level Interface
 *  Sequencer Middle Level Interface
 *  \ingroup Sequencer
 *  \{
 */

/**
 * \brief initialize event record
 * \param ev event record pointer
 */
#define snd_seq_ev_clear(ev) \
	memset(ev, 0, sizeof(snd_seq_event_t))

/**
 * \brief set the explicit destination
 * \param ev event record
 * \param c destination client id
 * \param p destination port id
 */
#define snd_seq_ev_set_dest(ev,c,p) \
	((ev)->dest.client = (c), (ev)->dest.port = (p))

/**
 * \brief set broadcasting to subscribers
 * \param ev event record
 */
#define snd_seq_ev_set_subs(ev) \
	((ev)->dest.client = SND_SEQ_ADDRESS_SUBSCRIBERS,\
	 (ev)->dest.port = SND_SEQ_ADDRESS_UNKNOWN)

/**
 * \brief set broadcasting to all clients/ports
 * \param ev event record
 */
#define snd_seq_ev_set_broadcast(ev) \
	((ev)->dest.client = SND_SEQ_ADDRESS_BROADCAST,\
	 (ev)->dest.port = SND_SEQ_ADDRESS_BROADCAST)

/**
 * \brief set the source port
 * \param ev event record
 * \param p source port id
 */
#define snd_seq_ev_set_source(ev,p) \
	((ev)->source.port = (p))

/**
 * \brief set direct passing mode (without queued)
 * \param ev event instance
 */
#define snd_seq_ev_set_direct(ev) \
	((ev)->queue = SND_SEQ_QUEUE_DIRECT)

/**
 * \brief set tick-scheduling mode on queue
 * \param ev event instance
 * \param q queue id to schedule
 * \param relative relative time-stamp if non-zero
 * \param ttick tick time-stap to be delivered
 */
#define snd_seq_ev_schedule_tick(ev, q, relative, ttick) \
	((ev)->flags &= ~(SND_SEQ_TIME_STAMP_MASK | SND_SEQ_TIME_MODE_MASK),\
	 (ev)->flags |= SND_SEQ_TIME_STAMP_TICK,\
	 (ev)->flags |= (relative) ? SND_SEQ_TIME_MODE_REL : SND_SEQ_TIME_MODE_ABS,\
	 (ev)->time.tick = (ttick),\
	 (ev)->queue = (q))

/**
 * \brief set real-time-scheduling mode on queue
 * \param ev event instance
 * \param q queue id to schedule
 * \param relative relative time-stamp if non-zero
 * \param rtime time-stamp to be delivered
 */
#define snd_seq_ev_schedule_real(ev, q, relative, rtime) \
	((ev)->flags &= ~(SND_SEQ_TIME_STAMP_MASK | SND_SEQ_TIME_MODE_MASK),\
	 (ev)->flags |= SND_SEQ_TIME_STAMP_REAL,\
	 (ev)->flags |= (relative) ? SND_SEQ_TIME_MODE_REL : SND_SEQ_TIME_MODE_ABS,\
	 (ev)->time.time = *(rtime),\
	 (ev)->queue = (q))

/**
 * \brief set event priority
 * \param ev event instance
 * \param high_prior 1 for high priority mode
 */
#define snd_seq_ev_set_priority(ev, high_prior) \
	((ev)->flags &= ~SND_SEQ_PRIORITY_MASK,\
	 (ev)->flags |= (high_prior) ? SND_SEQ_PRIORITY_HIGH : SND_SEQ_PRIORITY_NORMAL)

/**
 * \brief set fixed data
 * \param ev event instance
 *
 * Sets the event length mode as fixed size.
 */
#define snd_seq_ev_set_fixed(ev) \
	((ev)->flags &= ~SND_SEQ_EVENT_LENGTH_MASK,\
	 (ev)->flags |= SND_SEQ_EVENT_LENGTH_FIXED)

/**
 * \brief set variable data
 * \param ev event instance
 * \param datalen length of the external data
 * \param dataptr pointer of the external data
 *
 * Sets the event length mode as variable length and stores the data.
 */
#define snd_seq_ev_set_variable(ev, datalen, dataptr) \
	((ev)->flags &= ~SND_SEQ_EVENT_LENGTH_MASK,\
	 (ev)->flags |= SND_SEQ_EVENT_LENGTH_VARIABLE,\
	 (ev)->data.ext.len = (datalen),\
	 (ev)->data.ext.ptr = (dataptr))

/**
 * \brief set varusr data
 * \param ev event instance
 * \param len length of the external data
 * \param ptr pointer of the external data
 *
 * Sets the event length mode as variable user-space data and stores the data.
 */
#define snd_seq_ev_set_varusr(ev, datalen, dataptr) \
	((ev)->flags &= ~SND_SEQ_EVENT_LENGTH_MASK,\
	 (ev)->flags |= SND_SEQ_EVENT_LENGTH_VARUSR,\
	 (ev)->data.ext.len = (datalen),\
	 (ev)->data.ext.ptr = (dataptr))

/**
 * \brief set queue controls
 * \param ev event record
 * \param typ event type
 * \param q queue id
 * \param val control value
 */
#define snd_seq_ev_set_queue_control(ev, typ, q, val) \
	((ev)->type = (typ),\
	 snd_seq_ev_set_dest(ev, SND_SEQ_CLIENT_SYSTEM, SND_SEQ_PORT_SYSTEM_TIMER),\
	 (ev)->data.queue.queue = (q),\
	 (ev)->data.queue.param.value = (val))

/**
 * \brief set the start queue event
 * \param ev event record
 * \param q queud id to start
 */
#define snd_seq_ev_set_queue_start(ev, q) \
	snd_seq_ev_set_queue_control(ev, SND_SEQ_EVENT_START, q, 0)

/**
 * \brief set the stop queue event
 * \param ev event record
 * \param q queud id to stop
 */
#define snd_seq_ev_set_queue_stop(ev, q) \
	snd_seq_ev_set_queue_control(ev, SND_SEQ_EVENT_STOP, q, 0)

/**
 * \brief set the stop queue event
 * \param ev event record
 * \param q queud id to continue
 */
#define snd_seq_ev_set_queue_continue(ev, q) \
	snd_seq_ev_set_queue_control(ev, SND_SEQ_EVENT_CONTINUE, q, 0)

/**
 * \brief set the stop queue event
 * \param ev event record
 * \param q queud id to change tempo
 * \param val the new tempo value
 */
#define snd_seq_ev_set_queue_tempo(ev, q, val) \
	snd_seq_ev_set_queue_control(ev, SND_SEQ_EVENT_TEMPO, q, val)

/**
 * \brief set the real-time position of a queue
 * \param ev event record
 * \param q queud id to change tempo
 * \param rtime the new real-time pointer
 */
#define snd_seq_ev_set_queue_pos_real(ev, q, rtime) \
	((ev)->type = SND_SEQ_EVENT_SETPOS_TIME,\
	 snd_seq_ev_set_dest(ev, SND_SEQ_CLIENT_SYSTEM, SND_SEQ_PORT_SYSTEM_TIMER),\
	 (ev)->data.queue.queue = (q),\
	 (ev)->data.queue.param.time.time = *(rtime))

/**
 * \brief set the tick-time position of a queue
 * \param ev event record
 * \param q queud id to change tempo
 * \param ttime the new tick-time
 */
#define snd_seq_ev_set_queue_pos_tick(ev, q, ttime) \
	((ev)->type = SND_SEQ_EVENT_SETPOS_TICK,\
	 snd_seq_ev_set_dest(ev, SND_SEQ_CLIENT_SYSTEM, SND_SEQ_PORT_SYSTEM_TIMER),\
	 (ev)->data.queue.queue = (q),\
	 (ev)->data.queue.param.time.tick = (ttime))

/* set and send a queue control event */
int snd_seq_control_queue(snd_seq_t *seq, int q, int type, int value, snd_seq_event_t *ev);

/**
 * \brief start the specified queue
 * \param seq sequencer handle
 * \param q queue id to start
 * \param ev optional event record (see #snd_seq_control_queue)
 */
#define snd_seq_start_queue(seq, q, ev) \
	snd_seq_control_queue(seq, q, SND_SEQ_EVENT_START, 0, ev)

/**
 * \brief stop the specified queue
 * \param seq sequencer handle
 * \param q queue id to stop
 * \param ev optional event record (see #snd_seq_control_queue)
 */
#define snd_seq_stop_queue(seq, q, ev) \
	snd_seq_control_queue(seq, q, SND_SEQ_EVENT_STOP, 0, ev)

/**
 * \brief continue the specified queue
 * \param seq sequencer handle
 * \param q queue id to continue
 * \param ev optional event record (see #snd_seq_control_queue)
 */
#define snd_seq_continue_queue(seq, q, ev) \
	snd_seq_control_queue(seq, q, SND_SEQ_EVENT_CONTINUE, 0, ev)

/**
 * \brief change the tempo of the specified queue
 * \param seq sequencer handle
 * \param q queue id
 * \param tempo the new tempo value
 * \param ev optional event record (see #snd_seq_control_queue)
 */
#define snd_seq_change_queue_tempo(seq, q, tempo, ev) \
	snd_seq_control_queue(seq, q, SND_SEQ_EVENT_TEMPO, tempo, ev)

/* create a port - simple version - return the port number */
int snd_seq_create_simple_port(snd_seq_t *seq, const char *name,
			       unsigned int caps, unsigned int type);
/* delete the port */
int snd_seq_delete_simple_port(snd_seq_t *seq, int port);

/* simple subscription between this port and another port
   (w/o exclusive & time conversion)
   */
int snd_seq_connect_from(snd_seq_t *seq, int my_port, int src_client, int src_port);
int snd_seq_connect_to(snd_seq_t *seq, int my_port, int dest_client, int dest_port);
int snd_seq_disconnect_from(snd_seq_t *seq, int my_port, int src_client, int src_port);
int snd_seq_disconnect_to(snd_seq_t *seq, int my_port, int dest_client, int dest_port);

/*
 * set client information
 */
int snd_seq_set_client_name(snd_seq_t *seq, const char *name);
int snd_seq_set_client_event_filter(snd_seq_t *seq, int event_type);
int snd_seq_set_client_pool_output(snd_seq_t *seq, size_t size);
int snd_seq_set_client_pool_output_room(snd_seq_t *seq, size_t size);
int snd_seq_set_client_pool_input(snd_seq_t *seq, size_t size);

/*
 * parse the given string and get the sequencer address
 */
int snd_seq_parse_address(snd_seq_t *seq, snd_seq_addr_t *addr, const char *str);

/*
 * reset client input/output pool
 */
int snd_seq_reset_pool_output(snd_seq_t *seq);
int snd_seq_reset_pool_input(snd_seq_t *seq);

/**
 * \brief set note event
 * \param ev event record
 * \param ch channel number
 * \param key note key
 * \param vel velocity
 * \param dur duration (in tick or msec)
 */
#define snd_seq_ev_set_note(ev, ch, key, vel, dur) \
	((ev)->type = SND_SEQ_EVENT_NOTE,\
	 snd_seq_ev_set_fixed(ev),\
	 (ev)->data.note.channel = (ch),\
	 (ev)->data.note.note = (key),\
	 (ev)->data.note.velocity = (vel),\
	 (ev)->data.note.duration = (dur))

/**
 * \brief set note-on event
 * \param ev event record
 * \param ch channel number
 * \param key note key
 * \param vel velocity
 */
#define snd_seq_ev_set_noteon(ev, ch, key, vel) \
	((ev)->type = SND_SEQ_EVENT_NOTEON,\
	 snd_seq_ev_set_fixed(ev),\
	 (ev)->data.note.channel = (ch),\
	 (ev)->data.note.note = (key),\
	 (ev)->data.note.velocity = (vel))

/**
 * \brief set note-off event
 * \param ev event record
 * \param ch channel number
 * \param key note key
 * \param vel velocity
 */
#define snd_seq_ev_set_noteoff(ev, ch, key, vel) \
	((ev)->type = SND_SEQ_EVENT_NOTEOFF,\
	 snd_seq_ev_set_fixed(ev),\
	 (ev)->data.note.channel = (ch),\
	 (ev)->data.note.note = (key),\
	 (ev)->data.note.velocity = (vel))

/**
 * \brief set key-pressure event
 * \param ev event record
 * \param ch channel number
 * \param key note key
 * \param vel velocity
 */
#define snd_seq_ev_set_keypress(ev,ch,key,vel) \
	((ev)->type = SND_SEQ_EVENT_KEYPRESS,\
	 snd_seq_ev_set_fixed(ev),\
	 (ev)->data.note.channel = (ch),\
	 (ev)->data.note.note = (key),\
	 (ev)->data.note.velocity = (vel))

/**
 * \brief set MIDI controller event
 * \param ev event record
 * \param ch channel number
 * \param cc controller number
 * \param val control value
 */
#define snd_seq_ev_set_controller(ev,ch,cc,val) \
	((ev)->type = SND_SEQ_EVENT_CONTROLLER,\
	 snd_seq_ev_set_fixed(ev),\
	 (ev)->data.control.channel = (ch),\
	 (ev)->data.control.param = (cc),\
	 (ev)->data.control.value = (val))

/**
 * \brief set program change event
 * \param ev event record
 * \param ch channel number
 * \param val program number
 */
#define snd_seq_ev_set_pgmchange(ev,ch,val) \
	((ev)->type = SND_SEQ_EVENT_PGMCHANGE,\
	 snd_seq_ev_set_fixed(ev),\
	 (ev)->data.control.channel = (ch),\
	 (ev)->data.control.value = (val))

/**
 * \brief set pitchbend event
 * \param ev event record
 * \param ch channel number
 * \param val pitch bend; zero centered from -8192 to 8191
 */
#define snd_seq_ev_set_pitchbend(ev,ch,val) \
	((ev)->type = SND_SEQ_EVENT_PITCHBEND,\
	 snd_seq_ev_set_fixed(ev),\
	 (ev)->data.control.channel = (ch),\
	 (ev)->data.control.value = (val))

/**
 * \brief set channel pressure event
 * \param ev event record
 * \param ch channel number
 * \param val channel pressure value
 */
#define snd_seq_ev_set_chanpress(ev,ch,val) \
	((ev)->type = SND_SEQ_EVENT_CHANPRESS,\
	 snd_seq_ev_set_fixed(ev),\
	 (ev)->data.control.channel = (ch),\
	 (ev)->data.control.value = (val))

/**
 * \brief set sysex event
 * \param ev event record
 * \param datalen length of sysex data
 * \param dataptr sysex data pointer
 *
 * the sysex data must contain the start byte 0xf0 and the end byte 0xf7.
 */
#define snd_seq_ev_set_sysex(ev,datalen,dataptr) \
	((ev)->type = SND_SEQ_EVENT_SYSEX,\
	 snd_seq_ev_set_variable(ev, datalen, dataptr))

/** \} */

#ifdef __cplusplus
}
#endif

/****************************************************************************
 *                                                                          *
 *               Sequencer event <-> MIDI byte stream coder		    *
 *                                                                          *
 ****************************************************************************/

/**
 *  \defgroup MIDI_Event Sequencer event <-> MIDI byte stream coder
 *  \ingroup Sequencer
 *  Sequencer event <-> MIDI byte stream coder
 *  \{
 */

/** container for sequencer midi event parsers */
typedef struct snd_midi_event snd_midi_event_t;

#ifdef __cplusplus
extern "C" {
#endif

int snd_midi_event_new(size_t bufsize, snd_midi_event_t **rdev);
int snd_midi_event_resize_buffer(snd_midi_event_t *dev, size_t bufsize);
void snd_midi_event_free(snd_midi_event_t *dev);
void snd_midi_event_init(snd_midi_event_t *dev);
void snd_midi_event_reset_encode(snd_midi_event_t *dev);
void snd_midi_event_reset_decode(snd_midi_event_t *dev);
/* encode from byte stream - return number of written bytes if success */
long snd_midi_event_encode(snd_midi_event_t *dev, unsigned char *buf, long count, snd_seq_event_t *ev);
int snd_midi_event_encode_byte(snd_midi_event_t *dev, int c, snd_seq_event_t *ev);
/* decode from event to bytes - return number of written bytes if success */
long snd_midi_event_decode(snd_midi_event_t *dev, unsigned char *buf, long count, snd_seq_event_t *ev);

#ifdef __cplusplus
}
#endif

/** \} */

/****************************************************************************
 *                                                                          *
 *                                conv.h                                    *
 *                        Binary Value Conversion                           *
 *                                                                          *
 ****************************************************************************/

/**
 *  \defgroup BConv Binary Value Conversion
 *  Binary Value Conversion
 *  \{
 */

/** convert 16-bit value from host to Little Endian byte order */
#define snd_host_to_LE_16(val)	__cpu_to_le16(val)
/** convert 16-bit value from Little Endian to host byte order */
#define snd_LE_to_host_16(val)	__le16_to_cpu(val)
/** convert 32-bit value from host to Little Endian byte order */
#define snd_host_to_LE_32(val)	__cpu_to_le32(val)
/** convert 32-bit value from Little Endian to host byte order */
#define snd_LE_to_host_32(val)	__le32_to_cpu(val)
/** convert 16-bit value from host to Big Endian byte order */
#define snd_host_to_BE_16(val)	__cpu_to_be16(val)
/** convert 16-bit value from Big Endian to host byte order */
#define snd_BE_to_host_16(val)	__be16_to_cpu(val)
/** convert 32-bit value from host to Big Endian byte order */
#define snd_host_to_BE_32(val)	__cpu_to_be32(val)
/** convert 32-bit value from Big Endian to host byte order */
#define snd_BE_to_host_32(val)	__be32_to_cpu(val)

/** \} */

/****************************************************************************
 *                                                                          *
 *                               instr.h                                    *
 *                          Instrument Interface                            *
 *                                                                          *
 ****************************************************************************/

/**
 *  \defgroup Instrument Instrument Interface
 *  The Instrument Interface.
 *  \{
 */

/* instrument get/put */

#ifdef __cplusplus
extern "C" {
#endif

/** container for sequencer instrument header */
typedef struct _snd_instr_header snd_instr_header_t;

size_t snd_instr_header_sizeof(void);
#define snd_instr_header_alloca(ptr) \
do {\
	assert(ptr);\
	*ptr = (snd_instr_header_t *)alloca(snd_instr_header_sizeof());\
	memset(*ptr, 0, snd_instr_header_sizeof());\
} while (0) /**< allocate instrument header on stack */
int snd_instr_header_malloc(snd_instr_header_t **ptr, size_t len);
void snd_instr_header_free(snd_instr_header_t *ptr);
void snd_instr_header_copy(snd_instr_header_t *dst, const snd_instr_header_t *src);

const snd_seq_instr_t *snd_instr_header_get_id(const snd_instr_header_t *info);
snd_seq_instr_cluster_t snd_instr_header_get_cluster(const snd_instr_header_t *info);
unsigned int snd_instr_header_get_cmd(const snd_instr_header_t *info);
size_t snd_instr_header_get_len(const snd_instr_header_t *info);
const char *snd_instr_header_get_name(const snd_instr_header_t *info);
int snd_instr_header_get_type(const snd_instr_header_t *info);
const char *snd_instr_header_get_format(const snd_instr_header_t *info);
const snd_seq_instr_t *snd_instr_header_get_alias(const snd_instr_header_t *info);
void *snd_instr_header_get_data(const snd_instr_header_t *info);
int snd_instr_header_get_follow_alias(const snd_instr_header_t *info);

void snd_instr_header_set_id(snd_instr_header_t *info, const snd_seq_instr_t *id);
void snd_instr_header_set_cluster(snd_instr_header_t *info, snd_seq_instr_cluster_t cluster);
void snd_instr_header_set_cmd(snd_instr_header_t *info, unsigned int cmd);
void snd_instr_header_set_len(snd_instr_header_t *info, size_t len);
void snd_instr_header_set_name(snd_instr_header_t *info, const char *name);
void snd_instr_header_set_type(snd_instr_header_t *info, int type);
void snd_instr_header_set_format(snd_instr_header_t *info, const char *format);
void snd_instr_header_set_alias(snd_instr_header_t *info, const snd_seq_instr_t *instr);
void snd_instr_header_set_follow_alias(snd_instr_header_t *info, int val);

#ifdef __cplusplus
}
#endif

/**
 *  Instrument abstraction layer
 *     - based on events
 */

/** instrument types */
#define SND_SEQ_INSTR_ATYPE_DATA	SNDRV_SEQ_INSTR_ATYPE_DATA	/**< instrument data */
#define SND_SEQ_INSTR_ATYPE_ALIAS	SNDRV_SEQ_INSTR_ATYPE_ALIAS	/**< instrument alias */

/** instrument ASCII identifiers */
#define SND_SEQ_INSTR_ID_DLS1		SNDRV_SEQ_INSTR_ID_DLS1		/**< DLS1 */
#define SND_SEQ_INSTR_ID_DLS2		SNDRV_SEQ_INSTR_ID_DLS2		/**< DLS2 */
#define SND_SEQ_INSTR_ID_SIMPLE		SNDRV_SEQ_INSTR_ID_SIMPLE	/**< Simple Wave */
#define SND_SEQ_INSTR_ID_SOUNDFONT	SNDRV_SEQ_INSTR_ID_SOUNDFONT	/**< SoundFont */
#define SND_SEQ_INSTR_ID_GUS_PATCH	SNDRV_SEQ_INSTR_ID_GUS_PATCH	/**< Gravis Patch */
#define SND_SEQ_INSTR_ID_INTERWAVE	SNDRV_SEQ_INSTR_ID_INTERWAVE	/**< InterWave FFFF */
#define SND_SEQ_INSTR_ID_OPL2_3		SNDRV_SEQ_INSTR_ID_OPL2_3	/**< OPL2/3 FM */
#define SND_SEQ_INSTR_ID_OPL4		SNDRV_SEQ_INSTR_ID_OPL4		/**< OPL4 */

/** instrument types */
#define SND_SEQ_INSTR_TYPE0_DLS1	SNDRV_SEQ_INSTR_TYPE0_DLS1	/**< MIDI DLS v1 */
#define SND_SEQ_INSTR_TYPE0_DLS2	SNDRV_SEQ_INSTR_TYPE0_DLS2	/**< MIDI DLS v2 */
#define SND_SEQ_INSTR_TYPE1_SIMPLE	SNDRV_SEQ_INSTR_TYPE1_SIMPLE	/**< Simple Wave */
#define SND_SEQ_INSTR_TYPE1_SOUNDFONT	SNDRV_SEQ_INSTR_TYPE1_SOUNDFONT	/**< EMU SoundFont */
#define SND_SEQ_INSTR_TYPE1_GUS_PATCH	SNDRV_SEQ_INSTR_TYPE1_GUS_PATCH	/**< Gravis UltraSound Patch */
#define SND_SEQ_INSTR_TYPE1_INTERWAVE	SNDRV_SEQ_INSTR_TYPE1_INTERWAVE	/**< InterWave FFFF */
#define SND_SEQ_INSTR_TYPE2_OPL2_3	SNDRV_SEQ_INSTR_TYPE2_OPL2_3	/**< Yamaha OPL2/3 FM */
#define SND_SEQ_INSTR_TYPE2_OPL4	SNDRV_SEQ_INSTR_TYPE2_OPL4	/**< Yamaha OPL4 */

/** put commands */
#define SND_SEQ_INSTR_PUT_CMD_CREATE	SNDRV_SEQ_INSTR_PUT_CMD_CREATE	/**< create a new layer */
#define SND_SEQ_INSTR_PUT_CMD_REPLACE	SNDRV_SEQ_INSTR_PUT_CMD_REPLACE	/**< replace the old layer with new one */
#define SND_SEQ_INSTR_PUT_CMD_MODIFY	SNDRV_SEQ_INSTR_PUT_CMD_MODIFY	/**< modify the existing layer */
#define SND_SEQ_INSTR_PUT_CMD_ADD	SNDRV_SEQ_INSTR_PUT_CMD_ADD	/**< add one to the existing layer */
#define SND_SEQ_INSTR_PUT_CMD_REMOVE	SNDRV_SEQ_INSTR_PUT_CMD_REMOVE	/**< remove the layer */

/** get commands */
#define SND_SEQ_INSTR_GET_CMD_FULL	SNDRV_SEQ_INSTR_GET_CMD_FULL	/**< get the full data stream */
#define SND_SEQ_INSTR_GET_CMD_PARTIAL	SNDRV_SEQ_INSTR_GET_CMD_PARTIAL	/**< get the partial data stream */

/** free commands */
#define SND_SEQ_INSTR_FREE_CMD_ALL	SNDRV_SEQ_INSTR_FREE_CMD_ALL	/**< remove all matching instruments */
#define SND_SEQ_INSTR_FREE_CMD_PRIVATE	SNDRV_SEQ_INSTR_FREE_CMD_PRIVATE /**< remove only private instruments */
#define SND_SEQ_INSTR_FREE_CMD_CLUSTER	SNDRV_SEQ_INSTR_FREE_CMD_CLUSTER /**< remove only cluster instruments */
#define SND_SEQ_INSTR_FREE_CMD_SINGLE	SNDRV_SEQ_INSTR_FREE_CMD_SINGLE	/**< remove single instrument */


/**
 * FM instrument support
 */

/** FM instrument data structure */
typedef void snd_instr_fm_t;

#ifdef __cplusplus
extern "C" {
#endif

int snd_instr_fm_convert_to_stream(snd_instr_fm_t *fm, const char *name, snd_instr_header_t **put, size_t *size);
int snd_instr_fm_convert_from_stream(snd_instr_header_t *data, size_t size, snd_instr_fm_t **fm);
int snd_instr_fm_free(snd_instr_fm_t *fm);

#ifdef __cplusplus
}
#endif


/**
 * Simple Wave support
 */

/** simple instrument data structure */
typedef void snd_instr_simple_t;

#ifdef __cplusplus
extern "C" {
#endif

int snd_instr_simple_convert_to_stream(snd_instr_simple_t *simple, const char *name, snd_instr_header_t **put, size_t *size);
int snd_instr_simple_convert_from_stream(snd_instr_header_t *data, size_t size, snd_instr_simple_t **simple);
int snd_instr_simple_free(snd_instr_simple_t *simple);

#ifdef __cplusplus
}
#endif


/**
 * InterWave FFFF support
 */

/** IW FFFF instrument data structure */
typedef void snd_instr_iwffff_t;
/** IW FFFF handler */
typedef struct _snd_iwffff_handle snd_iwffff_handle_t;

#ifdef __cplusplus
extern "C" {
#endif

int snd_instr_iwffff_open(snd_iwffff_handle_t **handle, const char *name_fff, const char *name_dta);
int snd_instr_iwffff_open_rom(snd_iwffff_handle_t **handle, int card, int bank, int file);
int snd_instr_iwffff_open_rom_file(snd_iwffff_handle_t **handle, const char *name, int bank, int file);
int snd_instr_iwffff_close(snd_iwffff_handle_t *handle);
int snd_instr_iwffff_load(snd_iwffff_handle_t *handle, int bank, int prg, snd_instr_iwffff_t **iwffff);
int snd_instr_iwffff_convert_to_stream(snd_instr_iwffff_t *iwffff, const char *name, snd_instr_header_t **data, size_t *size);
int snd_instr_iwffff_convert_from_stream(snd_instr_header_t *data, size_t size, snd_instr_iwffff_t **iwffff);
int snd_instr_iwffff_free(snd_instr_iwffff_t *iwffff);

#ifdef __cplusplus
}
#endif

/** \} */

/*
 *
 */
 
#endif /* __ASOUNDLIB_H */
