/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for control of SoundBlaster cards
 */

#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "pcm.h"
#include "mixer.h"
#include "sb.h"

extern struct snd_stru_pcm_hardware snd_sb8_playback;
extern struct snd_stru_pcm_hardware snd_sb8_record;
extern struct snd_stru_pcm_hardware snd_sb16_playback;
extern struct snd_stru_pcm_hardware snd_sb16_record;

int snd_sbdsp_command( sbdsp_t *codec, unsigned char val )
{
  int i;
  
  for ( i = 10000; i; i-- )
    if ( ( INB( SBP( codec, STATUS ) ) & 0x80 ) == 0 ) {
      OUTB( val, SBP( codec, COMMAND ) );
      return 1;
    }
#ifdef SNDCFG_DEBUG
  printk( "snd_sb_dsp_command: timeout (0x%x)\n", val );
#endif
  return 0;
}

int snd_sbdsp_get_byte( sbdsp_t *codec )
{
  int i;
  
  for ( i = 1000; i; i-- )
    if ( INB( SBP( codec, DATA_AVAIL ) ) & 0x80 )
      return INB( SBP( codec, READ ) );
  PRINTD( "ultra sb get byte failed: 0x%x = 0x%x!!!\n", SBP( codec, DATA_AVAIL ), INB( SBP( codec, DATA_AVAIL ) ) );
  return -ENODEV;
}

void snd_sbdsp_mixer_write( sbdsp_t *codec, unsigned char reg, unsigned char data )
{
  unsigned long flags;
  
  CLI( &flags );
  OUTB( reg, SBP( codec, MIXER_ADDR ) );
  snd_delay( 1 );
  OUTB( data, SBP( codec, MIXER_DATA ) );
  snd_delay( 1 );
  STI( &flags );
}

unsigned char snd_sbdsp_mixer_read( sbdsp_t *codec, unsigned char reg )
{
  unsigned long flags;
  unsigned char result;
  
  CLI( &flags );
  OUTB( reg, SBP( codec, MIXER_ADDR ) );
  snd_delay( 1 );
  result = INB( SBP( codec, MIXER_DATA ) );
  snd_delay( 1 );
  STI( &flags );
  return result;
}

int snd_sbdsp_reset( sbdsp_t *codec )
{
  int i;

  OUTB( 1, SBP( codec, RESET ) );
  snd_delay( 1 );
  OUTB( 0, SBP( codec, RESET ) );
  snd_delay( 3 );
  for ( i = 0; i < 1000 && !(INB( SBP( codec, DATA_AVAIL ) ) & 0x80); i++ );
  if ( INB( SBP( codec, READ ) ) != 0xaa ) {
#ifdef SNDCFG_DEBUG
    PRINTK( "sb_reset: failed!!!\n" );
#endif
    return -ENODEV;
  }
  return 0;
}

int snd_sbdsp_version( snd_pcm_t *pcm )
{
  unsigned long flags;
  sbdsp_t *codec;
  int i;
  unsigned int result = -1;
  
  codec = (sbdsp_t *)pcm -> playback.hw.private_data;
  CLI( &flags );
  snd_sbdsp_command( codec, SB_DSP_GET_VERSION );
  for ( i = 100000; i; i-- ) {
    if ( INB( SBP( codec, DATA_AVAIL ) ) & 0x80 ) {
      result = (short)INB( SBP( codec, READ ) ) << 8;
      break;
    }
  }
  for ( i = 100000; i; i-- ) {
    if ( INB( SBP( codec, DATA_AVAIL ) ) & 0x80 ) {
      result |= (short)INB( SBP( codec, READ ) );
      break;
    }
  }
  STI( &flags );
  return result;
}

int snd_sbdsp_probe( snd_pcm_t *pcm )
{
  sbdsp_t *codec;
  int version, hw;
  char *str;

  /*
   *  initialization sequence
   */
  
  codec = (sbdsp_t *)pcm -> playback.hw.private_data;

  if ( snd_sbdsp_reset( codec ) < 0 ) {
    PRINTDD( "SB: [0x%x] reset failed... 0x%x\n", codec -> port, INB( SBP( codec, READ ) ) );
    return -ENODEV;
  }

  version = snd_sbdsp_version( pcm );
  if ( version < 0 ) return -ENODEV;
  
  PRINTDD( "SB [0x%x]: DSP chip found, version = %i.%i\n", codec -> port, version >> 8, version & 0xff );

  hw = SB_HW_AUTO;
  switch ( version >> 8 ) {
    case 1: hw = SB_HW_10; break;
    case 2: hw = !(version & 0xff) ? SB_HW_20 : SB_HW_201; break;
    case 3: hw = SB_HW_PRO; break;
    case 4: hw = SB_HW_16; break;
    default:
      PRINTK( "SB [0x%x]: uknown DSP chip version %i.%i\n", codec -> port, version >> 8, version & 0xff );
      return -ENODEV;
  }
  switch ( codec -> hardware = hw ) {
    case SB_HW_10: str = "Sound Blaster 1.0"; break;
    case SB_HW_20: str = "Sound Blaster 2.0"; break;
    case SB_HW_201: str = "Sound Blaster 2.01+"; break;
    case SB_HW_PRO: str = "Sound Blaster Pro"; break;
    case SB_HW_16: str = "Sound Blaster 16"; break;
    default: str = "???";
  }
  strcpy( pcm -> name, str );
  
  return 0;
}

int snd_sbdsp_sb16_configure( snd_pcm_t *pcm )
{
  sbdsp_t *codec;
  unsigned long flags;
  unsigned char irqreg = 0, dmareg = 0, realirq, realdma;
  
  codec = (sbdsp_t *)pcm -> playback.hw.private_data;
#if 0
  printk( "codec -> irq = %i, codec -> dma8 = %i, codec -> dma16 = %i\n", codec -> irq, codec -> dma8, codec -> dma16 );
#endif
  switch ( codec -> irq ) {
    case 2:
    case 9: irqreg |= 0x01; break;
    case 5: irqreg |= 0x02; break;
    case 7: irqreg |= 0x04; break;
    case 10: irqreg |= 0x10; break;
    default:
      return -EINVAL;
  }
  switch ( codec -> dma8 ) {
    case 0: dmareg |= 0x01; break;
    case 1: dmareg |= 0x02; break;
    case 3: dmareg |= 0x08; break;
    default:
      return -EINVAL;
  }
  switch ( codec -> dma16 ) {
    case 5: dmareg |= 0x20; break;
    case 6: dmareg |= 0x40; break;
    case 7: dmareg |= 0x80; break;
    default:
      return -EINVAL;
  }
  CLI( &flags );
  snd_sbdsp_mixer_write( codec, 0x80, irqreg );
  realirq = snd_sbdsp_mixer_read( codec, 0x80 );
  snd_sbdsp_mixer_write( codec, 0x81, dmareg );
  realdma = snd_sbdsp_mixer_read( codec, 0x81 );  
  STI( &flags );
  if ( (~realirq) & irqreg || (~realdma) & dmareg ) {
    PRINTK( "SB16 [0x%x]: unable to set DMA & IRQ (PnP device?)\n", codec -> port );
    PRINTD( "SB16 [0x%x]: irqreg=0x%x, dmareg=0x%x\n", codec -> port, realirq, realdma );
    return -ENODEV;
  }
  return 0;
}

void snd_sbdsp_free( void *private_data )
{
  snd_free( private_data, sizeof( sbdsp_t ) );
}

snd_pcm_t *snd_sbdsp_new_device( snd_card_t *card,
                                 unsigned short port,
				 unsigned short irqnum,
				 unsigned short dma8num,
				 unsigned short dma16num,
				 unsigned short hardware )
{
  snd_pcm_t *pcm;
  sbdsp_t *codec;

  pcm = snd_pcm_new_device( card );
  if ( !pcm ) return NULL;
  codec = (sbdsp_t *)snd_malloc( sizeof( sbdsp_t ) );
  if ( !codec ) return NULL;
  MEMSET( codec, 0, sizeof( sbdsp_t ) );
  codec -> pcm = pcm;
  codec -> card = pcm -> card;
  codec -> port = port;
  codec -> irqnum = irqnum;
  codec -> irq = pcm -> card -> irqs[ irqnum ] -> irq;
  codec -> dma8num = dma8num;
  if ( dma8num != SND_DMA_DISABLE )
    codec -> dma8 = pcm -> card -> dmas[ dma8num ] -> dma;
  codec -> dma16num = dma16num;
  if ( dma16num != SND_DMA_DISABLE )
    codec -> dma16 = pcm -> card -> dmas[ dma16num ] -> dma;
  codec -> hardware = hardware;
  pcm -> playback.hw.private_data = codec;
  pcm -> record.hw.private_data = codec;
  pcm -> record.hw.private_free = snd_sbdsp_free;
  strcpy( pcm -> id, "SB" );
  strcpy( pcm -> name, "Sound Blaster" );
  if ( snd_sbdsp_probe( pcm ) < 0 ) {
    snd_pcm_free( pcm );
    return NULL;
  }
  pcm -> flags |= SND_PCM_LFLG_USED;
  pcm -> info_flags = SND_PCM_INFO_CODEC | SND_PCM_INFO_MMAP |
                      SND_PCM_INFO_PLAYBACK | SND_PCM_INFO_RECORD;
  switch ( codec -> hardware ) {
    case SB_HW_10:
    case SB_HW_20:
    case SB_HW_201:
    case SB_HW_PRO:
      MEMCPY( &pcm -> playback.hw, &snd_sb8_playback, sizeof( snd_sb8_playback ) );
      MEMCPY( &pcm -> record.hw, &snd_sb8_record, sizeof( snd_sb8_record ) );
      break;
    case SB_HW_16:
      MEMCPY( &pcm -> playback.hw, &snd_sb16_playback, sizeof( snd_sb16_playback ) );
      MEMCPY( &pcm -> record.hw, &snd_sb16_record, sizeof( snd_sb16_record ) );
      break;
  }
  pcm -> playback.hw.private_data = codec;
  pcm -> record.hw.private_data = codec;
  pcm -> info_flags = SND_PCM_INFO_DSP | SND_PCM_INFO_PLAYBACK |
                      SND_PCM_INFO_RECORD | SND_PCM_INFO_MMAP;
  switch ( codec -> hardware ) {
    case SB_HW_10:
      pcm -> playback.hw.flags &= ~SND_PCM_HW_AUTODMA;	/* ugh, antique !!! */
      pcm -> record.hw.flags &= ~SND_PCM_HW_AUTODMA;	/* ugh, antique !!! */
      break;
    case SB_HW_PRO:
      pcm -> playback.hw.max_voices = 
      pcm -> record.hw.max_voices = 2;
      /* follow thru */
    case SB_HW_201:
      pcm -> playback.hw.max_rate = 44100;
      pcm -> record.hw.max_rate = 15000;
      break;
    case SB_HW_16:
      pcm -> info_flags |= SND_PCM_INFO_DUPLEX | SND_PCM_INFO_DUPLEX_LIMIT;
      break;
  }
  return pcm;
}

/*
 *  INIT part
 */
 
int init_module( void )
{
  return 0;
}

void cleanup_module( void )
{
}
