/*
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 *  Routines for control of SoundBlaster mixer
 */

#include "driver.h"
#include "pcm.h"
#include "mixer.h"
#include "sb.h"

static void snd_sbmixer_write( unsigned short port, unsigned char reg, unsigned char data )
{
  unsigned long flags;
  
  CLI( &flags );
  OUTB( reg, SBP1( port, MIXER_ADDR ) );
  snd_delay( 1 );
  OUTB( data, SBP1( port, MIXER_DATA ) );
  snd_delay( 1 );
  STI( &flags );
}

static unsigned char snd_sbmixer_read( unsigned short port, unsigned char reg )
{
  unsigned long flags;
  unsigned char result;
  
  CLI( &flags );
  OUTB( reg, SBP1( port, MIXER_ADDR ) );
  snd_delay( 1 );
  result = INB( SBP1( port, MIXER_DATA ) );
  snd_delay( 1 );
  STI( &flags );
  return result;
}

/*
 *    private_value
 *		0x000000ff - register
 *              0x00000f00 - left shift
 *              0x0000f000 - right shift
 */

static void snd_sbmixer_record_source( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, int enable )
{  
  unsigned long flags;
  unsigned char mixs = SB_DSP_MIXS_NONE, mask;
  
  switch ( channel -> hw.priority ) {
    case SND_MIXER_PRI_MIC:	mask = 1; break;
    case SND_MIXER_PRI_CD:	mask = 2; break;
    case SND_MIXER_PRI_LINE:	mask = 4; break;
    default: 			mask = 0; break;	/* master */
  }
  if ( enable )
    mixer -> private_value |= mask;
   else
    mixer -> private_value &= ~mask;

  if ( mixer -> private_value == 0 ) mixs = SB_DSP_MIXS_NONE; else
  if ( mixer -> private_value & 1 ) mixs = SB_DSP_MIXS_MIC; else
  if ( mixer -> private_value & 2 ) mixs = SB_DSP_MIXS_CD; else
  mixs = SB_DSP_MIXS_LINE;
        
  CLI( &flags );
  mixs |= snd_sbmixer_read( mixer -> port, SB_DSP_RECORD_SOURCE ) & ~6;
  snd_sbmixer_write( mixer -> port, SB_DSP_RECORD_SOURCE, mixs );
  STI( &flags );
}

static void snd_sbmixer_volume_level( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, int left, int right )
{
  unsigned long flags;
  unsigned char reg, left_shift, right_shift, data, mask;

  reg = (unsigned char)channel -> hw.private_value;
  left_shift = (unsigned char)((channel -> hw.private_value >> 8) & 0x0f);
  right_shift = (unsigned char)((channel -> hw.private_value >> 12) & 0x0f);
#if 0
  PRINTK( "volume: reg = 0x%x, left_shift = 0x%x, right_shift = 0x%x\n", reg, left_shift, right_shift );
#endif
  left <<= left_shift;
  mask = channel -> hw.max << left_shift;
  if ( left_shift != right_shift ) {
    left |= right << right_shift;
    mask |= channel -> hw.max << right_shift;
  }
  CLI( &flags );
  data = snd_sbmixer_read( mixer -> port, reg ) & ~mask;
  snd_sbmixer_write( mixer -> port, reg, left | data );
  STI( &flags );
}

#define SB_DSP_MIXS (sizeof(snd_sbmixer_pro_mixs)/sizeof(struct snd_stru_mixer_channel_hw))
#define SB_DSP_PRIVATE( reg, left_shift, right_shift ) (reg|(left_shift<<8)|(right_shift<<12))

static struct snd_stru_mixer_channel_hw snd_sbmixer_pro_mixs[] = {
  {
    SND_MIXER_PRI_MASTER,	/* priority */
    SND_MIXER_PRI_PARENT,	/* parent priority */
    SND_MIXER_ID_MASTER,	/* device name */
    SND_MIXER_VOLUME,		/* OSS device # */
    0, 1, 0, 0,	0,		/* mute/stereo/record/digital/input */
    0, 7,			/* max value */
    -4600, 0, 575,		/* min, max, step - dB */
    SB_DSP_PRIVATE( SB_DSP_MASTER_DEV, 5, 1 ),
    NULL,			/* compute dB -> linear */
    NULL,			/* compute linear -> dB */
    NULL,			/* record source */
    NULL,			/* set mute */
    snd_sbmixer_volume_level,	/* set volume level */
  },
  {
    SND_MIXER_PRI_PCM,		/* priority */
    SND_MIXER_PRI_MASTER,	/* parent priority */
    SND_MIXER_ID_PCM,		/* device name */
    SND_MIXER_PCM,		/* OSS device # */
    0, 1, 0, 0,	0,		/* mute/stereo/record/digital/input */
    0, 7,			/* max value */
    -4600, 0, 575,		/* min, max, step - dB */
    SB_DSP_PRIVATE( SB_DSP_PCM_DEV, 5, 1 ),
    NULL,			/* compute dB -> linear */
    NULL,			/* compute linear -> dB */
    NULL,			/* record source */
    NULL,			/* set mute */
    snd_sbmixer_volume_level,	/* set volume level */
  },
  {
    SND_MIXER_PRI_LINE,		/* priority */
    SND_MIXER_PRI_MASTER,	/* parent priority */
    SND_MIXER_ID_LINE,		/* device name */
    SND_MIXER_LINE,		/* OSS device # */
    0, 1, 1, 0,	1,		/* mute/stereo/record/digital/input */
    0, 7,			/* max value */
    -4600, 0, 575,		/* min, max, step - dB */
    SB_DSP_PRIVATE( SB_DSP_LINE_DEV, 5, 1 ),
    NULL,			/* compute dB -> linear */
    NULL,			/* compute linear -> dB */
    snd_sbmixer_record_source,	/* record source */
    NULL,			/* set mute */
    snd_sbmixer_volume_level,	/* set volume level */
  },
  {
    SND_MIXER_PRI_CD,		/* priority */
    SND_MIXER_PRI_MASTER,	/* parent priority */
    SND_MIXER_ID_CD,		/* device name */
    SND_MIXER_CD,		/* OSS device # */
    0, 1, 1, 0,	1,		/* mute/stereo/record/digital/input */
    0, 7,			/* max value */
    -4600, 0, 575,		/* min, max, step - dB */
    SB_DSP_PRIVATE( SB_DSP_CD_DEV, 5, 1 ),
    NULL,			/* compute dB -> linear */
    NULL,			/* compute linear -> dB */
    snd_sbmixer_record_source,	/* record source */
    NULL,			/* set mute */
    snd_sbmixer_volume_level,	/* set volume level */
  },
  {
    SND_MIXER_PRI_FM,		/* priority */
    SND_MIXER_PRI_MASTER,	/* parent priority */
    SND_MIXER_ID_FM,		/* device name */
    SND_MIXER_SYNTH,		/* OSS device # */
    0, 1, 0, 0,	0,		/* mute/stereo/record/digital/input */
    0, 7,			/* max value */
    -4600, 0, 575,		/* min, max, step - dB */
    SB_DSP_PRIVATE( SB_DSP_CD_DEV, 5, 1 ),
    NULL,			/* compute dB -> linear */
    NULL,			/* compute linear -> dB */
    snd_sbmixer_record_source,	/* record source */
    NULL,			/* set mute */
    snd_sbmixer_volume_level,	/* set volume level */
  },
  {
    SND_MIXER_PRI_MIC,		/* priority */
    SND_MIXER_PRI_MASTER,	/* parent priority */
    SND_MIXER_ID_MIC,		/* device name */
    SND_MIXER_MIC,		/* OSS device # */
    0, 0, 1, 0,	0,		/* mute/stereo/record/digital/input */
    0, 3,			/* max value */
    -4600, 0, 1150,		/* min, max, step - dB */
    SB_DSP_PRIVATE( SB_DSP_MIC_DEV, 1, 1 ),
    NULL,			/* compute dB -> linear */
    NULL,			/* compute linear -> dB */
    snd_sbmixer_record_source,	/* record source */
    NULL,			/* set mute */
    snd_sbmixer_volume_level,	/* set volume level */
  }
};

/*
 *    private_value
 *		0x000000ff - register
 *              0x00000f00 - left shift
 *              0x0000f000 - right shift
 */

static void snd_sb16mixer_record_source( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, int enable )
{  
  unsigned long flags;
  unsigned int mask;
  
  switch ( channel -> hw.priority ) {
    case SND_MIXER_PRI_MIC:	mask = 0x01; break;
    case SND_MIXER_PRI_CD:	mask = 0x04; break;
    case SND_MIXER_PRI_LINE:	mask = 0x10; break;
    case SND_MIXER_PRI_SYNTHESIZER: mask = 0x40; break;
    default: 			mask = 0; break;	/* master */
  }
  if ( enable )
    mixer -> private_value |= mask;
   else
    mixer -> private_value &= ~mask;

  CLI( &flags );
  snd_sbmixer_write( mixer -> port, SB_DSP4_INPUT_LEFT, mixer -> private_value & 0xff );
  snd_sbmixer_write( mixer -> port, SB_DSP4_INPUT_RIGHT, mixer -> private_value & 0xff );
  STI( &flags );
}

static void snd_sb16mixer_volume_level( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, int left, int right )
{
  unsigned long flags;
  unsigned char reg, left_shift, right_shift, data, mask;

  reg = (unsigned char)channel -> hw.private_value;
  left_shift = (unsigned char)((channel -> hw.private_value >> 8) & 0x0f);
  right_shift = (unsigned char)((channel -> hw.private_value >> 12) & 0x0f);
#if 0
  PRINTK( "volume: reg = 0x%x, left_shift = 0x%x, right_shift = 0x%x\n", reg, left_shift, right_shift );
#endif
  left <<= left_shift;
  mask = channel -> hw.max << left_shift;
  right <<= right_shift;
  CLI( &flags );
  data = snd_sbmixer_read( mixer -> port, reg ) & ~mask;
  snd_sbmixer_write( mixer -> port, reg, left | data );
  if ( channel -> hw.stereo ) {
    data = snd_sbmixer_read( mixer -> port, reg + 1 ) & ~mask;
    snd_sbmixer_write( mixer -> port, reg + 1, right | data );
  }
  STI( &flags );
}

#define SB_DSP4_MIXS (sizeof(snd_sbmixer_16_mixs)/sizeof(struct snd_stru_mixer_channel_hw))
#define SB_DSP4_PRIVATE( reg, left_shift, right_shift ) (reg|(left_shift<<8)|(right_shift<<12))

static struct snd_stru_mixer_channel_hw snd_sbmixer_16_mixs[] = {
  {
    SND_MIXER_PRI_MASTER,	/* priority */
    SND_MIXER_PRI_PARENT,	/* parent priority */
    SND_MIXER_ID_MASTER,	/* device name */
    SND_MIXER_VOLUME,		/* OSS device # */
    0, 1, 0, 0,	0,		/* mute/stereo/record/digital/input */
    0, 31,			/* min, max value */
    -4600, 0, 143,		/* min, max, step - dB */
    SB_DSP4_PRIVATE( SB_DSP4_PCM_DEV, 3, 3 ),
    NULL,			/* compute dB -> linear */
    NULL,			/* compute linear -> dB */
    NULL,			/* record source */
    NULL,			/* set mute */
    snd_sb16mixer_volume_level,	/* set volume level */
  },
  {
    SND_MIXER_PRI_BASS,		/* priority */
    SND_MIXER_PRI_MASTER,	/* parent priority */
    SND_MIXER_ID_BASS,		/* device name */
    SND_MIXER_BASS,		/* OSS device # */
    0, 1, 0, 0,	0,		/* mute/stereo/record/digital/input */
    0, 15,			/* min, max value */
    -4600, 0, 287,		/* min, max, step - dB */
    SB_DSP4_PRIVATE( SB_DSP4_BASS_DEV, 4, 4 ),
    NULL,			/* compute dB -> linear */
    NULL,			/* compute linear -> dB */
    NULL,			/* record source */
    NULL,			/* set mute */
    snd_sb16mixer_volume_level,	/* set volume level */
  },
  {
    SND_MIXER_PRI_TREBLE,	/* priority */
    SND_MIXER_PRI_MASTER,	/* parent priority */
    SND_MIXER_ID_TREBLE,	/* device name */
    SND_MIXER_TREBLE,		/* OSS device # */
    0, 1, 0, 0, 0,		/* mute/stereo/record/digital/input */
    0, 15,			/* min, max value */
    -4600, 0, 287,		/* min, max, step - dB */
    SB_DSP4_PRIVATE( SB_DSP4_TREBLE_DEV, 4, 4 ),
    NULL,			/* compute dB -> linear */
    NULL,			/* compute linear -> dB */
    NULL,			/* record source */
    NULL,			/* set mute */
    snd_sb16mixer_volume_level,	/* set volume level */
  },
  {
    SND_MIXER_PRI_SYNTHESIZER,	/* priority */
    SND_MIXER_PRI_MASTER,	/* parent priority */
    SND_MIXER_ID_SYNTHESIZER,	/* device name */
    SND_MIXER_SYNTH,		/* OSS device # */
    0, 1, 1, 0, 0,		/* mute/stereo/record/digital/input */
    0, 31,			/* min, max value */
    -4600, 0, 143,		/* min, max, step - dB */
    SB_DSP4_PRIVATE( SB_DSP4_SYNTH_DEV, 3, 3 ),
    NULL,			/* compute dB -> linear */
    NULL,			/* compute linear -> dB */
    snd_sb16mixer_record_source,/* record source */
    NULL,			/* set mute */
    snd_sb16mixer_volume_level,	/* set volume level */
  },
  {
    SND_MIXER_PRI_PCM,		/* priority */
    SND_MIXER_PRI_MASTER,	/* parent priority */
    SND_MIXER_ID_PCM,		/* device name */
    SND_MIXER_PCM,		/* OSS device # */
    0, 1, 0, 0,	0,		/* mute/stereo/record/digital/input */
    0, 31,			/* max value */
    -4600, 0, 143,		/* min, max, step - dB */
    SB_DSP4_PRIVATE( SB_DSP4_PCM_DEV, 3, 3 ),
    NULL,			/* compute dB -> linear */
    NULL,			/* compute linear -> dB */
    NULL,			/* record source */
    NULL,			/* set mute */
    snd_sb16mixer_volume_level,	/* set volume level */
  },
  {
    SND_MIXER_PRI_SPEAKER,	/* priority */
    SND_MIXER_PRI_PARENT,	/* parent priority */
    SND_MIXER_ID_SPEAKER,	/* device name */
    SND_MIXER_SPEAKER,		/* OSS device # */
    0, 0, 0, 0,	0,		/* mute/stereo/record/digital/input */
    0, 3,			/* max value */
    -4600, 0, 1150,		/* min, max, step - dB */
    SB_DSP4_PRIVATE( SB_DSP4_SPEAKER_DEV, 6, 6 ),
    NULL,			/* compute dB -> linear */
    NULL,			/* compute linear -> dB */
    NULL,			/* record source */
    NULL,			/* set mute */
    snd_sb16mixer_volume_level,	/* set volume level */
  },
  {
    SND_MIXER_PRI_LINE,		/* priority */
    SND_MIXER_PRI_PARENT,	/* parent priority */
    SND_MIXER_ID_LINE,		/* device name */
    SND_MIXER_LINE,		/* OSS device # */
    0, 1, 1, 0, 1,		/* mute/stereo/record/digital/input */
    0, 31,			/* max value */
    -4600, 0, 143,		/* min, max, step - dB */
    SB_DSP4_PRIVATE( SB_DSP4_LINE_DEV, 3, 3 ),
    NULL,			/* compute dB -> linear */
    NULL,			/* compute linear -> dB */
    snd_sb16mixer_record_source,/* record source */
    NULL,			/* set mute */
    snd_sb16mixer_volume_level,	/* set volume level */
  },
  {
    SND_MIXER_PRI_MIC,		/* priority */
    SND_MIXER_PRI_PARENT,	/* parent priority */
    SND_MIXER_ID_MIC,		/* device name */
    SND_MIXER_MIC,		/* OSS device # */
    0, 0, 1, 0,	1,		/* mute/stereo/record/digital/input */
    0, 31,			/* max value */
    -4600, 0, 143,		/* min, max, step - dB */
    SB_DSP4_PRIVATE( SB_DSP4_MIC_DEV, 3, 3 ),
    NULL,			/* compute dB -> linear */
    NULL,			/* compute linear -> dB */
    snd_sb16mixer_record_source,/* record source */
    NULL,			/* set mute */
    snd_sb16mixer_volume_level,	/* set volume level */
  },
  {
    SND_MIXER_PRI_CD,		/* priority */
    SND_MIXER_PRI_MASTER,	/* parent priority */
    SND_MIXER_ID_CD,		/* device name */
    SND_MIXER_CD,		/* OSS device # */
    0, 1, 1, 0, 1,		/* mute/stereo/record/digital/input */
    0, 31,			/* max value */
    -4600, 0, 143,		/* min, max, step - dB */
    SB_DSP4_PRIVATE( SB_DSP4_CD_DEV, 3, 3 ),
    NULL,			/* compute dB -> linear */
    NULL,			/* compute linear -> dB */
    snd_sb16mixer_record_source,/* record source */
    NULL,			/* set mute */
    snd_sb16mixer_volume_level,	/* set volume level */
  },
  {
    SND_MIXER_PRI_IGAIN,	/* priority */
    SND_MIXER_PRI_PARENT,	/* parent priority */
    SND_MIXER_ID_IGAIN,		/* device name */
    SND_MIXER_IGAIN,		/* OSS device # */
    0, 1, 0, 0, 1,		/* mute/stereo/record/digital/input */
    0, 3,			/* max value */
    -4600, 0, 1150,		/* min, max, step - dB */
    SB_DSP4_PRIVATE( SB_DSP4_IGAIN_DEV, 6, 6 ),
    NULL,			/* compute dB -> linear */
    NULL,			/* compute linear -> dB */
    NULL,			/* record source */
    NULL,			/* set mute */
    snd_sb16mixer_volume_level,	/* set volume level */
  },
  {
    SND_MIXER_PRI_OGAIN,	/* priority */
    SND_MIXER_PRI_PARENT,	/* parent priority */
    SND_MIXER_ID_OGAIN,		/* device name */
    SND_MIXER_OGAIN,		/* OSS device # */
    0, 1, 0, 0,	0,		/* mute/stereo/record/digital/input */
    0, 3,			/* max value */
    -4600, 0, 1150,		/* min, max, step - dB */
    SB_DSP4_PRIVATE( SB_DSP4_OGAIN_DEV, 6, 6 ),
    NULL,			/* compute dB -> linear */
    NULL,			/* compute linear -> dB */
    NULL,			/* record source */
    NULL,			/* set mute */
    snd_sb16mixer_volume_level,	/* set volume level */
  }
};

snd_kmixer_t *snd_sbdsp_new_mixer( snd_card_t *card,
                                   unsigned short port,
                                   char *name,
                                   unsigned short hardware )
{
  int idx, count;
  snd_kmixer_t *mixer;
  snd_kmixer_channel_t *channel;
  struct snd_stru_mixer_channel_hw *phw;

  switch ( hardware ) {
    case SB_HW_PRO:
      count = SB_DSP_MIXS;
      phw = snd_sbmixer_pro_mixs;
      break;
    case SB_HW_16:
      count = SB_DSP4_MIXS;
      phw = snd_sbmixer_16_mixs;
      break;
    default:
      return NULL;
  }
  if ( !card ) return NULL;
  mixer = snd_mixer_new( card, "SB", name, port );
  if ( !mixer ) return NULL;
  for ( idx = 0; idx < count; idx++ ) {
    channel = snd_mixer_new_channel( mixer, &phw[ idx ] );
    if ( !channel ) {
      snd_mixer_free( mixer );
      return NULL;
    }
  }
  mixer -> hw.caps = SND_MIXER_INFO_CAP_EXCL_RECORD;
  return mixer;
}
