/*
 *  Digital Audio (PCM) abstract layer
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 */

#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "pcm.h"
#include "control.h"
#include "ulaw.h"

snd_pcm_t *snd_pcm_devices[ SND_CARDS * SND_PCM_DEVICES ];
int (*snd_pcm_oss_register)( unsigned short minor ) = NULL;
int (*snd_pcm_oss_unregister)( unsigned short minor ) = NULL;

#if 0
#define SND_PCM_DEBUG_BUFFERS
#endif

static void snd_pcm_init_blocks( snd_pcm_t *pcm, int direction );

MUTEX_DEFINE_STATIC( register );

/*
 *
 */

int snd_pcm_dma_alloc( snd_card_t *card, snd_pcm_channel_t *pchn, int dmanum, char *ident, int lock )
{
  struct snd_stru_dma *pdma;

  if ( snd_dma_malloc( card, dmanum, ident, lock ) < 0 )
    return -EBUSY;
  pdma = card -> dmas[ dmanum ];  
  pchn -> _dma = pdma;
  pchn -> dma = pdma -> dma;
  pchn -> size = pdma -> usize;
  pchn -> buffer = pdma -> buf;
  pchn -> frag_size = 0;
  if ( pchn -> size < 4096 ) {
    PRINTD( "Invalid audio DMA size - %i\n", pchn -> size );
    snd_dma_free( card, dmanum, lock );
    return -ENOMEM;
  }
  pchn -> flags |= SND_PCM_FLG_DMAOK;
  return 0;
}

int snd_pcm_dma_free( snd_card_t *card, snd_pcm_channel_t *pchn, int dmanum, int lock )
{
  snd_dma_free( card, dmanum, lock );
  pchn -> flags &= ~SND_PCM_FLG_DMAOK;
  return 0;
}

void snd_pcm_clear_channel( snd_pcm_channel_t *pchn )
{
  pchn -> voices = 1;
  pchn -> mode = 0;
  pchn -> format = 0;
  pchn -> rate = 0;
  pchn -> real_rate = 0;
  pchn -> requested_block_size = 0;
  pchn -> requested_blocks = 0;
  pchn -> requested_subdivision = 0;
  pchn -> processed_bytes = 0;
  pchn -> interrupts = 0;
  pchn -> xruns = 0;
  pchn -> flags = 0;
  pchn -> used_size = 0;
  pchn -> neutral_byte = 0;
  pchn -> _dma = NULL;
  pchn -> dma = 0;
  pchn -> size = 0;
  pchn -> buffer = NULL;
  pchn -> blocks = 0;
  pchn -> block_size = 0;
  pchn -> used = 0;
  pchn -> frag_size = 0;
  pchn -> head = 0;
  pchn -> tail = 0;
  pchn -> blocks_max = 0;
  pchn -> blocks_room = 0;
  pchn -> blocks_min = 0;
  SND_PCM_CLEAR_TIME( pchn );
  SND_PCM_LOCKZERO( pchn );
}

void snd_pcm_fill_with_neutral( snd_pcm_t *pcm, snd_pcm_channel_t *pchn )
{
  unsigned int size;
  
  size = pchn -> used_size;
  if ( pchn -> hw.dma_neutral ) {	/* lowlevel driver does size conversion */
    pchn -> hw.dma_neutral( pcm, pchn -> buffer, 0, size, pchn -> neutral_byte );
    pchn -> flags &= ~SND_PCM_FLG_NEUTRAL;
    return;
  }
  if ( pchn -> hw.flags & SND_PCM_HW_8BITONLY ) {
    if ( pchn -> mode & SND_PCM_MODE_16 ) size >>= 1;
  }
  if ( pchn -> hw.flags & SND_PCM_HW_16BITONLY ) {
    if ( !(pchn -> mode & SND_PCM_MODE_16) ) size <<= 1;
  }
#ifdef SNDCFG_DEBUG
  if ( size > pchn -> size ) {
    PRINTK( "pcm: Oops, neutral fill size %i > DMA size %i\n", size, pchn -> size );
    size = pchn -> size;
  }
#endif
  MEMSET( pchn -> buffer, pchn -> neutral_byte, size );
  pchn -> flags &= ~SND_PCM_FLG_NEUTRAL;
}

/*
 *
 */

unsigned short snd_pcm_file_flags( unsigned short f_flags )
{
  switch ( f_flags & O_ACCMODE ) {
    case O_WRONLY:	return SND_PCM_LFLG_PLAY;
    case O_RDONLY:	return SND_PCM_LFLG_RECORD;
    default:		return SND_PCM_LFLG_BOTH;
  }
}

/*
 *  some shared routines for lowlevel driver
 */

void snd_pcm_playback_dma( snd_pcm_t *pcm,
			   unsigned char *buffer, unsigned int offset,
			   unsigned char *user, unsigned int count )
{
  MEMCPY_FROMFS( &buffer[ offset ], user, count );
}

void snd_pcm_playback_dma_ulaw( snd_pcm_t *pcm,
			        unsigned char *buffer, unsigned int offset,
			        unsigned char *user, unsigned int count )
{
  if ( pcm -> playback.mode & SND_PCM_MODE_ULAW )
    snd_translate_memcpy_fromfs( snd_ulaw_dsp, &buffer[ offset ], user, count );
   else
    MEMCPY_FROMFS( &buffer[ offset ], user, count );
}

void snd_pcm_playback_dma_neutral( snd_pcm_t *pcm,
				   unsigned char *buffer, unsigned int offset,
				   unsigned int count,
				   unsigned char neutral_byte )
{
  MEMSET( &buffer[ offset ], neutral_byte, count );
}

void snd_pcm_record_dma( snd_pcm_t *pcm,
			 unsigned char *buffer, unsigned int offset,
			 unsigned char *user, unsigned int count )
{
  MEMCPY_TOFS( user, &buffer[ offset ], count );
}

void snd_pcm_record_dma_ulaw( snd_pcm_t *pcm,
			      unsigned char *buffer, unsigned int offset,
			      unsigned char *user, unsigned int count )
{
  if ( pcm -> record.mode & SND_PCM_MODE_ULAW )
    snd_translate_memcpy_tofs( snd_dsp_ulaw, user, &buffer[ offset ], count );
   else
    MEMCPY_TOFS( user, &buffer[ offset ], count );
}

void snd_pcm_dma_move( snd_pcm_t *pcm,
                       unsigned char *buffer,
                       unsigned int dest_offset, unsigned int src_offset,
                       unsigned int count )
{
  MEMCPY( &buffer[ dest_offset ], &buffer[ src_offset ], count );
}

/*
 *  interrupt callbacks from lowlevel driver
 */

static inline int snd_pcm_playback_ok( snd_pcm_t *pcm )
{
  unsigned long flags;

  CLI( &flags );
  if ( pcm -> playback.used <= pcm -> playback.blocks_max &&
       pcm -> playback.blocks - pcm -> playback.used >= pcm -> playback.blocks_room ) {
    STI( &flags );
    return 1;
  }
  STI( &flags );
  return 0;
}

static inline int snd_pcm_record_ok( snd_pcm_t *pcm )
{
  unsigned long flags;

  CLI( &flags );
  if ( pcm -> record.used >= pcm -> record.blocks_min ) {
    STI( &flags );
    return 1;
  }
  STI( &flags );
  return 0;
}
 
static void snd_pcm_interrupt_playback( snd_pcm_t *pcm )
{
  unsigned long flags;
  snd_pcm_channel_t *pchn;
  int trigger = 0;
  
  pchn = &pcm -> playback;
  if ( !(pchn -> flags & SND_PCM_FLG_TRIGGER) ) {
    PRINTD( "snd_pcm_interrupt_playback: Oops, playback interrupt when playback isn't active!!!\n" );
    return;
  }
  if ( pchn -> hw.flags & SND_PCM_HW_AUTODMA ) {
    /* try clear first few bytes from next block */
    pchn -> hw.dma_neutral( pcm, pchn -> buffer, pchn -> tail * pchn -> block_size, 16, pchn -> neutral_byte );
  }
  if ( pchn -> used == 1 ) {
    if ( !(pchn -> flags & SND_PCM_FLG_SYNC) ) {
#ifdef SND_PCM_DEBUG_BUFFERS
      printk( "playback: underrun!!! - jiffies = %li\n", jiffies );
#endif
      pchn -> hw.xruns++;
      pchn -> xruns++;
    }
  }
  CLI( &flags );
  pchn -> interrupts++;
  if ( pchn -> used > 0 )
    pchn -> processed_bytes += pchn -> block_size;
  pchn -> tail++;
  pchn -> tail %= pchn -> blocks;
  pchn -> used--;
  if ( pchn -> used > 0 ) {
    SND_PCM_LOCK( pchn, pchn -> tail );
  } else {
    SND_PCM_LOCKZERO( pchn );
    pchn -> head = pchn -> tail = 0;
    STI( &flags );
    if ( pchn -> frag_size > 0 ) {
      pchn -> hw.dma_move( pcm, pchn -> buffer, 0, pchn -> head * pchn -> block_size, pchn -> frag_size );
    }
    CLI( &flags );
  }
  STI( &flags );
#if 0
  printk( "interrupt!!! tail = %i, used = %i\n", pchn -> tail, pchn -> used );
#endif
  if ( pchn -> used > 0 ) trigger = 1;
  if ( trigger && !(pchn -> hw.flags & SND_PCM_HW_AUTODMA) ) {
    pchn -> hw.prepare( pcm,
            		pchn -> buffer, pchn -> used_size,            			
            		pchn -> tail * pchn -> block_size,
                        pchn -> block_size );
    pchn -> hw.trigger( pcm, 1 );
  }
  if ( !trigger ) {
    if ( pchn -> hw.flags & SND_PCM_HW_AUTODMA ) {
      pchn -> hw.trigger( pcm, 0 );
      SND_PCM_CLEAR_TIME( pchn );
    }
    pchn -> flags &= ~SND_PCM_FLG_TRIGGER;
  }
  CLI( &flags );
  if ( (pchn -> flags & SND_PCM_FLG_SLEEP) &&
       snd_pcm_playback_ok( pcm ) ) {
    pchn -> flags &= ~SND_PCM_FLG_SLEEP;
    WAKEUP( pcm, playback );
  }
  STI( &flags );
}

static void snd_pcm_interrupt_record( snd_pcm_t *pcm )
{
  unsigned long flags;
  snd_pcm_channel_t *pchn;
  int trigger;
  
  pchn = &pcm -> record;
  if ( !(pchn -> flags & SND_PCM_FLG_TRIGGER) ) {
    PRINTD( "snd_pcm_interrupt_record: Oops, record interrupt when record isn't active!!!\n" );
    return;
  }
  trigger = 0;
  CLI( &flags );
  if ( pchn -> used < pchn -> blocks ) {
    pchn -> used++;
  } else {
    pchn -> hw.xruns++;
    pchn -> xruns++;
  }
  pchn -> interrupts++;
  pchn -> processed_bytes += pchn -> block_size;
  pchn -> head++;
  pchn -> head %= pchn -> blocks;
  SND_PCM_LOCK( pchn, pchn -> head );
  STI( &flags );
  if ( !(pchn -> flags & SND_PCM_FLG_SYNC) ) trigger = 1;
  if ( trigger && !(pchn -> hw.flags & SND_PCM_HW_AUTODMA) ) {
    pchn -> hw.prepare( pcm,
            		pchn -> buffer, pchn -> used_size,            			
            		pchn -> head * pchn -> block_size,
                        pchn -> block_size );
    pchn -> hw.trigger( pcm, 1 );
  }
  if ( !trigger ) { 
    if ( pchn -> hw.flags & SND_PCM_HW_AUTODMA ) {
      pchn -> hw.trigger( pcm, 0 );
      SND_PCM_CLEAR_TIME( pchn );
    }
    pchn -> flags &= ~SND_PCM_FLG_TRIGGER;
  }
  CLI( &flags );
  if ( (pchn -> flags & SND_PCM_FLG_SLEEP) &&
       snd_pcm_record_ok( pcm ) ) {
    pchn -> flags &= ~SND_PCM_FLG_SLEEP;
    WAKEUP( pcm, record );
  }
  STI( &flags );
}

/*
 *  trigger standard buffered playback
 */

static void snd_pcm_trigger_playback( snd_pcm_t *pcm, snd_pcm_channel_t *pchn )
{
  unsigned long flags;

  if ( !pchn -> used ) return;
  CLI( &flags );
  if ( !(pchn -> flags & SND_PCM_FLG_TRIGGER) ) {
    pchn -> flags |= SND_PCM_FLG_TRIGGERA;
    STI( &flags );
    pchn -> hw.prepare( pcm,
            		pchn -> buffer, pchn -> used_size,            			
            		pchn -> tail * pchn -> block_size,
                        pchn -> block_size );
    SND_PCM_LOCK( pchn, pchn -> tail );
    pchn -> hw.trigger( pcm, 1 );
    if ( pchn -> flags & SND_PCM_FLG_TIME ) {
      do_gettimeofday( &pchn -> time );
    }
    CLI( &flags );
  }
  STI( &flags );
}

/*
 *  user to dma
 */

static int snd_pcm_user_to_dma( struct file *file, snd_pcm_t *pcm, const char *buf, int count )
{
  unsigned long flags;
  snd_pcm_channel_t *pchn;
  int result, tmp;

  if ( count <= 0 || !buf ) return 0;
  if ( VERIFY_AREA( VERIFY_READ, buf, count ) ) return -EACCES;

  pchn = &pcm -> playback;
#if 0
  PRINTK( "pcm_user_to_dma: buf=0x%x, count=0x%x,%i (%s), used = %i, jiffies = %li\n", (int)buf, count, count, file -> f_flags & O_NONBLOCK ? "non blocked" : "blocked", pchn -> used, jiffies );
#endif
  count &= ~( (pchn -> voices * (pchn -> mode & SND_PCM_MODE_16 ? 2 : 1)) - 1 );
  pchn -> flags &= ~SND_PCM_FLG_ABORT;
  if ( pchn -> flags & SND_PCM_FLG_NEUTRAL )
    snd_pcm_fill_with_neutral( pcm, pchn );

  result = 0;

  /*
   * OK. This is patch for .au files which begin with .snd header...
   * This is a little bit hard - it's application work to do conversions...
   */
  if ( (pchn -> mode & SND_PCM_MODE_ULAW) && !pchn -> processed_bytes && !pchn -> used && count > 31 )
    {
      unsigned char buffer[ 8 ];
      
      MEMCPY_FROMFS( buffer, buf, 8 );
      if ( !memcmp( buffer, ".snd", 4 ) )
        {
          tmp = (buffer[4]<<24)|(buffer[5]<<16)|(buffer[6]<<8)|(buffer[7]);
          if ( tmp > count ) tmp = count;
          if ( tmp < 128 ) {
            buf += tmp;
            count -= tmp;
            result += tmp;
          }
        }
    }

  while ( count > 0 )
    {
      while ( !snd_pcm_playback_ok( pcm ) )
        {
          if ( !(pchn -> flags & SND_PCM_FLG_TRIGGER) )
            snd_pcm_trigger_playback( pcm, pchn );
          if ( file -> f_flags & O_NONBLOCK ) return result;
          if ( !(pchn -> flags & SND_PCM_FLG_ENABLE) ) return result;
          CLI( &flags );
          pchn -> flags |= SND_PCM_FLG_SLEEP;
          SLEEP( pcm, playback, 10 * HZ );
          pchn -> flags &= ~SND_PCM_FLG_SLEEP;
          STI( &flags );
          if ( TABORT( pcm, playback ) )
            {
              pchn -> flags |= SND_PCM_FLG_ABORT;
              return result;
            }
          if ( pchn -> used >= pchn -> blocks && TIMEOUT( pcm, playback ) )
            {
              PRINTD( "pcm_user_to_dma: timeout, new block discarded\n" );
              return -EIO;
            }
        }
        
      /* ok.. interresant part.. we must find first free & not locked block.. */
        
      tmp = 0;
      CLI( &flags );
      while ( SND_PCM_ISLOCK( pchn, pchn -> head ) )
        {
          pchn -> hw.discarded++;
#ifdef SND_PCM_DEBUG_BUFFERS
          printk( "playback: discarded!!! %i - %i\n", pchn -> block_lock, pchn -> head );
#endif
          pchn -> used++;
          pchn -> head++;
          pchn -> head %= pchn -> blocks;
          if ( pchn -> used >= pchn -> blocks )
            {
              tmp++;
              break;
            }
        }
      STI( &flags );
      if ( tmp ) continue;		/* go to sleep loop */

      /* ok.. now we have right block - fill it */
      
      tmp = pchn -> block_size - pchn -> frag_size;
      if ( tmp > count ) tmp = count;	/* correction */

#if 0
      printk( "to dma: size = %i\n", tmp );
#endif
      if ( VERIFY_AREA( VERIFY_READ, buf, tmp ) )
        return -EACCES;
      pchn -> hw.dma( pcm,
                      pchn -> buffer,
                      ( pchn -> head * pchn -> block_size ) + pchn -> frag_size,
                      (char *)buf, tmp );

      count -= tmp;
      result += tmp;
      buf += tmp;

      CLI( &flags );
      pchn -> frag_size += tmp;
      if ( pchn -> frag_size >= pchn -> block_size )
        {
#if 0
          printk( "block filled = %i, used = %i, blocks = %i\n", pchn -> head, pchn -> used, pchn -> blocks );
#endif
          pchn -> used++;
          pchn -> head++;
          pchn -> head %= pchn -> blocks;
          pchn -> processed_bytes += pchn -> frag_size;
          pchn -> frag_size = 0;
        }
      STI( &flags );
    }
#if 0
  PRINTK( "pcm_user_to_dma: end\n" );
#endif
  if ( pchn -> used )
    snd_pcm_trigger_playback( pcm, pchn );
  return result;
}

/*
 *  trigger standard buffered record
 */

static void snd_pcm_trigger_record( snd_pcm_t *pcm, snd_pcm_channel_t *pchn )
{
  unsigned long flags;

  if ( pchn -> used >= pchn -> blocks ) return;
  CLI( &flags );
  if ( !(pchn -> flags & SND_PCM_FLG_TRIGGER) ) {
    pchn -> flags |= SND_PCM_FLG_TRIGGERA;
    STI( &flags );
    pchn -> hw.prepare( pcm,
            		pchn -> buffer, pchn -> used_size,            			
            		pchn -> head * pchn -> block_size,
                        pchn -> block_size );
    SND_PCM_LOCK( pchn, pchn -> head );
    pchn -> hw.trigger( pcm, 1 );
    if ( pchn -> flags & SND_PCM_FLG_TIME ) {
      do_gettimeofday( &pchn -> time );
    }
    CLI( &flags );
  }
  STI( &flags );
}

/*
 *  dma to user
 */

static int snd_pcm_dma_to_user( struct file *file, snd_pcm_t *pcm, char *buf, int count )
{
  unsigned long flags;
  snd_pcm_channel_t *pchn;
  int result, tmp;

  if ( count <= 0 ) return 0;
  if ( VERIFY_AREA( VERIFY_WRITE, buf, count ) ) return -EACCES;
  
  pchn = &pcm -> record;
#if 0
  PRINTK( "pcm_user_from_dma: buf=0x%x, count=0x%x (%s)\n", (int)buf, count, file -> f_flags & O_NONBLOCK ? "non blocked" : "blocked" );
#endif
  count &= ~( (pchn -> voices * (pchn -> mode & SND_PCM_MODE_16 ? 2 : 1)) - 1 );
  pchn -> flags &= ~SND_PCM_FLG_ABORT;
  if ( pchn -> flags & SND_PCM_FLG_NEUTRAL )
    snd_pcm_fill_with_neutral( pcm, pchn );

  if ( !(pchn -> flags & SND_PCM_FLG_ENABLE) ) return 0;
  snd_pcm_trigger_record( pcm, pchn );
  result = 0;
  
  while ( count > 0 )
    {
      while ( !snd_pcm_record_ok( pcm ) )
        {
          if ( file -> f_flags & O_NONBLOCK ) return result;
          CLI( &flags );
          pchn -> flags |= SND_PCM_FLG_SLEEP;
          SLEEP( pcm, record, 10 * HZ );
          pchn -> flags &= ~SND_PCM_FLG_SLEEP;
          STI( &flags );
          if ( TABORT( pcm, record ) )
            {
              pchn -> flags |= SND_PCM_FLG_ABORT;
              return -EINTR;
            }
          if ( !pchn -> used && TIMEOUT( pcm, record ) )
            {
              PRINTD( "snd_pcm_dma_to_user: data timeout\n" );
              return -EIO;
            }
        }

      tmp = 0;
      CLI( &flags );
      while ( SND_PCM_ISLOCK( pchn, pchn -> tail ) )
        {
          pchn -> hw.discarded++;
          pchn -> used--;
          pchn -> tail++;
          pchn -> tail %= pchn -> blocks;
          if ( !pchn -> used )
            {
              tmp++;
              break;
            }
        }
      STI( &flags );
      if ( tmp ) continue;		/* go to sleep loop */
        
      tmp = count <= pchn -> frag_size ? count : pchn -> frag_size;
      if ( VERIFY_AREA( VERIFY_WRITE, buf, tmp ) )
        return -EACCES;
      pchn -> hw.dma( pcm,
                      pchn -> buffer,
      		      ( pchn -> tail * pchn -> block_size ) + ( pchn -> block_size - pchn -> frag_size ),
                      buf, tmp );

      buf += tmp;
      count -= tmp;
      result += tmp;

      CLI( &flags );
      pchn -> frag_size -= tmp;
      if ( !pchn -> frag_size )
        {
          pchn -> used--;
          pchn -> tail++;
          pchn -> tail %= pchn -> blocks;
          pchn -> frag_size = pchn -> block_size;
        }
      STI( &flags );
    }

  return result;
}

/*
 *  synchronize playback
 */

static int snd_pcm_drain_playback( snd_pcm_t *pcm )
{
  unsigned long flags;
  snd_pcm_channel_t *pchn;

  pchn = &pcm -> playback;
  CLI( &flags );
  if ( pchn -> flags & SND_PCM_FLG_TRIGGER ) {
    pchn -> hw.trigger( pcm, 0 );
    SND_PCM_CLEAR_TIME( pchn );
    pchn -> flags &= ~SND_PCM_FLG_TRIGGER;
  }
  STI( &flags );
  pchn -> flags |= SND_PCM_FLG_NEUTRAL;
  snd_pcm_fill_with_neutral( pcm, pchn );
  CLI( &flags );
  pchn -> flags &= ~SND_PCM_FLG_TRIGGER1;
  pchn -> head = pchn -> tail = pchn -> used = 0;
  SND_PCM_LOCKZERO( pchn );
  STI( &flags );
  return 0;
}

static int snd_pcm_flush_playback( snd_pcm_t *pcm )
{
  unsigned long flags;
  snd_pcm_channel_t *pchn;

  pchn = &pcm -> playback;
  if ( pchn -> flags & SND_PCM_FLG_ABORT ) {
    snd_pcm_drain_playback( pcm );
    return 0;
  }
  pchn -> flags |= SND_PCM_FLG_SYNC;
  if ( pchn -> used < pchn -> blocks && pchn -> frag_size > 0 )
    {
      unsigned int size = pchn -> frag_size;
      unsigned int count = pchn -> block_size - size;
      unsigned int offset;
      
      if ( count > 0 ) {
        offset = ( pchn -> head * pchn -> block_size ) + size;
        pchn -> hw.dma_neutral( pcm, pchn -> buffer, offset, count, pchn -> neutral_byte );
      }
      pchn -> used++;
      pchn -> head++;
      pchn -> head %= pchn -> blocks;
    }
  if ( !pchn -> used ) {
    pchn -> flags &= ~SND_PCM_FLG_SYNC;
    snd_pcm_drain_playback( pcm );
    return 0;
  }

  snd_pcm_trigger_playback( pcm, pchn );

  while ( pchn -> used ) {
    CLI( &flags );
    pchn -> flags |= SND_PCM_FLG_SLEEP;
    SLEEP( pcm, playback, 10 * HZ );
    pchn -> flags &= ~SND_PCM_FLG_SLEEP;
    STI( &flags );
    if ( TABORT( pcm, playback ) ) {
      pchn -> flags |= SND_PCM_FLG_ABORT;
      snd_pcm_drain_playback( pcm );
      pchn -> flags &= ~SND_PCM_FLG_SYNC;
      return -EINTR;
    }
    if ( pchn -> used && TIMEOUT( pcm, playback ) ) {
      PRINTD( "snd_pcm_flush_playback: timeout, skipping waiting blocks\n" );
      pchn -> flags &= ~SND_PCM_FLG_SYNC;
      return -EAGAIN;
    }
  }
  pchn -> flags &= ~SND_PCM_FLG_SYNC;
  snd_pcm_drain_playback( pcm );  
  return 0;
}

static int snd_pcm_drain_record( snd_pcm_t *pcm )
{
  unsigned long flags;
  snd_pcm_channel_t *pchn;

  pchn = &pcm -> record;
  CLI( &flags );
  if ( pchn -> flags & SND_PCM_FLG_TRIGGER ) {
    pchn -> hw.trigger( pcm, 0 );
    SND_PCM_CLEAR_TIME( pchn );
    pchn -> flags &= ~SND_PCM_FLG_TRIGGER;
  }
  STI( &flags );
  pchn -> flags |= SND_PCM_FLG_NEUTRAL;
  snd_pcm_fill_with_neutral( pcm, pchn );
  CLI( &flags );
  pchn -> flags &= ~SND_PCM_FLG_TRIGGER1;
  pchn -> head = pchn -> tail = pchn -> used = 0;
  pchn -> frag_size = pchn -> block_size;
  SND_PCM_LOCKZERO( pchn );
  STI( &flags );
  return 0;
}

static int snd_pcm_flush_record( snd_pcm_t *pcm )
{
  unsigned long flags;
  snd_pcm_channel_t *pchn;

  pchn = &pcm -> record;
  pchn -> flags |= SND_PCM_FLG_SYNC;
  while ( pchn -> flags & SND_PCM_FLG_TRIGGER ) {	/* still recording? */
    CLI( &flags );
    pchn -> flags |= SND_PCM_FLG_SLEEP;
    SLEEP( pcm, record, 10 * HZ );
    pchn -> flags &= ~SND_PCM_FLG_SLEEP;
    STI( &flags );
    if ( TABORT( pcm, record ) ) {
      pchn -> flags |= SND_PCM_FLG_ABORT;
      snd_pcm_drain_record( pcm );
      return -EINTR;
    }
    if ( (pchn -> flags & SND_PCM_FLG_TRIGGER) && TIMEOUT( pcm, record ) ) {
      PRINTD( "snd_pcm_flush_record: sync timeout\n" );
      pchn -> flags &= ~SND_PCM_FLG_SYNC;
      snd_pcm_drain_record( pcm );
      return -EAGAIN;
    }    
  }
  pchn -> flags &= ~SND_PCM_FLG_SYNC;
  snd_pcm_drain_playback( pcm );
  return 0;
}

/*
 *  other things
 */

static unsigned int snd_pcm_align( snd_pcm_t *pcm, int direction, unsigned int value )
{
  snd_pcm_channel_t *pchn;

  pchn = direction == SND_PCM_PLAYBACK ? &pcm -> playback : &pcm -> record;
  value &= ~(pchn -> voices * (pchn -> mode & SND_PCM_MODE_16 ? 2 : 1));
  value &= ~pchn -> hw.align;
  if ( !value ) value = pchn -> hw.align + 1;
  return value;
}

static void snd_pcm_init_blocks( snd_pcm_t *pcm, int direction )
{
  unsigned int block, blocks, bps, min;
  snd_pcm_channel_t *pchn;
  unsigned long flags;

  pchn = direction == SND_PCM_PLAYBACK ? &pcm -> playback : &pcm -> record;
  /* try make fragment size max 1sec long */
  CLI( &flags );
  pchn -> used_size = pchn -> size;
  if ( pchn -> hw.flags & SND_PCM_HW_8BITONLY ) {
    if ( pchn -> mode & SND_PCM_MODE_16 ) pchn -> used_size <<= 1;
  }
  if ( pchn -> hw.flags & SND_PCM_HW_16BITONLY ) {
    if ( !(pchn -> mode & SND_PCM_MODE_16) ) pchn -> used_size >>= 1;
  }                    
  for ( min = 1, bps = pchn -> hw.min_fragment; bps > 1; bps-- ) min <<= 1;
  bps = pchn -> rate * pchn -> voices;
  if ( pchn -> mode & SND_PCM_MODE_16 ) bps <<= 1;
  if ( pchn -> mode & SND_PCM_MODE_ADPCM ) bps >>= 2;
  block = pchn -> used_size;
  while ( block > bps ) block >>= 1;
  if ( block == pchn -> used_size ) block >>= 1;
  if ( direction == SND_PCM_RECORD )
    block /= 4;			/* small fragment when recording */
  block = snd_pcm_align( pcm, direction, block );
  if ( block < min ) block = min;
  blocks = pchn -> used_size / block;
  pchn -> used_size = blocks * block;
  pchn -> blocks = blocks;
  pchn -> block_size = block;
  pchn -> flags |= SND_PCM_FLG_NEUTRAL;
  if ( direction == SND_PCM_RECORD )
    pchn -> frag_size = pchn -> block_size;
  pchn -> blocks_max = blocks - 1;
  pchn -> blocks_room = 1;
  pchn -> blocks_min = 1;
  STI( &flags );
#ifdef SND_PCM_DEBUG_BUFFERS
  PRINTK( "used_size = %i, blocks = %i, blocks_size = %i\n", pchn -> used_size, pchn -> blocks, pchn -> block_size );
#endif
}

static int snd_pcm_info( snd_pcm_t *pcm, snd_pcm_info_t *xinfo, int space )
{
  snd_pcm_info_t ginfo, *info;

  info = xinfo;
  if ( space == SND_SP_USER ) {
    if ( VERIFY_AREA( VERIFY_WRITE, info, sizeof( snd_pcm_info_t ) ) ) return -EFAULT;
    info = &ginfo;
  }
  MEMSET( &ginfo, 0, sizeof( ginfo ) );
  ginfo.type = pcm -> card -> type;
  ginfo.flags = pcm -> info_flags;
  strncpy( ginfo.id, pcm -> id, sizeof( ginfo.id ) - 1 );
  strncpy( ginfo.name, pcm -> name, sizeof( ginfo.name ) - 1 );
  if ( space == SND_SP_USER ) {
    MEMCPY_TOFS( xinfo, &ginfo, sizeof( ginfo ) );
  }
  return 0;
}

static int snd_pcm_playback_info( snd_pcm_t *pcm, snd_pcm_playback_info_t *xinfo, int space )
{
  snd_pcm_playback_info_t ginfo, *info;
  snd_pcm_channel_t *pchn;
  int i;

  info = xinfo;
  if ( space == SND_SP_USER ) {
    if ( VERIFY_AREA( VERIFY_WRITE, xinfo, sizeof( snd_pcm_playback_info_t ) ) ) return -EFAULT;
    info = &ginfo;
  }
  MEMSET( &ginfo, 0, sizeof( ginfo ) );
  pchn = &pcm -> playback;
  ginfo.flags = pchn -> hw.flags & SND_PCM_HW_COPYMASK;
  ginfo.formats = pchn -> hw.formats;
  ginfo.min_rate = pchn -> hw.min_rate;
  ginfo.max_rate = pchn -> hw.max_rate;
  ginfo.min_channels = 1;
  ginfo.max_channels = pchn -> hw.max_voices;
  ginfo.buffer_size = pchn -> size;
  if ( pchn -> hw.flags & SND_PCM_HW_8BITONLY ) {
    if ( pchn -> mode & SND_PCM_MODE_16 ) ginfo.buffer_size <<= 1;
  }
  if ( pchn -> hw.flags & SND_PCM_HW_16BITONLY ) {
    if ( !(pchn -> mode & SND_PCM_MODE_16) ) ginfo.buffer_size >>= 1;
  }                    
  ginfo.min_fragment_size = 1;
  for ( i = 1; i < pchn -> hw.min_fragment; i++ ) ginfo.min_fragment_size <<= 1;
  ginfo.max_fragment_size = ginfo.buffer_size / 2;
  ginfo.fragment_align = pchn -> hw.align;
  if ( space == SND_SP_USER ) {
    MEMCPY_TOFS( xinfo, &ginfo, sizeof( ginfo ) );
  }
  return 0;
}

static int snd_pcm_record_info( snd_pcm_t *pcm, snd_pcm_record_info_t *xinfo, int space )
{
  snd_pcm_record_info_t ginfo, *info;
  snd_pcm_channel_t *pchn;
  int i;

  info = xinfo;
  if ( space == SND_SP_USER ) {
    if ( VERIFY_AREA( VERIFY_WRITE, xinfo, sizeof( snd_pcm_record_info_t ) ) ) return -EFAULT;
    info = &ginfo;
  }
  MEMSET( &ginfo, 0, sizeof( ginfo ) );
  pchn = &pcm -> record;
  ginfo.flags = pchn -> hw.flags & SND_PCM_HW_COPYMASK;
  ginfo.formats = pchn -> hw.formats;
  ginfo.min_rate = pchn -> hw.min_rate;
  ginfo.max_rate = pchn -> hw.max_rate;
  ginfo.min_channels = 1;
  ginfo.max_channels = pchn -> hw.max_voices;
  ginfo.buffer_size = pchn -> size;
  if ( pchn -> hw.flags & SND_PCM_HW_8BITONLY ) {
    if ( pchn -> mode & SND_PCM_MODE_16 ) ginfo.buffer_size <<= 1;
  }
  if ( pchn -> hw.flags & SND_PCM_HW_16BITONLY ) {
    if ( !(pchn -> mode & SND_PCM_MODE_16) ) ginfo.buffer_size >>= 1;
  }                    
  ginfo.min_fragment_size = 1;
  for ( i = 1; i < pchn -> hw.min_fragment; i++ ) ginfo.min_fragment_size <<= 1;
  ginfo.max_fragment_size = ginfo.buffer_size / 2;
  ginfo.fragment_align = pchn -> hw.align;
  if ( space == SND_SP_USER ) {
    MEMCPY_TOFS( xinfo, &ginfo, sizeof( ginfo ) );
  }
  return 0;
}

static int snd_pcm_format( snd_pcm_t *pcm, int direction, snd_pcm_format_t *format, int space )
{
  snd_pcm_format_t sformat;
  snd_pcm_channel_t *pchn;

  if ( space == SND_SP_USER ) {
    if ( VERIFY_AREA( VERIFY_READ, format, sizeof( *format ) ) ) return -EFAULT;
    MEMCPY_FROMFS( &sformat, format, sizeof( sformat ) );
    format = &sformat;
  }
  pchn = direction == SND_PCM_PLAYBACK ? &pcm -> playback : &pcm -> record;
  if ( format -> rate < pchn -> hw.min_rate ) format -> rate = pchn -> hw.min_rate;
  if ( format -> rate > pchn -> hw.max_rate ) format -> rate = pchn -> hw.max_rate;
  if ( format -> channels < 1 ) format -> channels = 1;
  if ( format -> channels > pchn -> hw.max_voices ) format -> channels = pchn -> hw.max_voices;
  if ( !(pchn -> hw.formats & (1<<format -> format)) ) return -EINVAL;
  if ( direction == SND_PCM_PLAYBACK )
    snd_pcm_flush_playback( pcm );
   else
    snd_pcm_flush_record( pcm );
  pchn -> mode &= ~SND_PCM_MODE_TYPE;
  switch ( format -> format ) {
    case SND_PCM_SFMT_MU_LAW:
      pchn -> mode |= SND_PCM_MODE_U | SND_PCM_MODE_ULAW;
      break;
    case SND_PCM_SFMT_A_LAW:
      pchn -> mode |= SND_PCM_MODE_U | SND_PCM_MODE_ALAW;
      break;
    case SND_PCM_SFMT_IMA_ADPCM:
      pchn -> mode |= SND_PCM_MODE_U | SND_PCM_MODE_ADPCM;
      break;
    case SND_PCM_SFMT_U8:
      pchn -> mode |= SND_PCM_MODE_U;
      break;
    case SND_PCM_SFMT_S16_LE:
      pchn -> mode |= SND_PCM_MODE_16;
      break;
    case SND_PCM_SFMT_S16_BE:
      pchn -> mode |= SND_PCM_MODE_16 | SND_PCM_MODE_BIG;
      break;
    case SND_PCM_SFMT_S8:
      break;
    case SND_PCM_SFMT_U16_LE:
      pchn -> mode |= SND_PCM_MODE_16 | SND_PCM_MODE_U;
      break;
    case SND_PCM_SFMT_U16_BE:
      pchn -> mode |= SND_PCM_MODE_16 | SND_PCM_MODE_BIG | SND_PCM_MODE_U;
      break;
    case SND_PCM_SFMT_MPEG:
      pchn -> mode |= SND_PCM_MODE_MPEG;
      break;
    case SND_PCM_SFMT_GSM:
      pchn -> mode |= SND_PCM_MODE_GSM;
      break;
    default:
      pchn -> mode |= SND_PCM_MODE_U | SND_PCM_MODE_ULAW;
      return -EINVAL;
  }
  pchn -> rate = format -> rate;
  pchn -> hw.compute_rate( pcm );
  pchn -> voices = format -> channels;
  pchn -> flags |= SND_PCM_FLG_BUFFERS;
  pchn -> neutral_byte = pchn -> mode & SND_PCM_MODE_16 ? 0x00 : 0x80;
  return 0;
}

static int snd_pcm_playback_params( snd_pcm_t *pcm, snd_pcm_playback_params_t *params, int space )
{
  snd_pcm_playback_params_t sparams;
  snd_pcm_channel_t *pchn;
  int i, tmp;
  unsigned long flags;

  if ( space == SND_SP_USER ) {
    if ( VERIFY_AREA( VERIFY_READ, params, sizeof( *params ) ) ) return -EFAULT;
    MEMCPY_FROMFS( &sparams, params, sizeof( sparams ) );
    params = &sparams;
  }
  snd_pcm_flush_playback( pcm );
  CLI( &flags );
  pchn = &pcm -> playback;
  for ( tmp = i = 1; i < pchn -> hw.min_fragment; i++ ) tmp <<= 1;
  if ( params -> fragment_size < tmp || params -> fragment_size > pchn -> size / 2 ) {
    STI( &flags );
    return -EINVAL;
  }
  pchn -> used_size = pchn -> size;
  if ( pchn -> hw.flags & SND_PCM_HW_8BITONLY ) {
    if ( pchn -> mode & SND_PCM_MODE_16 ) pchn -> used_size <<= 1;
  }
  if ( pchn -> hw.flags & SND_PCM_HW_16BITONLY ) {
    if ( !(pchn -> mode & SND_PCM_MODE_16) ) pchn -> used_size >>= 1;
  }                    
  pchn -> block_size = snd_pcm_align( pcm, SND_PCM_PLAYBACK, params -> fragment_size );
  pchn -> blocks = pchn -> used_size / pchn -> block_size;
  pchn -> used_size = pchn -> blocks * pchn -> block_size;
  if ( params -> fragments_max < 0 ) {
    pchn -> blocks_max = pchn -> blocks - params -> fragments_max;
    if ( pchn -> blocks_max < 1 ) pchn -> blocks_max = 1;
  } else {
    pchn -> blocks_max = params -> fragments_max;
    if ( pchn -> blocks_max > pchn -> blocks - 1 ) pchn -> blocks_max = pchn -> blocks - 1;
  }
  pchn -> blocks_room = params -> fragments_room;
  if ( pchn -> blocks_room > pchn -> blocks / 2 ) pchn -> blocks_room = pchn -> blocks / 2;
  STI( &flags );
  return 0;
}

static int snd_pcm_record_params( snd_pcm_t *pcm, snd_pcm_record_params_t *params, int space )
{
  snd_pcm_record_params_t sparams;
  snd_pcm_channel_t *pchn;
  int i, tmp;
  unsigned long flags;

  if ( space == SND_SP_USER ) {
    if ( VERIFY_AREA( VERIFY_READ, params, sizeof( *params ) ) ) return -EFAULT;
    MEMCPY_FROMFS( &sparams, params, sizeof( sparams ) );
    params = &sparams;
  }
  snd_pcm_flush_record( pcm );
  CLI( &flags );
  pchn = &pcm -> record;
  for ( tmp = i = 1; i < pchn -> hw.min_fragment; i++ ) tmp <<= 1;
  pchn -> used_size = pchn -> size;
  if ( pchn -> hw.flags & SND_PCM_HW_8BITONLY ) {
    if ( pchn -> mode & SND_PCM_MODE_16 ) pchn -> used_size <<= 1;
  }
  if ( pchn -> hw.flags & SND_PCM_HW_16BITONLY ) {
    if ( !(pchn -> mode & SND_PCM_MODE_16) ) pchn -> used_size >>= 1;
  }                    
  if ( params -> fragment_size < tmp || params -> fragment_size > pchn -> size / 2 ) {
    STI( &flags );
    return -EINVAL;
  }
  pchn -> block_size = snd_pcm_align( pcm, SND_PCM_RECORD, params -> fragment_size );
  pchn -> blocks = pchn -> used_size / pchn -> block_size;
  pchn -> used_size = pchn -> blocks * pchn -> block_size;
  pchn -> blocks_min = params -> fragments_min;
  if ( pchn -> blocks_min > pchn -> blocks / 2 ) pchn -> blocks_min = pchn -> blocks / 2;
  if ( pchn -> blocks_min < 1 ) pchn -> blocks_min = 1;
  STI( &flags );
  return 0;
}

static void snd_pcm_get_time( snd_pcm_channel_t *pchn, struct timeval *time, unsigned int bytes )
{
  unsigned int bps;	/* bytes per second */

  if ( pchn -> time.tv_sec == 0 ) {
    time -> tv_sec = time -> tv_usec = 0;
  } else {
    bps = pchn -> real_rate * pchn -> voices;
    if ( pchn -> mode & SND_PCM_MODE_16 ) bps <<= 1;
    if ( pchn -> mode & SND_PCM_MODE_ADPCM ) bps >>= 2;
    *time = pchn -> time;
    time -> tv_sec += bytes / bps;
    bytes %= bps;
    /* precision should be better :-((, but we have only 2^32 range */
    time -> tv_usec += (((10000UL * bytes) + (bps >> 1)) / bps) * 100UL;
    if ( time -> tv_usec >= 1000000UL ) {
      time -> tv_usec -= 1000000UL;
      time -> tv_sec++;
    } 
  }
}

static int snd_pcm_playback_status( snd_pcm_t *pcm, snd_pcm_playback_status_t *xstatus, int space )
{
  snd_pcm_playback_status_t sstatus, *status;
  snd_pcm_channel_t *pchn;
  unsigned long flags;
  int ptr;

  status = xstatus;
  if ( space == SND_SP_USER ) {
    if ( VERIFY_AREA( VERIFY_READ, xstatus, sizeof( *status ) ) ) return -EFAULT;
    MEMCPY_FROMFS( &sstatus, status, sizeof( sstatus ) );
    status = &sstatus;
  }
  pchn = &pcm -> playback;
  CLI( &flags );
  if ( pchn -> flags & SND_PCM_FLG_TRIGGER ) {
    ptr = pchn -> hw.pointer( pcm, pchn -> used_size );
    if ( pchn -> hw.flags & SND_PCM_HW_AUTODMA ) {
      ptr -= pchn -> tail * pchn -> block_size;
    }
    if ( ptr < 0 ) ptr = 0;
    if ( ptr > pchn -> block_size - 1 ) ptr = pchn -> block_size - 1;
  } else {
    ptr = 0;
  }
  STI( &flags );
  status -> rate = pchn -> real_rate;
  status -> fragments = pchn -> blocks;
  status -> fragment_size = pchn -> block_size;
  CLI( &flags );
  status -> count = (((pchn -> blocks_max + 1) - pchn -> used) * pchn -> block_size) - pchn -> frag_size;
  if ( status -> count < 0 ) status -> count = 0;
  status -> queue = ((pchn -> used * pchn -> block_size) + pchn -> frag_size) - ptr;
  status -> underrun = pchn -> xruns; pchn -> xruns = 0;
  if ( pchn -> flags & SND_PCM_FLG_TIME ) {
    snd_pcm_get_time( pchn, &status -> time, pchn -> processed_bytes + status -> queue );
    status -> stime = pchn -> time;
  } else {
    status -> time.tv_sec = status -> time.tv_usec =
    status -> stime.tv_sec = status -> time.tv_usec = 0;
  }
  STI( &flags );
  if ( space == SND_SP_USER ) {
    MEMCPY_TOFS( xstatus, status, sizeof( snd_pcm_playback_status_t ) );
  }
  return 0;
}

static int snd_pcm_record_status( snd_pcm_t *pcm, snd_pcm_record_status_t *xstatus, int space )
{
  snd_pcm_record_status_t sstatus, *status;
  snd_pcm_channel_t *pchn;
  unsigned long flags;
  int ptr;

  status = xstatus;
  if ( space == SND_SP_USER ) {
    if ( VERIFY_AREA( VERIFY_READ, xstatus, sizeof( *status ) ) ) return -EFAULT;
    MEMCPY_FROMFS( &sstatus, status, sizeof( sstatus ) );
    status = &sstatus;
  }
  pchn = &pcm -> record;
  CLI( &flags );
  if ( pchn -> flags & SND_PCM_FLG_TRIGGER ) {
    ptr = pchn -> hw.pointer( pcm, pchn -> used_size );
    if ( pchn -> hw.flags & SND_PCM_HW_AUTODMA ) {
      ptr -= pchn -> tail * pchn -> block_size;
    }
    if ( ptr < 0 ) ptr = 0;
    if ( ptr > pchn -> block_size - 1 ) ptr = pchn -> block_size - 1;
  } else {
    ptr = 0;
  }
  STI( &flags );
  status -> rate = pchn -> real_rate;
  status -> fragments = pchn -> blocks;
  status -> fragment_size = pchn -> block_size;
  CLI( &flags );
  if ( !pchn -> used )
    status -> count = 0;
   else
    status -> count = (pchn -> used - 1) * pchn -> block_size -
                      (pchn -> block_size - pchn -> frag_size);
  status -> free = ((pchn -> blocks - pchn -> used) * pchn -> block_size) - ptr;
  status -> overrun = pchn -> xruns; pchn -> xruns = 0;
  if ( pchn -> flags & SND_PCM_FLG_TIME ) {
    snd_pcm_get_time( pchn, &status -> time, pchn -> processed_bytes - status -> count );
    status -> stime = pchn -> time;
  } else {
    status -> time.tv_sec = status -> time.tv_usec =
    status -> stime.tv_sec = status -> time.tv_usec = 0;
  }
  STI( &flags );
  if ( space == SND_SP_USER ) {
    MEMCPY_TOFS( xstatus, status, sizeof( snd_pcm_playback_status_t ) );
  }
  return 0;
}

static int snd_pcm_open_card( snd_pcm_t *pcm, struct file *file )
{
  unsigned short flags;
  int res;
  snd_pcm_format_t format;

  flags = snd_pcm_file_flags( file -> f_flags );
#ifdef SND_PCM_DEBUG_BUFFERS
  PRINTK( "snd_pcm_open (1) - pcm = 0x%lx, minor = %i, flags = %i, pcm->flags = 0x%x\n", (long)pcm, minor, flags, pcm -> flags );
#endif
  if ( flags & pcm -> flags ) return -EBUSY;	/* channel(s) already used */
  if ( flags & SND_PCM_LFLG_PLAY )
    if ( !(pcm -> info_flags & SND_PCM_INFO_PLAYBACK) ) return -ENODEV;
  if ( flags & SND_PCM_LFLG_RECORD )
    if ( !(pcm -> info_flags & SND_PCM_INFO_RECORD) ) return -ENODEV;
  if ( flags == SND_PCM_LFLG_BOTH )
    if ( !(pcm -> info_flags & SND_PCM_INFO_DUPLEX) ) return -ENODEV;
  if ( flags & SND_PCM_LFLG_PLAY )
    snd_pcm_clear_channel( &pcm -> playback );
  if ( flags & SND_PCM_LFLG_RECORD )
    snd_pcm_clear_channel( &pcm -> record );
  format.format = SND_PCM_SFMT_MU_LAW;
  format.rate = SND_PCM_DEFAULT_RATE;
  format.channels = 1;
  if ( flags & SND_PCM_LFLG_PLAY )
    {
      if ( ( res = pcm -> playback.hw.open( pcm ) ) < 0 )
        return res;
      pcm -> playback.flags |= SND_PCM_FLG_ENABLE;
      pcm -> playback.ack = snd_pcm_interrupt_playback;
      snd_pcm_format( pcm, SND_PCM_PLAYBACK, &format, SND_SP_KERNEL );
      snd_pcm_init_blocks( pcm, SND_PCM_PLAYBACK );
    }
  if ( flags & SND_PCM_LFLG_RECORD )
    {
      if ( ( res = pcm -> record.hw.open( pcm ) ) < 0 )
        {
          if ( flags & SND_PCM_LFLG_PLAY )
            {
              pcm -> playback.hw.close( pcm );
              pcm -> playback.flags &= ~SND_PCM_FLG_ENABLE;
            }
          return res;
        }
      pcm -> record.flags |= SND_PCM_FLG_ENABLE;
      pcm -> record.ack = snd_pcm_interrupt_record;
      snd_pcm_format( pcm, SND_PCM_RECORD, &format, SND_SP_KERNEL );
      snd_pcm_init_blocks( pcm, SND_PCM_RECORD );
    }
  pcm -> flags |= flags;

#if 0
  printk( "pcm open - done...\n" );
#endif

  return 0;
}

static void snd_pcm_close_card( snd_pcm_t *pcm, struct file *file )
{
  unsigned short flags;
  
  if ( !pcm ) return;
  flags = snd_pcm_file_flags( file -> f_flags ) & pcm -> flags;
#if 0
  PRINTK( "snd_release_pcm - flags = %i\n", flags );
#endif
  if ( flags & SND_PCM_LFLG_PLAY )
    {
      snd_pcm_flush_playback( pcm );		/* synchronize playback */
      pcm -> playback.hw.trigger( pcm, 0 );	/* for sure */
      pcm -> playback.hw.close( pcm );
      pcm -> flags &= ~SND_PCM_LFLG_PLAY;
    }
  if ( flags & SND_PCM_LFLG_RECORD )
    {
      pcm -> playback.flags |= SND_PCM_FLG_ABORT;
      snd_pcm_drain_record( pcm );		/* and record, data can't be read, so flush isn't needed */
      pcm -> record.hw.trigger( pcm, 0 );	/* for sure */
      pcm -> record.hw.close( pcm );
      pcm -> flags &= ~SND_PCM_LFLG_RECORD;
    }
#if 0
  printk( "release pcm: done\n" );
#endif
}

static int snd_pcm_open( unsigned short minor, int cardnum, int device, struct file *file )
{
  int res;
  snd_pcm_t *pcm;

  pcm = snd_pcm_devices[ (cardnum * SND_PCM_DEVICES) + device ];
  if ( !pcm ) return -ENODEV;
  MUTEX_DOWN( pcm, open );
  if ( ( res = snd_pcm_open_card( pcm, file ) ) < 0 ) {
    MUTEX_UP( pcm, open );
    return res;
  }
  file -> private_data = pcm;
  MOD_INC_USE_COUNT;
  pcm -> card -> use_inc();
  MUTEX_UP( pcm, open );
  return 0;
}

static int snd_pcm_release( unsigned short minor, int cardnum, int device, struct file *file )
{
  snd_pcm_t *pcm;
 
  if ( file -> private_data ) {
    pcm = (snd_pcm_t *)file -> private_data;
    MUTEX_DOWN( pcm, open );
    snd_pcm_close_card( pcm, file );
    MUTEX_UP( pcm, open );
    pcm -> card -> use_dec();
  }
  MOD_DEC_USE_COUNT;
  return 0;
}

static int snd_pcm_ioctl( struct file *file, unsigned int cmd, unsigned long arg )
{
  snd_pcm_t *pcm;
  snd_card_t *card;
  unsigned short flags;
  int tmp;

  pcm = (snd_pcm_t *)file -> private_data;
  if ( !(pcm -> flags & SND_PCM_LFLG_VALID) ) return -ENODEV;
  card = pcm -> card;
  if ( ( ( cmd >> 8 ) & 0xff ) != 'A' ) return -EIO;
  flags = snd_pcm_file_flags( file -> f_flags );
#if 0
  printk( "cmd = 0x%x, arg = 0x%x, flags = 0x%x\n", cmd, SND_IOCTL_IN( arg ), flags );
#endif
  switch ( cmd ) {
    case SND_PCM_IOCTL_PVERSION:
      return SND_IOCTL_OUT( arg, SND_PCM_VERSION );
    case SND_PCM_IOCTL_INFO:
      return snd_pcm_info( pcm, (snd_pcm_info_t *)arg, SND_SP_USER );
    case SND_PCM_IOCTL_PLAYBACK_INFO:
      if ( !(flags & SND_PCM_LFLG_PLAY) ) return -EIO;
      return snd_pcm_playback_info( pcm, (snd_pcm_playback_info_t *)arg, SND_SP_USER );
    case SND_PCM_IOCTL_RECORD_INFO:
      if ( !(flags & SND_PCM_LFLG_RECORD) ) return -EIO;
      return snd_pcm_record_info( pcm, (snd_pcm_record_info_t *)arg, SND_SP_USER );
    case SND_PCM_IOCTL_PLAYBACK_FORMAT:
      if ( !(flags & SND_PCM_LFLG_PLAY) ) return -EIO;
      return snd_pcm_format( pcm, SND_PCM_PLAYBACK, (snd_pcm_format_t *)arg, SND_SP_USER );
    case SND_PCM_IOCTL_RECORD_FORMAT:
      if ( !(flags & SND_PCM_LFLG_RECORD) ) return -EIO;
      return snd_pcm_format( pcm, SND_PCM_RECORD, (snd_pcm_format_t *)arg, SND_SP_USER );
    case SND_PCM_IOCTL_PLAYBACK_PARAMS:
      if ( !(flags & SND_PCM_LFLG_PLAY) ) return -EIO;
      return snd_pcm_playback_params( pcm, (snd_pcm_playback_params_t *)arg, SND_SP_USER );
    case SND_PCM_IOCTL_RECORD_PARAMS:
      if ( !(flags & SND_PCM_LFLG_RECORD) ) return -EIO;
      return snd_pcm_record_params( pcm, (snd_pcm_record_params_t *)arg, SND_SP_USER );
    case SND_PCM_IOCTL_PLAYBACK_STATUS:
      if ( !(flags & SND_PCM_LFLG_PLAY) ) return -EIO;
      return snd_pcm_playback_status( pcm, (snd_pcm_playback_status_t *)arg, SND_SP_USER );
    case SND_PCM_IOCTL_RECORD_STATUS:
      if ( !(flags & SND_PCM_LFLG_RECORD) ) return -EIO;
      return snd_pcm_record_status( pcm, (snd_pcm_record_status_t *)arg, SND_SP_USER );
    case SND_PCM_IOCTL_DRAIN_PLAYBACK:
      if ( !(flags & SND_PCM_LFLG_PLAY) ) return -EIO;
      return snd_pcm_drain_playback( pcm );
    case SND_PCM_IOCTL_FLUSH_PLAYBACK:
      if ( !(flags & SND_PCM_LFLG_PLAY) ) return -EIO;
      return snd_pcm_flush_playback( pcm );
    case SND_PCM_IOCTL_FLUSH_RECORD:
      if ( !(flags & SND_PCM_LFLG_RECORD) ) return -EIO;
      return snd_pcm_flush_record( pcm );
    case SND_PCM_IOCTL_PLAYBACK_TIME:
      tmp = SND_IOCTL_IN( arg );
      if ( tmp )
        pcm -> playback.flags |= SND_PCM_FLG_TIME;
       else
        pcm -> playback.flags &= ~SND_PCM_FLG_TIME;
      SND_IOCTL_OUT( arg, tmp != 0 );
      return 0;
    case SND_PCM_IOCTL_RECORD_TIME:
      tmp = SND_IOCTL_IN( arg );
      if ( tmp )
        pcm -> record.flags |= SND_PCM_FLG_TIME;
       else
        pcm -> record.flags &= ~SND_PCM_FLG_TIME;
      SND_IOCTL_OUT( arg, tmp != 0 );
      return 0;
#ifdef SNDCFG_DEBUG
    default:
      PRINTK( "pcm: unknown command = 0x%x\n", cmd ); 
#endif
  }
  return -EIO;
}

static int snd_pcm_control_ioctl( snd_card_t *card, snd_control_t *control, unsigned int cmd, unsigned long arg )
{
  switch ( cmd ) {
    case SND_CTL_IOCTL_HW_INFO:
      {
        struct snd_ctl_hw_info *ptr = (struct snd_ctl_hw_info *)arg;
        if ( snd_pcm_devices[ (card -> number << 1) + 1 ] ) ptr -> pcmdevs = 2; else
        if ( snd_pcm_devices[ (card -> number << 1) + 0 ] ) ptr -> pcmdevs = 1; else
        ptr -> pcmdevs = 0;
        return 0;
      }
    case SND_CTL_IOCTL_PCM_DEVICE:
      {
        int val = SND_IOCTL_IN( arg );
        if ( val < 0 || val > 1 ) return -EINVAL;
        if ( !snd_pcm_devices[ (card -> number << 1) + 1 ] && val > 1 ) return -EINVAL;
        if ( !snd_pcm_devices[ (card -> number << 1) + 0 ] ) return -EINVAL;
        control -> pcm_device = val;
        return 0;
      }
    case SND_CTL_IOCTL_PCM_INFO:
      return snd_pcm_info( snd_pcm_devices[ (card -> number << 1) + control -> pcm_device ], (snd_pcm_info_t *)arg, SND_SP_USER );
  }
  return -EAGAIN;
}

static long snd_pcm_read( struct file *file, char *buf, long count )
{
  snd_pcm_t *pcm;

  pcm = (snd_pcm_t *)file -> private_data;
  if ( !pcm ) return -EIO;
  if ( !(pcm -> flags & SND_PCM_LFLG_VALID) ) return -ENODEV;
  if ( !( snd_pcm_file_flags( file -> f_flags ) & SND_PCM_LFLG_RECORD ) ||
       !( pcm -> flags & SND_PCM_LFLG_RECORD ) ) return -EIO;
  return snd_pcm_dma_to_user( file, pcm, buf, count );
}

static long snd_pcm_write( struct file *file, const char *buf, long count )
{
  snd_pcm_t *pcm;

  pcm = (snd_pcm_t *)file -> private_data;
  if ( !pcm ) return -EIO;
  if ( !(pcm -> flags & SND_PCM_LFLG_VALID) ) return -ENODEV;
  if ( !( snd_pcm_file_flags( file -> f_flags ) & SND_PCM_LFLG_PLAY ) ||
       !( pcm -> flags & SND_PCM_LFLG_PLAY ) ) return -EIO;
  return snd_pcm_user_to_dma( file, pcm, buf, count );
}

#ifdef SND_POLL
static unsigned int snd_pcm_poll( struct file *file, poll_table *wait )
{
  snd_pcm_t *pcm;
  unsigned long flags;
  unsigned int mask;
  snd_pcm_channel_t *record, *playback;
  unsigned short fflags;

  fflags = snd_pcm_file_flags( file -> f_flags );
  pcm = (snd_pcm_t *)file -> private_data;
  if ( !pcm ) return 0;
  if ( !(pcm -> flags & SND_PCM_LFLG_VALID) ) return 0;

  record = &pcm -> record;
  playback = &pcm -> playback;
  
  CLI( &flags );
  if ( fflags & SND_PCM_LFLG_RECORD ) {
    record -> flags |= SND_PCM_FLG_SLEEP;
    SLEEP_POLL( file, pcm, record, wait );
  }
  if ( fflags & SND_PCM_LFLG_PLAY ) {
    playback -> flags |= SND_PCM_FLG_SLEEP;
    SLEEP_POLL( file, pcm, playback, wait );
  }
  STI( &flags );

  mask = 0;
  if ( fflags & SND_PCM_LFLG_RECORD ) {
    if ( snd_pcm_record_ok( pcm ) ) mask |= POLLIN | POLLRDNORM;
  }
  if ( fflags & SND_PCM_LFLG_PLAY ) {
    if ( snd_pcm_playback_ok( pcm ) ) mask |= POLLOUT | POLLWRNORM;
  }

  return mask;
}
#else
static int snd_pcm_select( struct file *file, int sel_type, select_table *wait )
{
  snd_pcm_t *pcm;
  unsigned long flags;
  snd_pcm_channel_t *pchn;
  unsigned short fflags;

  fflags = snd_pcm_file_flags( file -> f_flags );
  pcm = (snd_pcm_t *)file -> private_data;
  if ( !pcm ) return -EIO;
  if ( !(pcm -> flags & SND_PCM_LFLG_VALID) ) return -ENODEV;
  switch ( sel_type ) {
    case SEL_IN:
      if ( !(fflags & SND_PCM_LFLG_RECORD) ) return 0;
      pchn = &pcm -> record;
      CLI( &flags );
      if ( !snd_pcm_record_ok( pcm ) )
        {
          pchn -> flags |= SND_PCM_FLG_SLEEP;
          SLEEPS( pcm, record, wait );
          STI( &flags );
          return 0;
        }
      pchn -> flags &= ~SND_PCM_FLG_SLEEP;
      STI( &flags );
      return 1;
    case SEL_OUT:
      if ( !(fflags & SND_PCM_LFLG_PLAY) ) return 0;
      pchn = &pcm -> playback;
      CLI( &flags );
      if ( !snd_pcm_playback_ok( pcm ) )
        {
          pchn -> flags |= SND_PCM_FLG_SLEEP;
          SLEEPS( pcm, playback, wait );
          STI( &flags );
          return 0;
        }
      pchn -> flags &= ~SND_PCM_FLG_SLEEP;
      STI( &flags );
      return 1;
    case SEL_EX:
      break;
  }
  return 0;
}
#endif

/*
 *  Register functions
 */

static snd_minor_t snd_pcm_reg = {
  "digital audio",
  
  NULL,				/* unregister */
  
  NULL,				/* lseek */
  snd_pcm_read,			/* read */
  snd_pcm_write,		/* write */
  snd_pcm_open,			/* open */
  snd_pcm_release,		/* release */
#ifdef SND_POLL
  snd_pcm_poll,			/* poll */
#else
  snd_pcm_select,		/* select */
#endif
  snd_pcm_ioctl,		/* ioctl */
  NULL,				/* mmap */
};

snd_pcm_t *snd_pcm_new_device( snd_card_t *card )
{
  snd_pcm_t *pcm;

  if ( !card ) return NULL;
  pcm = (snd_pcm_t *)snd_malloc( sizeof( snd_pcm_t ) );
  if ( !pcm ) return NULL;
  MEMSET( pcm, 0, sizeof( snd_pcm_t ) );
  pcm -> flags = SND_PCM_LFLG_NONE;
  pcm -> card = card;
  SLEEP_PREPARE( pcm, playback );
  SLEEP_PREPARE( pcm, record );
  MUTEX_PREPARE( pcm, open );
  return pcm;
}

int snd_pcm_free( snd_pcm_t *pcm )
{
  if ( !pcm ) return -EINVAL;
  if ( pcm -> playback.hw.private_free )
    pcm -> playback.hw.private_free( pcm -> playback.hw.private_data );
  if ( pcm -> record.hw.private_free )
    pcm -> record.hw.private_free( pcm -> record.hw.private_data );
  snd_free( pcm, sizeof( snd_pcm_t ) );
  return 0;  
}

int snd_pcm_register( snd_pcm_t *pcm, int pcm_device )
{
  int idx, err;
  unsigned short minor;

  MUTEX_DOWN_STATIC( register );
  if ( pcm_device < 0 || pcm_device >= SND_PCM_DEVICES ) {
    MUTEX_UP_STATIC( register );
    return -EINVAL;
  }
  idx = (pcm -> card -> number * SND_PCM_DEVICES) + pcm_device;
  if ( snd_pcm_devices[ idx ] ) {
    MUTEX_UP_STATIC( register );
    return -EBUSY;
  }
  pcm -> device = pcm_device;
  snd_pcm_devices[ idx ] = pcm;
  minor = SND_MINOR_PCM + (pcm -> card -> number * SND_PCM_DEVICES) + pcm_device;
  if ( (err = snd_register_minor( minor, &snd_pcm_reg )) < 0 ) {
    snd_pcm_devices[ idx ] = NULL;
    MUTEX_UP_STATIC( register );
    return err;
  }
  if ( pcm -> flags & SND_PCM_LFLG_USED )
    pcm -> flags |= SND_PCM_LFLG_VALID;
  if ( snd_pcm_oss_register )
    snd_pcm_oss_register( minor );
  MUTEX_UP_STATIC( register );
  return 0;
}

int snd_pcm_unregister( snd_pcm_t *pcm )
{
  int idx;
  
  if ( !pcm ) return -EINVAL;
  MUTEX_DOWN_STATIC( register );
  idx = (pcm -> card -> number * SND_PCM_DEVICES) + pcm -> device;
  if ( snd_pcm_devices[ idx ] != pcm ) {
    MUTEX_UP_STATIC( register );
    return -EINVAL;
  }
  snd_unregister_minor( SND_MINOR_PCM + idx );
  if ( snd_pcm_oss_unregister ) {
    snd_pcm_oss_unregister( SND_MINOR_PCM + idx );
  }
  snd_pcm_devices[ idx ] = NULL;
  MUTEX_UP_STATIC( register );
  return snd_pcm_free( pcm );
}

/*
 *  ENTRY functions
 */

int init_module( void )
{
  snd_control_register_ioctl( snd_pcm_control_ioctl );
  return 0;
}

void cleanup_module( void )
{
  snd_control_unregister_ioctl( snd_pcm_control_ioctl );
}
